#include "mbed.h"
#include "JSN_SR04.h"

JSN_SR04::JSN_SR04(PinName echoPin, PinName triggerPin) : echo(echoPin), trigger(triggerPin) {
    init();
}

void JSN_SR04::init() {
    distance = -1;										// Default distance
    minDistance = 2;										// Default min boundary
    maxDistance = 400;									// Default max boundary
    newDataReady = timerStarted = false;			// Indicators are false by default
}

void JSN_SR04::startTimer() {
	// Check if the timer started
   if (!timerStarted) {
		 // start the timer
       timer.start();
       timerStarted = true;

		 // Call the stopTimer() method after a predefined timeout (> 25 ms signal duration indicates that there is no obstacle)
       echoTimeout.attach_us(this, &JSN_SR04::stopTimer, SENSOR_TIMEOUT);

		 // Stop measurement and
       echo.fall(this, &JSN_SR04::stopTimer);
       echo.rise(NULL);
    }
}

void JSN_SR04::calculateDistance() {
	int timer_state;		// Current timer state (ticks amount)

	// Check if timer was even started
	if (timerStarted) {
		timer_state = timer.read();

		// If timeout occured (no obstacle)
		if (timer_state >= SENSOR_TIMEOUT) distance = -1;

		// If there is no timeout (an obstacle detected)
		else {
			// Calculate distance (duration of the signal * constant used in datasheet)
			distance = timer.read() * 1e6 / 58;

			// Set distance to the minimum if it's lower
			if (distance < minDistance)
				distance = minDistance;

				// Set distance to the maximum if it's higher
			if (distance > maxDistance)
				distance = maxDistance;
		}

		// Indicate that new data is ready
		newDataReady = true;
	 }
}

void JSN_SR04::stopTimer() {
	// Stop the timer
   timer.stop();

	// Call the calculation function
	calculateDistance();

	// Reset the timer
   timer.reset();
   timerStarted = false;

	// Switch off the timeout timer event
   echoTimeout.detach();
   echo.fall(NULL);
}

void JSN_SR04::turnOffTrigger() {
	// Switch off the trigger
   trigger = 0;
}

void JSN_SR04::startMeasurement() {
	// Trigger the sensor (by a signal described in the datasheet)
   trigger = 1;
   triggerTimeout.attach_us(this, &JSN_SR04::turnOffTrigger, 100); 

	// Start to measure a response
   echo.rise(this, &JSN_SR04::startTimer);

	// Outdate current response
   newDataReady = false;
}

float JSN_SR04::getDistance_cm() {
	// Indicate that the data was read
   newDataReady = false;

	// Return the data
   return distance;
}

float JSN_SR04::getDistance_mm() {
	// Indicate that the data was read
   newDataReady = false;

	// Return the data
	if (distance != -1)
		return distance * 10;

	else return distance;
}

bool JSN_SR04::isNewDataReady() {
	// Return the indicator
   return newDataReady;
}

void JSN_SR04::setRanges(float minRange, float maxRange) {
	// Set new boundaries that are in range of a sensors reading ability (datasheet)
    if (minRange < maxRange) {
        if (minRange >= 2 && minRange < 400) // bug from revs. 4 and 5 corrected
            minDistance = minRange;
        if (maxRange <= 400)
            maxDistance = maxRange;
    }
}

float JSN_SR04::getMinRange() {
	// Return a lower boundary
   return minDistance;
}

float JSN_SR04::getMaxRange() {
	// Return an upper boundary
   return maxDistance;
}
