// \file ds2482x_app.h \brief Dallas DS2482 I2C-to-Dallas1Wire Master Library. 
//*****************************************************************************
// Target MCU   : Atmel AVR Series
//*****************************************************************************
//

#ifndef DS2482_H
#define DS2482_H

    #include "mbed.h"

    #define OW_MASTER_START 0
    #define OW_MASTER_DEBUG 0
    
    // constants/macros/typdefs
    #define OW_I2C_DEVICE      0x30    // < Change this to the address of your 1-Wire master  

    #define FALSE 0
    #define TRUE  1
    
    #define NAK 0
    #define ACK 1
    
    #define I2C_READ 1
    #define I2C_WRITE 0
    
    #define POLL_LIMIT 100

    // Maximale Anzahl der Bausteine am Bus
    #define OW_MAX_DEVICES 16
    #define OW_SECOND_PORT 8

    // Memory definitions
    #define OW_SCRATCHPAD_BYTES 9
    #define OW_ROM_CODE_BYTES 8
    #define OW_CRC_BYTE 8

 
    //-----------------------------------------------------------------------------

    // Commands
    #define OW_CMD_SEARCH_ROM 0xF0      // Command Search ROM code
    #define OW_CMD_READ_ROM 0x33            // Command Read ROM code
    #define OW_CMD_MATCH_ROM 0x55       // Command Match ROM code
    #define OW_CMD_SKIP_ROM 0xCC            // Command Skip ROM code
    #define OW_CMD_ALARM_SEARCH 0xEC    // Command Alarm Search

    #define OW_CMD_CONVERT_T 0x44               // Command Initiate temperature conversion
    #define OW_CMD_WRITE_SCRATCHPAD 0x4E    // Command Write scratchpad
    #define OW_CMD_READ_SCRATCHPAD 0xBE     // Command Read scratchpad
    #define OW_CMD_COPY_SCRATCHPAD 0x48     // Command Copy scratchpad
    #define OW_CMD_RECALL_EE 0xB8               // Command Recall Th and Tl from EEPROM
    #define OW_CMD_READ_POWER_SUPPLY 0xB4 // Command Signal power supply mode

    // Error codes
    #define OW_RESET_ERROR      0x01
    #define OW_CRC_ERROR        0x02
    
    // DS2482 command defines
    #define DS2482_CMD_DRST     0xF0    //< DS2482 Device Reset
    #define DS2482_CMD_WCFG     0xD2    //< DS2482 Write Configuration
    #define DS2482_CMD_CHSL     0xC3    //< DS2482 Channel Select
    #define DS2482_CMD_SRP      0xE1    //< DS2482 Set Read Pointer
    #define DS2482_CMD_1WRS     0xB4    //< DS2482 1-Wire Reset
    #define DS2482_CMD_1WWB     0xA5    //< DS2482 1-Wire Write Byte
    #define DS2482_CMD_1WRB     0x96    //< DS2482 1-Wire Read Byte
    #define DS2482_CMD_1WSB     0x87    //< DS2482 1-Wire Single Bit
    #define DS2482_CMD_1WT      0x78    //< DS2482 1-Wire Triplet

    // DS2482 status register bit defines
    #define DS2482_STATUS_1WB   0x01    //< DS2482 Status 1-Wire Busy
    #define DS2482_STATUS_PPD   0x02    //< DS2482 Status Presence Pulse Detect
    #define DS2482_STATUS_SD    0x04    //< DS2482 Status Short Detected
    #define DS2482_STATUS_LL    0x08    //< DS2482 Status 1-Wire Logic Level
    #define DS2482_STATUS_RST   0x10    //< DS2482 Status Device Reset
    #define DS2482_STATUS_SBR   0x20    //< DS2482 Status Single Bit Result
    #define DS2482_STATUS_TSB   0x40    //< DS2482 Status Triplet Second Bit
    #define DS2482_STATUS_DIR   0x80    //< DS2482 Status Branch Direction Taken

    // DS2482 configuration register bit defines
    #define DS2482_CFG_APU      0x01    //< DS2482 Config Active Pull-Up
    #define DS2482_CFG_PPM      0x02    //< DS2482 Config Presence Pulse Masking
    #define DS2482_CFG_SPU      0x04    //< DS2482 Config Strong Pull-Up
    #define DS2482_CFG_1WS      0x08    //< DS2482 Config 1-Wire Speed

    // DS2482 channel selection code defines
    #define DS2482_CH_IO0       0xF0    //< DS2482 Select Channel IO0
    #define DS2482_CH_IO1       0xE1    //< DS2482 Select Channel IO1
    #define DS2482_CH_IO2       0xD2    //< DS2482 Select Channel IO2
    #define DS2482_CH_IO3       0xC3    //< DS2482 Select Channel IO3
    #define DS2482_CH_IO4       0xB4    //< DS2482 Select Channel IO4
    #define DS2482_CH_IO5       0xA5    //< DS2482 Select Channel IO5
    #define DS2482_CH_IO6       0x96    //< DS2482 Select Channel IO6
    #define DS2482_CH_IO7       0x87    //< DS2482 Select Channel IO7

    // DS2482 read pointer code defines
    #define DS2482_READPTR_SR   0xF0    //< DS2482 Status Register
    #define DS2482_READPTR_RDR  0xE1    //< DS2482 Read Data Register
    #define DS2482_READPTR_CSR  0xD2    //< DS2482 Channel Selection Register
    #define DS2482_READPTR_CR   0xC3    //< DS2482 Configuration Register


    // API mode bit flags
    #define MODE_STANDARD       0x00
    #define MODE_OVERDRIVE      0x01
    #define MODE_STRONG         0x02

    // One Wire
    #define OW_MATCH_ROM        0x55
    #define OW_SKIP_ROM         0xCC
    #define OW_CONVERT_TEMP     0x44
    #define OW_WR_SCRATCHPAD    0x4E
    #define OW_RD_SCRATCHPAD    0xBE
    #define OW_SEARCH_ROM       0xF0

    #define DS1820_LSB          0
    #define DS1820_MSB          1

    // ============================================================================
    
    class DS2482
    {
      public:
     
        // ROM code structure
        typedef struct sOW_ROM_CODE_ITEM
        {
            uint8_t adr;        // Adresse fuer den Baustein
            uint8_t status;     // Status fuer den Wandler
                                // bit 1 bis 4  0 = inaktiv
                                //              1 = Baustein erkannt
                                //              2 = wird als trigger verwendet
                                //              3 = Messung duchgeführt
                                //              4 = Übertagungsfehler
                                // bit 8 ist fuerBuskennung
            int16_t result;     // Ablage fuer Temperaturwert
            float   value;      // Ablage fuer Stromwert
            int16_t value_2;    // Ablage fuer Spannungwert
            uint8_t rom[8];     // 8 Bytes for ROM code
        } tOW_ROM_CODE_ITEM;
 
        typedef struct sOW
        {
            uint8_t devices;    // Number of devices
            uint8_t device_table_index;
            tOW_ROM_CODE_ITEM device_table[OW_MAX_DEVICES]; // OW-Device data table
        } tOW;

        // global vars
        tOW ow;
         
        /** Create an instance of the PCF8574 connected to specfied I2C pins, with the specified address.
        *
        * @param sda The I2C data pin
        * @param scl The I2C clock pin
        * @param address The I2C address for this DS2482
        */
        
        DS2482(PinName sda, PinName scl, int address);
        
        //-----------------------------------------------------------------------------
        // functions
      
        // crc buffer
        uint8_t crc_calc(uint8_t x);
        uint8_t crc_calc_buffer(uint8_t* pbuffer,uint8_t count);
    
        uint8_t detect(void);
        int reset(void);
        uint8_t write_config(uint8_t config);        
       
        //uint8_t DS2482_channel_select(uint8_t channel);
        uint8_t OWReset(void);
        void OWWriteBit(uint8_t sendbit);
        uint8_t OWReadBit(void);
        uint8_t OWTouchBit(uint8_t sendbit);
        void OWWriteByte(uint8_t sendbyte);
        uint8_t OWReadByte(void);
        void OWBlock(uint8_t *tran_buf, uint8_t tran_len);
        uint8_t OWTouchByte(uint8_t sendbyte);
        uint8_t OWFirst(void);
        uint8_t OWNext(void);
        int     OWVerify(void);
        void    OWTargetSetup(uint8_t family_code);
        void    OWFamilySkipSetup(void);
        uint8_t OWSearch(void);
        uint8_t search_triplet(uint8_t search_direction);
        uint8_t OWSpeed(uint8_t new_speed);
        uint8_t OWLevel(uint8_t new_level);
        uint8_t OWReadBitPower(uint8_t applyPowerResponse);
        uint8_t OWWriteBytePower(uint8_t sendbyte);
       
        void DS18XX_Read_Address(void);
        void start_conversion(void);
        
        bool ds1820_start_conversion(uint8_t command);
        bool ds18B20_read_hrtemp(void);
 
      protected:
 
        bool ow_read_rom(void);
        bool ow_read_scratchpad(void);
        bool ow_read_scratchpad_ds2438(uint8_t page);
        bool ow_write_scratchpad_ds18x20 (uint8_t th, uint8_t tl);
        bool ow_write_scratchpad_ds2438 (uint8_t th, uint8_t tl);
        bool ow_write_eeprom_ds18x20 (void);
        bool ow_write_eeprom_ds2438 (void);
        void ow_read_address (void);
        //-----------------------------------------------------------------------------
        // vars
        
        I2C i2c;
        int addr;
        uint8_t crc_value;
               
        // Search state
                                        // Variablen in einem Block ablegen
        uint8_t ow_scratchpad[OW_SCRATCHPAD_BYTES];     // Scratchpad memory
        uint8_t ow_rom_code[OW_ROM_CODE_BYTES];         // Temporary ROM code
        uint8_t ow_flags;
        uint8_t ROM_NO[8];                              // temporary ROM Code
        uint8_t LastDiscrepancy;
        uint8_t LastFamilyDiscrepancy;
        uint8_t LastDeviceFlag;

        // DS2482 state
        uint8_t c1WS, cSPU, cPPM, cAPU;
        uint8_t short_detected;
        
        bool ds1820_request_pending;
        
}; // end class

#endif
