/*
THIS CODE NEEDS TO BE PUT ONTO THE EMBED BEFORE LAUNCH FOR PROPER TSV DATA

By Cameron Rutherford

last updated: 4/27/2017

Ozone detector code for near space research project

This code collects data from the phototransistor with each LED on seperately,
as well as reading the data when none of the LEDs are on in a TSV text file in the
following format:

Time (s)    Red light   Green light No light
...         ...         ...         ...


Green LEDs connected in parallel to pin 23
Red LEDs connected in parallel to pin 22
Phototransistor connected to pin 20
SD card connected: MOSI - p5
                   MISO - p6
                   SCLK - p7
                   CS - p8
                   
*/

#include "mbed.h"
#include "SDFileSystem.h"
#include "ExtendedTimer.h" //Created by Dr Larkin to extend the timer object

AnalogIn light(p20); //PhotoTransistor
//Serial pc(USBTX,USBRX); this was used for debugging. uncomment this
//                        and other pc.printf lines to debug/print to console
SDFileSystem fs(p5, p6, p7, p8, "fs");
ExtendedTimer t; //Used to keep track of the time
Ticker sampleTime;
DigitalOut doneLED(LED1); //used to indicate when the program has finished
DigitalOut _2(LED2); //used to indicate if the mount was successful/unsuccessful
DigitalOut _4(LED4); //used to indicate if the file open was successful/unsuccessful
bool timeToRead;
DigitalOut green(p23); //used to turn the two green/red LEDs on and off
DigitalOut red(p22);
FILE *data;
void triggerCollection(); //used in the ticker to set timeToRead to true
void SDcardDONE(); //unmount the SD card
void take_reading(); //input data from the photransistor to the file
void LED_OFF(); //used to check the LEDs are off and nothing went wrong - probably useless
double collection_time = 24240; //time interval for collection to take place
float collection_interval = 5 ; //time difference between each data point (red to green to none time)
double num_averaged = 10000; //number of data points collected for each data reading
                             //this aims to eliminate as much noise as possible
float LED_on_time_delay = 1; //delay between each LED turning on and the data recording
                             //used to make sure all remaining light from the LEDs is gone

int main()
{
     // Mount the filesystem
     bool mountFailure = fs.mount();
     
     if (mountFailure != 0) { //if failed to mount
         //pc.printf("Failed to mount the SD card.\r\n");
         _2 = 1;
         wait(5);
         return -1;
     }
    
     data = fopen("/fs/data.txt","a");
     
     if (data == NULL) { //if failed to open the file
         //pc.printf("Failed to open the file.\r\n");
         fs.unmount();
         _4 = 1;
         wait(5);
         return -1;
     }
    
     // Write a header row
     fprintf(data, "Time (s)\tRed light\tGreen light\tNo light\r\n");
     //pc.printf("Time (s)\tRed light\tGreen light\tNo light\r\n");

     // Start the timer and ticker
     t.start();
     sampleTime.attach(&triggerCollection, collection_interval); // write data every collection_interval seconds
     timeToRead = true; // collect data at t = 0
     while (t.read() < collection_time) {
         if (timeToRead) {
             timeToRead = false; // reset data collection flag
             take_reading();
             fclose(data); //make sure the data is written to the SD card to avoid data loss if power is cycled
             data = fopen("/fs/data.txt","a");
         }
     }
    
     SDcardDONE();
     return 0;
}

void take_reading() {
    
    float time = t.read(); //Get the current time
    
    double data_red = 0, data_green = 0, data_none = 0;
    LED_OFF();
    
    red = 1; //Turn the red LEDs on and start recording data
    wait(LED_on_time_delay);
    for (double i = 0; i < num_averaged; i++) //average the data multiple times
        data_red += light;
    data_red = data_red / num_averaged;
    
    red = 0; //Turn the red LEDs off and the green LEDs on and start recording data
    green = 1;
    wait(LED_on_time_delay);
    for (double i = 0; i < num_averaged; i++)
        data_green += light;
    data_green = data_green / num_averaged;
    
    green = 0; //Have all the LEDs off and record data
    wait(LED_on_time_delay);
    for (double i = 0; i < num_averaged; i++)
        data_none += light;
    data_none = data_none / num_averaged;
    
    fprintf(data,"%-.2f\t %-.8f\t %-15.8f\t %-15.8f \r\n", time, data_red, data_green, data_none); //The minus sign makes it left aligned (I think...)
    //pc.printf("%-.2f\t %-.8f\t %-15.8f\t %-15.8f \r\n", time, data_red, data_green, data_none);
}

void triggerCollection() {
    timeToRead = true; // flag to collect data
}

void SDcardDONE() {
    // Close the file and unmount the file system so the SD card is happy
     fclose(data);
     fs.unmount();
     // Turn on LED to let user know it is safe to remove the SD card
     doneLED = 1;
}

void LED_OFF() {
    green = 0;
    red = 0;
    doneLED = 0;
}