/**
* Example app for using the Cayenne MQTT mbed library to send data from a TMP36 sensor. This example uses
* the X-NUCLEO-IDW01M1 WiFi expansion board via the X_NUCLEO_IDW01M1v2 library.
*/

#include "MQTTTimer.h"
#include "CayenneMQTTClient.h"
#include "MQTTNetworkIDW01M1.h"
#include "SpwfInterface.h"
#include "BME280.hpp"
#include "LED.hpp"
#include "Servo.h"
// WiFi network info.
char* ssid = "megu megu fire";
char* wifiPassword = "66666667";

// Cayenne authentication info. This should be obtained from the Cayenne Dashboard.
char* username = "1f764db0-122b-11e7-b235-9359079e690e";
char* password = "0d4e9d94e7897df9af861007853ed3f906b0627d";
char* clientID = "536277d0-8cb9-11e7-9727-55550d1a07e7";

LED led(D7);
Servo servo(D6);
SpwfSAInterface interface(D8, D2); // TX, RX
MQTTNetwork<SpwfSAInterface> network(interface);
CayenneMQTT::MQTTClient<MQTTNetwork<SpwfSAInterface>, MQTTTimer> mqttClient(network, username, password, clientID);

DigitalOut led1(LED1);

/**
* Print the message info.
* @param[in] message The message received from the Cayenne server.
*/
void outputMessage(CayenneMQTT::MessageData& message)
{
    int channel;
    int value ;
    
    switch (message.topic)  {
    case COMMAND_TOPIC:
        printf("topic=Command");
        break;
    case CONFIG_TOPIC:
        printf("topic=Config");
        break;
    default:
        printf("topic=%d", message.topic);
        break;
    }
    printf(" channel=%d", message.channel);
    channel = message.channel;
    if (message.clientID) {
        printf(" clientID=%s", message.clientID);
    }
    if (message.type) {
        printf(" type=%s", message.type);
    }
    printf("message.valueCount = %d\n",message.valueCount);
    for (size_t i = 0; i < message.valueCount; ++i) {
        if (message.getValue(i)) {
            printf(" value=%s", message.getValue(i));
            /*get trigger message*/
            value = (int)*message.getValue(i);
            //strcpy(value,*message.getValue(i));
        }
        if (message.getUnit(i)) {
            printf(" unit=%s", message.getUnit(i));
        }
    }
    if (message.id) {
        printf(" id=%s", message.id);
    }
    printf("\n");
    printf("get channel ==%d\nget value ==%c\n",channel,value);
    if(channel ==1 && value ==49){
        printf("LED has been Triger\n");
        led.On();
    }else if(channel ==2 && (value>=48 && value <=58)){
        servo = ((float)(value-48)/10);
    }else{
        led.Off();
    }
}

/**
* Handle messages received from the Cayenne server.
* @param[in] message The message received from the Cayenne server.
*/
void messageArrived(CayenneMQTT::MessageData& message)
{
    int error = 0;
    // Add code to process the message. Here we just ouput the message data.
    outputMessage(message);

    if (message.topic == COMMAND_TOPIC) {
        switch(message.channel) {
        case 0:
            // Set the onboard LED state
            led1 = atoi(message.getValue());
            // Publish the updated LED state
            if ((error = mqttClient.publishData(DATA_TOPIC, message.channel, NULL, NULL, message.getValue())) != CAYENNE_SUCCESS) {
                printf("Publish LED state failure, error: %d\n", error);
            }
            break;
        }
        
        // If this is a command message we publish a response. Here we are just sending a default 'OK' response.
        // An error response should be sent if there are issues processing the message.
        if ((error = mqttClient.publishResponse(message.id, NULL, message.clientID)) != CAYENNE_SUCCESS) {
            printf("Response failure, error: %d\n", error);
        }
    }
}

/**
* Connect to the Cayenne server.
* @return Returns CAYENNE_SUCCESS if the connection succeeds, or an error code otherwise.
*/
int connectClient(void)
{
    int error = 0;
    // Connect to the server.
    printf("Connecting to %s:%d\n", CAYENNE_DOMAIN, CAYENNE_PORT);
    while ((error = network.connect(CAYENNE_DOMAIN, CAYENNE_PORT)) != 0) {
        printf("TCP connect failed, error: %d\n", error);
        wait(2);
    }

    if ((error = mqttClient.connect()) != MQTT::SUCCESS) {
        printf("MQTT connect failed, error: %d\n", error);
        return error;
    }
    printf("Connected\n");

    // Subscribe to required topics.
    if ((error = mqttClient.subscribe(COMMAND_TOPIC, CAYENNE_ALL_CHANNELS)) != CAYENNE_SUCCESS) {
        printf("Subscription to Command topic failed, error: %d\n", error);
    }
    if ((error = mqttClient.subscribe(CONFIG_TOPIC, CAYENNE_ALL_CHANNELS)) != CAYENNE_SUCCESS) {
        printf("Subscription to Config topic failed, error:%d\n", error);
    }

    // Send device info. Here we just send some example values for the system info. These should be changed to use actual system data, or removed if not needed.
    mqttClient.publishData(SYS_VERSION_TOPIC, CAYENNE_NO_CHANNEL, NULL, NULL, CAYENNE_VERSION);
    mqttClient.publishData(SYS_MODEL_TOPIC, CAYENNE_NO_CHANNEL, NULL, NULL, "mbedDevice");
    //mqttClient.publishData(SYS_CPU_MODEL_TOPIC, CAYENNE_NO_CHANNEL, NULL, NULL, "CPU Model");
    //mqttClient.publishData(SYS_CPU_SPEED_TOPIC, CAYENNE_NO_CHANNEL, NULL, NULL, "1000000000");

    return CAYENNE_SUCCESS;
}

/**
* Main loop where MQTT code is run.
*/
void loop(void)
{
    // Start the countdown timer for publishing data every 5 seconds. Change the timeout parameter to publish at a different interval.
    MQTTTimer timer(5000);
    //TMP36 tmpSensor(A5);
    BME280 bmpSensor;
    
    
    
    while (true) {
        // Yield to allow MQTT message processing.
        mqttClient.yield(1000);
        //printf("%d\n",bmpSensor->Temp_read());
        // Check that we are still connected, if not, reconnect.
        if (!network.connected() || !mqttClient.connected()) {
            network.disconnect();
            mqttClient.disconnect();
            printf("Reconnecting\n");
            while (connectClient() != CAYENNE_SUCCESS) {
                wait(2);
                printf("Reconnect failed, retrying\n");
            }
        }
        printf("before you post data,your temp is %d\n",bmpSensor.Temp_read());
        // Publish some example data every few seconds. This should be changed to send your actual data to Cayenne.
        if (timer.expired()) {
            int error = 0;
            if ((error = mqttClient.publishData(DATA_TOPIC, 5, TYPE_TEMPERATURE, UNIT_CELSIUS,(float)bmpSensor.Temp_read())) != CAYENNE_SUCCESS) {
                printf("Publish temperature failed, error: %d\n", error);
            }
            // Restart the countdown timer for publishing data every 5 seconds. Change the timeout parameter to publish at a different interval.
            timer.countdown_ms(5000);
        }
    }
}

/**
* Main function.
*/
int main()
{   
    // Initialize the network interface.
    printf("Initializing interface\n");
    interface.connect(ssid, wifiPassword, NSAPI_SECURITY_WPA2);
    printf("connect AP successful\n");
    // Set the default function that receives Cayenne messages.
    mqttClient.setDefaultMessageHandler(messageArrived);

    // Connect to Cayenne.
    if (connectClient() == CAYENNE_SUCCESS) {
        // Run main loop.
        loop();
    }
    else {
        printf("Connection failed, exiting\n");
    }

    if (mqttClient.connected())
        printf("client disconnect\n");
        mqttClient.disconnect();
    if (network.connected())
        printf("network disconnect\n");
        network.disconnect();
        
        
    
    

}

