#ifndef TLC5955_H
#define TLC5955_H

//#include "FastPWM.h"
#include "mbed.h"

#define SHORTS_PER_CHANNEL 3
#define CHANNELS_PER_IC  16
#define NUMBER_OF_ICS 3
 
#define TRUE 1
#define FALSE 0

// slow this down 10x for testing
//#define SEQUENCER_RATE 0.5f
//#define SEQUENCER_RATE 0.025f
#define SEQUENCER_RATE 0.030f

/**
  * SPI speed used by the mbed to communicate with the TLC5955
  * The TLC5955 supports up to 30Mhz. This should be kept as high
  * as possible to ensure that data has time to be sent each reset cycle.
  */
  // 4Mhz is fastest supported rate on nRF51
#define SPI_SPEED 4000000

void rebuildGammaTables(uint8_t amplitude);

/**
  *  This class controls a TLC5955 PWM driver IC.
  *  It supports sending dot correction and grayscale data. However, it does not support error checking or writing the EEPROM.
  *  After
  *  4096 pulses, the private member funciton reset is called by the ticker. It resets the display by pulsing the BLANK pin. If new
  *  data has been set to be sent by the functions setNewGSData or setNewDCData, it is sent here. The definition GSCLK_SPEED in TLC5955.h
  *  controls how often this function is called. A higher GSCLK_SPEED will increase the rate at which the screen is updated but also increase
  *  CPU time spent in that function. The default value is 1Mhz. The rate at which the reset function is called can be calculated by: 
  *  (1/GSCLK_SPEED) * 4096.
  */
class TLC5955
{
public:

    typedef enum
    {
        I_3_2_MA = 0,
        I_8_0_MA = 1,
        I_11_2_MA = 2,
        I_15_9_MA = 3,
        I_19_1_MA = 4,
        I_23_9_MA = 5,
        I_27_1_MA = 6,
        I_31_9_MA = 7
    } led_power_t;

    /**
      *  Set up the TLC5955
      *  @param SCLK - The SCK pin of the SPI bus
      *  @param MOSI - The MOSI pin of the SPI bus
      *  @param GSCLK - The GSCLK pin of the TLC5955(s)
      *  @param BLANK - The BLANK pin of the TLC5955(s)  <- REMOVE
      *  @param XLAT - The XLAT pin of the TLC5955(s)
      *  @param DCPRG - The DCPRG pin of the TLC5955(s)  <- REMOVE
      *  @param VPRG - The VPRG pin of the TLC5955(s) <- REMOVE
      *  @param number - The number of TLC5955s (if you are daisy chaining)
      */
    TLC5955(PinName SCLK, PinName MOSI, PinName GSCLK, 
            PinName XLAT, const int number = 1);

    void setChannel(int channelNum, unsigned short red, unsigned short green, unsigned short blue);
    void latchData();
  
    
    /**
      *  Set the next chunk of grayscale data to be sent
      *  @param data - Array of 16 bit shorts containing 16 12 bit grayscale data chunks per TLC5955
      *  @note These must be in intervals of at least (1/GSCLK_SPEED) * 4096 to be sent
      */
    void setNewGSData(unsigned short* data);

    /**
      *  Set the next chunk of dot correction data to be sent
      *  @param data - Array of 8 bit chars containing 16 6 bit dot correction data chunks per TLC5955
      *  @note These must be in intervals of at least (1/GSCLK_SPEED) * 4096 to be sent. Also, this function is optional. If you do not
      *  use it, then the TLC5955 will use the EEPROM, which (by default) conatins the data 0x3F.
      */
void setNewControlData(unsigned short _globalBrightnessRed, unsigned short _globalBrightnessGreen, unsigned short _globalBrightnessBlue,
                                led_power_t _maximumCurrentRed, led_power_t _maximumCurrentGreen, led_power_t _maximumCurrentBlue, 
                                unsigned short* _dotCorrect);
    void clockOutData();
    void clearBit (unsigned short* value, int bitOffset); 
    void setBit (unsigned short* value, int bitOffset); 
    void packBit(unsigned int aBit);
    void packByte (unsigned int aByte);
    void packShort (unsigned int aShort);
protected:
    /**
      *  Set the next chunk of grayscale data to be sent while in the current reset cycle
      *  @note This is useful to send the next set of data right after the first is finished being displayed.
      *  The primary purpose for this is multiplexing, although it could be used for anything else.
      */
    virtual void setNextData() {}


    // Number of TLC5955s in series
    const int number;

private:

    // SPI port - only MOSI and SCK are used
    SPI spi;

    // PWM output
    DigitalOut gsclk;

    // Digital out pins used for the TLC5955
    DigitalOut xlat;

    // Call a reset function to manage sending data and GSCLK updating
    Ticker reset_ticker;

    int currentBitLocation;
    
    // Has new GS/Control data been loaded?
    volatile bool newGSData;
    volatile bool newControlData;

    // Do we need to send an XLAT pulse? (Was GS data clocked in last reset?)
    volatile bool need_xlat;

    // Buffer to store data until it is sent
    unsigned short gsBuffer[(SHORTS_PER_CHANNEL * CHANNELS_PER_IC * NUMBER_OF_ICS) + 1];  // one extra word to fit the MSB control bit. Extra bits will be clocked out.

    unsigned short internalData[SHORTS_PER_CHANNEL * CHANNELS_PER_IC * NUMBER_OF_ICS]; // internal data storage for use by per-channel setting function
    unsigned short globalBrightnessRed;
    unsigned short globalBrightnessGreen;
    unsigned short globalBrightnessBlue;
    unsigned short maximumCurrentRed;
    unsigned short maximumCurrentGreen; 
    unsigned short maximumCurrentBlue;
    unsigned short* dotCorrect;
    
    // Function to reset the display and send the next chunks of data
    void reset();
};


/**
  *  This class allows a TLC5955 to be multiplexed.
  *  It inherits the TLC5955 class and uses it to control the TLC5955 driver(s). It does not support sending dot corection data.
  *  This class sets the new grayscale data every iteration of the GSCLK reset loop. It then updates the current row using the
  *  user defined function SetRows. The framerate you will recieve using this function can be calculate by: 1 / (((1/GSCLK_SPEED) * 4096) * rows).
  *  I reccomend maintaining a framerate above 30fps. However, keep in mind that as your framerate increases, so does your CPU usage.
  *
  *  Using the TLC5955Mux class to control an 8x8 LED matrix:
  *  @code
  *  #include "mbed.h"
  *  #include "TLC5955.h"
  *  
  *  // Bus connecting to the rows of the LED matrix through PNP transistors
  *  BusOut rows(p22, p23, p24, p25, p26, p27, p28, p29);
  *  
  *  // Function to update the rows using the BusOut class
  *  void SetRows(int nextRow)
  *  {
  *      // I am using PNP transistors, so inversion is necessary
  *      rows = ~(1 << nextRow);
  *  }
  *  
  *  // Create the TLC5955Mux instance
  *  TLC5955Mux tlc(p7, p5, p21, p9, p10, p11, p12, 1, 8, &SetRows);
  *  
  *  int main()
  *  {   
  *      tlc[0][0] = 0xFFF; // Turn on the top left LED
  *      while(1)
  *      {
  *  
  *      }
  *  }
  *  @endcode
  */
class TLC5955Mux : private TLC5955
{
public:
    /**
      *  Set up the TLC5955
      *  @param SCLK - The SCK pin of the SPI bus
      *  @param MOSI - The MOSI pin of the SPI bus
      *  @param GSCLK - The GSCLK pin of the TLC5955(s)
      *  @param BLANK - The BLANK pin of the TLC5955(s)
      *  @param XLAT - The XLAT pin of the TLC5955(s)
      *  @param DCPRG - The DCPRG pin of the TLC5955(s)
      *  @param VPRG - The VPRG pin of the TLC5955(s)
      *  @param number - The number of TLC5955s (if you are daisy chaining)
      *  @param rows - The number of rows you are multiplexing
      *  @param SetRows - The function pointer to your function that sets the current row. 
      *  @note The SetRows function allows you to set exactly how you want your rows 
      *  to be updated. The TLC5955Mux class calls this function with an argument of int that contains the number of the row to 
      *  be turned on. If the TLC5955Mux class needs the first row to be turned on, the int will be 0.
      */
    TLC5955Mux(PinName SCLK, PinName MOSI, PinName GSCLK,
               PinName XLAT, const int number,
               const int rows, void (*SetRows)(int));
    
    // Destructor used to delete memory        
    ~TLC5955Mux();

    /**
      *  Set the contents of the buffer that contains the multiplexed data
      *  @param data - The data to set to the buffer containing 16 12 bit grayscale data chunks per TLC5955
      *  @returns The data provided
      */
    unsigned short* operator=(unsigned short* data);
    
    /**
      *  Get a pointer to one of the rows of the multiplexed data
      *  @param index - The row that you would like the contents of
      *  @returns A pointer to the data containing the requested row containing 16 12 bit grayscale data chunks per TLC5955
      *  @note This operator can also be used to change or get the value of an individual LED.
      *  For example:
      *  @code
      *  TLC5955Mux[0][0] = 0xFFF;
      *  @endcode
      */
    unsigned short* operator[](int index);
    
private:
    // Virtual function overriden from TLC5955 class
    virtual void setNextData();
    
    // Number of rows
    const int rows;

    // Function to set the current row
    void (*SetRows)(int);

    // The current row
    volatile int currentIndex;
};

#endif