#include "mbed.h"
#include "rtos.h"
#include "MDM.h"
#include "GPS.h"
#include "DeviceInfo.h"
#include "DeviceMemory.h"
#include "MbedAgent.h"
#include "GPSTracker.h"
#include "DeviceConfiguration.h"

#include <stdio.h>

// ----------------------------------------------------------------
// Cellular modem / SIM parameters
// ----------------------------------------------------------------
#include "MDM.h"
#define SIMPIN      NULL                    //!< Set your secret SIM pin here (e.g. "1234"). Check your SIM manual.
#define APN         "internet"              //!< The APN of your network operator SIM, sometimes it is "internet"
#define USERNAME    NULL                    //!< Set the user name for your APN, or NULL if not needed
#define PASSWORD    NULL                    //!< Set the password for your APN, or NULL if not needed

// ----------------------------------------------------------------
// PubNub Config
// ----------------------------------------------------------------
//#include "PubNub.h"
//const char pubkey[] = "pub-c-e3a95948-182a-46fd-b5f3-52c184eb3c12";
//const char subkey[] = "sub-c-0313f6b2-b2c0-11e4-ab0e-02ee2ddab7fe";
//const char channel[] = "mbed";
//const char channel_pub[] = "mbed_data";
//const char channel_sub[] = "mbed_cmd";

int main()
{
    MDMParser::DevStatus devStatus;
    uint8_t status = 0;

    MDMRtos<MDMSerial> mdm;
    GPSI2C gps;
    
    //mdm.setDebug(4);

    if (!mdm.init(SIMPIN, &devStatus))
        status = 1;
    else if (!gps.init())
        status = 2;
    
    DeviceIO io(gps);

    /** For debugging, you may find it useful to print memory usage
     * stats. AvailableMemory may be flaky, but the following is nice.
     * It will get printed to the USB serial port interface. 
     */
    //io.debugPrint("%d: ", __LINE__); __heapstats((__heapprt)fprintf, stdout);
    
    io.debugPrint("\r\nWiFi Scanner build " __DATE__ " " __TIME__ "\r\n");
    //io.debugPrint("CPU SystemCoreClock is %d Hz\r\n", SystemCoreClock);  
    
    switch (status) {
    case 1:
        io.debugPrint("MODEM INIT FAILURE. CHECK SIM");
        break;
    case 2:
        io.debugPrint("GPS INIT FAILURE");
        break;
    }
    
    if (status != 0)
        goto error;
    
    io.debugPrint("DEVICE INIT");
    
    //io.debugPrint("REGISTER NETWORK. IMEI: %s", devStatus.imei);

    if (!mdm.registerNet()) {
        io.debugPrint("NETWORK REG ERROR");
        goto error;
    }

    io.debugPrint("JOIN NETWORK");
#ifdef SIM_APN
    if (mdm.join(SIM_APN, SIM_USER, SIM_PASS) == NOIP) {
#else
    if (mdm.join() == NOIP) {
#endif
        io.debugPrint("NETWORK JOIN FAILURE");
        goto error;
    }
    
    {
        uint8_t tries;
        DeviceInfo deviceInfo(mdm, devStatus);
        DeviceMemory deviceMemory(mdm);
        
        if (io.userButtonPressed()) {
            if (deviceMemory.resetConfiguration())
                io.deviceFeedback().showSuccess();
            else
                io.deviceFeedback().showFailure();
            Thread::wait(1000);
            return 0;
        }

        MbedAgent agent(io, mdm, deviceInfo, deviceMemory);
    
        io.debugPrint("AGENT INIT");
        if (!agent.init()) {
            io.debugPrint("AGENT INIT FAILURE");
            goto error;
        }
        
        tries = 3;
        do {
            io.debugPrint("AGENT RUN");
            if (agent.run())
                break;
        } while (--tries > 0);

        if (tries == 0) {
            io.debugPrint("AGENT RUN FAILURE");
            goto error;
        }
    }
    
    mdm.disconnect();
    return 0;

error:
    mdm.disconnect();
    return 1;
}
