#include "mbed.h"
#include "BuddiService.h"
#include "Logger.h"
#include "Configuration.h"

namespace Nudge {
// Buddi base UUID: 92D7DC20-xxxx-4DF8-84B1-AD8AF6E1EA4A
#define buddi_UUID(x)   {0x92, 0xD7, 0xDC, 0x20, (((x) & 0xFF00) >> 8), ((x) & 0xFF), 0x4D, 0xF8, 0x82, 0xB1, 0xAD, 0x8A, 0xF6, 0xE1, 0xEA, 0x4A}
// UUID byte arrays
static const uint8_t buddiServiceUUID[]     = buddi_UUID(0xBA55);
static const uint8_t commandUUID[16]        = buddi_UUID(0xBB01);
static const uint8_t notificationUUID[]     = buddi_UUID(0xBB02);

// Storage for the value of the characteristics
typedef struct {
    uint8_t module;
    uint8_t opcode;
    uint8_t data[15];
} Command_t;
static Command_t commandPayload;
//uint8_t commandPayload[17] = {0,};

// Other things needed for operation
static BLEDevice*                   ble;

static GattCharacteristic commandCharacteristic(commandUUID, 
                                                (uint8_t *)&commandPayload, 
                                                sizeof(Command_t), //1, 
                                                sizeof(Command_t), //8, 
                                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | 
                                                GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE_WITHOUT_RESPONSE);

uint8_t notificationPayload[8] = {0,};

static GattCharacteristic notificationCharacteristic(notificationUUID, 
                                                    notificationPayload, 
                                                    1, 
                                                    8, 
                                                    GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY);

static GattCharacteristic *allChars[] = {&commandCharacteristic, &notificationCharacteristic};
static GattService buddiService(buddiServiceUUID, allChars, sizeof(allChars) / sizeof(GattCharacteristic *));

PwmOut      ledr(p22);                       /* Red LED on PCA10000 */
PwmOut      ledg(p23);                       /* Green LED on PCA10000 */ 
PwmOut      ledb(p21);                       /* Blue LED on PCA10000 */

void init(BLEDevice &bleDevice)
{
    ledr               = 1.0f;
    ledg               = 1.0f;
    ledb               = 1.0f;
    ble                = &bleDevice;
    ble->addService(buddiService);
    DEBUG("Added buddi service\r\n");
}

const uint8_t *getServiceUUIDp()
{
    return buddiServiceUUID;
}

void setLEDValue(uint8_t led, uint8_t value)
{
    float val = ((float)value)/255;
    DEBUG("Setting LED %u to %1.3f\r\n", led, val);
    switch(led)
    {
        case 0:
            ledg = val;
            break;
        case 1:
            ledr = val;
            break;
        case 2:
            ledb = val;
            break;
        default:
            DEBUG("Unknown LED %u selected\r\n", led);
            break;
    }
}

void handleDataWritten(uint16_t handle, const GattCharacteristicWriteCBParams *params)
{
    if (!ble) {
        return;
    }

    if (handle == commandCharacteristic.getValueAttribute().getHandle()) {
        //uint16_t len = sizeof(commandPayload);
        //ble->readCharacteristicValue(handle, (uint8_t *) &commandPayload, &len);
        if (params->len != sizeof(Command_t)) {
            DEBUG("invalid write into fileInfo characteristic\r\n");
            return;
        }
        memcpy(&commandPayload, params->data, params->len);
        
        DEBUG("RX: MOD: 0x%X OP: 0x%X D0: 0x%X D1: 0x%X", commandPayload.module, commandPayload.opcode, commandPayload.data[0], commandPayload.data[1]);
        //ble->updateCharacteristicValue(notificationCharacteristic.getValueAttribute().getHandle(), commandPayload, bytesRead);
        switch(commandPayload.module) {
            case MODULE_NEO_PIXEL:
                setLEDValue(0, commandPayload.data[2]);
                setLEDValue(1, commandPayload.data[3]);
                setLEDValue(2, commandPayload.data[4]);
                break;
            default:
                DEBUG("Module %u not implemented yet", commandPayload.module);
                break;
        }
    } else {
        DEBUG("Got data on unexpected characteristic handle %u!\r\n", handle);
    }
}

} // namespace nudge
