#include "mbed.h"
#include "mDot.h"
#include "MTSLog.h"
#include <string>
#include <vector>
#include <algorithm>
// these options must match the settings on your Conduit
// TTN Keys
static const uint8_t netowork_session_key_array[] = {0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C};
static const uint8_t data_session_key_array[] = {0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6, 0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C};
// uncomment the following lines and edit their values to match your configuration
//static const uint8_t network_address_array[] = {0x02, 0x01, 0xBA, 0x01}; // use yours based on http://thethingsnetwork.org/wiki/AddressSpace
static std::vector<uint8_t> netowork_session_key (netowork_session_key_array, netowork_session_key_array + sizeof(netowork_session_key_array) / sizeof(uint8_t));
static std::vector<uint8_t> data_session_key (data_session_key_array, data_session_key_array + sizeof(data_session_key_array) / sizeof(uint8_t));
static std::vector<uint8_t> network_address (network_address_array, network_address_array + sizeof(network_address_array) / sizeof(uint8_t));
static uint8_t config_frequency_sub_band = 4;

int main() {
    int32_t ret;
    mDot* dot;
    std::vector<uint8_t> data;
    std::string data_str = "hello ropu!";
    
    // get a mDot handle
    dot = mDot::getInstance();

    dot->resetConfig();

    dot->setLogLevel(mts::MTSLog::INFO_LEVEL);

    // too lazzy to check all errors
    dot->setJoinMode(mDot::MANUAL);
    dot->setPublicNetwork(true);
    dot->setFrequencySubBand(config_frequency_sub_band);
    dot->setNetworkSessionKey(netowork_session_key);
    dot->setDataSessionKey(data_session_key);
    dot->setNetworkAddress(network_address);
    
    
    // a higher spreading factor allows for longer range but lower throughput
    // in the 915 (US) frequency band, spreading factors 7 - 10 are available
    // in the 868 (EU) frequency band, spreading factors 7 - 12 are available
    logInfo("setting TX spreading factor");
    if ((ret = dot->setTxDataRate(mDot::SF_10)) != mDot::MDOT_OK) {
        logError("failed to set TX datarate %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    
    
    // request receive confirmation of packets from the gateway
    logInfo("enabling ACKs");
    if ((ret = dot->setAck(0)) != mDot::MDOT_OK) {
        logError("failed to enable ACKs %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // save this configuration to the mDot's NVM
    logInfo("saving config");
    if (! dot->saveConfig()) {
        logError("failed to save configuration");
    }
    //*******************************************
    // end of configuration
    //*******************************************

    // attempt to join the network
    logInfo("joining network");
    while ((ret = dot->joinNetwork()) != mDot::MDOT_OK) {
        logError("failed to join network %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        // in the 868 (EU) frequency band, we need to wait until another channel is available before transmitting again
        osDelay(std::max((uint32_t)1000, (uint32_t)dot->getNextTxMs()));
    }

    // format data for sending to the gateway
    for (std::string::iterator it = data_str.begin(); it != data_str.end(); it++)
        data.push_back((uint8_t) *it);

    while (true) {
        // send the data to the gateway
        if ((ret = dot->send(data)) != mDot::MDOT_OK) {
            logError("failed to send", ret, mDot::getReturnCodeString(ret).c_str());
        } else {
            logInfo("successfully sent data to gateway");
        }

        // in the 868 (EU) frequency band, we need to wait until another channel is available before transmitting again
        osDelay(std::max((uint32_t)5000, (uint32_t)dot->getNextTxMs()));
    }

    return 0;
}
