/* 
    Copyright (c) 2014 Romain Berrada
    
    Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
    and associated documentation files (the "Software"), to deal in the Software without restriction, 
    including without limitation the rights to use, copy, modify, merge, publish, distribute, 
    sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in all copies or 
    substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
    BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
    DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#include "Morse.h"
#include "mbed.h"

struct _morse_tree {
    char value;
    const struct _morse_tree* dot;
    const struct _morse_tree* dash;
};

// Depth 5 of the tree
static const struct _morse_tree _morse_tree_1 = {'1', 0, 0};
static const struct _morse_tree _morse_tree_2 = {'2', 0, 0};
static const struct _morse_tree _morse_tree_3 = {'3', 0, 0};
static const struct _morse_tree _morse_tree_4 = {'4', 0, 0};
static const struct _morse_tree _morse_tree_5 = {'5', 0, 0};
static const struct _morse_tree _morse_tree_6 = {'6', 0, 0};
static const struct _morse_tree _morse_tree_7 = {'7', 0, 0};
static const struct _morse_tree _morse_tree_8 = {'8', 0, 0};
static const struct _morse_tree _morse_tree_9 = {'9', 0, 0};
static const struct _morse_tree _morse_tree_0 = {'0', 0, 0};
static const struct _morse_tree _morse_tree_AR= {'@', 0, 0};                           // @ added as DOT,DOT,DOT,DASH,DOT
static const struct _morse_tree _morse_tree_IN= {'?', 0, 0};                           // ? added as DOT,DOT,DASH,DOT,DOT

// Depth 4 of the tree
static const struct _morse_tree _morse_tree_H = {'H', &_morse_tree_5, &_morse_tree_4};
static const struct _morse_tree _morse_tree_V = {'V',&_morse_tree_AR ,&_morse_tree_3};
static const struct _morse_tree _morse_tree_F = {'F',&_morse_tree_IN, 0};
static const struct _morse_tree _morse_tree_MN= {'-', 0,              &_morse_tree_2}; // - added as DOT,DOT,DASH,DASH
static const struct _morse_tree _morse_tree_L = {'L', 0,              0};
static const struct _morse_tree _morse_tree_SL= {'/', 0,              0};              // / added as DOT,DASH,DOT,DASH
static const struct _morse_tree _morse_tree_P = {'P', 0,              0};
static const struct _morse_tree _morse_tree_J = {'J', 0,              &_morse_tree_1};
static const struct _morse_tree _morse_tree_B = {'B', &_morse_tree_6, 0};
static const struct _morse_tree _morse_tree_X = {'X', 0, 0};
static const struct _morse_tree _morse_tree_C = {'C', 0, 0};
static const struct _morse_tree _morse_tree_Y = {'Y', 0, 0};
static const struct _morse_tree _morse_tree_Z = {'Z', &_morse_tree_7, 0};
static const struct _morse_tree _morse_tree_Q = {'Q', 0, 0};
static const struct _morse_tree _morse_tree_UN= {'_', &_morse_tree_8, 0};              // _ added as DASH,DASH,DASH,DOT
static const struct _morse_tree _morse_tree_DO= {'.', &_morse_tree_9, &_morse_tree_0}; // . added as DASH,DASH,DASH,DASH

// Depth 3 of the tree
static const struct _morse_tree _morse_tree_S = {'S', &_morse_tree_H, &_morse_tree_V};
static const struct _morse_tree _morse_tree_U = {'U', &_morse_tree_F, &_morse_tree_MN};
static const struct _morse_tree _morse_tree_R = {'R', &_morse_tree_L, &_morse_tree_SL};
static const struct _morse_tree _morse_tree_W = {'W', &_morse_tree_P, &_morse_tree_J};
static const struct _morse_tree _morse_tree_D = {'D', &_morse_tree_B, &_morse_tree_X};
static const struct _morse_tree _morse_tree_K = {'K', &_morse_tree_C, &_morse_tree_Y};
static const struct _morse_tree _morse_tree_G = {'G', &_morse_tree_Z, &_morse_tree_Q};
static const struct _morse_tree _morse_tree_O = {'O', &_morse_tree_UN,&_morse_tree_DO};

// Depth 2 of the tree
static const struct _morse_tree _morse_tree_I = {'I', &_morse_tree_S, &_morse_tree_U};
static const struct _morse_tree _morse_tree_A = {'A', &_morse_tree_R, &_morse_tree_W};
static const struct _morse_tree _morse_tree_N = {'N', &_morse_tree_D, &_morse_tree_K};
static const struct _morse_tree _morse_tree_M = {'M', &_morse_tree_G, &_morse_tree_O};

// Depth 1 of the tree
static const struct _morse_tree _morse_tree_E = {'E', &_morse_tree_I, &_morse_tree_A};
static const struct _morse_tree _morse_tree_T = {'T', &_morse_tree_N, &_morse_tree_M};

static const struct _morse_tree _morse_tree_root = {'#', &_morse_tree_E, &_morse_tree_T};

unsigned int morse_getWordSize(Morse_data* data) {
    unsigned int i;
    unsigned int blank_counter=0;
    unsigned int size=0;
    
    for (i=0; i<data->length; i++) {
        if (data->data[i]==false) blank_counter++;
        else { // else true
            // checks all the blanks so far, and then reset the counter
            if (blank_counter==3) size++; // if blanks to separate two chars
            else if (blank_counter==7) size+=2; // if blanks for a space : acknowledge the space and the char before the space
            blank_counter=0;
        }
    }
    size++; // acknowledge the last char
    if (blank_counter>=7) size++;
    return size;
}

void morse_decode(Morse_data* data, char* word) {
    unsigned int i;
    unsigned int char_i=0;
    unsigned int false_counter=0;
    unsigned int true_counter=0;
    const struct _morse_tree* tree = &_morse_tree_root; // CHANGE TO THE BINRARY TREE IMPLEMENTATION AND PUT THE ROOT INSTEAD

    for (i=0; i<data->length; i++) {
        if (data->data[i]==false) {
            // if false, increase the false_counter, manage the number of true and false before, and reset the true_counter
            false_counter++;
            
            if (true_counter==1) tree = tree->dot;
            else if (true_counter==3) tree = tree->dash;
            
            if (false_counter==3) {
                word[char_i++] = tree->value;
                tree = &_morse_tree_root;
            }
            else if (false_counter==7) word[char_i++] = ' ';

            true_counter=0;
        }
        else { // else true
            true_counter++;
            false_counter=0;
        }   
    }
    
    if (true_counter==1) word[char_i++] = tree->dot->value;
    else if (true_counter==3) word[char_i++] = tree->dash->value;

    word[char_i] = '\0'; // finally, end the char array
}
