/* 
    Copyright (c) 2014 Romain Berrada
    
    Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
    and associated documentation files (the "Software"), to deal in the Software without restriction, 
    including without limitation the rights to use, copy, modify, merge, publish, distribute, 
    sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
    furnished to do so, subject to the following conditions:

    The above copyright notice and this permission notice shall be included in all copies or 
    substantial portions of the Software.

    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
    BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
    DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef INCLUDE_MORSE_H
#define INCLUDE_MORSE_H

/** A  Morse Encoding/Decoding Library \n
    Transforms char arrays into bool arrays and vice-versa
    
    Morse code taken from http://en.wikipedia.org/wiki/Morse_code \n
    Added some more characters : \n
        minus : DOT DOT DASH DASH \n
        _ : DASH DASH DASH DOT \n
        . : DASH DASH DASH DASH \n
        / : DOT DASH DOT DASH \n
        @ : DOT DOT DOT DASH DOT \n
        ? : DOT DOT DASH DOT DOT \n
    
    Here is an quick hello-world that show how to use this library \n
    @code
    #include "mbed.h
    #include "Morse.h"
    
    Serial pc(USBTX, USBRX);
    
    int main() {
        int i;
        Morse_data* data;
        char message[] = "Hello World";
        
        data = morse_create(morse_getBoolSize(message));
        morse_encode(message, data);
        for (i=0; i<data->length; i++) pc.printf("%d", data->data[i]);
        
        morse_decode(data, message);
        pc.printf("\nMessage was : %s\n", message);
        
        while(1);
    }
    @endcode
*/

/** Morse_data Structure
*/
typedef struct {
    int length; /**< Length of the data field */
    bool* data; /**< data field containing 'length' booleans */
} Morse_data;

/** Give the size of the Morse_data that would be encoded from this word.
    @param word the word.
    @return the size of the encoded version of the word.
*/
unsigned int morse_getBoolSize(char* word);
/** Give the size of the char array that would be decoded from this Morse_data.
    @param data the data.
    @return the size of the decoded version of the data.
*/
unsigned int morse_getWordSize(Morse_data* data);

/** Encode the char array and put the result into the Morse_data structure.
    @param word the word to encode.
    @param data the Morse_data structure to store the result. This structure must have a sufficient size.
*/
void morse_encode(char* word, Morse_data* data);
/** Decode the Morse_data structure and put the result into the char array.
    @param data the data to decode.
    @param word the char array to store the result. The array must have a sufficient size. (Also think of the '\0')
*/
void morse_decode(Morse_data* data, char* word);

/** Allocate a Morse_data structure with the given size.
    @param length the size of the structure to create.
    @return the data pointer.
*/
Morse_data* morse_create(unsigned int length);
/** Free a Morse_data structure.
    @param data the data to free.
*/
void morse_destroy(Morse_data* data);

#endif