/* 
    Copyright (c) 2014 Romain Berrada
    
    Permission is hereby granted, free of charge, to any person obtaining a copy of this software 
    and associated documentation files (the "Software"), to deal in the Software without restriction, 
    including without limitation the rights to use, copy, modify, merge, publish, distribute, 
    sublicense, and/or sell copies of the Software, and to permit persons to whom the Software is 
    furnished to do so, subject to the following conditions:
 
    The above copyright notice and this permission notice shall be included in all copies or 
    substantial portions of the Software.
 
    THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING 
    BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
    NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, 
    DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
    OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
*/

#ifndef INCLUDE_MENU_H
#define INCLUDE_MENU_H

/** A simple Menu Library \n
    A Menu is light and easy to use. It's composed of a : \n
        - an output interface to show which program is selected, \n
        - a way to change the selection, \n
        - a way to validate the selection and launch the linked program. \n
        
    If you want a example on how to use it, look at the MenuExample program (http://mbed.org/users/rominos2/code/MenuExample/)
*/
class Menu {
private:
    /** A Program is reprensented with this nested class.
    */
    class Program {
        private:
        void* _output_interface_argument; /**< the output displayed when this program is seletected. For example, a LED Color or a message. */
        int(*_program)(); /**< the function to call when the program is validated. */
        Program* _next; /**< the next program in the chained list */
        
        Program(void* output_argument, int(*program)()); /**< Constructor of a Program */
        
        friend class Menu;
    };
    
    Program* _first; /**< First program of the chained list. */
    
protected:
    /** Start the Menu.
        Implementation can make LED blinks or print a message on a screen for example
    */
    virtual void startMenu() = 0;
    /** Implementation must links an input to know if the user change the selection.
        For example, a switch can be checked.
        @return 1 if the selection changes, else false.
    */
    virtual bool isSelectionChanging() = 0;
    /** Implementation must links an input to know if the user validate the selection.
        For example, a switch can be checked.
        @return 1 if the user validates, else false.
    */
    virtual bool isValidating() = 0;
    
    /** Implementation must links an output to show the selected program has changed.
        For example, the color of a LED can be changed or a message printed.
        @param output_argument the argument of the selected program. If NULL, the interface must clear.
    */
    virtual void displaySelectedProgram(void* output_argument) = 0;

public:
    /** Construcor of the Menu.
    */
    Menu();
    
    /** Add a program to the Menu.
        @param output_argument the output displayed when this program is seletected. For example, a LED Color or a message.
        @param the function to call when the program is validated.
    */
    void addProgram(void* output_argument, int(*program)());
    /** Free the alocated memory of the Bootstrap.
        All program will be removed from the list.
    */
    void clean();
    /** Start the Menu. \n
        First startMenu() is called. \n
        Then the output interface show the chosen program (with the output argument) and the Bootstrap check isSelectionChanging() and isValidating(). \n
        Once validate, the function of the program is called. \n
        @return the result of the program's function.
    */
    int launch();
};

#endif