#include "mbed.h"

DigitalOut ncs(p14);
SPI spi(p5,p6,p7);

// 23K256 data sheet at http://ww1.microchip.com/downloads/en/DeviceDoc/22100B.pdf

// Page-mode commands have not been implemented; I have found no need for them yet.

// Assumes spi mode is default (8,0).

// The 23K256 supports the mbed's maximum spi frequency of 12MHz.

// mode codes for 23K256
#define BYTE_MODE       0x00
#define SEQUENTIAL_MODE 0x40

// command codes for 23K256
#define READ            0x03
#define WRITE           0x02
#define READ_STATUS     0x05 // called RDSR in datasheet
#define WRITE_STATUS    0x01 // called WRSR in datasheet

inline void on() {
    ncs = 0;
}

inline void off() {
    ncs = 1;
}

void writeStatus(char status) {
    on();
    spi.write(WRITE_STATUS);
    spi.write(status);
    off();
}

int readStatus() {
    on();
    spi.write(READ_STATUS);
    int result = spi.write(0);
    off();
    return result;
}

inline void prepareCommand(char command, int address) {
    on();
    spi.write(command);
    spi.write(address >> 8);
    spi.write(address & 0xFF);
}

// write or read a single byte

void write(int address, char byte) {
    prepareCommand(WRITE, address);
    spi.write(byte);
    off();
}

char read(int address) {
    prepareCommand(READ, address);
    int result = spi.write(0);
    off();
    return (char) result;
}

// buffered write and read

/*
* the single-byte read and write assume the 23K256 is in its default byte-mode
* so sequential-model commands must switch the chip into sequential mode
* at the start and return it to byte mode at the end.
*/

void write(int address, char * buffer, int count) {
    writeStatus(SEQUENTIAL_MODE);
    prepareCommand(WRITE, address);
    for (int i = 0; i < count; i++) {
        spi.write(buffer[i]);
    }
    off();
    writeStatus(BYTE_MODE);
}

void read(int address, char * buffer, int count) {
    writeStatus(SEQUENTIAL_MODE);
    prepareCommand(READ, address);
    for (int i = 0; i < count; i++) {
        buffer[i] = spi.write(0);
    }
    off();
    writeStatus(BYTE_MODE);
}

int main() {
    off();
    printf("Status codes: 0 = byte mode (default), 0x40 = sequential mode. "); 
    printf("Status is currently %i\r\n", readStatus());
    char buff[50];
    write(0, 'h');
    write(1, 'i');
    write(2, '!');
    write(3, '\0');
    for (int address = 0; address < 4; address++) {
        buff[address] = read(address);
    }
    printf("mem = %s\r\n", buff);
    write(0, "Hello world!",12);
    read(0, buff, 12);
    buff[12]='\0';
    printf("now = %s\r\n", buff);
}