/* Test drive the MCP23S17 library
* Copyright (c) 2010 Romilly Cocking
* Released under the MIT License: http://mbed.org/license/mit
*
* See http://mbed.org/users/romilly/notebook/mcp23s17-addressable-16-bit-io-expander-with-spi/
*
* NB this code is intended to test the driver library, not the chip
* which is assumed to work as specified
* MCP23S17 datasheet http://ww1.microchip.com/downloads/en/DeviceDoc/21952b.pdf
* version 0.4
*/

#include "mbed.h"
#include "MCP23S17.h"

SPI spi(p5, p6, p7);

// A0, A1, A2 of the MCP23S17  are tied to ground on the breadboard, so the 8-bit address for writes is 0x40
// This is referred to as the opcode in the device datasheet
char writeOpcode = 0x40;
// mbed pin20 is connected to ~chipSelect of the MCP23S17
MCP23S17 chip = MCP23S17(spi, p20, writeOpcode); // create MCP23S17
DigitalOut    chipNotReset(p14); // connected to ~reset of the MCP23S17
DigitalInOut chipA0(p12);  // connected to Port A bit 0 of the MCP23S17
DigitalInOut chipA1(p30);     // connected to Port A bit 1 via 100k resistor
DigitalInOut  chipB0(p10); // connected to Port B bit 0 of the MCP23S17
DigitalIn   chipIntA(p16); // connected to INTA on the MCP23S17
DigitalIn   chipIntB(p18); // connected to INTB on the MCP23S17

void reset() {
    // I'm not sure we need the delays, but better safe than sorry.
    // Ensure mbed bi-directional pins set to input
    chipA0.input();
    chipA1.input();
    chipB0.input();
    chipNotReset = 0;  // reset chip
    wait_us(10);
    chipNotReset = 1;
    wait_us(10);
}

void checkEqual(int expected, int actual, char * description) {
    if (expected != actual) {
        printf("%s **TEST FAILED** - expected %i but get %i\r\n", description, expected, actual);
        exit(-1);
    }
}

void testInputFromPortA() {
    reset();
    chipA0.output(); // prepare to output from the mbed
    chip.direction(PORT_A,0x01); // chip Port A bit 0 set to input
    chipA0 = 0;
    checkEqual(0, chip.read(PORT_A) && 0x1,"read from A0");
    chipA0 = 1;
    checkEqual(1, chip.read(PORT_A) && 0x1,"read from A0");
}

void testInputFromPortB() {
    reset();
    chipB0.output(); // output from the mbed
    chip.direction(PORT_B,0x01); //  bit 0 set to input
    chipB0 = 0;
    checkEqual(0, chip.read(PORT_B) && 0x1,"read from B0");
    chipB0 = 1;
    checkEqual(1, chip.read(PORT_B) && 0x1,"read from B0");
}

void testOutputToPortA() {
    reset();
    chipA0.input(); // input to the mbed
    chip.direction(PORT_A, 0xFE); //  bit 0 set to output
    chip.write(PORT_A,0x00);
    checkEqual(0, int(chipA0),"write to A0");
    chip.write(PORT_A,0x01);
    checkEqual(1, int(chipA0),"write to A0");
}

void testOutputToPortB() {
    reset();
    chipB0.input(); // input to the mbed
    chip.direction(PORT_B, 0xFE); //  bit 0 set to output
    chip.write(PORT_B,0x00);
    checkEqual(0, int(chipB0),"write to B0");
    chip.write(PORT_B,0x01);
    checkEqual(1, int(chipB0),"write to B0");
}

void testInterruptEnableOnPortA() {
// NB by default both int pins are Active-LOW
    reset();
    chipA0.output(); // output from the mbed
    chip.direction(PORT_A, 0x01); //  bit 0 set to input
    chipA0 = 1;
    checkEqual(1, int(chipIntA),"interrupts not yet enabled");
    chipA0 = 0;
    chip.interruptEnable(PORT_A, 0x01); // interupt enabled on pin A0
    checkEqual(1, int(chipIntA), "value has not changed");
    chipA0 = 1;
    wait_us(1); // test fails without this - mbed is too darned fast!
    checkEqual(0, int(chipIntA), "value has changed");
}

void testInterruptEnableOnPortB() {
// NB by default both int pins are Active-LOW
    reset();
    chipB0.output(); // output from the mbed
    chip.direction(PORT_B, 0x01); //  bit 0 set to input
    chipB0 = 1;
    chip.interruptEnable(PORT_B, 0x01); // interupt enabled on pin 0
    wait_us(1);
    checkEqual(0, int(chipIntB), "interruptB");
}

void testMirrorInterrupts() {
    reset();
    chipB0.output(); // output from the mbed
    chip.direction(PORT_B, 0x01); //  bit 0 set to input
    chipB0 = 1;
    chip.interruptEnable(PORT_B, 0x01); // interupt enabled on pin 0
    wait_us(1);
    checkEqual(0, int(chipIntB), "interruptB");
    checkEqual(1, int(chipIntA), "before mirroring"); // no interrupt A yet
    chip.mirrorInterrupts(true);
    wait_us(1);
    checkEqual(0, int(chipIntA), "after mirroring");
    chip.mirrorInterrupts(false);
    wait_us(1);
    checkEqual(1, int(chipIntA), "after mirroring turned off");
}

void testInterruptPolarity() {
// NB by default both int pins are Active-LOW
// interrupt off (so HIGH) after POR
    reset();
    checkEqual(1, int(chipIntA),"interrupt ACTIVE_LOW by default");
    chip.interruptPolarity(ACTIVE_HIGH);
    wait_us(1);
    checkEqual(0, int(chipIntA), "interrupt ACTIVE_HIGH");
    chip.interruptPolarity(ACTIVE_LOW);
    wait_us(1);
    checkEqual(1, int(chipIntA), "interrupt ACTIVE_LOW");
}

void testInterruptControlAndDefaultValueOnPortA() {
    reset();
    chipA0.output(); // output from the mbed
    chip.direction(PORT_A, 0x01); //  bit 0 set to input
    chipA0 = 0;
    checkEqual(1, int(chipIntA),"interrupt ACTIVE_LOW by default");
    chip.interruptEnable(PORT_A, 0x01); // interupt enabled on pin 0
    chip.defaultValue(PORT_A, 0x01); // default value != input value
    checkEqual(1, int(chipIntA),"still no interrupt"); // interrupt control still set to interrupt on change
    chip.interruptControl(PORT_A, 0x01);
    wait_us(1);
    checkEqual(0, int(chipIntA), "expecting interrupt as default != input");
}

void testConfigurePullUpsOnPortA() {
    reset();
    chipA1.output();
    checkEqual(0, chip.read(PORT_A), "without pull-up input should be 0");
    chip.configurePullUps(PORT_A, 0x02); // pin A1 pull-up enabled
    checkEqual(2, chip.read(PORT_A), "pull-up should raise chip input to 1");
}

int main() {
    testInputFromPortA();
    testInputFromPortB();
    testOutputToPortA();
    testOutputToPortB();
    testInterruptEnableOnPortA();
    testInterruptEnableOnPortB();
    testInterruptPolarity();
    testMirrorInterrupts();
    testInterruptControlAndDefaultValueOnPortA();
    testConfigurePullUpsOnPortA();
    // testInterruptControlAndDefaultValueOnPortB();
    // testInterruptCaptureA();
    // testInterruptCaptureB();
    printf("all tests OK\r\n");
}
