// Mbed library to control WS2801-based RGB LED Strips
// some portions (c) 2011 Jelmer Tiete
// This library is ported from the Arduino implementation of Adafruit Industries
// found at: http://github.com/adafruit/LPD8806
// and their strips: http://www.adafruit.com/products/306
// Released under the MIT License: http://mbed.org/license/mit
//
/*****************************************************************************/

// Heavily modified by Jas Strong, 2012-10-04
// Changed to use a virtual base class and to use software SPI.
//
// Modified by Ned Konz, December 2013.
// Using three-phase DMA ala Paul Stoffegren's version.
//
// Modified by richard Thompson, March 2014.
// Uses 8-bit DMA transfers instead of 32-bit, uses 1/4 of the RAM (static)
// Now capable of running 240 LEDs on one pin

#ifndef MBED_WS2811_H
#define MBED_WS2811_H

#include "mbed.h"
#include "LedStrip.h"

//! Maximum number of LEDs per strip
#define MAX_LEDS_PER_STRIP 200

extern "C" void DMA0_IRQHandler();
extern "C" void TPM0_IRQHandler();

/**
* WS2811/WS2812/WS2812B
* LED Strip controller\n
* For FRDM-KL25Z and FRDM-KL46Z

 RAM usage:\n
 Per individual LED: 3 bytes (malloc'd) for RGB data

 Strip length per LED:\n
          24 bytes (static) for bit data\n
        + 24 bytes (static) for ones data\n
        = 48 bytes

        240 LEDs max per string = 11,520 bytes static

      One string:\n
        240 LEDs  : 11520 + 240*3 = 12,240 bytes

      Eight strings:\n
        240*8 LEDs: 11520 + (240*3) * 8 = 17,280 bytes

Example usage:
@code
#include "mbed.h"
#include "WS2811.h"
#include "Colors.h"

static void showRainbow(WS2811 &strip, float startHue, float sat, float brite, float hueShift)
{
    unsigned nLEDs = strip.numPixels();
    float hue = startHue;
    for (unsigned i = 0; i < nLEDs; i++) {
        uint8_t r, g, b;
        Colors::HSBtoRGB(hue, sat, brite, &r, &g, &b);
        strip.setPixelColor(i, r, g, b);
        hue += hueShift;
        if (hue > 1.0) hue = 0.0;
    }
    strip.show();
}

static void showSolidColor(WS2811 &strip, uint8_t r, uint8_t g, uint8_t b)
{
    unsigned nLEDs = strip.numPixels();
    for (unsigned i = 0; i < nLEDs; i++) {
        strip.setPixelColor(i, r, g, b);
    }
    strip.show();
}

int main(void)
{
    WS2811 lightStrip1(nLEDs, 2);
    WS2811 lightStrip2(nLEDs, 3);
    WS2811 lightStrip3(nLEDs, 4);

    lightStrip1.begin();
    lightStrip2.begin();
    lightStrip3.begin();

    uint8_t r =0;
    uint8_t g =0;
    uint8_t b =0;

    bool fadeUp = true;

    float startHue = 0.0;
    for (;;) {
        startHue += 0.01;
        if (startHue > 1.0) startHue = 0.0;
        if (fadeUp) {
            if (r == 255) fadeUp = false;
            else {
                ++r;
                ++g;
                ++b;
            }
        } else {
            if (r == 0) fadeUp = true;
            else {
                --r;
                --g;
                --b;
            }
        }

        // Solid fading white
        showSolidColor(lightStrip1, r, g, b);
        // Maximum saturation rainbow
        showRainbow(lightStrip2, startHue, 1.0, 1.0, 1.0/nLEDs);
        // Pastel rainbow
        showRainbow(lightStrip3, startHue, 0.5, 1.0, 1.0/nLEDs);

        WS2811::startDMA();
        wait_ms(25);
        frames++;
    }
}
@endcode
*/
class WS2811 : public LedStrip
{
public:
    /** Set up the LED strip
    @param pixelCount Number of RGB LEDs on the strip. No more than MAX_LEDS_PER_STRIP (240)
    @param pinNumber Pin number on PORTD. 0-7.
    */
    WS2811(uint16_t pixelCount, uint8_t pinNumber);

    virtual void begin();
    virtual void show();
    virtual void blank();

    /** Send a level update to all the WS2811 LED strips\n
    * All updates happen in parallel, ensure all (max 8) strips have complete data before calling this.
    */
    static void startDMA();

private:
    uint32_t pinMask;

    void writePixel(unsigned n, uint8_t *p);

    // Class Static:

    static bool initialized;
    static uint32_t enabledPins;
    static volatile bool dma_done;
    static void wait_for_dma_done() {
        while (!dma_done) __WFI();
    }

    static void writeByte(uint8_t byte, uint32_t mask, uint8_t *dest);

    static void hw_init();
    static void io_init();
    static void clock_init();
    static void dma_init();
    static void tpm_init();
    static void dma_data_init();

    friend void TPM0_IRQHandler();
};

#endif

