/*
 * Neopixel LED Cube library
 * by Robert Bui and Jose Oyola
 * UC Berkeley 2014
 */

#include "main.h"
#include "mbed.h"
#include "WS2811.h"
#include "Colors.h"
#include "TSISensor.h"
#include "MMA8451Q.h"
#include "ledCube.h"


#define MMA8451_I2C_ADDRESS (0x1d<<1)

#define PANEL1             0
#define PANEL2             1
#define PANEL3             2
#define nLEDs              200 //MAX_LEDS_PER_STRIP;
#define nROWs              10  //number of rows per cube panel 
#define nCOLs              10  //number of columns per cube panel
#define DATA_OUT_PIN1      2   // PTD2
#define DATA_OUT_PIN2      3   // PTD3 


LedCube::LedCube():X(0),Y(1),Z(2),ledStrip1(nLEDs, DATA_OUT_PIN1),ledStrip2(nLEDs, DATA_OUT_PIN2)
{
}

LedCube::~LedCube()
{
}

/*Sets the initial size and position of the lighted cube*/ 
void LedCube::Init(int x, int y, int z)
{
    size = 2;
    prevSize = size;
    pos[X] = prevPos[X] = x;
    pos[Y] = prevPos[Y] = y;
    pos[Z] = prevPos[Z] = z;
    r = 255*0.1;
    g = 255*0.1;
    b = 255*0.1;

    brightness = 0.5;
    saturation = 1.0;
    
    ledStrip1.begin();
    ledStrip2.begin();
}

/*Determines which LED should be lit based on its 
cartesian coordinate. The origin is the led at
bottom left of panel 1 when using a three panel cube
     ________
    /       /|
   /   3   / |
  /_______/ 2|
  |       |  | 
  |   1   | /
  |       |
  --------        
*/

int LedCube::getLedIndex(int panel, int x, int y) {
    if (panel == PANEL1) {
        if (y % 2 == 0) {
            return nCOLs*2 * y + x;
        }
        else {
            return nCOLs*2 * y + nCOLs + ((nCOLs - 1) - x);
        }
    }

    if (panel == PANEL2) {
        if (y % 2 == 0) {
            return nCOLs*2 * y + nCOLs + x;
        }
        else {
            return nCOLs*2 * y + ((nCOLs - 1) - x);
        }
    }
    
    if (panel == PANEL3) {
        if (y % 2 == 0) {
            return nCOLs * y + x;
        }
        else {
            return nCOLs * y + ((nCOLs - 1) - x);
        }
    }
    
    else return -1;
} 

void LedCube::updateLEDs(bool on, int size, int x, int y, int z) {
    //Panel 1
    double bright;
    bright = 1.0 / ((y + 1) * (y + 1));
    printf("Panel 1 Brightness: %f\r\n", bright);
    for(int i = x; i < x + size; i++) {
        for(int j = z; j < z + size; j++) {
            int led = getLedIndex(PANEL1, i, j);
            if(on) {
                ledStrip1.setPixelColor(led, r*bright, g*bright, b*bright);
            } else {
                ledStrip1.setPixelColor(led, 0, 0, 0);
            }
        }
    }
    
    //Panel 2
    bright = 1.0 / (((nCOLs-1) - x - (size-1) + 1) * ((nCOLs-1) - x - (size-1) + 1));
    printf("Panel 2 Brightness: %f\r\n", bright);
    for(int i = y; i < y + size; i++) {
        for(int j = z; j < z + size; j++) {
            int led = getLedIndex(PANEL2, i, j);
            if(on) {
                ledStrip1.setPixelColor(led, r*bright, g*bright, b*bright);
            } else {
                ledStrip1.setPixelColor(led, 0, 0, 0);
            }
        }
    }
    
    //Panel 3
    bright = 1.0 / (((nCOLs-1) - z - (size-1) + 1) * ((nCOLs-1) - z - (size-1) + 1));
    printf("Panel 3 Brightness: %f\r\n", bright);
    for(int i = x; i < x + size; i++) {
        for(int j = y; j < y + size; j++) {
            int led = getLedIndex(PANEL3, i, j);
            if(on) {
                ledStrip2.setPixelColor(led, r*bright, g*bright, b*bright);
            } else {
                ledStrip2.setPixelColor(led, 0, 0, 0);
            }
        }
    }
}

void LedCube::updateLEDsOld(bool on, int size, int x, int y, int z) {
    //Panel 1
    if(y == 0) {
        for(int i = x; i < x + size; i++) {
            for(int j = z; j < z + size; j++) {
                int led = getLedIndex(PANEL1, i, j);
                if(on) {
                    ledStrip1.setPixelColor(led, r, g, b);
                } else {
                    ledStrip1.setPixelColor(led, 0, 0, 0);
                }
            }
        }
    }
    
    //Panel 2
    if(x + size - 1 == (nCOLs - 1)) {
        for(int i = y; i < y + size; i++) {
            for(int j = z; j < z + size; j++) {
                int led = getLedIndex(PANEL2, i, j);
                if(on) {
                    ledStrip1.setPixelColor(led, r, g, b);
                } else {
                    ledStrip1.setPixelColor(led, 0, 0, 0);
                }
            }
        }
    }
    
    //Panel 3
    if(z + size - 1 == (nCOLs - 1)) {
        for(int i = x; i < x + size; i++) {
            for(int j = y; j < y + size; j++) {
                int led = getLedIndex(PANEL3, i, j);
                if(on) {
                    ledStrip2.setPixelColor(led, r, g, b);
                } else {
                    ledStrip2.setPixelColor(led, 0, 0, 0);
                }
            }
        }
    }
}


void LedCube::cubeUpdate() {
    updateLEDs(false, prevSize, prevPos[X], prevPos[Y], prevPos[Z]); //Turn off LEDs from previous state
    updateLEDs(true, size, pos[X], pos[Y], pos[Z]); //Turn on new LEDs for new state
    prevSize = size;
    prevPos[X] = pos[X];
    prevPos[Y] = pos[Y];
    prevPos[Z] = pos[Z];
    ledStrip1.show();
    ledStrip2.show();
    ledStrip1.startDMA();
    ledStrip2.startDMA();
    printf("Position: %d, %d, %d \t Size: %d\r\n", pos[X], pos[Y], pos[Z], size);
}

int LedCube::move(int deltaX, int deltaY, int deltaZ) {
    int retVal = -1;
    //Moving in X direction
    if(pos[Y] == 0 || pos[Z] + size - 1 == (nCOLs - 1)) { //Making sure square is "stuck" to panel 1 or 3
        if((pos[X] + size + deltaX - 1) < nCOLs && (pos[X] + deltaX) >= 0) {
            pos[X] += deltaX;
            if (deltaX != 0) retVal = 1;
        }
    }
    
    //Moving in Y direction
    if(pos[X] + size - 1 == (nCOLs - 1) || pos[Z] + size - 1 == (nCOLs - 1)) {//Making sure square is "stuck" to panel 2 or 3
        if((pos[Y] + size + deltaY - 1) < nCOLs && (pos[Y] + deltaY) >= 0) {
            pos[Y] += deltaY;
            if (deltaY != 0) retVal = 1;
        }
    }
    
    //Moving in Z direction
    if(pos[X] + size - 1 == (nCOLs - 1) || pos[Y] == 0) {//Making sure square is "stuck" to panel 1 or 2
        if((pos[Z] + size + deltaZ - 1) < nCOLs && (pos[Z] + deltaZ) >= 0) {
            pos[Z] += deltaZ;
            if (deltaZ != 0) retVal = 1;
        }
    }
    return retVal;
}

void LedCube::changeColor(float hue){
    Colors::HSBtoRGB(hue, saturation, brightness, &r, &g, &b);
}

void LedCube::changeSize(int newSize) {
    if((pos[X] + newSize) < nCOLs && (pos[Y] + newSize) < nCOLs && (pos[Z] + newSize) < nCOLs) {
        size = newSize;
    }
}

void LedCube::UpdateCube(int size, int deltaX, int deltaY, int deltaZ, float hue) {
     changeSize(size);
     move(deltaX, deltaY, deltaZ); 
     changeColor(hue); 
     cubeUpdate(); 
}



