/* ----------------------------------------------------------------------
 * Project:      CMSIS DSP Library
 * Title:        arm_dct4_init_q15.c
 * Description:  Initialization function of DCT-4 & IDCT4 Q15
 *
 * $Date:        27. January 2017
 * $Revision:    V.1.5.1
 *
 * Target Processor: Cortex-M cores
 * -------------------------------------------------------------------- */
/*
 * Copyright (C) 2010-2017 ARM Limited or its affiliates. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "arm_math.h"

/**
 * @ingroup DCT4_IDCT4
 */

/**
 * @addtogroup DCT4_IDCT4_Table DCT Type IV Tables
 * @{
 */

/*
* @brief  Weights Table
*/

/**
 * \par
 * Weights tables are generated using the formula : <pre>weights[n] = e^(-j*n*pi/(2*N))</pre>
 * \par
 * C command to generate the table
 * <pre>
 * for(i = 0; i< N; i++)
 * {
 *   weights[2*i]= cos(i*c);
 *   weights[(2*i)+1]= -sin(i * c);
 * } </pre>
 * \par
 * where <code>N</code> is the Number of weights to be calculated and <code>c</code> is <code>pi/(2*N)</code>
 * \par
 * Converted the output to q15 format by multiplying with 2^31 and saturated if required.
 * \par
 * In the tables below the real and imaginary values are placed alternatively, hence the
 * array length is <code>2*N</code>.
 */

static const q15_t ALIGN4 WeightsQ15_128[256] = {
  (q15_t)0x7fff, (q15_t)0x0, (q15_t)0x7ffd, (q15_t)0xfe6e, (q15_t)0x7ff6, (q15_t)0xfcdc, (q15_t)0x7fe9, (q15_t)0xfb4a,
  (q15_t)0x7fd8, (q15_t)0xf9b9, (q15_t)0x7fc2, (q15_t)0xf827, (q15_t)0x7fa7, (q15_t)0xf696, (q15_t)0x7f87, (q15_t)0xf505,
  (q15_t)0x7f62, (q15_t)0xf375, (q15_t)0x7f38, (q15_t)0xf1e5, (q15_t)0x7f09, (q15_t)0xf055, (q15_t)0x7ed5, (q15_t)0xeec7,
  (q15_t)0x7e9d, (q15_t)0xed38, (q15_t)0x7e5f, (q15_t)0xebab, (q15_t)0x7e1d, (q15_t)0xea1e, (q15_t)0x7dd6, (q15_t)0xe893,
  (q15_t)0x7d8a, (q15_t)0xe708, (q15_t)0x7d39, (q15_t)0xe57e, (q15_t)0x7ce3, (q15_t)0xe3f5, (q15_t)0x7c89, (q15_t)0xe26d,
  (q15_t)0x7c29, (q15_t)0xe0e7, (q15_t)0x7bc5, (q15_t)0xdf61, (q15_t)0x7b5d, (q15_t)0xdddd, (q15_t)0x7aef, (q15_t)0xdc5a,
  (q15_t)0x7a7d, (q15_t)0xdad8, (q15_t)0x7a05, (q15_t)0xd958, (q15_t)0x798a, (q15_t)0xd7da, (q15_t)0x7909, (q15_t)0xd65d,
  (q15_t)0x7884, (q15_t)0xd4e1, (q15_t)0x77fa, (q15_t)0xd368, (q15_t)0x776c, (q15_t)0xd1ef, (q15_t)0x76d9, (q15_t)0xd079,
  (q15_t)0x7641, (q15_t)0xcf05, (q15_t)0x75a5, (q15_t)0xcd92, (q15_t)0x7504, (q15_t)0xcc22, (q15_t)0x745f, (q15_t)0xcab3,
  (q15_t)0x73b5, (q15_t)0xc946, (q15_t)0x7307, (q15_t)0xc7dc, (q15_t)0x7255, (q15_t)0xc674, (q15_t)0x719e, (q15_t)0xc50e,
  (q15_t)0x70e2, (q15_t)0xc3aa, (q15_t)0x7023, (q15_t)0xc248, (q15_t)0x6f5f, (q15_t)0xc0e9, (q15_t)0x6e96, (q15_t)0xbf8d,
  (q15_t)0x6dca, (q15_t)0xbe32, (q15_t)0x6cf9, (q15_t)0xbcdb, (q15_t)0x6c24, (q15_t)0xbb86, (q15_t)0x6b4a, (q15_t)0xba33,
  (q15_t)0x6a6d, (q15_t)0xb8e4, (q15_t)0x698c, (q15_t)0xb797, (q15_t)0x68a6, (q15_t)0xb64c, (q15_t)0x67bd, (q15_t)0xb505,
  (q15_t)0x66cf, (q15_t)0xb3c1, (q15_t)0x65dd, (q15_t)0xb27f, (q15_t)0x64e8, (q15_t)0xb141, (q15_t)0x63ef, (q15_t)0xb005,
  (q15_t)0x62f2, (q15_t)0xaecd, (q15_t)0x61f1, (q15_t)0xad97, (q15_t)0x60ec, (q15_t)0xac65, (q15_t)0x5fe3, (q15_t)0xab36,
  (q15_t)0x5ed7, (q15_t)0xaa0b, (q15_t)0x5dc7, (q15_t)0xa8e3, (q15_t)0x5cb4, (q15_t)0xa7be, (q15_t)0x5b9d, (q15_t)0xa69c,
  (q15_t)0x5a82, (q15_t)0xa57e, (q15_t)0x5964, (q15_t)0xa463, (q15_t)0x5842, (q15_t)0xa34c, (q15_t)0x571d, (q15_t)0xa239,
  (q15_t)0x55f5, (q15_t)0xa129, (q15_t)0x54ca, (q15_t)0xa01d, (q15_t)0x539b, (q15_t)0x9f14, (q15_t)0x5269, (q15_t)0x9e0f,
  (q15_t)0x5133, (q15_t)0x9d0e, (q15_t)0x4ffb, (q15_t)0x9c11, (q15_t)0x4ebf, (q15_t)0x9b18, (q15_t)0x4d81, (q15_t)0x9a23,
  (q15_t)0x4c3f, (q15_t)0x9931, (q15_t)0x4afb, (q15_t)0x9843, (q15_t)0x49b4, (q15_t)0x975a, (q15_t)0x4869, (q15_t)0x9674,
  (q15_t)0x471c, (q15_t)0x9593, (q15_t)0x45cd, (q15_t)0x94b6, (q15_t)0x447a, (q15_t)0x93dc, (q15_t)0x4325, (q15_t)0x9307,
  (q15_t)0x41ce, (q15_t)0x9236, (q15_t)0x4073, (q15_t)0x916a, (q15_t)0x3f17, (q15_t)0x90a1, (q15_t)0x3db8, (q15_t)0x8fdd,
  (q15_t)0x3c56, (q15_t)0x8f1e, (q15_t)0x3af2, (q15_t)0x8e62, (q15_t)0x398c, (q15_t)0x8dab, (q15_t)0x3824, (q15_t)0x8cf9,
  (q15_t)0x36ba, (q15_t)0x8c4b, (q15_t)0x354d, (q15_t)0x8ba1, (q15_t)0x33de, (q15_t)0x8afc, (q15_t)0x326e, (q15_t)0x8a5b,
  (q15_t)0x30fb, (q15_t)0x89bf, (q15_t)0x2f87, (q15_t)0x8927, (q15_t)0x2e11, (q15_t)0x8894, (q15_t)0x2c98, (q15_t)0x8806,
  (q15_t)0x2b1f, (q15_t)0x877c, (q15_t)0x29a3, (q15_t)0x86f7, (q15_t)0x2826, (q15_t)0x8676, (q15_t)0x26a8, (q15_t)0x85fb,
  (q15_t)0x2528, (q15_t)0x8583, (q15_t)0x23a6, (q15_t)0x8511, (q15_t)0x2223, (q15_t)0x84a3, (q15_t)0x209f, (q15_t)0x843b,
  (q15_t)0x1f19, (q15_t)0x83d7, (q15_t)0x1d93, (q15_t)0x8377, (q15_t)0x1c0b, (q15_t)0x831d, (q15_t)0x1a82, (q15_t)0x82c7,
  (q15_t)0x18f8, (q15_t)0x8276, (q15_t)0x176d, (q15_t)0x822a, (q15_t)0x15e2, (q15_t)0x81e3, (q15_t)0x1455, (q15_t)0x81a1,
  (q15_t)0x12c8, (q15_t)0x8163, (q15_t)0x1139, (q15_t)0x812b, (q15_t)0xfab, (q15_t)0x80f7, (q15_t)0xe1b, (q15_t)0x80c8,
  (q15_t)0xc8b, (q15_t)0x809e, (q15_t)0xafb, (q15_t)0x8079, (q15_t)0x96a, (q15_t)0x8059, (q15_t)0x7d9, (q15_t)0x803e,
  (q15_t)0x647, (q15_t)0x8028, (q15_t)0x4b6, (q15_t)0x8017, (q15_t)0x324, (q15_t)0x800a, (q15_t)0x192, (q15_t)0x8003
};

static const q15_t ALIGN4 WeightsQ15_512[1024] = {
  (q15_t)0x7fff, (q15_t)0x0, (q15_t)0x7fff, (q15_t)0xff9c, (q15_t)0x7fff, (q15_t)0xff37, (q15_t)0x7ffe, (q15_t)0xfed3,
  (q15_t)0x7ffd, (q15_t)0xfe6e, (q15_t)0x7ffc, (q15_t)0xfe0a, (q15_t)0x7ffa, (q15_t)0xfda5, (q15_t)0x7ff8, (q15_t)0xfd41,
  (q15_t)0x7ff6, (q15_t)0xfcdc, (q15_t)0x7ff3, (q15_t)0xfc78, (q15_t)0x7ff0, (q15_t)0xfc13, (q15_t)0x7fed, (q15_t)0xfbaf,
  (q15_t)0x7fe9, (q15_t)0xfb4a, (q15_t)0x7fe5, (q15_t)0xfae6, (q15_t)0x7fe1, (q15_t)0xfa81, (q15_t)0x7fdd, (q15_t)0xfa1d,
  (q15_t)0x7fd8, (q15_t)0xf9b9, (q15_t)0x7fd3, (q15_t)0xf954, (q15_t)0x7fce, (q15_t)0xf8f0, (q15_t)0x7fc8, (q15_t)0xf88b,
  (q15_t)0x7fc2, (q15_t)0xf827, (q15_t)0x7fbc, (q15_t)0xf7c3, (q15_t)0x7fb5, (q15_t)0xf75e, (q15_t)0x7fae, (q15_t)0xf6fa,
  (q15_t)0x7fa7, (q15_t)0xf696, (q15_t)0x7f9f, (q15_t)0xf632, (q15_t)0x7f97, (q15_t)0xf5cd, (q15_t)0x7f8f, (q15_t)0xf569,
  (q15_t)0x7f87, (q15_t)0xf505, (q15_t)0x7f7e, (q15_t)0xf4a1, (q15_t)0x7f75, (q15_t)0xf43d, (q15_t)0x7f6b, (q15_t)0xf3d9,
  (q15_t)0x7f62, (q15_t)0xf375, (q15_t)0x7f58, (q15_t)0xf311, (q15_t)0x7f4d, (q15_t)0xf2ad, (q15_t)0x7f43, (q15_t)0xf249,
  (q15_t)0x7f38, (q15_t)0xf1e5, (q15_t)0x7f2d, (q15_t)0xf181, (q15_t)0x7f21, (q15_t)0xf11d, (q15_t)0x7f15, (q15_t)0xf0b9,
  (q15_t)0x7f09, (q15_t)0xf055, (q15_t)0x7efd, (q15_t)0xeff2, (q15_t)0x7ef0, (q15_t)0xef8e, (q15_t)0x7ee3, (q15_t)0xef2a,
  (q15_t)0x7ed5, (q15_t)0xeec7, (q15_t)0x7ec8, (q15_t)0xee63, (q15_t)0x7eba, (q15_t)0xedff, (q15_t)0x7eab, (q15_t)0xed9c,
  (q15_t)0x7e9d, (q15_t)0xed38, (q15_t)0x7e8e, (q15_t)0xecd5, (q15_t)0x7e7f, (q15_t)0xec72, (q15_t)0x7e6f, (q15_t)0xec0e,
  (q15_t)0x7e5f, (q15_t)0xebab, (q15_t)0x7e4f, (q15_t)0xeb48, (q15_t)0x7e3f, (q15_t)0xeae5, (q15_t)0x7e2e, (q15_t)0xea81,
  (q15_t)0x7e1d, (q15_t)0xea1e, (q15_t)0x7e0c, (q15_t)0xe9bb, (q15_t)0x7dfa, (q15_t)0xe958, (q15_t)0x7de8, (q15_t)0xe8f6,
  (q15_t)0x7dd6, (q15_t)0xe893, (q15_t)0x7dc3, (q15_t)0xe830, (q15_t)0x7db0, (q15_t)0xe7cd, (q15_t)0x7d9d, (q15_t)0xe76a,
  (q15_t)0x7d8a, (q15_t)0xe708, (q15_t)0x7d76, (q15_t)0xe6a5, (q15_t)0x7d62, (q15_t)0xe643, (q15_t)0x7d4e, (q15_t)0xe5e0,
  (q15_t)0x7d39, (q15_t)0xe57e, (q15_t)0x7d24, (q15_t)0xe51c, (q15_t)0x7d0f, (q15_t)0xe4b9, (q15_t)0x7cf9, (q15_t)0xe457,
  (q15_t)0x7ce3, (q15_t)0xe3f5, (q15_t)0x7ccd, (q15_t)0xe393, (q15_t)0x7cb7, (q15_t)0xe331, (q15_t)0x7ca0, (q15_t)0xe2cf,
  (q15_t)0x7c89, (q15_t)0xe26d, (q15_t)0x7c71, (q15_t)0xe20b, (q15_t)0x7c5a, (q15_t)0xe1aa, (q15_t)0x7c42, (q15_t)0xe148,
  (q15_t)0x7c29, (q15_t)0xe0e7, (q15_t)0x7c11, (q15_t)0xe085, (q15_t)0x7bf8, (q15_t)0xe024, (q15_t)0x7bdf, (q15_t)0xdfc2,
  (q15_t)0x7bc5, (q15_t)0xdf61, (q15_t)0x7bac, (q15_t)0xdf00, (q15_t)0x7b92, (q15_t)0xde9f, (q15_t)0x7b77, (q15_t)0xde3e,
  (q15_t)0x7b5d, (q15_t)0xdddd, (q15_t)0x7b42, (q15_t)0xdd7c, (q15_t)0x7b26, (q15_t)0xdd1b, (q15_t)0x7b0b, (q15_t)0xdcbb,
  (q15_t)0x7aef, (q15_t)0xdc5a, (q15_t)0x7ad3, (q15_t)0xdbf9, (q15_t)0x7ab6, (q15_t)0xdb99, (q15_t)0x7a9a, (q15_t)0xdb39,
  (q15_t)0x7a7d, (q15_t)0xdad8, (q15_t)0x7a5f, (q15_t)0xda78, (q15_t)0x7a42, (q15_t)0xda18, (q15_t)0x7a24, (q15_t)0xd9b8,
  (q15_t)0x7a05, (q15_t)0xd958, (q15_t)0x79e7, (q15_t)0xd8f9, (q15_t)0x79c8, (q15_t)0xd899, (q15_t)0x79a9, (q15_t)0xd839,
  (q15_t)0x798a, (q15_t)0xd7da, (q15_t)0x796a, (q15_t)0xd77a, (q15_t)0x794a, (q15_t)0xd71b, (q15_t)0x792a, (q15_t)0xd6bc,
  (q15_t)0x7909, (q15_t)0xd65d, (q15_t)0x78e8, (q15_t)0xd5fe, (q15_t)0x78c7, (q15_t)0xd59f, (q15_t)0x78a6, (q15_t)0xd540,
  (q15_t)0x7884, (q15_t)0xd4e1, (q15_t)0x7862, (q15_t)0xd483, (q15_t)0x7840, (q15_t)0xd424, (q15_t)0x781d, (q15_t)0xd3c6,
  (q15_t)0x77fa, (q15_t)0xd368, (q15_t)0x77d7, (q15_t)0xd309, (q15_t)0x77b4, (q15_t)0xd2ab, (q15_t)0x7790, (q15_t)0xd24d,
  (q15_t)0x776c, (q15_t)0xd1ef, (q15_t)0x7747, (q15_t)0xd192, (q15_t)0x7723, (q15_t)0xd134, (q15_t)0x76fe, (q15_t)0xd0d7,
  (q15_t)0x76d9, (q15_t)0xd079, (q15_t)0x76b3, (q15_t)0xd01c, (q15_t)0x768e, (q15_t)0xcfbf, (q15_t)0x7668, (q15_t)0xcf62,
  (q15_t)0x7641, (q15_t)0xcf05, (q15_t)0x761b, (q15_t)0xcea8, (q15_t)0x75f4, (q15_t)0xce4b, (q15_t)0x75cc, (q15_t)0xcdef,
  (q15_t)0x75a5, (q15_t)0xcd92, (q15_t)0x757d, (q15_t)0xcd36, (q15_t)0x7555, (q15_t)0xccda, (q15_t)0x752d, (q15_t)0xcc7e,
  (q15_t)0x7504, (q15_t)0xcc22, (q15_t)0x74db, (q15_t)0xcbc6, (q15_t)0x74b2, (q15_t)0xcb6a, (q15_t)0x7489, (q15_t)0xcb0e,
  (q15_t)0x745f, (q15_t)0xcab3, (q15_t)0x7435, (q15_t)0xca58, (q15_t)0x740b, (q15_t)0xc9fc, (q15_t)0x73e0, (q15_t)0xc9a1,
  (q15_t)0x73b5, (q15_t)0xc946, (q15_t)0x738a, (q15_t)0xc8ec, (q15_t)0x735f, (q15_t)0xc891, (q15_t)0x7333, (q15_t)0xc836,
  (q15_t)0x7307, (q15_t)0xc7dc, (q15_t)0x72db, (q15_t)0xc782, (q15_t)0x72af, (q15_t)0xc728, (q15_t)0x7282, (q15_t)0xc6ce,
  (q15_t)0x7255, (q15_t)0xc674, (q15_t)0x7227, (q15_t)0xc61a, (q15_t)0x71fa, (q15_t)0xc5c0, (q15_t)0x71cc, (q15_t)0xc567,
  (q15_t)0x719e, (q15_t)0xc50e, (q15_t)0x716f, (q15_t)0xc4b4, (q15_t)0x7141, (q15_t)0xc45b, (q15_t)0x7112, (q15_t)0xc403,
  (q15_t)0x70e2, (q15_t)0xc3aa, (q15_t)0x70b3, (q15_t)0xc351, (q15_t)0x7083, (q15_t)0xc2f9, (q15_t)0x7053, (q15_t)0xc2a0,
  (q15_t)0x7023, (q15_t)0xc248, (q15_t)0x6ff2, (q15_t)0xc1f0, (q15_t)0x6fc1, (q15_t)0xc198, (q15_t)0x6f90, (q15_t)0xc141,
  (q15_t)0x6f5f, (q15_t)0xc0e9, (q15_t)0x6f2d, (q15_t)0xc092, (q15_t)0x6efb, (q15_t)0xc03b, (q15_t)0x6ec9, (q15_t)0xbfe3,
  (q15_t)0x6e96, (q15_t)0xbf8d, (q15_t)0x6e63, (q15_t)0xbf36, (q15_t)0x6e30, (q15_t)0xbedf, (q15_t)0x6dfd, (q15_t)0xbe89,
  (q15_t)0x6dca, (q15_t)0xbe32, (q15_t)0x6d96, (q15_t)0xbddc, (q15_t)0x6d62, (q15_t)0xbd86, (q15_t)0x6d2d, (q15_t)0xbd30,
  (q15_t)0x6cf9, (q15_t)0xbcdb, (q15_t)0x6cc4, (q15_t)0xbc85, (q15_t)0x6c8f, (q15_t)0xbc30, (q15_t)0x6c59, (q15_t)0xbbdb,
  (q15_t)0x6c24, (q15_t)0xbb86, (q15_t)0x6bee, (q15_t)0xbb31, (q15_t)0x6bb8, (q15_t)0xbadc, (q15_t)0x6b81, (q15_t)0xba88,
  (q15_t)0x6b4a, (q15_t)0xba33, (q15_t)0x6b13, (q15_t)0xb9df, (q15_t)0x6adc, (q15_t)0xb98b, (q15_t)0x6aa5, (q15_t)0xb937,
  (q15_t)0x6a6d, (q15_t)0xb8e4, (q15_t)0x6a35, (q15_t)0xb890, (q15_t)0x69fd, (q15_t)0xb83d, (q15_t)0x69c4, (q15_t)0xb7ea,
  (q15_t)0x698c, (q15_t)0xb797, (q15_t)0x6953, (q15_t)0xb744, (q15_t)0x6919, (q15_t)0xb6f1, (q15_t)0x68e0, (q15_t)0xb69f,
  (q15_t)0x68a6, (q15_t)0xb64c, (q15_t)0x686c, (q15_t)0xb5fa, (q15_t)0x6832, (q15_t)0xb5a8, (q15_t)0x67f7, (q15_t)0xb557,
  (q15_t)0x67bd, (q15_t)0xb505, (q15_t)0x6782, (q15_t)0xb4b4, (q15_t)0x6746, (q15_t)0xb462, (q15_t)0x670b, (q15_t)0xb411,
  (q15_t)0x66cf, (q15_t)0xb3c1, (q15_t)0x6693, (q15_t)0xb370, (q15_t)0x6657, (q15_t)0xb31f, (q15_t)0x661a, (q15_t)0xb2cf,
  (q15_t)0x65dd, (q15_t)0xb27f, (q15_t)0x65a0, (q15_t)0xb22f, (q15_t)0x6563, (q15_t)0xb1df, (q15_t)0x6526, (q15_t)0xb190,
  (q15_t)0x64e8, (q15_t)0xb141, (q15_t)0x64aa, (q15_t)0xb0f1, (q15_t)0x646c, (q15_t)0xb0a2, (q15_t)0x642d, (q15_t)0xb054,
  (q15_t)0x63ef, (q15_t)0xb005, (q15_t)0x63b0, (q15_t)0xafb7, (q15_t)0x6371, (q15_t)0xaf69, (q15_t)0x6331, (q15_t)0xaf1b,
  (q15_t)0x62f2, (q15_t)0xaecd, (q15_t)0x62b2, (q15_t)0xae7f, (q15_t)0x6271, (q15_t)0xae32, (q15_t)0x6231, (q15_t)0xade4,
  (q15_t)0x61f1, (q15_t)0xad97, (q15_t)0x61b0, (q15_t)0xad4b, (q15_t)0x616f, (q15_t)0xacfe, (q15_t)0x612d, (q15_t)0xacb2,
  (q15_t)0x60ec, (q15_t)0xac65, (q15_t)0x60aa, (q15_t)0xac19, (q15_t)0x6068, (q15_t)0xabcd, (q15_t)0x6026, (q15_t)0xab82,
  (q15_t)0x5fe3, (q15_t)0xab36, (q15_t)0x5fa0, (q15_t)0xaaeb, (q15_t)0x5f5e, (q15_t)0xaaa0, (q15_t)0x5f1a, (q15_t)0xaa55,
  (q15_t)0x5ed7, (q15_t)0xaa0b, (q15_t)0x5e93, (q15_t)0xa9c0, (q15_t)0x5e50, (q15_t)0xa976, (q15_t)0x5e0b, (q15_t)0xa92c,
  (q15_t)0x5dc7, (q15_t)0xa8e3, (q15_t)0x5d83, (q15_t)0xa899, (q15_t)0x5d3e, (q15_t)0xa850, (q15_t)0x5cf9, (q15_t)0xa807,
  (q15_t)0x5cb4, (q15_t)0xa7be, (q15_t)0x5c6e, (q15_t)0xa775, (q15_t)0x5c29, (q15_t)0xa72c, (q15_t)0x5be3, (q15_t)0xa6e4,
  (q15_t)0x5b9d, (q15_t)0xa69c, (q15_t)0x5b56, (q15_t)0xa654, (q15_t)0x5b10, (q15_t)0xa60d, (q15_t)0x5ac9, (q15_t)0xa5c5,
  (q15_t)0x5a82, (q15_t)0xa57e, (q15_t)0x5a3b, (q15_t)0xa537, (q15_t)0x59f3, (q15_t)0xa4f0, (q15_t)0x59ac, (q15_t)0xa4aa,
  (q15_t)0x5964, (q15_t)0xa463, (q15_t)0x591c, (q15_t)0xa41d, (q15_t)0x58d4, (q15_t)0xa3d7, (q15_t)0x588b, (q15_t)0xa392,
  (q15_t)0x5842, (q15_t)0xa34c, (q15_t)0x57f9, (q15_t)0xa307, (q15_t)0x57b0, (q15_t)0xa2c2, (q15_t)0x5767, (q15_t)0xa27d,
  (q15_t)0x571d, (q15_t)0xa239, (q15_t)0x56d4, (q15_t)0xa1f5, (q15_t)0x568a, (q15_t)0xa1b0, (q15_t)0x5640, (q15_t)0xa16d,
  (q15_t)0x55f5, (q15_t)0xa129, (q15_t)0x55ab, (q15_t)0xa0e6, (q15_t)0x5560, (q15_t)0xa0a2, (q15_t)0x5515, (q15_t)0xa060,
  (q15_t)0x54ca, (q15_t)0xa01d, (q15_t)0x547e, (q15_t)0x9fda, (q15_t)0x5433, (q15_t)0x9f98, (q15_t)0x53e7, (q15_t)0x9f56,
  (q15_t)0x539b, (q15_t)0x9f14, (q15_t)0x534e, (q15_t)0x9ed3, (q15_t)0x5302, (q15_t)0x9e91, (q15_t)0x52b5, (q15_t)0x9e50,
  (q15_t)0x5269, (q15_t)0x9e0f, (q15_t)0x521c, (q15_t)0x9dcf, (q15_t)0x51ce, (q15_t)0x9d8f, (q15_t)0x5181, (q15_t)0x9d4e,
  (q15_t)0x5133, (q15_t)0x9d0e, (q15_t)0x50e5, (q15_t)0x9ccf, (q15_t)0x5097, (q15_t)0x9c8f, (q15_t)0x5049, (q15_t)0x9c50,
  (q15_t)0x4ffb, (q15_t)0x9c11, (q15_t)0x4fac, (q15_t)0x9bd3, (q15_t)0x4f5e, (q15_t)0x9b94, (q15_t)0x4f0f, (q15_t)0x9b56,
  (q15_t)0x4ebf, (q15_t)0x9b18, (q15_t)0x4e70, (q15_t)0x9ada, (q15_t)0x4e21, (q15_t)0x9a9d, (q15_t)0x4dd1, (q15_t)0x9a60,
  (q15_t)0x4d81, (q15_t)0x9a23, (q15_t)0x4d31, (q15_t)0x99e6, (q15_t)0x4ce1, (q15_t)0x99a9, (q15_t)0x4c90, (q15_t)0x996d,
  (q15_t)0x4c3f, (q15_t)0x9931, (q15_t)0x4bef, (q15_t)0x98f5, (q15_t)0x4b9e, (q15_t)0x98ba, (q15_t)0x4b4c, (q15_t)0x987e,
  (q15_t)0x4afb, (q15_t)0x9843, (q15_t)0x4aa9, (q15_t)0x9809, (q15_t)0x4a58, (q15_t)0x97ce, (q15_t)0x4a06, (q15_t)0x9794,
  (q15_t)0x49b4, (q15_t)0x975a, (q15_t)0x4961, (q15_t)0x9720, (q15_t)0x490f, (q15_t)0x96e7, (q15_t)0x48bc, (q15_t)0x96ad,
  (q15_t)0x4869, (q15_t)0x9674, (q15_t)0x4816, (q15_t)0x963c, (q15_t)0x47c3, (q15_t)0x9603, (q15_t)0x4770, (q15_t)0x95cb,
  (q15_t)0x471c, (q15_t)0x9593, (q15_t)0x46c9, (q15_t)0x955b, (q15_t)0x4675, (q15_t)0x9524, (q15_t)0x4621, (q15_t)0x94ed,
  (q15_t)0x45cd, (q15_t)0x94b6, (q15_t)0x4578, (q15_t)0x947f, (q15_t)0x4524, (q15_t)0x9448, (q15_t)0x44cf, (q15_t)0x9412,
  (q15_t)0x447a, (q15_t)0x93dc, (q15_t)0x4425, (q15_t)0x93a7, (q15_t)0x43d0, (q15_t)0x9371, (q15_t)0x437b, (q15_t)0x933c,
  (q15_t)0x4325, (q15_t)0x9307, (q15_t)0x42d0, (q15_t)0x92d3, (q15_t)0x427a, (q15_t)0x929e, (q15_t)0x4224, (q15_t)0x926a,
  (q15_t)0x41ce, (q15_t)0x9236, (q15_t)0x4177, (q15_t)0x9203, (q15_t)0x4121, (q15_t)0x91d0, (q15_t)0x40ca, (q15_t)0x919d,
  (q15_t)0x4073, (q15_t)0x916a, (q15_t)0x401d, (q15_t)0x9137, (q15_t)0x3fc5, (q15_t)0x9105, (q15_t)0x3f6e, (q15_t)0x90d3,
  (q15_t)0x3f17, (q15_t)0x90a1, (q15_t)0x3ebf, (q15_t)0x9070, (q15_t)0x3e68, (q15_t)0x903f, (q15_t)0x3e10, (q15_t)0x900e,
  (q15_t)0x3db8, (q15_t)0x8fdd, (q15_t)0x3d60, (q15_t)0x8fad, (q15_t)0x3d07, (q15_t)0x8f7d, (q15_t)0x3caf, (q15_t)0x8f4d,
  (q15_t)0x3c56, (q15_t)0x8f1e, (q15_t)0x3bfd, (q15_t)0x8eee, (q15_t)0x3ba5, (q15_t)0x8ebf, (q15_t)0x3b4c, (q15_t)0x8e91,
  (q15_t)0x3af2, (q15_t)0x8e62, (q15_t)0x3a99, (q15_t)0x8e34, (q15_t)0x3a40, (q15_t)0x8e06, (q15_t)0x39e6, (q15_t)0x8dd9,
  (q15_t)0x398c, (q15_t)0x8dab, (q15_t)0x3932, (q15_t)0x8d7e, (q15_t)0x38d8, (q15_t)0x8d51, (q15_t)0x387e, (q15_t)0x8d25,
  (q15_t)0x3824, (q15_t)0x8cf9, (q15_t)0x37ca, (q15_t)0x8ccd, (q15_t)0x376f, (q15_t)0x8ca1, (q15_t)0x3714, (q15_t)0x8c76,
  (q15_t)0x36ba, (q15_t)0x8c4b, (q15_t)0x365f, (q15_t)0x8c20, (q15_t)0x3604, (q15_t)0x8bf5, (q15_t)0x35a8, (q15_t)0x8bcb,
  (q15_t)0x354d, (q15_t)0x8ba1, (q15_t)0x34f2, (q15_t)0x8b77, (q15_t)0x3496, (q15_t)0x8b4e, (q15_t)0x343a, (q15_t)0x8b25,
  (q15_t)0x33de, (q15_t)0x8afc, (q15_t)0x3382, (q15_t)0x8ad3, (q15_t)0x3326, (q15_t)0x8aab, (q15_t)0x32ca, (q15_t)0x8a83,
  (q15_t)0x326e, (q15_t)0x8a5b, (q15_t)0x3211, (q15_t)0x8a34, (q15_t)0x31b5, (q15_t)0x8a0c, (q15_t)0x3158, (q15_t)0x89e5,
  (q15_t)0x30fb, (q15_t)0x89bf, (q15_t)0x309e, (q15_t)0x8998, (q15_t)0x3041, (q15_t)0x8972, (q15_t)0x2fe4, (q15_t)0x894d,
  (q15_t)0x2f87, (q15_t)0x8927, (q15_t)0x2f29, (q15_t)0x8902, (q15_t)0x2ecc, (q15_t)0x88dd, (q15_t)0x2e6e, (q15_t)0x88b9,
  (q15_t)0x2e11, (q15_t)0x8894, (q15_t)0x2db3, (q15_t)0x8870, (q15_t)0x2d55, (q15_t)0x884c, (q15_t)0x2cf7, (q15_t)0x8829,
  (q15_t)0x2c98, (q15_t)0x8806, (q15_t)0x2c3a, (q15_t)0x87e3, (q15_t)0x2bdc, (q15_t)0x87c0, (q15_t)0x2b7d, (q15_t)0x879e,
  (q15_t)0x2b1f, (q15_t)0x877c, (q15_t)0x2ac0, (q15_t)0x875a, (q15_t)0x2a61, (q15_t)0x8739, (q15_t)0x2a02, (q15_t)0x8718,
  (q15_t)0x29a3, (q15_t)0x86f7, (q15_t)0x2944, (q15_t)0x86d6, (q15_t)0x28e5, (q15_t)0x86b6, (q15_t)0x2886, (q15_t)0x8696,
  (q15_t)0x2826, (q15_t)0x8676, (q15_t)0x27c7, (q15_t)0x8657, (q15_t)0x2767, (q15_t)0x8638, (q15_t)0x2707, (q15_t)0x8619,
  (q15_t)0x26a8, (q15_t)0x85fb, (q15_t)0x2648, (q15_t)0x85dc, (q15_t)0x25e8, (q15_t)0x85be, (q15_t)0x2588, (q15_t)0x85a1,
  (q15_t)0x2528, (q15_t)0x8583, (q15_t)0x24c7, (q15_t)0x8566, (q15_t)0x2467, (q15_t)0x854a, (q15_t)0x2407, (q15_t)0x852d,
  (q15_t)0x23a6, (q15_t)0x8511, (q15_t)0x2345, (q15_t)0x84f5, (q15_t)0x22e5, (q15_t)0x84da, (q15_t)0x2284, (q15_t)0x84be,
  (q15_t)0x2223, (q15_t)0x84a3, (q15_t)0x21c2, (q15_t)0x8489, (q15_t)0x2161, (q15_t)0x846e, (q15_t)0x2100, (q15_t)0x8454,
  (q15_t)0x209f, (q15_t)0x843b, (q15_t)0x203e, (q15_t)0x8421, (q15_t)0x1fdc, (q15_t)0x8408, (q15_t)0x1f7b, (q15_t)0x83ef,
  (q15_t)0x1f19, (q15_t)0x83d7, (q15_t)0x1eb8, (q15_t)0x83be, (q15_t)0x1e56, (q15_t)0x83a6, (q15_t)0x1df5, (q15_t)0x838f,
  (q15_t)0x1d93, (q15_t)0x8377, (q15_t)0x1d31, (q15_t)0x8360, (q15_t)0x1ccf, (q15_t)0x8349, (q15_t)0x1c6d, (q15_t)0x8333,
  (q15_t)0x1c0b, (q15_t)0x831d, (q15_t)0x1ba9, (q15_t)0x8307, (q15_t)0x1b47, (q15_t)0x82f1, (q15_t)0x1ae4, (q15_t)0x82dc,
  (q15_t)0x1a82, (q15_t)0x82c7, (q15_t)0x1a20, (q15_t)0x82b2, (q15_t)0x19bd, (q15_t)0x829e, (q15_t)0x195b, (q15_t)0x828a,
  (q15_t)0x18f8, (q15_t)0x8276, (q15_t)0x1896, (q15_t)0x8263, (q15_t)0x1833, (q15_t)0x8250, (q15_t)0x17d0, (q15_t)0x823d,
  (q15_t)0x176d, (q15_t)0x822a, (q15_t)0x170a, (q15_t)0x8218, (q15_t)0x16a8, (q15_t)0x8206, (q15_t)0x1645, (q15_t)0x81f4,
  (q15_t)0x15e2, (q15_t)0x81e3, (q15_t)0x157f, (q15_t)0x81d2, (q15_t)0x151b, (q15_t)0x81c1, (q15_t)0x14b8, (q15_t)0x81b1,
  (q15_t)0x1455, (q15_t)0x81a1, (q15_t)0x13f2, (q15_t)0x8191, (q15_t)0x138e, (q15_t)0x8181, (q15_t)0x132b, (q15_t)0x8172,
  (q15_t)0x12c8, (q15_t)0x8163, (q15_t)0x1264, (q15_t)0x8155, (q15_t)0x1201, (q15_t)0x8146, (q15_t)0x119d, (q15_t)0x8138,
  (q15_t)0x1139, (q15_t)0x812b, (q15_t)0x10d6, (q15_t)0x811d, (q15_t)0x1072, (q15_t)0x8110, (q15_t)0x100e, (q15_t)0x8103,
  (q15_t)0xfab, (q15_t)0x80f7, (q15_t)0xf47, (q15_t)0x80eb, (q15_t)0xee3, (q15_t)0x80df, (q15_t)0xe7f, (q15_t)0x80d3,
  (q15_t)0xe1b, (q15_t)0x80c8, (q15_t)0xdb7, (q15_t)0x80bd, (q15_t)0xd53, (q15_t)0x80b3, (q15_t)0xcef, (q15_t)0x80a8,
  (q15_t)0xc8b, (q15_t)0x809e, (q15_t)0xc27, (q15_t)0x8095, (q15_t)0xbc3, (q15_t)0x808b, (q15_t)0xb5f, (q15_t)0x8082,
  (q15_t)0xafb, (q15_t)0x8079, (q15_t)0xa97, (q15_t)0x8071, (q15_t)0xa33, (q15_t)0x8069, (q15_t)0x9ce, (q15_t)0x8061,
  (q15_t)0x96a, (q15_t)0x8059, (q15_t)0x906, (q15_t)0x8052, (q15_t)0x8a2, (q15_t)0x804b, (q15_t)0x83d, (q15_t)0x8044,
  (q15_t)0x7d9, (q15_t)0x803e, (q15_t)0x775, (q15_t)0x8038, (q15_t)0x710, (q15_t)0x8032, (q15_t)0x6ac, (q15_t)0x802d,
  (q15_t)0x647, (q15_t)0x8028, (q15_t)0x5e3, (q15_t)0x8023, (q15_t)0x57f, (q15_t)0x801f, (q15_t)0x51a, (q15_t)0x801b,
  (q15_t)0x4b6, (q15_t)0x8017, (q15_t)0x451, (q15_t)0x8013, (q15_t)0x3ed, (q15_t)0x8010, (q15_t)0x388, (q15_t)0x800d,
  (q15_t)0x324, (q15_t)0x800a, (q15_t)0x2bf, (q15_t)0x8008, (q15_t)0x25b, (q15_t)0x8006, (q15_t)0x1f6, (q15_t)0x8004,
  (q15_t)0x192, (q15_t)0x8003, (q15_t)0x12d, (q15_t)0x8002, (q15_t)0xc9, (q15_t)0x8001, (q15_t)0x64, (q15_t)0x8001
};

static const q15_t ALIGN4 WeightsQ15_2048[4096] = {
  (q15_t)0x7fff, (q15_t)0x0, (q15_t)0x7fff, (q15_t)0xffe7, (q15_t)0x7fff, (q15_t)0xffce, (q15_t)0x7fff, (q15_t)0xffb5,
  (q15_t)0x7fff, (q15_t)0xff9c, (q15_t)0x7fff, (q15_t)0xff83, (q15_t)0x7fff, (q15_t)0xff6a, (q15_t)0x7fff, (q15_t)0xff51,
  (q15_t)0x7fff, (q15_t)0xff37, (q15_t)0x7fff, (q15_t)0xff1e, (q15_t)0x7fff, (q15_t)0xff05, (q15_t)0x7ffe, (q15_t)0xfeec,
  (q15_t)0x7ffe, (q15_t)0xfed3, (q15_t)0x7ffe, (q15_t)0xfeba, (q15_t)0x7ffe, (q15_t)0xfea1, (q15_t)0x7ffd, (q15_t)0xfe88,
  (q15_t)0x7ffd, (q15_t)0xfe6e, (q15_t)0x7ffd, (q15_t)0xfe55, (q15_t)0x7ffc, (q15_t)0xfe3c, (q15_t)0x7ffc, (q15_t)0xfe23,
  (q15_t)0x7ffc, (q15_t)0xfe0a, (q15_t)0x7ffb, (q15_t)0xfdf1, (q15_t)0x7ffb, (q15_t)0xfdd8, (q15_t)0x7ffa, (q15_t)0xfdbe,
  (q15_t)0x7ffa, (q15_t)0xfda5, (q15_t)0x7ff9, (q15_t)0xfd8c, (q15_t)0x7ff9, (q15_t)0xfd73, (q15_t)0x7ff8, (q15_t)0xfd5a,
  (q15_t)0x7ff8, (q15_t)0xfd41, (q15_t)0x7ff7, (q15_t)0xfd28, (q15_t)0x7ff7, (q15_t)0xfd0f, (q15_t)0x7ff6, (q15_t)0xfcf5,
  (q15_t)0x7ff6, (q15_t)0xfcdc, (q15_t)0x7ff5, (q15_t)0xfcc3, (q15_t)0x7ff4, (q15_t)0xfcaa, (q15_t)0x7ff4, (q15_t)0xfc91,
  (q15_t)0x7ff3, (q15_t)0xfc78, (q15_t)0x7ff2, (q15_t)0xfc5f, (q15_t)0x7ff2, (q15_t)0xfc46, (q15_t)0x7ff1, (q15_t)0xfc2c,
  (q15_t)0x7ff0, (q15_t)0xfc13, (q15_t)0x7fef, (q15_t)0xfbfa, (q15_t)0x7fee, (q15_t)0xfbe1, (q15_t)0x7fee, (q15_t)0xfbc8,
  (q15_t)0x7fed, (q15_t)0xfbaf, (q15_t)0x7fec, (q15_t)0xfb96, (q15_t)0x7feb, (q15_t)0xfb7d, (q15_t)0x7fea, (q15_t)0xfb64,
  (q15_t)0x7fe9, (q15_t)0xfb4a, (q15_t)0x7fe8, (q15_t)0xfb31, (q15_t)0x7fe7, (q15_t)0xfb18, (q15_t)0x7fe6, (q15_t)0xfaff,
  (q15_t)0x7fe5, (q15_t)0xfae6, (q15_t)0x7fe4, (q15_t)0xfacd, (q15_t)0x7fe3, (q15_t)0xfab4, (q15_t)0x7fe2, (q15_t)0xfa9b,
  (q15_t)0x7fe1, (q15_t)0xfa81, (q15_t)0x7fe0, (q15_t)0xfa68, (q15_t)0x7fdf, (q15_t)0xfa4f, (q15_t)0x7fde, (q15_t)0xfa36,
  (q15_t)0x7fdd, (q15_t)0xfa1d, (q15_t)0x7fdc, (q15_t)0xfa04, (q15_t)0x7fda, (q15_t)0xf9eb, (q15_t)0x7fd9, (q15_t)0xf9d2,
  (q15_t)0x7fd8, (q15_t)0xf9b9, (q15_t)0x7fd7, (q15_t)0xf9a0, (q15_t)0x7fd6, (q15_t)0xf986, (q15_t)0x7fd4, (q15_t)0xf96d,
  (q15_t)0x7fd3, (q15_t)0xf954, (q15_t)0x7fd2, (q15_t)0xf93b, (q15_t)0x7fd0, (q15_t)0xf922, (q15_t)0x7fcf, (q15_t)0xf909,
  (q15_t)0x7fce, (q15_t)0xf8f0, (q15_t)0x7fcc, (q15_t)0xf8d7, (q15_t)0x7fcb, (q15_t)0xf8be, (q15_t)0x7fc9, (q15_t)0xf8a5,
  (q15_t)0x7fc8, (q15_t)0xf88b, (q15_t)0x7fc6, (q15_t)0xf872, (q15_t)0x7fc5, (q15_t)0xf859, (q15_t)0x7fc3, (q15_t)0xf840,
  (q15_t)0x7fc2, (q15_t)0xf827, (q15_t)0x7fc0, (q15_t)0xf80e, (q15_t)0x7fbf, (q15_t)0xf7f5, (q15_t)0x7fbd, (q15_t)0xf7dc,
  (q15_t)0x7fbc, (q15_t)0xf7c3, (q15_t)0x7fba, (q15_t)0xf7aa, (q15_t)0x7fb8, (q15_t)0xf791, (q15_t)0x7fb7, (q15_t)0xf778,
  (q15_t)0x7fb5, (q15_t)0xf75e, (q15_t)0x7fb3, (q15_t)0xf745, (q15_t)0x7fb1, (q15_t)0xf72c, (q15_t)0x7fb0, (q15_t)0xf713,
  (q15_t)0x7fae, (q15_t)0xf6fa, (q15_t)0x7fac, (q15_t)0xf6e1, (q15_t)0x7faa, (q15_t)0xf6c8, (q15_t)0x7fa9, (q15_t)0xf6af,
  (q15_t)0x7fa7, (q15_t)0xf696, (q15_t)0x7fa5, (q15_t)0xf67d, (q15_t)0x7fa3, (q15_t)0xf664, (q15_t)0x7fa1, (q15_t)0xf64b,
  (q15_t)0x7f9f, (q15_t)0xf632, (q15_t)0x7f9d, (q15_t)0xf619, (q15_t)0x7f9b, (q15_t)0xf600, (q15_t)0x7f99, (q15_t)0xf5e7,
  (q15_t)0x7f97, (q15_t)0xf5cd, (q15_t)0x7f95, (q15_t)0xf5b4, (q15_t)0x7f93, (q15_t)0xf59b, (q15_t)0x7f91, (q15_t)0xf582,
  (q15_t)0x7f8f, (q15_t)0xf569, (q15_t)0x7f8d, (q15_t)0xf550, (q15_t)0x7f8b, (q15_t)0xf537, (q15_t)0x7f89, (q15_t)0xf51e,
  (q15_t)0x7f87, (q15_t)0xf505, (q15_t)0x7f85, (q15_t)0xf4ec, (q15_t)0x7f82, (q15_t)0xf4d3, (q15_t)0x7f80, (q15_t)0xf4ba,
  (q15_t)0x7f7e, (q15_t)0xf4a1, (q15_t)0x7f7c, (q15_t)0xf488, (q15_t)0x7f79, (q15_t)0xf46f, (q15_t)0x7f77, (q15_t)0xf456,
  (q15_t)0x7f75, (q15_t)0xf43d, (q15_t)0x7f72, (q15_t)0xf424, (q15_t)0x7f70, (q15_t)0xf40b, (q15_t)0x7f6e, (q15_t)0xf3f2,
  (q15_t)0x7f6b, (q15_t)0xf3d9, (q15_t)0x7f69, (q15_t)0xf3c0, (q15_t)0x7f67, (q15_t)0xf3a7, (q15_t)0x7f64, (q15_t)0xf38e,
  (q15_t)0x7f62, (q15_t)0xf375, (q15_t)0x7f5f, (q15_t)0xf35c, (q15_t)0x7f5d, (q15_t)0xf343, (q15_t)0x7f5a, (q15_t)0xf32a,
  (q15_t)0x7f58, (q15_t)0xf311, (q15_t)0x7f55, (q15_t)0xf2f8, (q15_t)0x7f53, (q15_t)0xf2df, (q15_t)0x7f50, (q15_t)0xf2c6,
  (q15_t)0x7f4d, (q15_t)0xf2ad, (q15_t)0x7f4b, (q15_t)0xf294, (q15_t)0x7f48, (q15_t)0xf27b, (q15_t)0x7f45, (q15_t)0xf262,
  (q15_t)0x7f43, (q15_t)0xf249, (q15_t)0x7f40, (q15_t)0xf230, (q15_t)0x7f3d, (q15_t)0xf217, (q15_t)0x7f3b, (q15_t)0xf1fe,
  (q15_t)0x7f38, (q15_t)0xf1e5, (q15_t)0x7f35, (q15_t)0xf1cc, (q15_t)0x7f32, (q15_t)0xf1b3, (q15_t)0x7f2f, (q15_t)0xf19a,
  (q15_t)0x7f2d, (q15_t)0xf181, (q15_t)0x7f2a, (q15_t)0xf168, (q15_t)0x7f27, (q15_t)0xf14f, (q15_t)0x7f24, (q15_t)0xf136,
  (q15_t)0x7f21, (q15_t)0xf11d, (q15_t)0x7f1e, (q15_t)0xf104, (q15_t)0x7f1b, (q15_t)0xf0eb, (q15_t)0x7f18, (q15_t)0xf0d2,
  (q15_t)0x7f15, (q15_t)0xf0b9, (q15_t)0x7f12, (q15_t)0xf0a0, (q15_t)0x7f0f, (q15_t)0xf087, (q15_t)0x7f0c, (q15_t)0xf06e,
  (q15_t)0x7f09, (q15_t)0xf055, (q15_t)0x7f06, (q15_t)0xf03c, (q15_t)0x7f03, (q15_t)0xf023, (q15_t)0x7f00, (q15_t)0xf00b,
  (q15_t)0x7efd, (q15_t)0xeff2, (q15_t)0x7ef9, (q15_t)0xefd9, (q15_t)0x7ef6, (q15_t)0xefc0, (q15_t)0x7ef3, (q15_t)0xefa7,
  (q15_t)0x7ef0, (q15_t)0xef8e, (q15_t)0x7eed, (q15_t)0xef75, (q15_t)0x7ee9, (q15_t)0xef5c, (q15_t)0x7ee6, (q15_t)0xef43,
  (q15_t)0x7ee3, (q15_t)0xef2a, (q15_t)0x7edf, (q15_t)0xef11, (q15_t)0x7edc, (q15_t)0xeef8, (q15_t)0x7ed9, (q15_t)0xeedf,
  (q15_t)0x7ed5, (q15_t)0xeec7, (q15_t)0x7ed2, (q15_t)0xeeae, (q15_t)0x7ecf, (q15_t)0xee95, (q15_t)0x7ecb, (q15_t)0xee7c,
  (q15_t)0x7ec8, (q15_t)0xee63, (q15_t)0x7ec4, (q15_t)0xee4a, (q15_t)0x7ec1, (q15_t)0xee31, (q15_t)0x7ebd, (q15_t)0xee18,
  (q15_t)0x7eba, (q15_t)0xedff, (q15_t)0x7eb6, (q15_t)0xede7, (q15_t)0x7eb3, (q15_t)0xedce, (q15_t)0x7eaf, (q15_t)0xedb5,
  (q15_t)0x7eab, (q15_t)0xed9c, (q15_t)0x7ea8, (q15_t)0xed83, (q15_t)0x7ea4, (q15_t)0xed6a, (q15_t)0x7ea1, (q15_t)0xed51,
  (q15_t)0x7e9d, (q15_t)0xed38, (q15_t)0x7e99, (q15_t)0xed20, (q15_t)0x7e95, (q15_t)0xed07, (q15_t)0x7e92, (q15_t)0xecee,
  (q15_t)0x7e8e, (q15_t)0xecd5, (q15_t)0x7e8a, (q15_t)0xecbc, (q15_t)0x7e86, (q15_t)0xeca3, (q15_t)0x7e83, (q15_t)0xec8a,
  (q15_t)0x7e7f, (q15_t)0xec72, (q15_t)0x7e7b, (q15_t)0xec59, (q15_t)0x7e77, (q15_t)0xec40, (q15_t)0x7e73, (q15_t)0xec27,
  (q15_t)0x7e6f, (q15_t)0xec0e, (q15_t)0x7e6b, (q15_t)0xebf5, (q15_t)0x7e67, (q15_t)0xebdd, (q15_t)0x7e63, (q15_t)0xebc4,
  (q15_t)0x7e5f, (q15_t)0xebab, (q15_t)0x7e5b, (q15_t)0xeb92, (q15_t)0x7e57, (q15_t)0xeb79, (q15_t)0x7e53, (q15_t)0xeb61,
  (q15_t)0x7e4f, (q15_t)0xeb48, (q15_t)0x7e4b, (q15_t)0xeb2f, (q15_t)0x7e47, (q15_t)0xeb16, (q15_t)0x7e43, (q15_t)0xeafd,
  (q15_t)0x7e3f, (q15_t)0xeae5, (q15_t)0x7e3b, (q15_t)0xeacc, (q15_t)0x7e37, (q15_t)0xeab3, (q15_t)0x7e32, (q15_t)0xea9a,
  (q15_t)0x7e2e, (q15_t)0xea81, (q15_t)0x7e2a, (q15_t)0xea69, (q15_t)0x7e26, (q15_t)0xea50, (q15_t)0x7e21, (q15_t)0xea37,
  (q15_t)0x7e1d, (q15_t)0xea1e, (q15_t)0x7e19, (q15_t)0xea06, (q15_t)0x7e14, (q15_t)0xe9ed, (q15_t)0x7e10, (q15_t)0xe9d4,
  (q15_t)0x7e0c, (q15_t)0xe9bb, (q15_t)0x7e07, (q15_t)0xe9a3, (q15_t)0x7e03, (q15_t)0xe98a, (q15_t)0x7dff, (q15_t)0xe971,
  (q15_t)0x7dfa, (q15_t)0xe958, (q15_t)0x7df6, (q15_t)0xe940, (q15_t)0x7df1, (q15_t)0xe927, (q15_t)0x7ded, (q15_t)0xe90e,
  (q15_t)0x7de8, (q15_t)0xe8f6, (q15_t)0x7de4, (q15_t)0xe8dd, (q15_t)0x7ddf, (q15_t)0xe8c4, (q15_t)0x7dda, (q15_t)0xe8ab,
  (q15_t)0x7dd6, (q15_t)0xe893, (q15_t)0x7dd1, (q15_t)0xe87a, (q15_t)0x7dcd, (q15_t)0xe861, (q15_t)0x7dc8, (q15_t)0xe849,
  (q15_t)0x7dc3, (q15_t)0xe830, (q15_t)0x7dbf, (q15_t)0xe817, (q15_t)0x7dba, (q15_t)0xe7fe, (q15_t)0x7db5, (q15_t)0xe7e6,
  (q15_t)0x7db0, (q15_t)0xe7cd, (q15_t)0x7dac, (q15_t)0xe7b4, (q15_t)0x7da7, (q15_t)0xe79c, (q15_t)0x7da2, (q15_t)0xe783,
  (q15_t)0x7d9d, (q15_t)0xe76a, (q15_t)0x7d98, (q15_t)0xe752, (q15_t)0x7d94, (q15_t)0xe739, (q15_t)0x7d8f, (q15_t)0xe720,
  (q15_t)0x7d8a, (q15_t)0xe708, (q15_t)0x7d85, (q15_t)0xe6ef, (q15_t)0x7d80, (q15_t)0xe6d6, (q15_t)0x7d7b, (q15_t)0xe6be,
  (q15_t)0x7d76, (q15_t)0xe6a5, (q15_t)0x7d71, (q15_t)0xe68d, (q15_t)0x7d6c, (q15_t)0xe674, (q15_t)0x7d67, (q15_t)0xe65b,
  (q15_t)0x7d62, (q15_t)0xe643, (q15_t)0x7d5d, (q15_t)0xe62a, (q15_t)0x7d58, (q15_t)0xe611, (q15_t)0x7d53, (q15_t)0xe5f9,
  (q15_t)0x7d4e, (q15_t)0xe5e0, (q15_t)0x7d49, (q15_t)0xe5c8, (q15_t)0x7d43, (q15_t)0xe5af, (q15_t)0x7d3e, (q15_t)0xe596,
  (q15_t)0x7d39, (q15_t)0xe57e, (q15_t)0x7d34, (q15_t)0xe565, (q15_t)0x7d2f, (q15_t)0xe54d, (q15_t)0x7d29, (q15_t)0xe534,
  (q15_t)0x7d24, (q15_t)0xe51c, (q15_t)0x7d1f, (q15_t)0xe503, (q15_t)0x7d19, (q15_t)0xe4ea, (q15_t)0x7d14, (q15_t)0xe4d2,
  (q15_t)0x7d0f, (q15_t)0xe4b9, (q15_t)0x7d09, (q15_t)0xe4a1, (q15_t)0x7d04, (q15_t)0xe488, (q15_t)0x7cff, (q15_t)0xe470,
  (q15_t)0x7cf9, (q15_t)0xe457, (q15_t)0x7cf4, (q15_t)0xe43f, (q15_t)0x7cee, (q15_t)0xe426, (q15_t)0x7ce9, (q15_t)0xe40e,
  (q15_t)0x7ce3, (q15_t)0xe3f5, (q15_t)0x7cde, (q15_t)0xe3dc, (q15_t)0x7cd8, (q15_t)0xe3c4, (q15_t)0x7cd3, (q15_t)0xe3ab,
  (q15_t)0x7ccd, (q15_t)0xe393, (q15_t)0x7cc8, (q15_t)0xe37a, (q15_t)0x7cc2, (q15_t)0xe362, (q15_t)0x7cbc, (q15_t)0xe349,
  (q15_t)0x7cb7, (q15_t)0xe331, (q15_t)0x7cb1, (q15_t)0xe318, (q15_t)0x7cab, (q15_t)0xe300, (q15_t)0x7ca6, (q15_t)0xe2e8,
  (q15_t)0x7ca0, (q15_t)0xe2cf, (q15_t)0x7c9a, (q15_t)0xe2b7, (q15_t)0x7c94, (q15_t)0xe29e, (q15_t)0x7c8f, (q15_t)0xe286,
  (q15_t)0x7c89, (q15_t)0xe26d, (q15_t)0x7c83, (q15_t)0xe255, (q15_t)0x7c7d, (q15_t)0xe23c, (q15_t)0x7c77, (q15_t)0xe224,
  (q15_t)0x7c71, (q15_t)0xe20b, (q15_t)0x7c6c, (q15_t)0xe1f3, (q15_t)0x7c66, (q15_t)0xe1db, (q15_t)0x7c60, (q15_t)0xe1c2,
  (q15_t)0x7c5a, (q15_t)0xe1aa, (q15_t)0x7c54, (q15_t)0xe191, (q15_t)0x7c4e, (q15_t)0xe179, (q15_t)0x7c48, (q15_t)0xe160,
  (q15_t)0x7c42, (q15_t)0xe148, (q15_t)0x7c3c, (q15_t)0xe130, (q15_t)0x7c36, (q15_t)0xe117, (q15_t)0x7c30, (q15_t)0xe0ff,
  (q15_t)0x7c29, (q15_t)0xe0e7, (q15_t)0x7c23, (q15_t)0xe0ce, (q15_t)0x7c1d, (q15_t)0xe0b6, (q15_t)0x7c17, (q15_t)0xe09d,
  (q15_t)0x7c11, (q15_t)0xe085, (q15_t)0x7c0b, (q15_t)0xe06d, (q15_t)0x7c05, (q15_t)0xe054, (q15_t)0x7bfe, (q15_t)0xe03c,
  (q15_t)0x7bf8, (q15_t)0xe024, (q15_t)0x7bf2, (q15_t)0xe00b, (q15_t)0x7beb, (q15_t)0xdff3, (q15_t)0x7be5, (q15_t)0xdfdb,
  (q15_t)0x7bdf, (q15_t)0xdfc2, (q15_t)0x7bd9, (q15_t)0xdfaa, (q15_t)0x7bd2, (q15_t)0xdf92, (q15_t)0x7bcc, (q15_t)0xdf79,
  (q15_t)0x7bc5, (q15_t)0xdf61, (q15_t)0x7bbf, (q15_t)0xdf49, (q15_t)0x7bb9, (q15_t)0xdf30, (q15_t)0x7bb2, (q15_t)0xdf18,
  (q15_t)0x7bac, (q15_t)0xdf00, (q15_t)0x7ba5, (q15_t)0xdee8, (q15_t)0x7b9f, (q15_t)0xdecf, (q15_t)0x7b98, (q15_t)0xdeb7,
  (q15_t)0x7b92, (q15_t)0xde9f, (q15_t)0x7b8b, (q15_t)0xde87, (q15_t)0x7b84, (q15_t)0xde6e, (q15_t)0x7b7e, (q15_t)0xde56,
  (q15_t)0x7b77, (q15_t)0xde3e, (q15_t)0x7b71, (q15_t)0xde26, (q15_t)0x7b6a, (q15_t)0xde0d, (q15_t)0x7b63, (q15_t)0xddf5,
  (q15_t)0x7b5d, (q15_t)0xdddd, (q15_t)0x7b56, (q15_t)0xddc5, (q15_t)0x7b4f, (q15_t)0xddac, (q15_t)0x7b48, (q15_t)0xdd94,
  (q15_t)0x7b42, (q15_t)0xdd7c, (q15_t)0x7b3b, (q15_t)0xdd64, (q15_t)0x7b34, (q15_t)0xdd4c, (q15_t)0x7b2d, (q15_t)0xdd33,
  (q15_t)0x7b26, (q15_t)0xdd1b, (q15_t)0x7b1f, (q15_t)0xdd03, (q15_t)0x7b19, (q15_t)0xdceb, (q15_t)0x7b12, (q15_t)0xdcd3,
  (q15_t)0x7b0b, (q15_t)0xdcbb, (q15_t)0x7b04, (q15_t)0xdca2, (q15_t)0x7afd, (q15_t)0xdc8a, (q15_t)0x7af6, (q15_t)0xdc72,
  (q15_t)0x7aef, (q15_t)0xdc5a, (q15_t)0x7ae8, (q15_t)0xdc42, (q15_t)0x7ae1, (q15_t)0xdc2a, (q15_t)0x7ada, (q15_t)0xdc12,
  (q15_t)0x7ad3, (q15_t)0xdbf9, (q15_t)0x7acc, (q15_t)0xdbe1, (q15_t)0x7ac5, (q15_t)0xdbc9, (q15_t)0x7abd, (q15_t)0xdbb1,
  (q15_t)0x7ab6, (q15_t)0xdb99, (q15_t)0x7aaf, (q15_t)0xdb81, (q15_t)0x7aa8, (q15_t)0xdb69, (q15_t)0x7aa1, (q15_t)0xdb51,
  (q15_t)0x7a9a, (q15_t)0xdb39, (q15_t)0x7a92, (q15_t)0xdb21, (q15_t)0x7a8b, (q15_t)0xdb09, (q15_t)0x7a84, (q15_t)0xdaf1,
  (q15_t)0x7a7d, (q15_t)0xdad8, (q15_t)0x7a75, (q15_t)0xdac0, (q15_t)0x7a6e, (q15_t)0xdaa8, (q15_t)0x7a67, (q15_t)0xda90,
  (q15_t)0x7a5f, (q15_t)0xda78, (q15_t)0x7a58, (q15_t)0xda60, (q15_t)0x7a50, (q15_t)0xda48, (q15_t)0x7a49, (q15_t)0xda30,
  (q15_t)0x7a42, (q15_t)0xda18, (q15_t)0x7a3a, (q15_t)0xda00, (q15_t)0x7a33, (q15_t)0xd9e8, (q15_t)0x7a2b, (q15_t)0xd9d0,
  (q15_t)0x7a24, (q15_t)0xd9b8, (q15_t)0x7a1c, (q15_t)0xd9a0, (q15_t)0x7a15, (q15_t)0xd988, (q15_t)0x7a0d, (q15_t)0xd970,
  (q15_t)0x7a05, (q15_t)0xd958, (q15_t)0x79fe, (q15_t)0xd940, (q15_t)0x79f6, (q15_t)0xd928, (q15_t)0x79ef, (q15_t)0xd911,
  (q15_t)0x79e7, (q15_t)0xd8f9, (q15_t)0x79df, (q15_t)0xd8e1, (q15_t)0x79d8, (q15_t)0xd8c9, (q15_t)0x79d0, (q15_t)0xd8b1,
  (q15_t)0x79c8, (q15_t)0xd899, (q15_t)0x79c0, (q15_t)0xd881, (q15_t)0x79b9, (q15_t)0xd869, (q15_t)0x79b1, (q15_t)0xd851,
  (q15_t)0x79a9, (q15_t)0xd839, (q15_t)0x79a1, (q15_t)0xd821, (q15_t)0x7999, (q15_t)0xd80a, (q15_t)0x7992, (q15_t)0xd7f2,
  (q15_t)0x798a, (q15_t)0xd7da, (q15_t)0x7982, (q15_t)0xd7c2, (q15_t)0x797a, (q15_t)0xd7aa, (q15_t)0x7972, (q15_t)0xd792,
  (q15_t)0x796a, (q15_t)0xd77a, (q15_t)0x7962, (q15_t)0xd763, (q15_t)0x795a, (q15_t)0xd74b, (q15_t)0x7952, (q15_t)0xd733,
  (q15_t)0x794a, (q15_t)0xd71b, (q15_t)0x7942, (q15_t)0xd703, (q15_t)0x793a, (q15_t)0xd6eb, (q15_t)0x7932, (q15_t)0xd6d4,
  (q15_t)0x792a, (q15_t)0xd6bc, (q15_t)0x7922, (q15_t)0xd6a4, (q15_t)0x7919, (q15_t)0xd68c, (q15_t)0x7911, (q15_t)0xd675,
  (q15_t)0x7909, (q15_t)0xd65d, (q15_t)0x7901, (q15_t)0xd645, (q15_t)0x78f9, (q15_t)0xd62d, (q15_t)0x78f1, (q15_t)0xd615,
  (q15_t)0x78e8, (q15_t)0xd5fe, (q15_t)0x78e0, (q15_t)0xd5e6, (q15_t)0x78d8, (q15_t)0xd5ce, (q15_t)0x78cf, (q15_t)0xd5b7,
  (q15_t)0x78c7, (q15_t)0xd59f, (q15_t)0x78bf, (q15_t)0xd587, (q15_t)0x78b6, (q15_t)0xd56f, (q15_t)0x78ae, (q15_t)0xd558,
  (q15_t)0x78a6, (q15_t)0xd540, (q15_t)0x789d, (q15_t)0xd528, (q15_t)0x7895, (q15_t)0xd511, (q15_t)0x788c, (q15_t)0xd4f9,
  (q15_t)0x7884, (q15_t)0xd4e1, (q15_t)0x787c, (q15_t)0xd4ca, (q15_t)0x7873, (q15_t)0xd4b2, (q15_t)0x786b, (q15_t)0xd49a,
  (q15_t)0x7862, (q15_t)0xd483, (q15_t)0x7859, (q15_t)0xd46b, (q15_t)0x7851, (q15_t)0xd453, (q15_t)0x7848, (q15_t)0xd43c,
  (q15_t)0x7840, (q15_t)0xd424, (q15_t)0x7837, (q15_t)0xd40d, (q15_t)0x782e, (q15_t)0xd3f5, (q15_t)0x7826, (q15_t)0xd3dd,
  (q15_t)0x781d, (q15_t)0xd3c6, (q15_t)0x7814, (q15_t)0xd3ae, (q15_t)0x780c, (q15_t)0xd397, (q15_t)0x7803, (q15_t)0xd37f,
  (q15_t)0x77fa, (q15_t)0xd368, (q15_t)0x77f1, (q15_t)0xd350, (q15_t)0x77e9, (q15_t)0xd338, (q15_t)0x77e0, (q15_t)0xd321,
  (q15_t)0x77d7, (q15_t)0xd309, (q15_t)0x77ce, (q15_t)0xd2f2, (q15_t)0x77c5, (q15_t)0xd2da, (q15_t)0x77bc, (q15_t)0xd2c3,
  (q15_t)0x77b4, (q15_t)0xd2ab, (q15_t)0x77ab, (q15_t)0xd294, (q15_t)0x77a2, (q15_t)0xd27c, (q15_t)0x7799, (q15_t)0xd265,
  (q15_t)0x7790, (q15_t)0xd24d, (q15_t)0x7787, (q15_t)0xd236, (q15_t)0x777e, (q15_t)0xd21e, (q15_t)0x7775, (q15_t)0xd207,
  (q15_t)0x776c, (q15_t)0xd1ef, (q15_t)0x7763, (q15_t)0xd1d8, (q15_t)0x775a, (q15_t)0xd1c1, (q15_t)0x7751, (q15_t)0xd1a9,
  (q15_t)0x7747, (q15_t)0xd192, (q15_t)0x773e, (q15_t)0xd17a, (q15_t)0x7735, (q15_t)0xd163, (q15_t)0x772c, (q15_t)0xd14b,
  (q15_t)0x7723, (q15_t)0xd134, (q15_t)0x771a, (q15_t)0xd11d, (q15_t)0x7710, (q15_t)0xd105, (q15_t)0x7707, (q15_t)0xd0ee,
  (q15_t)0x76fe, (q15_t)0xd0d7, (q15_t)0x76f5, (q15_t)0xd0bf, (q15_t)0x76eb, (q15_t)0xd0a8, (q15_t)0x76e2, (q15_t)0xd091,
  (q15_t)0x76d9, (q15_t)0xd079, (q15_t)0x76cf, (q15_t)0xd062, (q15_t)0x76c6, (q15_t)0xd04b, (q15_t)0x76bd, (q15_t)0xd033,
  (q15_t)0x76b3, (q15_t)0xd01c, (q15_t)0x76aa, (q15_t)0xd005, (q15_t)0x76a0, (q15_t)0xcfed, (q15_t)0x7697, (q15_t)0xcfd6,
  (q15_t)0x768e, (q15_t)0xcfbf, (q15_t)0x7684, (q15_t)0xcfa7, (q15_t)0x767b, (q15_t)0xcf90, (q15_t)0x7671, (q15_t)0xcf79,
  (q15_t)0x7668, (q15_t)0xcf62, (q15_t)0x765e, (q15_t)0xcf4a, (q15_t)0x7654, (q15_t)0xcf33, (q15_t)0x764b, (q15_t)0xcf1c,
  (q15_t)0x7641, (q15_t)0xcf05, (q15_t)0x7638, (q15_t)0xceee, (q15_t)0x762e, (q15_t)0xced6, (q15_t)0x7624, (q15_t)0xcebf,
  (q15_t)0x761b, (q15_t)0xcea8, (q15_t)0x7611, (q15_t)0xce91, (q15_t)0x7607, (q15_t)0xce7a, (q15_t)0x75fd, (q15_t)0xce62,
  (q15_t)0x75f4, (q15_t)0xce4b, (q15_t)0x75ea, (q15_t)0xce34, (q15_t)0x75e0, (q15_t)0xce1d, (q15_t)0x75d6, (q15_t)0xce06,
  (q15_t)0x75cc, (q15_t)0xcdef, (q15_t)0x75c3, (q15_t)0xcdd8, (q15_t)0x75b9, (q15_t)0xcdc0, (q15_t)0x75af, (q15_t)0xcda9,
  (q15_t)0x75a5, (q15_t)0xcd92, (q15_t)0x759b, (q15_t)0xcd7b, (q15_t)0x7591, (q15_t)0xcd64, (q15_t)0x7587, (q15_t)0xcd4d,
  (q15_t)0x757d, (q15_t)0xcd36, (q15_t)0x7573, (q15_t)0xcd1f, (q15_t)0x7569, (q15_t)0xcd08, (q15_t)0x755f, (q15_t)0xccf1,
  (q15_t)0x7555, (q15_t)0xccda, (q15_t)0x754b, (q15_t)0xccc3, (q15_t)0x7541, (q15_t)0xccac, (q15_t)0x7537, (q15_t)0xcc95,
  (q15_t)0x752d, (q15_t)0xcc7e, (q15_t)0x7523, (q15_t)0xcc67, (q15_t)0x7519, (q15_t)0xcc50, (q15_t)0x750f, (q15_t)0xcc39,
  (q15_t)0x7504, (q15_t)0xcc22, (q15_t)0x74fa, (q15_t)0xcc0b, (q15_t)0x74f0, (q15_t)0xcbf4, (q15_t)0x74e6, (q15_t)0xcbdd,
  (q15_t)0x74db, (q15_t)0xcbc6, (q15_t)0x74d1, (q15_t)0xcbaf, (q15_t)0x74c7, (q15_t)0xcb98, (q15_t)0x74bd, (q15_t)0xcb81,
  (q15_t)0x74b2, (q15_t)0xcb6a, (q15_t)0x74a8, (q15_t)0xcb53, (q15_t)0x749e, (q15_t)0xcb3c, (q15_t)0x7493, (q15_t)0xcb25,
  (q15_t)0x7489, (q15_t)0xcb0e, (q15_t)0x747e, (q15_t)0xcaf8, (q15_t)0x7474, (q15_t)0xcae1, (q15_t)0x746a, (q15_t)0xcaca,
  (q15_t)0x745f, (q15_t)0xcab3, (q15_t)0x7455, (q15_t)0xca9c, (q15_t)0x744a, (q15_t)0xca85, (q15_t)0x7440, (q15_t)0xca6e,
  (q15_t)0x7435, (q15_t)0xca58, (q15_t)0x742b, (q15_t)0xca41, (q15_t)0x7420, (q15_t)0xca2a, (q15_t)0x7415, (q15_t)0xca13,
  (q15_t)0x740b, (q15_t)0xc9fc, (q15_t)0x7400, (q15_t)0xc9e6, (q15_t)0x73f6, (q15_t)0xc9cf, (q15_t)0x73eb, (q15_t)0xc9b8,
  (q15_t)0x73e0, (q15_t)0xc9a1, (q15_t)0x73d6, (q15_t)0xc98b, (q15_t)0x73cb, (q15_t)0xc974, (q15_t)0x73c0, (q15_t)0xc95d,
  (q15_t)0x73b5, (q15_t)0xc946, (q15_t)0x73ab, (q15_t)0xc930, (q15_t)0x73a0, (q15_t)0xc919, (q15_t)0x7395, (q15_t)0xc902,
  (q15_t)0x738a, (q15_t)0xc8ec, (q15_t)0x737f, (q15_t)0xc8d5, (q15_t)0x7375, (q15_t)0xc8be, (q15_t)0x736a, (q15_t)0xc8a8,
  (q15_t)0x735f, (q15_t)0xc891, (q15_t)0x7354, (q15_t)0xc87a, (q15_t)0x7349, (q15_t)0xc864, (q15_t)0x733e, (q15_t)0xc84d,
  (q15_t)0x7333, (q15_t)0xc836, (q15_t)0x7328, (q15_t)0xc820, (q15_t)0x731d, (q15_t)0xc809, (q15_t)0x7312, (q15_t)0xc7f3,
  (q15_t)0x7307, (q15_t)0xc7dc, (q15_t)0x72fc, (q15_t)0xc7c5, (q15_t)0x72f1, (q15_t)0xc7af, (q15_t)0x72e6, (q15_t)0xc798,
  (q15_t)0x72db, (q15_t)0xc782, (q15_t)0x72d0, (q15_t)0xc76b, (q15_t)0x72c5, (q15_t)0xc755, (q15_t)0x72ba, (q15_t)0xc73e,
  (q15_t)0x72af, (q15_t)0xc728, (q15_t)0x72a3, (q15_t)0xc711, (q15_t)0x7298, (q15_t)0xc6fa, (q15_t)0x728d, (q15_t)0xc6e4,
  (q15_t)0x7282, (q15_t)0xc6ce, (q15_t)0x7276, (q15_t)0xc6b7, (q15_t)0x726b, (q15_t)0xc6a1, (q15_t)0x7260, (q15_t)0xc68a,
  (q15_t)0x7255, (q15_t)0xc674, (q15_t)0x7249, (q15_t)0xc65d, (q15_t)0x723e, (q15_t)0xc647, (q15_t)0x7233, (q15_t)0xc630,
  (q15_t)0x7227, (q15_t)0xc61a, (q15_t)0x721c, (q15_t)0xc603, (q15_t)0x7211, (q15_t)0xc5ed, (q15_t)0x7205, (q15_t)0xc5d7,
  (q15_t)0x71fa, (q15_t)0xc5c0, (q15_t)0x71ee, (q15_t)0xc5aa, (q15_t)0x71e3, (q15_t)0xc594, (q15_t)0x71d7, (q15_t)0xc57d,
  (q15_t)0x71cc, (q15_t)0xc567, (q15_t)0x71c0, (q15_t)0xc551, (q15_t)0x71b5, (q15_t)0xc53a, (q15_t)0x71a9, (q15_t)0xc524,
  (q15_t)0x719e, (q15_t)0xc50e, (q15_t)0x7192, (q15_t)0xc4f7, (q15_t)0x7186, (q15_t)0xc4e1, (q15_t)0x717b, (q15_t)0xc4cb,
  (q15_t)0x716f, (q15_t)0xc4b4, (q15_t)0x7164, (q15_t)0xc49e, (q15_t)0x7158, (q15_t)0xc488, (q15_t)0x714c, (q15_t)0xc472,
  (q15_t)0x7141, (q15_t)0xc45b, (q15_t)0x7135, (q15_t)0xc445, (q15_t)0x7129, (q15_t)0xc42f, (q15_t)0x711d, (q15_t)0xc419,
  (q15_t)0x7112, (q15_t)0xc403, (q15_t)0x7106, (q15_t)0xc3ec, (q15_t)0x70fa, (q15_t)0xc3d6, (q15_t)0x70ee, (q15_t)0xc3c0,
  (q15_t)0x70e2, (q15_t)0xc3aa, (q15_t)0x70d6, (q15_t)0xc394, (q15_t)0x70cb, (q15_t)0xc37d, (q15_t)0x70bf, (q15_t)0xc367,
  (q15_t)0x70b3, (q15_t)0xc351, (q15_t)0x70a7, (q15_t)0xc33b, (q15_t)0x709b, (q15_t)0xc325, (q15_t)0x708f, (q15_t)0xc30f,
  (q15_t)0x7083, (q15_t)0xc2f9, (q15_t)0x7077, (q15_t)0xc2e3, (q15_t)0x706b, (q15_t)0xc2cd, (q15_t)0x705f, (q15_t)0xc2b7,
  (q15_t)0x7053, (q15_t)0xc2a0, (q15_t)0x7047, (q15_t)0xc28a, (q15_t)0x703b, (q15_t)0xc274, (q15_t)0x702f, (q15_t)0xc25e,
  (q15_t)0x7023, (q15_t)0xc248, (q15_t)0x7016, (q15_t)0xc232, (q15_t)0x700a, (q15_t)0xc21c, (q15_t)0x6ffe, (q15_t)0xc206,
  (q15_t)0x6ff2, (q15_t)0xc1f0, (q15_t)0x6fe6, (q15_t)0xc1da, (q15_t)0x6fda, (q15_t)0xc1c4, (q15_t)0x6fcd, (q15_t)0xc1ae,
  (q15_t)0x6fc1, (q15_t)0xc198, (q15_t)0x6fb5, (q15_t)0xc183, (q15_t)0x6fa9, (q15_t)0xc16d, (q15_t)0x6f9c, (q15_t)0xc157,
  (q15_t)0x6f90, (q15_t)0xc141, (q15_t)0x6f84, (q15_t)0xc12b, (q15_t)0x6f77, (q15_t)0xc115, (q15_t)0x6f6b, (q15_t)0xc0ff,
  (q15_t)0x6f5f, (q15_t)0xc0e9, (q15_t)0x6f52, (q15_t)0xc0d3, (q15_t)0x6f46, (q15_t)0xc0bd, (q15_t)0x6f39, (q15_t)0xc0a8,
  (q15_t)0x6f2d, (q15_t)0xc092, (q15_t)0x6f20, (q15_t)0xc07c, (q15_t)0x6f14, (q15_t)0xc066, (q15_t)0x6f07, (q15_t)0xc050,
  (q15_t)0x6efb, (q15_t)0xc03b, (q15_t)0x6eee, (q15_t)0xc025, (q15_t)0x6ee2, (q15_t)0xc00f, (q15_t)0x6ed5, (q15_t)0xbff9,
  (q15_t)0x6ec9, (q15_t)0xbfe3, (q15_t)0x6ebc, (q15_t)0xbfce, (q15_t)0x6eaf, (q15_t)0xbfb8, (q15_t)0x6ea3, (q15_t)0xbfa2,
  (q15_t)0x6e96, (q15_t)0xbf8d, (q15_t)0x6e89, (q15_t)0xbf77, (q15_t)0x6e7d, (q15_t)0xbf61, (q15_t)0x6e70, (q15_t)0xbf4b,
  (q15_t)0x6e63, (q15_t)0xbf36, (q15_t)0x6e57, (q15_t)0xbf20, (q15_t)0x6e4a, (q15_t)0xbf0a, (q15_t)0x6e3d, (q15_t)0xbef5,
  (q15_t)0x6e30, (q15_t)0xbedf, (q15_t)0x6e24, (q15_t)0xbeca, (q15_t)0x6e17, (q15_t)0xbeb4, (q15_t)0x6e0a, (q15_t)0xbe9e,
  (q15_t)0x6dfd, (q15_t)0xbe89, (q15_t)0x6df0, (q15_t)0xbe73, (q15_t)0x6de3, (q15_t)0xbe5e, (q15_t)0x6dd6, (q15_t)0xbe48,
  (q15_t)0x6dca, (q15_t)0xbe32, (q15_t)0x6dbd, (q15_t)0xbe1d, (q15_t)0x6db0, (q15_t)0xbe07, (q15_t)0x6da3, (q15_t)0xbdf2,
  (q15_t)0x6d96, (q15_t)0xbddc, (q15_t)0x6d89, (q15_t)0xbdc7, (q15_t)0x6d7c, (q15_t)0xbdb1, (q15_t)0x6d6f, (q15_t)0xbd9c,
  (q15_t)0x6d62, (q15_t)0xbd86, (q15_t)0x6d55, (q15_t)0xbd71, (q15_t)0x6d48, (q15_t)0xbd5b, (q15_t)0x6d3a, (q15_t)0xbd46,
  (q15_t)0x6d2d, (q15_t)0xbd30, (q15_t)0x6d20, (q15_t)0xbd1b, (q15_t)0x6d13, (q15_t)0xbd06, (q15_t)0x6d06, (q15_t)0xbcf0,
  (q15_t)0x6cf9, (q15_t)0xbcdb, (q15_t)0x6cec, (q15_t)0xbcc5, (q15_t)0x6cde, (q15_t)0xbcb0, (q15_t)0x6cd1, (q15_t)0xbc9b,
  (q15_t)0x6cc4, (q15_t)0xbc85, (q15_t)0x6cb7, (q15_t)0xbc70, (q15_t)0x6ca9, (q15_t)0xbc5b, (q15_t)0x6c9c, (q15_t)0xbc45,
  (q15_t)0x6c8f, (q15_t)0xbc30, (q15_t)0x6c81, (q15_t)0xbc1b, (q15_t)0x6c74, (q15_t)0xbc05, (q15_t)0x6c67, (q15_t)0xbbf0,
  (q15_t)0x6c59, (q15_t)0xbbdb, (q15_t)0x6c4c, (q15_t)0xbbc5, (q15_t)0x6c3f, (q15_t)0xbbb0, (q15_t)0x6c31, (q15_t)0xbb9b,
  (q15_t)0x6c24, (q15_t)0xbb86, (q15_t)0x6c16, (q15_t)0xbb70, (q15_t)0x6c09, (q15_t)0xbb5b, (q15_t)0x6bfb, (q15_t)0xbb46,
  (q15_t)0x6bee, (q15_t)0xbb31, (q15_t)0x6be0, (q15_t)0xbb1c, (q15_t)0x6bd3, (q15_t)0xbb06, (q15_t)0x6bc5, (q15_t)0xbaf1,
  (q15_t)0x6bb8, (q15_t)0xbadc, (q15_t)0x6baa, (q15_t)0xbac7, (q15_t)0x6b9c, (q15_t)0xbab2, (q15_t)0x6b8f, (q15_t)0xba9d,
  (q15_t)0x6b81, (q15_t)0xba88, (q15_t)0x6b73, (q15_t)0xba73, (q15_t)0x6b66, (q15_t)0xba5d, (q15_t)0x6b58, (q15_t)0xba48,
  (q15_t)0x6b4a, (q15_t)0xba33, (q15_t)0x6b3d, (q15_t)0xba1e, (q15_t)0x6b2f, (q15_t)0xba09, (q15_t)0x6b21, (q15_t)0xb9f4,
  (q15_t)0x6b13, (q15_t)0xb9df, (q15_t)0x6b06, (q15_t)0xb9ca, (q15_t)0x6af8, (q15_t)0xb9b5, (q15_t)0x6aea, (q15_t)0xb9a0,
  (q15_t)0x6adc, (q15_t)0xb98b, (q15_t)0x6ace, (q15_t)0xb976, (q15_t)0x6ac1, (q15_t)0xb961, (q15_t)0x6ab3, (q15_t)0xb94c,
  (q15_t)0x6aa5, (q15_t)0xb937, (q15_t)0x6a97, (q15_t)0xb922, (q15_t)0x6a89, (q15_t)0xb90d, (q15_t)0x6a7b, (q15_t)0xb8f8,
  (q15_t)0x6a6d, (q15_t)0xb8e4, (q15_t)0x6a5f, (q15_t)0xb8cf, (q15_t)0x6a51, (q15_t)0xb8ba, (q15_t)0x6a43, (q15_t)0xb8a5,
  (q15_t)0x6a35, (q15_t)0xb890, (q15_t)0x6a27, (q15_t)0xb87b, (q15_t)0x6a19, (q15_t)0xb866, (q15_t)0x6a0b, (q15_t)0xb852,
  (q15_t)0x69fd, (q15_t)0xb83d, (q15_t)0x69ef, (q15_t)0xb828, (q15_t)0x69e1, (q15_t)0xb813, (q15_t)0x69d3, (q15_t)0xb7fe,
  (q15_t)0x69c4, (q15_t)0xb7ea, (q15_t)0x69b6, (q15_t)0xb7d5, (q15_t)0x69a8, (q15_t)0xb7c0, (q15_t)0x699a, (q15_t)0xb7ab,
  (q15_t)0x698c, (q15_t)0xb797, (q15_t)0x697d, (q15_t)0xb782, (q15_t)0x696f, (q15_t)0xb76d, (q15_t)0x6961, (q15_t)0xb758,
  (q15_t)0x6953, (q15_t)0xb744, (q15_t)0x6944, (q15_t)0xb72f, (q15_t)0x6936, (q15_t)0xb71a, (q15_t)0x6928, (q15_t)0xb706,
  (q15_t)0x6919, (q15_t)0xb6f1, (q15_t)0x690b, (q15_t)0xb6dd, (q15_t)0x68fd, (q15_t)0xb6c8, (q15_t)0x68ee, (q15_t)0xb6b3,
  (q15_t)0x68e0, (q15_t)0xb69f, (q15_t)0x68d1, (q15_t)0xb68a, (q15_t)0x68c3, (q15_t)0xb676, (q15_t)0x68b5, (q15_t)0xb661,
  (q15_t)0x68a6, (q15_t)0xb64c, (q15_t)0x6898, (q15_t)0xb638, (q15_t)0x6889, (q15_t)0xb623, (q15_t)0x687b, (q15_t)0xb60f,
  (q15_t)0x686c, (q15_t)0xb5fa, (q15_t)0x685e, (q15_t)0xb5e6, (q15_t)0x684f, (q15_t)0xb5d1, (q15_t)0x6840, (q15_t)0xb5bd,
  (q15_t)0x6832, (q15_t)0xb5a8, (q15_t)0x6823, (q15_t)0xb594, (q15_t)0x6815, (q15_t)0xb57f, (q15_t)0x6806, (q15_t)0xb56b,
  (q15_t)0x67f7, (q15_t)0xb557, (q15_t)0x67e9, (q15_t)0xb542, (q15_t)0x67da, (q15_t)0xb52e, (q15_t)0x67cb, (q15_t)0xb519,
  (q15_t)0x67bd, (q15_t)0xb505, (q15_t)0x67ae, (q15_t)0xb4f1, (q15_t)0x679f, (q15_t)0xb4dc, (q15_t)0x6790, (q15_t)0xb4c8,
  (q15_t)0x6782, (q15_t)0xb4b4, (q15_t)0x6773, (q15_t)0xb49f, (q15_t)0x6764, (q15_t)0xb48b, (q15_t)0x6755, (q15_t)0xb477,
  (q15_t)0x6746, (q15_t)0xb462, (q15_t)0x6737, (q15_t)0xb44e, (q15_t)0x6729, (q15_t)0xb43a, (q15_t)0x671a, (q15_t)0xb426,
  (q15_t)0x670b, (q15_t)0xb411, (q15_t)0x66fc, (q15_t)0xb3fd, (q15_t)0x66ed, (q15_t)0xb3e9, (q15_t)0x66de, (q15_t)0xb3d5,
  (q15_t)0x66cf, (q15_t)0xb3c1, (q15_t)0x66c0, (q15_t)0xb3ac, (q15_t)0x66b1, (q15_t)0xb398, (q15_t)0x66a2, (q15_t)0xb384,
  (q15_t)0x6693, (q15_t)0xb370, (q15_t)0x6684, (q15_t)0xb35c, (q15_t)0x6675, (q15_t)0xb348, (q15_t)0x6666, (q15_t)0xb334,
  (q15_t)0x6657, (q15_t)0xb31f, (q15_t)0x6648, (q15_t)0xb30b, (q15_t)0x6639, (q15_t)0xb2f7, (q15_t)0x6629, (q15_t)0xb2e3,
  (q15_t)0x661a, (q15_t)0xb2cf, (q15_t)0x660b, (q15_t)0xb2bb, (q15_t)0x65fc, (q15_t)0xb2a7, (q15_t)0x65ed, (q15_t)0xb293,
  (q15_t)0x65dd, (q15_t)0xb27f, (q15_t)0x65ce, (q15_t)0xb26b, (q15_t)0x65bf, (q15_t)0xb257, (q15_t)0x65b0, (q15_t)0xb243,
  (q15_t)0x65a0, (q15_t)0xb22f, (q15_t)0x6591, (q15_t)0xb21b, (q15_t)0x6582, (q15_t)0xb207, (q15_t)0x6573, (q15_t)0xb1f3,
  (q15_t)0x6563, (q15_t)0xb1df, (q15_t)0x6554, (q15_t)0xb1cc, (q15_t)0x6545, (q15_t)0xb1b8, (q15_t)0x6535, (q15_t)0xb1a4,
  (q15_t)0x6526, (q15_t)0xb190, (q15_t)0x6516, (q15_t)0xb17c, (q15_t)0x6507, (q15_t)0xb168, (q15_t)0x64f7, (q15_t)0xb154,
  (q15_t)0x64e8, (q15_t)0xb141, (q15_t)0x64d9, (q15_t)0xb12d, (q15_t)0x64c9, (q15_t)0xb119, (q15_t)0x64ba, (q15_t)0xb105,
  (q15_t)0x64aa, (q15_t)0xb0f1, (q15_t)0x649b, (q15_t)0xb0de, (q15_t)0x648b, (q15_t)0xb0ca, (q15_t)0x647b, (q15_t)0xb0b6,
  (q15_t)0x646c, (q15_t)0xb0a2, (q15_t)0x645c, (q15_t)0xb08f, (q15_t)0x644d, (q15_t)0xb07b, (q15_t)0x643d, (q15_t)0xb067,
  (q15_t)0x642d, (q15_t)0xb054, (q15_t)0x641e, (q15_t)0xb040, (q15_t)0x640e, (q15_t)0xb02c, (q15_t)0x63fe, (q15_t)0xb019,
  (q15_t)0x63ef, (q15_t)0xb005, (q15_t)0x63df, (q15_t)0xaff1, (q15_t)0x63cf, (q15_t)0xafde, (q15_t)0x63c0, (q15_t)0xafca,
  (q15_t)0x63b0, (q15_t)0xafb7, (q15_t)0x63a0, (q15_t)0xafa3, (q15_t)0x6390, (q15_t)0xaf90, (q15_t)0x6380, (q15_t)0xaf7c,
  (q15_t)0x6371, (q15_t)0xaf69, (q15_t)0x6361, (q15_t)0xaf55, (q15_t)0x6351, (q15_t)0xaf41, (q15_t)0x6341, (q15_t)0xaf2e,
  (q15_t)0x6331, (q15_t)0xaf1b, (q15_t)0x6321, (q15_t)0xaf07, (q15_t)0x6311, (q15_t)0xaef4, (q15_t)0x6301, (q15_t)0xaee0,
  (q15_t)0x62f2, (q15_t)0xaecd, (q15_t)0x62e2, (q15_t)0xaeb9, (q15_t)0x62d2, (q15_t)0xaea6, (q15_t)0x62c2, (q15_t)0xae92,
  (q15_t)0x62b2, (q15_t)0xae7f, (q15_t)0x62a2, (q15_t)0xae6c, (q15_t)0x6292, (q15_t)0xae58, (q15_t)0x6282, (q15_t)0xae45,
  (q15_t)0x6271, (q15_t)0xae32, (q15_t)0x6261, (q15_t)0xae1e, (q15_t)0x6251, (q15_t)0xae0b, (q15_t)0x6241, (q15_t)0xadf8,
  (q15_t)0x6231, (q15_t)0xade4, (q15_t)0x6221, (q15_t)0xadd1, (q15_t)0x6211, (q15_t)0xadbe, (q15_t)0x6201, (q15_t)0xadab,
  (q15_t)0x61f1, (q15_t)0xad97, (q15_t)0x61e0, (q15_t)0xad84, (q15_t)0x61d0, (q15_t)0xad71, (q15_t)0x61c0, (q15_t)0xad5e,
  (q15_t)0x61b0, (q15_t)0xad4b, (q15_t)0x619f, (q15_t)0xad37, (q15_t)0x618f, (q15_t)0xad24, (q15_t)0x617f, (q15_t)0xad11,
  (q15_t)0x616f, (q15_t)0xacfe, (q15_t)0x615e, (q15_t)0xaceb, (q15_t)0x614e, (q15_t)0xacd8, (q15_t)0x613e, (q15_t)0xacc5,
  (q15_t)0x612d, (q15_t)0xacb2, (q15_t)0x611d, (q15_t)0xac9e, (q15_t)0x610d, (q15_t)0xac8b, (q15_t)0x60fc, (q15_t)0xac78,
  (q15_t)0x60ec, (q15_t)0xac65, (q15_t)0x60db, (q15_t)0xac52, (q15_t)0x60cb, (q15_t)0xac3f, (q15_t)0x60ba, (q15_t)0xac2c,
  (q15_t)0x60aa, (q15_t)0xac19, (q15_t)0x6099, (q15_t)0xac06, (q15_t)0x6089, (q15_t)0xabf3, (q15_t)0x6078, (q15_t)0xabe0,
  (q15_t)0x6068, (q15_t)0xabcd, (q15_t)0x6057, (q15_t)0xabbb, (q15_t)0x6047, (q15_t)0xaba8, (q15_t)0x6036, (q15_t)0xab95,
  (q15_t)0x6026, (q15_t)0xab82, (q15_t)0x6015, (q15_t)0xab6f, (q15_t)0x6004, (q15_t)0xab5c, (q15_t)0x5ff4, (q15_t)0xab49,
  (q15_t)0x5fe3, (q15_t)0xab36, (q15_t)0x5fd3, (q15_t)0xab24, (q15_t)0x5fc2, (q15_t)0xab11, (q15_t)0x5fb1, (q15_t)0xaafe,
  (q15_t)0x5fa0, (q15_t)0xaaeb, (q15_t)0x5f90, (q15_t)0xaad8, (q15_t)0x5f7f, (q15_t)0xaac6, (q15_t)0x5f6e, (q15_t)0xaab3,
  (q15_t)0x5f5e, (q15_t)0xaaa0, (q15_t)0x5f4d, (q15_t)0xaa8e, (q15_t)0x5f3c, (q15_t)0xaa7b, (q15_t)0x5f2b, (q15_t)0xaa68,
  (q15_t)0x5f1a, (q15_t)0xaa55, (q15_t)0x5f0a, (q15_t)0xaa43, (q15_t)0x5ef9, (q15_t)0xaa30, (q15_t)0x5ee8, (q15_t)0xaa1d,
  (q15_t)0x5ed7, (q15_t)0xaa0b, (q15_t)0x5ec6, (q15_t)0xa9f8, (q15_t)0x5eb5, (q15_t)0xa9e6, (q15_t)0x5ea4, (q15_t)0xa9d3,
  (q15_t)0x5e93, (q15_t)0xa9c0, (q15_t)0x5e82, (q15_t)0xa9ae, (q15_t)0x5e71, (q15_t)0xa99b, (q15_t)0x5e60, (q15_t)0xa989,
  (q15_t)0x5e50, (q15_t)0xa976, (q15_t)0x5e3f, (q15_t)0xa964, (q15_t)0x5e2d, (q15_t)0xa951, (q15_t)0x5e1c, (q15_t)0xa93f,
  (q15_t)0x5e0b, (q15_t)0xa92c, (q15_t)0x5dfa, (q15_t)0xa91a, (q15_t)0x5de9, (q15_t)0xa907, (q15_t)0x5dd8, (q15_t)0xa8f5,
  (q15_t)0x5dc7, (q15_t)0xa8e3, (q15_t)0x5db6, (q15_t)0xa8d0, (q15_t)0x5da5, (q15_t)0xa8be, (q15_t)0x5d94, (q15_t)0xa8ab,
  (q15_t)0x5d83, (q15_t)0xa899, (q15_t)0x5d71, (q15_t)0xa887, (q15_t)0x5d60, (q15_t)0xa874, (q15_t)0x5d4f, (q15_t)0xa862,
  (q15_t)0x5d3e, (q15_t)0xa850, (q15_t)0x5d2d, (q15_t)0xa83d, (q15_t)0x5d1b, (q15_t)0xa82b, (q15_t)0x5d0a, (q15_t)0xa819,
  (q15_t)0x5cf9, (q15_t)0xa807, (q15_t)0x5ce8, (q15_t)0xa7f4, (q15_t)0x5cd6, (q15_t)0xa7e2, (q15_t)0x5cc5, (q15_t)0xa7d0,
  (q15_t)0x5cb4, (q15_t)0xa7be, (q15_t)0x5ca2, (q15_t)0xa7ab, (q15_t)0x5c91, (q15_t)0xa799, (q15_t)0x5c80, (q15_t)0xa787,
  (q15_t)0x5c6e, (q15_t)0xa775, (q15_t)0x5c5d, (q15_t)0xa763, (q15_t)0x5c4b, (q15_t)0xa751, (q15_t)0x5c3a, (q15_t)0xa73f,
  (q15_t)0x5c29, (q15_t)0xa72c, (q15_t)0x5c17, (q15_t)0xa71a, (q15_t)0x5c06, (q15_t)0xa708, (q15_t)0x5bf4, (q15_t)0xa6f6,
  (q15_t)0x5be3, (q15_t)0xa6e4, (q15_t)0x5bd1, (q15_t)0xa6d2, (q15_t)0x5bc0, (q15_t)0xa6c0, (q15_t)0x5bae, (q15_t)0xa6ae,
  (q15_t)0x5b9d, (q15_t)0xa69c, (q15_t)0x5b8b, (q15_t)0xa68a, (q15_t)0x5b79, (q15_t)0xa678, (q15_t)0x5b68, (q15_t)0xa666,
  (q15_t)0x5b56, (q15_t)0xa654, (q15_t)0x5b45, (q15_t)0xa642, (q15_t)0x5b33, (q15_t)0xa630, (q15_t)0x5b21, (q15_t)0xa61f,
  (q15_t)0x5b10, (q15_t)0xa60d, (q15_t)0x5afe, (q15_t)0xa5fb, (q15_t)0x5aec, (q15_t)0xa5e9, (q15_t)0x5adb, (q15_t)0xa5d7,
  (q15_t)0x5ac9, (q15_t)0xa5c5, (q15_t)0x5ab7, (q15_t)0xa5b3, (q15_t)0x5aa5, (q15_t)0xa5a2, (q15_t)0x5a94, (q15_t)0xa590,
  (q15_t)0x5a82, (q15_t)0xa57e, (q15_t)0x5a70, (q15_t)0xa56c, (q15_t)0x5a5e, (q15_t)0xa55b, (q15_t)0x5a4d, (q15_t)0xa549,
  (q15_t)0x5a3b, (q15_t)0xa537, (q15_t)0x5a29, (q15_t)0xa525, (q15_t)0x5a17, (q15_t)0xa514, (q15_t)0x5a05, (q15_t)0xa502,
  (q15_t)0x59f3, (q15_t)0xa4f0, (q15_t)0x59e1, (q15_t)0xa4df, (q15_t)0x59d0, (q15_t)0xa4cd, (q15_t)0x59be, (q15_t)0xa4bb,
  (q15_t)0x59ac, (q15_t)0xa4aa, (q15_t)0x599a, (q15_t)0xa498, (q15_t)0x5988, (q15_t)0xa487, (q15_t)0x5976, (q15_t)0xa475,
  (q15_t)0x5964, (q15_t)0xa463, (q15_t)0x5952, (q15_t)0xa452, (q15_t)0x5940, (q15_t)0xa440, (q15_t)0x592e, (q15_t)0xa42f,
  (q15_t)0x591c, (q15_t)0xa41d, (q15_t)0x590a, (q15_t)0xa40c, (q15_t)0x58f8, (q15_t)0xa3fa, (q15_t)0x58e6, (q15_t)0xa3e9,
  (q15_t)0x58d4, (q15_t)0xa3d7, (q15_t)0x58c1, (q15_t)0xa3c6, (q15_t)0x58af, (q15_t)0xa3b5, (q15_t)0x589d, (q15_t)0xa3a3,
  (q15_t)0x588b, (q15_t)0xa392, (q15_t)0x5879, (q15_t)0xa380, (q15_t)0x5867, (q15_t)0xa36f, (q15_t)0x5855, (q15_t)0xa35e,
  (q15_t)0x5842, (q15_t)0xa34c, (q15_t)0x5830, (q15_t)0xa33b, (q15_t)0x581e, (q15_t)0xa32a, (q15_t)0x580c, (q15_t)0xa318,
  (q15_t)0x57f9, (q15_t)0xa307, (q15_t)0x57e7, (q15_t)0xa2f6, (q15_t)0x57d5, (q15_t)0xa2e5, (q15_t)0x57c3, (q15_t)0xa2d3,
  (q15_t)0x57b0, (q15_t)0xa2c2, (q15_t)0x579e, (q15_t)0xa2b1, (q15_t)0x578c, (q15_t)0xa2a0, (q15_t)0x5779, (q15_t)0xa28f,
  (q15_t)0x5767, (q15_t)0xa27d, (q15_t)0x5755, (q15_t)0xa26c, (q15_t)0x5742, (q15_t)0xa25b, (q15_t)0x5730, (q15_t)0xa24a,
  (q15_t)0x571d, (q15_t)0xa239, (q15_t)0x570b, (q15_t)0xa228, (q15_t)0x56f9, (q15_t)0xa217, (q15_t)0x56e6, (q15_t)0xa206,
  (q15_t)0x56d4, (q15_t)0xa1f5, (q15_t)0x56c1, (q15_t)0xa1e4, (q15_t)0x56af, (q15_t)0xa1d3, (q15_t)0x569c, (q15_t)0xa1c1,
  (q15_t)0x568a, (q15_t)0xa1b0, (q15_t)0x5677, (q15_t)0xa1a0, (q15_t)0x5665, (q15_t)0xa18f, (q15_t)0x5652, (q15_t)0xa17e,
  (q15_t)0x5640, (q15_t)0xa16d, (q15_t)0x562d, (q15_t)0xa15c, (q15_t)0x561a, (q15_t)0xa14b, (q15_t)0x5608, (q15_t)0xa13a,
  (q15_t)0x55f5, (q15_t)0xa129, (q15_t)0x55e3, (q15_t)0xa118, (q15_t)0x55d0, (q15_t)0xa107, (q15_t)0x55bd, (q15_t)0xa0f6,
  (q15_t)0x55ab, (q15_t)0xa0e6, (q15_t)0x5598, (q15_t)0xa0d5, (q15_t)0x5585, (q15_t)0xa0c4, (q15_t)0x5572, (q15_t)0xa0b3,
  (q15_t)0x5560, (q15_t)0xa0a2, (q15_t)0x554d, (q15_t)0xa092, (q15_t)0x553a, (q15_t)0xa081, (q15_t)0x5528, (q15_t)0xa070,
  (q15_t)0x5515, (q15_t)0xa060, (q15_t)0x5502, (q15_t)0xa04f, (q15_t)0x54ef, (q15_t)0xa03e, (q15_t)0x54dc, (q15_t)0xa02d,
  (q15_t)0x54ca, (q15_t)0xa01d, (q15_t)0x54b7, (q15_t)0xa00c, (q15_t)0x54a4, (q15_t)0x9ffc, (q15_t)0x5491, (q15_t)0x9feb,
  (q15_t)0x547e, (q15_t)0x9fda, (q15_t)0x546b, (q15_t)0x9fca, (q15_t)0x5458, (q15_t)0x9fb9, (q15_t)0x5445, (q15_t)0x9fa9,
  (q15_t)0x5433, (q15_t)0x9f98, (q15_t)0x5420, (q15_t)0x9f88, (q15_t)0x540d, (q15_t)0x9f77, (q15_t)0x53fa, (q15_t)0x9f67,
  (q15_t)0x53e7, (q15_t)0x9f56, (q15_t)0x53d4, (q15_t)0x9f46, (q15_t)0x53c1, (q15_t)0x9f35, (q15_t)0x53ae, (q15_t)0x9f25,
  (q15_t)0x539b, (q15_t)0x9f14, (q15_t)0x5388, (q15_t)0x9f04, (q15_t)0x5375, (q15_t)0x9ef3, (q15_t)0x5362, (q15_t)0x9ee3,
  (q15_t)0x534e, (q15_t)0x9ed3, (q15_t)0x533b, (q15_t)0x9ec2, (q15_t)0x5328, (q15_t)0x9eb2, (q15_t)0x5315, (q15_t)0x9ea2,
  (q15_t)0x5302, (q15_t)0x9e91, (q15_t)0x52ef, (q15_t)0x9e81, (q15_t)0x52dc, (q15_t)0x9e71, (q15_t)0x52c9, (q15_t)0x9e61,
  (q15_t)0x52b5, (q15_t)0x9e50, (q15_t)0x52a2, (q15_t)0x9e40, (q15_t)0x528f, (q15_t)0x9e30, (q15_t)0x527c, (q15_t)0x9e20,
  (q15_t)0x5269, (q15_t)0x9e0f, (q15_t)0x5255, (q15_t)0x9dff, (q15_t)0x5242, (q15_t)0x9def, (q15_t)0x522f, (q15_t)0x9ddf,
  (q15_t)0x521c, (q15_t)0x9dcf, (q15_t)0x5208, (q15_t)0x9dbf, (q15_t)0x51f5, (q15_t)0x9daf, (q15_t)0x51e2, (q15_t)0x9d9f,
  (q15_t)0x51ce, (q15_t)0x9d8f, (q15_t)0x51bb, (q15_t)0x9d7e, (q15_t)0x51a8, (q15_t)0x9d6e, (q15_t)0x5194, (q15_t)0x9d5e,
  (q15_t)0x5181, (q15_t)0x9d4e, (q15_t)0x516e, (q15_t)0x9d3e, (q15_t)0x515a, (q15_t)0x9d2e, (q15_t)0x5147, (q15_t)0x9d1e,
  (q15_t)0x5133, (q15_t)0x9d0e, (q15_t)0x5120, (q15_t)0x9cff, (q15_t)0x510c, (q15_t)0x9cef, (q15_t)0x50f9, (q15_t)0x9cdf,
  (q15_t)0x50e5, (q15_t)0x9ccf, (q15_t)0x50d2, (q15_t)0x9cbf, (q15_t)0x50bf, (q15_t)0x9caf, (q15_t)0x50ab, (q15_t)0x9c9f,
  (q15_t)0x5097, (q15_t)0x9c8f, (q15_t)0x5084, (q15_t)0x9c80, (q15_t)0x5070, (q15_t)0x9c70, (q15_t)0x505d, (q15_t)0x9c60,
  (q15_t)0x5049, (q15_t)0x9c50, (q15_t)0x5036, (q15_t)0x9c40, (q15_t)0x5022, (q15_t)0x9c31, (q15_t)0x500f, (q15_t)0x9c21,
  (q15_t)0x4ffb, (q15_t)0x9c11, (q15_t)0x4fe7, (q15_t)0x9c02, (q15_t)0x4fd4, (q15_t)0x9bf2, (q15_t)0x4fc0, (q15_t)0x9be2,
  (q15_t)0x4fac, (q15_t)0x9bd3, (q15_t)0x4f99, (q15_t)0x9bc3, (q15_t)0x4f85, (q15_t)0x9bb3, (q15_t)0x4f71, (q15_t)0x9ba4,
  (q15_t)0x4f5e, (q15_t)0x9b94, (q15_t)0x4f4a, (q15_t)0x9b85, (q15_t)0x4f36, (q15_t)0x9b75, (q15_t)0x4f22, (q15_t)0x9b65,
  (q15_t)0x4f0f, (q15_t)0x9b56, (q15_t)0x4efb, (q15_t)0x9b46, (q15_t)0x4ee7, (q15_t)0x9b37, (q15_t)0x4ed3, (q15_t)0x9b27,
  (q15_t)0x4ebf, (q15_t)0x9b18, (q15_t)0x4eac, (q15_t)0x9b09, (q15_t)0x4e98, (q15_t)0x9af9, (q15_t)0x4e84, (q15_t)0x9aea,
  (q15_t)0x4e70, (q15_t)0x9ada, (q15_t)0x4e5c, (q15_t)0x9acb, (q15_t)0x4e48, (q15_t)0x9abb, (q15_t)0x4e34, (q15_t)0x9aac,
  (q15_t)0x4e21, (q15_t)0x9a9d, (q15_t)0x4e0d, (q15_t)0x9a8d, (q15_t)0x4df9, (q15_t)0x9a7e, (q15_t)0x4de5, (q15_t)0x9a6f,
  (q15_t)0x4dd1, (q15_t)0x9a60, (q15_t)0x4dbd, (q15_t)0x9a50, (q15_t)0x4da9, (q15_t)0x9a41, (q15_t)0x4d95, (q15_t)0x9a32,
  (q15_t)0x4d81, (q15_t)0x9a23, (q15_t)0x4d6d, (q15_t)0x9a13, (q15_t)0x4d59, (q15_t)0x9a04, (q15_t)0x4d45, (q15_t)0x99f5,
  (q15_t)0x4d31, (q15_t)0x99e6, (q15_t)0x4d1d, (q15_t)0x99d7, (q15_t)0x4d09, (q15_t)0x99c7, (q15_t)0x4cf5, (q15_t)0x99b8,
  (q15_t)0x4ce1, (q15_t)0x99a9, (q15_t)0x4ccc, (q15_t)0x999a, (q15_t)0x4cb8, (q15_t)0x998b, (q15_t)0x4ca4, (q15_t)0x997c,
  (q15_t)0x4c90, (q15_t)0x996d, (q15_t)0x4c7c, (q15_t)0x995e, (q15_t)0x4c68, (q15_t)0x994f, (q15_t)0x4c54, (q15_t)0x9940,
  (q15_t)0x4c3f, (q15_t)0x9931, (q15_t)0x4c2b, (q15_t)0x9922, (q15_t)0x4c17, (q15_t)0x9913, (q15_t)0x4c03, (q15_t)0x9904,
  (q15_t)0x4bef, (q15_t)0x98f5, (q15_t)0x4bda, (q15_t)0x98e6, (q15_t)0x4bc6, (q15_t)0x98d7, (q15_t)0x4bb2, (q15_t)0x98c9,
  (q15_t)0x4b9e, (q15_t)0x98ba, (q15_t)0x4b89, (q15_t)0x98ab, (q15_t)0x4b75, (q15_t)0x989c, (q15_t)0x4b61, (q15_t)0x988d,
  (q15_t)0x4b4c, (q15_t)0x987e, (q15_t)0x4b38, (q15_t)0x9870, (q15_t)0x4b24, (q15_t)0x9861, (q15_t)0x4b0f, (q15_t)0x9852,
  (q15_t)0x4afb, (q15_t)0x9843, (q15_t)0x4ae7, (q15_t)0x9835, (q15_t)0x4ad2, (q15_t)0x9826, (q15_t)0x4abe, (q15_t)0x9817,
  (q15_t)0x4aa9, (q15_t)0x9809, (q15_t)0x4a95, (q15_t)0x97fa, (q15_t)0x4a81, (q15_t)0x97eb, (q15_t)0x4a6c, (q15_t)0x97dd,
  (q15_t)0x4a58, (q15_t)0x97ce, (q15_t)0x4a43, (q15_t)0x97c0, (q15_t)0x4a2f, (q15_t)0x97b1, (q15_t)0x4a1a, (q15_t)0x97a2,
  (q15_t)0x4a06, (q15_t)0x9794, (q15_t)0x49f1, (q15_t)0x9785, (q15_t)0x49dd, (q15_t)0x9777, (q15_t)0x49c8, (q15_t)0x9768,
  (q15_t)0x49b4, (q15_t)0x975a, (q15_t)0x499f, (q15_t)0x974b, (q15_t)0x498a, (q15_t)0x973d, (q15_t)0x4976, (q15_t)0x972f,
  (q15_t)0x4961, (q15_t)0x9720, (q15_t)0x494d, (q15_t)0x9712, (q15_t)0x4938, (q15_t)0x9703, (q15_t)0x4923, (q15_t)0x96f5,
  (q15_t)0x490f, (q15_t)0x96e7, (q15_t)0x48fa, (q15_t)0x96d8, (q15_t)0x48e6, (q15_t)0x96ca, (q15_t)0x48d1, (q15_t)0x96bc,
  (q15_t)0x48bc, (q15_t)0x96ad, (q15_t)0x48a8, (q15_t)0x969f, (q15_t)0x4893, (q15_t)0x9691, (q15_t)0x487e, (q15_t)0x9683,
  (q15_t)0x4869, (q15_t)0x9674, (q15_t)0x4855, (q15_t)0x9666, (q15_t)0x4840, (q15_t)0x9658, (q15_t)0x482b, (q15_t)0x964a,
  (q15_t)0x4816, (q15_t)0x963c, (q15_t)0x4802, (q15_t)0x962d, (q15_t)0x47ed, (q15_t)0x961f, (q15_t)0x47d8, (q15_t)0x9611,
  (q15_t)0x47c3, (q15_t)0x9603, (q15_t)0x47ae, (q15_t)0x95f5, (q15_t)0x479a, (q15_t)0x95e7, (q15_t)0x4785, (q15_t)0x95d9,
  (q15_t)0x4770, (q15_t)0x95cb, (q15_t)0x475b, (q15_t)0x95bd, (q15_t)0x4746, (q15_t)0x95af, (q15_t)0x4731, (q15_t)0x95a1,
  (q15_t)0x471c, (q15_t)0x9593, (q15_t)0x4708, (q15_t)0x9585, (q15_t)0x46f3, (q15_t)0x9577, (q15_t)0x46de, (q15_t)0x9569,
  (q15_t)0x46c9, (q15_t)0x955b, (q15_t)0x46b4, (q15_t)0x954d, (q15_t)0x469f, (q15_t)0x953f, (q15_t)0x468a, (q15_t)0x9532,
  (q15_t)0x4675, (q15_t)0x9524, (q15_t)0x4660, (q15_t)0x9516, (q15_t)0x464b, (q15_t)0x9508, (q15_t)0x4636, (q15_t)0x94fa,
  (q15_t)0x4621, (q15_t)0x94ed, (q15_t)0x460c, (q15_t)0x94df, (q15_t)0x45f7, (q15_t)0x94d1, (q15_t)0x45e2, (q15_t)0x94c3,
  (q15_t)0x45cd, (q15_t)0x94b6, (q15_t)0x45b8, (q15_t)0x94a8, (q15_t)0x45a3, (q15_t)0x949a, (q15_t)0x458d, (q15_t)0x948d,
  (q15_t)0x4578, (q15_t)0x947f, (q15_t)0x4563, (q15_t)0x9471, (q15_t)0x454e, (q15_t)0x9464, (q15_t)0x4539, (q15_t)0x9456,
  (q15_t)0x4524, (q15_t)0x9448, (q15_t)0x450f, (q15_t)0x943b, (q15_t)0x44fa, (q15_t)0x942d, (q15_t)0x44e4, (q15_t)0x9420,
  (q15_t)0x44cf, (q15_t)0x9412, (q15_t)0x44ba, (q15_t)0x9405, (q15_t)0x44a5, (q15_t)0x93f7, (q15_t)0x4490, (q15_t)0x93ea,
  (q15_t)0x447a, (q15_t)0x93dc, (q15_t)0x4465, (q15_t)0x93cf, (q15_t)0x4450, (q15_t)0x93c1, (q15_t)0x443b, (q15_t)0x93b4,
  (q15_t)0x4425, (q15_t)0x93a7, (q15_t)0x4410, (q15_t)0x9399, (q15_t)0x43fb, (q15_t)0x938c, (q15_t)0x43e5, (q15_t)0x937f,
  (q15_t)0x43d0, (q15_t)0x9371, (q15_t)0x43bb, (q15_t)0x9364, (q15_t)0x43a5, (q15_t)0x9357, (q15_t)0x4390, (q15_t)0x9349,
  (q15_t)0x437b, (q15_t)0x933c, (q15_t)0x4365, (q15_t)0x932f, (q15_t)0x4350, (q15_t)0x9322, (q15_t)0x433b, (q15_t)0x9314,
  (q15_t)0x4325, (q15_t)0x9307, (q15_t)0x4310, (q15_t)0x92fa, (q15_t)0x42fa, (q15_t)0x92ed, (q15_t)0x42e5, (q15_t)0x92e0,
  (q15_t)0x42d0, (q15_t)0x92d3, (q15_t)0x42ba, (q15_t)0x92c6, (q15_t)0x42a5, (q15_t)0x92b8, (q15_t)0x428f, (q15_t)0x92ab,
  (q15_t)0x427a, (q15_t)0x929e, (q15_t)0x4264, (q15_t)0x9291, (q15_t)0x424f, (q15_t)0x9284, (q15_t)0x4239, (q15_t)0x9277,
  (q15_t)0x4224, (q15_t)0x926a, (q15_t)0x420e, (q15_t)0x925d, (q15_t)0x41f9, (q15_t)0x9250, (q15_t)0x41e3, (q15_t)0x9243,
  (q15_t)0x41ce, (q15_t)0x9236, (q15_t)0x41b8, (q15_t)0x922a, (q15_t)0x41a2, (q15_t)0x921d, (q15_t)0x418d, (q15_t)0x9210,
  (q15_t)0x4177, (q15_t)0x9203, (q15_t)0x4162, (q15_t)0x91f6, (q15_t)0x414c, (q15_t)0x91e9, (q15_t)0x4136, (q15_t)0x91dc,
  (q15_t)0x4121, (q15_t)0x91d0, (q15_t)0x410b, (q15_t)0x91c3, (q15_t)0x40f6, (q15_t)0x91b6, (q15_t)0x40e0, (q15_t)0x91a9,
  (q15_t)0x40ca, (q15_t)0x919d, (q15_t)0x40b5, (q15_t)0x9190, (q15_t)0x409f, (q15_t)0x9183, (q15_t)0x4089, (q15_t)0x9177,
  (q15_t)0x4073, (q15_t)0x916a, (q15_t)0x405e, (q15_t)0x915d, (q15_t)0x4048, (q15_t)0x9151, (q15_t)0x4032, (q15_t)0x9144,
  (q15_t)0x401d, (q15_t)0x9137, (q15_t)0x4007, (q15_t)0x912b, (q15_t)0x3ff1, (q15_t)0x911e, (q15_t)0x3fdb, (q15_t)0x9112,
  (q15_t)0x3fc5, (q15_t)0x9105, (q15_t)0x3fb0, (q15_t)0x90f9, (q15_t)0x3f9a, (q15_t)0x90ec, (q15_t)0x3f84, (q15_t)0x90e0,
  (q15_t)0x3f6e, (q15_t)0x90d3, (q15_t)0x3f58, (q15_t)0x90c7, (q15_t)0x3f43, (q15_t)0x90ba, (q15_t)0x3f2d, (q15_t)0x90ae,
  (q15_t)0x3f17, (q15_t)0x90a1, (q15_t)0x3f01, (q15_t)0x9095, (q15_t)0x3eeb, (q15_t)0x9089, (q15_t)0x3ed5, (q15_t)0x907c,
  (q15_t)0x3ebf, (q15_t)0x9070, (q15_t)0x3ea9, (q15_t)0x9064, (q15_t)0x3e93, (q15_t)0x9057, (q15_t)0x3e7d, (q15_t)0x904b,
  (q15_t)0x3e68, (q15_t)0x903f, (q15_t)0x3e52, (q15_t)0x9033, (q15_t)0x3e3c, (q15_t)0x9026, (q15_t)0x3e26, (q15_t)0x901a,
  (q15_t)0x3e10, (q15_t)0x900e, (q15_t)0x3dfa, (q15_t)0x9002, (q15_t)0x3de4, (q15_t)0x8ff6, (q15_t)0x3dce, (q15_t)0x8fea,
  (q15_t)0x3db8, (q15_t)0x8fdd, (q15_t)0x3da2, (q15_t)0x8fd1, (q15_t)0x3d8c, (q15_t)0x8fc5, (q15_t)0x3d76, (q15_t)0x8fb9,
  (q15_t)0x3d60, (q15_t)0x8fad, (q15_t)0x3d49, (q15_t)0x8fa1, (q15_t)0x3d33, (q15_t)0x8f95, (q15_t)0x3d1d, (q15_t)0x8f89,
  (q15_t)0x3d07, (q15_t)0x8f7d, (q15_t)0x3cf1, (q15_t)0x8f71, (q15_t)0x3cdb, (q15_t)0x8f65, (q15_t)0x3cc5, (q15_t)0x8f59,
  (q15_t)0x3caf, (q15_t)0x8f4d, (q15_t)0x3c99, (q15_t)0x8f41, (q15_t)0x3c83, (q15_t)0x8f35, (q15_t)0x3c6c, (q15_t)0x8f2a,
  (q15_t)0x3c56, (q15_t)0x8f1e, (q15_t)0x3c40, (q15_t)0x8f12, (q15_t)0x3c2a, (q15_t)0x8f06, (q15_t)0x3c14, (q15_t)0x8efa,
  (q15_t)0x3bfd, (q15_t)0x8eee, (q15_t)0x3be7, (q15_t)0x8ee3, (q15_t)0x3bd1, (q15_t)0x8ed7, (q15_t)0x3bbb, (q15_t)0x8ecb,
  (q15_t)0x3ba5, (q15_t)0x8ebf, (q15_t)0x3b8e, (q15_t)0x8eb4, (q15_t)0x3b78, (q15_t)0x8ea8, (q15_t)0x3b62, (q15_t)0x8e9c,
  (q15_t)0x3b4c, (q15_t)0x8e91, (q15_t)0x3b35, (q15_t)0x8e85, (q15_t)0x3b1f, (q15_t)0x8e7a, (q15_t)0x3b09, (q15_t)0x8e6e,
  (q15_t)0x3af2, (q15_t)0x8e62, (q15_t)0x3adc, (q15_t)0x8e57, (q15_t)0x3ac6, (q15_t)0x8e4b, (q15_t)0x3aaf, (q15_t)0x8e40,
  (q15_t)0x3a99, (q15_t)0x8e34, (q15_t)0x3a83, (q15_t)0x8e29, (q15_t)0x3a6c, (q15_t)0x8e1d, (q15_t)0x3a56, (q15_t)0x8e12,
  (q15_t)0x3a40, (q15_t)0x8e06, (q15_t)0x3a29, (q15_t)0x8dfb, (q15_t)0x3a13, (q15_t)0x8def, (q15_t)0x39fd, (q15_t)0x8de4,
  (q15_t)0x39e6, (q15_t)0x8dd9, (q15_t)0x39d0, (q15_t)0x8dcd, (q15_t)0x39b9, (q15_t)0x8dc2, (q15_t)0x39a3, (q15_t)0x8db7,
  (q15_t)0x398c, (q15_t)0x8dab, (q15_t)0x3976, (q15_t)0x8da0, (q15_t)0x395f, (q15_t)0x8d95, (q15_t)0x3949, (q15_t)0x8d8a,
  (q15_t)0x3932, (q15_t)0x8d7e, (q15_t)0x391c, (q15_t)0x8d73, (q15_t)0x3906, (q15_t)0x8d68, (q15_t)0x38ef, (q15_t)0x8d5d,
  (q15_t)0x38d8, (q15_t)0x8d51, (q15_t)0x38c2, (q15_t)0x8d46, (q15_t)0x38ab, (q15_t)0x8d3b, (q15_t)0x3895, (q15_t)0x8d30,
  (q15_t)0x387e, (q15_t)0x8d25, (q15_t)0x3868, (q15_t)0x8d1a, (q15_t)0x3851, (q15_t)0x8d0f, (q15_t)0x383b, (q15_t)0x8d04,
  (q15_t)0x3824, (q15_t)0x8cf9, (q15_t)0x380d, (q15_t)0x8cee, (q15_t)0x37f7, (q15_t)0x8ce3, (q15_t)0x37e0, (q15_t)0x8cd8,
  (q15_t)0x37ca, (q15_t)0x8ccd, (q15_t)0x37b3, (q15_t)0x8cc2, (q15_t)0x379c, (q15_t)0x8cb7, (q15_t)0x3786, (q15_t)0x8cac,
  (q15_t)0x376f, (q15_t)0x8ca1, (q15_t)0x3758, (q15_t)0x8c96, (q15_t)0x3742, (q15_t)0x8c8b, (q15_t)0x372b, (q15_t)0x8c81,
  (q15_t)0x3714, (q15_t)0x8c76, (q15_t)0x36fe, (q15_t)0x8c6b, (q15_t)0x36e7, (q15_t)0x8c60, (q15_t)0x36d0, (q15_t)0x8c55,
  (q15_t)0x36ba, (q15_t)0x8c4b, (q15_t)0x36a3, (q15_t)0x8c40, (q15_t)0x368c, (q15_t)0x8c35, (q15_t)0x3675, (q15_t)0x8c2a,
  (q15_t)0x365f, (q15_t)0x8c20, (q15_t)0x3648, (q15_t)0x8c15, (q15_t)0x3631, (q15_t)0x8c0a, (q15_t)0x361a, (q15_t)0x8c00,
  (q15_t)0x3604, (q15_t)0x8bf5, (q15_t)0x35ed, (q15_t)0x8beb, (q15_t)0x35d6, (q15_t)0x8be0, (q15_t)0x35bf, (q15_t)0x8bd5,
  (q15_t)0x35a8, (q15_t)0x8bcb, (q15_t)0x3592, (q15_t)0x8bc0, (q15_t)0x357b, (q15_t)0x8bb6, (q15_t)0x3564, (q15_t)0x8bab,
  (q15_t)0x354d, (q15_t)0x8ba1, (q15_t)0x3536, (q15_t)0x8b96, (q15_t)0x351f, (q15_t)0x8b8c, (q15_t)0x3508, (q15_t)0x8b82,
  (q15_t)0x34f2, (q15_t)0x8b77, (q15_t)0x34db, (q15_t)0x8b6d, (q15_t)0x34c4, (q15_t)0x8b62, (q15_t)0x34ad, (q15_t)0x8b58,
  (q15_t)0x3496, (q15_t)0x8b4e, (q15_t)0x347f, (q15_t)0x8b43, (q15_t)0x3468, (q15_t)0x8b39, (q15_t)0x3451, (q15_t)0x8b2f,
  (q15_t)0x343a, (q15_t)0x8b25, (q15_t)0x3423, (q15_t)0x8b1a, (q15_t)0x340c, (q15_t)0x8b10, (q15_t)0x33f5, (q15_t)0x8b06,
  (q15_t)0x33de, (q15_t)0x8afc, (q15_t)0x33c7, (q15_t)0x8af1, (q15_t)0x33b0, (q15_t)0x8ae7, (q15_t)0x3399, (q15_t)0x8add,
  (q15_t)0x3382, (q15_t)0x8ad3, (q15_t)0x336b, (q15_t)0x8ac9, (q15_t)0x3354, (q15_t)0x8abf, (q15_t)0x333d, (q15_t)0x8ab5,
  (q15_t)0x3326, (q15_t)0x8aab, (q15_t)0x330f, (q15_t)0x8aa1, (q15_t)0x32f8, (q15_t)0x8a97, (q15_t)0x32e1, (q15_t)0x8a8d,
  (q15_t)0x32ca, (q15_t)0x8a83, (q15_t)0x32b3, (q15_t)0x8a79, (q15_t)0x329c, (q15_t)0x8a6f, (q15_t)0x3285, (q15_t)0x8a65,
  (q15_t)0x326e, (q15_t)0x8a5b, (q15_t)0x3257, (q15_t)0x8a51, (q15_t)0x3240, (q15_t)0x8a47, (q15_t)0x3228, (q15_t)0x8a3d,
  (q15_t)0x3211, (q15_t)0x8a34, (q15_t)0x31fa, (q15_t)0x8a2a, (q15_t)0x31e3, (q15_t)0x8a20, (q15_t)0x31cc, (q15_t)0x8a16,
  (q15_t)0x31b5, (q15_t)0x8a0c, (q15_t)0x319e, (q15_t)0x8a03, (q15_t)0x3186, (q15_t)0x89f9, (q15_t)0x316f, (q15_t)0x89ef,
  (q15_t)0x3158, (q15_t)0x89e5, (q15_t)0x3141, (q15_t)0x89dc, (q15_t)0x312a, (q15_t)0x89d2, (q15_t)0x3112, (q15_t)0x89c8,
  (q15_t)0x30fb, (q15_t)0x89bf, (q15_t)0x30e4, (q15_t)0x89b5, (q15_t)0x30cd, (q15_t)0x89ac, (q15_t)0x30b6, (q15_t)0x89a2,
  (q15_t)0x309e, (q15_t)0x8998, (q15_t)0x3087, (q15_t)0x898f, (q15_t)0x3070, (q15_t)0x8985, (q15_t)0x3059, (q15_t)0x897c,
  (q15_t)0x3041, (q15_t)0x8972, (q15_t)0x302a, (q15_t)0x8969, (q15_t)0x3013, (q15_t)0x8960, (q15_t)0x2ffb, (q15_t)0x8956,
  (q15_t)0x2fe4, (q15_t)0x894d, (q15_t)0x2fcd, (q15_t)0x8943, (q15_t)0x2fb5, (q15_t)0x893a, (q15_t)0x2f9e, (q15_t)0x8931,
  (q15_t)0x2f87, (q15_t)0x8927, (q15_t)0x2f6f, (q15_t)0x891e, (q15_t)0x2f58, (q15_t)0x8915, (q15_t)0x2f41, (q15_t)0x890b,
  (q15_t)0x2f29, (q15_t)0x8902, (q15_t)0x2f12, (q15_t)0x88f9, (q15_t)0x2efb, (q15_t)0x88f0, (q15_t)0x2ee3, (q15_t)0x88e6,
  (q15_t)0x2ecc, (q15_t)0x88dd, (q15_t)0x2eb5, (q15_t)0x88d4, (q15_t)0x2e9d, (q15_t)0x88cb, (q15_t)0x2e86, (q15_t)0x88c2,
  (q15_t)0x2e6e, (q15_t)0x88b9, (q15_t)0x2e57, (q15_t)0x88af, (q15_t)0x2e3f, (q15_t)0x88a6, (q15_t)0x2e28, (q15_t)0x889d,
  (q15_t)0x2e11, (q15_t)0x8894, (q15_t)0x2df9, (q15_t)0x888b, (q15_t)0x2de2, (q15_t)0x8882, (q15_t)0x2dca, (q15_t)0x8879,
  (q15_t)0x2db3, (q15_t)0x8870, (q15_t)0x2d9b, (q15_t)0x8867, (q15_t)0x2d84, (q15_t)0x885e, (q15_t)0x2d6c, (q15_t)0x8855,
  (q15_t)0x2d55, (q15_t)0x884c, (q15_t)0x2d3d, (q15_t)0x8844, (q15_t)0x2d26, (q15_t)0x883b, (q15_t)0x2d0e, (q15_t)0x8832,
  (q15_t)0x2cf7, (q15_t)0x8829, (q15_t)0x2cdf, (q15_t)0x8820, (q15_t)0x2cc8, (q15_t)0x8817, (q15_t)0x2cb0, (q15_t)0x880f,
  (q15_t)0x2c98, (q15_t)0x8806, (q15_t)0x2c81, (q15_t)0x87fd, (q15_t)0x2c69, (q15_t)0x87f4, (q15_t)0x2c52, (q15_t)0x87ec,
  (q15_t)0x2c3a, (q15_t)0x87e3, (q15_t)0x2c23, (q15_t)0x87da, (q15_t)0x2c0b, (q15_t)0x87d2, (q15_t)0x2bf3, (q15_t)0x87c9,
  (q15_t)0x2bdc, (q15_t)0x87c0, (q15_t)0x2bc4, (q15_t)0x87b8, (q15_t)0x2bad, (q15_t)0x87af, (q15_t)0x2b95, (q15_t)0x87a7,
  (q15_t)0x2b7d, (q15_t)0x879e, (q15_t)0x2b66, (q15_t)0x8795, (q15_t)0x2b4e, (q15_t)0x878d, (q15_t)0x2b36, (q15_t)0x8784,
  (q15_t)0x2b1f, (q15_t)0x877c, (q15_t)0x2b07, (q15_t)0x8774, (q15_t)0x2aef, (q15_t)0x876b, (q15_t)0x2ad8, (q15_t)0x8763,
  (q15_t)0x2ac0, (q15_t)0x875a, (q15_t)0x2aa8, (q15_t)0x8752, (q15_t)0x2a91, (q15_t)0x874a, (q15_t)0x2a79, (q15_t)0x8741,
  (q15_t)0x2a61, (q15_t)0x8739, (q15_t)0x2a49, (q15_t)0x8731, (q15_t)0x2a32, (q15_t)0x8728, (q15_t)0x2a1a, (q15_t)0x8720,
  (q15_t)0x2a02, (q15_t)0x8718, (q15_t)0x29eb, (q15_t)0x870f, (q15_t)0x29d3, (q15_t)0x8707, (q15_t)0x29bb, (q15_t)0x86ff,
  (q15_t)0x29a3, (q15_t)0x86f7, (q15_t)0x298b, (q15_t)0x86ef, (q15_t)0x2974, (q15_t)0x86e7, (q15_t)0x295c, (q15_t)0x86de,
  (q15_t)0x2944, (q15_t)0x86d6, (q15_t)0x292c, (q15_t)0x86ce, (q15_t)0x2915, (q15_t)0x86c6, (q15_t)0x28fd, (q15_t)0x86be,
  (q15_t)0x28e5, (q15_t)0x86b6, (q15_t)0x28cd, (q15_t)0x86ae, (q15_t)0x28b5, (q15_t)0x86a6, (q15_t)0x289d, (q15_t)0x869e,
  (q15_t)0x2886, (q15_t)0x8696, (q15_t)0x286e, (q15_t)0x868e, (q15_t)0x2856, (q15_t)0x8686, (q15_t)0x283e, (q15_t)0x867e,
  (q15_t)0x2826, (q15_t)0x8676, (q15_t)0x280e, (q15_t)0x866e, (q15_t)0x27f6, (q15_t)0x8667, (q15_t)0x27df, (q15_t)0x865f,
  (q15_t)0x27c7, (q15_t)0x8657, (q15_t)0x27af, (q15_t)0x864f, (q15_t)0x2797, (q15_t)0x8647, (q15_t)0x277f, (q15_t)0x8640,
  (q15_t)0x2767, (q15_t)0x8638, (q15_t)0x274f, (q15_t)0x8630, (q15_t)0x2737, (q15_t)0x8628, (q15_t)0x271f, (q15_t)0x8621,
  (q15_t)0x2707, (q15_t)0x8619, (q15_t)0x26ef, (q15_t)0x8611, (q15_t)0x26d8, (q15_t)0x860a, (q15_t)0x26c0, (q15_t)0x8602,
  (q15_t)0x26a8, (q15_t)0x85fb, (q15_t)0x2690, (q15_t)0x85f3, (q15_t)0x2678, (q15_t)0x85eb, (q15_t)0x2660, (q15_t)0x85e4,
  (q15_t)0x2648, (q15_t)0x85dc, (q15_t)0x2630, (q15_t)0x85d5, (q15_t)0x2618, (q15_t)0x85cd, (q15_t)0x2600, (q15_t)0x85c6,
  (q15_t)0x25e8, (q15_t)0x85be, (q15_t)0x25d0, (q15_t)0x85b7, (q15_t)0x25b8, (q15_t)0x85b0, (q15_t)0x25a0, (q15_t)0x85a8,
  (q15_t)0x2588, (q15_t)0x85a1, (q15_t)0x2570, (q15_t)0x8599, (q15_t)0x2558, (q15_t)0x8592, (q15_t)0x2540, (q15_t)0x858b,
  (q15_t)0x2528, (q15_t)0x8583, (q15_t)0x250f, (q15_t)0x857c, (q15_t)0x24f7, (q15_t)0x8575, (q15_t)0x24df, (q15_t)0x856e,
  (q15_t)0x24c7, (q15_t)0x8566, (q15_t)0x24af, (q15_t)0x855f, (q15_t)0x2497, (q15_t)0x8558, (q15_t)0x247f, (q15_t)0x8551,
  (q15_t)0x2467, (q15_t)0x854a, (q15_t)0x244f, (q15_t)0x8543, (q15_t)0x2437, (q15_t)0x853b, (q15_t)0x241f, (q15_t)0x8534,
  (q15_t)0x2407, (q15_t)0x852d, (q15_t)0x23ee, (q15_t)0x8526, (q15_t)0x23d6, (q15_t)0x851f, (q15_t)0x23be, (q15_t)0x8518,
  (q15_t)0x23a6, (q15_t)0x8511, (q15_t)0x238e, (q15_t)0x850a, (q15_t)0x2376, (q15_t)0x8503, (q15_t)0x235e, (q15_t)0x84fc,
  (q15_t)0x2345, (q15_t)0x84f5, (q15_t)0x232d, (q15_t)0x84ee, (q15_t)0x2315, (q15_t)0x84e7, (q15_t)0x22fd, (q15_t)0x84e1,
  (q15_t)0x22e5, (q15_t)0x84da, (q15_t)0x22cd, (q15_t)0x84d3, (q15_t)0x22b4, (q15_t)0x84cc, (q15_t)0x229c, (q15_t)0x84c5,
  (q15_t)0x2284, (q15_t)0x84be, (q15_t)0x226c, (q15_t)0x84b8, (q15_t)0x2254, (q15_t)0x84b1, (q15_t)0x223b, (q15_t)0x84aa,
  (q15_t)0x2223, (q15_t)0x84a3, (q15_t)0x220b, (q15_t)0x849d, (q15_t)0x21f3, (q15_t)0x8496, (q15_t)0x21da, (q15_t)0x848f,
  (q15_t)0x21c2, (q15_t)0x8489, (q15_t)0x21aa, (q15_t)0x8482, (q15_t)0x2192, (q15_t)0x847c, (q15_t)0x2179, (q15_t)0x8475,
  (q15_t)0x2161, (q15_t)0x846e, (q15_t)0x2149, (q15_t)0x8468, (q15_t)0x2131, (q15_t)0x8461, (q15_t)0x2118, (q15_t)0x845b,
  (q15_t)0x2100, (q15_t)0x8454, (q15_t)0x20e8, (q15_t)0x844e, (q15_t)0x20d0, (q15_t)0x8447, (q15_t)0x20b7, (q15_t)0x8441,
  (q15_t)0x209f, (q15_t)0x843b, (q15_t)0x2087, (q15_t)0x8434, (q15_t)0x206e, (q15_t)0x842e, (q15_t)0x2056, (q15_t)0x8427,
  (q15_t)0x203e, (q15_t)0x8421, (q15_t)0x2025, (q15_t)0x841b, (q15_t)0x200d, (q15_t)0x8415, (q15_t)0x1ff5, (q15_t)0x840e,
  (q15_t)0x1fdc, (q15_t)0x8408, (q15_t)0x1fc4, (q15_t)0x8402, (q15_t)0x1fac, (q15_t)0x83fb, (q15_t)0x1f93, (q15_t)0x83f5,
  (q15_t)0x1f7b, (q15_t)0x83ef, (q15_t)0x1f63, (q15_t)0x83e9, (q15_t)0x1f4a, (q15_t)0x83e3, (q15_t)0x1f32, (q15_t)0x83dd,
  (q15_t)0x1f19, (q15_t)0x83d7, (q15_t)0x1f01, (q15_t)0x83d0, (q15_t)0x1ee9, (q15_t)0x83ca, (q15_t)0x1ed0, (q15_t)0x83c4,
  (q15_t)0x1eb8, (q15_t)0x83be, (q15_t)0x1ea0, (q15_t)0x83b8, (q15_t)0x1e87, (q15_t)0x83b2, (q15_t)0x1e6f, (q15_t)0x83ac,
  (q15_t)0x1e56, (q15_t)0x83a6, (q15_t)0x1e3e, (q15_t)0x83a0, (q15_t)0x1e25, (q15_t)0x839a, (q15_t)0x1e0d, (q15_t)0x8394,
  (q15_t)0x1df5, (q15_t)0x838f, (q15_t)0x1ddc, (q15_t)0x8389, (q15_t)0x1dc4, (q15_t)0x8383, (q15_t)0x1dab, (q15_t)0x837d,
  (q15_t)0x1d93, (q15_t)0x8377, (q15_t)0x1d7a, (q15_t)0x8371, (q15_t)0x1d62, (q15_t)0x836c, (q15_t)0x1d49, (q15_t)0x8366,
  (q15_t)0x1d31, (q15_t)0x8360, (q15_t)0x1d18, (q15_t)0x835a, (q15_t)0x1d00, (q15_t)0x8355, (q15_t)0x1ce8, (q15_t)0x834f,
  (q15_t)0x1ccf, (q15_t)0x8349, (q15_t)0x1cb7, (q15_t)0x8344, (q15_t)0x1c9e, (q15_t)0x833e, (q15_t)0x1c86, (q15_t)0x8338,
  (q15_t)0x1c6d, (q15_t)0x8333, (q15_t)0x1c55, (q15_t)0x832d, (q15_t)0x1c3c, (q15_t)0x8328, (q15_t)0x1c24, (q15_t)0x8322,
  (q15_t)0x1c0b, (q15_t)0x831d, (q15_t)0x1bf2, (q15_t)0x8317, (q15_t)0x1bda, (q15_t)0x8312, (q15_t)0x1bc1, (q15_t)0x830c,
  (q15_t)0x1ba9, (q15_t)0x8307, (q15_t)0x1b90, (q15_t)0x8301, (q15_t)0x1b78, (q15_t)0x82fc, (q15_t)0x1b5f, (q15_t)0x82f7,
  (q15_t)0x1b47, (q15_t)0x82f1, (q15_t)0x1b2e, (q15_t)0x82ec, (q15_t)0x1b16, (q15_t)0x82e7, (q15_t)0x1afd, (q15_t)0x82e1,
  (q15_t)0x1ae4, (q15_t)0x82dc, (q15_t)0x1acc, (q15_t)0x82d7, (q15_t)0x1ab3, (q15_t)0x82d1, (q15_t)0x1a9b, (q15_t)0x82cc,
  (q15_t)0x1a82, (q15_t)0x82c7, (q15_t)0x1a6a, (q15_t)0x82c2, (q15_t)0x1a51, (q15_t)0x82bd, (q15_t)0x1a38, (q15_t)0x82b7,
  (q15_t)0x1a20, (q15_t)0x82b2, (q15_t)0x1a07, (q15_t)0x82ad, (q15_t)0x19ef, (q15_t)0x82a8, (q15_t)0x19d6, (q15_t)0x82a3,
  (q15_t)0x19bd, (q15_t)0x829e, (q15_t)0x19a5, (q15_t)0x8299, (q15_t)0x198c, (q15_t)0x8294, (q15_t)0x1973, (q15_t)0x828f,
  (q15_t)0x195b, (q15_t)0x828a, (q15_t)0x1942, (q15_t)0x8285, (q15_t)0x192a, (q15_t)0x8280, (q15_t)0x1911, (q15_t)0x827b,
  (q15_t)0x18f8, (q15_t)0x8276, (q15_t)0x18e0, (q15_t)0x8271, (q15_t)0x18c7, (q15_t)0x826c, (q15_t)0x18ae, (q15_t)0x8268,
  (q15_t)0x1896, (q15_t)0x8263, (q15_t)0x187d, (q15_t)0x825e, (q15_t)0x1864, (q15_t)0x8259, (q15_t)0x184c, (q15_t)0x8254,
  (q15_t)0x1833, (q15_t)0x8250, (q15_t)0x181a, (q15_t)0x824b, (q15_t)0x1802, (q15_t)0x8246, (q15_t)0x17e9, (q15_t)0x8241,
  (q15_t)0x17d0, (q15_t)0x823d, (q15_t)0x17b7, (q15_t)0x8238, (q15_t)0x179f, (q15_t)0x8233, (q15_t)0x1786, (q15_t)0x822f,
  (q15_t)0x176d, (q15_t)0x822a, (q15_t)0x1755, (q15_t)0x8226, (q15_t)0x173c, (q15_t)0x8221, (q15_t)0x1723, (q15_t)0x821c,
  (q15_t)0x170a, (q15_t)0x8218, (q15_t)0x16f2, (q15_t)0x8213, (q15_t)0x16d9, (q15_t)0x820f, (q15_t)0x16c0, (q15_t)0x820a,
  (q15_t)0x16a8, (q15_t)0x8206, (q15_t)0x168f, (q15_t)0x8201, (q15_t)0x1676, (q15_t)0x81fd, (q15_t)0x165d, (q15_t)0x81f9,
  (q15_t)0x1645, (q15_t)0x81f4, (q15_t)0x162c, (q15_t)0x81f0, (q15_t)0x1613, (q15_t)0x81ec, (q15_t)0x15fa, (q15_t)0x81e7,
  (q15_t)0x15e2, (q15_t)0x81e3, (q15_t)0x15c9, (q15_t)0x81df, (q15_t)0x15b0, (q15_t)0x81da, (q15_t)0x1597, (q15_t)0x81d6,
  (q15_t)0x157f, (q15_t)0x81d2, (q15_t)0x1566, (q15_t)0x81ce, (q15_t)0x154d, (q15_t)0x81c9, (q15_t)0x1534, (q15_t)0x81c5,
  (q15_t)0x151b, (q15_t)0x81c1, (q15_t)0x1503, (q15_t)0x81bd, (q15_t)0x14ea, (q15_t)0x81b9, (q15_t)0x14d1, (q15_t)0x81b5,
  (q15_t)0x14b8, (q15_t)0x81b1, (q15_t)0x149f, (q15_t)0x81ad, (q15_t)0x1487, (q15_t)0x81a9, (q15_t)0x146e, (q15_t)0x81a5,
  (q15_t)0x1455, (q15_t)0x81a1, (q15_t)0x143c, (q15_t)0x819d, (q15_t)0x1423, (q15_t)0x8199, (q15_t)0x140b, (q15_t)0x8195,
  (q15_t)0x13f2, (q15_t)0x8191, (q15_t)0x13d9, (q15_t)0x818d, (q15_t)0x13c0, (q15_t)0x8189, (q15_t)0x13a7, (q15_t)0x8185,
  (q15_t)0x138e, (q15_t)0x8181, (q15_t)0x1376, (q15_t)0x817d, (q15_t)0x135d, (q15_t)0x817a, (q15_t)0x1344, (q15_t)0x8176,
  (q15_t)0x132b, (q15_t)0x8172, (q15_t)0x1312, (q15_t)0x816e, (q15_t)0x12f9, (q15_t)0x816b, (q15_t)0x12e0, (q15_t)0x8167,
  (q15_t)0x12c8, (q15_t)0x8163, (q15_t)0x12af, (q15_t)0x815f, (q15_t)0x1296, (q15_t)0x815c, (q15_t)0x127d, (q15_t)0x8158,
  (q15_t)0x1264, (q15_t)0x8155, (q15_t)0x124b, (q15_t)0x8151, (q15_t)0x1232, (q15_t)0x814d, (q15_t)0x1219, (q15_t)0x814a,
  (q15_t)0x1201, (q15_t)0x8146, (q15_t)0x11e8, (q15_t)0x8143, (q15_t)0x11cf, (q15_t)0x813f, (q15_t)0x11b6, (q15_t)0x813c,
  (q15_t)0x119d, (q15_t)0x8138, (q15_t)0x1184, (q15_t)0x8135, (q15_t)0x116b, (q15_t)0x8131, (q15_t)0x1152, (q15_t)0x812e,
  (q15_t)0x1139, (q15_t)0x812b, (q15_t)0x1121, (q15_t)0x8127, (q15_t)0x1108, (q15_t)0x8124, (q15_t)0x10ef, (q15_t)0x8121,
  (q15_t)0x10d6, (q15_t)0x811d, (q15_t)0x10bd, (q15_t)0x811a, (q15_t)0x10a4, (q15_t)0x8117, (q15_t)0x108b, (q15_t)0x8113,
  (q15_t)0x1072, (q15_t)0x8110, (q15_t)0x1059, (q15_t)0x810d, (q15_t)0x1040, (q15_t)0x810a, (q15_t)0x1027, (q15_t)0x8107,
  (q15_t)0x100e, (q15_t)0x8103, (q15_t)0xff5, (q15_t)0x8100, (q15_t)0xfdd, (q15_t)0x80fd, (q15_t)0xfc4, (q15_t)0x80fa,
  (q15_t)0xfab, (q15_t)0x80f7, (q15_t)0xf92, (q15_t)0x80f4, (q15_t)0xf79, (q15_t)0x80f1, (q15_t)0xf60, (q15_t)0x80ee,
  (q15_t)0xf47, (q15_t)0x80eb, (q15_t)0xf2e, (q15_t)0x80e8, (q15_t)0xf15, (q15_t)0x80e5, (q15_t)0xefc, (q15_t)0x80e2,
  (q15_t)0xee3, (q15_t)0x80df, (q15_t)0xeca, (q15_t)0x80dc, (q15_t)0xeb1, (q15_t)0x80d9, (q15_t)0xe98, (q15_t)0x80d6,
  (q15_t)0xe7f, (q15_t)0x80d3, (q15_t)0xe66, (q15_t)0x80d1, (q15_t)0xe4d, (q15_t)0x80ce, (q15_t)0xe34, (q15_t)0x80cb,
  (q15_t)0xe1b, (q15_t)0x80c8, (q15_t)0xe02, (q15_t)0x80c5, (q15_t)0xde9, (q15_t)0x80c3, (q15_t)0xdd0, (q15_t)0x80c0,
  (q15_t)0xdb7, (q15_t)0x80bd, (q15_t)0xd9e, (q15_t)0x80bb, (q15_t)0xd85, (q15_t)0x80b8, (q15_t)0xd6c, (q15_t)0x80b5,
  (q15_t)0xd53, (q15_t)0x80b3, (q15_t)0xd3a, (q15_t)0x80b0, (q15_t)0xd21, (q15_t)0x80ad, (q15_t)0xd08, (q15_t)0x80ab,
  (q15_t)0xcef, (q15_t)0x80a8, (q15_t)0xcd6, (q15_t)0x80a6, (q15_t)0xcbd, (q15_t)0x80a3, (q15_t)0xca4, (q15_t)0x80a1,
  (q15_t)0xc8b, (q15_t)0x809e, (q15_t)0xc72, (q15_t)0x809c, (q15_t)0xc59, (q15_t)0x8099, (q15_t)0xc40, (q15_t)0x8097,
  (q15_t)0xc27, (q15_t)0x8095, (q15_t)0xc0e, (q15_t)0x8092, (q15_t)0xbf5, (q15_t)0x8090, (q15_t)0xbdc, (q15_t)0x808e,
  (q15_t)0xbc3, (q15_t)0x808b, (q15_t)0xbaa, (q15_t)0x8089, (q15_t)0xb91, (q15_t)0x8087, (q15_t)0xb78, (q15_t)0x8084,
  (q15_t)0xb5f, (q15_t)0x8082, (q15_t)0xb46, (q15_t)0x8080, (q15_t)0xb2d, (q15_t)0x807e, (q15_t)0xb14, (q15_t)0x807b,
  (q15_t)0xafb, (q15_t)0x8079, (q15_t)0xae2, (q15_t)0x8077, (q15_t)0xac9, (q15_t)0x8075, (q15_t)0xab0, (q15_t)0x8073,
  (q15_t)0xa97, (q15_t)0x8071, (q15_t)0xa7e, (q15_t)0x806f, (q15_t)0xa65, (q15_t)0x806d, (q15_t)0xa4c, (q15_t)0x806b,
  (q15_t)0xa33, (q15_t)0x8069, (q15_t)0xa19, (q15_t)0x8067, (q15_t)0xa00, (q15_t)0x8065, (q15_t)0x9e7, (q15_t)0x8063,
  (q15_t)0x9ce, (q15_t)0x8061, (q15_t)0x9b5, (q15_t)0x805f, (q15_t)0x99c, (q15_t)0x805d, (q15_t)0x983, (q15_t)0x805b,
  (q15_t)0x96a, (q15_t)0x8059, (q15_t)0x951, (q15_t)0x8057, (q15_t)0x938, (q15_t)0x8056, (q15_t)0x91f, (q15_t)0x8054,
  (q15_t)0x906, (q15_t)0x8052, (q15_t)0x8ed, (q15_t)0x8050, (q15_t)0x8d4, (q15_t)0x804f, (q15_t)0x8bb, (q15_t)0x804d,
  (q15_t)0x8a2, (q15_t)0x804b, (q15_t)0x888, (q15_t)0x8049, (q15_t)0x86f, (q15_t)0x8048, (q15_t)0x856, (q15_t)0x8046,
  (q15_t)0x83d, (q15_t)0x8044, (q15_t)0x824, (q15_t)0x8043, (q15_t)0x80b, (q15_t)0x8041, (q15_t)0x7f2, (q15_t)0x8040,
  (q15_t)0x7d9, (q15_t)0x803e, (q15_t)0x7c0, (q15_t)0x803d, (q15_t)0x7a7, (q15_t)0x803b, (q15_t)0x78e, (q15_t)0x803a,
  (q15_t)0x775, (q15_t)0x8038, (q15_t)0x75b, (q15_t)0x8037, (q15_t)0x742, (q15_t)0x8035, (q15_t)0x729, (q15_t)0x8034,
  (q15_t)0x710, (q15_t)0x8032, (q15_t)0x6f7, (q15_t)0x8031, (q15_t)0x6de, (q15_t)0x8030, (q15_t)0x6c5, (q15_t)0x802e,
  (q15_t)0x6ac, (q15_t)0x802d, (q15_t)0x693, (q15_t)0x802c, (q15_t)0x67a, (q15_t)0x802a, (q15_t)0x660, (q15_t)0x8029,
  (q15_t)0x647, (q15_t)0x8028, (q15_t)0x62e, (q15_t)0x8027, (q15_t)0x615, (q15_t)0x8026, (q15_t)0x5fc, (q15_t)0x8024,
  (q15_t)0x5e3, (q15_t)0x8023, (q15_t)0x5ca, (q15_t)0x8022, (q15_t)0x5b1, (q15_t)0x8021, (q15_t)0x598, (q15_t)0x8020,
  (q15_t)0x57f, (q15_t)0x801f, (q15_t)0x565, (q15_t)0x801e, (q15_t)0x54c, (q15_t)0x801d, (q15_t)0x533, (q15_t)0x801c,
  (q15_t)0x51a, (q15_t)0x801b, (q15_t)0x501, (q15_t)0x801a, (q15_t)0x4e8, (q15_t)0x8019, (q15_t)0x4cf, (q15_t)0x8018,
  (q15_t)0x4b6, (q15_t)0x8017, (q15_t)0x49c, (q15_t)0x8016, (q15_t)0x483, (q15_t)0x8015, (q15_t)0x46a, (q15_t)0x8014,
  (q15_t)0x451, (q15_t)0x8013, (q15_t)0x438, (q15_t)0x8012, (q15_t)0x41f, (q15_t)0x8012, (q15_t)0x406, (q15_t)0x8011,
  (q15_t)0x3ed, (q15_t)0x8010, (q15_t)0x3d4, (q15_t)0x800f, (q15_t)0x3ba, (q15_t)0x800e, (q15_t)0x3a1, (q15_t)0x800e,
  (q15_t)0x388, (q15_t)0x800d, (q15_t)0x36f, (q15_t)0x800c, (q15_t)0x356, (q15_t)0x800c, (q15_t)0x33d, (q15_t)0x800b,
  (q15_t)0x324, (q15_t)0x800a, (q15_t)0x30b, (q15_t)0x800a, (q15_t)0x2f1, (q15_t)0x8009, (q15_t)0x2d8, (q15_t)0x8009,
  (q15_t)0x2bf, (q15_t)0x8008, (q15_t)0x2a6, (q15_t)0x8008, (q15_t)0x28d, (q15_t)0x8007, (q15_t)0x274, (q15_t)0x8007,
  (q15_t)0x25b, (q15_t)0x8006, (q15_t)0x242, (q15_t)0x8006, (q15_t)0x228, (q15_t)0x8005, (q15_t)0x20f, (q15_t)0x8005,
  (q15_t)0x1f6, (q15_t)0x8004, (q15_t)0x1dd, (q15_t)0x8004, (q15_t)0x1c4, (q15_t)0x8004, (q15_t)0x1ab, (q15_t)0x8003,
  (q15_t)0x192, (q15_t)0x8003, (q15_t)0x178, (q15_t)0x8003, (q15_t)0x15f, (q15_t)0x8002, (q15_t)0x146, (q15_t)0x8002,
  (q15_t)0x12d, (q15_t)0x8002, (q15_t)0x114, (q15_t)0x8002, (q15_t)0xfb, (q15_t)0x8001, (q15_t)0xe2, (q15_t)0x8001,
  (q15_t)0xc9, (q15_t)0x8001, (q15_t)0xaf, (q15_t)0x8001, (q15_t)0x96, (q15_t)0x8001, (q15_t)0x7d, (q15_t)0x8001,
  (q15_t)0x64, (q15_t)0x8001, (q15_t)0x4b, (q15_t)0x8001, (q15_t)0x32, (q15_t)0x8001, (q15_t)0x19, (q15_t)0x8001
};

static const q15_t ALIGN4 WeightsQ15_8192[16384] = {
  (q15_t)0x7fff, (q15_t)0x0, (q15_t)0x7fff, (q15_t)0xfffa, (q15_t)0x7fff, (q15_t)0xfff4, (q15_t)0x7fff, (q15_t)0xffee,
  (q15_t)0x7fff, (q15_t)0xffe7, (q15_t)0x7fff, (q15_t)0xffe1, (q15_t)0x7fff, (q15_t)0xffdb, (q15_t)0x7fff, (q15_t)0xffd5,
  (q15_t)0x7fff, (q15_t)0xffce, (q15_t)0x7fff, (q15_t)0xffc8, (q15_t)0x7fff, (q15_t)0xffc2, (q15_t)0x7fff, (q15_t)0xffbb,
  (q15_t)0x7fff, (q15_t)0xffb5, (q15_t)0x7fff, (q15_t)0xffaf, (q15_t)0x7fff, (q15_t)0xffa9, (q15_t)0x7fff, (q15_t)0xffa2,
  (q15_t)0x7fff, (q15_t)0xff9c, (q15_t)0x7fff, (q15_t)0xff96, (q15_t)0x7fff, (q15_t)0xff8f, (q15_t)0x7fff, (q15_t)0xff89,
  (q15_t)0x7fff, (q15_t)0xff83, (q15_t)0x7fff, (q15_t)0xff7d, (q15_t)0x7fff, (q15_t)0xff76, (q15_t)0x7fff, (q15_t)0xff70,
  (q15_t)0x7fff, (q15_t)0xff6a, (q15_t)0x7fff, (q15_t)0xff63, (q15_t)0x7fff, (q15_t)0xff5d, (q15_t)0x7fff, (q15_t)0xff57,
  (q15_t)0x7fff, (q15_t)0xff51, (q15_t)0x7fff, (q15_t)0xff4a, (q15_t)0x7fff, (q15_t)0xff44, (q15_t)0x7fff, (q15_t)0xff3e,
  (q15_t)0x7fff, (q15_t)0xff37, (q15_t)0x7fff, (q15_t)0xff31, (q15_t)0x7fff, (q15_t)0xff2b, (q15_t)0x7fff, (q15_t)0xff25,
  (q15_t)0x7fff, (q15_t)0xff1e, (q15_t)0x7fff, (q15_t)0xff18, (q15_t)0x7fff, (q15_t)0xff12, (q15_t)0x7fff, (q15_t)0xff0b,
  (q15_t)0x7fff, (q15_t)0xff05, (q15_t)0x7ffe, (q15_t)0xfeff, (q15_t)0x7ffe, (q15_t)0xfef9, (q15_t)0x7ffe, (q15_t)0xfef2,
  (q15_t)0x7ffe, (q15_t)0xfeec, (q15_t)0x7ffe, (q15_t)0xfee6, (q15_t)0x7ffe, (q15_t)0xfedf, (q15_t)0x7ffe, (q15_t)0xfed9,
  (q15_t)0x7ffe, (q15_t)0xfed3, (q15_t)0x7ffe, (q15_t)0xfecd, (q15_t)0x7ffe, (q15_t)0xfec6, (q15_t)0x7ffe, (q15_t)0xfec0,
  (q15_t)0x7ffe, (q15_t)0xfeba, (q15_t)0x7ffe, (q15_t)0xfeb3, (q15_t)0x7ffe, (q15_t)0xfead, (q15_t)0x7ffe, (q15_t)0xfea7,
  (q15_t)0x7ffe, (q15_t)0xfea1, (q15_t)0x7ffe, (q15_t)0xfe9a, (q15_t)0x7ffd, (q15_t)0xfe94, (q15_t)0x7ffd, (q15_t)0xfe8e,
  (q15_t)0x7ffd, (q15_t)0xfe88, (q15_t)0x7ffd, (q15_t)0xfe81, (q15_t)0x7ffd, (q15_t)0xfe7b, (q15_t)0x7ffd, (q15_t)0xfe75,
  (q15_t)0x7ffd, (q15_t)0xfe6e, (q15_t)0x7ffd, (q15_t)0xfe68, (q15_t)0x7ffd, (q15_t)0xfe62, (q15_t)0x7ffd, (q15_t)0xfe5c,
  (q15_t)0x7ffd, (q15_t)0xfe55, (q15_t)0x7ffd, (q15_t)0xfe4f, (q15_t)0x7ffd, (q15_t)0xfe49, (q15_t)0x7ffc, (q15_t)0xfe42,
  (q15_t)0x7ffc, (q15_t)0xfe3c, (q15_t)0x7ffc, (q15_t)0xfe36, (q15_t)0x7ffc, (q15_t)0xfe30, (q15_t)0x7ffc, (q15_t)0xfe29,
  (q15_t)0x7ffc, (q15_t)0xfe23, (q15_t)0x7ffc, (q15_t)0xfe1d, (q15_t)0x7ffc, (q15_t)0xfe16, (q15_t)0x7ffc, (q15_t)0xfe10,
  (q15_t)0x7ffc, (q15_t)0xfe0a, (q15_t)0x7ffc, (q15_t)0xfe04, (q15_t)0x7ffb, (q15_t)0xfdfd, (q15_t)0x7ffb, (q15_t)0xfdf7,
  (q15_t)0x7ffb, (q15_t)0xfdf1, (q15_t)0x7ffb, (q15_t)0xfdea, (q15_t)0x7ffb, (q15_t)0xfde4, (q15_t)0x7ffb, (q15_t)0xfdde,
  (q15_t)0x7ffb, (q15_t)0xfdd8, (q15_t)0x7ffb, (q15_t)0xfdd1, (q15_t)0x7ffb, (q15_t)0xfdcb, (q15_t)0x7ffb, (q15_t)0xfdc5,
  (q15_t)0x7ffa, (q15_t)0xfdbe, (q15_t)0x7ffa, (q15_t)0xfdb8, (q15_t)0x7ffa, (q15_t)0xfdb2, (q15_t)0x7ffa, (q15_t)0xfdac,
  (q15_t)0x7ffa, (q15_t)0xfda5, (q15_t)0x7ffa, (q15_t)0xfd9f, (q15_t)0x7ffa, (q15_t)0xfd99, (q15_t)0x7ffa, (q15_t)0xfd93,
  (q15_t)0x7ff9, (q15_t)0xfd8c, (q15_t)0x7ff9, (q15_t)0xfd86, (q15_t)0x7ff9, (q15_t)0xfd80, (q15_t)0x7ff9, (q15_t)0xfd79,
  (q15_t)0x7ff9, (q15_t)0xfd73, (q15_t)0x7ff9, (q15_t)0xfd6d, (q15_t)0x7ff9, (q15_t)0xfd67, (q15_t)0x7ff9, (q15_t)0xfd60,
  (q15_t)0x7ff8, (q15_t)0xfd5a, (q15_t)0x7ff8, (q15_t)0xfd54, (q15_t)0x7ff8, (q15_t)0xfd4d, (q15_t)0x7ff8, (q15_t)0xfd47,
  (q15_t)0x7ff8, (q15_t)0xfd41, (q15_t)0x7ff8, (q15_t)0xfd3b, (q15_t)0x7ff8, (q15_t)0xfd34, (q15_t)0x7ff8, (q15_t)0xfd2e,
  (q15_t)0x7ff7, (q15_t)0xfd28, (q15_t)0x7ff7, (q15_t)0xfd21, (q15_t)0x7ff7, (q15_t)0xfd1b, (q15_t)0x7ff7, (q15_t)0xfd15,
  (q15_t)0x7ff7, (q15_t)0xfd0f, (q15_t)0x7ff7, (q15_t)0xfd08, (q15_t)0x7ff7, (q15_t)0xfd02, (q15_t)0x7ff6, (q15_t)0xfcfc,
  (q15_t)0x7ff6, (q15_t)0xfcf5, (q15_t)0x7ff6, (q15_t)0xfcef, (q15_t)0x7ff6, (q15_t)0xfce9, (q15_t)0x7ff6, (q15_t)0xfce3,
  (q15_t)0x7ff6, (q15_t)0xfcdc, (q15_t)0x7ff5, (q15_t)0xfcd6, (q15_t)0x7ff5, (q15_t)0xfcd0, (q15_t)0x7ff5, (q15_t)0xfcc9,
  (q15_t)0x7ff5, (q15_t)0xfcc3, (q15_t)0x7ff5, (q15_t)0xfcbd, (q15_t)0x7ff5, (q15_t)0xfcb7, (q15_t)0x7ff5, (q15_t)0xfcb0,
  (q15_t)0x7ff4, (q15_t)0xfcaa, (q15_t)0x7ff4, (q15_t)0xfca4, (q15_t)0x7ff4, (q15_t)0xfc9e, (q15_t)0x7ff4, (q15_t)0xfc97,
  (q15_t)0x7ff4, (q15_t)0xfc91, (q15_t)0x7ff4, (q15_t)0xfc8b, (q15_t)0x7ff3, (q15_t)0xfc84, (q15_t)0x7ff3, (q15_t)0xfc7e,
  (q15_t)0x7ff3, (q15_t)0xfc78, (q15_t)0x7ff3, (q15_t)0xfc72, (q15_t)0x7ff3, (q15_t)0xfc6b, (q15_t)0x7ff2, (q15_t)0xfc65,
  (q15_t)0x7ff2, (q15_t)0xfc5f, (q15_t)0x7ff2, (q15_t)0xfc58, (q15_t)0x7ff2, (q15_t)0xfc52, (q15_t)0x7ff2, (q15_t)0xfc4c,
  (q15_t)0x7ff2, (q15_t)0xfc46, (q15_t)0x7ff1, (q15_t)0xfc3f, (q15_t)0x7ff1, (q15_t)0xfc39, (q15_t)0x7ff1, (q15_t)0xfc33,
  (q15_t)0x7ff1, (q15_t)0xfc2c, (q15_t)0x7ff1, (q15_t)0xfc26, (q15_t)0x7ff0, (q15_t)0xfc20, (q15_t)0x7ff0, (q15_t)0xfc1a,
  (q15_t)0x7ff0, (q15_t)0xfc13, (q15_t)0x7ff0, (q15_t)0xfc0d, (q15_t)0x7ff0, (q15_t)0xfc07, (q15_t)0x7fef, (q15_t)0xfc01,
  (q15_t)0x7fef, (q15_t)0xfbfa, (q15_t)0x7fef, (q15_t)0xfbf4, (q15_t)0x7fef, (q15_t)0xfbee, (q15_t)0x7fef, (q15_t)0xfbe7,
  (q15_t)0x7fee, (q15_t)0xfbe1, (q15_t)0x7fee, (q15_t)0xfbdb, (q15_t)0x7fee, (q15_t)0xfbd5, (q15_t)0x7fee, (q15_t)0xfbce,
  (q15_t)0x7fee, (q15_t)0xfbc8, (q15_t)0x7fed, (q15_t)0xfbc2, (q15_t)0x7fed, (q15_t)0xfbbb, (q15_t)0x7fed, (q15_t)0xfbb5,
  (q15_t)0x7fed, (q15_t)0xfbaf, (q15_t)0x7fed, (q15_t)0xfba9, (q15_t)0x7fec, (q15_t)0xfba2, (q15_t)0x7fec, (q15_t)0xfb9c,
  (q15_t)0x7fec, (q15_t)0xfb96, (q15_t)0x7fec, (q15_t)0xfb8f, (q15_t)0x7fec, (q15_t)0xfb89, (q15_t)0x7feb, (q15_t)0xfb83,
  (q15_t)0x7feb, (q15_t)0xfb7d, (q15_t)0x7feb, (q15_t)0xfb76, (q15_t)0x7feb, (q15_t)0xfb70, (q15_t)0x7fea, (q15_t)0xfb6a,
  (q15_t)0x7fea, (q15_t)0xfb64, (q15_t)0x7fea, (q15_t)0xfb5d, (q15_t)0x7fea, (q15_t)0xfb57, (q15_t)0x7fea, (q15_t)0xfb51,
  (q15_t)0x7fe9, (q15_t)0xfb4a, (q15_t)0x7fe9, (q15_t)0xfb44, (q15_t)0x7fe9, (q15_t)0xfb3e, (q15_t)0x7fe9, (q15_t)0xfb38,
  (q15_t)0x7fe8, (q15_t)0xfb31, (q15_t)0x7fe8, (q15_t)0xfb2b, (q15_t)0x7fe8, (q15_t)0xfb25, (q15_t)0x7fe8, (q15_t)0xfb1e,
  (q15_t)0x7fe7, (q15_t)0xfb18, (q15_t)0x7fe7, (q15_t)0xfb12, (q15_t)0x7fe7, (q15_t)0xfb0c, (q15_t)0x7fe7, (q15_t)0xfb05,
  (q15_t)0x7fe6, (q15_t)0xfaff, (q15_t)0x7fe6, (q15_t)0xfaf9, (q15_t)0x7fe6, (q15_t)0xfaf3, (q15_t)0x7fe6, (q15_t)0xfaec,
  (q15_t)0x7fe5, (q15_t)0xfae6, (q15_t)0x7fe5, (q15_t)0xfae0, (q15_t)0x7fe5, (q15_t)0xfad9, (q15_t)0x7fe5, (q15_t)0xfad3,
  (q15_t)0x7fe4, (q15_t)0xfacd, (q15_t)0x7fe4, (q15_t)0xfac7, (q15_t)0x7fe4, (q15_t)0xfac0, (q15_t)0x7fe4, (q15_t)0xfaba,
  (q15_t)0x7fe3, (q15_t)0xfab4, (q15_t)0x7fe3, (q15_t)0xfaad, (q15_t)0x7fe3, (q15_t)0xfaa7, (q15_t)0x7fe3, (q15_t)0xfaa1,
  (q15_t)0x7fe2, (q15_t)0xfa9b, (q15_t)0x7fe2, (q15_t)0xfa94, (q15_t)0x7fe2, (q15_t)0xfa8e, (q15_t)0x7fe2, (q15_t)0xfa88,
  (q15_t)0x7fe1, (q15_t)0xfa81, (q15_t)0x7fe1, (q15_t)0xfa7b, (q15_t)0x7fe1, (q15_t)0xfa75, (q15_t)0x7fe0, (q15_t)0xfa6f,
  (q15_t)0x7fe0, (q15_t)0xfa68, (q15_t)0x7fe0, (q15_t)0xfa62, (q15_t)0x7fe0, (q15_t)0xfa5c, (q15_t)0x7fdf, (q15_t)0xfa56,
  (q15_t)0x7fdf, (q15_t)0xfa4f, (q15_t)0x7fdf, (q15_t)0xfa49, (q15_t)0x7fdf, (q15_t)0xfa43, (q15_t)0x7fde, (q15_t)0xfa3c,
  (q15_t)0x7fde, (q15_t)0xfa36, (q15_t)0x7fde, (q15_t)0xfa30, (q15_t)0x7fdd, (q15_t)0xfa2a, (q15_t)0x7fdd, (q15_t)0xfa23,
  (q15_t)0x7fdd, (q15_t)0xfa1d, (q15_t)0x7fdd, (q15_t)0xfa17, (q15_t)0x7fdc, (q15_t)0xfa11, (q15_t)0x7fdc, (q15_t)0xfa0a,
  (q15_t)0x7fdc, (q15_t)0xfa04, (q15_t)0x7fdb, (q15_t)0xf9fe, (q15_t)0x7fdb, (q15_t)0xf9f7, (q15_t)0x7fdb, (q15_t)0xf9f1,
  (q15_t)0x7fda, (q15_t)0xf9eb, (q15_t)0x7fda, (q15_t)0xf9e5, (q15_t)0x7fda, (q15_t)0xf9de, (q15_t)0x7fda, (q15_t)0xf9d8,
  (q15_t)0x7fd9, (q15_t)0xf9d2, (q15_t)0x7fd9, (q15_t)0xf9cb, (q15_t)0x7fd9, (q15_t)0xf9c5, (q15_t)0x7fd8, (q15_t)0xf9bf,
  (q15_t)0x7fd8, (q15_t)0xf9b9, (q15_t)0x7fd8, (q15_t)0xf9b2, (q15_t)0x7fd7, (q15_t)0xf9ac, (q15_t)0x7fd7, (q15_t)0xf9a6,
  (q15_t)0x7fd7, (q15_t)0xf9a0, (q15_t)0x7fd6, (q15_t)0xf999, (q15_t)0x7fd6, (q15_t)0xf993, (q15_t)0x7fd6, (q15_t)0xf98d,
  (q15_t)0x7fd6, (q15_t)0xf986, (q15_t)0x7fd5, (q15_t)0xf980, (q15_t)0x7fd5, (q15_t)0xf97a, (q15_t)0x7fd5, (q15_t)0xf974,
  (q15_t)0x7fd4, (q15_t)0xf96d, (q15_t)0x7fd4, (q15_t)0xf967, (q15_t)0x7fd4, (q15_t)0xf961, (q15_t)0x7fd3, (q15_t)0xf95b,
  (q15_t)0x7fd3, (q15_t)0xf954, (q15_t)0x7fd3, (q15_t)0xf94e, (q15_t)0x7fd2, (q15_t)0xf948, (q15_t)0x7fd2, (q15_t)0xf941,
  (q15_t)0x7fd2, (q15_t)0xf93b, (q15_t)0x7fd1, (q15_t)0xf935, (q15_t)0x7fd1, (q15_t)0xf92f, (q15_t)0x7fd1, (q15_t)0xf928,
  (q15_t)0x7fd0, (q15_t)0xf922, (q15_t)0x7fd0, (q15_t)0xf91c, (q15_t)0x7fd0, (q15_t)0xf916, (q15_t)0x7fcf, (q15_t)0xf90f,
  (q15_t)0x7fcf, (q15_t)0xf909, (q15_t)0x7fcf, (q15_t)0xf903, (q15_t)0x7fce, (q15_t)0xf8fc, (q15_t)0x7fce, (q15_t)0xf8f6,
  (q15_t)0x7fce, (q15_t)0xf8f0, (q15_t)0x7fcd, (q15_t)0xf8ea, (q15_t)0x7fcd, (q15_t)0xf8e3, (q15_t)0x7fcd, (q15_t)0xf8dd,
  (q15_t)0x7fcc, (q15_t)0xf8d7, (q15_t)0x7fcc, (q15_t)0xf8d0, (q15_t)0x7fcb, (q15_t)0xf8ca, (q15_t)0x7fcb, (q15_t)0xf8c4,
  (q15_t)0x7fcb, (q15_t)0xf8be, (q15_t)0x7fca, (q15_t)0xf8b7, (q15_t)0x7fca, (q15_t)0xf8b1, (q15_t)0x7fca, (q15_t)0xf8ab,
  (q15_t)0x7fc9, (q15_t)0xf8a5, (q15_t)0x7fc9, (q15_t)0xf89e, (q15_t)0x7fc9, (q15_t)0xf898, (q15_t)0x7fc8, (q15_t)0xf892,
  (q15_t)0x7fc8, (q15_t)0xf88b, (q15_t)0x7fc7, (q15_t)0xf885, (q15_t)0x7fc7, (q15_t)0xf87f, (q15_t)0x7fc7, (q15_t)0xf879,
  (q15_t)0x7fc6, (q15_t)0xf872, (q15_t)0x7fc6, (q15_t)0xf86c, (q15_t)0x7fc6, (q15_t)0xf866, (q15_t)0x7fc5, (q15_t)0xf860,
  (q15_t)0x7fc5, (q15_t)0xf859, (q15_t)0x7fc5, (q15_t)0xf853, (q15_t)0x7fc4, (q15_t)0xf84d, (q15_t)0x7fc4, (q15_t)0xf846,
  (q15_t)0x7fc3, (q15_t)0xf840, (q15_t)0x7fc3, (q15_t)0xf83a, (q15_t)0x7fc3, (q15_t)0xf834, (q15_t)0x7fc2, (q15_t)0xf82d,
  (q15_t)0x7fc2, (q15_t)0xf827, (q15_t)0x7fc1, (q15_t)0xf821, (q15_t)0x7fc1, (q15_t)0xf81b, (q15_t)0x7fc1, (q15_t)0xf814,
  (q15_t)0x7fc0, (q15_t)0xf80e, (q15_t)0x7fc0, (q15_t)0xf808, (q15_t)0x7fc0, (q15_t)0xf802, (q15_t)0x7fbf, (q15_t)0xf7fb,
  (q15_t)0x7fbf, (q15_t)0xf7f5, (q15_t)0x7fbe, (q15_t)0xf7ef, (q15_t)0x7fbe, (q15_t)0xf7e8, (q15_t)0x7fbe, (q15_t)0xf7e2,
  (q15_t)0x7fbd, (q15_t)0xf7dc, (q15_t)0x7fbd, (q15_t)0xf7d6, (q15_t)0x7fbc, (q15_t)0xf7cf, (q15_t)0x7fbc, (q15_t)0xf7c9,
  (q15_t)0x7fbc, (q15_t)0xf7c3, (q15_t)0x7fbb, (q15_t)0xf7bd, (q15_t)0x7fbb, (q15_t)0xf7b6, (q15_t)0x7fba, (q15_t)0xf7b0,
  (q15_t)0x7fba, (q15_t)0xf7aa, (q15_t)0x7fb9, (q15_t)0xf7a3, (q15_t)0x7fb9, (q15_t)0xf79d, (q15_t)0x7fb9, (q15_t)0xf797,
  (q15_t)0x7fb8, (q15_t)0xf791, (q15_t)0x7fb8, (q15_t)0xf78a, (q15_t)0x7fb7, (q15_t)0xf784, (q15_t)0x7fb7, (q15_t)0xf77e,
  (q15_t)0x7fb7, (q15_t)0xf778, (q15_t)0x7fb6, (q15_t)0xf771, (q15_t)0x7fb6, (q15_t)0xf76b, (q15_t)0x7fb5, (q15_t)0xf765,
  (q15_t)0x7fb5, (q15_t)0xf75e, (q15_t)0x7fb4, (q15_t)0xf758, (q15_t)0x7fb4, (q15_t)0xf752, (q15_t)0x7fb4, (q15_t)0xf74c,
  (q15_t)0x7fb3, (q15_t)0xf745, (q15_t)0x7fb3, (q15_t)0xf73f, (q15_t)0x7fb2, (q15_t)0xf739, (q15_t)0x7fb2, (q15_t)0xf733,
  (q15_t)0x7fb1, (q15_t)0xf72c, (q15_t)0x7fb1, (q15_t)0xf726, (q15_t)0x7fb1, (q15_t)0xf720, (q15_t)0x7fb0, (q15_t)0xf71a,
  (q15_t)0x7fb0, (q15_t)0xf713, (q15_t)0x7faf, (q15_t)0xf70d, (q15_t)0x7faf, (q15_t)0xf707, (q15_t)0x7fae, (q15_t)0xf700,
  (q15_t)0x7fae, (q15_t)0xf6fa, (q15_t)0x7fae, (q15_t)0xf6f4, (q15_t)0x7fad, (q15_t)0xf6ee, (q15_t)0x7fad, (q15_t)0xf6e7,
  (q15_t)0x7fac, (q15_t)0xf6e1, (q15_t)0x7fac, (q15_t)0xf6db, (q15_t)0x7fab, (q15_t)0xf6d5, (q15_t)0x7fab, (q15_t)0xf6ce,
  (q15_t)0x7faa, (q15_t)0xf6c8, (q15_t)0x7faa, (q15_t)0xf6c2, (q15_t)0x7fa9, (q15_t)0xf6bc, (q15_t)0x7fa9, (q15_t)0xf6b5,
  (q15_t)0x7fa9, (q15_t)0xf6af, (q15_t)0x7fa8, (q15_t)0xf6a9, (q15_t)0x7fa8, (q15_t)0xf6a2, (q15_t)0x7fa7, (q15_t)0xf69c,
  (q15_t)0x7fa7, (q15_t)0xf696, (q15_t)0x7fa6, (q15_t)0xf690, (q15_t)0x7fa6, (q15_t)0xf689, (q15_t)0x7fa5, (q15_t)0xf683,
  (q15_t)0x7fa5, (q15_t)0xf67d, (q15_t)0x7fa4, (q15_t)0xf677, (q15_t)0x7fa4, (q15_t)0xf670, (q15_t)0x7fa3, (q15_t)0xf66a,
  (q15_t)0x7fa3, (q15_t)0xf664, (q15_t)0x7fa3, (q15_t)0xf65e, (q15_t)0x7fa2, (q15_t)0xf657, (q15_t)0x7fa2, (q15_t)0xf651,
  (q15_t)0x7fa1, (q15_t)0xf64b, (q15_t)0x7fa1, (q15_t)0xf644, (q15_t)0x7fa0, (q15_t)0xf63e, (q15_t)0x7fa0, (q15_t)0xf638,
  (q15_t)0x7f9f, (q15_t)0xf632, (q15_t)0x7f9f, (q15_t)0xf62b, (q15_t)0x7f9e, (q15_t)0xf625, (q15_t)0x7f9e, (q15_t)0xf61f,
  (q15_t)0x7f9d, (q15_t)0xf619, (q15_t)0x7f9d, (q15_t)0xf612, (q15_t)0x7f9c, (q15_t)0xf60c, (q15_t)0x7f9c, (q15_t)0xf606,
  (q15_t)0x7f9b, (q15_t)0xf600, (q15_t)0x7f9b, (q15_t)0xf5f9, (q15_t)0x7f9a, (q15_t)0xf5f3, (q15_t)0x7f9a, (q15_t)0xf5ed,
  (q15_t)0x7f99, (q15_t)0xf5e7, (q15_t)0x7f99, (q15_t)0xf5e0, (q15_t)0x7f98, (q15_t)0xf5da, (q15_t)0x7f98, (q15_t)0xf5d4,
  (q15_t)0x7f97, (q15_t)0xf5cd, (q15_t)0x7f97, (q15_t)0xf5c7, (q15_t)0x7f96, (q15_t)0xf5c1, (q15_t)0x7f96, (q15_t)0xf5bb,
  (q15_t)0x7f95, (q15_t)0xf5b4, (q15_t)0x7f95, (q15_t)0xf5ae, (q15_t)0x7f94, (q15_t)0xf5a8, (q15_t)0x7f94, (q15_t)0xf5a2,
  (q15_t)0x7f93, (q15_t)0xf59b, (q15_t)0x7f93, (q15_t)0xf595, (q15_t)0x7f92, (q15_t)0xf58f, (q15_t)0x7f92, (q15_t)0xf589,
  (q15_t)0x7f91, (q15_t)0xf582, (q15_t)0x7f91, (q15_t)0xf57c, (q15_t)0x7f90, (q15_t)0xf576, (q15_t)0x7f90, (q15_t)0xf570,
  (q15_t)0x7f8f, (q15_t)0xf569, (q15_t)0x7f8f, (q15_t)0xf563, (q15_t)0x7f8e, (q15_t)0xf55d, (q15_t)0x7f8e, (q15_t)0xf556,
  (q15_t)0x7f8d, (q15_t)0xf550, (q15_t)0x7f8d, (q15_t)0xf54a, (q15_t)0x7f8c, (q15_t)0xf544, (q15_t)0x7f8b, (q15_t)0xf53d,
  (q15_t)0x7f8b, (q15_t)0xf537, (q15_t)0x7f8a, (q15_t)0xf531, (q15_t)0x7f8a, (q15_t)0xf52b, (q15_t)0x7f89, (q15_t)0xf524,
  (q15_t)0x7f89, (q15_t)0xf51e, (q15_t)0x7f88, (q15_t)0xf518, (q15_t)0x7f88, (q15_t)0xf512, (q15_t)0x7f87, (q15_t)0xf50b,
  (q15_t)0x7f87, (q15_t)0xf505, (q15_t)0x7f86, (q15_t)0xf4ff, (q15_t)0x7f86, (q15_t)0xf4f9, (q15_t)0x7f85, (q15_t)0xf4f2,
  (q15_t)0x7f85, (q15_t)0xf4ec, (q15_t)0x7f84, (q15_t)0xf4e6, (q15_t)0x7f83, (q15_t)0xf4e0, (q15_t)0x7f83, (q15_t)0xf4d9,
  (q15_t)0x7f82, (q15_t)0xf4d3, (q15_t)0x7f82, (q15_t)0xf4cd, (q15_t)0x7f81, (q15_t)0xf4c6, (q15_t)0x7f81, (q15_t)0xf4c0,
  (q15_t)0x7f80, (q15_t)0xf4ba, (q15_t)0x7f80, (q15_t)0xf4b4, (q15_t)0x7f7f, (q15_t)0xf4ad, (q15_t)0x7f7e, (q15_t)0xf4a7,
  (q15_t)0x7f7e, (q15_t)0xf4a1, (q15_t)0x7f7d, (q15_t)0xf49b, (q15_t)0x7f7d, (q15_t)0xf494, (q15_t)0x7f7c, (q15_t)0xf48e,
  (q15_t)0x7f7c, (q15_t)0xf488, (q15_t)0x7f7b, (q15_t)0xf482, (q15_t)0x7f7b, (q15_t)0xf47b, (q15_t)0x7f7a, (q15_t)0xf475,
  (q15_t)0x7f79, (q15_t)0xf46f, (q15_t)0x7f79, (q15_t)0xf469, (q15_t)0x7f78, (q15_t)0xf462, (q15_t)0x7f78, (q15_t)0xf45c,
  (q15_t)0x7f77, (q15_t)0xf456, (q15_t)0x7f77, (q15_t)0xf450, (q15_t)0x7f76, (q15_t)0xf449, (q15_t)0x7f75, (q15_t)0xf443,
  (q15_t)0x7f75, (q15_t)0xf43d, (q15_t)0x7f74, (q15_t)0xf437, (q15_t)0x7f74, (q15_t)0xf430, (q15_t)0x7f73, (q15_t)0xf42a,
  (q15_t)0x7f72, (q15_t)0xf424, (q15_t)0x7f72, (q15_t)0xf41e, (q15_t)0x7f71, (q15_t)0xf417, (q15_t)0x7f71, (q15_t)0xf411,
  (q15_t)0x7f70, (q15_t)0xf40b, (q15_t)0x7f70, (q15_t)0xf405, (q15_t)0x7f6f, (q15_t)0xf3fe, (q15_t)0x7f6e, (q15_t)0xf3f8,
  (q15_t)0x7f6e, (q15_t)0xf3f2, (q15_t)0x7f6d, (q15_t)0xf3ec, (q15_t)0x7f6d, (q15_t)0xf3e5, (q15_t)0x7f6c, (q15_t)0xf3df,
  (q15_t)0x7f6b, (q15_t)0xf3d9, (q15_t)0x7f6b, (q15_t)0xf3d2, (q15_t)0x7f6a, (q15_t)0xf3cc, (q15_t)0x7f6a, (q15_t)0xf3c6,
  (q15_t)0x7f69, (q15_t)0xf3c0, (q15_t)0x7f68, (q15_t)0xf3b9, (q15_t)0x7f68, (q15_t)0xf3b3, (q15_t)0x7f67, (q15_t)0xf3ad,
  (q15_t)0x7f67, (q15_t)0xf3a7, (q15_t)0x7f66, (q15_t)0xf3a0, (q15_t)0x7f65, (q15_t)0xf39a, (q15_t)0x7f65, (q15_t)0xf394,
  (q15_t)0x7f64, (q15_t)0xf38e, (q15_t)0x7f64, (q15_t)0xf387, (q15_t)0x7f63, (q15_t)0xf381, (q15_t)0x7f62, (q15_t)0xf37b,
  (q15_t)0x7f62, (q15_t)0xf375, (q15_t)0x7f61, (q15_t)0xf36e, (q15_t)0x7f60, (q15_t)0xf368, (q15_t)0x7f60, (q15_t)0xf362,
  (q15_t)0x7f5f, (q15_t)0xf35c, (q15_t)0x7f5f, (q15_t)0xf355, (q15_t)0x7f5e, (q15_t)0xf34f, (q15_t)0x7f5d, (q15_t)0xf349,
  (q15_t)0x7f5d, (q15_t)0xf343, (q15_t)0x7f5c, (q15_t)0xf33c, (q15_t)0x7f5b, (q15_t)0xf336, (q15_t)0x7f5b, (q15_t)0xf330,
  (q15_t)0x7f5a, (q15_t)0xf32a, (q15_t)0x7f5a, (q15_t)0xf323, (q15_t)0x7f59, (q15_t)0xf31d, (q15_t)0x7f58, (q15_t)0xf317,
  (q15_t)0x7f58, (q15_t)0xf311, (q15_t)0x7f57, (q15_t)0xf30a, (q15_t)0x7f56, (q15_t)0xf304, (q15_t)0x7f56, (q15_t)0xf2fe,
  (q15_t)0x7f55, (q15_t)0xf2f8, (q15_t)0x7f55, (q15_t)0xf2f1, (q15_t)0x7f54, (q15_t)0xf2eb, (q15_t)0x7f53, (q15_t)0xf2e5,
  (q15_t)0x7f53, (q15_t)0xf2df, (q15_t)0x7f52, (q15_t)0xf2d8, (q15_t)0x7f51, (q15_t)0xf2d2, (q15_t)0x7f51, (q15_t)0xf2cc,
  (q15_t)0x7f50, (q15_t)0xf2c6, (q15_t)0x7f4f, (q15_t)0xf2bf, (q15_t)0x7f4f, (q15_t)0xf2b9, (q15_t)0x7f4e, (q15_t)0xf2b3,
  (q15_t)0x7f4d, (q15_t)0xf2ad, (q15_t)0x7f4d, (q15_t)0xf2a6, (q15_t)0x7f4c, (q15_t)0xf2a0, (q15_t)0x7f4b, (q15_t)0xf29a,
  (q15_t)0x7f4b, (q15_t)0xf294, (q15_t)0x7f4a, (q15_t)0xf28d, (q15_t)0x7f49, (q15_t)0xf287, (q15_t)0x7f49, (q15_t)0xf281,
  (q15_t)0x7f48, (q15_t)0xf27b, (q15_t)0x7f47, (q15_t)0xf274, (q15_t)0x7f47, (q15_t)0xf26e, (q15_t)0x7f46, (q15_t)0xf268,
  (q15_t)0x7f45, (q15_t)0xf262, (q15_t)0x7f45, (q15_t)0xf25b, (q15_t)0x7f44, (q15_t)0xf255, (q15_t)0x7f43, (q15_t)0xf24f,
  (q15_t)0x7f43, (q15_t)0xf249, (q15_t)0x7f42, (q15_t)0xf242, (q15_t)0x7f41, (q15_t)0xf23c, (q15_t)0x7f41, (q15_t)0xf236,
  (q15_t)0x7f40, (q15_t)0xf230, (q15_t)0x7f3f, (q15_t)0xf229, (q15_t)0x7f3f, (q15_t)0xf223, (q15_t)0x7f3e, (q15_t)0xf21d,
  (q15_t)0x7f3d, (q15_t)0xf217, (q15_t)0x7f3d, (q15_t)0xf210, (q15_t)0x7f3c, (q15_t)0xf20a, (q15_t)0x7f3b, (q15_t)0xf204,
  (q15_t)0x7f3b, (q15_t)0xf1fe, (q15_t)0x7f3a, (q15_t)0xf1f7, (q15_t)0x7f39, (q15_t)0xf1f1, (q15_t)0x7f39, (q15_t)0xf1eb,
  (q15_t)0x7f38, (q15_t)0xf1e5, (q15_t)0x7f37, (q15_t)0xf1de, (q15_t)0x7f36, (q15_t)0xf1d8, (q15_t)0x7f36, (q15_t)0xf1d2,
  (q15_t)0x7f35, (q15_t)0xf1cc, (q15_t)0x7f34, (q15_t)0xf1c6, (q15_t)0x7f34, (q15_t)0xf1bf, (q15_t)0x7f33, (q15_t)0xf1b9,
  (q15_t)0x7f32, (q15_t)0xf1b3, (q15_t)0x7f32, (q15_t)0xf1ad, (q15_t)0x7f31, (q15_t)0xf1a6, (q15_t)0x7f30, (q15_t)0xf1a0,
  (q15_t)0x7f2f, (q15_t)0xf19a, (q15_t)0x7f2f, (q15_t)0xf194, (q15_t)0x7f2e, (q15_t)0xf18d, (q15_t)0x7f2d, (q15_t)0xf187,
  (q15_t)0x7f2d, (q15_t)0xf181, (q15_t)0x7f2c, (q15_t)0xf17b, (q15_t)0x7f2b, (q15_t)0xf174, (q15_t)0x7f2a, (q15_t)0xf16e,
  (q15_t)0x7f2a, (q15_t)0xf168, (q15_t)0x7f29, (q15_t)0xf162, (q15_t)0x7f28, (q15_t)0xf15b, (q15_t)0x7f28, (q15_t)0xf155,
  (q15_t)0x7f27, (q15_t)0xf14f, (q15_t)0x7f26, (q15_t)0xf149, (q15_t)0x7f25, (q15_t)0xf142, (q15_t)0x7f25, (q15_t)0xf13c,
  (q15_t)0x7f24, (q15_t)0xf136, (q15_t)0x7f23, (q15_t)0xf130, (q15_t)0x7f23, (q15_t)0xf129, (q15_t)0x7f22, (q15_t)0xf123,
  (q15_t)0x7f21, (q15_t)0xf11d, (q15_t)0x7f20, (q15_t)0xf117, (q15_t)0x7f20, (q15_t)0xf110, (q15_t)0x7f1f, (q15_t)0xf10a,
  (q15_t)0x7f1e, (q15_t)0xf104, (q15_t)0x7f1d, (q15_t)0xf0fe, (q15_t)0x7f1d, (q15_t)0xf0f8, (q15_t)0x7f1c, (q15_t)0xf0f1,
  (q15_t)0x7f1b, (q15_t)0xf0eb, (q15_t)0x7f1a, (q15_t)0xf0e5, (q15_t)0x7f1a, (q15_t)0xf0df, (q15_t)0x7f19, (q15_t)0xf0d8,
  (q15_t)0x7f18, (q15_t)0xf0d2, (q15_t)0x7f17, (q15_t)0xf0cc, (q15_t)0x7f17, (q15_t)0xf0c6, (q15_t)0x7f16, (q15_t)0xf0bf,
  (q15_t)0x7f15, (q15_t)0xf0b9, (q15_t)0x7f14, (q15_t)0xf0b3, (q15_t)0x7f14, (q15_t)0xf0ad, (q15_t)0x7f13, (q15_t)0xf0a6,
  (q15_t)0x7f12, (q15_t)0xf0a0, (q15_t)0x7f11, (q15_t)0xf09a, (q15_t)0x7f11, (q15_t)0xf094, (q15_t)0x7f10, (q15_t)0xf08d,
  (q15_t)0x7f0f, (q15_t)0xf087, (q15_t)0x7f0e, (q15_t)0xf081, (q15_t)0x7f0e, (q15_t)0xf07b, (q15_t)0x7f0d, (q15_t)0xf075,
  (q15_t)0x7f0c, (q15_t)0xf06e, (q15_t)0x7f0b, (q15_t)0xf068, (q15_t)0x7f0b, (q15_t)0xf062, (q15_t)0x7f0a, (q15_t)0xf05c,
  (q15_t)0x7f09, (q15_t)0xf055, (q15_t)0x7f08, (q15_t)0xf04f, (q15_t)0x7f08, (q15_t)0xf049, (q15_t)0x7f07, (q15_t)0xf043,
  (q15_t)0x7f06, (q15_t)0xf03c, (q15_t)0x7f05, (q15_t)0xf036, (q15_t)0x7f04, (q15_t)0xf030, (q15_t)0x7f04, (q15_t)0xf02a,
  (q15_t)0x7f03, (q15_t)0xf023, (q15_t)0x7f02, (q15_t)0xf01d, (q15_t)0x7f01, (q15_t)0xf017, (q15_t)0x7f01, (q15_t)0xf011,
  (q15_t)0x7f00, (q15_t)0xf00b, (q15_t)0x7eff, (q15_t)0xf004, (q15_t)0x7efe, (q15_t)0xeffe, (q15_t)0x7efd, (q15_t)0xeff8,
  (q15_t)0x7efd, (q15_t)0xeff2, (q15_t)0x7efc, (q15_t)0xefeb, (q15_t)0x7efb, (q15_t)0xefe5, (q15_t)0x7efa, (q15_t)0xefdf,
  (q15_t)0x7ef9, (q15_t)0xefd9, (q15_t)0x7ef9, (q15_t)0xefd2, (q15_t)0x7ef8, (q15_t)0xefcc, (q15_t)0x7ef7, (q15_t)0xefc6,
  (q15_t)0x7ef6, (q15_t)0xefc0, (q15_t)0x7ef5, (q15_t)0xefb9, (q15_t)0x7ef5, (q15_t)0xefb3, (q15_t)0x7ef4, (q15_t)0xefad,
  (q15_t)0x7ef3, (q15_t)0xefa7, (q15_t)0x7ef2, (q15_t)0xefa1, (q15_t)0x7ef1, (q15_t)0xef9a, (q15_t)0x7ef1, (q15_t)0xef94,
  (q15_t)0x7ef0, (q15_t)0xef8e, (q15_t)0x7eef, (q15_t)0xef88, (q15_t)0x7eee, (q15_t)0xef81, (q15_t)0x7eed, (q15_t)0xef7b,
  (q15_t)0x7eed, (q15_t)0xef75, (q15_t)0x7eec, (q15_t)0xef6f, (q15_t)0x7eeb, (q15_t)0xef68, (q15_t)0x7eea, (q15_t)0xef62,
  (q15_t)0x7ee9, (q15_t)0xef5c, (q15_t)0x7ee9, (q15_t)0xef56, (q15_t)0x7ee8, (q15_t)0xef50, (q15_t)0x7ee7, (q15_t)0xef49,
  (q15_t)0x7ee6, (q15_t)0xef43, (q15_t)0x7ee5, (q15_t)0xef3d, (q15_t)0x7ee4, (q15_t)0xef37, (q15_t)0x7ee4, (q15_t)0xef30,
  (q15_t)0x7ee3, (q15_t)0xef2a, (q15_t)0x7ee2, (q15_t)0xef24, (q15_t)0x7ee1, (q15_t)0xef1e, (q15_t)0x7ee0, (q15_t)0xef18,
  (q15_t)0x7edf, (q15_t)0xef11, (q15_t)0x7edf, (q15_t)0xef0b, (q15_t)0x7ede, (q15_t)0xef05, (q15_t)0x7edd, (q15_t)0xeeff,
  (q15_t)0x7edc, (q15_t)0xeef8, (q15_t)0x7edb, (q15_t)0xeef2, (q15_t)0x7eda, (q15_t)0xeeec, (q15_t)0x7eda, (q15_t)0xeee6,
  (q15_t)0x7ed9, (q15_t)0xeedf, (q15_t)0x7ed8, (q15_t)0xeed9, (q15_t)0x7ed7, (q15_t)0xeed3, (q15_t)0x7ed6, (q15_t)0xeecd,
  (q15_t)0x7ed5, (q15_t)0xeec7, (q15_t)0x7ed5, (q15_t)0xeec0, (q15_t)0x7ed4, (q15_t)0xeeba, (q15_t)0x7ed3, (q15_t)0xeeb4,
  (q15_t)0x7ed2, (q15_t)0xeeae, (q15_t)0x7ed1, (q15_t)0xeea7, (q15_t)0x7ed0, (q15_t)0xeea1, (q15_t)0x7ecf, (q15_t)0xee9b,
  (q15_t)0x7ecf, (q15_t)0xee95, (q15_t)0x7ece, (q15_t)0xee8f, (q15_t)0x7ecd, (q15_t)0xee88, (q15_t)0x7ecc, (q15_t)0xee82,
  (q15_t)0x7ecb, (q15_t)0xee7c, (q15_t)0x7eca, (q15_t)0xee76, (q15_t)0x7ec9, (q15_t)0xee6f, (q15_t)0x7ec9, (q15_t)0xee69,
  (q15_t)0x7ec8, (q15_t)0xee63, (q15_t)0x7ec7, (q15_t)0xee5d, (q15_t)0x7ec6, (q15_t)0xee57, (q15_t)0x7ec5, (q15_t)0xee50,
  (q15_t)0x7ec4, (q15_t)0xee4a, (q15_t)0x7ec3, (q15_t)0xee44, (q15_t)0x7ec3, (q15_t)0xee3e, (q15_t)0x7ec2, (q15_t)0xee37,
  (q15_t)0x7ec1, (q15_t)0xee31, (q15_t)0x7ec0, (q15_t)0xee2b, (q15_t)0x7ebf, (q15_t)0xee25, (q15_t)0x7ebe, (q15_t)0xee1f,
  (q15_t)0x7ebd, (q15_t)0xee18, (q15_t)0x7ebc, (q15_t)0xee12, (q15_t)0x7ebb, (q15_t)0xee0c, (q15_t)0x7ebb, (q15_t)0xee06,
  (q15_t)0x7eba, (q15_t)0xedff, (q15_t)0x7eb9, (q15_t)0xedf9, (q15_t)0x7eb8, (q15_t)0xedf3, (q15_t)0x7eb7, (q15_t)0xeded,
  (q15_t)0x7eb6, (q15_t)0xede7, (q15_t)0x7eb5, (q15_t)0xede0, (q15_t)0x7eb4, (q15_t)0xedda, (q15_t)0x7eb4, (q15_t)0xedd4,
  (q15_t)0x7eb3, (q15_t)0xedce, (q15_t)0x7eb2, (q15_t)0xedc7, (q15_t)0x7eb1, (q15_t)0xedc1, (q15_t)0x7eb0, (q15_t)0xedbb,
  (q15_t)0x7eaf, (q15_t)0xedb5, (q15_t)0x7eae, (q15_t)0xedaf, (q15_t)0x7ead, (q15_t)0xeda8, (q15_t)0x7eac, (q15_t)0xeda2,
  (q15_t)0x7eab, (q15_t)0xed9c, (q15_t)0x7eab, (q15_t)0xed96, (q15_t)0x7eaa, (q15_t)0xed8f, (q15_t)0x7ea9, (q15_t)0xed89,
  (q15_t)0x7ea8, (q15_t)0xed83, (q15_t)0x7ea7, (q15_t)0xed7d, (q15_t)0x7ea6, (q15_t)0xed77, (q15_t)0x7ea5, (q15_t)0xed70,
  (q15_t)0x7ea4, (q15_t)0xed6a, (q15_t)0x7ea3, (q15_t)0xed64, (q15_t)0x7ea2, (q15_t)0xed5e, (q15_t)0x7ea1, (q15_t)0xed58,
  (q15_t)0x7ea1, (q15_t)0xed51, (q15_t)0x7ea0, (q15_t)0xed4b, (q15_t)0x7e9f, (q15_t)0xed45, (q15_t)0x7e9e, (q15_t)0xed3f,
  (q15_t)0x7e9d, (q15_t)0xed38, (q15_t)0x7e9c, (q15_t)0xed32, (q15_t)0x7e9b, (q15_t)0xed2c, (q15_t)0x7e9a, (q15_t)0xed26,
  (q15_t)0x7e99, (q15_t)0xed20, (q15_t)0x7e98, (q15_t)0xed19, (q15_t)0x7e97, (q15_t)0xed13, (q15_t)0x7e96, (q15_t)0xed0d,
  (q15_t)0x7e95, (q15_t)0xed07, (q15_t)0x7e94, (q15_t)0xed01, (q15_t)0x7e94, (q15_t)0xecfa, (q15_t)0x7e93, (q15_t)0xecf4,
  (q15_t)0x7e92, (q15_t)0xecee, (q15_t)0x7e91, (q15_t)0xece8, (q15_t)0x7e90, (q15_t)0xece1, (q15_t)0x7e8f, (q15_t)0xecdb,
  (q15_t)0x7e8e, (q15_t)0xecd5, (q15_t)0x7e8d, (q15_t)0xeccf, (q15_t)0x7e8c, (q15_t)0xecc9, (q15_t)0x7e8b, (q15_t)0xecc2,
  (q15_t)0x7e8a, (q15_t)0xecbc, (q15_t)0x7e89, (q15_t)0xecb6, (q15_t)0x7e88, (q15_t)0xecb0, (q15_t)0x7e87, (q15_t)0xecaa,
  (q15_t)0x7e86, (q15_t)0xeca3, (q15_t)0x7e85, (q15_t)0xec9d, (q15_t)0x7e84, (q15_t)0xec97, (q15_t)0x7e84, (q15_t)0xec91,
  (q15_t)0x7e83, (q15_t)0xec8a, (q15_t)0x7e82, (q15_t)0xec84, (q15_t)0x7e81, (q15_t)0xec7e, (q15_t)0x7e80, (q15_t)0xec78,
  (q15_t)0x7e7f, (q15_t)0xec72, (q15_t)0x7e7e, (q15_t)0xec6b, (q15_t)0x7e7d, (q15_t)0xec65, (q15_t)0x7e7c, (q15_t)0xec5f,
  (q15_t)0x7e7b, (q15_t)0xec59, (q15_t)0x7e7a, (q15_t)0xec53, (q15_t)0x7e79, (q15_t)0xec4c, (q15_t)0x7e78, (q15_t)0xec46,
  (q15_t)0x7e77, (q15_t)0xec40, (q15_t)0x7e76, (q15_t)0xec3a, (q15_t)0x7e75, (q15_t)0xec34, (q15_t)0x7e74, (q15_t)0xec2d,
  (q15_t)0x7e73, (q15_t)0xec27, (q15_t)0x7e72, (q15_t)0xec21, (q15_t)0x7e71, (q15_t)0xec1b, (q15_t)0x7e70, (q15_t)0xec15,
  (q15_t)0x7e6f, (q15_t)0xec0e, (q15_t)0x7e6e, (q15_t)0xec08, (q15_t)0x7e6d, (q15_t)0xec02, (q15_t)0x7e6c, (q15_t)0xebfc,
  (q15_t)0x7e6b, (q15_t)0xebf5, (q15_t)0x7e6a, (q15_t)0xebef, (q15_t)0x7e69, (q15_t)0xebe9, (q15_t)0x7e68, (q15_t)0xebe3,
  (q15_t)0x7e67, (q15_t)0xebdd, (q15_t)0x7e66, (q15_t)0xebd6, (q15_t)0x7e65, (q15_t)0xebd0, (q15_t)0x7e64, (q15_t)0xebca,
  (q15_t)0x7e63, (q15_t)0xebc4, (q15_t)0x7e62, (q15_t)0xebbe, (q15_t)0x7e61, (q15_t)0xebb7, (q15_t)0x7e60, (q15_t)0xebb1,
  (q15_t)0x7e5f, (q15_t)0xebab, (q15_t)0x7e5e, (q15_t)0xeba5, (q15_t)0x7e5d, (q15_t)0xeb9f, (q15_t)0x7e5c, (q15_t)0xeb98,
  (q15_t)0x7e5b, (q15_t)0xeb92, (q15_t)0x7e5a, (q15_t)0xeb8c, (q15_t)0x7e59, (q15_t)0xeb86, (q15_t)0x7e58, (q15_t)0xeb80,
  (q15_t)0x7e57, (q15_t)0xeb79, (q15_t)0x7e56, (q15_t)0xeb73, (q15_t)0x7e55, (q15_t)0xeb6d, (q15_t)0x7e54, (q15_t)0xeb67,
  (q15_t)0x7e53, (q15_t)0xeb61, (q15_t)0x7e52, (q15_t)0xeb5a, (q15_t)0x7e51, (q15_t)0xeb54, (q15_t)0x7e50, (q15_t)0xeb4e,
  (q15_t)0x7e4f, (q15_t)0xeb48, (q15_t)0x7e4e, (q15_t)0xeb42, (q15_t)0x7e4d, (q15_t)0xeb3b, (q15_t)0x7e4c, (q15_t)0xeb35,
  (q15_t)0x7e4b, (q15_t)0xeb2f, (q15_t)0x7e4a, (q15_t)0xeb29, (q15_t)0x7e49, (q15_t)0xeb23, (q15_t)0x7e48, (q15_t)0xeb1c,
  (q15_t)0x7e47, (q15_t)0xeb16, (q15_t)0x7e46, (q15_t)0xeb10, (q15_t)0x7e45, (q15_t)0xeb0a, (q15_t)0x7e44, (q15_t)0xeb04,
  (q15_t)0x7e43, (q15_t)0xeafd, (q15_t)0x7e42, (q15_t)0xeaf7, (q15_t)0x7e41, (q15_t)0xeaf1, (q15_t)0x7e40, (q15_t)0xeaeb,
  (q15_t)0x7e3f, (q15_t)0xeae5, (q15_t)0x7e3e, (q15_t)0xeade, (q15_t)0x7e3d, (q15_t)0xead8, (q15_t)0x7e3c, (q15_t)0xead2,
  (q15_t)0x7e3b, (q15_t)0xeacc, (q15_t)0x7e3a, (q15_t)0xeac6, (q15_t)0x7e39, (q15_t)0xeabf, (q15_t)0x7e38, (q15_t)0xeab9,
  (q15_t)0x7e37, (q15_t)0xeab3, (q15_t)0x7e35, (q15_t)0xeaad, (q15_t)0x7e34, (q15_t)0xeaa7, (q15_t)0x7e33, (q15_t)0xeaa0,
  (q15_t)0x7e32, (q15_t)0xea9a, (q15_t)0x7e31, (q15_t)0xea94, (q15_t)0x7e30, (q15_t)0xea8e, (q15_t)0x7e2f, (q15_t)0xea88,
  (q15_t)0x7e2e, (q15_t)0xea81, (q15_t)0x7e2d, (q15_t)0xea7b, (q15_t)0x7e2c, (q15_t)0xea75, (q15_t)0x7e2b, (q15_t)0xea6f,
  (q15_t)0x7e2a, (q15_t)0xea69, (q15_t)0x7e29, (q15_t)0xea63, (q15_t)0x7e28, (q15_t)0xea5c, (q15_t)0x7e27, (q15_t)0xea56,
  (q15_t)0x7e26, (q15_t)0xea50, (q15_t)0x7e25, (q15_t)0xea4a, (q15_t)0x7e24, (q15_t)0xea44, (q15_t)0x7e22, (q15_t)0xea3d,
  (q15_t)0x7e21, (q15_t)0xea37, (q15_t)0x7e20, (q15_t)0xea31, (q15_t)0x7e1f, (q15_t)0xea2b, (q15_t)0x7e1e, (q15_t)0xea25,
  (q15_t)0x7e1d, (q15_t)0xea1e, (q15_t)0x7e1c, (q15_t)0xea18, (q15_t)0x7e1b, (q15_t)0xea12, (q15_t)0x7e1a, (q15_t)0xea0c,
  (q15_t)0x7e19, (q15_t)0xea06, (q15_t)0x7e18, (q15_t)0xe9ff, (q15_t)0x7e17, (q15_t)0xe9f9, (q15_t)0x7e16, (q15_t)0xe9f3,
  (q15_t)0x7e14, (q15_t)0xe9ed, (q15_t)0x7e13, (q15_t)0xe9e7, (q15_t)0x7e12, (q15_t)0xe9e1, (q15_t)0x7e11, (q15_t)0xe9da,
  (q15_t)0x7e10, (q15_t)0xe9d4, (q15_t)0x7e0f, (q15_t)0xe9ce, (q15_t)0x7e0e, (q15_t)0xe9c8, (q15_t)0x7e0d, (q15_t)0xe9c2,
  (q15_t)0x7e0c, (q15_t)0xe9bb, (q15_t)0x7e0b, (q15_t)0xe9b5, (q15_t)0x7e0a, (q15_t)0xe9af, (q15_t)0x7e08, (q15_t)0xe9a9,
  (q15_t)0x7e07, (q15_t)0xe9a3, (q15_t)0x7e06, (q15_t)0xe99c, (q15_t)0x7e05, (q15_t)0xe996, (q15_t)0x7e04, (q15_t)0xe990,
  (q15_t)0x7e03, (q15_t)0xe98a, (q15_t)0x7e02, (q15_t)0xe984, (q15_t)0x7e01, (q15_t)0xe97e, (q15_t)0x7e00, (q15_t)0xe977,
  (q15_t)0x7dff, (q15_t)0xe971, (q15_t)0x7dfd, (q15_t)0xe96b, (q15_t)0x7dfc, (q15_t)0xe965, (q15_t)0x7dfb, (q15_t)0xe95f,
  (q15_t)0x7dfa, (q15_t)0xe958, (q15_t)0x7df9, (q15_t)0xe952, (q15_t)0x7df8, (q15_t)0xe94c, (q15_t)0x7df7, (q15_t)0xe946,
  (q15_t)0x7df6, (q15_t)0xe940, (q15_t)0x7df5, (q15_t)0xe93a, (q15_t)0x7df3, (q15_t)0xe933, (q15_t)0x7df2, (q15_t)0xe92d,
  (q15_t)0x7df1, (q15_t)0xe927, (q15_t)0x7df0, (q15_t)0xe921, (q15_t)0x7def, (q15_t)0xe91b, (q15_t)0x7dee, (q15_t)0xe914,
  (q15_t)0x7ded, (q15_t)0xe90e, (q15_t)0x7dec, (q15_t)0xe908, (q15_t)0x7dea, (q15_t)0xe902, (q15_t)0x7de9, (q15_t)0xe8fc,
  (q15_t)0x7de8, (q15_t)0xe8f6, (q15_t)0x7de7, (q15_t)0xe8ef, (q15_t)0x7de6, (q15_t)0xe8e9, (q15_t)0x7de5, (q15_t)0xe8e3,
  (q15_t)0x7de4, (q15_t)0xe8dd, (q15_t)0x7de2, (q15_t)0xe8d7, (q15_t)0x7de1, (q15_t)0xe8d0, (q15_t)0x7de0, (q15_t)0xe8ca,
  (q15_t)0x7ddf, (q15_t)0xe8c4, (q15_t)0x7dde, (q15_t)0xe8be, (q15_t)0x7ddd, (q15_t)0xe8b8, (q15_t)0x7ddc, (q15_t)0xe8b2,
  (q15_t)0x7dda, (q15_t)0xe8ab, (q15_t)0x7dd9, (q15_t)0xe8a5, (q15_t)0x7dd8, (q15_t)0xe89f, (q15_t)0x7dd7, (q15_t)0xe899,
  (q15_t)0x7dd6, (q15_t)0xe893, (q15_t)0x7dd5, (q15_t)0xe88c, (q15_t)0x7dd4, (q15_t)0xe886, (q15_t)0x7dd2, (q15_t)0xe880,
  (q15_t)0x7dd1, (q15_t)0xe87a, (q15_t)0x7dd0, (q15_t)0xe874, (q15_t)0x7dcf, (q15_t)0xe86e, (q15_t)0x7dce, (q15_t)0xe867,
  (q15_t)0x7dcd, (q15_t)0xe861, (q15_t)0x7dcc, (q15_t)0xe85b, (q15_t)0x7dca, (q15_t)0xe855, (q15_t)0x7dc9, (q15_t)0xe84f,
  (q15_t)0x7dc8, (q15_t)0xe849, (q15_t)0x7dc7, (q15_t)0xe842, (q15_t)0x7dc6, (q15_t)0xe83c, (q15_t)0x7dc5, (q15_t)0xe836,
  (q15_t)0x7dc3, (q15_t)0xe830, (q15_t)0x7dc2, (q15_t)0xe82a, (q15_t)0x7dc1, (q15_t)0xe823, (q15_t)0x7dc0, (q15_t)0xe81d,
  (q15_t)0x7dbf, (q15_t)0xe817, (q15_t)0x7dbd, (q15_t)0xe811, (q15_t)0x7dbc, (q15_t)0xe80b, (q15_t)0x7dbb, (q15_t)0xe805,
  (q15_t)0x7dba, (q15_t)0xe7fe, (q15_t)0x7db9, (q15_t)0xe7f8, (q15_t)0x7db8, (q15_t)0xe7f2, (q15_t)0x7db6, (q15_t)0xe7ec,
  (q15_t)0x7db5, (q15_t)0xe7e6, (q15_t)0x7db4, (q15_t)0xe7e0, (q15_t)0x7db3, (q15_t)0xe7d9, (q15_t)0x7db2, (q15_t)0xe7d3,
  (q15_t)0x7db0, (q15_t)0xe7cd, (q15_t)0x7daf, (q15_t)0xe7c7, (q15_t)0x7dae, (q15_t)0xe7c1, (q15_t)0x7dad, (q15_t)0xe7bb,
  (q15_t)0x7dac, (q15_t)0xe7b4, (q15_t)0x7dab, (q15_t)0xe7ae, (q15_t)0x7da9, (q15_t)0xe7a8, (q15_t)0x7da8, (q15_t)0xe7a2,
  (q15_t)0x7da7, (q15_t)0xe79c, (q15_t)0x7da6, (q15_t)0xe796, (q15_t)0x7da5, (q15_t)0xe78f, (q15_t)0x7da3, (q15_t)0xe789,
  (q15_t)0x7da2, (q15_t)0xe783, (q15_t)0x7da1, (q15_t)0xe77d, (q15_t)0x7da0, (q15_t)0xe777, (q15_t)0x7d9f, (q15_t)0xe771,
  (q15_t)0x7d9d, (q15_t)0xe76a, (q15_t)0x7d9c, (q15_t)0xe764, (q15_t)0x7d9b, (q15_t)0xe75e, (q15_t)0x7d9a, (q15_t)0xe758,
  (q15_t)0x7d98, (q15_t)0xe752, (q15_t)0x7d97, (q15_t)0xe74c, (q15_t)0x7d96, (q15_t)0xe745, (q15_t)0x7d95, (q15_t)0xe73f,
  (q15_t)0x7d94, (q15_t)0xe739, (q15_t)0x7d92, (q15_t)0xe733, (q15_t)0x7d91, (q15_t)0xe72d, (q15_t)0x7d90, (q15_t)0xe727,
  (q15_t)0x7d8f, (q15_t)0xe720, (q15_t)0x7d8e, (q15_t)0xe71a, (q15_t)0x7d8c, (q15_t)0xe714, (q15_t)0x7d8b, (q15_t)0xe70e,
  (q15_t)0x7d8a, (q15_t)0xe708, (q15_t)0x7d89, (q15_t)0xe702, (q15_t)0x7d87, (q15_t)0xe6fb, (q15_t)0x7d86, (q15_t)0xe6f5,
  (q15_t)0x7d85, (q15_t)0xe6ef, (q15_t)0x7d84, (q15_t)0xe6e9, (q15_t)0x7d82, (q15_t)0xe6e3, (q15_t)0x7d81, (q15_t)0xe6dd,
  (q15_t)0x7d80, (q15_t)0xe6d6, (q15_t)0x7d7f, (q15_t)0xe6d0, (q15_t)0x7d7e, (q15_t)0xe6ca, (q15_t)0x7d7c, (q15_t)0xe6c4,
  (q15_t)0x7d7b, (q15_t)0xe6be, (q15_t)0x7d7a, (q15_t)0xe6b8, (q15_t)0x7d79, (q15_t)0xe6b2, (q15_t)0x7d77, (q15_t)0xe6ab,
  (q15_t)0x7d76, (q15_t)0xe6a5, (q15_t)0x7d75, (q15_t)0xe69f, (q15_t)0x7d74, (q15_t)0xe699, (q15_t)0x7d72, (q15_t)0xe693,
  (q15_t)0x7d71, (q15_t)0xe68d, (q15_t)0x7d70, (q15_t)0xe686, (q15_t)0x7d6f, (q15_t)0xe680, (q15_t)0x7d6d, (q15_t)0xe67a,
  (q15_t)0x7d6c, (q15_t)0xe674, (q15_t)0x7d6b, (q15_t)0xe66e, (q15_t)0x7d6a, (q15_t)0xe668, (q15_t)0x7d68, (q15_t)0xe661,
  (q15_t)0x7d67, (q15_t)0xe65b, (q15_t)0x7d66, (q15_t)0xe655, (q15_t)0x7d65, (q15_t)0xe64f, (q15_t)0x7d63, (q15_t)0xe649,
  (q15_t)0x7d62, (q15_t)0xe643, (q15_t)0x7d61, (q15_t)0xe63d, (q15_t)0x7d60, (q15_t)0xe636, (q15_t)0x7d5e, (q15_t)0xe630,
  (q15_t)0x7d5d, (q15_t)0xe62a, (q15_t)0x7d5c, (q15_t)0xe624, (q15_t)0x7d5a, (q15_t)0xe61e, (q15_t)0x7d59, (q15_t)0xe618,
  (q15_t)0x7d58, (q15_t)0xe611, (q15_t)0x7d57, (q15_t)0xe60b, (q15_t)0x7d55, (q15_t)0xe605, (q15_t)0x7d54, (q15_t)0xe5ff,
  (q15_t)0x7d53, (q15_t)0xe5f9, (q15_t)0x7d52, (q15_t)0xe5f3, (q15_t)0x7d50, (q15_t)0xe5ed, (q15_t)0x7d4f, (q15_t)0xe5e6,
  (q15_t)0x7d4e, (q15_t)0xe5e0, (q15_t)0x7d4c, (q15_t)0xe5da, (q15_t)0x7d4b, (q15_t)0xe5d4, (q15_t)0x7d4a, (q15_t)0xe5ce,
  (q15_t)0x7d49, (q15_t)0xe5c8, (q15_t)0x7d47, (q15_t)0xe5c2, (q15_t)0x7d46, (q15_t)0xe5bb, (q15_t)0x7d45, (q15_t)0xe5b5,
  (q15_t)0x7d43, (q15_t)0xe5af, (q15_t)0x7d42, (q15_t)0xe5a9, (q15_t)0x7d41, (q15_t)0xe5a3, (q15_t)0x7d3f, (q15_t)0xe59d,
  (q15_t)0x7d3e, (q15_t)0xe596, (q15_t)0x7d3d, (q15_t)0xe590, (q15_t)0x7d3c, (q15_t)0xe58a, (q15_t)0x7d3a, (q15_t)0xe584,
  (q15_t)0x7d39, (q15_t)0xe57e, (q15_t)0x7d38, (q15_t)0xe578, (q15_t)0x7d36, (q15_t)0xe572, (q15_t)0x7d35, (q15_t)0xe56b,
  (q15_t)0x7d34, (q15_t)0xe565, (q15_t)0x7d32, (q15_t)0xe55f, (q15_t)0x7d31, (q15_t)0xe559, (q15_t)0x7d30, (q15_t)0xe553,
  (q15_t)0x7d2f, (q15_t)0xe54d, (q15_t)0x7d2d, (q15_t)0xe547, (q15_t)0x7d2c, (q15_t)0xe540, (q15_t)0x7d2b, (q15_t)0xe53a,
  (q15_t)0x7d29, (q15_t)0xe534, (q15_t)0x7d28, (q15_t)0xe52e, (q15_t)0x7d27, (q15_t)0xe528, (q15_t)0x7d25, (q15_t)0xe522,
  (q15_t)0x7d24, (q15_t)0xe51c, (q15_t)0x7d23, (q15_t)0xe515, (q15_t)0x7d21, (q15_t)0xe50f, (q15_t)0x7d20, (q15_t)0xe509,
  (q15_t)0x7d1f, (q15_t)0xe503, (q15_t)0x7d1d, (q15_t)0xe4fd, (q15_t)0x7d1c, (q15_t)0xe4f7, (q15_t)0x7d1b, (q15_t)0xe4f1,
  (q15_t)0x7d19, (q15_t)0xe4ea, (q15_t)0x7d18, (q15_t)0xe4e4, (q15_t)0x7d17, (q15_t)0xe4de, (q15_t)0x7d15, (q15_t)0xe4d8,
  (q15_t)0x7d14, (q15_t)0xe4d2, (q15_t)0x7d13, (q15_t)0xe4cc, (q15_t)0x7d11, (q15_t)0xe4c6, (q15_t)0x7d10, (q15_t)0xe4bf,
  (q15_t)0x7d0f, (q15_t)0xe4b9, (q15_t)0x7d0d, (q15_t)0xe4b3, (q15_t)0x7d0c, (q15_t)0xe4ad, (q15_t)0x7d0b, (q15_t)0xe4a7,
  (q15_t)0x7d09, (q15_t)0xe4a1, (q15_t)0x7d08, (q15_t)0xe49b, (q15_t)0x7d07, (q15_t)0xe494, (q15_t)0x7d05, (q15_t)0xe48e,
  (q15_t)0x7d04, (q15_t)0xe488, (q15_t)0x7d03, (q15_t)0xe482, (q15_t)0x7d01, (q15_t)0xe47c, (q15_t)0x7d00, (q15_t)0xe476,
  (q15_t)0x7cff, (q15_t)0xe470, (q15_t)0x7cfd, (q15_t)0xe46a, (q15_t)0x7cfc, (q15_t)0xe463, (q15_t)0x7cfb, (q15_t)0xe45d,
  (q15_t)0x7cf9, (q15_t)0xe457, (q15_t)0x7cf8, (q15_t)0xe451, (q15_t)0x7cf6, (q15_t)0xe44b, (q15_t)0x7cf5, (q15_t)0xe445,
  (q15_t)0x7cf4, (q15_t)0xe43f, (q15_t)0x7cf2, (q15_t)0xe438, (q15_t)0x7cf1, (q15_t)0xe432, (q15_t)0x7cf0, (q15_t)0xe42c,
  (q15_t)0x7cee, (q15_t)0xe426, (q15_t)0x7ced, (q15_t)0xe420, (q15_t)0x7cec, (q15_t)0xe41a, (q15_t)0x7cea, (q15_t)0xe414,
  (q15_t)0x7ce9, (q15_t)0xe40e, (q15_t)0x7ce7, (q15_t)0xe407, (q15_t)0x7ce6, (q15_t)0xe401, (q15_t)0x7ce5, (q15_t)0xe3fb,
  (q15_t)0x7ce3, (q15_t)0xe3f5, (q15_t)0x7ce2, (q15_t)0xe3ef, (q15_t)0x7ce1, (q15_t)0xe3e9, (q15_t)0x7cdf, (q15_t)0xe3e3,
  (q15_t)0x7cde, (q15_t)0xe3dc, (q15_t)0x7cdc, (q15_t)0xe3d6, (q15_t)0x7cdb, (q15_t)0xe3d0, (q15_t)0x7cda, (q15_t)0xe3ca,
  (q15_t)0x7cd8, (q15_t)0xe3c4, (q15_t)0x7cd7, (q15_t)0xe3be, (q15_t)0x7cd5, (q15_t)0xe3b8, (q15_t)0x7cd4, (q15_t)0xe3b2,
  (q15_t)0x7cd3, (q15_t)0xe3ab, (q15_t)0x7cd1, (q15_t)0xe3a5, (q15_t)0x7cd0, (q15_t)0xe39f, (q15_t)0x7ccf, (q15_t)0xe399,
  (q15_t)0x7ccd, (q15_t)0xe393, (q15_t)0x7ccc, (q15_t)0xe38d, (q15_t)0x7cca, (q15_t)0xe387, (q15_t)0x7cc9, (q15_t)0xe381,
  (q15_t)0x7cc8, (q15_t)0xe37a, (q15_t)0x7cc6, (q15_t)0xe374, (q15_t)0x7cc5, (q15_t)0xe36e, (q15_t)0x7cc3, (q15_t)0xe368,
  (q15_t)0x7cc2, (q15_t)0xe362, (q15_t)0x7cc1, (q15_t)0xe35c, (q15_t)0x7cbf, (q15_t)0xe356, (q15_t)0x7cbe, (q15_t)0xe350,
  (q15_t)0x7cbc, (q15_t)0xe349, (q15_t)0x7cbb, (q15_t)0xe343, (q15_t)0x7cb9, (q15_t)0xe33d, (q15_t)0x7cb8, (q15_t)0xe337,
  (q15_t)0x7cb7, (q15_t)0xe331, (q15_t)0x7cb5, (q15_t)0xe32b, (q15_t)0x7cb4, (q15_t)0xe325, (q15_t)0x7cb2, (q15_t)0xe31f,
  (q15_t)0x7cb1, (q15_t)0xe318, (q15_t)0x7cb0, (q15_t)0xe312, (q15_t)0x7cae, (q15_t)0xe30c, (q15_t)0x7cad, (q15_t)0xe306,
  (q15_t)0x7cab, (q15_t)0xe300, (q15_t)0x7caa, (q15_t)0xe2fa, (q15_t)0x7ca8, (q15_t)0xe2f4, (q15_t)0x7ca7, (q15_t)0xe2ee,
  (q15_t)0x7ca6, (q15_t)0xe2e8, (q15_t)0x7ca4, (q15_t)0xe2e1, (q15_t)0x7ca3, (q15_t)0xe2db, (q15_t)0x7ca1, (q15_t)0xe2d5,
  (q15_t)0x7ca0, (q15_t)0xe2cf, (q15_t)0x7c9e, (q15_t)0xe2c9, (q15_t)0x7c9d, (q15_t)0xe2c3, (q15_t)0x7c9c, (q15_t)0xe2bd,
  (q15_t)0x7c9a, (q15_t)0xe2b7, (q15_t)0x7c99, (q15_t)0xe2b0, (q15_t)0x7c97, (q15_t)0xe2aa, (q15_t)0x7c96, (q15_t)0xe2a4,
  (q15_t)0x7c94, (q15_t)0xe29e, (q15_t)0x7c93, (q15_t)0xe298, (q15_t)0x7c91, (q15_t)0xe292, (q15_t)0x7c90, (q15_t)0xe28c,
  (q15_t)0x7c8f, (q15_t)0xe286, (q15_t)0x7c8d, (q15_t)0xe280, (q15_t)0x7c8c, (q15_t)0xe279, (q15_t)0x7c8a, (q15_t)0xe273,
  (q15_t)0x7c89, (q15_t)0xe26d, (q15_t)0x7c87, (q15_t)0xe267, (q15_t)0x7c86, (q15_t)0xe261, (q15_t)0x7c84, (q15_t)0xe25b,
  (q15_t)0x7c83, (q15_t)0xe255, (q15_t)0x7c82, (q15_t)0xe24f, (q15_t)0x7c80, (q15_t)0xe249, (q15_t)0x7c7f, (q15_t)0xe242,
  (q15_t)0x7c7d, (q15_t)0xe23c, (q15_t)0x7c7c, (q15_t)0xe236, (q15_t)0x7c7a, (q15_t)0xe230, (q15_t)0x7c79, (q15_t)0xe22a,
  (q15_t)0x7c77, (q15_t)0xe224, (q15_t)0x7c76, (q15_t)0xe21e, (q15_t)0x7c74, (q15_t)0xe218, (q15_t)0x7c73, (q15_t)0xe212,
  (q15_t)0x7c71, (q15_t)0xe20b, (q15_t)0x7c70, (q15_t)0xe205, (q15_t)0x7c6e, (q15_t)0xe1ff, (q15_t)0x7c6d, (q15_t)0xe1f9,
  (q15_t)0x7c6c, (q15_t)0xe1f3, (q15_t)0x7c6a, (q15_t)0xe1ed, (q15_t)0x7c69, (q15_t)0xe1e7, (q15_t)0x7c67, (q15_t)0xe1e1,
  (q15_t)0x7c66, (q15_t)0xe1db, (q15_t)0x7c64, (q15_t)0xe1d4, (q15_t)0x7c63, (q15_t)0xe1ce, (q15_t)0x7c61, (q15_t)0xe1c8,
  (q15_t)0x7c60, (q15_t)0xe1c2, (q15_t)0x7c5e, (q15_t)0xe1bc, (q15_t)0x7c5d, (q15_t)0xe1b6, (q15_t)0x7c5b, (q15_t)0xe1b0,
  (q15_t)0x7c5a, (q15_t)0xe1aa, (q15_t)0x7c58, (q15_t)0xe1a4, (q15_t)0x7c57, (q15_t)0xe19e, (q15_t)0x7c55, (q15_t)0xe197,
  (q15_t)0x7c54, (q15_t)0xe191, (q15_t)0x7c52, (q15_t)0xe18b, (q15_t)0x7c51, (q15_t)0xe185, (q15_t)0x7c4f, (q15_t)0xe17f,
  (q15_t)0x7c4e, (q15_t)0xe179, (q15_t)0x7c4c, (q15_t)0xe173, (q15_t)0x7c4b, (q15_t)0xe16d, (q15_t)0x7c49, (q15_t)0xe167,
  (q15_t)0x7c48, (q15_t)0xe160, (q15_t)0x7c46, (q15_t)0xe15a, (q15_t)0x7c45, (q15_t)0xe154, (q15_t)0x7c43, (q15_t)0xe14e,
  (q15_t)0x7c42, (q15_t)0xe148, (q15_t)0x7c40, (q15_t)0xe142, (q15_t)0x7c3f, (q15_t)0xe13c, (q15_t)0x7c3d, (q15_t)0xe136,
  (q15_t)0x7c3c, (q15_t)0xe130, (q15_t)0x7c3a, (q15_t)0xe12a, (q15_t)0x7c39, (q15_t)0xe123, (q15_t)0x7c37, (q15_t)0xe11d,
  (q15_t)0x7c36, (q15_t)0xe117, (q15_t)0x7c34, (q15_t)0xe111, (q15_t)0x7c33, (q15_t)0xe10b, (q15_t)0x7c31, (q15_t)0xe105,
  (q15_t)0x7c30, (q15_t)0xe0ff, (q15_t)0x7c2e, (q15_t)0xe0f9, (q15_t)0x7c2d, (q15_t)0xe0f3, (q15_t)0x7c2b, (q15_t)0xe0ed,
  (q15_t)0x7c29, (q15_t)0xe0e7, (q15_t)0x7c28, (q15_t)0xe0e0, (q15_t)0x7c26, (q15_t)0xe0da, (q15_t)0x7c25, (q15_t)0xe0d4,
  (q15_t)0x7c23, (q15_t)0xe0ce, (q15_t)0x7c22, (q15_t)0xe0c8, (q15_t)0x7c20, (q15_t)0xe0c2, (q15_t)0x7c1f, (q15_t)0xe0bc,
  (q15_t)0x7c1d, (q15_t)0xe0b6, (q15_t)0x7c1c, (q15_t)0xe0b0, (q15_t)0x7c1a, (q15_t)0xe0aa, (q15_t)0x7c19, (q15_t)0xe0a3,
  (q15_t)0x7c17, (q15_t)0xe09d, (q15_t)0x7c16, (q15_t)0xe097, (q15_t)0x7c14, (q15_t)0xe091, (q15_t)0x7c12, (q15_t)0xe08b,
  (q15_t)0x7c11, (q15_t)0xe085, (q15_t)0x7c0f, (q15_t)0xe07f, (q15_t)0x7c0e, (q15_t)0xe079, (q15_t)0x7c0c, (q15_t)0xe073,
  (q15_t)0x7c0b, (q15_t)0xe06d, (q15_t)0x7c09, (q15_t)0xe067, (q15_t)0x7c08, (q15_t)0xe061, (q15_t)0x7c06, (q15_t)0xe05a,
  (q15_t)0x7c05, (q15_t)0xe054, (q15_t)0x7c03, (q15_t)0xe04e, (q15_t)0x7c01, (q15_t)0xe048, (q15_t)0x7c00, (q15_t)0xe042,
  (q15_t)0x7bfe, (q15_t)0xe03c, (q15_t)0x7bfd, (q15_t)0xe036, (q15_t)0x7bfb, (q15_t)0xe030, (q15_t)0x7bfa, (q15_t)0xe02a,
  (q15_t)0x7bf8, (q15_t)0xe024, (q15_t)0x7bf6, (q15_t)0xe01e, (q15_t)0x7bf5, (q15_t)0xe017, (q15_t)0x7bf3, (q15_t)0xe011,
  (q15_t)0x7bf2, (q15_t)0xe00b, (q15_t)0x7bf0, (q15_t)0xe005, (q15_t)0x7bef, (q15_t)0xdfff, (q15_t)0x7bed, (q15_t)0xdff9,
  (q15_t)0x7beb, (q15_t)0xdff3, (q15_t)0x7bea, (q15_t)0xdfed, (q15_t)0x7be8, (q15_t)0xdfe7, (q15_t)0x7be7, (q15_t)0xdfe1,
  (q15_t)0x7be5, (q15_t)0xdfdb, (q15_t)0x7be4, (q15_t)0xdfd5, (q15_t)0x7be2, (q15_t)0xdfce, (q15_t)0x7be0, (q15_t)0xdfc8,
  (q15_t)0x7bdf, (q15_t)0xdfc2, (q15_t)0x7bdd, (q15_t)0xdfbc, (q15_t)0x7bdc, (q15_t)0xdfb6, (q15_t)0x7bda, (q15_t)0xdfb0,
  (q15_t)0x7bd9, (q15_t)0xdfaa, (q15_t)0x7bd7, (q15_t)0xdfa4, (q15_t)0x7bd5, (q15_t)0xdf9e, (q15_t)0x7bd4, (q15_t)0xdf98,
  (q15_t)0x7bd2, (q15_t)0xdf92, (q15_t)0x7bd1, (q15_t)0xdf8c, (q15_t)0x7bcf, (q15_t)0xdf86, (q15_t)0x7bcd, (q15_t)0xdf7f,
  (q15_t)0x7bcc, (q15_t)0xdf79, (q15_t)0x7bca, (q15_t)0xdf73, (q15_t)0x7bc9, (q15_t)0xdf6d, (q15_t)0x7bc7, (q15_t)0xdf67,
  (q15_t)0x7bc5, (q15_t)0xdf61, (q15_t)0x7bc4, (q15_t)0xdf5b, (q15_t)0x7bc2, (q15_t)0xdf55, (q15_t)0x7bc1, (q15_t)0xdf4f,
  (q15_t)0x7bbf, (q15_t)0xdf49, (q15_t)0x7bbd, (q15_t)0xdf43, (q15_t)0x7bbc, (q15_t)0xdf3d, (q15_t)0x7bba, (q15_t)0xdf37,
  (q15_t)0x7bb9, (q15_t)0xdf30, (q15_t)0x7bb7, (q15_t)0xdf2a, (q15_t)0x7bb5, (q15_t)0xdf24, (q15_t)0x7bb4, (q15_t)0xdf1e,
  (q15_t)0x7bb2, (q15_t)0xdf18, (q15_t)0x7bb0, (q15_t)0xdf12, (q15_t)0x7baf, (q15_t)0xdf0c, (q15_t)0x7bad, (q15_t)0xdf06,
  (q15_t)0x7bac, (q15_t)0xdf00, (q15_t)0x7baa, (q15_t)0xdefa, (q15_t)0x7ba8, (q15_t)0xdef4, (q15_t)0x7ba7, (q15_t)0xdeee,
  (q15_t)0x7ba5, (q15_t)0xdee8, (q15_t)0x7ba3, (q15_t)0xdee2, (q15_t)0x7ba2, (q15_t)0xdedb, (q15_t)0x7ba0, (q15_t)0xded5,
  (q15_t)0x7b9f, (q15_t)0xdecf, (q15_t)0x7b9d, (q15_t)0xdec9, (q15_t)0x7b9b, (q15_t)0xdec3, (q15_t)0x7b9a, (q15_t)0xdebd,
  (q15_t)0x7b98, (q15_t)0xdeb7, (q15_t)0x7b96, (q15_t)0xdeb1, (q15_t)0x7b95, (q15_t)0xdeab, (q15_t)0x7b93, (q15_t)0xdea5,
  (q15_t)0x7b92, (q15_t)0xde9f, (q15_t)0x7b90, (q15_t)0xde99, (q15_t)0x7b8e, (q15_t)0xde93, (q15_t)0x7b8d, (q15_t)0xde8d,
  (q15_t)0x7b8b, (q15_t)0xde87, (q15_t)0x7b89, (q15_t)0xde80, (q15_t)0x7b88, (q15_t)0xde7a, (q15_t)0x7b86, (q15_t)0xde74,
  (q15_t)0x7b84, (q15_t)0xde6e, (q15_t)0x7b83, (q15_t)0xde68, (q15_t)0x7b81, (q15_t)0xde62, (q15_t)0x7b7f, (q15_t)0xde5c,
  (q15_t)0x7b7e, (q15_t)0xde56, (q15_t)0x7b7c, (q15_t)0xde50, (q15_t)0x7b7a, (q15_t)0xde4a, (q15_t)0x7b79, (q15_t)0xde44,
  (q15_t)0x7b77, (q15_t)0xde3e, (q15_t)0x7b76, (q15_t)0xde38, (q15_t)0x7b74, (q15_t)0xde32, (q15_t)0x7b72, (q15_t)0xde2c,
  (q15_t)0x7b71, (q15_t)0xde26, (q15_t)0x7b6f, (q15_t)0xde1f, (q15_t)0x7b6d, (q15_t)0xde19, (q15_t)0x7b6c, (q15_t)0xde13,
  (q15_t)0x7b6a, (q15_t)0xde0d, (q15_t)0x7b68, (q15_t)0xde07, (q15_t)0x7b67, (q15_t)0xde01, (q15_t)0x7b65, (q15_t)0xddfb,
  (q15_t)0x7b63, (q15_t)0xddf5, (q15_t)0x7b62, (q15_t)0xddef, (q15_t)0x7b60, (q15_t)0xdde9, (q15_t)0x7b5e, (q15_t)0xdde3,
  (q15_t)0x7b5d, (q15_t)0xdddd, (q15_t)0x7b5b, (q15_t)0xddd7, (q15_t)0x7b59, (q15_t)0xddd1, (q15_t)0x7b57, (q15_t)0xddcb,
  (q15_t)0x7b56, (q15_t)0xddc5, (q15_t)0x7b54, (q15_t)0xddbf, (q15_t)0x7b52, (q15_t)0xddb9, (q15_t)0x7b51, (q15_t)0xddb2,
  (q15_t)0x7b4f, (q15_t)0xddac, (q15_t)0x7b4d, (q15_t)0xdda6, (q15_t)0x7b4c, (q15_t)0xdda0, (q15_t)0x7b4a, (q15_t)0xdd9a,
  (q15_t)0x7b48, (q15_t)0xdd94, (q15_t)0x7b47, (q15_t)0xdd8e, (q15_t)0x7b45, (q15_t)0xdd88, (q15_t)0x7b43, (q15_t)0xdd82,
  (q15_t)0x7b42, (q15_t)0xdd7c, (q15_t)0x7b40, (q15_t)0xdd76, (q15_t)0x7b3e, (q15_t)0xdd70, (q15_t)0x7b3c, (q15_t)0xdd6a,
  (q15_t)0x7b3b, (q15_t)0xdd64, (q15_t)0x7b39, (q15_t)0xdd5e, (q15_t)0x7b37, (q15_t)0xdd58, (q15_t)0x7b36, (q15_t)0xdd52,
  (q15_t)0x7b34, (q15_t)0xdd4c, (q15_t)0x7b32, (q15_t)0xdd46, (q15_t)0x7b31, (q15_t)0xdd40, (q15_t)0x7b2f, (q15_t)0xdd39,
  (q15_t)0x7b2d, (q15_t)0xdd33, (q15_t)0x7b2b, (q15_t)0xdd2d, (q15_t)0x7b2a, (q15_t)0xdd27, (q15_t)0x7b28, (q15_t)0xdd21,
  (q15_t)0x7b26, (q15_t)0xdd1b, (q15_t)0x7b25, (q15_t)0xdd15, (q15_t)0x7b23, (q15_t)0xdd0f, (q15_t)0x7b21, (q15_t)0xdd09,
  (q15_t)0x7b1f, (q15_t)0xdd03, (q15_t)0x7b1e, (q15_t)0xdcfd, (q15_t)0x7b1c, (q15_t)0xdcf7, (q15_t)0x7b1a, (q15_t)0xdcf1,
  (q15_t)0x7b19, (q15_t)0xdceb, (q15_t)0x7b17, (q15_t)0xdce5, (q15_t)0x7b15, (q15_t)0xdcdf, (q15_t)0x7b13, (q15_t)0xdcd9,
  (q15_t)0x7b12, (q15_t)0xdcd3, (q15_t)0x7b10, (q15_t)0xdccd, (q15_t)0x7b0e, (q15_t)0xdcc7, (q15_t)0x7b0c, (q15_t)0xdcc1,
  (q15_t)0x7b0b, (q15_t)0xdcbb, (q15_t)0x7b09, (q15_t)0xdcb5, (q15_t)0x7b07, (q15_t)0xdcae, (q15_t)0x7b06, (q15_t)0xdca8,
  (q15_t)0x7b04, (q15_t)0xdca2, (q15_t)0x7b02, (q15_t)0xdc9c, (q15_t)0x7b00, (q15_t)0xdc96, (q15_t)0x7aff, (q15_t)0xdc90,
  (q15_t)0x7afd, (q15_t)0xdc8a, (q15_t)0x7afb, (q15_t)0xdc84, (q15_t)0x7af9, (q15_t)0xdc7e, (q15_t)0x7af8, (q15_t)0xdc78,
  (q15_t)0x7af6, (q15_t)0xdc72, (q15_t)0x7af4, (q15_t)0xdc6c, (q15_t)0x7af2, (q15_t)0xdc66, (q15_t)0x7af1, (q15_t)0xdc60,
  (q15_t)0x7aef, (q15_t)0xdc5a, (q15_t)0x7aed, (q15_t)0xdc54, (q15_t)0x7aeb, (q15_t)0xdc4e, (q15_t)0x7aea, (q15_t)0xdc48,
  (q15_t)0x7ae8, (q15_t)0xdc42, (q15_t)0x7ae6, (q15_t)0xdc3c, (q15_t)0x7ae4, (q15_t)0xdc36, (q15_t)0x7ae3, (q15_t)0xdc30,
  (q15_t)0x7ae1, (q15_t)0xdc2a, (q15_t)0x7adf, (q15_t)0xdc24, (q15_t)0x7add, (q15_t)0xdc1e, (q15_t)0x7adc, (q15_t)0xdc18,
  (q15_t)0x7ada, (q15_t)0xdc12, (q15_t)0x7ad8, (q15_t)0xdc0c, (q15_t)0x7ad6, (q15_t)0xdc06, (q15_t)0x7ad5, (q15_t)0xdbff,
  (q15_t)0x7ad3, (q15_t)0xdbf9, (q15_t)0x7ad1, (q15_t)0xdbf3, (q15_t)0x7acf, (q15_t)0xdbed, (q15_t)0x7acd, (q15_t)0xdbe7,
  (q15_t)0x7acc, (q15_t)0xdbe1, (q15_t)0x7aca, (q15_t)0xdbdb, (q15_t)0x7ac8, (q15_t)0xdbd5, (q15_t)0x7ac6, (q15_t)0xdbcf,
  (q15_t)0x7ac5, (q15_t)0xdbc9, (q15_t)0x7ac3, (q15_t)0xdbc3, (q15_t)0x7ac1, (q15_t)0xdbbd, (q15_t)0x7abf, (q15_t)0xdbb7,
  (q15_t)0x7abd, (q15_t)0xdbb1, (q15_t)0x7abc, (q15_t)0xdbab, (q15_t)0x7aba, (q15_t)0xdba5, (q15_t)0x7ab8, (q15_t)0xdb9f,
  (q15_t)0x7ab6, (q15_t)0xdb99, (q15_t)0x7ab5, (q15_t)0xdb93, (q15_t)0x7ab3, (q15_t)0xdb8d, (q15_t)0x7ab1, (q15_t)0xdb87,
  (q15_t)0x7aaf, (q15_t)0xdb81, (q15_t)0x7aad, (q15_t)0xdb7b, (q15_t)0x7aac, (q15_t)0xdb75, (q15_t)0x7aaa, (q15_t)0xdb6f,
  (q15_t)0x7aa8, (q15_t)0xdb69, (q15_t)0x7aa6, (q15_t)0xdb63, (q15_t)0x7aa4, (q15_t)0xdb5d, (q15_t)0x7aa3, (q15_t)0xdb57,
  (q15_t)0x7aa1, (q15_t)0xdb51, (q15_t)0x7a9f, (q15_t)0xdb4b, (q15_t)0x7a9d, (q15_t)0xdb45, (q15_t)0x7a9b, (q15_t)0xdb3f,
  (q15_t)0x7a9a, (q15_t)0xdb39, (q15_t)0x7a98, (q15_t)0xdb33, (q15_t)0x7a96, (q15_t)0xdb2d, (q15_t)0x7a94, (q15_t)0xdb27,
  (q15_t)0x7a92, (q15_t)0xdb21, (q15_t)0x7a91, (q15_t)0xdb1b, (q15_t)0x7a8f, (q15_t)0xdb15, (q15_t)0x7a8d, (q15_t)0xdb0f,
  (q15_t)0x7a8b, (q15_t)0xdb09, (q15_t)0x7a89, (q15_t)0xdb03, (q15_t)0x7a87, (q15_t)0xdafd, (q15_t)0x7a86, (q15_t)0xdaf7,
  (q15_t)0x7a84, (q15_t)0xdaf1, (q15_t)0x7a82, (q15_t)0xdaea, (q15_t)0x7a80, (q15_t)0xdae4, (q15_t)0x7a7e, (q15_t)0xdade,
  (q15_t)0x7a7d, (q15_t)0xdad8, (q15_t)0x7a7b, (q15_t)0xdad2, (q15_t)0x7a79, (q15_t)0xdacc, (q15_t)0x7a77, (q15_t)0xdac6,
  (q15_t)0x7a75, (q15_t)0xdac0, (q15_t)0x7a73, (q15_t)0xdaba, (q15_t)0x7a72, (q15_t)0xdab4, (q15_t)0x7a70, (q15_t)0xdaae,
  (q15_t)0x7a6e, (q15_t)0xdaa8, (q15_t)0x7a6c, (q15_t)0xdaa2, (q15_t)0x7a6a, (q15_t)0xda9c, (q15_t)0x7a68, (q15_t)0xda96,
  (q15_t)0x7a67, (q15_t)0xda90, (q15_t)0x7a65, (q15_t)0xda8a, (q15_t)0x7a63, (q15_t)0xda84, (q15_t)0x7a61, (q15_t)0xda7e,
  (q15_t)0x7a5f, (q15_t)0xda78, (q15_t)0x7a5d, (q15_t)0xda72, (q15_t)0x7a5c, (q15_t)0xda6c, (q15_t)0x7a5a, (q15_t)0xda66,
  (q15_t)0x7a58, (q15_t)0xda60, (q15_t)0x7a56, (q15_t)0xda5a, (q15_t)0x7a54, (q15_t)0xda54, (q15_t)0x7a52, (q15_t)0xda4e,
  (q15_t)0x7a50, (q15_t)0xda48, (q15_t)0x7a4f, (q15_t)0xda42, (q15_t)0x7a4d, (q15_t)0xda3c, (q15_t)0x7a4b, (q15_t)0xda36,
  (q15_t)0x7a49, (q15_t)0xda30, (q15_t)0x7a47, (q15_t)0xda2a, (q15_t)0x7a45, (q15_t)0xda24, (q15_t)0x7a43, (q15_t)0xda1e,
  (q15_t)0x7a42, (q15_t)0xda18, (q15_t)0x7a40, (q15_t)0xda12, (q15_t)0x7a3e, (q15_t)0xda0c, (q15_t)0x7a3c, (q15_t)0xda06,
  (q15_t)0x7a3a, (q15_t)0xda00, (q15_t)0x7a38, (q15_t)0xd9fa, (q15_t)0x7a36, (q15_t)0xd9f4, (q15_t)0x7a35, (q15_t)0xd9ee,
  (q15_t)0x7a33, (q15_t)0xd9e8, (q15_t)0x7a31, (q15_t)0xd9e2, (q15_t)0x7a2f, (q15_t)0xd9dc, (q15_t)0x7a2d, (q15_t)0xd9d6,
  (q15_t)0x7a2b, (q15_t)0xd9d0, (q15_t)0x7a29, (q15_t)0xd9ca, (q15_t)0x7a27, (q15_t)0xd9c4, (q15_t)0x7a26, (q15_t)0xd9be,
  (q15_t)0x7a24, (q15_t)0xd9b8, (q15_t)0x7a22, (q15_t)0xd9b2, (q15_t)0x7a20, (q15_t)0xd9ac, (q15_t)0x7a1e, (q15_t)0xd9a6,
  (q15_t)0x7a1c, (q15_t)0xd9a0, (q15_t)0x7a1a, (q15_t)0xd99a, (q15_t)0x7a18, (q15_t)0xd994, (q15_t)0x7a16, (q15_t)0xd98e,
  (q15_t)0x7a15, (q15_t)0xd988, (q15_t)0x7a13, (q15_t)0xd982, (q15_t)0x7a11, (q15_t)0xd97c, (q15_t)0x7a0f, (q15_t)0xd976,
  (q15_t)0x7a0d, (q15_t)0xd970, (q15_t)0x7a0b, (q15_t)0xd96a, (q15_t)0x7a09, (q15_t)0xd964, (q15_t)0x7a07, (q15_t)0xd95e,
  (q15_t)0x7a05, (q15_t)0xd958, (q15_t)0x7a04, (q15_t)0xd952, (q15_t)0x7a02, (q15_t)0xd94c, (q15_t)0x7a00, (q15_t)0xd946,
  (q15_t)0x79fe, (q15_t)0xd940, (q15_t)0x79fc, (q15_t)0xd93a, (q15_t)0x79fa, (q15_t)0xd934, (q15_t)0x79f8, (q15_t)0xd92e,
  (q15_t)0x79f6, (q15_t)0xd928, (q15_t)0x79f4, (q15_t)0xd922, (q15_t)0x79f2, (q15_t)0xd91c, (q15_t)0x79f0, (q15_t)0xd917,
  (q15_t)0x79ef, (q15_t)0xd911, (q15_t)0x79ed, (q15_t)0xd90b, (q15_t)0x79eb, (q15_t)0xd905, (q15_t)0x79e9, (q15_t)0xd8ff,
  (q15_t)0x79e7, (q15_t)0xd8f9, (q15_t)0x79e5, (q15_t)0xd8f3, (q15_t)0x79e3, (q15_t)0xd8ed, (q15_t)0x79e1, (q15_t)0xd8e7,
  (q15_t)0x79df, (q15_t)0xd8e1, (q15_t)0x79dd, (q15_t)0xd8db, (q15_t)0x79db, (q15_t)0xd8d5, (q15_t)0x79d9, (q15_t)0xd8cf,
  (q15_t)0x79d8, (q15_t)0xd8c9, (q15_t)0x79d6, (q15_t)0xd8c3, (q15_t)0x79d4, (q15_t)0xd8bd, (q15_t)0x79d2, (q15_t)0xd8b7,
  (q15_t)0x79d0, (q15_t)0xd8b1, (q15_t)0x79ce, (q15_t)0xd8ab, (q15_t)0x79cc, (q15_t)0xd8a5, (q15_t)0x79ca, (q15_t)0xd89f,
  (q15_t)0x79c8, (q15_t)0xd899, (q15_t)0x79c6, (q15_t)0xd893, (q15_t)0x79c4, (q15_t)0xd88d, (q15_t)0x79c2, (q15_t)0xd887,
  (q15_t)0x79c0, (q15_t)0xd881, (q15_t)0x79be, (q15_t)0xd87b, (q15_t)0x79bc, (q15_t)0xd875, (q15_t)0x79bb, (q15_t)0xd86f,
  (q15_t)0x79b9, (q15_t)0xd869, (q15_t)0x79b7, (q15_t)0xd863, (q15_t)0x79b5, (q15_t)0xd85d, (q15_t)0x79b3, (q15_t)0xd857,
  (q15_t)0x79b1, (q15_t)0xd851, (q15_t)0x79af, (q15_t)0xd84b, (q15_t)0x79ad, (q15_t)0xd845, (q15_t)0x79ab, (q15_t)0xd83f,
  (q15_t)0x79a9, (q15_t)0xd839, (q15_t)0x79a7, (q15_t)0xd833, (q15_t)0x79a5, (q15_t)0xd82d, (q15_t)0x79a3, (q15_t)0xd827,
  (q15_t)0x79a1, (q15_t)0xd821, (q15_t)0x799f, (q15_t)0xd81b, (q15_t)0x799d, (q15_t)0xd815, (q15_t)0x799b, (q15_t)0xd80f,
  (q15_t)0x7999, (q15_t)0xd80a, (q15_t)0x7997, (q15_t)0xd804, (q15_t)0x7995, (q15_t)0xd7fe, (q15_t)0x7993, (q15_t)0xd7f8,
  (q15_t)0x7992, (q15_t)0xd7f2, (q15_t)0x7990, (q15_t)0xd7ec, (q15_t)0x798e, (q15_t)0xd7e6, (q15_t)0x798c, (q15_t)0xd7e0,
  (q15_t)0x798a, (q15_t)0xd7da, (q15_t)0x7988, (q15_t)0xd7d4, (q15_t)0x7986, (q15_t)0xd7ce, (q15_t)0x7984, (q15_t)0xd7c8,
  (q15_t)0x7982, (q15_t)0xd7c2, (q15_t)0x7980, (q15_t)0xd7bc, (q15_t)0x797e, (q15_t)0xd7b6, (q15_t)0x797c, (q15_t)0xd7b0,
  (q15_t)0x797a, (q15_t)0xd7aa, (q15_t)0x7978, (q15_t)0xd7a4, (q15_t)0x7976, (q15_t)0xd79e, (q15_t)0x7974, (q15_t)0xd798,
  (q15_t)0x7972, (q15_t)0xd792, (q15_t)0x7970, (q15_t)0xd78c, (q15_t)0x796e, (q15_t)0xd786, (q15_t)0x796c, (q15_t)0xd780,
  (q15_t)0x796a, (q15_t)0xd77a, (q15_t)0x7968, (q15_t)0xd774, (q15_t)0x7966, (q15_t)0xd76e, (q15_t)0x7964, (q15_t)0xd768,
  (q15_t)0x7962, (q15_t)0xd763, (q15_t)0x7960, (q15_t)0xd75d, (q15_t)0x795e, (q15_t)0xd757, (q15_t)0x795c, (q15_t)0xd751,
  (q15_t)0x795a, (q15_t)0xd74b, (q15_t)0x7958, (q15_t)0xd745, (q15_t)0x7956, (q15_t)0xd73f, (q15_t)0x7954, (q15_t)0xd739,
  (q15_t)0x7952, (q15_t)0xd733, (q15_t)0x7950, (q15_t)0xd72d, (q15_t)0x794e, (q15_t)0xd727, (q15_t)0x794c, (q15_t)0xd721,
  (q15_t)0x794a, (q15_t)0xd71b, (q15_t)0x7948, (q15_t)0xd715, (q15_t)0x7946, (q15_t)0xd70f, (q15_t)0x7944, (q15_t)0xd709,
  (q15_t)0x7942, (q15_t)0xd703, (q15_t)0x7940, (q15_t)0xd6fd, (q15_t)0x793e, (q15_t)0xd6f7, (q15_t)0x793c, (q15_t)0xd6f1,
  (q15_t)0x793a, (q15_t)0xd6eb, (q15_t)0x7938, (q15_t)0xd6e5, (q15_t)0x7936, (q15_t)0xd6e0, (q15_t)0x7934, (q15_t)0xd6da,
  (q15_t)0x7932, (q15_t)0xd6d4, (q15_t)0x7930, (q15_t)0xd6ce, (q15_t)0x792e, (q15_t)0xd6c8, (q15_t)0x792c, (q15_t)0xd6c2,
  (q15_t)0x792a, (q15_t)0xd6bc, (q15_t)0x7928, (q15_t)0xd6b6, (q15_t)0x7926, (q15_t)0xd6b0, (q15_t)0x7924, (q15_t)0xd6aa,
  (q15_t)0x7922, (q15_t)0xd6a4, (q15_t)0x7920, (q15_t)0xd69e, (q15_t)0x791e, (q15_t)0xd698, (q15_t)0x791c, (q15_t)0xd692,
  (q15_t)0x7919, (q15_t)0xd68c, (q15_t)0x7917, (q15_t)0xd686, (q15_t)0x7915, (q15_t)0xd680, (q15_t)0x7913, (q15_t)0xd67a,
  (q15_t)0x7911, (q15_t)0xd675, (q15_t)0x790f, (q15_t)0xd66f, (q15_t)0x790d, (q15_t)0xd669, (q15_t)0x790b, (q15_t)0xd663,
  (q15_t)0x7909, (q15_t)0xd65d, (q15_t)0x7907, (q15_t)0xd657, (q15_t)0x7905, (q15_t)0xd651, (q15_t)0x7903, (q15_t)0xd64b,
  (q15_t)0x7901, (q15_t)0xd645, (q15_t)0x78ff, (q15_t)0xd63f, (q15_t)0x78fd, (q15_t)0xd639, (q15_t)0x78fb, (q15_t)0xd633,
  (q15_t)0x78f9, (q15_t)0xd62d, (q15_t)0x78f7, (q15_t)0xd627, (q15_t)0x78f5, (q15_t)0xd621, (q15_t)0x78f3, (q15_t)0xd61b,
  (q15_t)0x78f1, (q15_t)0xd615, (q15_t)0x78ee, (q15_t)0xd610, (q15_t)0x78ec, (q15_t)0xd60a, (q15_t)0x78ea, (q15_t)0xd604,
  (q15_t)0x78e8, (q15_t)0xd5fe, (q15_t)0x78e6, (q15_t)0xd5f8, (q15_t)0x78e4, (q15_t)0xd5f2, (q15_t)0x78e2, (q15_t)0xd5ec,
  (q15_t)0x78e0, (q15_t)0xd5e6, (q15_t)0x78de, (q15_t)0xd5e0, (q15_t)0x78dc, (q15_t)0xd5da, (q15_t)0x78da, (q15_t)0xd5d4,
  (q15_t)0x78d8, (q15_t)0xd5ce, (q15_t)0x78d6, (q15_t)0xd5c8, (q15_t)0x78d4, (q15_t)0xd5c2, (q15_t)0x78d2, (q15_t)0xd5bc,
  (q15_t)0x78cf, (q15_t)0xd5b7, (q15_t)0x78cd, (q15_t)0xd5b1, (q15_t)0x78cb, (q15_t)0xd5ab, (q15_t)0x78c9, (q15_t)0xd5a5,
  (q15_t)0x78c7, (q15_t)0xd59f, (q15_t)0x78c5, (q15_t)0xd599, (q15_t)0x78c3, (q15_t)0xd593, (q15_t)0x78c1, (q15_t)0xd58d,
  (q15_t)0x78bf, (q15_t)0xd587, (q15_t)0x78bd, (q15_t)0xd581, (q15_t)0x78bb, (q15_t)0xd57b, (q15_t)0x78b9, (q15_t)0xd575,
  (q15_t)0x78b6, (q15_t)0xd56f, (q15_t)0x78b4, (q15_t)0xd569, (q15_t)0x78b2, (q15_t)0xd564, (q15_t)0x78b0, (q15_t)0xd55e,
  (q15_t)0x78ae, (q15_t)0xd558, (q15_t)0x78ac, (q15_t)0xd552, (q15_t)0x78aa, (q15_t)0xd54c, (q15_t)0x78a8, (q15_t)0xd546,
  (q15_t)0x78a6, (q15_t)0xd540, (q15_t)0x78a4, (q15_t)0xd53a, (q15_t)0x78a2, (q15_t)0xd534, (q15_t)0x789f, (q15_t)0xd52e,
  (q15_t)0x789d, (q15_t)0xd528, (q15_t)0x789b, (q15_t)0xd522, (q15_t)0x7899, (q15_t)0xd51c, (q15_t)0x7897, (q15_t)0xd517,
  (q15_t)0x7895, (q15_t)0xd511, (q15_t)0x7893, (q15_t)0xd50b, (q15_t)0x7891, (q15_t)0xd505, (q15_t)0x788f, (q15_t)0xd4ff,
  (q15_t)0x788c, (q15_t)0xd4f9, (q15_t)0x788a, (q15_t)0xd4f3, (q15_t)0x7888, (q15_t)0xd4ed, (q15_t)0x7886, (q15_t)0xd4e7,
  (q15_t)0x7884, (q15_t)0xd4e1, (q15_t)0x7882, (q15_t)0xd4db, (q15_t)0x7880, (q15_t)0xd4d5, (q15_t)0x787e, (q15_t)0xd4d0,
  (q15_t)0x787c, (q15_t)0xd4ca, (q15_t)0x7879, (q15_t)0xd4c4, (q15_t)0x7877, (q15_t)0xd4be, (q15_t)0x7875, (q15_t)0xd4b8,
  (q15_t)0x7873, (q15_t)0xd4b2, (q15_t)0x7871, (q15_t)0xd4ac, (q15_t)0x786f, (q15_t)0xd4a6, (q15_t)0x786d, (q15_t)0xd4a0,
  (q15_t)0x786b, (q15_t)0xd49a, (q15_t)0x7868, (q15_t)0xd494, (q15_t)0x7866, (q15_t)0xd48f, (q15_t)0x7864, (q15_t)0xd489,
  (q15_t)0x7862, (q15_t)0xd483, (q15_t)0x7860, (q15_t)0xd47d, (q15_t)0x785e, (q15_t)0xd477, (q15_t)0x785c, (q15_t)0xd471,
  (q15_t)0x7859, (q15_t)0xd46b, (q15_t)0x7857, (q15_t)0xd465, (q15_t)0x7855, (q15_t)0xd45f, (q15_t)0x7853, (q15_t)0xd459,
  (q15_t)0x7851, (q15_t)0xd453, (q15_t)0x784f, (q15_t)0xd44e, (q15_t)0x784d, (q15_t)0xd448, (q15_t)0x784a, (q15_t)0xd442,
  (q15_t)0x7848, (q15_t)0xd43c, (q15_t)0x7846, (q15_t)0xd436, (q15_t)0x7844, (q15_t)0xd430, (q15_t)0x7842, (q15_t)0xd42a,
  (q15_t)0x7840, (q15_t)0xd424, (q15_t)0x783e, (q15_t)0xd41e, (q15_t)0x783b, (q15_t)0xd418, (q15_t)0x7839, (q15_t)0xd412,
  (q15_t)0x7837, (q15_t)0xd40d, (q15_t)0x7835, (q15_t)0xd407, (q15_t)0x7833, (q15_t)0xd401, (q15_t)0x7831, (q15_t)0xd3fb,
  (q15_t)0x782e, (q15_t)0xd3f5, (q15_t)0x782c, (q15_t)0xd3ef, (q15_t)0x782a, (q15_t)0xd3e9, (q15_t)0x7828, (q15_t)0xd3e3,
  (q15_t)0x7826, (q15_t)0xd3dd, (q15_t)0x7824, (q15_t)0xd3d7, (q15_t)0x7821, (q15_t)0xd3d2, (q15_t)0x781f, (q15_t)0xd3cc,
  (q15_t)0x781d, (q15_t)0xd3c6, (q15_t)0x781b, (q15_t)0xd3c0, (q15_t)0x7819, (q15_t)0xd3ba, (q15_t)0x7817, (q15_t)0xd3b4,
  (q15_t)0x7814, (q15_t)0xd3ae, (q15_t)0x7812, (q15_t)0xd3a8, (q15_t)0x7810, (q15_t)0xd3a2, (q15_t)0x780e, (q15_t)0xd39d,
  (q15_t)0x780c, (q15_t)0xd397, (q15_t)0x780a, (q15_t)0xd391, (q15_t)0x7807, (q15_t)0xd38b, (q15_t)0x7805, (q15_t)0xd385,
  (q15_t)0x7803, (q15_t)0xd37f, (q15_t)0x7801, (q15_t)0xd379, (q15_t)0x77ff, (q15_t)0xd373, (q15_t)0x77fc, (q15_t)0xd36d,
  (q15_t)0x77fa, (q15_t)0xd368, (q15_t)0x77f8, (q15_t)0xd362, (q15_t)0x77f6, (q15_t)0xd35c, (q15_t)0x77f4, (q15_t)0xd356,
  (q15_t)0x77f1, (q15_t)0xd350, (q15_t)0x77ef, (q15_t)0xd34a, (q15_t)0x77ed, (q15_t)0xd344, (q15_t)0x77eb, (q15_t)0xd33e,
  (q15_t)0x77e9, (q15_t)0xd338, (q15_t)0x77e6, (q15_t)0xd333, (q15_t)0x77e4, (q15_t)0xd32d, (q15_t)0x77e2, (q15_t)0xd327,
  (q15_t)0x77e0, (q15_t)0xd321, (q15_t)0x77de, (q15_t)0xd31b, (q15_t)0x77db, (q15_t)0xd315, (q15_t)0x77d9, (q15_t)0xd30f,
  (q15_t)0x77d7, (q15_t)0xd309, (q15_t)0x77d5, (q15_t)0xd303, (q15_t)0x77d3, (q15_t)0xd2fe, (q15_t)0x77d0, (q15_t)0xd2f8,
  (q15_t)0x77ce, (q15_t)0xd2f2, (q15_t)0x77cc, (q15_t)0xd2ec, (q15_t)0x77ca, (q15_t)0xd2e6, (q15_t)0x77c8, (q15_t)0xd2e0,
  (q15_t)0x77c5, (q15_t)0xd2da, (q15_t)0x77c3, (q15_t)0xd2d4, (q15_t)0x77c1, (q15_t)0xd2cf, (q15_t)0x77bf, (q15_t)0xd2c9,
  (q15_t)0x77bc, (q15_t)0xd2c3, (q15_t)0x77ba, (q15_t)0xd2bd, (q15_t)0x77b8, (q15_t)0xd2b7, (q15_t)0x77b6, (q15_t)0xd2b1,
  (q15_t)0x77b4, (q15_t)0xd2ab, (q15_t)0x77b1, (q15_t)0xd2a5, (q15_t)0x77af, (q15_t)0xd2a0, (q15_t)0x77ad, (q15_t)0xd29a,
  (q15_t)0x77ab, (q15_t)0xd294, (q15_t)0x77a8, (q15_t)0xd28e, (q15_t)0x77a6, (q15_t)0xd288, (q15_t)0x77a4, (q15_t)0xd282,
  (q15_t)0x77a2, (q15_t)0xd27c, (q15_t)0x77a0, (q15_t)0xd276, (q15_t)0x779d, (q15_t)0xd271, (q15_t)0x779b, (q15_t)0xd26b,
  (q15_t)0x7799, (q15_t)0xd265, (q15_t)0x7797, (q15_t)0xd25f, (q15_t)0x7794, (q15_t)0xd259, (q15_t)0x7792, (q15_t)0xd253,
  (q15_t)0x7790, (q15_t)0xd24d, (q15_t)0x778e, (q15_t)0xd247, (q15_t)0x778b, (q15_t)0xd242, (q15_t)0x7789, (q15_t)0xd23c,
  (q15_t)0x7787, (q15_t)0xd236, (q15_t)0x7785, (q15_t)0xd230, (q15_t)0x7782, (q15_t)0xd22a, (q15_t)0x7780, (q15_t)0xd224,
  (q15_t)0x777e, (q15_t)0xd21e, (q15_t)0x777c, (q15_t)0xd219, (q15_t)0x7779, (q15_t)0xd213, (q15_t)0x7777, (q15_t)0xd20d,
  (q15_t)0x7775, (q15_t)0xd207, (q15_t)0x7773, (q15_t)0xd201, (q15_t)0x7770, (q15_t)0xd1fb, (q15_t)0x776e, (q15_t)0xd1f5,
  (q15_t)0x776c, (q15_t)0xd1ef, (q15_t)0x776a, (q15_t)0xd1ea, (q15_t)0x7767, (q15_t)0xd1e4, (q15_t)0x7765, (q15_t)0xd1de,
  (q15_t)0x7763, (q15_t)0xd1d8, (q15_t)0x7760, (q15_t)0xd1d2, (q15_t)0x775e, (q15_t)0xd1cc, (q15_t)0x775c, (q15_t)0xd1c6,
  (q15_t)0x775a, (q15_t)0xd1c1, (q15_t)0x7757, (q15_t)0xd1bb, (q15_t)0x7755, (q15_t)0xd1b5, (q15_t)0x7753, (q15_t)0xd1af,
  (q15_t)0x7751, (q15_t)0xd1a9, (q15_t)0x774e, (q15_t)0xd1a3, (q15_t)0x774c, (q15_t)0xd19d, (q15_t)0x774a, (q15_t)0xd198,
  (q15_t)0x7747, (q15_t)0xd192, (q15_t)0x7745, (q15_t)0xd18c, (q15_t)0x7743, (q15_t)0xd186, (q15_t)0x7741, (q15_t)0xd180,
  (q15_t)0x773e, (q15_t)0xd17a, (q15_t)0x773c, (q15_t)0xd174, (q15_t)0x773a, (q15_t)0xd16f, (q15_t)0x7738, (q15_t)0xd169,
  (q15_t)0x7735, (q15_t)0xd163, (q15_t)0x7733, (q15_t)0xd15d, (q15_t)0x7731, (q15_t)0xd157, (q15_t)0x772e, (q15_t)0xd151,
  (q15_t)0x772c, (q15_t)0xd14b, (q15_t)0x772a, (q15_t)0xd146, (q15_t)0x7727, (q15_t)0xd140, (q15_t)0x7725, (q15_t)0xd13a,
  (q15_t)0x7723, (q15_t)0xd134, (q15_t)0x7721, (q15_t)0xd12e, (q15_t)0x771e, (q15_t)0xd128, (q15_t)0x771c, (q15_t)0xd123,
  (q15_t)0x771a, (q15_t)0xd11d, (q15_t)0x7717, (q15_t)0xd117, (q15_t)0x7715, (q15_t)0xd111, (q15_t)0x7713, (q15_t)0xd10b,
  (q15_t)0x7710, (q15_t)0xd105, (q15_t)0x770e, (q15_t)0xd0ff, (q15_t)0x770c, (q15_t)0xd0fa, (q15_t)0x770a, (q15_t)0xd0f4,
  (q15_t)0x7707, (q15_t)0xd0ee, (q15_t)0x7705, (q15_t)0xd0e8, (q15_t)0x7703, (q15_t)0xd0e2, (q15_t)0x7700, (q15_t)0xd0dc,
  (q15_t)0x76fe, (q15_t)0xd0d7, (q15_t)0x76fc, (q15_t)0xd0d1, (q15_t)0x76f9, (q15_t)0xd0cb, (q15_t)0x76f7, (q15_t)0xd0c5,
  (q15_t)0x76f5, (q15_t)0xd0bf, (q15_t)0x76f2, (q15_t)0xd0b9, (q15_t)0x76f0, (q15_t)0xd0b4, (q15_t)0x76ee, (q15_t)0xd0ae,
  (q15_t)0x76eb, (q15_t)0xd0a8, (q15_t)0x76e9, (q15_t)0xd0a2, (q15_t)0x76e7, (q15_t)0xd09c, (q15_t)0x76e4, (q15_t)0xd096,
  (q15_t)0x76e2, (q15_t)0xd091, (q15_t)0x76e0, (q15_t)0xd08b, (q15_t)0x76dd, (q15_t)0xd085, (q15_t)0x76db, (q15_t)0xd07f,
  (q15_t)0x76d9, (q15_t)0xd079, (q15_t)0x76d6, (q15_t)0xd073, (q15_t)0x76d4, (q15_t)0xd06e, (q15_t)0x76d2, (q15_t)0xd068,
  (q15_t)0x76cf, (q15_t)0xd062, (q15_t)0x76cd, (q15_t)0xd05c, (q15_t)0x76cb, (q15_t)0xd056, (q15_t)0x76c8, (q15_t)0xd050,
  (q15_t)0x76c6, (q15_t)0xd04b, (q15_t)0x76c4, (q15_t)0xd045, (q15_t)0x76c1, (q15_t)0xd03f, (q15_t)0x76bf, (q15_t)0xd039,
  (q15_t)0x76bd, (q15_t)0xd033, (q15_t)0x76ba, (q15_t)0xd02d, (q15_t)0x76b8, (q15_t)0xd028, (q15_t)0x76b6, (q15_t)0xd022,
  (q15_t)0x76b3, (q15_t)0xd01c, (q15_t)0x76b1, (q15_t)0xd016, (q15_t)0x76af, (q15_t)0xd010, (q15_t)0x76ac, (q15_t)0xd00a,
  (q15_t)0x76aa, (q15_t)0xd005, (q15_t)0x76a8, (q15_t)0xcfff, (q15_t)0x76a5, (q15_t)0xcff9, (q15_t)0x76a3, (q15_t)0xcff3,
  (q15_t)0x76a0, (q15_t)0xcfed, (q15_t)0x769e, (q15_t)0xcfe7, (q15_t)0x769c, (q15_t)0xcfe2, (q15_t)0x7699, (q15_t)0xcfdc,
  (q15_t)0x7697, (q15_t)0xcfd6, (q15_t)0x7695, (q15_t)0xcfd0, (q15_t)0x7692, (q15_t)0xcfca, (q15_t)0x7690, (q15_t)0xcfc5,
  (q15_t)0x768e, (q15_t)0xcfbf, (q15_t)0x768b, (q15_t)0xcfb9, (q15_t)0x7689, (q15_t)0xcfb3, (q15_t)0x7686, (q15_t)0xcfad,
  (q15_t)0x7684, (q15_t)0xcfa7, (q15_t)0x7682, (q15_t)0xcfa2, (q15_t)0x767f, (q15_t)0xcf9c, (q15_t)0x767d, (q15_t)0xcf96,
  (q15_t)0x767b, (q15_t)0xcf90, (q15_t)0x7678, (q15_t)0xcf8a, (q15_t)0x7676, (q15_t)0xcf85, (q15_t)0x7673, (q15_t)0xcf7f,
  (q15_t)0x7671, (q15_t)0xcf79, (q15_t)0x766f, (q15_t)0xcf73, (q15_t)0x766c, (q15_t)0xcf6d, (q15_t)0x766a, (q15_t)0xcf67,
  (q15_t)0x7668, (q15_t)0xcf62, (q15_t)0x7665, (q15_t)0xcf5c, (q15_t)0x7663, (q15_t)0xcf56, (q15_t)0x7660, (q15_t)0xcf50,
  (q15_t)0x765e, (q15_t)0xcf4a, (q15_t)0x765c, (q15_t)0xcf45, (q15_t)0x7659, (q15_t)0xcf3f, (q15_t)0x7657, (q15_t)0xcf39,
  (q15_t)0x7654, (q15_t)0xcf33, (q15_t)0x7652, (q15_t)0xcf2d, (q15_t)0x7650, (q15_t)0xcf28, (q15_t)0x764d, (q15_t)0xcf22,
  (q15_t)0x764b, (q15_t)0xcf1c, (q15_t)0x7648, (q15_t)0xcf16, (q15_t)0x7646, (q15_t)0xcf10, (q15_t)0x7644, (q15_t)0xcf0b,
  (q15_t)0x7641, (q15_t)0xcf05, (q15_t)0x763f, (q15_t)0xceff, (q15_t)0x763c, (q15_t)0xcef9, (q15_t)0x763a, (q15_t)0xcef3,
  (q15_t)0x7638, (q15_t)0xceee, (q15_t)0x7635, (q15_t)0xcee8, (q15_t)0x7633, (q15_t)0xcee2, (q15_t)0x7630, (q15_t)0xcedc,
  (q15_t)0x762e, (q15_t)0xced6, (q15_t)0x762b, (q15_t)0xced1, (q15_t)0x7629, (q15_t)0xcecb, (q15_t)0x7627, (q15_t)0xcec5,
  (q15_t)0x7624, (q15_t)0xcebf, (q15_t)0x7622, (q15_t)0xceb9, (q15_t)0x761f, (q15_t)0xceb4, (q15_t)0x761d, (q15_t)0xceae,
  (q15_t)0x761b, (q15_t)0xcea8, (q15_t)0x7618, (q15_t)0xcea2, (q15_t)0x7616, (q15_t)0xce9c, (q15_t)0x7613, (q15_t)0xce97,
  (q15_t)0x7611, (q15_t)0xce91, (q15_t)0x760e, (q15_t)0xce8b, (q15_t)0x760c, (q15_t)0xce85, (q15_t)0x760a, (q15_t)0xce7f,
  (q15_t)0x7607, (q15_t)0xce7a, (q15_t)0x7605, (q15_t)0xce74, (q15_t)0x7602, (q15_t)0xce6e, (q15_t)0x7600, (q15_t)0xce68,
  (q15_t)0x75fd, (q15_t)0xce62, (q15_t)0x75fb, (q15_t)0xce5d, (q15_t)0x75f9, (q15_t)0xce57, (q15_t)0x75f6, (q15_t)0xce51,
  (q15_t)0x75f4, (q15_t)0xce4b, (q15_t)0x75f1, (q15_t)0xce45, (q15_t)0x75ef, (q15_t)0xce40, (q15_t)0x75ec, (q15_t)0xce3a,
  (q15_t)0x75ea, (q15_t)0xce34, (q15_t)0x75e7, (q15_t)0xce2e, (q15_t)0x75e5, (q15_t)0xce28, (q15_t)0x75e3, (q15_t)0xce23,
  (q15_t)0x75e0, (q15_t)0xce1d, (q15_t)0x75de, (q15_t)0xce17, (q15_t)0x75db, (q15_t)0xce11, (q15_t)0x75d9, (q15_t)0xce0c,
  (q15_t)0x75d6, (q15_t)0xce06, (q15_t)0x75d4, (q15_t)0xce00, (q15_t)0x75d1, (q15_t)0xcdfa, (q15_t)0x75cf, (q15_t)0xcdf4,
  (q15_t)0x75cc, (q15_t)0xcdef, (q15_t)0x75ca, (q15_t)0xcde9, (q15_t)0x75c8, (q15_t)0xcde3, (q15_t)0x75c5, (q15_t)0xcddd,
  (q15_t)0x75c3, (q15_t)0xcdd8, (q15_t)0x75c0, (q15_t)0xcdd2, (q15_t)0x75be, (q15_t)0xcdcc, (q15_t)0x75bb, (q15_t)0xcdc6,
  (q15_t)0x75b9, (q15_t)0xcdc0, (q15_t)0x75b6, (q15_t)0xcdbb, (q15_t)0x75b4, (q15_t)0xcdb5, (q15_t)0x75b1, (q15_t)0xcdaf,
  (q15_t)0x75af, (q15_t)0xcda9, (q15_t)0x75ac, (q15_t)0xcda3, (q15_t)0x75aa, (q15_t)0xcd9e, (q15_t)0x75a7, (q15_t)0xcd98,
  (q15_t)0x75a5, (q15_t)0xcd92, (q15_t)0x75a3, (q15_t)0xcd8c, (q15_t)0x75a0, (q15_t)0xcd87, (q15_t)0x759e, (q15_t)0xcd81,
  (q15_t)0x759b, (q15_t)0xcd7b, (q15_t)0x7599, (q15_t)0xcd75, (q15_t)0x7596, (q15_t)0xcd70, (q15_t)0x7594, (q15_t)0xcd6a,
  (q15_t)0x7591, (q15_t)0xcd64, (q15_t)0x758f, (q15_t)0xcd5e, (q15_t)0x758c, (q15_t)0xcd58, (q15_t)0x758a, (q15_t)0xcd53,
  (q15_t)0x7587, (q15_t)0xcd4d, (q15_t)0x7585, (q15_t)0xcd47, (q15_t)0x7582, (q15_t)0xcd41, (q15_t)0x7580, (q15_t)0xcd3c,
  (q15_t)0x757d, (q15_t)0xcd36, (q15_t)0x757b, (q15_t)0xcd30, (q15_t)0x7578, (q15_t)0xcd2a, (q15_t)0x7576, (q15_t)0xcd25,
  (q15_t)0x7573, (q15_t)0xcd1f, (q15_t)0x7571, (q15_t)0xcd19, (q15_t)0x756e, (q15_t)0xcd13, (q15_t)0x756c, (q15_t)0xcd0d,
  (q15_t)0x7569, (q15_t)0xcd08, (q15_t)0x7567, (q15_t)0xcd02, (q15_t)0x7564, (q15_t)0xccfc, (q15_t)0x7562, (q15_t)0xccf6,
  (q15_t)0x755f, (q15_t)0xccf1, (q15_t)0x755d, (q15_t)0xcceb, (q15_t)0x755a, (q15_t)0xcce5, (q15_t)0x7558, (q15_t)0xccdf,
  (q15_t)0x7555, (q15_t)0xccda, (q15_t)0x7553, (q15_t)0xccd4, (q15_t)0x7550, (q15_t)0xccce, (q15_t)0x754e, (q15_t)0xccc8,
  (q15_t)0x754b, (q15_t)0xccc3, (q15_t)0x7549, (q15_t)0xccbd, (q15_t)0x7546, (q15_t)0xccb7, (q15_t)0x7544, (q15_t)0xccb1,
  (q15_t)0x7541, (q15_t)0xccac, (q15_t)0x753f, (q15_t)0xcca6, (q15_t)0x753c, (q15_t)0xcca0, (q15_t)0x753a, (q15_t)0xcc9a,
  (q15_t)0x7537, (q15_t)0xcc95, (q15_t)0x7535, (q15_t)0xcc8f, (q15_t)0x7532, (q15_t)0xcc89, (q15_t)0x752f, (q15_t)0xcc83,
  (q15_t)0x752d, (q15_t)0xcc7e, (q15_t)0x752a, (q15_t)0xcc78, (q15_t)0x7528, (q15_t)0xcc72, (q15_t)0x7525, (q15_t)0xcc6c,
  (q15_t)0x7523, (q15_t)0xcc67, (q15_t)0x7520, (q15_t)0xcc61, (q15_t)0x751e, (q15_t)0xcc5b, (q15_t)0x751b, (q15_t)0xcc55,
  (q15_t)0x7519, (q15_t)0xcc50, (q15_t)0x7516, (q15_t)0xcc4a, (q15_t)0x7514, (q15_t)0xcc44, (q15_t)0x7511, (q15_t)0xcc3e,
  (q15_t)0x750f, (q15_t)0xcc39, (q15_t)0x750c, (q15_t)0xcc33, (q15_t)0x7509, (q15_t)0xcc2d, (q15_t)0x7507, (q15_t)0xcc27,
  (q15_t)0x7504, (q15_t)0xcc22, (q15_t)0x7502, (q15_t)0xcc1c, (q15_t)0x74ff, (q15_t)0xcc16, (q15_t)0x74fd, (q15_t)0xcc10,
  (q15_t)0x74fa, (q15_t)0xcc0b, (q15_t)0x74f8, (q15_t)0xcc05, (q15_t)0x74f5, (q15_t)0xcbff, (q15_t)0x74f2, (q15_t)0xcbf9,
  (q15_t)0x74f0, (q15_t)0xcbf4, (q15_t)0x74ed, (q15_t)0xcbee, (q15_t)0x74eb, (q15_t)0xcbe8, (q15_t)0x74e8, (q15_t)0xcbe2,
  (q15_t)0x74e6, (q15_t)0xcbdd, (q15_t)0x74e3, (q15_t)0xcbd7, (q15_t)0x74e1, (q15_t)0xcbd1, (q15_t)0x74de, (q15_t)0xcbcb,
  (q15_t)0x74db, (q15_t)0xcbc6, (q15_t)0x74d9, (q15_t)0xcbc0, (q15_t)0x74d6, (q15_t)0xcbba, (q15_t)0x74d4, (q15_t)0xcbb5,
  (q15_t)0x74d1, (q15_t)0xcbaf, (q15_t)0x74cf, (q15_t)0xcba9, (q15_t)0x74cc, (q15_t)0xcba3, (q15_t)0x74c9, (q15_t)0xcb9e,
  (q15_t)0x74c7, (q15_t)0xcb98, (q15_t)0x74c4, (q15_t)0xcb92, (q15_t)0x74c2, (q15_t)0xcb8c, (q15_t)0x74bf, (q15_t)0xcb87,
  (q15_t)0x74bd, (q15_t)0xcb81, (q15_t)0x74ba, (q15_t)0xcb7b, (q15_t)0x74b7, (q15_t)0xcb75, (q15_t)0x74b5, (q15_t)0xcb70,
  (q15_t)0x74b2, (q15_t)0xcb6a, (q15_t)0x74b0, (q15_t)0xcb64, (q15_t)0x74ad, (q15_t)0xcb5f, (q15_t)0x74ab, (q15_t)0xcb59,
  (q15_t)0x74a8, (q15_t)0xcb53, (q15_t)0x74a5, (q15_t)0xcb4d, (q15_t)0x74a3, (q15_t)0xcb48, (q15_t)0x74a0, (q15_t)0xcb42,
  (q15_t)0x749e, (q15_t)0xcb3c, (q15_t)0x749b, (q15_t)0xcb36, (q15_t)0x7498, (q15_t)0xcb31, (q15_t)0x7496, (q15_t)0xcb2b,
  (q15_t)0x7493, (q15_t)0xcb25, (q15_t)0x7491, (q15_t)0xcb20, (q15_t)0x748e, (q15_t)0xcb1a, (q15_t)0x748b, (q15_t)0xcb14,
  (q15_t)0x7489, (q15_t)0xcb0e, (q15_t)0x7486, (q15_t)0xcb09, (q15_t)0x7484, (q15_t)0xcb03, (q15_t)0x7481, (q15_t)0xcafd,
  (q15_t)0x747e, (q15_t)0xcaf8, (q15_t)0x747c, (q15_t)0xcaf2, (q15_t)0x7479, (q15_t)0xcaec, (q15_t)0x7477, (q15_t)0xcae6,
  (q15_t)0x7474, (q15_t)0xcae1, (q15_t)0x7471, (q15_t)0xcadb, (q15_t)0x746f, (q15_t)0xcad5, (q15_t)0x746c, (q15_t)0xcad0,
  (q15_t)0x746a, (q15_t)0xcaca, (q15_t)0x7467, (q15_t)0xcac4, (q15_t)0x7464, (q15_t)0xcabe, (q15_t)0x7462, (q15_t)0xcab9,
  (q15_t)0x745f, (q15_t)0xcab3, (q15_t)0x745c, (q15_t)0xcaad, (q15_t)0x745a, (q15_t)0xcaa8, (q15_t)0x7457, (q15_t)0xcaa2,
  (q15_t)0x7455, (q15_t)0xca9c, (q15_t)0x7452, (q15_t)0xca96, (q15_t)0x744f, (q15_t)0xca91, (q15_t)0x744d, (q15_t)0xca8b,
  (q15_t)0x744a, (q15_t)0xca85, (q15_t)0x7448, (q15_t)0xca80, (q15_t)0x7445, (q15_t)0xca7a, (q15_t)0x7442, (q15_t)0xca74,
  (q15_t)0x7440, (q15_t)0xca6e, (q15_t)0x743d, (q15_t)0xca69, (q15_t)0x743a, (q15_t)0xca63, (q15_t)0x7438, (q15_t)0xca5d,
  (q15_t)0x7435, (q15_t)0xca58, (q15_t)0x7432, (q15_t)0xca52, (q15_t)0x7430, (q15_t)0xca4c, (q15_t)0x742d, (q15_t)0xca46,
  (q15_t)0x742b, (q15_t)0xca41, (q15_t)0x7428, (q15_t)0xca3b, (q15_t)0x7425, (q15_t)0xca35, (q15_t)0x7423, (q15_t)0xca30,
  (q15_t)0x7420, (q15_t)0xca2a, (q15_t)0x741d, (q15_t)0xca24, (q15_t)0x741b, (q15_t)0xca1f, (q15_t)0x7418, (q15_t)0xca19,
  (q15_t)0x7415, (q15_t)0xca13, (q15_t)0x7413, (q15_t)0xca0d, (q15_t)0x7410, (q15_t)0xca08, (q15_t)0x740d, (q15_t)0xca02,
  (q15_t)0x740b, (q15_t)0xc9fc, (q15_t)0x7408, (q15_t)0xc9f7, (q15_t)0x7406, (q15_t)0xc9f1, (q15_t)0x7403, (q15_t)0xc9eb,
  (q15_t)0x7400, (q15_t)0xc9e6, (q15_t)0x73fe, (q15_t)0xc9e0, (q15_t)0x73fb, (q15_t)0xc9da, (q15_t)0x73f8, (q15_t)0xc9d5,
  (q15_t)0x73f6, (q15_t)0xc9cf, (q15_t)0x73f3, (q15_t)0xc9c9, (q15_t)0x73f0, (q15_t)0xc9c3, (q15_t)0x73ee, (q15_t)0xc9be,
  (q15_t)0x73eb, (q15_t)0xc9b8, (q15_t)0x73e8, (q15_t)0xc9b2, (q15_t)0x73e6, (q15_t)0xc9ad, (q15_t)0x73e3, (q15_t)0xc9a7,
  (q15_t)0x73e0, (q15_t)0xc9a1, (q15_t)0x73de, (q15_t)0xc99c, (q15_t)0x73db, (q15_t)0xc996, (q15_t)0x73d8, (q15_t)0xc990,
  (q15_t)0x73d6, (q15_t)0xc98b, (q15_t)0x73d3, (q15_t)0xc985, (q15_t)0x73d0, (q15_t)0xc97f, (q15_t)0x73ce, (q15_t)0xc97a,
  (q15_t)0x73cb, (q15_t)0xc974, (q15_t)0x73c8, (q15_t)0xc96e, (q15_t)0x73c6, (q15_t)0xc968, (q15_t)0x73c3, (q15_t)0xc963,
  (q15_t)0x73c0, (q15_t)0xc95d, (q15_t)0x73bd, (q15_t)0xc957, (q15_t)0x73bb, (q15_t)0xc952, (q15_t)0x73b8, (q15_t)0xc94c,
  (q15_t)0x73b5, (q15_t)0xc946, (q15_t)0x73b3, (q15_t)0xc941, (q15_t)0x73b0, (q15_t)0xc93b, (q15_t)0x73ad, (q15_t)0xc935,
  (q15_t)0x73ab, (q15_t)0xc930, (q15_t)0x73a8, (q15_t)0xc92a, (q15_t)0x73a5, (q15_t)0xc924, (q15_t)0x73a3, (q15_t)0xc91f,
  (q15_t)0x73a0, (q15_t)0xc919, (q15_t)0x739d, (q15_t)0xc913, (q15_t)0x739b, (q15_t)0xc90e, (q15_t)0x7398, (q15_t)0xc908,
  (q15_t)0x7395, (q15_t)0xc902, (q15_t)0x7392, (q15_t)0xc8fd, (q15_t)0x7390, (q15_t)0xc8f7, (q15_t)0x738d, (q15_t)0xc8f1,
  (q15_t)0x738a, (q15_t)0xc8ec, (q15_t)0x7388, (q15_t)0xc8e6, (q15_t)0x7385, (q15_t)0xc8e0, (q15_t)0x7382, (q15_t)0xc8db,
  (q15_t)0x737f, (q15_t)0xc8d5, (q15_t)0x737d, (q15_t)0xc8cf, (q15_t)0x737a, (q15_t)0xc8ca, (q15_t)0x7377, (q15_t)0xc8c4,
  (q15_t)0x7375, (q15_t)0xc8be, (q15_t)0x7372, (q15_t)0xc8b9, (q15_t)0x736f, (q15_t)0xc8b3, (q15_t)0x736c, (q15_t)0xc8ad,
  (q15_t)0x736a, (q15_t)0xc8a8, (q15_t)0x7367, (q15_t)0xc8a2, (q15_t)0x7364, (q15_t)0xc89c, (q15_t)0x7362, (q15_t)0xc897,
  (q15_t)0x735f, (q15_t)0xc891, (q15_t)0x735c, (q15_t)0xc88b, (q15_t)0x7359, (q15_t)0xc886, (q15_t)0x7357, (q15_t)0xc880,
  (q15_t)0x7354, (q15_t)0xc87a, (q15_t)0x7351, (q15_t)0xc875, (q15_t)0x734f, (q15_t)0xc86f, (q15_t)0x734c, (q15_t)0xc869,
  (q15_t)0x7349, (q15_t)0xc864, (q15_t)0x7346, (q15_t)0xc85e, (q15_t)0x7344, (q15_t)0xc858, (q15_t)0x7341, (q15_t)0xc853,
  (q15_t)0x733e, (q15_t)0xc84d, (q15_t)0x733b, (q15_t)0xc847, (q15_t)0x7339, (q15_t)0xc842, (q15_t)0x7336, (q15_t)0xc83c,
  (q15_t)0x7333, (q15_t)0xc836, (q15_t)0x7330, (q15_t)0xc831, (q15_t)0x732e, (q15_t)0xc82b, (q15_t)0x732b, (q15_t)0xc825,
  (q15_t)0x7328, (q15_t)0xc820, (q15_t)0x7326, (q15_t)0xc81a, (q15_t)0x7323, (q15_t)0xc814, (q15_t)0x7320, (q15_t)0xc80f,
  (q15_t)0x731d, (q15_t)0xc809, (q15_t)0x731b, (q15_t)0xc803, (q15_t)0x7318, (q15_t)0xc7fe, (q15_t)0x7315, (q15_t)0xc7f8,
  (q15_t)0x7312, (q15_t)0xc7f3, (q15_t)0x7310, (q15_t)0xc7ed, (q15_t)0x730d, (q15_t)0xc7e7, (q15_t)0x730a, (q15_t)0xc7e2,
  (q15_t)0x7307, (q15_t)0xc7dc, (q15_t)0x7305, (q15_t)0xc7d6, (q15_t)0x7302, (q15_t)0xc7d1, (q15_t)0x72ff, (q15_t)0xc7cb,
  (q15_t)0x72fc, (q15_t)0xc7c5, (q15_t)0x72f9, (q15_t)0xc7c0, (q15_t)0x72f7, (q15_t)0xc7ba, (q15_t)0x72f4, (q15_t)0xc7b4,
  (q15_t)0x72f1, (q15_t)0xc7af, (q15_t)0x72ee, (q15_t)0xc7a9, (q15_t)0x72ec, (q15_t)0xc7a3, (q15_t)0x72e9, (q15_t)0xc79e,
  (q15_t)0x72e6, (q15_t)0xc798, (q15_t)0x72e3, (q15_t)0xc793, (q15_t)0x72e1, (q15_t)0xc78d, (q15_t)0x72de, (q15_t)0xc787,
  (q15_t)0x72db, (q15_t)0xc782, (q15_t)0x72d8, (q15_t)0xc77c, (q15_t)0x72d5, (q15_t)0xc776, (q15_t)0x72d3, (q15_t)0xc771,
  (q15_t)0x72d0, (q15_t)0xc76b, (q15_t)0x72cd, (q15_t)0xc765, (q15_t)0x72ca, (q15_t)0xc760, (q15_t)0x72c8, (q15_t)0xc75a,
  (q15_t)0x72c5, (q15_t)0xc755, (q15_t)0x72c2, (q15_t)0xc74f, (q15_t)0x72bf, (q15_t)0xc749, (q15_t)0x72bc, (q15_t)0xc744,
  (q15_t)0x72ba, (q15_t)0xc73e, (q15_t)0x72b7, (q15_t)0xc738, (q15_t)0x72b4, (q15_t)0xc733, (q15_t)0x72b1, (q15_t)0xc72d,
  (q15_t)0x72af, (q15_t)0xc728, (q15_t)0x72ac, (q15_t)0xc722, (q15_t)0x72a9, (q15_t)0xc71c, (q15_t)0x72a6, (q15_t)0xc717,
  (q15_t)0x72a3, (q15_t)0xc711, (q15_t)0x72a1, (q15_t)0xc70b, (q15_t)0x729e, (q15_t)0xc706, (q15_t)0x729b, (q15_t)0xc700,
  (q15_t)0x7298, (q15_t)0xc6fa, (q15_t)0x7295, (q15_t)0xc6f5, (q15_t)0x7293, (q15_t)0xc6ef, (q15_t)0x7290, (q15_t)0xc6ea,
  (q15_t)0x728d, (q15_t)0xc6e4, (q15_t)0x728a, (q15_t)0xc6de, (q15_t)0x7287, (q15_t)0xc6d9, (q15_t)0x7285, (q15_t)0xc6d3,
  (q15_t)0x7282, (q15_t)0xc6ce, (q15_t)0x727f, (q15_t)0xc6c8, (q15_t)0x727c, (q15_t)0xc6c2, (q15_t)0x7279, (q15_t)0xc6bd,
  (q15_t)0x7276, (q15_t)0xc6b7, (q15_t)0x7274, (q15_t)0xc6b1, (q15_t)0x7271, (q15_t)0xc6ac, (q15_t)0x726e, (q15_t)0xc6a6,
  (q15_t)0x726b, (q15_t)0xc6a1, (q15_t)0x7268, (q15_t)0xc69b, (q15_t)0x7266, (q15_t)0xc695, (q15_t)0x7263, (q15_t)0xc690,
  (q15_t)0x7260, (q15_t)0xc68a, (q15_t)0x725d, (q15_t)0xc684, (q15_t)0x725a, (q15_t)0xc67f, (q15_t)0x7257, (q15_t)0xc679,
  (q15_t)0x7255, (q15_t)0xc674, (q15_t)0x7252, (q15_t)0xc66e, (q15_t)0x724f, (q15_t)0xc668, (q15_t)0x724c, (q15_t)0xc663,
  (q15_t)0x7249, (q15_t)0xc65d, (q15_t)0x7247, (q15_t)0xc658, (q15_t)0x7244, (q15_t)0xc652, (q15_t)0x7241, (q15_t)0xc64c,
  (q15_t)0x723e, (q15_t)0xc647, (q15_t)0x723b, (q15_t)0xc641, (q15_t)0x7238, (q15_t)0xc63c, (q15_t)0x7236, (q15_t)0xc636,
  (q15_t)0x7233, (q15_t)0xc630, (q15_t)0x7230, (q15_t)0xc62b, (q15_t)0x722d, (q15_t)0xc625, (q15_t)0x722a, (q15_t)0xc620,
  (q15_t)0x7227, (q15_t)0xc61a, (q15_t)0x7224, (q15_t)0xc614, (q15_t)0x7222, (q15_t)0xc60f, (q15_t)0x721f, (q15_t)0xc609,
  (q15_t)0x721c, (q15_t)0xc603, (q15_t)0x7219, (q15_t)0xc5fe, (q15_t)0x7216, (q15_t)0xc5f8, (q15_t)0x7213, (q15_t)0xc5f3,
  (q15_t)0x7211, (q15_t)0xc5ed, (q15_t)0x720e, (q15_t)0xc5e7, (q15_t)0x720b, (q15_t)0xc5e2, (q15_t)0x7208, (q15_t)0xc5dc,
  (q15_t)0x7205, (q15_t)0xc5d7, (q15_t)0x7202, (q15_t)0xc5d1, (q15_t)0x71ff, (q15_t)0xc5cc, (q15_t)0x71fd, (q15_t)0xc5c6,
  (q15_t)0x71fa, (q15_t)0xc5c0, (q15_t)0x71f7, (q15_t)0xc5bb, (q15_t)0x71f4, (q15_t)0xc5b5, (q15_t)0x71f1, (q15_t)0xc5b0,
  (q15_t)0x71ee, (q15_t)0xc5aa, (q15_t)0x71eb, (q15_t)0xc5a4, (q15_t)0x71e9, (q15_t)0xc59f, (q15_t)0x71e6, (q15_t)0xc599,
  (q15_t)0x71e3, (q15_t)0xc594, (q15_t)0x71e0, (q15_t)0xc58e, (q15_t)0x71dd, (q15_t)0xc588, (q15_t)0x71da, (q15_t)0xc583,
  (q15_t)0x71d7, (q15_t)0xc57d, (q15_t)0x71d4, (q15_t)0xc578, (q15_t)0x71d2, (q15_t)0xc572, (q15_t)0x71cf, (q15_t)0xc56c,
  (q15_t)0x71cc, (q15_t)0xc567, (q15_t)0x71c9, (q15_t)0xc561, (q15_t)0x71c6, (q15_t)0xc55c, (q15_t)0x71c3, (q15_t)0xc556,
  (q15_t)0x71c0, (q15_t)0xc551, (q15_t)0x71bd, (q15_t)0xc54b, (q15_t)0x71bb, (q15_t)0xc545, (q15_t)0x71b8, (q15_t)0xc540,
  (q15_t)0x71b5, (q15_t)0xc53a, (q15_t)0x71b2, (q15_t)0xc535, (q15_t)0x71af, (q15_t)0xc52f, (q15_t)0x71ac, (q15_t)0xc529,
  (q15_t)0x71a9, (q15_t)0xc524, (q15_t)0x71a6, (q15_t)0xc51e, (q15_t)0x71a3, (q15_t)0xc519, (q15_t)0x71a1, (q15_t)0xc513,
  (q15_t)0x719e, (q15_t)0xc50e, (q15_t)0x719b, (q15_t)0xc508, (q15_t)0x7198, (q15_t)0xc502, (q15_t)0x7195, (q15_t)0xc4fd,
  (q15_t)0x7192, (q15_t)0xc4f7, (q15_t)0x718f, (q15_t)0xc4f2, (q15_t)0x718c, (q15_t)0xc4ec, (q15_t)0x7189, (q15_t)0xc4e7,
  (q15_t)0x7186, (q15_t)0xc4e1, (q15_t)0x7184, (q15_t)0xc4db, (q15_t)0x7181, (q15_t)0xc4d6, (q15_t)0x717e, (q15_t)0xc4d0,
  (q15_t)0x717b, (q15_t)0xc4cb, (q15_t)0x7178, (q15_t)0xc4c5, (q15_t)0x7175, (q15_t)0xc4c0, (q15_t)0x7172, (q15_t)0xc4ba,
  (q15_t)0x716f, (q15_t)0xc4b4, (q15_t)0x716c, (q15_t)0xc4af, (q15_t)0x7169, (q15_t)0xc4a9, (q15_t)0x7167, (q15_t)0xc4a4,
  (q15_t)0x7164, (q15_t)0xc49e, (q15_t)0x7161, (q15_t)0xc499, (q15_t)0x715e, (q15_t)0xc493, (q15_t)0x715b, (q15_t)0xc48d,
  (q15_t)0x7158, (q15_t)0xc488, (q15_t)0x7155, (q15_t)0xc482, (q15_t)0x7152, (q15_t)0xc47d, (q15_t)0x714f, (q15_t)0xc477,
  (q15_t)0x714c, (q15_t)0xc472, (q15_t)0x7149, (q15_t)0xc46c, (q15_t)0x7146, (q15_t)0xc467, (q15_t)0x7143, (q15_t)0xc461,
  (q15_t)0x7141, (q15_t)0xc45b, (q15_t)0x713e, (q15_t)0xc456, (q15_t)0x713b, (q15_t)0xc450, (q15_t)0x7138, (q15_t)0xc44b,
  (q15_t)0x7135, (q15_t)0xc445, (q15_t)0x7132, (q15_t)0xc440, (q15_t)0x712f, (q15_t)0xc43a, (q15_t)0x712c, (q15_t)0xc434,
  (q15_t)0x7129, (q15_t)0xc42f, (q15_t)0x7126, (q15_t)0xc429, (q15_t)0x7123, (q15_t)0xc424, (q15_t)0x7120, (q15_t)0xc41e,
  (q15_t)0x711d, (q15_t)0xc419, (q15_t)0x711a, (q15_t)0xc413, (q15_t)0x7117, (q15_t)0xc40e, (q15_t)0x7114, (q15_t)0xc408,
  (q15_t)0x7112, (q15_t)0xc403, (q15_t)0x710f, (q15_t)0xc3fd, (q15_t)0x710c, (q15_t)0xc3f7, (q15_t)0x7109, (q15_t)0xc3f2,
  (q15_t)0x7106, (q15_t)0xc3ec, (q15_t)0x7103, (q15_t)0xc3e7, (q15_t)0x7100, (q15_t)0xc3e1, (q15_t)0x70fd, (q15_t)0xc3dc,
  (q15_t)0x70fa, (q15_t)0xc3d6, (q15_t)0x70f7, (q15_t)0xc3d1, (q15_t)0x70f4, (q15_t)0xc3cb, (q15_t)0x70f1, (q15_t)0xc3c5,
  (q15_t)0x70ee, (q15_t)0xc3c0, (q15_t)0x70eb, (q15_t)0xc3ba, (q15_t)0x70e8, (q15_t)0xc3b5, (q15_t)0x70e5, (q15_t)0xc3af,
  (q15_t)0x70e2, (q15_t)0xc3aa, (q15_t)0x70df, (q15_t)0xc3a4, (q15_t)0x70dc, (q15_t)0xc39f, (q15_t)0x70d9, (q15_t)0xc399,
  (q15_t)0x70d6, (q15_t)0xc394, (q15_t)0x70d3, (q15_t)0xc38e, (q15_t)0x70d1, (q15_t)0xc389, (q15_t)0x70ce, (q15_t)0xc383,
  (q15_t)0x70cb, (q15_t)0xc37d, (q15_t)0x70c8, (q15_t)0xc378, (q15_t)0x70c5, (q15_t)0xc372, (q15_t)0x70c2, (q15_t)0xc36d,
  (q15_t)0x70bf, (q15_t)0xc367, (q15_t)0x70bc, (q15_t)0xc362, (q15_t)0x70b9, (q15_t)0xc35c, (q15_t)0x70b6, (q15_t)0xc357,
  (q15_t)0x70b3, (q15_t)0xc351, (q15_t)0x70b0, (q15_t)0xc34c, (q15_t)0x70ad, (q15_t)0xc346, (q15_t)0x70aa, (q15_t)0xc341,
  (q15_t)0x70a7, (q15_t)0xc33b, (q15_t)0x70a4, (q15_t)0xc336, (q15_t)0x70a1, (q15_t)0xc330, (q15_t)0x709e, (q15_t)0xc32a,
  (q15_t)0x709b, (q15_t)0xc325, (q15_t)0x7098, (q15_t)0xc31f, (q15_t)0x7095, (q15_t)0xc31a, (q15_t)0x7092, (q15_t)0xc314,
  (q15_t)0x708f, (q15_t)0xc30f, (q15_t)0x708c, (q15_t)0xc309, (q15_t)0x7089, (q15_t)0xc304, (q15_t)0x7086, (q15_t)0xc2fe,
  (q15_t)0x7083, (q15_t)0xc2f9, (q15_t)0x7080, (q15_t)0xc2f3, (q15_t)0x707d, (q15_t)0xc2ee, (q15_t)0x707a, (q15_t)0xc2e8,
  (q15_t)0x7077, (q15_t)0xc2e3, (q15_t)0x7074, (q15_t)0xc2dd, (q15_t)0x7071, (q15_t)0xc2d8, (q15_t)0x706e, (q15_t)0xc2d2,
  (q15_t)0x706b, (q15_t)0xc2cd, (q15_t)0x7068, (q15_t)0xc2c7, (q15_t)0x7065, (q15_t)0xc2c2, (q15_t)0x7062, (q15_t)0xc2bc,
  (q15_t)0x705f, (q15_t)0xc2b7, (q15_t)0x705c, (q15_t)0xc2b1, (q15_t)0x7059, (q15_t)0xc2ab, (q15_t)0x7056, (q15_t)0xc2a6,
  (q15_t)0x7053, (q15_t)0xc2a0, (q15_t)0x7050, (q15_t)0xc29b, (q15_t)0x704d, (q15_t)0xc295, (q15_t)0x704a, (q15_t)0xc290,
  (q15_t)0x7047, (q15_t)0xc28a, (q15_t)0x7044, (q15_t)0xc285, (q15_t)0x7041, (q15_t)0xc27f, (q15_t)0x703e, (q15_t)0xc27a,
  (q15_t)0x703b, (q15_t)0xc274, (q15_t)0x7038, (q15_t)0xc26f, (q15_t)0x7035, (q15_t)0xc269, (q15_t)0x7032, (q15_t)0xc264,
  (q15_t)0x702f, (q15_t)0xc25e, (q15_t)0x702c, (q15_t)0xc259, (q15_t)0x7029, (q15_t)0xc253, (q15_t)0x7026, (q15_t)0xc24e,
  (q15_t)0x7023, (q15_t)0xc248, (q15_t)0x7020, (q15_t)0xc243, (q15_t)0x701d, (q15_t)0xc23d, (q15_t)0x7019, (q15_t)0xc238,
  (q15_t)0x7016, (q15_t)0xc232, (q15_t)0x7013, (q15_t)0xc22d, (q15_t)0x7010, (q15_t)0xc227, (q15_t)0x700d, (q15_t)0xc222,
  (q15_t)0x700a, (q15_t)0xc21c, (q15_t)0x7007, (q15_t)0xc217, (q15_t)0x7004, (q15_t)0xc211, (q15_t)0x7001, (q15_t)0xc20c,
  (q15_t)0x6ffe, (q15_t)0xc206, (q15_t)0x6ffb, (q15_t)0xc201, (q15_t)0x6ff8, (q15_t)0xc1fb, (q15_t)0x6ff5, (q15_t)0xc1f6,
  (q15_t)0x6ff2, (q15_t)0xc1f0, (q15_t)0x6fef, (q15_t)0xc1eb, (q15_t)0x6fec, (q15_t)0xc1e5, (q15_t)0x6fe9, (q15_t)0xc1e0,
  (q15_t)0x6fe6, (q15_t)0xc1da, (q15_t)0x6fe3, (q15_t)0xc1d5, (q15_t)0x6fe0, (q15_t)0xc1cf, (q15_t)0x6fdd, (q15_t)0xc1ca,
  (q15_t)0x6fda, (q15_t)0xc1c4, (q15_t)0x6fd6, (q15_t)0xc1bf, (q15_t)0x6fd3, (q15_t)0xc1b9, (q15_t)0x6fd0, (q15_t)0xc1b4,
  (q15_t)0x6fcd, (q15_t)0xc1ae, (q15_t)0x6fca, (q15_t)0xc1a9, (q15_t)0x6fc7, (q15_t)0xc1a3, (q15_t)0x6fc4, (q15_t)0xc19e,
  (q15_t)0x6fc1, (q15_t)0xc198, (q15_t)0x6fbe, (q15_t)0xc193, (q15_t)0x6fbb, (q15_t)0xc18d, (q15_t)0x6fb8, (q15_t)0xc188,
  (q15_t)0x6fb5, (q15_t)0xc183, (q15_t)0x6fb2, (q15_t)0xc17d, (q15_t)0x6faf, (q15_t)0xc178, (q15_t)0x6fac, (q15_t)0xc172,
  (q15_t)0x6fa9, (q15_t)0xc16d, (q15_t)0x6fa5, (q15_t)0xc167, (q15_t)0x6fa2, (q15_t)0xc162, (q15_t)0x6f9f, (q15_t)0xc15c,
  (q15_t)0x6f9c, (q15_t)0xc157, (q15_t)0x6f99, (q15_t)0xc151, (q15_t)0x6f96, (q15_t)0xc14c, (q15_t)0x6f93, (q15_t)0xc146,
  (q15_t)0x6f90, (q15_t)0xc141, (q15_t)0x6f8d, (q15_t)0xc13b, (q15_t)0x6f8a, (q15_t)0xc136, (q15_t)0x6f87, (q15_t)0xc130,
  (q15_t)0x6f84, (q15_t)0xc12b, (q15_t)0x6f81, (q15_t)0xc125, (q15_t)0x6f7d, (q15_t)0xc120, (q15_t)0x6f7a, (q15_t)0xc11a,
  (q15_t)0x6f77, (q15_t)0xc115, (q15_t)0x6f74, (q15_t)0xc10f, (q15_t)0x6f71, (q15_t)0xc10a, (q15_t)0x6f6e, (q15_t)0xc105,
  (q15_t)0x6f6b, (q15_t)0xc0ff, (q15_t)0x6f68, (q15_t)0xc0fa, (q15_t)0x6f65, (q15_t)0xc0f4, (q15_t)0x6f62, (q15_t)0xc0ef,
  (q15_t)0x6f5f, (q15_t)0xc0e9, (q15_t)0x6f5b, (q15_t)0xc0e4, (q15_t)0x6f58, (q15_t)0xc0de, (q15_t)0x6f55, (q15_t)0xc0d9,
  (q15_t)0x6f52, (q15_t)0xc0d3, (q15_t)0x6f4f, (q15_t)0xc0ce, (q15_t)0x6f4c, (q15_t)0xc0c8, (q15_t)0x6f49, (q15_t)0xc0c3,
  (q15_t)0x6f46, (q15_t)0xc0bd, (q15_t)0x6f43, (q15_t)0xc0b8, (q15_t)0x6f3f, (q15_t)0xc0b3, (q15_t)0x6f3c, (q15_t)0xc0ad,
  (q15_t)0x6f39, (q15_t)0xc0a8, (q15_t)0x6f36, (q15_t)0xc0a2, (q15_t)0x6f33, (q15_t)0xc09d, (q15_t)0x6f30, (q15_t)0xc097,
  (q15_t)0x6f2d, (q15_t)0xc092, (q15_t)0x6f2a, (q15_t)0xc08c, (q15_t)0x6f27, (q15_t)0xc087, (q15_t)0x6f23, (q15_t)0xc081,
  (q15_t)0x6f20, (q15_t)0xc07c, (q15_t)0x6f1d, (q15_t)0xc077, (q15_t)0x6f1a, (q15_t)0xc071, (q15_t)0x6f17, (q15_t)0xc06c,
  (q15_t)0x6f14, (q15_t)0xc066, (q15_t)0x6f11, (q15_t)0xc061, (q15_t)0x6f0e, (q15_t)0xc05b, (q15_t)0x6f0b, (q15_t)0xc056,
  (q15_t)0x6f07, (q15_t)0xc050, (q15_t)0x6f04, (q15_t)0xc04b, (q15_t)0x6f01, (q15_t)0xc045, (q15_t)0x6efe, (q15_t)0xc040,
  (q15_t)0x6efb, (q15_t)0xc03b, (q15_t)0x6ef8, (q15_t)0xc035, (q15_t)0x6ef5, (q15_t)0xc030, (q15_t)0x6ef1, (q15_t)0xc02a,
  (q15_t)0x6eee, (q15_t)0xc025, (q15_t)0x6eeb, (q15_t)0xc01f, (q15_t)0x6ee8, (q15_t)0xc01a, (q15_t)0x6ee5, (q15_t)0xc014,
  (q15_t)0x6ee2, (q15_t)0xc00f, (q15_t)0x6edf, (q15_t)0xc00a, (q15_t)0x6edc, (q15_t)0xc004, (q15_t)0x6ed8, (q15_t)0xbfff,
  (q15_t)0x6ed5, (q15_t)0xbff9, (q15_t)0x6ed2, (q15_t)0xbff4, (q15_t)0x6ecf, (q15_t)0xbfee, (q15_t)0x6ecc, (q15_t)0xbfe9,
  (q15_t)0x6ec9, (q15_t)0xbfe3, (q15_t)0x6ec6, (q15_t)0xbfde, (q15_t)0x6ec2, (q15_t)0xbfd9, (q15_t)0x6ebf, (q15_t)0xbfd3,
  (q15_t)0x6ebc, (q15_t)0xbfce, (q15_t)0x6eb9, (q15_t)0xbfc8, (q15_t)0x6eb6, (q15_t)0xbfc3, (q15_t)0x6eb3, (q15_t)0xbfbd,
  (q15_t)0x6eaf, (q15_t)0xbfb8, (q15_t)0x6eac, (q15_t)0xbfb3, (q15_t)0x6ea9, (q15_t)0xbfad, (q15_t)0x6ea6, (q15_t)0xbfa8,
  (q15_t)0x6ea3, (q15_t)0xbfa2, (q15_t)0x6ea0, (q15_t)0xbf9d, (q15_t)0x6e9c, (q15_t)0xbf97, (q15_t)0x6e99, (q15_t)0xbf92,
  (q15_t)0x6e96, (q15_t)0xbf8d, (q15_t)0x6e93, (q15_t)0xbf87, (q15_t)0x6e90, (q15_t)0xbf82, (q15_t)0x6e8d, (q15_t)0xbf7c,
  (q15_t)0x6e89, (q15_t)0xbf77, (q15_t)0x6e86, (q15_t)0xbf71, (q15_t)0x6e83, (q15_t)0xbf6c, (q15_t)0x6e80, (q15_t)0xbf67,
  (q15_t)0x6e7d, (q15_t)0xbf61, (q15_t)0x6e7a, (q15_t)0xbf5c, (q15_t)0x6e76, (q15_t)0xbf56, (q15_t)0x6e73, (q15_t)0xbf51,
  (q15_t)0x6e70, (q15_t)0xbf4b, (q15_t)0x6e6d, (q15_t)0xbf46, (q15_t)0x6e6a, (q15_t)0xbf41, (q15_t)0x6e67, (q15_t)0xbf3b,
  (q15_t)0x6e63, (q15_t)0xbf36, (q15_t)0x6e60, (q15_t)0xbf30, (q15_t)0x6e5d, (q15_t)0xbf2b, (q15_t)0x6e5a, (q15_t)0xbf26,
  (q15_t)0x6e57, (q15_t)0xbf20, (q15_t)0x6e53, (q15_t)0xbf1b, (q15_t)0x6e50, (q15_t)0xbf15, (q15_t)0x6e4d, (q15_t)0xbf10,
  (q15_t)0x6e4a, (q15_t)0xbf0a, (q15_t)0x6e47, (q15_t)0xbf05, (q15_t)0x6e44, (q15_t)0xbf00, (q15_t)0x6e40, (q15_t)0xbefa,
  (q15_t)0x6e3d, (q15_t)0xbef5, (q15_t)0x6e3a, (q15_t)0xbeef, (q15_t)0x6e37, (q15_t)0xbeea, (q15_t)0x6e34, (q15_t)0xbee5,
  (q15_t)0x6e30, (q15_t)0xbedf, (q15_t)0x6e2d, (q15_t)0xbeda, (q15_t)0x6e2a, (q15_t)0xbed4, (q15_t)0x6e27, (q15_t)0xbecf,
  (q15_t)0x6e24, (q15_t)0xbeca, (q15_t)0x6e20, (q15_t)0xbec4, (q15_t)0x6e1d, (q15_t)0xbebf, (q15_t)0x6e1a, (q15_t)0xbeb9,
  (q15_t)0x6e17, (q15_t)0xbeb4, (q15_t)0x6e14, (q15_t)0xbeae, (q15_t)0x6e10, (q15_t)0xbea9, (q15_t)0x6e0d, (q15_t)0xbea4,
  (q15_t)0x6e0a, (q15_t)0xbe9e, (q15_t)0x6e07, (q15_t)0xbe99, (q15_t)0x6e04, (q15_t)0xbe93, (q15_t)0x6e00, (q15_t)0xbe8e,
  (q15_t)0x6dfd, (q15_t)0xbe89, (q15_t)0x6dfa, (q15_t)0xbe83, (q15_t)0x6df7, (q15_t)0xbe7e, (q15_t)0x6df3, (q15_t)0xbe78,
  (q15_t)0x6df0, (q15_t)0xbe73, (q15_t)0x6ded, (q15_t)0xbe6e, (q15_t)0x6dea, (q15_t)0xbe68, (q15_t)0x6de7, (q15_t)0xbe63,
  (q15_t)0x6de3, (q15_t)0xbe5e, (q15_t)0x6de0, (q15_t)0xbe58, (q15_t)0x6ddd, (q15_t)0xbe53, (q15_t)0x6dda, (q15_t)0xbe4d,
  (q15_t)0x6dd6, (q15_t)0xbe48, (q15_t)0x6dd3, (q15_t)0xbe43, (q15_t)0x6dd0, (q15_t)0xbe3d, (q15_t)0x6dcd, (q15_t)0xbe38,
  (q15_t)0x6dca, (q15_t)0xbe32, (q15_t)0x6dc6, (q15_t)0xbe2d, (q15_t)0x6dc3, (q15_t)0xbe28, (q15_t)0x6dc0, (q15_t)0xbe22,
  (q15_t)0x6dbd, (q15_t)0xbe1d, (q15_t)0x6db9, (q15_t)0xbe17, (q15_t)0x6db6, (q15_t)0xbe12, (q15_t)0x6db3, (q15_t)0xbe0d,
  (q15_t)0x6db0, (q15_t)0xbe07, (q15_t)0x6dac, (q15_t)0xbe02, (q15_t)0x6da9, (q15_t)0xbdfd, (q15_t)0x6da6, (q15_t)0xbdf7,
  (q15_t)0x6da3, (q15_t)0xbdf2, (q15_t)0x6d9f, (q15_t)0xbdec, (q15_t)0x6d9c, (q15_t)0xbde7, (q15_t)0x6d99, (q15_t)0xbde2,
  (q15_t)0x6d96, (q15_t)0xbddc, (q15_t)0x6d92, (q15_t)0xbdd7, (q15_t)0x6d8f, (q15_t)0xbdd1, (q15_t)0x6d8c, (q15_t)0xbdcc,
  (q15_t)0x6d89, (q15_t)0xbdc7, (q15_t)0x6d85, (q15_t)0xbdc1, (q15_t)0x6d82, (q15_t)0xbdbc, (q15_t)0x6d7f, (q15_t)0xbdb7,
  (q15_t)0x6d7c, (q15_t)0xbdb1, (q15_t)0x6d78, (q15_t)0xbdac, (q15_t)0x6d75, (q15_t)0xbda6, (q15_t)0x6d72, (q15_t)0xbda1,
  (q15_t)0x6d6f, (q15_t)0xbd9c, (q15_t)0x6d6b, (q15_t)0xbd96, (q15_t)0x6d68, (q15_t)0xbd91, (q15_t)0x6d65, (q15_t)0xbd8c,
  (q15_t)0x6d62, (q15_t)0xbd86, (q15_t)0x6d5e, (q15_t)0xbd81, (q15_t)0x6d5b, (q15_t)0xbd7c, (q15_t)0x6d58, (q15_t)0xbd76,
  (q15_t)0x6d55, (q15_t)0xbd71, (q15_t)0x6d51, (q15_t)0xbd6b, (q15_t)0x6d4e, (q15_t)0xbd66, (q15_t)0x6d4b, (q15_t)0xbd61,
  (q15_t)0x6d48, (q15_t)0xbd5b, (q15_t)0x6d44, (q15_t)0xbd56, (q15_t)0x6d41, (q15_t)0xbd51, (q15_t)0x6d3e, (q15_t)0xbd4b,
  (q15_t)0x6d3a, (q15_t)0xbd46, (q15_t)0x6d37, (q15_t)0xbd40, (q15_t)0x6d34, (q15_t)0xbd3b, (q15_t)0x6d31, (q15_t)0xbd36,
  (q15_t)0x6d2d, (q15_t)0xbd30, (q15_t)0x6d2a, (q15_t)0xbd2b, (q15_t)0x6d27, (q15_t)0xbd26, (q15_t)0x6d23, (q15_t)0xbd20,
  (q15_t)0x6d20, (q15_t)0xbd1b, (q15_t)0x6d1d, (q15_t)0xbd16, (q15_t)0x6d1a, (q15_t)0xbd10, (q15_t)0x6d16, (q15_t)0xbd0b,
  (q15_t)0x6d13, (q15_t)0xbd06, (q15_t)0x6d10, (q15_t)0xbd00, (q15_t)0x6d0c, (q15_t)0xbcfb, (q15_t)0x6d09, (q15_t)0xbcf5,
  (q15_t)0x6d06, (q15_t)0xbcf0, (q15_t)0x6d03, (q15_t)0xbceb, (q15_t)0x6cff, (q15_t)0xbce5, (q15_t)0x6cfc, (q15_t)0xbce0,
  (q15_t)0x6cf9, (q15_t)0xbcdb, (q15_t)0x6cf5, (q15_t)0xbcd5, (q15_t)0x6cf2, (q15_t)0xbcd0, (q15_t)0x6cef, (q15_t)0xbccb,
  (q15_t)0x6cec, (q15_t)0xbcc5, (q15_t)0x6ce8, (q15_t)0xbcc0, (q15_t)0x6ce5, (q15_t)0xbcbb, (q15_t)0x6ce2, (q15_t)0xbcb5,
  (q15_t)0x6cde, (q15_t)0xbcb0, (q15_t)0x6cdb, (q15_t)0xbcab, (q15_t)0x6cd8, (q15_t)0xbca5, (q15_t)0x6cd4, (q15_t)0xbca0,
  (q15_t)0x6cd1, (q15_t)0xbc9b, (q15_t)0x6cce, (q15_t)0xbc95, (q15_t)0x6cca, (q15_t)0xbc90, (q15_t)0x6cc7, (q15_t)0xbc8b,
  (q15_t)0x6cc4, (q15_t)0xbc85, (q15_t)0x6cc1, (q15_t)0xbc80, (q15_t)0x6cbd, (q15_t)0xbc7b, (q15_t)0x6cba, (q15_t)0xbc75,
  (q15_t)0x6cb7, (q15_t)0xbc70, (q15_t)0x6cb3, (q15_t)0xbc6b, (q15_t)0x6cb0, (q15_t)0xbc65, (q15_t)0x6cad, (q15_t)0xbc60,
  (q15_t)0x6ca9, (q15_t)0xbc5b, (q15_t)0x6ca6, (q15_t)0xbc55, (q15_t)0x6ca3, (q15_t)0xbc50, (q15_t)0x6c9f, (q15_t)0xbc4b,
  (q15_t)0x6c9c, (q15_t)0xbc45, (q15_t)0x6c99, (q15_t)0xbc40, (q15_t)0x6c95, (q15_t)0xbc3b, (q15_t)0x6c92, (q15_t)0xbc35,
  (q15_t)0x6c8f, (q15_t)0xbc30, (q15_t)0x6c8b, (q15_t)0xbc2b, (q15_t)0x6c88, (q15_t)0xbc25, (q15_t)0x6c85, (q15_t)0xbc20,
  (q15_t)0x6c81, (q15_t)0xbc1b, (q15_t)0x6c7e, (q15_t)0xbc15, (q15_t)0x6c7b, (q15_t)0xbc10, (q15_t)0x6c77, (q15_t)0xbc0b,
  (q15_t)0x6c74, (q15_t)0xbc05, (q15_t)0x6c71, (q15_t)0xbc00, (q15_t)0x6c6d, (q15_t)0xbbfb, (q15_t)0x6c6a, (q15_t)0xbbf5,
  (q15_t)0x6c67, (q15_t)0xbbf0, (q15_t)0x6c63, (q15_t)0xbbeb, (q15_t)0x6c60, (q15_t)0xbbe5, (q15_t)0x6c5d, (q15_t)0xbbe0,
  (q15_t)0x6c59, (q15_t)0xbbdb, (q15_t)0x6c56, (q15_t)0xbbd5, (q15_t)0x6c53, (q15_t)0xbbd0, (q15_t)0x6c4f, (q15_t)0xbbcb,
  (q15_t)0x6c4c, (q15_t)0xbbc5, (q15_t)0x6c49, (q15_t)0xbbc0, (q15_t)0x6c45, (q15_t)0xbbbb, (q15_t)0x6c42, (q15_t)0xbbb5,
  (q15_t)0x6c3f, (q15_t)0xbbb0, (q15_t)0x6c3b, (q15_t)0xbbab, (q15_t)0x6c38, (q15_t)0xbba6, (q15_t)0x6c34, (q15_t)0xbba0,
  (q15_t)0x6c31, (q15_t)0xbb9b, (q15_t)0x6c2e, (q15_t)0xbb96, (q15_t)0x6c2a, (q15_t)0xbb90, (q15_t)0x6c27, (q15_t)0xbb8b,
  (q15_t)0x6c24, (q15_t)0xbb86, (q15_t)0x6c20, (q15_t)0xbb80, (q15_t)0x6c1d, (q15_t)0xbb7b, (q15_t)0x6c1a, (q15_t)0xbb76,
  (q15_t)0x6c16, (q15_t)0xbb70, (q15_t)0x6c13, (q15_t)0xbb6b, (q15_t)0x6c0f, (q15_t)0xbb66, (q15_t)0x6c0c, (q15_t)0xbb61,
  (q15_t)0x6c09, (q15_t)0xbb5b, (q15_t)0x6c05, (q15_t)0xbb56, (q15_t)0x6c02, (q15_t)0xbb51, (q15_t)0x6bff, (q15_t)0xbb4b,
  (q15_t)0x6bfb, (q15_t)0xbb46, (q15_t)0x6bf8, (q15_t)0xbb41, (q15_t)0x6bf5, (q15_t)0xbb3b, (q15_t)0x6bf1, (q15_t)0xbb36,
  (q15_t)0x6bee, (q15_t)0xbb31, (q15_t)0x6bea, (q15_t)0xbb2c, (q15_t)0x6be7, (q15_t)0xbb26, (q15_t)0x6be4, (q15_t)0xbb21,
  (q15_t)0x6be0, (q15_t)0xbb1c, (q15_t)0x6bdd, (q15_t)0xbb16, (q15_t)0x6bd9, (q15_t)0xbb11, (q15_t)0x6bd6, (q15_t)0xbb0c,
  (q15_t)0x6bd3, (q15_t)0xbb06, (q15_t)0x6bcf, (q15_t)0xbb01, (q15_t)0x6bcc, (q15_t)0xbafc, (q15_t)0x6bc9, (q15_t)0xbaf7,
  (q15_t)0x6bc5, (q15_t)0xbaf1, (q15_t)0x6bc2, (q15_t)0xbaec, (q15_t)0x6bbe, (q15_t)0xbae7, (q15_t)0x6bbb, (q15_t)0xbae1,
  (q15_t)0x6bb8, (q15_t)0xbadc, (q15_t)0x6bb4, (q15_t)0xbad7, (q15_t)0x6bb1, (q15_t)0xbad2, (q15_t)0x6bad, (q15_t)0xbacc,
  (q15_t)0x6baa, (q15_t)0xbac7, (q15_t)0x6ba7, (q15_t)0xbac2, (q15_t)0x6ba3, (q15_t)0xbabc, (q15_t)0x6ba0, (q15_t)0xbab7,
  (q15_t)0x6b9c, (q15_t)0xbab2, (q15_t)0x6b99, (q15_t)0xbaad, (q15_t)0x6b96, (q15_t)0xbaa7, (q15_t)0x6b92, (q15_t)0xbaa2,
  (q15_t)0x6b8f, (q15_t)0xba9d, (q15_t)0x6b8b, (q15_t)0xba97, (q15_t)0x6b88, (q15_t)0xba92, (q15_t)0x6b85, (q15_t)0xba8d,
  (q15_t)0x6b81, (q15_t)0xba88, (q15_t)0x6b7e, (q15_t)0xba82, (q15_t)0x6b7a, (q15_t)0xba7d, (q15_t)0x6b77, (q15_t)0xba78,
  (q15_t)0x6b73, (q15_t)0xba73, (q15_t)0x6b70, (q15_t)0xba6d, (q15_t)0x6b6d, (q15_t)0xba68, (q15_t)0x6b69, (q15_t)0xba63,
  (q15_t)0x6b66, (q15_t)0xba5d, (q15_t)0x6b62, (q15_t)0xba58, (q15_t)0x6b5f, (q15_t)0xba53, (q15_t)0x6b5c, (q15_t)0xba4e,
  (q15_t)0x6b58, (q15_t)0xba48, (q15_t)0x6b55, (q15_t)0xba43, (q15_t)0x6b51, (q15_t)0xba3e, (q15_t)0x6b4e, (q15_t)0xba39,
  (q15_t)0x6b4a, (q15_t)0xba33, (q15_t)0x6b47, (q15_t)0xba2e, (q15_t)0x6b44, (q15_t)0xba29, (q15_t)0x6b40, (q15_t)0xba23,
  (q15_t)0x6b3d, (q15_t)0xba1e, (q15_t)0x6b39, (q15_t)0xba19, (q15_t)0x6b36, (q15_t)0xba14, (q15_t)0x6b32, (q15_t)0xba0e,
  (q15_t)0x6b2f, (q15_t)0xba09, (q15_t)0x6b2c, (q15_t)0xba04, (q15_t)0x6b28, (q15_t)0xb9ff, (q15_t)0x6b25, (q15_t)0xb9f9,
  (q15_t)0x6b21, (q15_t)0xb9f4, (q15_t)0x6b1e, (q15_t)0xb9ef, (q15_t)0x6b1a, (q15_t)0xb9ea, (q15_t)0x6b17, (q15_t)0xb9e4,
  (q15_t)0x6b13, (q15_t)0xb9df, (q15_t)0x6b10, (q15_t)0xb9da, (q15_t)0x6b0d, (q15_t)0xb9d5, (q15_t)0x6b09, (q15_t)0xb9cf,
  (q15_t)0x6b06, (q15_t)0xb9ca, (q15_t)0x6b02, (q15_t)0xb9c5, (q15_t)0x6aff, (q15_t)0xb9c0, (q15_t)0x6afb, (q15_t)0xb9ba,
  (q15_t)0x6af8, (q15_t)0xb9b5, (q15_t)0x6af4, (q15_t)0xb9b0, (q15_t)0x6af1, (q15_t)0xb9ab, (q15_t)0x6aee, (q15_t)0xb9a5,
  (q15_t)0x6aea, (q15_t)0xb9a0, (q15_t)0x6ae7, (q15_t)0xb99b, (q15_t)0x6ae3, (q15_t)0xb996, (q15_t)0x6ae0, (q15_t)0xb990,
  (q15_t)0x6adc, (q15_t)0xb98b, (q15_t)0x6ad9, (q15_t)0xb986, (q15_t)0x6ad5, (q15_t)0xb981, (q15_t)0x6ad2, (q15_t)0xb97b,
  (q15_t)0x6ace, (q15_t)0xb976, (q15_t)0x6acb, (q15_t)0xb971, (q15_t)0x6ac8, (q15_t)0xb96c, (q15_t)0x6ac4, (q15_t)0xb966,
  (q15_t)0x6ac1, (q15_t)0xb961, (q15_t)0x6abd, (q15_t)0xb95c, (q15_t)0x6aba, (q15_t)0xb957, (q15_t)0x6ab6, (q15_t)0xb951,
  (q15_t)0x6ab3, (q15_t)0xb94c, (q15_t)0x6aaf, (q15_t)0xb947, (q15_t)0x6aac, (q15_t)0xb942, (q15_t)0x6aa8, (q15_t)0xb93c,
  (q15_t)0x6aa5, (q15_t)0xb937, (q15_t)0x6aa1, (q15_t)0xb932, (q15_t)0x6a9e, (q15_t)0xb92d, (q15_t)0x6a9a, (q15_t)0xb928,
  (q15_t)0x6a97, (q15_t)0xb922, (q15_t)0x6a93, (q15_t)0xb91d, (q15_t)0x6a90, (q15_t)0xb918, (q15_t)0x6a8c, (q15_t)0xb913,
  (q15_t)0x6a89, (q15_t)0xb90d, (q15_t)0x6a86, (q15_t)0xb908, (q15_t)0x6a82, (q15_t)0xb903, (q15_t)0x6a7f, (q15_t)0xb8fe,
  (q15_t)0x6a7b, (q15_t)0xb8f8, (q15_t)0x6a78, (q15_t)0xb8f3, (q15_t)0x6a74, (q15_t)0xb8ee, (q15_t)0x6a71, (q15_t)0xb8e9,
  (q15_t)0x6a6d, (q15_t)0xb8e4, (q15_t)0x6a6a, (q15_t)0xb8de, (q15_t)0x6a66, (q15_t)0xb8d9, (q15_t)0x6a63, (q15_t)0xb8d4,
  (q15_t)0x6a5f, (q15_t)0xb8cf, (q15_t)0x6a5c, (q15_t)0xb8c9, (q15_t)0x6a58, (q15_t)0xb8c4, (q15_t)0x6a55, (q15_t)0xb8bf,
  (q15_t)0x6a51, (q15_t)0xb8ba, (q15_t)0x6a4e, (q15_t)0xb8b5, (q15_t)0x6a4a, (q15_t)0xb8af, (q15_t)0x6a47, (q15_t)0xb8aa,
  (q15_t)0x6a43, (q15_t)0xb8a5, (q15_t)0x6a40, (q15_t)0xb8a0, (q15_t)0x6a3c, (q15_t)0xb89b, (q15_t)0x6a39, (q15_t)0xb895,
  (q15_t)0x6a35, (q15_t)0xb890, (q15_t)0x6a32, (q15_t)0xb88b, (q15_t)0x6a2e, (q15_t)0xb886, (q15_t)0x6a2b, (q15_t)0xb880,
  (q15_t)0x6a27, (q15_t)0xb87b, (q15_t)0x6a24, (q15_t)0xb876, (q15_t)0x6a20, (q15_t)0xb871, (q15_t)0x6a1d, (q15_t)0xb86c,
  (q15_t)0x6a19, (q15_t)0xb866, (q15_t)0x6a16, (q15_t)0xb861, (q15_t)0x6a12, (q15_t)0xb85c, (q15_t)0x6a0e, (q15_t)0xb857,
  (q15_t)0x6a0b, (q15_t)0xb852, (q15_t)0x6a07, (q15_t)0xb84c, (q15_t)0x6a04, (q15_t)0xb847, (q15_t)0x6a00, (q15_t)0xb842,
  (q15_t)0x69fd, (q15_t)0xb83d, (q15_t)0x69f9, (q15_t)0xb838, (q15_t)0x69f6, (q15_t)0xb832, (q15_t)0x69f2, (q15_t)0xb82d,
  (q15_t)0x69ef, (q15_t)0xb828, (q15_t)0x69eb, (q15_t)0xb823, (q15_t)0x69e8, (q15_t)0xb81e, (q15_t)0x69e4, (q15_t)0xb818,
  (q15_t)0x69e1, (q15_t)0xb813, (q15_t)0x69dd, (q15_t)0xb80e, (q15_t)0x69da, (q15_t)0xb809, (q15_t)0x69d6, (q15_t)0xb804,
  (q15_t)0x69d3, (q15_t)0xb7fe, (q15_t)0x69cf, (q15_t)0xb7f9, (q15_t)0x69cb, (q15_t)0xb7f4, (q15_t)0x69c8, (q15_t)0xb7ef,
  (q15_t)0x69c4, (q15_t)0xb7ea, (q15_t)0x69c1, (q15_t)0xb7e4, (q15_t)0x69bd, (q15_t)0xb7df, (q15_t)0x69ba, (q15_t)0xb7da,
  (q15_t)0x69b6, (q15_t)0xb7d5, (q15_t)0x69b3, (q15_t)0xb7d0, (q15_t)0x69af, (q15_t)0xb7ca, (q15_t)0x69ac, (q15_t)0xb7c5,
  (q15_t)0x69a8, (q15_t)0xb7c0, (q15_t)0x69a5, (q15_t)0xb7bb, (q15_t)0x69a1, (q15_t)0xb7b6, (q15_t)0x699d, (q15_t)0xb7b1,
  (q15_t)0x699a, (q15_t)0xb7ab, (q15_t)0x6996, (q15_t)0xb7a6, (q15_t)0x6993, (q15_t)0xb7a1, (q15_t)0x698f, (q15_t)0xb79c,
  (q15_t)0x698c, (q15_t)0xb797, (q15_t)0x6988, (q15_t)0xb791, (q15_t)0x6985, (q15_t)0xb78c, (q15_t)0x6981, (q15_t)0xb787,
  (q15_t)0x697d, (q15_t)0xb782, (q15_t)0x697a, (q15_t)0xb77d, (q15_t)0x6976, (q15_t)0xb778, (q15_t)0x6973, (q15_t)0xb772,
  (q15_t)0x696f, (q15_t)0xb76d, (q15_t)0x696c, (q15_t)0xb768, (q15_t)0x6968, (q15_t)0xb763, (q15_t)0x6964, (q15_t)0xb75e,
  (q15_t)0x6961, (q15_t)0xb758, (q15_t)0x695d, (q15_t)0xb753, (q15_t)0x695a, (q15_t)0xb74e, (q15_t)0x6956, (q15_t)0xb749,
  (q15_t)0x6953, (q15_t)0xb744, (q15_t)0x694f, (q15_t)0xb73f, (q15_t)0x694b, (q15_t)0xb739, (q15_t)0x6948, (q15_t)0xb734,
  (q15_t)0x6944, (q15_t)0xb72f, (q15_t)0x6941, (q15_t)0xb72a, (q15_t)0x693d, (q15_t)0xb725, (q15_t)0x693a, (q15_t)0xb720,
  (q15_t)0x6936, (q15_t)0xb71a, (q15_t)0x6932, (q15_t)0xb715, (q15_t)0x692f, (q15_t)0xb710, (q15_t)0x692b, (q15_t)0xb70b,
  (q15_t)0x6928, (q15_t)0xb706, (q15_t)0x6924, (q15_t)0xb701, (q15_t)0x6921, (q15_t)0xb6fb, (q15_t)0x691d, (q15_t)0xb6f6,
  (q15_t)0x6919, (q15_t)0xb6f1, (q15_t)0x6916, (q15_t)0xb6ec, (q15_t)0x6912, (q15_t)0xb6e7, (q15_t)0x690f, (q15_t)0xb6e2,
  (q15_t)0x690b, (q15_t)0xb6dd, (q15_t)0x6907, (q15_t)0xb6d7, (q15_t)0x6904, (q15_t)0xb6d2, (q15_t)0x6900, (q15_t)0xb6cd,
  (q15_t)0x68fd, (q15_t)0xb6c8, (q15_t)0x68f9, (q15_t)0xb6c3, (q15_t)0x68f5, (q15_t)0xb6be, (q15_t)0x68f2, (q15_t)0xb6b8,
  (q15_t)0x68ee, (q15_t)0xb6b3, (q15_t)0x68eb, (q15_t)0xb6ae, (q15_t)0x68e7, (q15_t)0xb6a9, (q15_t)0x68e3, (q15_t)0xb6a4,
  (q15_t)0x68e0, (q15_t)0xb69f, (q15_t)0x68dc, (q15_t)0xb69a, (q15_t)0x68d9, (q15_t)0xb694, (q15_t)0x68d5, (q15_t)0xb68f,
  (q15_t)0x68d1, (q15_t)0xb68a, (q15_t)0x68ce, (q15_t)0xb685, (q15_t)0x68ca, (q15_t)0xb680, (q15_t)0x68c7, (q15_t)0xb67b,
  (q15_t)0x68c3, (q15_t)0xb676, (q15_t)0x68bf, (q15_t)0xb670, (q15_t)0x68bc, (q15_t)0xb66b, (q15_t)0x68b8, (q15_t)0xb666,
  (q15_t)0x68b5, (q15_t)0xb661, (q15_t)0x68b1, (q15_t)0xb65c, (q15_t)0x68ad, (q15_t)0xb657, (q15_t)0x68aa, (q15_t)0xb652,
  (q15_t)0x68a6, (q15_t)0xb64c, (q15_t)0x68a3, (q15_t)0xb647, (q15_t)0x689f, (q15_t)0xb642, (q15_t)0x689b, (q15_t)0xb63d,
  (q15_t)0x6898, (q15_t)0xb638, (q15_t)0x6894, (q15_t)0xb633, (q15_t)0x6890, (q15_t)0xb62e, (q15_t)0x688d, (q15_t)0xb628,
  (q15_t)0x6889, (q15_t)0xb623, (q15_t)0x6886, (q15_t)0xb61e, (q15_t)0x6882, (q15_t)0xb619, (q15_t)0x687e, (q15_t)0xb614,
  (q15_t)0x687b, (q15_t)0xb60f, (q15_t)0x6877, (q15_t)0xb60a, (q15_t)0x6873, (q15_t)0xb605, (q15_t)0x6870, (q15_t)0xb5ff,
  (q15_t)0x686c, (q15_t)0xb5fa, (q15_t)0x6868, (q15_t)0xb5f5, (q15_t)0x6865, (q15_t)0xb5f0, (q15_t)0x6861, (q15_t)0xb5eb,
  (q15_t)0x685e, (q15_t)0xb5e6, (q15_t)0x685a, (q15_t)0xb5e1, (q15_t)0x6856, (q15_t)0xb5dc, (q15_t)0x6853, (q15_t)0xb5d6,
  (q15_t)0x684f, (q15_t)0xb5d1, (q15_t)0x684b, (q15_t)0xb5cc, (q15_t)0x6848, (q15_t)0xb5c7, (q15_t)0x6844, (q15_t)0xb5c2,
  (q15_t)0x6840, (q15_t)0xb5bd, (q15_t)0x683d, (q15_t)0xb5b8, (q15_t)0x6839, (q15_t)0xb5b3, (q15_t)0x6835, (q15_t)0xb5ae,
  (q15_t)0x6832, (q15_t)0xb5a8, (q15_t)0x682e, (q15_t)0xb5a3, (q15_t)0x682b, (q15_t)0xb59e, (q15_t)0x6827, (q15_t)0xb599,
  (q15_t)0x6823, (q15_t)0xb594, (q15_t)0x6820, (q15_t)0xb58f, (q15_t)0x681c, (q15_t)0xb58a, (q15_t)0x6818, (q15_t)0xb585,
  (q15_t)0x6815, (q15_t)0xb57f, (q15_t)0x6811, (q15_t)0xb57a, (q15_t)0x680d, (q15_t)0xb575, (q15_t)0x680a, (q15_t)0xb570,
  (q15_t)0x6806, (q15_t)0xb56b, (q15_t)0x6802, (q15_t)0xb566, (q15_t)0x67ff, (q15_t)0xb561, (q15_t)0x67fb, (q15_t)0xb55c,
  (q15_t)0x67f7, (q15_t)0xb557, (q15_t)0x67f4, (q15_t)0xb552, (q15_t)0x67f0, (q15_t)0xb54c, (q15_t)0x67ec, (q15_t)0xb547,
  (q15_t)0x67e9, (q15_t)0xb542, (q15_t)0x67e5, (q15_t)0xb53d, (q15_t)0x67e1, (q15_t)0xb538, (q15_t)0x67de, (q15_t)0xb533,
  (q15_t)0x67da, (q15_t)0xb52e, (q15_t)0x67d6, (q15_t)0xb529, (q15_t)0x67d3, (q15_t)0xb524, (q15_t)0x67cf, (q15_t)0xb51f,
  (q15_t)0x67cb, (q15_t)0xb519, (q15_t)0x67c8, (q15_t)0xb514, (q15_t)0x67c4, (q15_t)0xb50f, (q15_t)0x67c0, (q15_t)0xb50a,
  (q15_t)0x67bd, (q15_t)0xb505, (q15_t)0x67b9, (q15_t)0xb500, (q15_t)0x67b5, (q15_t)0xb4fb, (q15_t)0x67b2, (q15_t)0xb4f6,
  (q15_t)0x67ae, (q15_t)0xb4f1, (q15_t)0x67aa, (q15_t)0xb4ec, (q15_t)0x67a6, (q15_t)0xb4e7, (q15_t)0x67a3, (q15_t)0xb4e1,
  (q15_t)0x679f, (q15_t)0xb4dc, (q15_t)0x679b, (q15_t)0xb4d7, (q15_t)0x6798, (q15_t)0xb4d2, (q15_t)0x6794, (q15_t)0xb4cd,
  (q15_t)0x6790, (q15_t)0xb4c8, (q15_t)0x678d, (q15_t)0xb4c3, (q15_t)0x6789, (q15_t)0xb4be, (q15_t)0x6785, (q15_t)0xb4b9,
  (q15_t)0x6782, (q15_t)0xb4b4, (q15_t)0x677e, (q15_t)0xb4af, (q15_t)0x677a, (q15_t)0xb4aa, (q15_t)0x6776, (q15_t)0xb4a4,
  (q15_t)0x6773, (q15_t)0xb49f, (q15_t)0x676f, (q15_t)0xb49a, (q15_t)0x676b, (q15_t)0xb495, (q15_t)0x6768, (q15_t)0xb490,
  (q15_t)0x6764, (q15_t)0xb48b, (q15_t)0x6760, (q15_t)0xb486, (q15_t)0x675d, (q15_t)0xb481, (q15_t)0x6759, (q15_t)0xb47c,
  (q15_t)0x6755, (q15_t)0xb477, (q15_t)0x6751, (q15_t)0xb472, (q15_t)0x674e, (q15_t)0xb46d, (q15_t)0x674a, (q15_t)0xb468,
  (q15_t)0x6746, (q15_t)0xb462, (q15_t)0x6743, (q15_t)0xb45d, (q15_t)0x673f, (q15_t)0xb458, (q15_t)0x673b, (q15_t)0xb453,
  (q15_t)0x6737, (q15_t)0xb44e, (q15_t)0x6734, (q15_t)0xb449, (q15_t)0x6730, (q15_t)0xb444, (q15_t)0x672c, (q15_t)0xb43f,
  (q15_t)0x6729, (q15_t)0xb43a, (q15_t)0x6725, (q15_t)0xb435, (q15_t)0x6721, (q15_t)0xb430, (q15_t)0x671d, (q15_t)0xb42b,
  (q15_t)0x671a, (q15_t)0xb426, (q15_t)0x6716, (q15_t)0xb421, (q15_t)0x6712, (q15_t)0xb41c, (q15_t)0x670e, (q15_t)0xb417,
  (q15_t)0x670b, (q15_t)0xb411, (q15_t)0x6707, (q15_t)0xb40c, (q15_t)0x6703, (q15_t)0xb407, (q15_t)0x6700, (q15_t)0xb402,
  (q15_t)0x66fc, (q15_t)0xb3fd, (q15_t)0x66f8, (q15_t)0xb3f8, (q15_t)0x66f4, (q15_t)0xb3f3, (q15_t)0x66f1, (q15_t)0xb3ee,
  (q15_t)0x66ed, (q15_t)0xb3e9, (q15_t)0x66e9, (q15_t)0xb3e4, (q15_t)0x66e5, (q15_t)0xb3df, (q15_t)0x66e2, (q15_t)0xb3da,
  (q15_t)0x66de, (q15_t)0xb3d5, (q15_t)0x66da, (q15_t)0xb3d0, (q15_t)0x66d6, (q15_t)0xb3cb, (q15_t)0x66d3, (q15_t)0xb3c6,
  (q15_t)0x66cf, (q15_t)0xb3c1, (q15_t)0x66cb, (q15_t)0xb3bc, (q15_t)0x66c8, (q15_t)0xb3b7, (q15_t)0x66c4, (q15_t)0xb3b1,
  (q15_t)0x66c0, (q15_t)0xb3ac, (q15_t)0x66bc, (q15_t)0xb3a7, (q15_t)0x66b9, (q15_t)0xb3a2, (q15_t)0x66b5, (q15_t)0xb39d,
  (q15_t)0x66b1, (q15_t)0xb398, (q15_t)0x66ad, (q15_t)0xb393, (q15_t)0x66aa, (q15_t)0xb38e, (q15_t)0x66a6, (q15_t)0xb389,
  (q15_t)0x66a2, (q15_t)0xb384, (q15_t)0x669e, (q15_t)0xb37f, (q15_t)0x669b, (q15_t)0xb37a, (q15_t)0x6697, (q15_t)0xb375,
  (q15_t)0x6693, (q15_t)0xb370, (q15_t)0x668f, (q15_t)0xb36b, (q15_t)0x668b, (q15_t)0xb366, (q15_t)0x6688, (q15_t)0xb361,
  (q15_t)0x6684, (q15_t)0xb35c, (q15_t)0x6680, (q15_t)0xb357, (q15_t)0x667c, (q15_t)0xb352, (q15_t)0x6679, (q15_t)0xb34d,
  (q15_t)0x6675, (q15_t)0xb348, (q15_t)0x6671, (q15_t)0xb343, (q15_t)0x666d, (q15_t)0xb33e, (q15_t)0x666a, (q15_t)0xb339,
  (q15_t)0x6666, (q15_t)0xb334, (q15_t)0x6662, (q15_t)0xb32f, (q15_t)0x665e, (q15_t)0xb32a, (q15_t)0x665b, (q15_t)0xb325,
  (q15_t)0x6657, (q15_t)0xb31f, (q15_t)0x6653, (q15_t)0xb31a, (q15_t)0x664f, (q15_t)0xb315, (q15_t)0x664b, (q15_t)0xb310,
  (q15_t)0x6648, (q15_t)0xb30b, (q15_t)0x6644, (q15_t)0xb306, (q15_t)0x6640, (q15_t)0xb301, (q15_t)0x663c, (q15_t)0xb2fc,
  (q15_t)0x6639, (q15_t)0xb2f7, (q15_t)0x6635, (q15_t)0xb2f2, (q15_t)0x6631, (q15_t)0xb2ed, (q15_t)0x662d, (q15_t)0xb2e8,
  (q15_t)0x6629, (q15_t)0xb2e3, (q15_t)0x6626, (q15_t)0xb2de, (q15_t)0x6622, (q15_t)0xb2d9, (q15_t)0x661e, (q15_t)0xb2d4,
  (q15_t)0x661a, (q15_t)0xb2cf, (q15_t)0x6616, (q15_t)0xb2ca, (q15_t)0x6613, (q15_t)0xb2c5, (q15_t)0x660f, (q15_t)0xb2c0,
  (q15_t)0x660b, (q15_t)0xb2bb, (q15_t)0x6607, (q15_t)0xb2b6, (q15_t)0x6603, (q15_t)0xb2b1, (q15_t)0x6600, (q15_t)0xb2ac,
  (q15_t)0x65fc, (q15_t)0xb2a7, (q15_t)0x65f8, (q15_t)0xb2a2, (q15_t)0x65f4, (q15_t)0xb29d, (q15_t)0x65f0, (q15_t)0xb298,
  (q15_t)0x65ed, (q15_t)0xb293, (q15_t)0x65e9, (q15_t)0xb28e, (q15_t)0x65e5, (q15_t)0xb289, (q15_t)0x65e1, (q15_t)0xb284,
  (q15_t)0x65dd, (q15_t)0xb27f, (q15_t)0x65da, (q15_t)0xb27a, (q15_t)0x65d6, (q15_t)0xb275, (q15_t)0x65d2, (q15_t)0xb270,
  (q15_t)0x65ce, (q15_t)0xb26b, (q15_t)0x65ca, (q15_t)0xb266, (q15_t)0x65c7, (q15_t)0xb261, (q15_t)0x65c3, (q15_t)0xb25c,
  (q15_t)0x65bf, (q15_t)0xb257, (q15_t)0x65bb, (q15_t)0xb252, (q15_t)0x65b7, (q15_t)0xb24d, (q15_t)0x65b4, (q15_t)0xb248,
  (q15_t)0x65b0, (q15_t)0xb243, (q15_t)0x65ac, (q15_t)0xb23e, (q15_t)0x65a8, (q15_t)0xb239, (q15_t)0x65a4, (q15_t)0xb234,
  (q15_t)0x65a0, (q15_t)0xb22f, (q15_t)0x659d, (q15_t)0xb22a, (q15_t)0x6599, (q15_t)0xb225, (q15_t)0x6595, (q15_t)0xb220,
  (q15_t)0x6591, (q15_t)0xb21b, (q15_t)0x658d, (q15_t)0xb216, (q15_t)0x658a, (q15_t)0xb211, (q15_t)0x6586, (q15_t)0xb20c,
  (q15_t)0x6582, (q15_t)0xb207, (q15_t)0x657e, (q15_t)0xb202, (q15_t)0x657a, (q15_t)0xb1fd, (q15_t)0x6576, (q15_t)0xb1f8,
  (q15_t)0x6573, (q15_t)0xb1f3, (q15_t)0x656f, (q15_t)0xb1ee, (q15_t)0x656b, (q15_t)0xb1e9, (q15_t)0x6567, (q15_t)0xb1e4,
  (q15_t)0x6563, (q15_t)0xb1df, (q15_t)0x655f, (q15_t)0xb1da, (q15_t)0x655c, (q15_t)0xb1d6, (q15_t)0x6558, (q15_t)0xb1d1,
  (q15_t)0x6554, (q15_t)0xb1cc, (q15_t)0x6550, (q15_t)0xb1c7, (q15_t)0x654c, (q15_t)0xb1c2, (q15_t)0x6548, (q15_t)0xb1bd,
  (q15_t)0x6545, (q15_t)0xb1b8, (q15_t)0x6541, (q15_t)0xb1b3, (q15_t)0x653d, (q15_t)0xb1ae, (q15_t)0x6539, (q15_t)0xb1a9,
  (q15_t)0x6535, (q15_t)0xb1a4, (q15_t)0x6531, (q15_t)0xb19f, (q15_t)0x652d, (q15_t)0xb19a, (q15_t)0x652a, (q15_t)0xb195,
  (q15_t)0x6526, (q15_t)0xb190, (q15_t)0x6522, (q15_t)0xb18b, (q15_t)0x651e, (q15_t)0xb186, (q15_t)0x651a, (q15_t)0xb181,
  (q15_t)0x6516, (q15_t)0xb17c, (q15_t)0x6513, (q15_t)0xb177, (q15_t)0x650f, (q15_t)0xb172, (q15_t)0x650b, (q15_t)0xb16d,
  (q15_t)0x6507, (q15_t)0xb168, (q15_t)0x6503, (q15_t)0xb163, (q15_t)0x64ff, (q15_t)0xb15e, (q15_t)0x64fb, (q15_t)0xb159,
  (q15_t)0x64f7, (q15_t)0xb154, (q15_t)0x64f4, (q15_t)0xb14f, (q15_t)0x64f0, (q15_t)0xb14a, (q15_t)0x64ec, (q15_t)0xb146,
  (q15_t)0x64e8, (q15_t)0xb141, (q15_t)0x64e4, (q15_t)0xb13c, (q15_t)0x64e0, (q15_t)0xb137, (q15_t)0x64dc, (q15_t)0xb132,
  (q15_t)0x64d9, (q15_t)0xb12d, (q15_t)0x64d5, (q15_t)0xb128, (q15_t)0x64d1, (q15_t)0xb123, (q15_t)0x64cd, (q15_t)0xb11e,
  (q15_t)0x64c9, (q15_t)0xb119, (q15_t)0x64c5, (q15_t)0xb114, (q15_t)0x64c1, (q15_t)0xb10f, (q15_t)0x64bd, (q15_t)0xb10a,
  (q15_t)0x64ba, (q15_t)0xb105, (q15_t)0x64b6, (q15_t)0xb100, (q15_t)0x64b2, (q15_t)0xb0fb, (q15_t)0x64ae, (q15_t)0xb0f6,
  (q15_t)0x64aa, (q15_t)0xb0f1, (q15_t)0x64a6, (q15_t)0xb0ec, (q15_t)0x64a2, (q15_t)0xb0e8, (q15_t)0x649e, (q15_t)0xb0e3,
  (q15_t)0x649b, (q15_t)0xb0de, (q15_t)0x6497, (q15_t)0xb0d9, (q15_t)0x6493, (q15_t)0xb0d4, (q15_t)0x648f, (q15_t)0xb0cf,
  (q15_t)0x648b, (q15_t)0xb0ca, (q15_t)0x6487, (q15_t)0xb0c5, (q15_t)0x6483, (q15_t)0xb0c0, (q15_t)0x647f, (q15_t)0xb0bb,
  (q15_t)0x647b, (q15_t)0xb0b6, (q15_t)0x6478, (q15_t)0xb0b1, (q15_t)0x6474, (q15_t)0xb0ac, (q15_t)0x6470, (q15_t)0xb0a7,
  (q15_t)0x646c, (q15_t)0xb0a2, (q15_t)0x6468, (q15_t)0xb09e, (q15_t)0x6464, (q15_t)0xb099, (q15_t)0x6460, (q15_t)0xb094,
  (q15_t)0x645c, (q15_t)0xb08f, (q15_t)0x6458, (q15_t)0xb08a, (q15_t)0x6454, (q15_t)0xb085, (q15_t)0x6451, (q15_t)0xb080,
  (q15_t)0x644d, (q15_t)0xb07b, (q15_t)0x6449, (q15_t)0xb076, (q15_t)0x6445, (q15_t)0xb071, (q15_t)0x6441, (q15_t)0xb06c,
  (q15_t)0x643d, (q15_t)0xb067, (q15_t)0x6439, (q15_t)0xb062, (q15_t)0x6435, (q15_t)0xb05e, (q15_t)0x6431, (q15_t)0xb059,
  (q15_t)0x642d, (q15_t)0xb054, (q15_t)0x6429, (q15_t)0xb04f, (q15_t)0x6426, (q15_t)0xb04a, (q15_t)0x6422, (q15_t)0xb045,
  (q15_t)0x641e, (q15_t)0xb040, (q15_t)0x641a, (q15_t)0xb03b, (q15_t)0x6416, (q15_t)0xb036, (q15_t)0x6412, (q15_t)0xb031,
  (q15_t)0x640e, (q15_t)0xb02c, (q15_t)0x640a, (q15_t)0xb027, (q15_t)0x6406, (q15_t)0xb023, (q15_t)0x6402, (q15_t)0xb01e,
  (q15_t)0x63fe, (q15_t)0xb019, (q15_t)0x63fa, (q15_t)0xb014, (q15_t)0x63f7, (q15_t)0xb00f, (q15_t)0x63f3, (q15_t)0xb00a,
  (q15_t)0x63ef, (q15_t)0xb005, (q15_t)0x63eb, (q15_t)0xb000, (q15_t)0x63e7, (q15_t)0xaffb, (q15_t)0x63e3, (q15_t)0xaff6,
  (q15_t)0x63df, (q15_t)0xaff1, (q15_t)0x63db, (q15_t)0xafed, (q15_t)0x63d7, (q15_t)0xafe8, (q15_t)0x63d3, (q15_t)0xafe3,
  (q15_t)0x63cf, (q15_t)0xafde, (q15_t)0x63cb, (q15_t)0xafd9, (q15_t)0x63c7, (q15_t)0xafd4, (q15_t)0x63c3, (q15_t)0xafcf,
  (q15_t)0x63c0, (q15_t)0xafca, (q15_t)0x63bc, (q15_t)0xafc5, (q15_t)0x63b8, (q15_t)0xafc1, (q15_t)0x63b4, (q15_t)0xafbc,
  (q15_t)0x63b0, (q15_t)0xafb7, (q15_t)0x63ac, (q15_t)0xafb2, (q15_t)0x63a8, (q15_t)0xafad, (q15_t)0x63a4, (q15_t)0xafa8,
  (q15_t)0x63a0, (q15_t)0xafa3, (q15_t)0x639c, (q15_t)0xaf9e, (q15_t)0x6398, (q15_t)0xaf99, (q15_t)0x6394, (q15_t)0xaf94,
  (q15_t)0x6390, (q15_t)0xaf90, (q15_t)0x638c, (q15_t)0xaf8b, (q15_t)0x6388, (q15_t)0xaf86, (q15_t)0x6384, (q15_t)0xaf81,
  (q15_t)0x6380, (q15_t)0xaf7c, (q15_t)0x637c, (q15_t)0xaf77, (q15_t)0x6378, (q15_t)0xaf72, (q15_t)0x6375, (q15_t)0xaf6d,
  (q15_t)0x6371, (q15_t)0xaf69, (q15_t)0x636d, (q15_t)0xaf64, (q15_t)0x6369, (q15_t)0xaf5f, (q15_t)0x6365, (q15_t)0xaf5a,
  (q15_t)0x6361, (q15_t)0xaf55, (q15_t)0x635d, (q15_t)0xaf50, (q15_t)0x6359, (q15_t)0xaf4b, (q15_t)0x6355, (q15_t)0xaf46,
  (q15_t)0x6351, (q15_t)0xaf41, (q15_t)0x634d, (q15_t)0xaf3d, (q15_t)0x6349, (q15_t)0xaf38, (q15_t)0x6345, (q15_t)0xaf33,
  (q15_t)0x6341, (q15_t)0xaf2e, (q15_t)0x633d, (q15_t)0xaf29, (q15_t)0x6339, (q15_t)0xaf24, (q15_t)0x6335, (q15_t)0xaf1f,
  (q15_t)0x6331, (q15_t)0xaf1b, (q15_t)0x632d, (q15_t)0xaf16, (q15_t)0x6329, (q15_t)0xaf11, (q15_t)0x6325, (q15_t)0xaf0c,
  (q15_t)0x6321, (q15_t)0xaf07, (q15_t)0x631d, (q15_t)0xaf02, (q15_t)0x6319, (q15_t)0xaefd, (q15_t)0x6315, (q15_t)0xaef8,
  (q15_t)0x6311, (q15_t)0xaef4, (q15_t)0x630d, (q15_t)0xaeef, (q15_t)0x6309, (q15_t)0xaeea, (q15_t)0x6305, (q15_t)0xaee5,
  (q15_t)0x6301, (q15_t)0xaee0, (q15_t)0x62fd, (q15_t)0xaedb, (q15_t)0x62f9, (q15_t)0xaed6, (q15_t)0x62f5, (q15_t)0xaed2,
  (q15_t)0x62f2, (q15_t)0xaecd, (q15_t)0x62ee, (q15_t)0xaec8, (q15_t)0x62ea, (q15_t)0xaec3, (q15_t)0x62e6, (q15_t)0xaebe,
  (q15_t)0x62e2, (q15_t)0xaeb9, (q15_t)0x62de, (q15_t)0xaeb4, (q15_t)0x62da, (q15_t)0xaeb0, (q15_t)0x62d6, (q15_t)0xaeab,
  (q15_t)0x62d2, (q15_t)0xaea6, (q15_t)0x62ce, (q15_t)0xaea1, (q15_t)0x62ca, (q15_t)0xae9c, (q15_t)0x62c6, (q15_t)0xae97,
  (q15_t)0x62c2, (q15_t)0xae92, (q15_t)0x62be, (q15_t)0xae8e, (q15_t)0x62ba, (q15_t)0xae89, (q15_t)0x62b6, (q15_t)0xae84,
  (q15_t)0x62b2, (q15_t)0xae7f, (q15_t)0x62ae, (q15_t)0xae7a, (q15_t)0x62aa, (q15_t)0xae75, (q15_t)0x62a6, (q15_t)0xae71,
  (q15_t)0x62a2, (q15_t)0xae6c, (q15_t)0x629e, (q15_t)0xae67, (q15_t)0x629a, (q15_t)0xae62, (q15_t)0x6296, (q15_t)0xae5d,
  (q15_t)0x6292, (q15_t)0xae58, (q15_t)0x628e, (q15_t)0xae54, (q15_t)0x628a, (q15_t)0xae4f, (q15_t)0x6286, (q15_t)0xae4a,
  (q15_t)0x6282, (q15_t)0xae45, (q15_t)0x627e, (q15_t)0xae40, (q15_t)0x627a, (q15_t)0xae3b, (q15_t)0x6275, (q15_t)0xae37,
  (q15_t)0x6271, (q15_t)0xae32, (q15_t)0x626d, (q15_t)0xae2d, (q15_t)0x6269, (q15_t)0xae28, (q15_t)0x6265, (q15_t)0xae23,
  (q15_t)0x6261, (q15_t)0xae1e, (q15_t)0x625d, (q15_t)0xae1a, (q15_t)0x6259, (q15_t)0xae15, (q15_t)0x6255, (q15_t)0xae10,
  (q15_t)0x6251, (q15_t)0xae0b, (q15_t)0x624d, (q15_t)0xae06, (q15_t)0x6249, (q15_t)0xae01, (q15_t)0x6245, (q15_t)0xadfd,
  (q15_t)0x6241, (q15_t)0xadf8, (q15_t)0x623d, (q15_t)0xadf3, (q15_t)0x6239, (q15_t)0xadee, (q15_t)0x6235, (q15_t)0xade9,
  (q15_t)0x6231, (q15_t)0xade4, (q15_t)0x622d, (q15_t)0xade0, (q15_t)0x6229, (q15_t)0xaddb, (q15_t)0x6225, (q15_t)0xadd6,
  (q15_t)0x6221, (q15_t)0xadd1, (q15_t)0x621d, (q15_t)0xadcc, (q15_t)0x6219, (q15_t)0xadc8, (q15_t)0x6215, (q15_t)0xadc3,
  (q15_t)0x6211, (q15_t)0xadbe, (q15_t)0x620d, (q15_t)0xadb9, (q15_t)0x6209, (q15_t)0xadb4, (q15_t)0x6205, (q15_t)0xadaf,
  (q15_t)0x6201, (q15_t)0xadab, (q15_t)0x61fd, (q15_t)0xada6, (q15_t)0x61f9, (q15_t)0xada1, (q15_t)0x61f5, (q15_t)0xad9c,
  (q15_t)0x61f1, (q15_t)0xad97, (q15_t)0x61ec, (q15_t)0xad93, (q15_t)0x61e8, (q15_t)0xad8e, (q15_t)0x61e4, (q15_t)0xad89,
  (q15_t)0x61e0, (q15_t)0xad84, (q15_t)0x61dc, (q15_t)0xad7f, (q15_t)0x61d8, (q15_t)0xad7b, (q15_t)0x61d4, (q15_t)0xad76,
  (q15_t)0x61d0, (q15_t)0xad71, (q15_t)0x61cc, (q15_t)0xad6c, (q15_t)0x61c8, (q15_t)0xad67, (q15_t)0x61c4, (q15_t)0xad63,
  (q15_t)0x61c0, (q15_t)0xad5e, (q15_t)0x61bc, (q15_t)0xad59, (q15_t)0x61b8, (q15_t)0xad54, (q15_t)0x61b4, (q15_t)0xad4f,
  (q15_t)0x61b0, (q15_t)0xad4b, (q15_t)0x61ac, (q15_t)0xad46, (q15_t)0x61a8, (q15_t)0xad41, (q15_t)0x61a3, (q15_t)0xad3c,
  (q15_t)0x619f, (q15_t)0xad37, (q15_t)0x619b, (q15_t)0xad33, (q15_t)0x6197, (q15_t)0xad2e, (q15_t)0x6193, (q15_t)0xad29,
  (q15_t)0x618f, (q15_t)0xad24, (q15_t)0x618b, (q15_t)0xad1f, (q15_t)0x6187, (q15_t)0xad1b, (q15_t)0x6183, (q15_t)0xad16,
  (q15_t)0x617f, (q15_t)0xad11, (q15_t)0x617b, (q15_t)0xad0c, (q15_t)0x6177, (q15_t)0xad08, (q15_t)0x6173, (q15_t)0xad03,
  (q15_t)0x616f, (q15_t)0xacfe, (q15_t)0x616b, (q15_t)0xacf9, (q15_t)0x6166, (q15_t)0xacf4, (q15_t)0x6162, (q15_t)0xacf0,
  (q15_t)0x615e, (q15_t)0xaceb, (q15_t)0x615a, (q15_t)0xace6, (q15_t)0x6156, (q15_t)0xace1, (q15_t)0x6152, (q15_t)0xacdd,
  (q15_t)0x614e, (q15_t)0xacd8, (q15_t)0x614a, (q15_t)0xacd3, (q15_t)0x6146, (q15_t)0xacce, (q15_t)0x6142, (q15_t)0xacc9,
  (q15_t)0x613e, (q15_t)0xacc5, (q15_t)0x613a, (q15_t)0xacc0, (q15_t)0x6135, (q15_t)0xacbb, (q15_t)0x6131, (q15_t)0xacb6,
  (q15_t)0x612d, (q15_t)0xacb2, (q15_t)0x6129, (q15_t)0xacad, (q15_t)0x6125, (q15_t)0xaca8, (q15_t)0x6121, (q15_t)0xaca3,
  (q15_t)0x611d, (q15_t)0xac9e, (q15_t)0x6119, (q15_t)0xac9a, (q15_t)0x6115, (q15_t)0xac95, (q15_t)0x6111, (q15_t)0xac90,
  (q15_t)0x610d, (q15_t)0xac8b, (q15_t)0x6108, (q15_t)0xac87, (q15_t)0x6104, (q15_t)0xac82, (q15_t)0x6100, (q15_t)0xac7d,
  (q15_t)0x60fc, (q15_t)0xac78, (q15_t)0x60f8, (q15_t)0xac74, (q15_t)0x60f4, (q15_t)0xac6f, (q15_t)0x60f0, (q15_t)0xac6a,
  (q15_t)0x60ec, (q15_t)0xac65, (q15_t)0x60e8, (q15_t)0xac61, (q15_t)0x60e4, (q15_t)0xac5c, (q15_t)0x60df, (q15_t)0xac57,
  (q15_t)0x60db, (q15_t)0xac52, (q15_t)0x60d7, (q15_t)0xac4e, (q15_t)0x60d3, (q15_t)0xac49, (q15_t)0x60cf, (q15_t)0xac44,
  (q15_t)0x60cb, (q15_t)0xac3f, (q15_t)0x60c7, (q15_t)0xac3b, (q15_t)0x60c3, (q15_t)0xac36, (q15_t)0x60bf, (q15_t)0xac31,
  (q15_t)0x60ba, (q15_t)0xac2c, (q15_t)0x60b6, (q15_t)0xac28, (q15_t)0x60b2, (q15_t)0xac23, (q15_t)0x60ae, (q15_t)0xac1e,
  (q15_t)0x60aa, (q15_t)0xac19, (q15_t)0x60a6, (q15_t)0xac15, (q15_t)0x60a2, (q15_t)0xac10, (q15_t)0x609e, (q15_t)0xac0b,
  (q15_t)0x6099, (q15_t)0xac06, (q15_t)0x6095, (q15_t)0xac02, (q15_t)0x6091, (q15_t)0xabfd, (q15_t)0x608d, (q15_t)0xabf8,
  (q15_t)0x6089, (q15_t)0xabf3, (q15_t)0x6085, (q15_t)0xabef, (q15_t)0x6081, (q15_t)0xabea, (q15_t)0x607d, (q15_t)0xabe5,
  (q15_t)0x6078, (q15_t)0xabe0, (q15_t)0x6074, (q15_t)0xabdc, (q15_t)0x6070, (q15_t)0xabd7, (q15_t)0x606c, (q15_t)0xabd2,
  (q15_t)0x6068, (q15_t)0xabcd, (q15_t)0x6064, (q15_t)0xabc9, (q15_t)0x6060, (q15_t)0xabc4, (q15_t)0x605c, (q15_t)0xabbf,
  (q15_t)0x6057, (q15_t)0xabbb, (q15_t)0x6053, (q15_t)0xabb6, (q15_t)0x604f, (q15_t)0xabb1, (q15_t)0x604b, (q15_t)0xabac,
  (q15_t)0x6047, (q15_t)0xaba8, (q15_t)0x6043, (q15_t)0xaba3, (q15_t)0x603f, (q15_t)0xab9e, (q15_t)0x603a, (q15_t)0xab99,
  (q15_t)0x6036, (q15_t)0xab95, (q15_t)0x6032, (q15_t)0xab90, (q15_t)0x602e, (q15_t)0xab8b, (q15_t)0x602a, (q15_t)0xab87,
  (q15_t)0x6026, (q15_t)0xab82, (q15_t)0x6022, (q15_t)0xab7d, (q15_t)0x601d, (q15_t)0xab78, (q15_t)0x6019, (q15_t)0xab74,
  (q15_t)0x6015, (q15_t)0xab6f, (q15_t)0x6011, (q15_t)0xab6a, (q15_t)0x600d, (q15_t)0xab66, (q15_t)0x6009, (q15_t)0xab61,
  (q15_t)0x6004, (q15_t)0xab5c, (q15_t)0x6000, (q15_t)0xab57, (q15_t)0x5ffc, (q15_t)0xab53, (q15_t)0x5ff8, (q15_t)0xab4e,
  (q15_t)0x5ff4, (q15_t)0xab49, (q15_t)0x5ff0, (q15_t)0xab45, (q15_t)0x5fec, (q15_t)0xab40, (q15_t)0x5fe7, (q15_t)0xab3b,
  (q15_t)0x5fe3, (q15_t)0xab36, (q15_t)0x5fdf, (q15_t)0xab32, (q15_t)0x5fdb, (q15_t)0xab2d, (q15_t)0x5fd7, (q15_t)0xab28,
  (q15_t)0x5fd3, (q15_t)0xab24, (q15_t)0x5fce, (q15_t)0xab1f, (q15_t)0x5fca, (q15_t)0xab1a, (q15_t)0x5fc6, (q15_t)0xab16,
  (q15_t)0x5fc2, (q15_t)0xab11, (q15_t)0x5fbe, (q15_t)0xab0c, (q15_t)0x5fba, (q15_t)0xab07, (q15_t)0x5fb5, (q15_t)0xab03,
  (q15_t)0x5fb1, (q15_t)0xaafe, (q15_t)0x5fad, (q15_t)0xaaf9, (q15_t)0x5fa9, (q15_t)0xaaf5, (q15_t)0x5fa5, (q15_t)0xaaf0,
  (q15_t)0x5fa0, (q15_t)0xaaeb, (q15_t)0x5f9c, (q15_t)0xaae7, (q15_t)0x5f98, (q15_t)0xaae2, (q15_t)0x5f94, (q15_t)0xaadd,
  (q15_t)0x5f90, (q15_t)0xaad8, (q15_t)0x5f8c, (q15_t)0xaad4, (q15_t)0x5f87, (q15_t)0xaacf, (q15_t)0x5f83, (q15_t)0xaaca,
  (q15_t)0x5f7f, (q15_t)0xaac6, (q15_t)0x5f7b, (q15_t)0xaac1, (q15_t)0x5f77, (q15_t)0xaabc, (q15_t)0x5f72, (q15_t)0xaab8,
  (q15_t)0x5f6e, (q15_t)0xaab3, (q15_t)0x5f6a, (q15_t)0xaaae, (q15_t)0x5f66, (q15_t)0xaaaa, (q15_t)0x5f62, (q15_t)0xaaa5,
  (q15_t)0x5f5e, (q15_t)0xaaa0, (q15_t)0x5f59, (q15_t)0xaa9c, (q15_t)0x5f55, (q15_t)0xaa97, (q15_t)0x5f51, (q15_t)0xaa92,
  (q15_t)0x5f4d, (q15_t)0xaa8e, (q15_t)0x5f49, (q15_t)0xaa89, (q15_t)0x5f44, (q15_t)0xaa84, (q15_t)0x5f40, (q15_t)0xaa7f,
  (q15_t)0x5f3c, (q15_t)0xaa7b, (q15_t)0x5f38, (q15_t)0xaa76, (q15_t)0x5f34, (q15_t)0xaa71, (q15_t)0x5f2f, (q15_t)0xaa6d,
  (q15_t)0x5f2b, (q15_t)0xaa68, (q15_t)0x5f27, (q15_t)0xaa63, (q15_t)0x5f23, (q15_t)0xaa5f, (q15_t)0x5f1f, (q15_t)0xaa5a,
  (q15_t)0x5f1a, (q15_t)0xaa55, (q15_t)0x5f16, (q15_t)0xaa51, (q15_t)0x5f12, (q15_t)0xaa4c, (q15_t)0x5f0e, (q15_t)0xaa47,
  (q15_t)0x5f0a, (q15_t)0xaa43, (q15_t)0x5f05, (q15_t)0xaa3e, (q15_t)0x5f01, (q15_t)0xaa39, (q15_t)0x5efd, (q15_t)0xaa35,
  (q15_t)0x5ef9, (q15_t)0xaa30, (q15_t)0x5ef5, (q15_t)0xaa2b, (q15_t)0x5ef0, (q15_t)0xaa27, (q15_t)0x5eec, (q15_t)0xaa22,
  (q15_t)0x5ee8, (q15_t)0xaa1d, (q15_t)0x5ee4, (q15_t)0xaa19, (q15_t)0x5edf, (q15_t)0xaa14, (q15_t)0x5edb, (q15_t)0xaa10,
  (q15_t)0x5ed7, (q15_t)0xaa0b, (q15_t)0x5ed3, (q15_t)0xaa06, (q15_t)0x5ecf, (q15_t)0xaa02, (q15_t)0x5eca, (q15_t)0xa9fd,
  (q15_t)0x5ec6, (q15_t)0xa9f8, (q15_t)0x5ec2, (q15_t)0xa9f4, (q15_t)0x5ebe, (q15_t)0xa9ef, (q15_t)0x5eb9, (q15_t)0xa9ea,
  (q15_t)0x5eb5, (q15_t)0xa9e6, (q15_t)0x5eb1, (q15_t)0xa9e1, (q15_t)0x5ead, (q15_t)0xa9dc, (q15_t)0x5ea9, (q15_t)0xa9d8,
  (q15_t)0x5ea4, (q15_t)0xa9d3, (q15_t)0x5ea0, (q15_t)0xa9ce, (q15_t)0x5e9c, (q15_t)0xa9ca, (q15_t)0x5e98, (q15_t)0xa9c5,
  (q15_t)0x5e93, (q15_t)0xa9c0, (q15_t)0x5e8f, (q15_t)0xa9bc, (q15_t)0x5e8b, (q15_t)0xa9b7, (q15_t)0x5e87, (q15_t)0xa9b3,
  (q15_t)0x5e82, (q15_t)0xa9ae, (q15_t)0x5e7e, (q15_t)0xa9a9, (q15_t)0x5e7a, (q15_t)0xa9a5, (q15_t)0x5e76, (q15_t)0xa9a0,
  (q15_t)0x5e71, (q15_t)0xa99b, (q15_t)0x5e6d, (q15_t)0xa997, (q15_t)0x5e69, (q15_t)0xa992, (q15_t)0x5e65, (q15_t)0xa98d,
  (q15_t)0x5e60, (q15_t)0xa989, (q15_t)0x5e5c, (q15_t)0xa984, (q15_t)0x5e58, (q15_t)0xa980, (q15_t)0x5e54, (q15_t)0xa97b,
  (q15_t)0x5e50, (q15_t)0xa976, (q15_t)0x5e4b, (q15_t)0xa972, (q15_t)0x5e47, (q15_t)0xa96d, (q15_t)0x5e43, (q15_t)0xa968,
  (q15_t)0x5e3f, (q15_t)0xa964, (q15_t)0x5e3a, (q15_t)0xa95f, (q15_t)0x5e36, (q15_t)0xa95b, (q15_t)0x5e32, (q15_t)0xa956,
  (q15_t)0x5e2d, (q15_t)0xa951, (q15_t)0x5e29, (q15_t)0xa94d, (q15_t)0x5e25, (q15_t)0xa948, (q15_t)0x5e21, (q15_t)0xa943,
  (q15_t)0x5e1c, (q15_t)0xa93f, (q15_t)0x5e18, (q15_t)0xa93a, (q15_t)0x5e14, (q15_t)0xa936, (q15_t)0x5e10, (q15_t)0xa931,
  (q15_t)0x5e0b, (q15_t)0xa92c, (q15_t)0x5e07, (q15_t)0xa928, (q15_t)0x5e03, (q15_t)0xa923, (q15_t)0x5dff, (q15_t)0xa91e,
  (q15_t)0x5dfa, (q15_t)0xa91a, (q15_t)0x5df6, (q15_t)0xa915, (q15_t)0x5df2, (q15_t)0xa911, (q15_t)0x5dee, (q15_t)0xa90c,
  (q15_t)0x5de9, (q15_t)0xa907, (q15_t)0x5de5, (q15_t)0xa903, (q15_t)0x5de1, (q15_t)0xa8fe, (q15_t)0x5ddc, (q15_t)0xa8fa,
  (q15_t)0x5dd8, (q15_t)0xa8f5, (q15_t)0x5dd4, (q15_t)0xa8f0, (q15_t)0x5dd0, (q15_t)0xa8ec, (q15_t)0x5dcb, (q15_t)0xa8e7,
  (q15_t)0x5dc7, (q15_t)0xa8e3, (q15_t)0x5dc3, (q15_t)0xa8de, (q15_t)0x5dbf, (q15_t)0xa8d9, (q15_t)0x5dba, (q15_t)0xa8d5,
  (q15_t)0x5db6, (q15_t)0xa8d0, (q15_t)0x5db2, (q15_t)0xa8cc, (q15_t)0x5dad, (q15_t)0xa8c7, (q15_t)0x5da9, (q15_t)0xa8c2,
  (q15_t)0x5da5, (q15_t)0xa8be, (q15_t)0x5da1, (q15_t)0xa8b9, (q15_t)0x5d9c, (q15_t)0xa8b5, (q15_t)0x5d98, (q15_t)0xa8b0,
  (q15_t)0x5d94, (q15_t)0xa8ab, (q15_t)0x5d8f, (q15_t)0xa8a7, (q15_t)0x5d8b, (q15_t)0xa8a2, (q15_t)0x5d87, (q15_t)0xa89e,
  (q15_t)0x5d83, (q15_t)0xa899, (q15_t)0x5d7e, (q15_t)0xa894, (q15_t)0x5d7a, (q15_t)0xa890, (q15_t)0x5d76, (q15_t)0xa88b,
  (q15_t)0x5d71, (q15_t)0xa887, (q15_t)0x5d6d, (q15_t)0xa882, (q15_t)0x5d69, (q15_t)0xa87d, (q15_t)0x5d65, (q15_t)0xa879,
  (q15_t)0x5d60, (q15_t)0xa874, (q15_t)0x5d5c, (q15_t)0xa870, (q15_t)0x5d58, (q15_t)0xa86b, (q15_t)0x5d53, (q15_t)0xa867,
  (q15_t)0x5d4f, (q15_t)0xa862, (q15_t)0x5d4b, (q15_t)0xa85d, (q15_t)0x5d46, (q15_t)0xa859, (q15_t)0x5d42, (q15_t)0xa854,
  (q15_t)0x5d3e, (q15_t)0xa850, (q15_t)0x5d3a, (q15_t)0xa84b, (q15_t)0x5d35, (q15_t)0xa847, (q15_t)0x5d31, (q15_t)0xa842,
  (q15_t)0x5d2d, (q15_t)0xa83d, (q15_t)0x5d28, (q15_t)0xa839, (q15_t)0x5d24, (q15_t)0xa834, (q15_t)0x5d20, (q15_t)0xa830,
  (q15_t)0x5d1b, (q15_t)0xa82b, (q15_t)0x5d17, (q15_t)0xa827, (q15_t)0x5d13, (q15_t)0xa822, (q15_t)0x5d0e, (q15_t)0xa81d,
  (q15_t)0x5d0a, (q15_t)0xa819, (q15_t)0x5d06, (q15_t)0xa814, (q15_t)0x5d01, (q15_t)0xa810, (q15_t)0x5cfd, (q15_t)0xa80b,
  (q15_t)0x5cf9, (q15_t)0xa807, (q15_t)0x5cf5, (q15_t)0xa802, (q15_t)0x5cf0, (q15_t)0xa7fd, (q15_t)0x5cec, (q15_t)0xa7f9,
  (q15_t)0x5ce8, (q15_t)0xa7f4, (q15_t)0x5ce3, (q15_t)0xa7f0, (q15_t)0x5cdf, (q15_t)0xa7eb, (q15_t)0x5cdb, (q15_t)0xa7e7,
  (q15_t)0x5cd6, (q15_t)0xa7e2, (q15_t)0x5cd2, (q15_t)0xa7de, (q15_t)0x5cce, (q15_t)0xa7d9, (q15_t)0x5cc9, (q15_t)0xa7d4,
  (q15_t)0x5cc5, (q15_t)0xa7d0, (q15_t)0x5cc1, (q15_t)0xa7cb, (q15_t)0x5cbc, (q15_t)0xa7c7, (q15_t)0x5cb8, (q15_t)0xa7c2,
  (q15_t)0x5cb4, (q15_t)0xa7be, (q15_t)0x5caf, (q15_t)0xa7b9, (q15_t)0x5cab, (q15_t)0xa7b5, (q15_t)0x5ca7, (q15_t)0xa7b0,
  (q15_t)0x5ca2, (q15_t)0xa7ab, (q15_t)0x5c9e, (q15_t)0xa7a7, (q15_t)0x5c9a, (q15_t)0xa7a2, (q15_t)0x5c95, (q15_t)0xa79e,
  (q15_t)0x5c91, (q15_t)0xa799, (q15_t)0x5c8d, (q15_t)0xa795, (q15_t)0x5c88, (q15_t)0xa790, (q15_t)0x5c84, (q15_t)0xa78c,
  (q15_t)0x5c80, (q15_t)0xa787, (q15_t)0x5c7b, (q15_t)0xa783, (q15_t)0x5c77, (q15_t)0xa77e, (q15_t)0x5c73, (q15_t)0xa779,
  (q15_t)0x5c6e, (q15_t)0xa775, (q15_t)0x5c6a, (q15_t)0xa770, (q15_t)0x5c66, (q15_t)0xa76c, (q15_t)0x5c61, (q15_t)0xa767,
  (q15_t)0x5c5d, (q15_t)0xa763, (q15_t)0x5c58, (q15_t)0xa75e, (q15_t)0x5c54, (q15_t)0xa75a, (q15_t)0x5c50, (q15_t)0xa755,
  (q15_t)0x5c4b, (q15_t)0xa751, (q15_t)0x5c47, (q15_t)0xa74c, (q15_t)0x5c43, (q15_t)0xa748, (q15_t)0x5c3e, (q15_t)0xa743,
  (q15_t)0x5c3a, (q15_t)0xa73f, (q15_t)0x5c36, (q15_t)0xa73a, (q15_t)0x5c31, (q15_t)0xa735, (q15_t)0x5c2d, (q15_t)0xa731,
  (q15_t)0x5c29, (q15_t)0xa72c, (q15_t)0x5c24, (q15_t)0xa728, (q15_t)0x5c20, (q15_t)0xa723, (q15_t)0x5c1b, (q15_t)0xa71f,
  (q15_t)0x5c17, (q15_t)0xa71a, (q15_t)0x5c13, (q15_t)0xa716, (q15_t)0x5c0e, (q15_t)0xa711, (q15_t)0x5c0a, (q15_t)0xa70d,
  (q15_t)0x5c06, (q15_t)0xa708, (q15_t)0x5c01, (q15_t)0xa704, (q15_t)0x5bfd, (q15_t)0xa6ff, (q15_t)0x5bf9, (q15_t)0xa6fb,
  (q15_t)0x5bf4, (q15_t)0xa6f6, (q15_t)0x5bf0, (q15_t)0xa6f2, (q15_t)0x5beb, (q15_t)0xa6ed, (q15_t)0x5be7, (q15_t)0xa6e9,
  (q15_t)0x5be3, (q15_t)0xa6e4, (q15_t)0x5bde, (q15_t)0xa6e0, (q15_t)0x5bda, (q15_t)0xa6db, (q15_t)0x5bd6, (q15_t)0xa6d7,
  (q15_t)0x5bd1, (q15_t)0xa6d2, (q15_t)0x5bcd, (q15_t)0xa6ce, (q15_t)0x5bc8, (q15_t)0xa6c9, (q15_t)0x5bc4, (q15_t)0xa6c5,
  (q15_t)0x5bc0, (q15_t)0xa6c0, (q15_t)0x5bbb, (q15_t)0xa6bc, (q15_t)0x5bb7, (q15_t)0xa6b7, (q15_t)0x5bb2, (q15_t)0xa6b3,
  (q15_t)0x5bae, (q15_t)0xa6ae, (q15_t)0x5baa, (q15_t)0xa6aa, (q15_t)0x5ba5, (q15_t)0xa6a5, (q15_t)0x5ba1, (q15_t)0xa6a1,
  (q15_t)0x5b9d, (q15_t)0xa69c, (q15_t)0x5b98, (q15_t)0xa698, (q15_t)0x5b94, (q15_t)0xa693, (q15_t)0x5b8f, (q15_t)0xa68f,
  (q15_t)0x5b8b, (q15_t)0xa68a, (q15_t)0x5b87, (q15_t)0xa686, (q15_t)0x5b82, (q15_t)0xa681, (q15_t)0x5b7e, (q15_t)0xa67d,
  (q15_t)0x5b79, (q15_t)0xa678, (q15_t)0x5b75, (q15_t)0xa674, (q15_t)0x5b71, (q15_t)0xa66f, (q15_t)0x5b6c, (q15_t)0xa66b,
  (q15_t)0x5b68, (q15_t)0xa666, (q15_t)0x5b63, (q15_t)0xa662, (q15_t)0x5b5f, (q15_t)0xa65d, (q15_t)0x5b5b, (q15_t)0xa659,
  (q15_t)0x5b56, (q15_t)0xa654, (q15_t)0x5b52, (q15_t)0xa650, (q15_t)0x5b4d, (q15_t)0xa64b, (q15_t)0x5b49, (q15_t)0xa647,
  (q15_t)0x5b45, (q15_t)0xa642, (q15_t)0x5b40, (q15_t)0xa63e, (q15_t)0x5b3c, (q15_t)0xa639, (q15_t)0x5b37, (q15_t)0xa635,
  (q15_t)0x5b33, (q15_t)0xa630, (q15_t)0x5b2f, (q15_t)0xa62c, (q15_t)0x5b2a, (q15_t)0xa627, (q15_t)0x5b26, (q15_t)0xa623,
  (q15_t)0x5b21, (q15_t)0xa61f, (q15_t)0x5b1d, (q15_t)0xa61a, (q15_t)0x5b19, (q15_t)0xa616, (q15_t)0x5b14, (q15_t)0xa611,
  (q15_t)0x5b10, (q15_t)0xa60d, (q15_t)0x5b0b, (q15_t)0xa608, (q15_t)0x5b07, (q15_t)0xa604, (q15_t)0x5b02, (q15_t)0xa5ff,
  (q15_t)0x5afe, (q15_t)0xa5fb, (q15_t)0x5afa, (q15_t)0xa5f6, (q15_t)0x5af5, (q15_t)0xa5f2, (q15_t)0x5af1, (q15_t)0xa5ed,
  (q15_t)0x5aec, (q15_t)0xa5e9, (q15_t)0x5ae8, (q15_t)0xa5e4, (q15_t)0x5ae4, (q15_t)0xa5e0, (q15_t)0x5adf, (q15_t)0xa5dc,
  (q15_t)0x5adb, (q15_t)0xa5d7, (q15_t)0x5ad6, (q15_t)0xa5d3, (q15_t)0x5ad2, (q15_t)0xa5ce, (q15_t)0x5acd, (q15_t)0xa5ca,
  (q15_t)0x5ac9, (q15_t)0xa5c5, (q15_t)0x5ac5, (q15_t)0xa5c1, (q15_t)0x5ac0, (q15_t)0xa5bc, (q15_t)0x5abc, (q15_t)0xa5b8,
  (q15_t)0x5ab7, (q15_t)0xa5b3, (q15_t)0x5ab3, (q15_t)0xa5af, (q15_t)0x5aae, (q15_t)0xa5aa, (q15_t)0x5aaa, (q15_t)0xa5a6,
  (q15_t)0x5aa5, (q15_t)0xa5a2, (q15_t)0x5aa1, (q15_t)0xa59d, (q15_t)0x5a9d, (q15_t)0xa599, (q15_t)0x5a98, (q15_t)0xa594,
  (q15_t)0x5a94, (q15_t)0xa590, (q15_t)0x5a8f, (q15_t)0xa58b, (q15_t)0x5a8b, (q15_t)0xa587, (q15_t)0x5a86, (q15_t)0xa582,
  (q15_t)0x5a82, (q15_t)0xa57e, (q15_t)0x5a7e, (q15_t)0xa57a, (q15_t)0x5a79, (q15_t)0xa575, (q15_t)0x5a75, (q15_t)0xa571,
  (q15_t)0x5a70, (q15_t)0xa56c, (q15_t)0x5a6c, (q15_t)0xa568, (q15_t)0x5a67, (q15_t)0xa563, (q15_t)0x5a63, (q15_t)0xa55f,
  (q15_t)0x5a5e, (q15_t)0xa55b, (q15_t)0x5a5a, (q15_t)0xa556, (q15_t)0x5a56, (q15_t)0xa552, (q15_t)0x5a51, (q15_t)0xa54d,
  (q15_t)0x5a4d, (q15_t)0xa549, (q15_t)0x5a48, (q15_t)0xa544, (q15_t)0x5a44, (q15_t)0xa540, (q15_t)0x5a3f, (q15_t)0xa53b,
  (q15_t)0x5a3b, (q15_t)0xa537, (q15_t)0x5a36, (q15_t)0xa533, (q15_t)0x5a32, (q15_t)0xa52e, (q15_t)0x5a2d, (q15_t)0xa52a,
  (q15_t)0x5a29, (q15_t)0xa525, (q15_t)0x5a24, (q15_t)0xa521, (q15_t)0x5a20, (q15_t)0xa51c, (q15_t)0x5a1c, (q15_t)0xa518,
  (q15_t)0x5a17, (q15_t)0xa514, (q15_t)0x5a13, (q15_t)0xa50f, (q15_t)0x5a0e, (q15_t)0xa50b, (q15_t)0x5a0a, (q15_t)0xa506,
  (q15_t)0x5a05, (q15_t)0xa502, (q15_t)0x5a01, (q15_t)0xa4fe, (q15_t)0x59fc, (q15_t)0xa4f9, (q15_t)0x59f8, (q15_t)0xa4f5,
  (q15_t)0x59f3, (q15_t)0xa4f0, (q15_t)0x59ef, (q15_t)0xa4ec, (q15_t)0x59ea, (q15_t)0xa4e7, (q15_t)0x59e6, (q15_t)0xa4e3,
  (q15_t)0x59e1, (q15_t)0xa4df, (q15_t)0x59dd, (q15_t)0xa4da, (q15_t)0x59d9, (q15_t)0xa4d6, (q15_t)0x59d4, (q15_t)0xa4d1,
  (q15_t)0x59d0, (q15_t)0xa4cd, (q15_t)0x59cb, (q15_t)0xa4c9, (q15_t)0x59c7, (q15_t)0xa4c4, (q15_t)0x59c2, (q15_t)0xa4c0,
  (q15_t)0x59be, (q15_t)0xa4bb, (q15_t)0x59b9, (q15_t)0xa4b7, (q15_t)0x59b5, (q15_t)0xa4b3, (q15_t)0x59b0, (q15_t)0xa4ae,
  (q15_t)0x59ac, (q15_t)0xa4aa, (q15_t)0x59a7, (q15_t)0xa4a5, (q15_t)0x59a3, (q15_t)0xa4a1, (q15_t)0x599e, (q15_t)0xa49d,
  (q15_t)0x599a, (q15_t)0xa498, (q15_t)0x5995, (q15_t)0xa494, (q15_t)0x5991, (q15_t)0xa48f, (q15_t)0x598c, (q15_t)0xa48b,
  (q15_t)0x5988, (q15_t)0xa487, (q15_t)0x5983, (q15_t)0xa482, (q15_t)0x597f, (q15_t)0xa47e, (q15_t)0x597a, (q15_t)0xa479,
  (q15_t)0x5976, (q15_t)0xa475, (q15_t)0x5971, (q15_t)0xa471, (q15_t)0x596d, (q15_t)0xa46c, (q15_t)0x5968, (q15_t)0xa468,
  (q15_t)0x5964, (q15_t)0xa463, (q15_t)0x595f, (q15_t)0xa45f, (q15_t)0x595b, (q15_t)0xa45b, (q15_t)0x5956, (q15_t)0xa456,
  (q15_t)0x5952, (q15_t)0xa452, (q15_t)0x594d, (q15_t)0xa44e, (q15_t)0x5949, (q15_t)0xa449, (q15_t)0x5944, (q15_t)0xa445,
  (q15_t)0x5940, (q15_t)0xa440, (q15_t)0x593b, (q15_t)0xa43c, (q15_t)0x5937, (q15_t)0xa438, (q15_t)0x5932, (q15_t)0xa433,
  (q15_t)0x592e, (q15_t)0xa42f, (q15_t)0x5929, (q15_t)0xa42a, (q15_t)0x5925, (q15_t)0xa426, (q15_t)0x5920, (q15_t)0xa422,
  (q15_t)0x591c, (q15_t)0xa41d, (q15_t)0x5917, (q15_t)0xa419, (q15_t)0x5913, (q15_t)0xa415, (q15_t)0x590e, (q15_t)0xa410,
  (q15_t)0x590a, (q15_t)0xa40c, (q15_t)0x5905, (q15_t)0xa407, (q15_t)0x5901, (q15_t)0xa403, (q15_t)0x58fc, (q15_t)0xa3ff,
  (q15_t)0x58f8, (q15_t)0xa3fa, (q15_t)0x58f3, (q15_t)0xa3f6, (q15_t)0x58ef, (q15_t)0xa3f2, (q15_t)0x58ea, (q15_t)0xa3ed,
  (q15_t)0x58e6, (q15_t)0xa3e9, (q15_t)0x58e1, (q15_t)0xa3e5, (q15_t)0x58dd, (q15_t)0xa3e0, (q15_t)0x58d8, (q15_t)0xa3dc,
  (q15_t)0x58d4, (q15_t)0xa3d7, (q15_t)0x58cf, (q15_t)0xa3d3, (q15_t)0x58cb, (q15_t)0xa3cf, (q15_t)0x58c6, (q15_t)0xa3ca,
  (q15_t)0x58c1, (q15_t)0xa3c6, (q15_t)0x58bd, (q15_t)0xa3c2, (q15_t)0x58b8, (q15_t)0xa3bd, (q15_t)0x58b4, (q15_t)0xa3b9,
  (q15_t)0x58af, (q15_t)0xa3b5, (q15_t)0x58ab, (q15_t)0xa3b0, (q15_t)0x58a6, (q15_t)0xa3ac, (q15_t)0x58a2, (q15_t)0xa3a8,
  (q15_t)0x589d, (q15_t)0xa3a3, (q15_t)0x5899, (q15_t)0xa39f, (q15_t)0x5894, (q15_t)0xa39a, (q15_t)0x5890, (q15_t)0xa396,
  (q15_t)0x588b, (q15_t)0xa392, (q15_t)0x5887, (q15_t)0xa38d, (q15_t)0x5882, (q15_t)0xa389, (q15_t)0x587d, (q15_t)0xa385,
  (q15_t)0x5879, (q15_t)0xa380, (q15_t)0x5874, (q15_t)0xa37c, (q15_t)0x5870, (q15_t)0xa378, (q15_t)0x586b, (q15_t)0xa373,
  (q15_t)0x5867, (q15_t)0xa36f, (q15_t)0x5862, (q15_t)0xa36b, (q15_t)0x585e, (q15_t)0xa366, (q15_t)0x5859, (q15_t)0xa362,
  (q15_t)0x5855, (q15_t)0xa35e, (q15_t)0x5850, (q15_t)0xa359, (q15_t)0x584b, (q15_t)0xa355, (q15_t)0x5847, (q15_t)0xa351,
  (q15_t)0x5842, (q15_t)0xa34c, (q15_t)0x583e, (q15_t)0xa348, (q15_t)0x5839, (q15_t)0xa344, (q15_t)0x5835, (q15_t)0xa33f,
  (q15_t)0x5830, (q15_t)0xa33b, (q15_t)0x582c, (q15_t)0xa337, (q15_t)0x5827, (q15_t)0xa332, (q15_t)0x5822, (q15_t)0xa32e,
  (q15_t)0x581e, (q15_t)0xa32a, (q15_t)0x5819, (q15_t)0xa325, (q15_t)0x5815, (q15_t)0xa321, (q15_t)0x5810, (q15_t)0xa31d,
  (q15_t)0x580c, (q15_t)0xa318, (q15_t)0x5807, (q15_t)0xa314, (q15_t)0x5803, (q15_t)0xa310, (q15_t)0x57fe, (q15_t)0xa30b,
  (q15_t)0x57f9, (q15_t)0xa307, (q15_t)0x57f5, (q15_t)0xa303, (q15_t)0x57f0, (q15_t)0xa2ff, (q15_t)0x57ec, (q15_t)0xa2fa,
  (q15_t)0x57e7, (q15_t)0xa2f6, (q15_t)0x57e3, (q15_t)0xa2f2, (q15_t)0x57de, (q15_t)0xa2ed, (q15_t)0x57d9, (q15_t)0xa2e9,
  (q15_t)0x57d5, (q15_t)0xa2e5, (q15_t)0x57d0, (q15_t)0xa2e0, (q15_t)0x57cc, (q15_t)0xa2dc, (q15_t)0x57c7, (q15_t)0xa2d8,
  (q15_t)0x57c3, (q15_t)0xa2d3, (q15_t)0x57be, (q15_t)0xa2cf, (q15_t)0x57b9, (q15_t)0xa2cb, (q15_t)0x57b5, (q15_t)0xa2c6,
  (q15_t)0x57b0, (q15_t)0xa2c2, (q15_t)0x57ac, (q15_t)0xa2be, (q15_t)0x57a7, (q15_t)0xa2ba, (q15_t)0x57a3, (q15_t)0xa2b5,
  (q15_t)0x579e, (q15_t)0xa2b1, (q15_t)0x5799, (q15_t)0xa2ad, (q15_t)0x5795, (q15_t)0xa2a8, (q15_t)0x5790, (q15_t)0xa2a4,
  (q15_t)0x578c, (q15_t)0xa2a0, (q15_t)0x5787, (q15_t)0xa29b, (q15_t)0x5783, (q15_t)0xa297, (q15_t)0x577e, (q15_t)0xa293,
  (q15_t)0x5779, (q15_t)0xa28f, (q15_t)0x5775, (q15_t)0xa28a, (q15_t)0x5770, (q15_t)0xa286, (q15_t)0x576c, (q15_t)0xa282,
  (q15_t)0x5767, (q15_t)0xa27d, (q15_t)0x5762, (q15_t)0xa279, (q15_t)0x575e, (q15_t)0xa275, (q15_t)0x5759, (q15_t)0xa271,
  (q15_t)0x5755, (q15_t)0xa26c, (q15_t)0x5750, (q15_t)0xa268, (q15_t)0x574b, (q15_t)0xa264, (q15_t)0x5747, (q15_t)0xa25f,
  (q15_t)0x5742, (q15_t)0xa25b, (q15_t)0x573e, (q15_t)0xa257, (q15_t)0x5739, (q15_t)0xa253, (q15_t)0x5734, (q15_t)0xa24e,
  (q15_t)0x5730, (q15_t)0xa24a, (q15_t)0x572b, (q15_t)0xa246, (q15_t)0x5727, (q15_t)0xa241, (q15_t)0x5722, (q15_t)0xa23d,
  (q15_t)0x571d, (q15_t)0xa239, (q15_t)0x5719, (q15_t)0xa235, (q15_t)0x5714, (q15_t)0xa230, (q15_t)0x5710, (q15_t)0xa22c,
  (q15_t)0x570b, (q15_t)0xa228, (q15_t)0x5706, (q15_t)0xa224, (q15_t)0x5702, (q15_t)0xa21f, (q15_t)0x56fd, (q15_t)0xa21b,
  (q15_t)0x56f9, (q15_t)0xa217, (q15_t)0x56f4, (q15_t)0xa212, (q15_t)0x56ef, (q15_t)0xa20e, (q15_t)0x56eb, (q15_t)0xa20a,
  (q15_t)0x56e6, (q15_t)0xa206, (q15_t)0x56e2, (q15_t)0xa201, (q15_t)0x56dd, (q15_t)0xa1fd, (q15_t)0x56d8, (q15_t)0xa1f9,
  (q15_t)0x56d4, (q15_t)0xa1f5, (q15_t)0x56cf, (q15_t)0xa1f0, (q15_t)0x56ca, (q15_t)0xa1ec, (q15_t)0x56c6, (q15_t)0xa1e8,
  (q15_t)0x56c1, (q15_t)0xa1e4, (q15_t)0x56bd, (q15_t)0xa1df, (q15_t)0x56b8, (q15_t)0xa1db, (q15_t)0x56b3, (q15_t)0xa1d7,
  (q15_t)0x56af, (q15_t)0xa1d3, (q15_t)0x56aa, (q15_t)0xa1ce, (q15_t)0x56a5, (q15_t)0xa1ca, (q15_t)0x56a1, (q15_t)0xa1c6,
  (q15_t)0x569c, (q15_t)0xa1c1, (q15_t)0x5698, (q15_t)0xa1bd, (q15_t)0x5693, (q15_t)0xa1b9, (q15_t)0x568e, (q15_t)0xa1b5,
  (q15_t)0x568a, (q15_t)0xa1b0, (q15_t)0x5685, (q15_t)0xa1ac, (q15_t)0x5680, (q15_t)0xa1a8, (q15_t)0x567c, (q15_t)0xa1a4,
  (q15_t)0x5677, (q15_t)0xa1a0, (q15_t)0x5673, (q15_t)0xa19b, (q15_t)0x566e, (q15_t)0xa197, (q15_t)0x5669, (q15_t)0xa193,
  (q15_t)0x5665, (q15_t)0xa18f, (q15_t)0x5660, (q15_t)0xa18a, (q15_t)0x565b, (q15_t)0xa186, (q15_t)0x5657, (q15_t)0xa182,
  (q15_t)0x5652, (q15_t)0xa17e, (q15_t)0x564d, (q15_t)0xa179, (q15_t)0x5649, (q15_t)0xa175, (q15_t)0x5644, (q15_t)0xa171,
  (q15_t)0x5640, (q15_t)0xa16d, (q15_t)0x563b, (q15_t)0xa168, (q15_t)0x5636, (q15_t)0xa164, (q15_t)0x5632, (q15_t)0xa160,
  (q15_t)0x562d, (q15_t)0xa15c, (q15_t)0x5628, (q15_t)0xa157, (q15_t)0x5624, (q15_t)0xa153, (q15_t)0x561f, (q15_t)0xa14f,
  (q15_t)0x561a, (q15_t)0xa14b, (q15_t)0x5616, (q15_t)0xa147, (q15_t)0x5611, (q15_t)0xa142, (q15_t)0x560c, (q15_t)0xa13e,
  (q15_t)0x5608, (q15_t)0xa13a, (q15_t)0x5603, (q15_t)0xa136, (q15_t)0x55fe, (q15_t)0xa131, (q15_t)0x55fa, (q15_t)0xa12d,
  (q15_t)0x55f5, (q15_t)0xa129, (q15_t)0x55f0, (q15_t)0xa125, (q15_t)0x55ec, (q15_t)0xa121, (q15_t)0x55e7, (q15_t)0xa11c,
  (q15_t)0x55e3, (q15_t)0xa118, (q15_t)0x55de, (q15_t)0xa114, (q15_t)0x55d9, (q15_t)0xa110, (q15_t)0x55d5, (q15_t)0xa10b,
  (q15_t)0x55d0, (q15_t)0xa107, (q15_t)0x55cb, (q15_t)0xa103, (q15_t)0x55c7, (q15_t)0xa0ff, (q15_t)0x55c2, (q15_t)0xa0fb,
  (q15_t)0x55bd, (q15_t)0xa0f6, (q15_t)0x55b9, (q15_t)0xa0f2, (q15_t)0x55b4, (q15_t)0xa0ee, (q15_t)0x55af, (q15_t)0xa0ea,
  (q15_t)0x55ab, (q15_t)0xa0e6, (q15_t)0x55a6, (q15_t)0xa0e1, (q15_t)0x55a1, (q15_t)0xa0dd, (q15_t)0x559d, (q15_t)0xa0d9,
  (q15_t)0x5598, (q15_t)0xa0d5, (q15_t)0x5593, (q15_t)0xa0d1, (q15_t)0x558f, (q15_t)0xa0cc, (q15_t)0x558a, (q15_t)0xa0c8,
  (q15_t)0x5585, (q15_t)0xa0c4, (q15_t)0x5581, (q15_t)0xa0c0, (q15_t)0x557c, (q15_t)0xa0bc, (q15_t)0x5577, (q15_t)0xa0b7,
  (q15_t)0x5572, (q15_t)0xa0b3, (q15_t)0x556e, (q15_t)0xa0af, (q15_t)0x5569, (q15_t)0xa0ab, (q15_t)0x5564, (q15_t)0xa0a7,
  (q15_t)0x5560, (q15_t)0xa0a2, (q15_t)0x555b, (q15_t)0xa09e, (q15_t)0x5556, (q15_t)0xa09a, (q15_t)0x5552, (q15_t)0xa096,
  (q15_t)0x554d, (q15_t)0xa092, (q15_t)0x5548, (q15_t)0xa08e, (q15_t)0x5544, (q15_t)0xa089, (q15_t)0x553f, (q15_t)0xa085,
  (q15_t)0x553a, (q15_t)0xa081, (q15_t)0x5536, (q15_t)0xa07d, (q15_t)0x5531, (q15_t)0xa079, (q15_t)0x552c, (q15_t)0xa074,
  (q15_t)0x5528, (q15_t)0xa070, (q15_t)0x5523, (q15_t)0xa06c, (q15_t)0x551e, (q15_t)0xa068, (q15_t)0x5519, (q15_t)0xa064,
  (q15_t)0x5515, (q15_t)0xa060, (q15_t)0x5510, (q15_t)0xa05b, (q15_t)0x550b, (q15_t)0xa057, (q15_t)0x5507, (q15_t)0xa053,
  (q15_t)0x5502, (q15_t)0xa04f, (q15_t)0x54fd, (q15_t)0xa04b, (q15_t)0x54f9, (q15_t)0xa046, (q15_t)0x54f4, (q15_t)0xa042,
  (q15_t)0x54ef, (q15_t)0xa03e, (q15_t)0x54ea, (q15_t)0xa03a, (q15_t)0x54e6, (q15_t)0xa036, (q15_t)0x54e1, (q15_t)0xa032,
  (q15_t)0x54dc, (q15_t)0xa02d, (q15_t)0x54d8, (q15_t)0xa029, (q15_t)0x54d3, (q15_t)0xa025, (q15_t)0x54ce, (q15_t)0xa021,
  (q15_t)0x54ca, (q15_t)0xa01d, (q15_t)0x54c5, (q15_t)0xa019, (q15_t)0x54c0, (q15_t)0xa014, (q15_t)0x54bb, (q15_t)0xa010,
  (q15_t)0x54b7, (q15_t)0xa00c, (q15_t)0x54b2, (q15_t)0xa008, (q15_t)0x54ad, (q15_t)0xa004, (q15_t)0x54a9, (q15_t)0xa000,
  (q15_t)0x54a4, (q15_t)0x9ffc, (q15_t)0x549f, (q15_t)0x9ff7, (q15_t)0x549a, (q15_t)0x9ff3, (q15_t)0x5496, (q15_t)0x9fef,
  (q15_t)0x5491, (q15_t)0x9feb, (q15_t)0x548c, (q15_t)0x9fe7, (q15_t)0x5488, (q15_t)0x9fe3, (q15_t)0x5483, (q15_t)0x9fde,
  (q15_t)0x547e, (q15_t)0x9fda, (q15_t)0x5479, (q15_t)0x9fd6, (q15_t)0x5475, (q15_t)0x9fd2, (q15_t)0x5470, (q15_t)0x9fce,
  (q15_t)0x546b, (q15_t)0x9fca, (q15_t)0x5467, (q15_t)0x9fc6, (q15_t)0x5462, (q15_t)0x9fc1, (q15_t)0x545d, (q15_t)0x9fbd,
  (q15_t)0x5458, (q15_t)0x9fb9, (q15_t)0x5454, (q15_t)0x9fb5, (q15_t)0x544f, (q15_t)0x9fb1, (q15_t)0x544a, (q15_t)0x9fad,
  (q15_t)0x5445, (q15_t)0x9fa9, (q15_t)0x5441, (q15_t)0x9fa4, (q15_t)0x543c, (q15_t)0x9fa0, (q15_t)0x5437, (q15_t)0x9f9c,
  (q15_t)0x5433, (q15_t)0x9f98, (q15_t)0x542e, (q15_t)0x9f94, (q15_t)0x5429, (q15_t)0x9f90, (q15_t)0x5424, (q15_t)0x9f8c,
  (q15_t)0x5420, (q15_t)0x9f88, (q15_t)0x541b, (q15_t)0x9f83, (q15_t)0x5416, (q15_t)0x9f7f, (q15_t)0x5411, (q15_t)0x9f7b,
  (q15_t)0x540d, (q15_t)0x9f77, (q15_t)0x5408, (q15_t)0x9f73, (q15_t)0x5403, (q15_t)0x9f6f, (q15_t)0x53fe, (q15_t)0x9f6b,
  (q15_t)0x53fa, (q15_t)0x9f67, (q15_t)0x53f5, (q15_t)0x9f62, (q15_t)0x53f0, (q15_t)0x9f5e, (q15_t)0x53eb, (q15_t)0x9f5a,
  (q15_t)0x53e7, (q15_t)0x9f56, (q15_t)0x53e2, (q15_t)0x9f52, (q15_t)0x53dd, (q15_t)0x9f4e, (q15_t)0x53d8, (q15_t)0x9f4a,
  (q15_t)0x53d4, (q15_t)0x9f46, (q15_t)0x53cf, (q15_t)0x9f41, (q15_t)0x53ca, (q15_t)0x9f3d, (q15_t)0x53c5, (q15_t)0x9f39,
  (q15_t)0x53c1, (q15_t)0x9f35, (q15_t)0x53bc, (q15_t)0x9f31, (q15_t)0x53b7, (q15_t)0x9f2d, (q15_t)0x53b2, (q15_t)0x9f29,
  (q15_t)0x53ae, (q15_t)0x9f25, (q15_t)0x53a9, (q15_t)0x9f21, (q15_t)0x53a4, (q15_t)0x9f1c, (q15_t)0x539f, (q15_t)0x9f18,
  (q15_t)0x539b, (q15_t)0x9f14, (q15_t)0x5396, (q15_t)0x9f10, (q15_t)0x5391, (q15_t)0x9f0c, (q15_t)0x538c, (q15_t)0x9f08,
  (q15_t)0x5388, (q15_t)0x9f04, (q15_t)0x5383, (q15_t)0x9f00, (q15_t)0x537e, (q15_t)0x9efc, (q15_t)0x5379, (q15_t)0x9ef8,
  (q15_t)0x5375, (q15_t)0x9ef3, (q15_t)0x5370, (q15_t)0x9eef, (q15_t)0x536b, (q15_t)0x9eeb, (q15_t)0x5366, (q15_t)0x9ee7,
  (q15_t)0x5362, (q15_t)0x9ee3, (q15_t)0x535d, (q15_t)0x9edf, (q15_t)0x5358, (q15_t)0x9edb, (q15_t)0x5353, (q15_t)0x9ed7,
  (q15_t)0x534e, (q15_t)0x9ed3, (q15_t)0x534a, (q15_t)0x9ecf, (q15_t)0x5345, (q15_t)0x9ecb, (q15_t)0x5340, (q15_t)0x9ec6,
  (q15_t)0x533b, (q15_t)0x9ec2, (q15_t)0x5337, (q15_t)0x9ebe, (q15_t)0x5332, (q15_t)0x9eba, (q15_t)0x532d, (q15_t)0x9eb6,
  (q15_t)0x5328, (q15_t)0x9eb2, (q15_t)0x5323, (q15_t)0x9eae, (q15_t)0x531f, (q15_t)0x9eaa, (q15_t)0x531a, (q15_t)0x9ea6,
  (q15_t)0x5315, (q15_t)0x9ea2, (q15_t)0x5310, (q15_t)0x9e9e, (q15_t)0x530c, (q15_t)0x9e9a, (q15_t)0x5307, (q15_t)0x9e95,
  (q15_t)0x5302, (q15_t)0x9e91, (q15_t)0x52fd, (q15_t)0x9e8d, (q15_t)0x52f8, (q15_t)0x9e89, (q15_t)0x52f4, (q15_t)0x9e85,
  (q15_t)0x52ef, (q15_t)0x9e81, (q15_t)0x52ea, (q15_t)0x9e7d, (q15_t)0x52e5, (q15_t)0x9e79, (q15_t)0x52e1, (q15_t)0x9e75,
  (q15_t)0x52dc, (q15_t)0x9e71, (q15_t)0x52d7, (q15_t)0x9e6d, (q15_t)0x52d2, (q15_t)0x9e69, (q15_t)0x52cd, (q15_t)0x9e65,
  (q15_t)0x52c9, (q15_t)0x9e61, (q15_t)0x52c4, (q15_t)0x9e5d, (q15_t)0x52bf, (q15_t)0x9e58, (q15_t)0x52ba, (q15_t)0x9e54,
  (q15_t)0x52b5, (q15_t)0x9e50, (q15_t)0x52b1, (q15_t)0x9e4c, (q15_t)0x52ac, (q15_t)0x9e48, (q15_t)0x52a7, (q15_t)0x9e44,
  (q15_t)0x52a2, (q15_t)0x9e40, (q15_t)0x529d, (q15_t)0x9e3c, (q15_t)0x5299, (q15_t)0x9e38, (q15_t)0x5294, (q15_t)0x9e34,
  (q15_t)0x528f, (q15_t)0x9e30, (q15_t)0x528a, (q15_t)0x9e2c, (q15_t)0x5285, (q15_t)0x9e28, (q15_t)0x5281, (q15_t)0x9e24,
  (q15_t)0x527c, (q15_t)0x9e20, (q15_t)0x5277, (q15_t)0x9e1c, (q15_t)0x5272, (q15_t)0x9e18, (q15_t)0x526d, (q15_t)0x9e14,
  (q15_t)0x5269, (q15_t)0x9e0f, (q15_t)0x5264, (q15_t)0x9e0b, (q15_t)0x525f, (q15_t)0x9e07, (q15_t)0x525a, (q15_t)0x9e03,
  (q15_t)0x5255, (q15_t)0x9dff, (q15_t)0x5251, (q15_t)0x9dfb, (q15_t)0x524c, (q15_t)0x9df7, (q15_t)0x5247, (q15_t)0x9df3,
  (q15_t)0x5242, (q15_t)0x9def, (q15_t)0x523d, (q15_t)0x9deb, (q15_t)0x5238, (q15_t)0x9de7, (q15_t)0x5234, (q15_t)0x9de3,
  (q15_t)0x522f, (q15_t)0x9ddf, (q15_t)0x522a, (q15_t)0x9ddb, (q15_t)0x5225, (q15_t)0x9dd7, (q15_t)0x5220, (q15_t)0x9dd3,
  (q15_t)0x521c, (q15_t)0x9dcf, (q15_t)0x5217, (q15_t)0x9dcb, (q15_t)0x5212, (q15_t)0x9dc7, (q15_t)0x520d, (q15_t)0x9dc3,
  (q15_t)0x5208, (q15_t)0x9dbf, (q15_t)0x5203, (q15_t)0x9dbb, (q15_t)0x51ff, (q15_t)0x9db7, (q15_t)0x51fa, (q15_t)0x9db3,
  (q15_t)0x51f5, (q15_t)0x9daf, (q15_t)0x51f0, (q15_t)0x9dab, (q15_t)0x51eb, (q15_t)0x9da7, (q15_t)0x51e6, (q15_t)0x9da3,
  (q15_t)0x51e2, (q15_t)0x9d9f, (q15_t)0x51dd, (q15_t)0x9d9b, (q15_t)0x51d8, (q15_t)0x9d97, (q15_t)0x51d3, (q15_t)0x9d93,
  (q15_t)0x51ce, (q15_t)0x9d8f, (q15_t)0x51c9, (q15_t)0x9d8b, (q15_t)0x51c5, (q15_t)0x9d86, (q15_t)0x51c0, (q15_t)0x9d82,
  (q15_t)0x51bb, (q15_t)0x9d7e, (q15_t)0x51b6, (q15_t)0x9d7a, (q15_t)0x51b1, (q15_t)0x9d76, (q15_t)0x51ac, (q15_t)0x9d72,
  (q15_t)0x51a8, (q15_t)0x9d6e, (q15_t)0x51a3, (q15_t)0x9d6a, (q15_t)0x519e, (q15_t)0x9d66, (q15_t)0x5199, (q15_t)0x9d62,
  (q15_t)0x5194, (q15_t)0x9d5e, (q15_t)0x518f, (q15_t)0x9d5a, (q15_t)0x518b, (q15_t)0x9d56, (q15_t)0x5186, (q15_t)0x9d52,
  (q15_t)0x5181, (q15_t)0x9d4e, (q15_t)0x517c, (q15_t)0x9d4a, (q15_t)0x5177, (q15_t)0x9d46, (q15_t)0x5172, (q15_t)0x9d42,
  (q15_t)0x516e, (q15_t)0x9d3e, (q15_t)0x5169, (q15_t)0x9d3a, (q15_t)0x5164, (q15_t)0x9d36, (q15_t)0x515f, (q15_t)0x9d32,
  (q15_t)0x515a, (q15_t)0x9d2e, (q15_t)0x5155, (q15_t)0x9d2a, (q15_t)0x5150, (q15_t)0x9d26, (q15_t)0x514c, (q15_t)0x9d22,
  (q15_t)0x5147, (q15_t)0x9d1e, (q15_t)0x5142, (q15_t)0x9d1a, (q15_t)0x513d, (q15_t)0x9d16, (q15_t)0x5138, (q15_t)0x9d12,
  (q15_t)0x5133, (q15_t)0x9d0e, (q15_t)0x512e, (q15_t)0x9d0b, (q15_t)0x512a, (q15_t)0x9d07, (q15_t)0x5125, (q15_t)0x9d03,
  (q15_t)0x5120, (q15_t)0x9cff, (q15_t)0x511b, (q15_t)0x9cfb, (q15_t)0x5116, (q15_t)0x9cf7, (q15_t)0x5111, (q15_t)0x9cf3,
  (q15_t)0x510c, (q15_t)0x9cef, (q15_t)0x5108, (q15_t)0x9ceb, (q15_t)0x5103, (q15_t)0x9ce7, (q15_t)0x50fe, (q15_t)0x9ce3,
  (q15_t)0x50f9, (q15_t)0x9cdf, (q15_t)0x50f4, (q15_t)0x9cdb, (q15_t)0x50ef, (q15_t)0x9cd7, (q15_t)0x50ea, (q15_t)0x9cd3,
  (q15_t)0x50e5, (q15_t)0x9ccf, (q15_t)0x50e1, (q15_t)0x9ccb, (q15_t)0x50dc, (q15_t)0x9cc7, (q15_t)0x50d7, (q15_t)0x9cc3,
  (q15_t)0x50d2, (q15_t)0x9cbf, (q15_t)0x50cd, (q15_t)0x9cbb, (q15_t)0x50c8, (q15_t)0x9cb7, (q15_t)0x50c3, (q15_t)0x9cb3,
  (q15_t)0x50bf, (q15_t)0x9caf, (q15_t)0x50ba, (q15_t)0x9cab, (q15_t)0x50b5, (q15_t)0x9ca7, (q15_t)0x50b0, (q15_t)0x9ca3,
  (q15_t)0x50ab, (q15_t)0x9c9f, (q15_t)0x50a6, (q15_t)0x9c9b, (q15_t)0x50a1, (q15_t)0x9c97, (q15_t)0x509c, (q15_t)0x9c93,
  (q15_t)0x5097, (q15_t)0x9c8f, (q15_t)0x5093, (q15_t)0x9c8b, (q15_t)0x508e, (q15_t)0x9c88, (q15_t)0x5089, (q15_t)0x9c84,
  (q15_t)0x5084, (q15_t)0x9c80, (q15_t)0x507f, (q15_t)0x9c7c, (q15_t)0x507a, (q15_t)0x9c78, (q15_t)0x5075, (q15_t)0x9c74,
  (q15_t)0x5070, (q15_t)0x9c70, (q15_t)0x506c, (q15_t)0x9c6c, (q15_t)0x5067, (q15_t)0x9c68, (q15_t)0x5062, (q15_t)0x9c64,
  (q15_t)0x505d, (q15_t)0x9c60, (q15_t)0x5058, (q15_t)0x9c5c, (q15_t)0x5053, (q15_t)0x9c58, (q15_t)0x504e, (q15_t)0x9c54,
  (q15_t)0x5049, (q15_t)0x9c50, (q15_t)0x5044, (q15_t)0x9c4c, (q15_t)0x503f, (q15_t)0x9c48, (q15_t)0x503b, (q15_t)0x9c44,
  (q15_t)0x5036, (q15_t)0x9c40, (q15_t)0x5031, (q15_t)0x9c3d, (q15_t)0x502c, (q15_t)0x9c39, (q15_t)0x5027, (q15_t)0x9c35,
  (q15_t)0x5022, (q15_t)0x9c31, (q15_t)0x501d, (q15_t)0x9c2d, (q15_t)0x5018, (q15_t)0x9c29, (q15_t)0x5013, (q15_t)0x9c25,
  (q15_t)0x500f, (q15_t)0x9c21, (q15_t)0x500a, (q15_t)0x9c1d, (q15_t)0x5005, (q15_t)0x9c19, (q15_t)0x5000, (q15_t)0x9c15,
  (q15_t)0x4ffb, (q15_t)0x9c11, (q15_t)0x4ff6, (q15_t)0x9c0d, (q15_t)0x4ff1, (q15_t)0x9c09, (q15_t)0x4fec, (q15_t)0x9c06,
  (q15_t)0x4fe7, (q15_t)0x9c02, (q15_t)0x4fe2, (q15_t)0x9bfe, (q15_t)0x4fdd, (q15_t)0x9bfa, (q15_t)0x4fd9, (q15_t)0x9bf6,
  (q15_t)0x4fd4, (q15_t)0x9bf2, (q15_t)0x4fcf, (q15_t)0x9bee, (q15_t)0x4fca, (q15_t)0x9bea, (q15_t)0x4fc5, (q15_t)0x9be6,
  (q15_t)0x4fc0, (q15_t)0x9be2, (q15_t)0x4fbb, (q15_t)0x9bde, (q15_t)0x4fb6, (q15_t)0x9bda, (q15_t)0x4fb1, (q15_t)0x9bd7,
  (q15_t)0x4fac, (q15_t)0x9bd3, (q15_t)0x4fa7, (q15_t)0x9bcf, (q15_t)0x4fa2, (q15_t)0x9bcb, (q15_t)0x4f9e, (q15_t)0x9bc7,
  (q15_t)0x4f99, (q15_t)0x9bc3, (q15_t)0x4f94, (q15_t)0x9bbf, (q15_t)0x4f8f, (q15_t)0x9bbb, (q15_t)0x4f8a, (q15_t)0x9bb7,
  (q15_t)0x4f85, (q15_t)0x9bb3, (q15_t)0x4f80, (q15_t)0x9baf, (q15_t)0x4f7b, (q15_t)0x9bac, (q15_t)0x4f76, (q15_t)0x9ba8,
  (q15_t)0x4f71, (q15_t)0x9ba4, (q15_t)0x4f6c, (q15_t)0x9ba0, (q15_t)0x4f67, (q15_t)0x9b9c, (q15_t)0x4f62, (q15_t)0x9b98,
  (q15_t)0x4f5e, (q15_t)0x9b94, (q15_t)0x4f59, (q15_t)0x9b90, (q15_t)0x4f54, (q15_t)0x9b8c, (q15_t)0x4f4f, (q15_t)0x9b88,
  (q15_t)0x4f4a, (q15_t)0x9b85, (q15_t)0x4f45, (q15_t)0x9b81, (q15_t)0x4f40, (q15_t)0x9b7d, (q15_t)0x4f3b, (q15_t)0x9b79,
  (q15_t)0x4f36, (q15_t)0x9b75, (q15_t)0x4f31, (q15_t)0x9b71, (q15_t)0x4f2c, (q15_t)0x9b6d, (q15_t)0x4f27, (q15_t)0x9b69,
  (q15_t)0x4f22, (q15_t)0x9b65, (q15_t)0x4f1d, (q15_t)0x9b62, (q15_t)0x4f18, (q15_t)0x9b5e, (q15_t)0x4f14, (q15_t)0x9b5a,
  (q15_t)0x4f0f, (q15_t)0x9b56, (q15_t)0x4f0a, (q15_t)0x9b52, (q15_t)0x4f05, (q15_t)0x9b4e, (q15_t)0x4f00, (q15_t)0x9b4a,
  (q15_t)0x4efb, (q15_t)0x9b46, (q15_t)0x4ef6, (q15_t)0x9b43, (q15_t)0x4ef1, (q15_t)0x9b3f, (q15_t)0x4eec, (q15_t)0x9b3b,
  (q15_t)0x4ee7, (q15_t)0x9b37, (q15_t)0x4ee2, (q15_t)0x9b33, (q15_t)0x4edd, (q15_t)0x9b2f, (q15_t)0x4ed8, (q15_t)0x9b2b,
  (q15_t)0x4ed3, (q15_t)0x9b27, (q15_t)0x4ece, (q15_t)0x9b24, (q15_t)0x4ec9, (q15_t)0x9b20, (q15_t)0x4ec4, (q15_t)0x9b1c,
  (q15_t)0x4ebf, (q15_t)0x9b18, (q15_t)0x4eba, (q15_t)0x9b14, (q15_t)0x4eb6, (q15_t)0x9b10, (q15_t)0x4eb1, (q15_t)0x9b0c,
  (q15_t)0x4eac, (q15_t)0x9b09, (q15_t)0x4ea7, (q15_t)0x9b05, (q15_t)0x4ea2, (q15_t)0x9b01, (q15_t)0x4e9d, (q15_t)0x9afd,
  (q15_t)0x4e98, (q15_t)0x9af9, (q15_t)0x4e93, (q15_t)0x9af5, (q15_t)0x4e8e, (q15_t)0x9af1, (q15_t)0x4e89, (q15_t)0x9aed,
  (q15_t)0x4e84, (q15_t)0x9aea, (q15_t)0x4e7f, (q15_t)0x9ae6, (q15_t)0x4e7a, (q15_t)0x9ae2, (q15_t)0x4e75, (q15_t)0x9ade,
  (q15_t)0x4e70, (q15_t)0x9ada, (q15_t)0x4e6b, (q15_t)0x9ad6, (q15_t)0x4e66, (q15_t)0x9ad3, (q15_t)0x4e61, (q15_t)0x9acf,
  (q15_t)0x4e5c, (q15_t)0x9acb, (q15_t)0x4e57, (q15_t)0x9ac7, (q15_t)0x4e52, (q15_t)0x9ac3, (q15_t)0x4e4d, (q15_t)0x9abf,
  (q15_t)0x4e48, (q15_t)0x9abb, (q15_t)0x4e43, (q15_t)0x9ab8, (q15_t)0x4e3e, (q15_t)0x9ab4, (q15_t)0x4e39, (q15_t)0x9ab0,
  (q15_t)0x4e34, (q15_t)0x9aac, (q15_t)0x4e2f, (q15_t)0x9aa8, (q15_t)0x4e2a, (q15_t)0x9aa4, (q15_t)0x4e26, (q15_t)0x9aa1,
  (q15_t)0x4e21, (q15_t)0x9a9d, (q15_t)0x4e1c, (q15_t)0x9a99, (q15_t)0x4e17, (q15_t)0x9a95, (q15_t)0x4e12, (q15_t)0x9a91,
  (q15_t)0x4e0d, (q15_t)0x9a8d, (q15_t)0x4e08, (q15_t)0x9a8a, (q15_t)0x4e03, (q15_t)0x9a86, (q15_t)0x4dfe, (q15_t)0x9a82,
  (q15_t)0x4df9, (q15_t)0x9a7e, (q15_t)0x4df4, (q15_t)0x9a7a, (q15_t)0x4def, (q15_t)0x9a76, (q15_t)0x4dea, (q15_t)0x9a73,
  (q15_t)0x4de5, (q15_t)0x9a6f, (q15_t)0x4de0, (q15_t)0x9a6b, (q15_t)0x4ddb, (q15_t)0x9a67, (q15_t)0x4dd6, (q15_t)0x9a63,
  (q15_t)0x4dd1, (q15_t)0x9a60, (q15_t)0x4dcc, (q15_t)0x9a5c, (q15_t)0x4dc7, (q15_t)0x9a58, (q15_t)0x4dc2, (q15_t)0x9a54,
  (q15_t)0x4dbd, (q15_t)0x9a50, (q15_t)0x4db8, (q15_t)0x9a4c, (q15_t)0x4db3, (q15_t)0x9a49, (q15_t)0x4dae, (q15_t)0x9a45,
  (q15_t)0x4da9, (q15_t)0x9a41, (q15_t)0x4da4, (q15_t)0x9a3d, (q15_t)0x4d9f, (q15_t)0x9a39, (q15_t)0x4d9a, (q15_t)0x9a36,
  (q15_t)0x4d95, (q15_t)0x9a32, (q15_t)0x4d90, (q15_t)0x9a2e, (q15_t)0x4d8b, (q15_t)0x9a2a, (q15_t)0x4d86, (q15_t)0x9a26,
  (q15_t)0x4d81, (q15_t)0x9a23, (q15_t)0x4d7c, (q15_t)0x9a1f, (q15_t)0x4d77, (q15_t)0x9a1b, (q15_t)0x4d72, (q15_t)0x9a17,
  (q15_t)0x4d6d, (q15_t)0x9a13, (q15_t)0x4d68, (q15_t)0x9a10, (q15_t)0x4d63, (q15_t)0x9a0c, (q15_t)0x4d5e, (q15_t)0x9a08,
  (q15_t)0x4d59, (q15_t)0x9a04, (q15_t)0x4d54, (q15_t)0x9a00, (q15_t)0x4d4f, (q15_t)0x99fd, (q15_t)0x4d4a, (q15_t)0x99f9,
  (q15_t)0x4d45, (q15_t)0x99f5, (q15_t)0x4d40, (q15_t)0x99f1, (q15_t)0x4d3b, (q15_t)0x99ed, (q15_t)0x4d36, (q15_t)0x99ea,
  (q15_t)0x4d31, (q15_t)0x99e6, (q15_t)0x4d2c, (q15_t)0x99e2, (q15_t)0x4d27, (q15_t)0x99de, (q15_t)0x4d22, (q15_t)0x99da,
  (q15_t)0x4d1d, (q15_t)0x99d7, (q15_t)0x4d18, (q15_t)0x99d3, (q15_t)0x4d13, (q15_t)0x99cf, (q15_t)0x4d0e, (q15_t)0x99cb,
  (q15_t)0x4d09, (q15_t)0x99c7, (q15_t)0x4d04, (q15_t)0x99c4, (q15_t)0x4cff, (q15_t)0x99c0, (q15_t)0x4cfa, (q15_t)0x99bc,
  (q15_t)0x4cf5, (q15_t)0x99b8, (q15_t)0x4cf0, (q15_t)0x99b5, (q15_t)0x4ceb, (q15_t)0x99b1, (q15_t)0x4ce6, (q15_t)0x99ad,
  (q15_t)0x4ce1, (q15_t)0x99a9, (q15_t)0x4cdb, (q15_t)0x99a5, (q15_t)0x4cd6, (q15_t)0x99a2, (q15_t)0x4cd1, (q15_t)0x999e,
  (q15_t)0x4ccc, (q15_t)0x999a, (q15_t)0x4cc7, (q15_t)0x9996, (q15_t)0x4cc2, (q15_t)0x9993, (q15_t)0x4cbd, (q15_t)0x998f,
  (q15_t)0x4cb8, (q15_t)0x998b, (q15_t)0x4cb3, (q15_t)0x9987, (q15_t)0x4cae, (q15_t)0x9984, (q15_t)0x4ca9, (q15_t)0x9980,
  (q15_t)0x4ca4, (q15_t)0x997c, (q15_t)0x4c9f, (q15_t)0x9978, (q15_t)0x4c9a, (q15_t)0x9975, (q15_t)0x4c95, (q15_t)0x9971,
  (q15_t)0x4c90, (q15_t)0x996d, (q15_t)0x4c8b, (q15_t)0x9969, (q15_t)0x4c86, (q15_t)0x9965, (q15_t)0x4c81, (q15_t)0x9962,
  (q15_t)0x4c7c, (q15_t)0x995e, (q15_t)0x4c77, (q15_t)0x995a, (q15_t)0x4c72, (q15_t)0x9956, (q15_t)0x4c6d, (q15_t)0x9953,
  (q15_t)0x4c68, (q15_t)0x994f, (q15_t)0x4c63, (q15_t)0x994b, (q15_t)0x4c5e, (q15_t)0x9947, (q15_t)0x4c59, (q15_t)0x9944,
  (q15_t)0x4c54, (q15_t)0x9940, (q15_t)0x4c4f, (q15_t)0x993c, (q15_t)0x4c49, (q15_t)0x9938, (q15_t)0x4c44, (q15_t)0x9935,
  (q15_t)0x4c3f, (q15_t)0x9931, (q15_t)0x4c3a, (q15_t)0x992d, (q15_t)0x4c35, (q15_t)0x992a, (q15_t)0x4c30, (q15_t)0x9926,
  (q15_t)0x4c2b, (q15_t)0x9922, (q15_t)0x4c26, (q15_t)0x991e, (q15_t)0x4c21, (q15_t)0x991b, (q15_t)0x4c1c, (q15_t)0x9917,
  (q15_t)0x4c17, (q15_t)0x9913, (q15_t)0x4c12, (q15_t)0x990f, (q15_t)0x4c0d, (q15_t)0x990c, (q15_t)0x4c08, (q15_t)0x9908,
  (q15_t)0x4c03, (q15_t)0x9904, (q15_t)0x4bfe, (q15_t)0x9900, (q15_t)0x4bf9, (q15_t)0x98fd, (q15_t)0x4bf4, (q15_t)0x98f9,
  (q15_t)0x4bef, (q15_t)0x98f5, (q15_t)0x4be9, (q15_t)0x98f2, (q15_t)0x4be4, (q15_t)0x98ee, (q15_t)0x4bdf, (q15_t)0x98ea,
  (q15_t)0x4bda, (q15_t)0x98e6, (q15_t)0x4bd5, (q15_t)0x98e3, (q15_t)0x4bd0, (q15_t)0x98df, (q15_t)0x4bcb, (q15_t)0x98db,
  (q15_t)0x4bc6, (q15_t)0x98d7, (q15_t)0x4bc1, (q15_t)0x98d4, (q15_t)0x4bbc, (q15_t)0x98d0, (q15_t)0x4bb7, (q15_t)0x98cc,
  (q15_t)0x4bb2, (q15_t)0x98c9, (q15_t)0x4bad, (q15_t)0x98c5, (q15_t)0x4ba8, (q15_t)0x98c1, (q15_t)0x4ba3, (q15_t)0x98bd,
  (q15_t)0x4b9e, (q15_t)0x98ba, (q15_t)0x4b98, (q15_t)0x98b6, (q15_t)0x4b93, (q15_t)0x98b2, (q15_t)0x4b8e, (q15_t)0x98af,
  (q15_t)0x4b89, (q15_t)0x98ab, (q15_t)0x4b84, (q15_t)0x98a7, (q15_t)0x4b7f, (q15_t)0x98a3, (q15_t)0x4b7a, (q15_t)0x98a0,
  (q15_t)0x4b75, (q15_t)0x989c, (q15_t)0x4b70, (q15_t)0x9898, (q15_t)0x4b6b, (q15_t)0x9895, (q15_t)0x4b66, (q15_t)0x9891,
  (q15_t)0x4b61, (q15_t)0x988d, (q15_t)0x4b5c, (q15_t)0x988a, (q15_t)0x4b56, (q15_t)0x9886, (q15_t)0x4b51, (q15_t)0x9882,
  (q15_t)0x4b4c, (q15_t)0x987e, (q15_t)0x4b47, (q15_t)0x987b, (q15_t)0x4b42, (q15_t)0x9877, (q15_t)0x4b3d, (q15_t)0x9873,
  (q15_t)0x4b38, (q15_t)0x9870, (q15_t)0x4b33, (q15_t)0x986c, (q15_t)0x4b2e, (q15_t)0x9868, (q15_t)0x4b29, (q15_t)0x9865,
  (q15_t)0x4b24, (q15_t)0x9861, (q15_t)0x4b1f, (q15_t)0x985d, (q15_t)0x4b19, (q15_t)0x985a, (q15_t)0x4b14, (q15_t)0x9856,
  (q15_t)0x4b0f, (q15_t)0x9852, (q15_t)0x4b0a, (q15_t)0x984e, (q15_t)0x4b05, (q15_t)0x984b, (q15_t)0x4b00, (q15_t)0x9847,
  (q15_t)0x4afb, (q15_t)0x9843, (q15_t)0x4af6, (q15_t)0x9840, (q15_t)0x4af1, (q15_t)0x983c, (q15_t)0x4aec, (q15_t)0x9838,
  (q15_t)0x4ae7, (q15_t)0x9835, (q15_t)0x4ae1, (q15_t)0x9831, (q15_t)0x4adc, (q15_t)0x982d, (q15_t)0x4ad7, (q15_t)0x982a,
  (q15_t)0x4ad2, (q15_t)0x9826, (q15_t)0x4acd, (q15_t)0x9822, (q15_t)0x4ac8, (q15_t)0x981f, (q15_t)0x4ac3, (q15_t)0x981b,
  (q15_t)0x4abe, (q15_t)0x9817, (q15_t)0x4ab9, (q15_t)0x9814, (q15_t)0x4ab4, (q15_t)0x9810, (q15_t)0x4aae, (q15_t)0x980c,
  (q15_t)0x4aa9, (q15_t)0x9809, (q15_t)0x4aa4, (q15_t)0x9805, (q15_t)0x4a9f, (q15_t)0x9801, (q15_t)0x4a9a, (q15_t)0x97fe,
  (q15_t)0x4a95, (q15_t)0x97fa, (q15_t)0x4a90, (q15_t)0x97f6, (q15_t)0x4a8b, (q15_t)0x97f3, (q15_t)0x4a86, (q15_t)0x97ef,
  (q15_t)0x4a81, (q15_t)0x97eb, (q15_t)0x4a7b, (q15_t)0x97e8, (q15_t)0x4a76, (q15_t)0x97e4, (q15_t)0x4a71, (q15_t)0x97e0,
  (q15_t)0x4a6c, (q15_t)0x97dd, (q15_t)0x4a67, (q15_t)0x97d9, (q15_t)0x4a62, (q15_t)0x97d5, (q15_t)0x4a5d, (q15_t)0x97d2,
  (q15_t)0x4a58, (q15_t)0x97ce, (q15_t)0x4a52, (q15_t)0x97cb, (q15_t)0x4a4d, (q15_t)0x97c7, (q15_t)0x4a48, (q15_t)0x97c3,
  (q15_t)0x4a43, (q15_t)0x97c0, (q15_t)0x4a3e, (q15_t)0x97bc, (q15_t)0x4a39, (q15_t)0x97b8, (q15_t)0x4a34, (q15_t)0x97b5,
  (q15_t)0x4a2f, (q15_t)0x97b1, (q15_t)0x4a2a, (q15_t)0x97ad, (q15_t)0x4a24, (q15_t)0x97aa, (q15_t)0x4a1f, (q15_t)0x97a6,
  (q15_t)0x4a1a, (q15_t)0x97a2, (q15_t)0x4a15, (q15_t)0x979f, (q15_t)0x4a10, (q15_t)0x979b, (q15_t)0x4a0b, (q15_t)0x9798,
  (q15_t)0x4a06, (q15_t)0x9794, (q15_t)0x4a01, (q15_t)0x9790, (q15_t)0x49fb, (q15_t)0x978d, (q15_t)0x49f6, (q15_t)0x9789,
  (q15_t)0x49f1, (q15_t)0x9785, (q15_t)0x49ec, (q15_t)0x9782, (q15_t)0x49e7, (q15_t)0x977e, (q15_t)0x49e2, (q15_t)0x977a,
  (q15_t)0x49dd, (q15_t)0x9777, (q15_t)0x49d8, (q15_t)0x9773, (q15_t)0x49d2, (q15_t)0x9770, (q15_t)0x49cd, (q15_t)0x976c,
  (q15_t)0x49c8, (q15_t)0x9768, (q15_t)0x49c3, (q15_t)0x9765, (q15_t)0x49be, (q15_t)0x9761, (q15_t)0x49b9, (q15_t)0x975d,
  (q15_t)0x49b4, (q15_t)0x975a, (q15_t)0x49ae, (q15_t)0x9756, (q15_t)0x49a9, (q15_t)0x9753, (q15_t)0x49a4, (q15_t)0x974f,
  (q15_t)0x499f, (q15_t)0x974b, (q15_t)0x499a, (q15_t)0x9748, (q15_t)0x4995, (q15_t)0x9744, (q15_t)0x4990, (q15_t)0x9741,
  (q15_t)0x498a, (q15_t)0x973d, (q15_t)0x4985, (q15_t)0x9739, (q15_t)0x4980, (q15_t)0x9736, (q15_t)0x497b, (q15_t)0x9732,
  (q15_t)0x4976, (q15_t)0x972f, (q15_t)0x4971, (q15_t)0x972b, (q15_t)0x496c, (q15_t)0x9727, (q15_t)0x4966, (q15_t)0x9724,
  (q15_t)0x4961, (q15_t)0x9720, (q15_t)0x495c, (q15_t)0x971d, (q15_t)0x4957, (q15_t)0x9719, (q15_t)0x4952, (q15_t)0x9715,
  (q15_t)0x494d, (q15_t)0x9712, (q15_t)0x4948, (q15_t)0x970e, (q15_t)0x4942, (q15_t)0x970b, (q15_t)0x493d, (q15_t)0x9707,
  (q15_t)0x4938, (q15_t)0x9703, (q15_t)0x4933, (q15_t)0x9700, (q15_t)0x492e, (q15_t)0x96fc, (q15_t)0x4929, (q15_t)0x96f9,
  (q15_t)0x4923, (q15_t)0x96f5, (q15_t)0x491e, (q15_t)0x96f1, (q15_t)0x4919, (q15_t)0x96ee, (q15_t)0x4914, (q15_t)0x96ea,
  (q15_t)0x490f, (q15_t)0x96e7, (q15_t)0x490a, (q15_t)0x96e3, (q15_t)0x4905, (q15_t)0x96df, (q15_t)0x48ff, (q15_t)0x96dc,
  (q15_t)0x48fa, (q15_t)0x96d8, (q15_t)0x48f5, (q15_t)0x96d5, (q15_t)0x48f0, (q15_t)0x96d1, (q15_t)0x48eb, (q15_t)0x96ce,
  (q15_t)0x48e6, (q15_t)0x96ca, (q15_t)0x48e0, (q15_t)0x96c6, (q15_t)0x48db, (q15_t)0x96c3, (q15_t)0x48d6, (q15_t)0x96bf,
  (q15_t)0x48d1, (q15_t)0x96bc, (q15_t)0x48cc, (q15_t)0x96b8, (q15_t)0x48c7, (q15_t)0x96b5, (q15_t)0x48c1, (q15_t)0x96b1,
  (q15_t)0x48bc, (q15_t)0x96ad, (q15_t)0x48b7, (q15_t)0x96aa, (q15_t)0x48b2, (q15_t)0x96a6, (q15_t)0x48ad, (q15_t)0x96a3,
  (q15_t)0x48a8, (q15_t)0x969f, (q15_t)0x48a2, (q15_t)0x969c, (q15_t)0x489d, (q15_t)0x9698, (q15_t)0x4898, (q15_t)0x9694,
  (q15_t)0x4893, (q15_t)0x9691, (q15_t)0x488e, (q15_t)0x968d, (q15_t)0x4888, (q15_t)0x968a, (q15_t)0x4883, (q15_t)0x9686,
  (q15_t)0x487e, (q15_t)0x9683, (q15_t)0x4879, (q15_t)0x967f, (q15_t)0x4874, (q15_t)0x967b, (q15_t)0x486f, (q15_t)0x9678,
  (q15_t)0x4869, (q15_t)0x9674, (q15_t)0x4864, (q15_t)0x9671, (q15_t)0x485f, (q15_t)0x966d, (q15_t)0x485a, (q15_t)0x966a,
  (q15_t)0x4855, (q15_t)0x9666, (q15_t)0x484f, (q15_t)0x9663, (q15_t)0x484a, (q15_t)0x965f, (q15_t)0x4845, (q15_t)0x965b,
  (q15_t)0x4840, (q15_t)0x9658, (q15_t)0x483b, (q15_t)0x9654, (q15_t)0x4836, (q15_t)0x9651, (q15_t)0x4830, (q15_t)0x964d,
  (q15_t)0x482b, (q15_t)0x964a, (q15_t)0x4826, (q15_t)0x9646, (q15_t)0x4821, (q15_t)0x9643, (q15_t)0x481c, (q15_t)0x963f,
  (q15_t)0x4816, (q15_t)0x963c, (q15_t)0x4811, (q15_t)0x9638, (q15_t)0x480c, (q15_t)0x9635, (q15_t)0x4807, (q15_t)0x9631,
  (q15_t)0x4802, (q15_t)0x962d, (q15_t)0x47fc, (q15_t)0x962a, (q15_t)0x47f7, (q15_t)0x9626, (q15_t)0x47f2, (q15_t)0x9623,
  (q15_t)0x47ed, (q15_t)0x961f, (q15_t)0x47e8, (q15_t)0x961c, (q15_t)0x47e2, (q15_t)0x9618, (q15_t)0x47dd, (q15_t)0x9615,
  (q15_t)0x47d8, (q15_t)0x9611, (q15_t)0x47d3, (q15_t)0x960e, (q15_t)0x47ce, (q15_t)0x960a, (q15_t)0x47c8, (q15_t)0x9607,
  (q15_t)0x47c3, (q15_t)0x9603, (q15_t)0x47be, (q15_t)0x9600, (q15_t)0x47b9, (q15_t)0x95fc, (q15_t)0x47b4, (q15_t)0x95f9,
  (q15_t)0x47ae, (q15_t)0x95f5, (q15_t)0x47a9, (q15_t)0x95f2, (q15_t)0x47a4, (q15_t)0x95ee, (q15_t)0x479f, (q15_t)0x95ea,
  (q15_t)0x479a, (q15_t)0x95e7, (q15_t)0x4794, (q15_t)0x95e3, (q15_t)0x478f, (q15_t)0x95e0, (q15_t)0x478a, (q15_t)0x95dc,
  (q15_t)0x4785, (q15_t)0x95d9, (q15_t)0x4780, (q15_t)0x95d5, (q15_t)0x477a, (q15_t)0x95d2, (q15_t)0x4775, (q15_t)0x95ce,
  (q15_t)0x4770, (q15_t)0x95cb, (q15_t)0x476b, (q15_t)0x95c7, (q15_t)0x4765, (q15_t)0x95c4, (q15_t)0x4760, (q15_t)0x95c0,
  (q15_t)0x475b, (q15_t)0x95bd, (q15_t)0x4756, (q15_t)0x95b9, (q15_t)0x4751, (q15_t)0x95b6, (q15_t)0x474b, (q15_t)0x95b2,
  (q15_t)0x4746, (q15_t)0x95af, (q15_t)0x4741, (q15_t)0x95ab, (q15_t)0x473c, (q15_t)0x95a8, (q15_t)0x4737, (q15_t)0x95a4,
  (q15_t)0x4731, (q15_t)0x95a1, (q15_t)0x472c, (q15_t)0x959d, (q15_t)0x4727, (q15_t)0x959a, (q15_t)0x4722, (q15_t)0x9596,
  (q15_t)0x471c, (q15_t)0x9593, (q15_t)0x4717, (q15_t)0x958f, (q15_t)0x4712, (q15_t)0x958c, (q15_t)0x470d, (q15_t)0x9588,
  (q15_t)0x4708, (q15_t)0x9585, (q15_t)0x4702, (q15_t)0x9581, (q15_t)0x46fd, (q15_t)0x957e, (q15_t)0x46f8, (q15_t)0x957a,
  (q15_t)0x46f3, (q15_t)0x9577, (q15_t)0x46ed, (q15_t)0x9574, (q15_t)0x46e8, (q15_t)0x9570, (q15_t)0x46e3, (q15_t)0x956d,
  (q15_t)0x46de, (q15_t)0x9569, (q15_t)0x46d8, (q15_t)0x9566, (q15_t)0x46d3, (q15_t)0x9562, (q15_t)0x46ce, (q15_t)0x955f,
  (q15_t)0x46c9, (q15_t)0x955b, (q15_t)0x46c4, (q15_t)0x9558, (q15_t)0x46be, (q15_t)0x9554, (q15_t)0x46b9, (q15_t)0x9551,
  (q15_t)0x46b4, (q15_t)0x954d, (q15_t)0x46af, (q15_t)0x954a, (q15_t)0x46a9, (q15_t)0x9546, (q15_t)0x46a4, (q15_t)0x9543,
  (q15_t)0x469f, (q15_t)0x953f, (q15_t)0x469a, (q15_t)0x953c, (q15_t)0x4694, (q15_t)0x9538, (q15_t)0x468f, (q15_t)0x9535,
  (q15_t)0x468a, (q15_t)0x9532, (q15_t)0x4685, (q15_t)0x952e, (q15_t)0x467f, (q15_t)0x952b, (q15_t)0x467a, (q15_t)0x9527,
  (q15_t)0x4675, (q15_t)0x9524, (q15_t)0x4670, (q15_t)0x9520, (q15_t)0x466a, (q15_t)0x951d, (q15_t)0x4665, (q15_t)0x9519,
  (q15_t)0x4660, (q15_t)0x9516, (q15_t)0x465b, (q15_t)0x9512, (q15_t)0x4655, (q15_t)0x950f, (q15_t)0x4650, (q15_t)0x950c,
  (q15_t)0x464b, (q15_t)0x9508, (q15_t)0x4646, (q15_t)0x9505, (q15_t)0x4640, (q15_t)0x9501, (q15_t)0x463b, (q15_t)0x94fe,
  (q15_t)0x4636, (q15_t)0x94fa, (q15_t)0x4631, (q15_t)0x94f7, (q15_t)0x462b, (q15_t)0x94f3, (q15_t)0x4626, (q15_t)0x94f0,
  (q15_t)0x4621, (q15_t)0x94ed, (q15_t)0x461c, (q15_t)0x94e9, (q15_t)0x4616, (q15_t)0x94e6, (q15_t)0x4611, (q15_t)0x94e2,
  (q15_t)0x460c, (q15_t)0x94df, (q15_t)0x4607, (q15_t)0x94db, (q15_t)0x4601, (q15_t)0x94d8, (q15_t)0x45fc, (q15_t)0x94d4,
  (q15_t)0x45f7, (q15_t)0x94d1, (q15_t)0x45f2, (q15_t)0x94ce, (q15_t)0x45ec, (q15_t)0x94ca, (q15_t)0x45e7, (q15_t)0x94c7,
  (q15_t)0x45e2, (q15_t)0x94c3, (q15_t)0x45dd, (q15_t)0x94c0, (q15_t)0x45d7, (q15_t)0x94bc, (q15_t)0x45d2, (q15_t)0x94b9,
  (q15_t)0x45cd, (q15_t)0x94b6, (q15_t)0x45c7, (q15_t)0x94b2, (q15_t)0x45c2, (q15_t)0x94af, (q15_t)0x45bd, (q15_t)0x94ab,
  (q15_t)0x45b8, (q15_t)0x94a8, (q15_t)0x45b2, (q15_t)0x94a4, (q15_t)0x45ad, (q15_t)0x94a1, (q15_t)0x45a8, (q15_t)0x949e,
  (q15_t)0x45a3, (q15_t)0x949a, (q15_t)0x459d, (q15_t)0x9497, (q15_t)0x4598, (q15_t)0x9493, (q15_t)0x4593, (q15_t)0x9490,
  (q15_t)0x458d, (q15_t)0x948d, (q15_t)0x4588, (q15_t)0x9489, (q15_t)0x4583, (q15_t)0x9486, (q15_t)0x457e, (q15_t)0x9482,
  (q15_t)0x4578, (q15_t)0x947f, (q15_t)0x4573, (q15_t)0x947b, (q15_t)0x456e, (q15_t)0x9478, (q15_t)0x4569, (q15_t)0x9475,
  (q15_t)0x4563, (q15_t)0x9471, (q15_t)0x455e, (q15_t)0x946e, (q15_t)0x4559, (q15_t)0x946a, (q15_t)0x4553, (q15_t)0x9467,
  (q15_t)0x454e, (q15_t)0x9464, (q15_t)0x4549, (q15_t)0x9460, (q15_t)0x4544, (q15_t)0x945d, (q15_t)0x453e, (q15_t)0x9459,
  (q15_t)0x4539, (q15_t)0x9456, (q15_t)0x4534, (q15_t)0x9453, (q15_t)0x452e, (q15_t)0x944f, (q15_t)0x4529, (q15_t)0x944c,
  (q15_t)0x4524, (q15_t)0x9448, (q15_t)0x451f, (q15_t)0x9445, (q15_t)0x4519, (q15_t)0x9442, (q15_t)0x4514, (q15_t)0x943e,
  (q15_t)0x450f, (q15_t)0x943b, (q15_t)0x4509, (q15_t)0x9437, (q15_t)0x4504, (q15_t)0x9434, (q15_t)0x44ff, (q15_t)0x9431,
  (q15_t)0x44fa, (q15_t)0x942d, (q15_t)0x44f4, (q15_t)0x942a, (q15_t)0x44ef, (q15_t)0x9427, (q15_t)0x44ea, (q15_t)0x9423,
  (q15_t)0x44e4, (q15_t)0x9420, (q15_t)0x44df, (q15_t)0x941c, (q15_t)0x44da, (q15_t)0x9419, (q15_t)0x44d4, (q15_t)0x9416,
  (q15_t)0x44cf, (q15_t)0x9412, (q15_t)0x44ca, (q15_t)0x940f, (q15_t)0x44c5, (q15_t)0x940b, (q15_t)0x44bf, (q15_t)0x9408,
  (q15_t)0x44ba, (q15_t)0x9405, (q15_t)0x44b5, (q15_t)0x9401, (q15_t)0x44af, (q15_t)0x93fe, (q15_t)0x44aa, (q15_t)0x93fb,
  (q15_t)0x44a5, (q15_t)0x93f7, (q15_t)0x449f, (q15_t)0x93f4, (q15_t)0x449a, (q15_t)0x93f1, (q15_t)0x4495, (q15_t)0x93ed,
  (q15_t)0x4490, (q15_t)0x93ea, (q15_t)0x448a, (q15_t)0x93e6, (q15_t)0x4485, (q15_t)0x93e3, (q15_t)0x4480, (q15_t)0x93e0,
  (q15_t)0x447a, (q15_t)0x93dc, (q15_t)0x4475, (q15_t)0x93d9, (q15_t)0x4470, (q15_t)0x93d6, (q15_t)0x446a, (q15_t)0x93d2,
  (q15_t)0x4465, (q15_t)0x93cf, (q15_t)0x4460, (q15_t)0x93cc, (q15_t)0x445a, (q15_t)0x93c8, (q15_t)0x4455, (q15_t)0x93c5,
  (q15_t)0x4450, (q15_t)0x93c1, (q15_t)0x444b, (q15_t)0x93be, (q15_t)0x4445, (q15_t)0x93bb, (q15_t)0x4440, (q15_t)0x93b7,
  (q15_t)0x443b, (q15_t)0x93b4, (q15_t)0x4435, (q15_t)0x93b1, (q15_t)0x4430, (q15_t)0x93ad, (q15_t)0x442b, (q15_t)0x93aa,
  (q15_t)0x4425, (q15_t)0x93a7, (q15_t)0x4420, (q15_t)0x93a3, (q15_t)0x441b, (q15_t)0x93a0, (q15_t)0x4415, (q15_t)0x939d,
  (q15_t)0x4410, (q15_t)0x9399, (q15_t)0x440b, (q15_t)0x9396, (q15_t)0x4405, (q15_t)0x9393, (q15_t)0x4400, (q15_t)0x938f,
  (q15_t)0x43fb, (q15_t)0x938c, (q15_t)0x43f5, (q15_t)0x9389, (q15_t)0x43f0, (q15_t)0x9385, (q15_t)0x43eb, (q15_t)0x9382,
  (q15_t)0x43e5, (q15_t)0x937f, (q15_t)0x43e0, (q15_t)0x937b, (q15_t)0x43db, (q15_t)0x9378, (q15_t)0x43d5, (q15_t)0x9375,
  (q15_t)0x43d0, (q15_t)0x9371, (q15_t)0x43cb, (q15_t)0x936e, (q15_t)0x43c5, (q15_t)0x936b, (q15_t)0x43c0, (q15_t)0x9367,
  (q15_t)0x43bb, (q15_t)0x9364, (q15_t)0x43b5, (q15_t)0x9361, (q15_t)0x43b0, (q15_t)0x935d, (q15_t)0x43ab, (q15_t)0x935a,
  (q15_t)0x43a5, (q15_t)0x9357, (q15_t)0x43a0, (q15_t)0x9353, (q15_t)0x439b, (q15_t)0x9350, (q15_t)0x4395, (q15_t)0x934d,
  (q15_t)0x4390, (q15_t)0x9349, (q15_t)0x438b, (q15_t)0x9346, (q15_t)0x4385, (q15_t)0x9343, (q15_t)0x4380, (q15_t)0x933f,
  (q15_t)0x437b, (q15_t)0x933c, (q15_t)0x4375, (q15_t)0x9339, (q15_t)0x4370, (q15_t)0x9336, (q15_t)0x436b, (q15_t)0x9332,
  (q15_t)0x4365, (q15_t)0x932f, (q15_t)0x4360, (q15_t)0x932c, (q15_t)0x435b, (q15_t)0x9328, (q15_t)0x4355, (q15_t)0x9325,
  (q15_t)0x4350, (q15_t)0x9322, (q15_t)0x434b, (q15_t)0x931e, (q15_t)0x4345, (q15_t)0x931b, (q15_t)0x4340, (q15_t)0x9318,
  (q15_t)0x433b, (q15_t)0x9314, (q15_t)0x4335, (q15_t)0x9311, (q15_t)0x4330, (q15_t)0x930e, (q15_t)0x432b, (q15_t)0x930b,
  (q15_t)0x4325, (q15_t)0x9307, (q15_t)0x4320, (q15_t)0x9304, (q15_t)0x431b, (q15_t)0x9301, (q15_t)0x4315, (q15_t)0x92fd,
  (q15_t)0x4310, (q15_t)0x92fa, (q15_t)0x430b, (q15_t)0x92f7, (q15_t)0x4305, (q15_t)0x92f4, (q15_t)0x4300, (q15_t)0x92f0,
  (q15_t)0x42fa, (q15_t)0x92ed, (q15_t)0x42f5, (q15_t)0x92ea, (q15_t)0x42f0, (q15_t)0x92e6, (q15_t)0x42ea, (q15_t)0x92e3,
  (q15_t)0x42e5, (q15_t)0x92e0, (q15_t)0x42e0, (q15_t)0x92dd, (q15_t)0x42da, (q15_t)0x92d9, (q15_t)0x42d5, (q15_t)0x92d6,
  (q15_t)0x42d0, (q15_t)0x92d3, (q15_t)0x42ca, (q15_t)0x92cf, (q15_t)0x42c5, (q15_t)0x92cc, (q15_t)0x42c0, (q15_t)0x92c9,
  (q15_t)0x42ba, (q15_t)0x92c6, (q15_t)0x42b5, (q15_t)0x92c2, (q15_t)0x42af, (q15_t)0x92bf, (q15_t)0x42aa, (q15_t)0x92bc,
  (q15_t)0x42a5, (q15_t)0x92b8, (q15_t)0x429f, (q15_t)0x92b5, (q15_t)0x429a, (q15_t)0x92b2, (q15_t)0x4295, (q15_t)0x92af,
  (q15_t)0x428f, (q15_t)0x92ab, (q15_t)0x428a, (q15_t)0x92a8, (q15_t)0x4284, (q15_t)0x92a5, (q15_t)0x427f, (q15_t)0x92a2,
  (q15_t)0x427a, (q15_t)0x929e, (q15_t)0x4274, (q15_t)0x929b, (q15_t)0x426f, (q15_t)0x9298, (q15_t)0x426a, (q15_t)0x9295,
  (q15_t)0x4264, (q15_t)0x9291, (q15_t)0x425f, (q15_t)0x928e, (q15_t)0x425a, (q15_t)0x928b, (q15_t)0x4254, (q15_t)0x9288,
  (q15_t)0x424f, (q15_t)0x9284, (q15_t)0x4249, (q15_t)0x9281, (q15_t)0x4244, (q15_t)0x927e, (q15_t)0x423f, (q15_t)0x927b,
  (q15_t)0x4239, (q15_t)0x9277, (q15_t)0x4234, (q15_t)0x9274, (q15_t)0x422f, (q15_t)0x9271, (q15_t)0x4229, (q15_t)0x926e,
  (q15_t)0x4224, (q15_t)0x926a, (q15_t)0x421e, (q15_t)0x9267, (q15_t)0x4219, (q15_t)0x9264, (q15_t)0x4214, (q15_t)0x9261,
  (q15_t)0x420e, (q15_t)0x925d, (q15_t)0x4209, (q15_t)0x925a, (q15_t)0x4203, (q15_t)0x9257, (q15_t)0x41fe, (q15_t)0x9254,
  (q15_t)0x41f9, (q15_t)0x9250, (q15_t)0x41f3, (q15_t)0x924d, (q15_t)0x41ee, (q15_t)0x924a, (q15_t)0x41e9, (q15_t)0x9247,
  (q15_t)0x41e3, (q15_t)0x9243, (q15_t)0x41de, (q15_t)0x9240, (q15_t)0x41d8, (q15_t)0x923d, (q15_t)0x41d3, (q15_t)0x923a,
  (q15_t)0x41ce, (q15_t)0x9236, (q15_t)0x41c8, (q15_t)0x9233, (q15_t)0x41c3, (q15_t)0x9230, (q15_t)0x41bd, (q15_t)0x922d,
  (q15_t)0x41b8, (q15_t)0x922a, (q15_t)0x41b3, (q15_t)0x9226, (q15_t)0x41ad, (q15_t)0x9223, (q15_t)0x41a8, (q15_t)0x9220,
  (q15_t)0x41a2, (q15_t)0x921d, (q15_t)0x419d, (q15_t)0x9219, (q15_t)0x4198, (q15_t)0x9216, (q15_t)0x4192, (q15_t)0x9213,
  (q15_t)0x418d, (q15_t)0x9210, (q15_t)0x4188, (q15_t)0x920d, (q15_t)0x4182, (q15_t)0x9209, (q15_t)0x417d, (q15_t)0x9206,
  (q15_t)0x4177, (q15_t)0x9203, (q15_t)0x4172, (q15_t)0x9200, (q15_t)0x416d, (q15_t)0x91fc, (q15_t)0x4167, (q15_t)0x91f9,
  (q15_t)0x4162, (q15_t)0x91f6, (q15_t)0x415c, (q15_t)0x91f3, (q15_t)0x4157, (q15_t)0x91f0, (q15_t)0x4152, (q15_t)0x91ec,
  (q15_t)0x414c, (q15_t)0x91e9, (q15_t)0x4147, (q15_t)0x91e6, (q15_t)0x4141, (q15_t)0x91e3, (q15_t)0x413c, (q15_t)0x91e0,
  (q15_t)0x4136, (q15_t)0x91dc, (q15_t)0x4131, (q15_t)0x91d9, (q15_t)0x412c, (q15_t)0x91d6, (q15_t)0x4126, (q15_t)0x91d3,
  (q15_t)0x4121, (q15_t)0x91d0, (q15_t)0x411b, (q15_t)0x91cc, (q15_t)0x4116, (q15_t)0x91c9, (q15_t)0x4111, (q15_t)0x91c6,
  (q15_t)0x410b, (q15_t)0x91c3, (q15_t)0x4106, (q15_t)0x91c0, (q15_t)0x4100, (q15_t)0x91bc, (q15_t)0x40fb, (q15_t)0x91b9,
  (q15_t)0x40f6, (q15_t)0x91b6, (q15_t)0x40f0, (q15_t)0x91b3, (q15_t)0x40eb, (q15_t)0x91b0, (q15_t)0x40e5, (q15_t)0x91ad,
  (q15_t)0x40e0, (q15_t)0x91a9, (q15_t)0x40da, (q15_t)0x91a6, (q15_t)0x40d5, (q15_t)0x91a3, (q15_t)0x40d0, (q15_t)0x91a0,
  (q15_t)0x40ca, (q15_t)0x919d, (q15_t)0x40c5, (q15_t)0x9199, (q15_t)0x40bf, (q15_t)0x9196, (q15_t)0x40ba, (q15_t)0x9193,
  (q15_t)0x40b5, (q15_t)0x9190, (q15_t)0x40af, (q15_t)0x918d, (q15_t)0x40aa, (q15_t)0x918a, (q15_t)0x40a4, (q15_t)0x9186,
  (q15_t)0x409f, (q15_t)0x9183, (q15_t)0x4099, (q15_t)0x9180, (q15_t)0x4094, (q15_t)0x917d, (q15_t)0x408f, (q15_t)0x917a,
  (q15_t)0x4089, (q15_t)0x9177, (q15_t)0x4084, (q15_t)0x9173, (q15_t)0x407e, (q15_t)0x9170, (q15_t)0x4079, (q15_t)0x916d,
  (q15_t)0x4073, (q15_t)0x916a, (q15_t)0x406e, (q15_t)0x9167, (q15_t)0x4069, (q15_t)0x9164, (q15_t)0x4063, (q15_t)0x9160,
  (q15_t)0x405e, (q15_t)0x915d, (q15_t)0x4058, (q15_t)0x915a, (q15_t)0x4053, (q15_t)0x9157, (q15_t)0x404d, (q15_t)0x9154,
  (q15_t)0x4048, (q15_t)0x9151, (q15_t)0x4043, (q15_t)0x914d, (q15_t)0x403d, (q15_t)0x914a, (q15_t)0x4038, (q15_t)0x9147,
  (q15_t)0x4032, (q15_t)0x9144, (q15_t)0x402d, (q15_t)0x9141, (q15_t)0x4027, (q15_t)0x913e, (q15_t)0x4022, (q15_t)0x913a,
  (q15_t)0x401d, (q15_t)0x9137, (q15_t)0x4017, (q15_t)0x9134, (q15_t)0x4012, (q15_t)0x9131, (q15_t)0x400c, (q15_t)0x912e,
  (q15_t)0x4007, (q15_t)0x912b, (q15_t)0x4001, (q15_t)0x9128, (q15_t)0x3ffc, (q15_t)0x9124, (q15_t)0x3ff6, (q15_t)0x9121,
  (q15_t)0x3ff1, (q15_t)0x911e, (q15_t)0x3fec, (q15_t)0x911b, (q15_t)0x3fe6, (q15_t)0x9118, (q15_t)0x3fe1, (q15_t)0x9115,
  (q15_t)0x3fdb, (q15_t)0x9112, (q15_t)0x3fd6, (q15_t)0x910f, (q15_t)0x3fd0, (q15_t)0x910b, (q15_t)0x3fcb, (q15_t)0x9108,
  (q15_t)0x3fc5, (q15_t)0x9105, (q15_t)0x3fc0, (q15_t)0x9102, (q15_t)0x3fbb, (q15_t)0x90ff, (q15_t)0x3fb5, (q15_t)0x90fc,
  (q15_t)0x3fb0, (q15_t)0x90f9, (q15_t)0x3faa, (q15_t)0x90f5, (q15_t)0x3fa5, (q15_t)0x90f2, (q15_t)0x3f9f, (q15_t)0x90ef,
  (q15_t)0x3f9a, (q15_t)0x90ec, (q15_t)0x3f94, (q15_t)0x90e9, (q15_t)0x3f8f, (q15_t)0x90e6, (q15_t)0x3f89, (q15_t)0x90e3,
  (q15_t)0x3f84, (q15_t)0x90e0, (q15_t)0x3f7f, (q15_t)0x90dd, (q15_t)0x3f79, (q15_t)0x90d9, (q15_t)0x3f74, (q15_t)0x90d6,
  (q15_t)0x3f6e, (q15_t)0x90d3, (q15_t)0x3f69, (q15_t)0x90d0, (q15_t)0x3f63, (q15_t)0x90cd, (q15_t)0x3f5e, (q15_t)0x90ca,
  (q15_t)0x3f58, (q15_t)0x90c7, (q15_t)0x3f53, (q15_t)0x90c4, (q15_t)0x3f4d, (q15_t)0x90c1, (q15_t)0x3f48, (q15_t)0x90bd,
  (q15_t)0x3f43, (q15_t)0x90ba, (q15_t)0x3f3d, (q15_t)0x90b7, (q15_t)0x3f38, (q15_t)0x90b4, (q15_t)0x3f32, (q15_t)0x90b1,
  (q15_t)0x3f2d, (q15_t)0x90ae, (q15_t)0x3f27, (q15_t)0x90ab, (q15_t)0x3f22, (q15_t)0x90a8, (q15_t)0x3f1c, (q15_t)0x90a5,
  (q15_t)0x3f17, (q15_t)0x90a1, (q15_t)0x3f11, (q15_t)0x909e, (q15_t)0x3f0c, (q15_t)0x909b, (q15_t)0x3f06, (q15_t)0x9098,
  (q15_t)0x3f01, (q15_t)0x9095, (q15_t)0x3efb, (q15_t)0x9092, (q15_t)0x3ef6, (q15_t)0x908f, (q15_t)0x3ef1, (q15_t)0x908c,
  (q15_t)0x3eeb, (q15_t)0x9089, (q15_t)0x3ee6, (q15_t)0x9086, (q15_t)0x3ee0, (q15_t)0x9083, (q15_t)0x3edb, (q15_t)0x907f,
  (q15_t)0x3ed5, (q15_t)0x907c, (q15_t)0x3ed0, (q15_t)0x9079, (q15_t)0x3eca, (q15_t)0x9076, (q15_t)0x3ec5, (q15_t)0x9073,
  (q15_t)0x3ebf, (q15_t)0x9070, (q15_t)0x3eba, (q15_t)0x906d, (q15_t)0x3eb4, (q15_t)0x906a, (q15_t)0x3eaf, (q15_t)0x9067,
  (q15_t)0x3ea9, (q15_t)0x9064, (q15_t)0x3ea4, (q15_t)0x9061, (q15_t)0x3e9e, (q15_t)0x905e, (q15_t)0x3e99, (q15_t)0x905b,
  (q15_t)0x3e93, (q15_t)0x9057, (q15_t)0x3e8e, (q15_t)0x9054, (q15_t)0x3e88, (q15_t)0x9051, (q15_t)0x3e83, (q15_t)0x904e,
  (q15_t)0x3e7d, (q15_t)0x904b, (q15_t)0x3e78, (q15_t)0x9048, (q15_t)0x3e73, (q15_t)0x9045, (q15_t)0x3e6d, (q15_t)0x9042,
  (q15_t)0x3e68, (q15_t)0x903f, (q15_t)0x3e62, (q15_t)0x903c, (q15_t)0x3e5d, (q15_t)0x9039, (q15_t)0x3e57, (q15_t)0x9036,
  (q15_t)0x3e52, (q15_t)0x9033, (q15_t)0x3e4c, (q15_t)0x9030, (q15_t)0x3e47, (q15_t)0x902d, (q15_t)0x3e41, (q15_t)0x902a,
  (q15_t)0x3e3c, (q15_t)0x9026, (q15_t)0x3e36, (q15_t)0x9023, (q15_t)0x3e31, (q15_t)0x9020, (q15_t)0x3e2b, (q15_t)0x901d,
  (q15_t)0x3e26, (q15_t)0x901a, (q15_t)0x3e20, (q15_t)0x9017, (q15_t)0x3e1b, (q15_t)0x9014, (q15_t)0x3e15, (q15_t)0x9011,
  (q15_t)0x3e10, (q15_t)0x900e, (q15_t)0x3e0a, (q15_t)0x900b, (q15_t)0x3e05, (q15_t)0x9008, (q15_t)0x3dff, (q15_t)0x9005,
  (q15_t)0x3dfa, (q15_t)0x9002, (q15_t)0x3df4, (q15_t)0x8fff, (q15_t)0x3def, (q15_t)0x8ffc, (q15_t)0x3de9, (q15_t)0x8ff9,
  (q15_t)0x3de4, (q15_t)0x8ff6, (q15_t)0x3dde, (q15_t)0x8ff3, (q15_t)0x3dd9, (q15_t)0x8ff0, (q15_t)0x3dd3, (q15_t)0x8fed,
  (q15_t)0x3dce, (q15_t)0x8fea, (q15_t)0x3dc8, (q15_t)0x8fe7, (q15_t)0x3dc3, (q15_t)0x8fe3, (q15_t)0x3dbd, (q15_t)0x8fe0,
  (q15_t)0x3db8, (q15_t)0x8fdd, (q15_t)0x3db2, (q15_t)0x8fda, (q15_t)0x3dad, (q15_t)0x8fd7, (q15_t)0x3da7, (q15_t)0x8fd4,
  (q15_t)0x3da2, (q15_t)0x8fd1, (q15_t)0x3d9c, (q15_t)0x8fce, (q15_t)0x3d97, (q15_t)0x8fcb, (q15_t)0x3d91, (q15_t)0x8fc8,
  (q15_t)0x3d8c, (q15_t)0x8fc5, (q15_t)0x3d86, (q15_t)0x8fc2, (q15_t)0x3d81, (q15_t)0x8fbf, (q15_t)0x3d7b, (q15_t)0x8fbc,
  (q15_t)0x3d76, (q15_t)0x8fb9, (q15_t)0x3d70, (q15_t)0x8fb6, (q15_t)0x3d6b, (q15_t)0x8fb3, (q15_t)0x3d65, (q15_t)0x8fb0,
  (q15_t)0x3d60, (q15_t)0x8fad, (q15_t)0x3d5a, (q15_t)0x8faa, (q15_t)0x3d55, (q15_t)0x8fa7, (q15_t)0x3d4f, (q15_t)0x8fa4,
  (q15_t)0x3d49, (q15_t)0x8fa1, (q15_t)0x3d44, (q15_t)0x8f9e, (q15_t)0x3d3e, (q15_t)0x8f9b, (q15_t)0x3d39, (q15_t)0x8f98,
  (q15_t)0x3d33, (q15_t)0x8f95, (q15_t)0x3d2e, (q15_t)0x8f92, (q15_t)0x3d28, (q15_t)0x8f8f, (q15_t)0x3d23, (q15_t)0x8f8c,
  (q15_t)0x3d1d, (q15_t)0x8f89, (q15_t)0x3d18, (q15_t)0x8f86, (q15_t)0x3d12, (q15_t)0x8f83, (q15_t)0x3d0d, (q15_t)0x8f80,
  (q15_t)0x3d07, (q15_t)0x8f7d, (q15_t)0x3d02, (q15_t)0x8f7a, (q15_t)0x3cfc, (q15_t)0x8f77, (q15_t)0x3cf7, (q15_t)0x8f74,
  (q15_t)0x3cf1, (q15_t)0x8f71, (q15_t)0x3cec, (q15_t)0x8f6e, (q15_t)0x3ce6, (q15_t)0x8f6b, (q15_t)0x3ce1, (q15_t)0x8f68,
  (q15_t)0x3cdb, (q15_t)0x8f65, (q15_t)0x3cd6, (q15_t)0x8f62, (q15_t)0x3cd0, (q15_t)0x8f5f, (q15_t)0x3cca, (q15_t)0x8f5c,
  (q15_t)0x3cc5, (q15_t)0x8f59, (q15_t)0x3cbf, (q15_t)0x8f56, (q15_t)0x3cba, (q15_t)0x8f53, (q15_t)0x3cb4, (q15_t)0x8f50,
  (q15_t)0x3caf, (q15_t)0x8f4d, (q15_t)0x3ca9, (q15_t)0x8f4a, (q15_t)0x3ca4, (q15_t)0x8f47, (q15_t)0x3c9e, (q15_t)0x8f44,
  (q15_t)0x3c99, (q15_t)0x8f41, (q15_t)0x3c93, (q15_t)0x8f3e, (q15_t)0x3c8e, (q15_t)0x8f3b, (q15_t)0x3c88, (q15_t)0x8f38,
  (q15_t)0x3c83, (q15_t)0x8f35, (q15_t)0x3c7d, (q15_t)0x8f32, (q15_t)0x3c77, (q15_t)0x8f2f, (q15_t)0x3c72, (q15_t)0x8f2d,
  (q15_t)0x3c6c, (q15_t)0x8f2a, (q15_t)0x3c67, (q15_t)0x8f27, (q15_t)0x3c61, (q15_t)0x8f24, (q15_t)0x3c5c, (q15_t)0x8f21,
  (q15_t)0x3c56, (q15_t)0x8f1e, (q15_t)0x3c51, (q15_t)0x8f1b, (q15_t)0x3c4b, (q15_t)0x8f18, (q15_t)0x3c46, (q15_t)0x8f15,
  (q15_t)0x3c40, (q15_t)0x8f12, (q15_t)0x3c3b, (q15_t)0x8f0f, (q15_t)0x3c35, (q15_t)0x8f0c, (q15_t)0x3c2f, (q15_t)0x8f09,
  (q15_t)0x3c2a, (q15_t)0x8f06, (q15_t)0x3c24, (q15_t)0x8f03, (q15_t)0x3c1f, (q15_t)0x8f00, (q15_t)0x3c19, (q15_t)0x8efd,
  (q15_t)0x3c14, (q15_t)0x8efa, (q15_t)0x3c0e, (q15_t)0x8ef7, (q15_t)0x3c09, (q15_t)0x8ef4, (q15_t)0x3c03, (q15_t)0x8ef1,
  (q15_t)0x3bfd, (q15_t)0x8eee, (q15_t)0x3bf8, (q15_t)0x8eec, (q15_t)0x3bf2, (q15_t)0x8ee9, (q15_t)0x3bed, (q15_t)0x8ee6,
  (q15_t)0x3be7, (q15_t)0x8ee3, (q15_t)0x3be2, (q15_t)0x8ee0, (q15_t)0x3bdc, (q15_t)0x8edd, (q15_t)0x3bd7, (q15_t)0x8eda,
  (q15_t)0x3bd1, (q15_t)0x8ed7, (q15_t)0x3bcc, (q15_t)0x8ed4, (q15_t)0x3bc6, (q15_t)0x8ed1, (q15_t)0x3bc0, (q15_t)0x8ece,
  (q15_t)0x3bbb, (q15_t)0x8ecb, (q15_t)0x3bb5, (q15_t)0x8ec8, (q15_t)0x3bb0, (q15_t)0x8ec5, (q15_t)0x3baa, (q15_t)0x8ec2,
  (q15_t)0x3ba5, (q15_t)0x8ebf, (q15_t)0x3b9f, (q15_t)0x8ebd, (q15_t)0x3b99, (q15_t)0x8eba, (q15_t)0x3b94, (q15_t)0x8eb7,
  (q15_t)0x3b8e, (q15_t)0x8eb4, (q15_t)0x3b89, (q15_t)0x8eb1, (q15_t)0x3b83, (q15_t)0x8eae, (q15_t)0x3b7e, (q15_t)0x8eab,
  (q15_t)0x3b78, (q15_t)0x8ea8, (q15_t)0x3b73, (q15_t)0x8ea5, (q15_t)0x3b6d, (q15_t)0x8ea2, (q15_t)0x3b67, (q15_t)0x8e9f,
  (q15_t)0x3b62, (q15_t)0x8e9c, (q15_t)0x3b5c, (q15_t)0x8e99, (q15_t)0x3b57, (q15_t)0x8e97, (q15_t)0x3b51, (q15_t)0x8e94,
  (q15_t)0x3b4c, (q15_t)0x8e91, (q15_t)0x3b46, (q15_t)0x8e8e, (q15_t)0x3b40, (q15_t)0x8e8b, (q15_t)0x3b3b, (q15_t)0x8e88,
  (q15_t)0x3b35, (q15_t)0x8e85, (q15_t)0x3b30, (q15_t)0x8e82, (q15_t)0x3b2a, (q15_t)0x8e7f, (q15_t)0x3b25, (q15_t)0x8e7c,
  (q15_t)0x3b1f, (q15_t)0x8e7a, (q15_t)0x3b19, (q15_t)0x8e77, (q15_t)0x3b14, (q15_t)0x8e74, (q15_t)0x3b0e, (q15_t)0x8e71,
  (q15_t)0x3b09, (q15_t)0x8e6e, (q15_t)0x3b03, (q15_t)0x8e6b, (q15_t)0x3afe, (q15_t)0x8e68, (q15_t)0x3af8, (q15_t)0x8e65,
  (q15_t)0x3af2, (q15_t)0x8e62, (q15_t)0x3aed, (q15_t)0x8e5f, (q15_t)0x3ae7, (q15_t)0x8e5d, (q15_t)0x3ae2, (q15_t)0x8e5a,
  (q15_t)0x3adc, (q15_t)0x8e57, (q15_t)0x3ad7, (q15_t)0x8e54, (q15_t)0x3ad1, (q15_t)0x8e51, (q15_t)0x3acb, (q15_t)0x8e4e,
  (q15_t)0x3ac6, (q15_t)0x8e4b, (q15_t)0x3ac0, (q15_t)0x8e48, (q15_t)0x3abb, (q15_t)0x8e45, (q15_t)0x3ab5, (q15_t)0x8e43,
  (q15_t)0x3aaf, (q15_t)0x8e40, (q15_t)0x3aaa, (q15_t)0x8e3d, (q15_t)0x3aa4, (q15_t)0x8e3a, (q15_t)0x3a9f, (q15_t)0x8e37,
  (q15_t)0x3a99, (q15_t)0x8e34, (q15_t)0x3a94, (q15_t)0x8e31, (q15_t)0x3a8e, (q15_t)0x8e2e, (q15_t)0x3a88, (q15_t)0x8e2c,
  (q15_t)0x3a83, (q15_t)0x8e29, (q15_t)0x3a7d, (q15_t)0x8e26, (q15_t)0x3a78, (q15_t)0x8e23, (q15_t)0x3a72, (q15_t)0x8e20,
  (q15_t)0x3a6c, (q15_t)0x8e1d, (q15_t)0x3a67, (q15_t)0x8e1a, (q15_t)0x3a61, (q15_t)0x8e17, (q15_t)0x3a5c, (q15_t)0x8e15,
  (q15_t)0x3a56, (q15_t)0x8e12, (q15_t)0x3a50, (q15_t)0x8e0f, (q15_t)0x3a4b, (q15_t)0x8e0c, (q15_t)0x3a45, (q15_t)0x8e09,
  (q15_t)0x3a40, (q15_t)0x8e06, (q15_t)0x3a3a, (q15_t)0x8e03, (q15_t)0x3a34, (q15_t)0x8e01, (q15_t)0x3a2f, (q15_t)0x8dfe,
  (q15_t)0x3a29, (q15_t)0x8dfb, (q15_t)0x3a24, (q15_t)0x8df8, (q15_t)0x3a1e, (q15_t)0x8df5, (q15_t)0x3a19, (q15_t)0x8df2,
  (q15_t)0x3a13, (q15_t)0x8def, (q15_t)0x3a0d, (q15_t)0x8ded, (q15_t)0x3a08, (q15_t)0x8dea, (q15_t)0x3a02, (q15_t)0x8de7,
  (q15_t)0x39fd, (q15_t)0x8de4, (q15_t)0x39f7, (q15_t)0x8de1, (q15_t)0x39f1, (q15_t)0x8dde, (q15_t)0x39ec, (q15_t)0x8ddc,
  (q15_t)0x39e6, (q15_t)0x8dd9, (q15_t)0x39e0, (q15_t)0x8dd6, (q15_t)0x39db, (q15_t)0x8dd3, (q15_t)0x39d5, (q15_t)0x8dd0,
  (q15_t)0x39d0, (q15_t)0x8dcd, (q15_t)0x39ca, (q15_t)0x8dca, (q15_t)0x39c4, (q15_t)0x8dc8, (q15_t)0x39bf, (q15_t)0x8dc5,
  (q15_t)0x39b9, (q15_t)0x8dc2, (q15_t)0x39b4, (q15_t)0x8dbf, (q15_t)0x39ae, (q15_t)0x8dbc, (q15_t)0x39a8, (q15_t)0x8db9,
  (q15_t)0x39a3, (q15_t)0x8db7, (q15_t)0x399d, (q15_t)0x8db4, (q15_t)0x3998, (q15_t)0x8db1, (q15_t)0x3992, (q15_t)0x8dae,
  (q15_t)0x398c, (q15_t)0x8dab, (q15_t)0x3987, (q15_t)0x8da9, (q15_t)0x3981, (q15_t)0x8da6, (q15_t)0x397c, (q15_t)0x8da3,
  (q15_t)0x3976, (q15_t)0x8da0, (q15_t)0x3970, (q15_t)0x8d9d, (q15_t)0x396b, (q15_t)0x8d9a, (q15_t)0x3965, (q15_t)0x8d98,
  (q15_t)0x395f, (q15_t)0x8d95, (q15_t)0x395a, (q15_t)0x8d92, (q15_t)0x3954, (q15_t)0x8d8f, (q15_t)0x394f, (q15_t)0x8d8c,
  (q15_t)0x3949, (q15_t)0x8d8a, (q15_t)0x3943, (q15_t)0x8d87, (q15_t)0x393e, (q15_t)0x8d84, (q15_t)0x3938, (q15_t)0x8d81,
  (q15_t)0x3932, (q15_t)0x8d7e, (q15_t)0x392d, (q15_t)0x8d7b, (q15_t)0x3927, (q15_t)0x8d79, (q15_t)0x3922, (q15_t)0x8d76,
  (q15_t)0x391c, (q15_t)0x8d73, (q15_t)0x3916, (q15_t)0x8d70, (q15_t)0x3911, (q15_t)0x8d6d, (q15_t)0x390b, (q15_t)0x8d6b,
  (q15_t)0x3906, (q15_t)0x8d68, (q15_t)0x3900, (q15_t)0x8d65, (q15_t)0x38fa, (q15_t)0x8d62, (q15_t)0x38f5, (q15_t)0x8d5f,
  (q15_t)0x38ef, (q15_t)0x8d5d, (q15_t)0x38e9, (q15_t)0x8d5a, (q15_t)0x38e4, (q15_t)0x8d57, (q15_t)0x38de, (q15_t)0x8d54,
  (q15_t)0x38d8, (q15_t)0x8d51, (q15_t)0x38d3, (q15_t)0x8d4f, (q15_t)0x38cd, (q15_t)0x8d4c, (q15_t)0x38c8, (q15_t)0x8d49,
  (q15_t)0x38c2, (q15_t)0x8d46, (q15_t)0x38bc, (q15_t)0x8d44, (q15_t)0x38b7, (q15_t)0x8d41, (q15_t)0x38b1, (q15_t)0x8d3e,
  (q15_t)0x38ab, (q15_t)0x8d3b, (q15_t)0x38a6, (q15_t)0x8d38, (q15_t)0x38a0, (q15_t)0x8d36, (q15_t)0x389b, (q15_t)0x8d33,
  (q15_t)0x3895, (q15_t)0x8d30, (q15_t)0x388f, (q15_t)0x8d2d, (q15_t)0x388a, (q15_t)0x8d2b, (q15_t)0x3884, (q15_t)0x8d28,
  (q15_t)0x387e, (q15_t)0x8d25, (q15_t)0x3879, (q15_t)0x8d22, (q15_t)0x3873, (q15_t)0x8d1f, (q15_t)0x386d, (q15_t)0x8d1d,
  (q15_t)0x3868, (q15_t)0x8d1a, (q15_t)0x3862, (q15_t)0x8d17, (q15_t)0x385d, (q15_t)0x8d14, (q15_t)0x3857, (q15_t)0x8d12,
  (q15_t)0x3851, (q15_t)0x8d0f, (q15_t)0x384c, (q15_t)0x8d0c, (q15_t)0x3846, (q15_t)0x8d09, (q15_t)0x3840, (q15_t)0x8d07,
  (q15_t)0x383b, (q15_t)0x8d04, (q15_t)0x3835, (q15_t)0x8d01, (q15_t)0x382f, (q15_t)0x8cfe, (q15_t)0x382a, (q15_t)0x8cfb,
  (q15_t)0x3824, (q15_t)0x8cf9, (q15_t)0x381e, (q15_t)0x8cf6, (q15_t)0x3819, (q15_t)0x8cf3, (q15_t)0x3813, (q15_t)0x8cf0,
  (q15_t)0x380d, (q15_t)0x8cee, (q15_t)0x3808, (q15_t)0x8ceb, (q15_t)0x3802, (q15_t)0x8ce8, (q15_t)0x37fd, (q15_t)0x8ce5,
  (q15_t)0x37f7, (q15_t)0x8ce3, (q15_t)0x37f1, (q15_t)0x8ce0, (q15_t)0x37ec, (q15_t)0x8cdd, (q15_t)0x37e6, (q15_t)0x8cda,
  (q15_t)0x37e0, (q15_t)0x8cd8, (q15_t)0x37db, (q15_t)0x8cd5, (q15_t)0x37d5, (q15_t)0x8cd2, (q15_t)0x37cf, (q15_t)0x8cd0,
  (q15_t)0x37ca, (q15_t)0x8ccd, (q15_t)0x37c4, (q15_t)0x8cca, (q15_t)0x37be, (q15_t)0x8cc7, (q15_t)0x37b9, (q15_t)0x8cc5,
  (q15_t)0x37b3, (q15_t)0x8cc2, (q15_t)0x37ad, (q15_t)0x8cbf, (q15_t)0x37a8, (q15_t)0x8cbc, (q15_t)0x37a2, (q15_t)0x8cba,
  (q15_t)0x379c, (q15_t)0x8cb7, (q15_t)0x3797, (q15_t)0x8cb4, (q15_t)0x3791, (q15_t)0x8cb1, (q15_t)0x378b, (q15_t)0x8caf,
  (q15_t)0x3786, (q15_t)0x8cac, (q15_t)0x3780, (q15_t)0x8ca9, (q15_t)0x377a, (q15_t)0x8ca7, (q15_t)0x3775, (q15_t)0x8ca4,
  (q15_t)0x376f, (q15_t)0x8ca1, (q15_t)0x3769, (q15_t)0x8c9e, (q15_t)0x3764, (q15_t)0x8c9c, (q15_t)0x375e, (q15_t)0x8c99,
  (q15_t)0x3758, (q15_t)0x8c96, (q15_t)0x3753, (q15_t)0x8c94, (q15_t)0x374d, (q15_t)0x8c91, (q15_t)0x3747, (q15_t)0x8c8e,
  (q15_t)0x3742, (q15_t)0x8c8b, (q15_t)0x373c, (q15_t)0x8c89, (q15_t)0x3736, (q15_t)0x8c86, (q15_t)0x3731, (q15_t)0x8c83,
  (q15_t)0x372b, (q15_t)0x8c81, (q15_t)0x3725, (q15_t)0x8c7e, (q15_t)0x3720, (q15_t)0x8c7b, (q15_t)0x371a, (q15_t)0x8c78,
  (q15_t)0x3714, (q15_t)0x8c76, (q15_t)0x370f, (q15_t)0x8c73, (q15_t)0x3709, (q15_t)0x8c70, (q15_t)0x3703, (q15_t)0x8c6e,
  (q15_t)0x36fe, (q15_t)0x8c6b, (q15_t)0x36f8, (q15_t)0x8c68, (q15_t)0x36f2, (q15_t)0x8c65, (q15_t)0x36ed, (q15_t)0x8c63,
  (q15_t)0x36e7, (q15_t)0x8c60, (q15_t)0x36e1, (q15_t)0x8c5d, (q15_t)0x36dc, (q15_t)0x8c5b, (q15_t)0x36d6, (q15_t)0x8c58,
  (q15_t)0x36d0, (q15_t)0x8c55, (q15_t)0x36cb, (q15_t)0x8c53, (q15_t)0x36c5, (q15_t)0x8c50, (q15_t)0x36bf, (q15_t)0x8c4d,
  (q15_t)0x36ba, (q15_t)0x8c4b, (q15_t)0x36b4, (q15_t)0x8c48, (q15_t)0x36ae, (q15_t)0x8c45, (q15_t)0x36a9, (q15_t)0x8c43,
  (q15_t)0x36a3, (q15_t)0x8c40, (q15_t)0x369d, (q15_t)0x8c3d, (q15_t)0x3698, (q15_t)0x8c3a, (q15_t)0x3692, (q15_t)0x8c38,
  (q15_t)0x368c, (q15_t)0x8c35, (q15_t)0x3686, (q15_t)0x8c32, (q15_t)0x3681, (q15_t)0x8c30, (q15_t)0x367b, (q15_t)0x8c2d,
  (q15_t)0x3675, (q15_t)0x8c2a, (q15_t)0x3670, (q15_t)0x8c28, (q15_t)0x366a, (q15_t)0x8c25, (q15_t)0x3664, (q15_t)0x8c22,
  (q15_t)0x365f, (q15_t)0x8c20, (q15_t)0x3659, (q15_t)0x8c1d, (q15_t)0x3653, (q15_t)0x8c1a, (q15_t)0x364e, (q15_t)0x8c18,
  (q15_t)0x3648, (q15_t)0x8c15, (q15_t)0x3642, (q15_t)0x8c12, (q15_t)0x363d, (q15_t)0x8c10, (q15_t)0x3637, (q15_t)0x8c0d,
  (q15_t)0x3631, (q15_t)0x8c0a, (q15_t)0x362b, (q15_t)0x8c08, (q15_t)0x3626, (q15_t)0x8c05, (q15_t)0x3620, (q15_t)0x8c02,
  (q15_t)0x361a, (q15_t)0x8c00, (q15_t)0x3615, (q15_t)0x8bfd, (q15_t)0x360f, (q15_t)0x8bfa, (q15_t)0x3609, (q15_t)0x8bf8,
  (q15_t)0x3604, (q15_t)0x8bf5, (q15_t)0x35fe, (q15_t)0x8bf3, (q15_t)0x35f8, (q15_t)0x8bf0, (q15_t)0x35f3, (q15_t)0x8bed,
  (q15_t)0x35ed, (q15_t)0x8beb, (q15_t)0x35e7, (q15_t)0x8be8, (q15_t)0x35e1, (q15_t)0x8be5, (q15_t)0x35dc, (q15_t)0x8be3,
  (q15_t)0x35d6, (q15_t)0x8be0, (q15_t)0x35d0, (q15_t)0x8bdd, (q15_t)0x35cb, (q15_t)0x8bdb, (q15_t)0x35c5, (q15_t)0x8bd8,
  (q15_t)0x35bf, (q15_t)0x8bd5, (q15_t)0x35ba, (q15_t)0x8bd3, (q15_t)0x35b4, (q15_t)0x8bd0, (q15_t)0x35ae, (q15_t)0x8bce,
  (q15_t)0x35a8, (q15_t)0x8bcb, (q15_t)0x35a3, (q15_t)0x8bc8, (q15_t)0x359d, (q15_t)0x8bc6, (q15_t)0x3597, (q15_t)0x8bc3,
  (q15_t)0x3592, (q15_t)0x8bc0, (q15_t)0x358c, (q15_t)0x8bbe, (q15_t)0x3586, (q15_t)0x8bbb, (q15_t)0x3580, (q15_t)0x8bb8,
  (q15_t)0x357b, (q15_t)0x8bb6, (q15_t)0x3575, (q15_t)0x8bb3, (q15_t)0x356f, (q15_t)0x8bb1, (q15_t)0x356a, (q15_t)0x8bae,
  (q15_t)0x3564, (q15_t)0x8bab, (q15_t)0x355e, (q15_t)0x8ba9, (q15_t)0x3558, (q15_t)0x8ba6, (q15_t)0x3553, (q15_t)0x8ba4,
  (q15_t)0x354d, (q15_t)0x8ba1, (q15_t)0x3547, (q15_t)0x8b9e, (q15_t)0x3542, (q15_t)0x8b9c, (q15_t)0x353c, (q15_t)0x8b99,
  (q15_t)0x3536, (q15_t)0x8b96, (q15_t)0x3530, (q15_t)0x8b94, (q15_t)0x352b, (q15_t)0x8b91, (q15_t)0x3525, (q15_t)0x8b8f,
  (q15_t)0x351f, (q15_t)0x8b8c, (q15_t)0x351a, (q15_t)0x8b89, (q15_t)0x3514, (q15_t)0x8b87, (q15_t)0x350e, (q15_t)0x8b84,
  (q15_t)0x3508, (q15_t)0x8b82, (q15_t)0x3503, (q15_t)0x8b7f, (q15_t)0x34fd, (q15_t)0x8b7c, (q15_t)0x34f7, (q15_t)0x8b7a,
  (q15_t)0x34f2, (q15_t)0x8b77, (q15_t)0x34ec, (q15_t)0x8b75, (q15_t)0x34e6, (q15_t)0x8b72, (q15_t)0x34e0, (q15_t)0x8b6f,
  (q15_t)0x34db, (q15_t)0x8b6d, (q15_t)0x34d5, (q15_t)0x8b6a, (q15_t)0x34cf, (q15_t)0x8b68, (q15_t)0x34ca, (q15_t)0x8b65,
  (q15_t)0x34c4, (q15_t)0x8b62, (q15_t)0x34be, (q15_t)0x8b60, (q15_t)0x34b8, (q15_t)0x8b5d, (q15_t)0x34b3, (q15_t)0x8b5b,
  (q15_t)0x34ad, (q15_t)0x8b58, (q15_t)0x34a7, (q15_t)0x8b55, (q15_t)0x34a1, (q15_t)0x8b53, (q15_t)0x349c, (q15_t)0x8b50,
  (q15_t)0x3496, (q15_t)0x8b4e, (q15_t)0x3490, (q15_t)0x8b4b, (q15_t)0x348b, (q15_t)0x8b49, (q15_t)0x3485, (q15_t)0x8b46,
  (q15_t)0x347f, (q15_t)0x8b43, (q15_t)0x3479, (q15_t)0x8b41, (q15_t)0x3474, (q15_t)0x8b3e, (q15_t)0x346e, (q15_t)0x8b3c,
  (q15_t)0x3468, (q15_t)0x8b39, (q15_t)0x3462, (q15_t)0x8b37, (q15_t)0x345d, (q15_t)0x8b34, (q15_t)0x3457, (q15_t)0x8b31,
  (q15_t)0x3451, (q15_t)0x8b2f, (q15_t)0x344b, (q15_t)0x8b2c, (q15_t)0x3446, (q15_t)0x8b2a, (q15_t)0x3440, (q15_t)0x8b27,
  (q15_t)0x343a, (q15_t)0x8b25, (q15_t)0x3435, (q15_t)0x8b22, (q15_t)0x342f, (q15_t)0x8b1f, (q15_t)0x3429, (q15_t)0x8b1d,
  (q15_t)0x3423, (q15_t)0x8b1a, (q15_t)0x341e, (q15_t)0x8b18, (q15_t)0x3418, (q15_t)0x8b15, (q15_t)0x3412, (q15_t)0x8b13,
  (q15_t)0x340c, (q15_t)0x8b10, (q15_t)0x3407, (q15_t)0x8b0e, (q15_t)0x3401, (q15_t)0x8b0b, (q15_t)0x33fb, (q15_t)0x8b08,
  (q15_t)0x33f5, (q15_t)0x8b06, (q15_t)0x33f0, (q15_t)0x8b03, (q15_t)0x33ea, (q15_t)0x8b01, (q15_t)0x33e4, (q15_t)0x8afe,
  (q15_t)0x33de, (q15_t)0x8afc, (q15_t)0x33d9, (q15_t)0x8af9, (q15_t)0x33d3, (q15_t)0x8af7, (q15_t)0x33cd, (q15_t)0x8af4,
  (q15_t)0x33c7, (q15_t)0x8af1, (q15_t)0x33c2, (q15_t)0x8aef, (q15_t)0x33bc, (q15_t)0x8aec, (q15_t)0x33b6, (q15_t)0x8aea,
  (q15_t)0x33b0, (q15_t)0x8ae7, (q15_t)0x33ab, (q15_t)0x8ae5, (q15_t)0x33a5, (q15_t)0x8ae2, (q15_t)0x339f, (q15_t)0x8ae0,
  (q15_t)0x3399, (q15_t)0x8add, (q15_t)0x3394, (q15_t)0x8adb, (q15_t)0x338e, (q15_t)0x8ad8, (q15_t)0x3388, (q15_t)0x8ad6,
  (q15_t)0x3382, (q15_t)0x8ad3, (q15_t)0x337d, (q15_t)0x8ad1, (q15_t)0x3377, (q15_t)0x8ace, (q15_t)0x3371, (q15_t)0x8acb,
  (q15_t)0x336b, (q15_t)0x8ac9, (q15_t)0x3366, (q15_t)0x8ac6, (q15_t)0x3360, (q15_t)0x8ac4, (q15_t)0x335a, (q15_t)0x8ac1,
  (q15_t)0x3354, (q15_t)0x8abf, (q15_t)0x334f, (q15_t)0x8abc, (q15_t)0x3349, (q15_t)0x8aba, (q15_t)0x3343, (q15_t)0x8ab7,
  (q15_t)0x333d, (q15_t)0x8ab5, (q15_t)0x3338, (q15_t)0x8ab2, (q15_t)0x3332, (q15_t)0x8ab0, (q15_t)0x332c, (q15_t)0x8aad,
  (q15_t)0x3326, (q15_t)0x8aab, (q15_t)0x3321, (q15_t)0x8aa8, (q15_t)0x331b, (q15_t)0x8aa6, (q15_t)0x3315, (q15_t)0x8aa3,
  (q15_t)0x330f, (q15_t)0x8aa1, (q15_t)0x330a, (q15_t)0x8a9e, (q15_t)0x3304, (q15_t)0x8a9c, (q15_t)0x32fe, (q15_t)0x8a99,
  (q15_t)0x32f8, (q15_t)0x8a97, (q15_t)0x32f3, (q15_t)0x8a94, (q15_t)0x32ed, (q15_t)0x8a92, (q15_t)0x32e7, (q15_t)0x8a8f,
  (q15_t)0x32e1, (q15_t)0x8a8d, (q15_t)0x32db, (q15_t)0x8a8a, (q15_t)0x32d6, (q15_t)0x8a88, (q15_t)0x32d0, (q15_t)0x8a85,
  (q15_t)0x32ca, (q15_t)0x8a83, (q15_t)0x32c4, (q15_t)0x8a80, (q15_t)0x32bf, (q15_t)0x8a7e, (q15_t)0x32b9, (q15_t)0x8a7b,
  (q15_t)0x32b3, (q15_t)0x8a79, (q15_t)0x32ad, (q15_t)0x8a76, (q15_t)0x32a8, (q15_t)0x8a74, (q15_t)0x32a2, (q15_t)0x8a71,
  (q15_t)0x329c, (q15_t)0x8a6f, (q15_t)0x3296, (q15_t)0x8a6c, (q15_t)0x3290, (q15_t)0x8a6a, (q15_t)0x328b, (q15_t)0x8a67,
  (q15_t)0x3285, (q15_t)0x8a65, (q15_t)0x327f, (q15_t)0x8a62, (q15_t)0x3279, (q15_t)0x8a60, (q15_t)0x3274, (q15_t)0x8a5d,
  (q15_t)0x326e, (q15_t)0x8a5b, (q15_t)0x3268, (q15_t)0x8a59, (q15_t)0x3262, (q15_t)0x8a56, (q15_t)0x325d, (q15_t)0x8a54,
  (q15_t)0x3257, (q15_t)0x8a51, (q15_t)0x3251, (q15_t)0x8a4f, (q15_t)0x324b, (q15_t)0x8a4c, (q15_t)0x3245, (q15_t)0x8a4a,
  (q15_t)0x3240, (q15_t)0x8a47, (q15_t)0x323a, (q15_t)0x8a45, (q15_t)0x3234, (q15_t)0x8a42, (q15_t)0x322e, (q15_t)0x8a40,
  (q15_t)0x3228, (q15_t)0x8a3d, (q15_t)0x3223, (q15_t)0x8a3b, (q15_t)0x321d, (q15_t)0x8a38, (q15_t)0x3217, (q15_t)0x8a36,
  (q15_t)0x3211, (q15_t)0x8a34, (q15_t)0x320c, (q15_t)0x8a31, (q15_t)0x3206, (q15_t)0x8a2f, (q15_t)0x3200, (q15_t)0x8a2c,
  (q15_t)0x31fa, (q15_t)0x8a2a, (q15_t)0x31f4, (q15_t)0x8a27, (q15_t)0x31ef, (q15_t)0x8a25, (q15_t)0x31e9, (q15_t)0x8a22,
  (q15_t)0x31e3, (q15_t)0x8a20, (q15_t)0x31dd, (q15_t)0x8a1d, (q15_t)0x31d8, (q15_t)0x8a1b, (q15_t)0x31d2, (q15_t)0x8a19,
  (q15_t)0x31cc, (q15_t)0x8a16, (q15_t)0x31c6, (q15_t)0x8a14, (q15_t)0x31c0, (q15_t)0x8a11, (q15_t)0x31bb, (q15_t)0x8a0f,
  (q15_t)0x31b5, (q15_t)0x8a0c, (q15_t)0x31af, (q15_t)0x8a0a, (q15_t)0x31a9, (q15_t)0x8a07, (q15_t)0x31a3, (q15_t)0x8a05,
  (q15_t)0x319e, (q15_t)0x8a03, (q15_t)0x3198, (q15_t)0x8a00, (q15_t)0x3192, (q15_t)0x89fe, (q15_t)0x318c, (q15_t)0x89fb,
  (q15_t)0x3186, (q15_t)0x89f9, (q15_t)0x3181, (q15_t)0x89f6, (q15_t)0x317b, (q15_t)0x89f4, (q15_t)0x3175, (q15_t)0x89f2,
  (q15_t)0x316f, (q15_t)0x89ef, (q15_t)0x3169, (q15_t)0x89ed, (q15_t)0x3164, (q15_t)0x89ea, (q15_t)0x315e, (q15_t)0x89e8,
  (q15_t)0x3158, (q15_t)0x89e5, (q15_t)0x3152, (q15_t)0x89e3, (q15_t)0x314c, (q15_t)0x89e1, (q15_t)0x3147, (q15_t)0x89de,
  (q15_t)0x3141, (q15_t)0x89dc, (q15_t)0x313b, (q15_t)0x89d9, (q15_t)0x3135, (q15_t)0x89d7, (q15_t)0x312f, (q15_t)0x89d5,
  (q15_t)0x312a, (q15_t)0x89d2, (q15_t)0x3124, (q15_t)0x89d0, (q15_t)0x311e, (q15_t)0x89cd, (q15_t)0x3118, (q15_t)0x89cb,
  (q15_t)0x3112, (q15_t)0x89c8, (q15_t)0x310d, (q15_t)0x89c6, (q15_t)0x3107, (q15_t)0x89c4, (q15_t)0x3101, (q15_t)0x89c1,
  (q15_t)0x30fb, (q15_t)0x89bf, (q15_t)0x30f5, (q15_t)0x89bc, (q15_t)0x30f0, (q15_t)0x89ba, (q15_t)0x30ea, (q15_t)0x89b8,
  (q15_t)0x30e4, (q15_t)0x89b5, (q15_t)0x30de, (q15_t)0x89b3, (q15_t)0x30d8, (q15_t)0x89b0, (q15_t)0x30d3, (q15_t)0x89ae,
  (q15_t)0x30cd, (q15_t)0x89ac, (q15_t)0x30c7, (q15_t)0x89a9, (q15_t)0x30c1, (q15_t)0x89a7, (q15_t)0x30bb, (q15_t)0x89a4,
  (q15_t)0x30b6, (q15_t)0x89a2, (q15_t)0x30b0, (q15_t)0x89a0, (q15_t)0x30aa, (q15_t)0x899d, (q15_t)0x30a4, (q15_t)0x899b,
  (q15_t)0x309e, (q15_t)0x8998, (q15_t)0x3099, (q15_t)0x8996, (q15_t)0x3093, (q15_t)0x8994, (q15_t)0x308d, (q15_t)0x8991,
  (q15_t)0x3087, (q15_t)0x898f, (q15_t)0x3081, (q15_t)0x898d, (q15_t)0x307b, (q15_t)0x898a, (q15_t)0x3076, (q15_t)0x8988,
  (q15_t)0x3070, (q15_t)0x8985, (q15_t)0x306a, (q15_t)0x8983, (q15_t)0x3064, (q15_t)0x8981, (q15_t)0x305e, (q15_t)0x897e,
  (q15_t)0x3059, (q15_t)0x897c, (q15_t)0x3053, (q15_t)0x897a, (q15_t)0x304d, (q15_t)0x8977, (q15_t)0x3047, (q15_t)0x8975,
  (q15_t)0x3041, (q15_t)0x8972, (q15_t)0x303b, (q15_t)0x8970, (q15_t)0x3036, (q15_t)0x896e, (q15_t)0x3030, (q15_t)0x896b,
  (q15_t)0x302a, (q15_t)0x8969, (q15_t)0x3024, (q15_t)0x8967, (q15_t)0x301e, (q15_t)0x8964, (q15_t)0x3019, (q15_t)0x8962,
  (q15_t)0x3013, (q15_t)0x8960, (q15_t)0x300d, (q15_t)0x895d, (q15_t)0x3007, (q15_t)0x895b, (q15_t)0x3001, (q15_t)0x8958,
  (q15_t)0x2ffb, (q15_t)0x8956, (q15_t)0x2ff6, (q15_t)0x8954, (q15_t)0x2ff0, (q15_t)0x8951, (q15_t)0x2fea, (q15_t)0x894f,
  (q15_t)0x2fe4, (q15_t)0x894d, (q15_t)0x2fde, (q15_t)0x894a, (q15_t)0x2fd8, (q15_t)0x8948, (q15_t)0x2fd3, (q15_t)0x8946,
  (q15_t)0x2fcd, (q15_t)0x8943, (q15_t)0x2fc7, (q15_t)0x8941, (q15_t)0x2fc1, (q15_t)0x893f, (q15_t)0x2fbb, (q15_t)0x893c,
  (q15_t)0x2fb5, (q15_t)0x893a, (q15_t)0x2fb0, (q15_t)0x8938, (q15_t)0x2faa, (q15_t)0x8935, (q15_t)0x2fa4, (q15_t)0x8933,
  (q15_t)0x2f9e, (q15_t)0x8931, (q15_t)0x2f98, (q15_t)0x892e, (q15_t)0x2f92, (q15_t)0x892c, (q15_t)0x2f8d, (q15_t)0x892a,
  (q15_t)0x2f87, (q15_t)0x8927, (q15_t)0x2f81, (q15_t)0x8925, (q15_t)0x2f7b, (q15_t)0x8923, (q15_t)0x2f75, (q15_t)0x8920,
  (q15_t)0x2f6f, (q15_t)0x891e, (q15_t)0x2f6a, (q15_t)0x891c, (q15_t)0x2f64, (q15_t)0x8919, (q15_t)0x2f5e, (q15_t)0x8917,
  (q15_t)0x2f58, (q15_t)0x8915, (q15_t)0x2f52, (q15_t)0x8912, (q15_t)0x2f4c, (q15_t)0x8910, (q15_t)0x2f47, (q15_t)0x890e,
  (q15_t)0x2f41, (q15_t)0x890b, (q15_t)0x2f3b, (q15_t)0x8909, (q15_t)0x2f35, (q15_t)0x8907, (q15_t)0x2f2f, (q15_t)0x8904,
  (q15_t)0x2f29, (q15_t)0x8902, (q15_t)0x2f24, (q15_t)0x8900, (q15_t)0x2f1e, (q15_t)0x88fd, (q15_t)0x2f18, (q15_t)0x88fb,
  (q15_t)0x2f12, (q15_t)0x88f9, (q15_t)0x2f0c, (q15_t)0x88f6, (q15_t)0x2f06, (q15_t)0x88f4, (q15_t)0x2f01, (q15_t)0x88f2,
  (q15_t)0x2efb, (q15_t)0x88f0, (q15_t)0x2ef5, (q15_t)0x88ed, (q15_t)0x2eef, (q15_t)0x88eb, (q15_t)0x2ee9, (q15_t)0x88e9,
  (q15_t)0x2ee3, (q15_t)0x88e6, (q15_t)0x2edd, (q15_t)0x88e4, (q15_t)0x2ed8, (q15_t)0x88e2, (q15_t)0x2ed2, (q15_t)0x88df,
  (q15_t)0x2ecc, (q15_t)0x88dd, (q15_t)0x2ec6, (q15_t)0x88db, (q15_t)0x2ec0, (q15_t)0x88d9, (q15_t)0x2eba, (q15_t)0x88d6,
  (q15_t)0x2eb5, (q15_t)0x88d4, (q15_t)0x2eaf, (q15_t)0x88d2, (q15_t)0x2ea9, (q15_t)0x88cf, (q15_t)0x2ea3, (q15_t)0x88cd,
  (q15_t)0x2e9d, (q15_t)0x88cb, (q15_t)0x2e97, (q15_t)0x88c8, (q15_t)0x2e91, (q15_t)0x88c6, (q15_t)0x2e8c, (q15_t)0x88c4,
  (q15_t)0x2e86, (q15_t)0x88c2, (q15_t)0x2e80, (q15_t)0x88bf, (q15_t)0x2e7a, (q15_t)0x88bd, (q15_t)0x2e74, (q15_t)0x88bb,
  (q15_t)0x2e6e, (q15_t)0x88b9, (q15_t)0x2e68, (q15_t)0x88b6, (q15_t)0x2e63, (q15_t)0x88b4, (q15_t)0x2e5d, (q15_t)0x88b2,
  (q15_t)0x2e57, (q15_t)0x88af, (q15_t)0x2e51, (q15_t)0x88ad, (q15_t)0x2e4b, (q15_t)0x88ab, (q15_t)0x2e45, (q15_t)0x88a9,
  (q15_t)0x2e3f, (q15_t)0x88a6, (q15_t)0x2e3a, (q15_t)0x88a4, (q15_t)0x2e34, (q15_t)0x88a2, (q15_t)0x2e2e, (q15_t)0x88a0,
  (q15_t)0x2e28, (q15_t)0x889d, (q15_t)0x2e22, (q15_t)0x889b, (q15_t)0x2e1c, (q15_t)0x8899, (q15_t)0x2e16, (q15_t)0x8896,
  (q15_t)0x2e11, (q15_t)0x8894, (q15_t)0x2e0b, (q15_t)0x8892, (q15_t)0x2e05, (q15_t)0x8890, (q15_t)0x2dff, (q15_t)0x888d,
  (q15_t)0x2df9, (q15_t)0x888b, (q15_t)0x2df3, (q15_t)0x8889, (q15_t)0x2ded, (q15_t)0x8887, (q15_t)0x2de7, (q15_t)0x8884,
  (q15_t)0x2de2, (q15_t)0x8882, (q15_t)0x2ddc, (q15_t)0x8880, (q15_t)0x2dd6, (q15_t)0x887e, (q15_t)0x2dd0, (q15_t)0x887b,
  (q15_t)0x2dca, (q15_t)0x8879, (q15_t)0x2dc4, (q15_t)0x8877, (q15_t)0x2dbe, (q15_t)0x8875, (q15_t)0x2db9, (q15_t)0x8872,
  (q15_t)0x2db3, (q15_t)0x8870, (q15_t)0x2dad, (q15_t)0x886e, (q15_t)0x2da7, (q15_t)0x886c, (q15_t)0x2da1, (q15_t)0x8869,
  (q15_t)0x2d9b, (q15_t)0x8867, (q15_t)0x2d95, (q15_t)0x8865, (q15_t)0x2d8f, (q15_t)0x8863, (q15_t)0x2d8a, (q15_t)0x8860,
  (q15_t)0x2d84, (q15_t)0x885e, (q15_t)0x2d7e, (q15_t)0x885c, (q15_t)0x2d78, (q15_t)0x885a, (q15_t)0x2d72, (q15_t)0x8858,
  (q15_t)0x2d6c, (q15_t)0x8855, (q15_t)0x2d66, (q15_t)0x8853, (q15_t)0x2d60, (q15_t)0x8851, (q15_t)0x2d5b, (q15_t)0x884f,
  (q15_t)0x2d55, (q15_t)0x884c, (q15_t)0x2d4f, (q15_t)0x884a, (q15_t)0x2d49, (q15_t)0x8848, (q15_t)0x2d43, (q15_t)0x8846,
  (q15_t)0x2d3d, (q15_t)0x8844, (q15_t)0x2d37, (q15_t)0x8841, (q15_t)0x2d31, (q15_t)0x883f, (q15_t)0x2d2c, (q15_t)0x883d,
  (q15_t)0x2d26, (q15_t)0x883b, (q15_t)0x2d20, (q15_t)0x8838, (q15_t)0x2d1a, (q15_t)0x8836, (q15_t)0x2d14, (q15_t)0x8834,
  (q15_t)0x2d0e, (q15_t)0x8832, (q15_t)0x2d08, (q15_t)0x8830, (q15_t)0x2d02, (q15_t)0x882d, (q15_t)0x2cfd, (q15_t)0x882b,
  (q15_t)0x2cf7, (q15_t)0x8829, (q15_t)0x2cf1, (q15_t)0x8827, (q15_t)0x2ceb, (q15_t)0x8825, (q15_t)0x2ce5, (q15_t)0x8822,
  (q15_t)0x2cdf, (q15_t)0x8820, (q15_t)0x2cd9, (q15_t)0x881e, (q15_t)0x2cd3, (q15_t)0x881c, (q15_t)0x2ccd, (q15_t)0x881a,
  (q15_t)0x2cc8, (q15_t)0x8817, (q15_t)0x2cc2, (q15_t)0x8815, (q15_t)0x2cbc, (q15_t)0x8813, (q15_t)0x2cb6, (q15_t)0x8811,
  (q15_t)0x2cb0, (q15_t)0x880f, (q15_t)0x2caa, (q15_t)0x880c, (q15_t)0x2ca4, (q15_t)0x880a, (q15_t)0x2c9e, (q15_t)0x8808,
  (q15_t)0x2c98, (q15_t)0x8806, (q15_t)0x2c93, (q15_t)0x8804, (q15_t)0x2c8d, (q15_t)0x8801, (q15_t)0x2c87, (q15_t)0x87ff,
  (q15_t)0x2c81, (q15_t)0x87fd, (q15_t)0x2c7b, (q15_t)0x87fb, (q15_t)0x2c75, (q15_t)0x87f9, (q15_t)0x2c6f, (q15_t)0x87f6,
  (q15_t)0x2c69, (q15_t)0x87f4, (q15_t)0x2c63, (q15_t)0x87f2, (q15_t)0x2c5e, (q15_t)0x87f0, (q15_t)0x2c58, (q15_t)0x87ee,
  (q15_t)0x2c52, (q15_t)0x87ec, (q15_t)0x2c4c, (q15_t)0x87e9, (q15_t)0x2c46, (q15_t)0x87e7, (q15_t)0x2c40, (q15_t)0x87e5,
  (q15_t)0x2c3a, (q15_t)0x87e3, (q15_t)0x2c34, (q15_t)0x87e1, (q15_t)0x2c2e, (q15_t)0x87df, (q15_t)0x2c29, (q15_t)0x87dc,
  (q15_t)0x2c23, (q15_t)0x87da, (q15_t)0x2c1d, (q15_t)0x87d8, (q15_t)0x2c17, (q15_t)0x87d6, (q15_t)0x2c11, (q15_t)0x87d4,
  (q15_t)0x2c0b, (q15_t)0x87d2, (q15_t)0x2c05, (q15_t)0x87cf, (q15_t)0x2bff, (q15_t)0x87cd, (q15_t)0x2bf9, (q15_t)0x87cb,
  (q15_t)0x2bf3, (q15_t)0x87c9, (q15_t)0x2bee, (q15_t)0x87c7, (q15_t)0x2be8, (q15_t)0x87c5, (q15_t)0x2be2, (q15_t)0x87c2,
  (q15_t)0x2bdc, (q15_t)0x87c0, (q15_t)0x2bd6, (q15_t)0x87be, (q15_t)0x2bd0, (q15_t)0x87bc, (q15_t)0x2bca, (q15_t)0x87ba,
  (q15_t)0x2bc4, (q15_t)0x87b8, (q15_t)0x2bbe, (q15_t)0x87b6, (q15_t)0x2bb8, (q15_t)0x87b3, (q15_t)0x2bb2, (q15_t)0x87b1,
  (q15_t)0x2bad, (q15_t)0x87af, (q15_t)0x2ba7, (q15_t)0x87ad, (q15_t)0x2ba1, (q15_t)0x87ab, (q15_t)0x2b9b, (q15_t)0x87a9,
  (q15_t)0x2b95, (q15_t)0x87a7, (q15_t)0x2b8f, (q15_t)0x87a4, (q15_t)0x2b89, (q15_t)0x87a2, (q15_t)0x2b83, (q15_t)0x87a0,
  (q15_t)0x2b7d, (q15_t)0x879e, (q15_t)0x2b77, (q15_t)0x879c, (q15_t)0x2b71, (q15_t)0x879a, (q15_t)0x2b6c, (q15_t)0x8798,
  (q15_t)0x2b66, (q15_t)0x8795, (q15_t)0x2b60, (q15_t)0x8793, (q15_t)0x2b5a, (q15_t)0x8791, (q15_t)0x2b54, (q15_t)0x878f,
  (q15_t)0x2b4e, (q15_t)0x878d, (q15_t)0x2b48, (q15_t)0x878b, (q15_t)0x2b42, (q15_t)0x8789, (q15_t)0x2b3c, (q15_t)0x8787,
  (q15_t)0x2b36, (q15_t)0x8784, (q15_t)0x2b30, (q15_t)0x8782, (q15_t)0x2b2b, (q15_t)0x8780, (q15_t)0x2b25, (q15_t)0x877e,
  (q15_t)0x2b1f, (q15_t)0x877c, (q15_t)0x2b19, (q15_t)0x877a, (q15_t)0x2b13, (q15_t)0x8778, (q15_t)0x2b0d, (q15_t)0x8776,
  (q15_t)0x2b07, (q15_t)0x8774, (q15_t)0x2b01, (q15_t)0x8771, (q15_t)0x2afb, (q15_t)0x876f, (q15_t)0x2af5, (q15_t)0x876d,
  (q15_t)0x2aef, (q15_t)0x876b, (q15_t)0x2ae9, (q15_t)0x8769, (q15_t)0x2ae4, (q15_t)0x8767, (q15_t)0x2ade, (q15_t)0x8765,
  (q15_t)0x2ad8, (q15_t)0x8763, (q15_t)0x2ad2, (q15_t)0x8761, (q15_t)0x2acc, (q15_t)0x875e, (q15_t)0x2ac6, (q15_t)0x875c,
  (q15_t)0x2ac0, (q15_t)0x875a, (q15_t)0x2aba, (q15_t)0x8758, (q15_t)0x2ab4, (q15_t)0x8756, (q15_t)0x2aae, (q15_t)0x8754,
  (q15_t)0x2aa8, (q15_t)0x8752, (q15_t)0x2aa2, (q15_t)0x8750, (q15_t)0x2a9c, (q15_t)0x874e, (q15_t)0x2a97, (q15_t)0x874c,
  (q15_t)0x2a91, (q15_t)0x874a, (q15_t)0x2a8b, (q15_t)0x8747, (q15_t)0x2a85, (q15_t)0x8745, (q15_t)0x2a7f, (q15_t)0x8743,
  (q15_t)0x2a79, (q15_t)0x8741, (q15_t)0x2a73, (q15_t)0x873f, (q15_t)0x2a6d, (q15_t)0x873d, (q15_t)0x2a67, (q15_t)0x873b,
  (q15_t)0x2a61, (q15_t)0x8739, (q15_t)0x2a5b, (q15_t)0x8737, (q15_t)0x2a55, (q15_t)0x8735, (q15_t)0x2a4f, (q15_t)0x8733,
  (q15_t)0x2a49, (q15_t)0x8731, (q15_t)0x2a44, (q15_t)0x872e, (q15_t)0x2a3e, (q15_t)0x872c, (q15_t)0x2a38, (q15_t)0x872a,
  (q15_t)0x2a32, (q15_t)0x8728, (q15_t)0x2a2c, (q15_t)0x8726, (q15_t)0x2a26, (q15_t)0x8724, (q15_t)0x2a20, (q15_t)0x8722,
  (q15_t)0x2a1a, (q15_t)0x8720, (q15_t)0x2a14, (q15_t)0x871e, (q15_t)0x2a0e, (q15_t)0x871c, (q15_t)0x2a08, (q15_t)0x871a,
  (q15_t)0x2a02, (q15_t)0x8718, (q15_t)0x29fc, (q15_t)0x8716, (q15_t)0x29f6, (q15_t)0x8714, (q15_t)0x29f0, (q15_t)0x8712,
  (q15_t)0x29eb, (q15_t)0x870f, (q15_t)0x29e5, (q15_t)0x870d, (q15_t)0x29df, (q15_t)0x870b, (q15_t)0x29d9, (q15_t)0x8709,
  (q15_t)0x29d3, (q15_t)0x8707, (q15_t)0x29cd, (q15_t)0x8705, (q15_t)0x29c7, (q15_t)0x8703, (q15_t)0x29c1, (q15_t)0x8701,
  (q15_t)0x29bb, (q15_t)0x86ff, (q15_t)0x29b5, (q15_t)0x86fd, (q15_t)0x29af, (q15_t)0x86fb, (q15_t)0x29a9, (q15_t)0x86f9,
  (q15_t)0x29a3, (q15_t)0x86f7, (q15_t)0x299d, (q15_t)0x86f5, (q15_t)0x2997, (q15_t)0x86f3, (q15_t)0x2991, (q15_t)0x86f1,
  (q15_t)0x298b, (q15_t)0x86ef, (q15_t)0x2986, (q15_t)0x86ed, (q15_t)0x2980, (q15_t)0x86eb, (q15_t)0x297a, (q15_t)0x86e9,
  (q15_t)0x2974, (q15_t)0x86e7, (q15_t)0x296e, (q15_t)0x86e4, (q15_t)0x2968, (q15_t)0x86e2, (q15_t)0x2962, (q15_t)0x86e0,
  (q15_t)0x295c, (q15_t)0x86de, (q15_t)0x2956, (q15_t)0x86dc, (q15_t)0x2950, (q15_t)0x86da, (q15_t)0x294a, (q15_t)0x86d8,
  (q15_t)0x2944, (q15_t)0x86d6, (q15_t)0x293e, (q15_t)0x86d4, (q15_t)0x2938, (q15_t)0x86d2, (q15_t)0x2932, (q15_t)0x86d0,
  (q15_t)0x292c, (q15_t)0x86ce, (q15_t)0x2926, (q15_t)0x86cc, (q15_t)0x2920, (q15_t)0x86ca, (q15_t)0x291b, (q15_t)0x86c8,
  (q15_t)0x2915, (q15_t)0x86c6, (q15_t)0x290f, (q15_t)0x86c4, (q15_t)0x2909, (q15_t)0x86c2, (q15_t)0x2903, (q15_t)0x86c0,
  (q15_t)0x28fd, (q15_t)0x86be, (q15_t)0x28f7, (q15_t)0x86bc, (q15_t)0x28f1, (q15_t)0x86ba, (q15_t)0x28eb, (q15_t)0x86b8,
  (q15_t)0x28e5, (q15_t)0x86b6, (q15_t)0x28df, (q15_t)0x86b4, (q15_t)0x28d9, (q15_t)0x86b2, (q15_t)0x28d3, (q15_t)0x86b0,
  (q15_t)0x28cd, (q15_t)0x86ae, (q15_t)0x28c7, (q15_t)0x86ac, (q15_t)0x28c1, (q15_t)0x86aa, (q15_t)0x28bb, (q15_t)0x86a8,
  (q15_t)0x28b5, (q15_t)0x86a6, (q15_t)0x28af, (q15_t)0x86a4, (q15_t)0x28a9, (q15_t)0x86a2, (q15_t)0x28a3, (q15_t)0x86a0,
  (q15_t)0x289d, (q15_t)0x869e, (q15_t)0x2898, (q15_t)0x869c, (q15_t)0x2892, (q15_t)0x869a, (q15_t)0x288c, (q15_t)0x8698,
  (q15_t)0x2886, (q15_t)0x8696, (q15_t)0x2880, (q15_t)0x8694, (q15_t)0x287a, (q15_t)0x8692, (q15_t)0x2874, (q15_t)0x8690,
  (q15_t)0x286e, (q15_t)0x868e, (q15_t)0x2868, (q15_t)0x868c, (q15_t)0x2862, (q15_t)0x868a, (q15_t)0x285c, (q15_t)0x8688,
  (q15_t)0x2856, (q15_t)0x8686, (q15_t)0x2850, (q15_t)0x8684, (q15_t)0x284a, (q15_t)0x8682, (q15_t)0x2844, (q15_t)0x8680,
  (q15_t)0x283e, (q15_t)0x867e, (q15_t)0x2838, (q15_t)0x867c, (q15_t)0x2832, (q15_t)0x867a, (q15_t)0x282c, (q15_t)0x8678,
  (q15_t)0x2826, (q15_t)0x8676, (q15_t)0x2820, (q15_t)0x8674, (q15_t)0x281a, (q15_t)0x8672, (q15_t)0x2814, (q15_t)0x8670,
  (q15_t)0x280e, (q15_t)0x866e, (q15_t)0x2808, (q15_t)0x866d, (q15_t)0x2802, (q15_t)0x866b, (q15_t)0x27fc, (q15_t)0x8669,
  (q15_t)0x27f6, (q15_t)0x8667, (q15_t)0x27f1, (q15_t)0x8665, (q15_t)0x27eb, (q15_t)0x8663, (q15_t)0x27e5, (q15_t)0x8661,
  (q15_t)0x27df, (q15_t)0x865f, (q15_t)0x27d9, (q15_t)0x865d, (q15_t)0x27d3, (q15_t)0x865b, (q15_t)0x27cd, (q15_t)0x8659,
  (q15_t)0x27c7, (q15_t)0x8657, (q15_t)0x27c1, (q15_t)0x8655, (q15_t)0x27bb, (q15_t)0x8653, (q15_t)0x27b5, (q15_t)0x8651,
  (q15_t)0x27af, (q15_t)0x864f, (q15_t)0x27a9, (q15_t)0x864d, (q15_t)0x27a3, (q15_t)0x864b, (q15_t)0x279d, (q15_t)0x8649,
  (q15_t)0x2797, (q15_t)0x8647, (q15_t)0x2791, (q15_t)0x8645, (q15_t)0x278b, (q15_t)0x8644, (q15_t)0x2785, (q15_t)0x8642,
  (q15_t)0x277f, (q15_t)0x8640, (q15_t)0x2779, (q15_t)0x863e, (q15_t)0x2773, (q15_t)0x863c, (q15_t)0x276d, (q15_t)0x863a,
  (q15_t)0x2767, (q15_t)0x8638, (q15_t)0x2761, (q15_t)0x8636, (q15_t)0x275b, (q15_t)0x8634, (q15_t)0x2755, (q15_t)0x8632,
  (q15_t)0x274f, (q15_t)0x8630, (q15_t)0x2749, (q15_t)0x862e, (q15_t)0x2743, (q15_t)0x862c, (q15_t)0x273d, (q15_t)0x862a,
  (q15_t)0x2737, (q15_t)0x8628, (q15_t)0x2731, (q15_t)0x8627, (q15_t)0x272b, (q15_t)0x8625, (q15_t)0x2725, (q15_t)0x8623,
  (q15_t)0x271f, (q15_t)0x8621, (q15_t)0x2719, (q15_t)0x861f, (q15_t)0x2713, (q15_t)0x861d, (q15_t)0x270d, (q15_t)0x861b,
  (q15_t)0x2707, (q15_t)0x8619, (q15_t)0x2701, (q15_t)0x8617, (q15_t)0x26fb, (q15_t)0x8615, (q15_t)0x26f5, (q15_t)0x8613,
  (q15_t)0x26ef, (q15_t)0x8611, (q15_t)0x26e9, (q15_t)0x8610, (q15_t)0x26e4, (q15_t)0x860e, (q15_t)0x26de, (q15_t)0x860c,
  (q15_t)0x26d8, (q15_t)0x860a, (q15_t)0x26d2, (q15_t)0x8608, (q15_t)0x26cc, (q15_t)0x8606, (q15_t)0x26c6, (q15_t)0x8604,
  (q15_t)0x26c0, (q15_t)0x8602, (q15_t)0x26ba, (q15_t)0x8600, (q15_t)0x26b4, (q15_t)0x85fe, (q15_t)0x26ae, (q15_t)0x85fc,
  (q15_t)0x26a8, (q15_t)0x85fb, (q15_t)0x26a2, (q15_t)0x85f9, (q15_t)0x269c, (q15_t)0x85f7, (q15_t)0x2696, (q15_t)0x85f5,
  (q15_t)0x2690, (q15_t)0x85f3, (q15_t)0x268a, (q15_t)0x85f1, (q15_t)0x2684, (q15_t)0x85ef, (q15_t)0x267e, (q15_t)0x85ed,
  (q15_t)0x2678, (q15_t)0x85eb, (q15_t)0x2672, (q15_t)0x85ea, (q15_t)0x266c, (q15_t)0x85e8, (q15_t)0x2666, (q15_t)0x85e6,
  (q15_t)0x2660, (q15_t)0x85e4, (q15_t)0x265a, (q15_t)0x85e2, (q15_t)0x2654, (q15_t)0x85e0, (q15_t)0x264e, (q15_t)0x85de,
  (q15_t)0x2648, (q15_t)0x85dc, (q15_t)0x2642, (q15_t)0x85da, (q15_t)0x263c, (q15_t)0x85d9, (q15_t)0x2636, (q15_t)0x85d7,
  (q15_t)0x2630, (q15_t)0x85d5, (q15_t)0x262a, (q15_t)0x85d3, (q15_t)0x2624, (q15_t)0x85d1, (q15_t)0x261e, (q15_t)0x85cf,
  (q15_t)0x2618, (q15_t)0x85cd, (q15_t)0x2612, (q15_t)0x85cb, (q15_t)0x260c, (q15_t)0x85ca, (q15_t)0x2606, (q15_t)0x85c8,
  (q15_t)0x2600, (q15_t)0x85c6, (q15_t)0x25fa, (q15_t)0x85c4, (q15_t)0x25f4, (q15_t)0x85c2, (q15_t)0x25ee, (q15_t)0x85c0,
  (q15_t)0x25e8, (q15_t)0x85be, (q15_t)0x25e2, (q15_t)0x85bd, (q15_t)0x25dc, (q15_t)0x85bb, (q15_t)0x25d6, (q15_t)0x85b9,
  (q15_t)0x25d0, (q15_t)0x85b7, (q15_t)0x25ca, (q15_t)0x85b5, (q15_t)0x25c4, (q15_t)0x85b3, (q15_t)0x25be, (q15_t)0x85b1,
  (q15_t)0x25b8, (q15_t)0x85b0, (q15_t)0x25b2, (q15_t)0x85ae, (q15_t)0x25ac, (q15_t)0x85ac, (q15_t)0x25a6, (q15_t)0x85aa,
  (q15_t)0x25a0, (q15_t)0x85a8, (q15_t)0x259a, (q15_t)0x85a6, (q15_t)0x2594, (q15_t)0x85a4, (q15_t)0x258e, (q15_t)0x85a3,
  (q15_t)0x2588, (q15_t)0x85a1, (q15_t)0x2582, (q15_t)0x859f, (q15_t)0x257c, (q15_t)0x859d, (q15_t)0x2576, (q15_t)0x859b,
  (q15_t)0x2570, (q15_t)0x8599, (q15_t)0x256a, (q15_t)0x8598, (q15_t)0x2564, (q15_t)0x8596, (q15_t)0x255e, (q15_t)0x8594,
  (q15_t)0x2558, (q15_t)0x8592, (q15_t)0x2552, (q15_t)0x8590, (q15_t)0x254c, (q15_t)0x858e, (q15_t)0x2546, (q15_t)0x858d,
  (q15_t)0x2540, (q15_t)0x858b, (q15_t)0x253a, (q15_t)0x8589, (q15_t)0x2534, (q15_t)0x8587, (q15_t)0x252e, (q15_t)0x8585,
  (q15_t)0x2528, (q15_t)0x8583, (q15_t)0x2522, (q15_t)0x8582, (q15_t)0x251c, (q15_t)0x8580, (q15_t)0x2516, (q15_t)0x857e,
  (q15_t)0x250f, (q15_t)0x857c, (q15_t)0x2509, (q15_t)0x857a, (q15_t)0x2503, (q15_t)0x8579, (q15_t)0x24fd, (q15_t)0x8577,
  (q15_t)0x24f7, (q15_t)0x8575, (q15_t)0x24f1, (q15_t)0x8573, (q15_t)0x24eb, (q15_t)0x8571, (q15_t)0x24e5, (q15_t)0x856f,
  (q15_t)0x24df, (q15_t)0x856e, (q15_t)0x24d9, (q15_t)0x856c, (q15_t)0x24d3, (q15_t)0x856a, (q15_t)0x24cd, (q15_t)0x8568,
  (q15_t)0x24c7, (q15_t)0x8566, (q15_t)0x24c1, (q15_t)0x8565, (q15_t)0x24bb, (q15_t)0x8563, (q15_t)0x24b5, (q15_t)0x8561,
  (q15_t)0x24af, (q15_t)0x855f, (q15_t)0x24a9, (q15_t)0x855d, (q15_t)0x24a3, (q15_t)0x855c, (q15_t)0x249d, (q15_t)0x855a,
  (q15_t)0x2497, (q15_t)0x8558, (q15_t)0x2491, (q15_t)0x8556, (q15_t)0x248b, (q15_t)0x8554, (q15_t)0x2485, (q15_t)0x8553,
  (q15_t)0x247f, (q15_t)0x8551, (q15_t)0x2479, (q15_t)0x854f, (q15_t)0x2473, (q15_t)0x854d, (q15_t)0x246d, (q15_t)0x854b,
  (q15_t)0x2467, (q15_t)0x854a, (q15_t)0x2461, (q15_t)0x8548, (q15_t)0x245b, (q15_t)0x8546, (q15_t)0x2455, (q15_t)0x8544,
  (q15_t)0x244f, (q15_t)0x8543, (q15_t)0x2449, (q15_t)0x8541, (q15_t)0x2443, (q15_t)0x853f, (q15_t)0x243d, (q15_t)0x853d,
  (q15_t)0x2437, (q15_t)0x853b, (q15_t)0x2431, (q15_t)0x853a, (q15_t)0x242b, (q15_t)0x8538, (q15_t)0x2425, (q15_t)0x8536,
  (q15_t)0x241f, (q15_t)0x8534, (q15_t)0x2419, (q15_t)0x8533, (q15_t)0x2413, (q15_t)0x8531, (q15_t)0x240d, (q15_t)0x852f,
  (q15_t)0x2407, (q15_t)0x852d, (q15_t)0x2401, (q15_t)0x852b, (q15_t)0x23fa, (q15_t)0x852a, (q15_t)0x23f4, (q15_t)0x8528,
  (q15_t)0x23ee, (q15_t)0x8526, (q15_t)0x23e8, (q15_t)0x8524, (q15_t)0x23e2, (q15_t)0x8523, (q15_t)0x23dc, (q15_t)0x8521,
  (q15_t)0x23d6, (q15_t)0x851f, (q15_t)0x23d0, (q15_t)0x851d, (q15_t)0x23ca, (q15_t)0x851c, (q15_t)0x23c4, (q15_t)0x851a,
  (q15_t)0x23be, (q15_t)0x8518, (q15_t)0x23b8, (q15_t)0x8516, (q15_t)0x23b2, (q15_t)0x8515, (q15_t)0x23ac, (q15_t)0x8513,
  (q15_t)0x23a6, (q15_t)0x8511, (q15_t)0x23a0, (q15_t)0x850f, (q15_t)0x239a, (q15_t)0x850e, (q15_t)0x2394, (q15_t)0x850c,
  (q15_t)0x238e, (q15_t)0x850a, (q15_t)0x2388, (q15_t)0x8508, (q15_t)0x2382, (q15_t)0x8507, (q15_t)0x237c, (q15_t)0x8505,
  (q15_t)0x2376, (q15_t)0x8503, (q15_t)0x2370, (q15_t)0x8501, (q15_t)0x236a, (q15_t)0x8500, (q15_t)0x2364, (q15_t)0x84fe,
  (q15_t)0x235e, (q15_t)0x84fc, (q15_t)0x2358, (q15_t)0x84fa, (q15_t)0x2352, (q15_t)0x84f9, (q15_t)0x234b, (q15_t)0x84f7,
  (q15_t)0x2345, (q15_t)0x84f5, (q15_t)0x233f, (q15_t)0x84f4, (q15_t)0x2339, (q15_t)0x84f2, (q15_t)0x2333, (q15_t)0x84f0,
  (q15_t)0x232d, (q15_t)0x84ee, (q15_t)0x2327, (q15_t)0x84ed, (q15_t)0x2321, (q15_t)0x84eb, (q15_t)0x231b, (q15_t)0x84e9,
  (q15_t)0x2315, (q15_t)0x84e7, (q15_t)0x230f, (q15_t)0x84e6, (q15_t)0x2309, (q15_t)0x84e4, (q15_t)0x2303, (q15_t)0x84e2,
  (q15_t)0x22fd, (q15_t)0x84e1, (q15_t)0x22f7, (q15_t)0x84df, (q15_t)0x22f1, (q15_t)0x84dd, (q15_t)0x22eb, (q15_t)0x84db,
  (q15_t)0x22e5, (q15_t)0x84da, (q15_t)0x22df, (q15_t)0x84d8, (q15_t)0x22d9, (q15_t)0x84d6, (q15_t)0x22d3, (q15_t)0x84d5,
  (q15_t)0x22cd, (q15_t)0x84d3, (q15_t)0x22c7, (q15_t)0x84d1, (q15_t)0x22c0, (q15_t)0x84cf, (q15_t)0x22ba, (q15_t)0x84ce,
  (q15_t)0x22b4, (q15_t)0x84cc, (q15_t)0x22ae, (q15_t)0x84ca, (q15_t)0x22a8, (q15_t)0x84c9, (q15_t)0x22a2, (q15_t)0x84c7,
  (q15_t)0x229c, (q15_t)0x84c5, (q15_t)0x2296, (q15_t)0x84c4, (q15_t)0x2290, (q15_t)0x84c2, (q15_t)0x228a, (q15_t)0x84c0,
  (q15_t)0x2284, (q15_t)0x84be, (q15_t)0x227e, (q15_t)0x84bd, (q15_t)0x2278, (q15_t)0x84bb, (q15_t)0x2272, (q15_t)0x84b9,
  (q15_t)0x226c, (q15_t)0x84b8, (q15_t)0x2266, (q15_t)0x84b6, (q15_t)0x2260, (q15_t)0x84b4, (q15_t)0x225a, (q15_t)0x84b3,
  (q15_t)0x2254, (q15_t)0x84b1, (q15_t)0x224e, (q15_t)0x84af, (q15_t)0x2247, (q15_t)0x84ae, (q15_t)0x2241, (q15_t)0x84ac,
  (q15_t)0x223b, (q15_t)0x84aa, (q15_t)0x2235, (q15_t)0x84a9, (q15_t)0x222f, (q15_t)0x84a7, (q15_t)0x2229, (q15_t)0x84a5,
  (q15_t)0x2223, (q15_t)0x84a3, (q15_t)0x221d, (q15_t)0x84a2, (q15_t)0x2217, (q15_t)0x84a0, (q15_t)0x2211, (q15_t)0x849e,
  (q15_t)0x220b, (q15_t)0x849d, (q15_t)0x2205, (q15_t)0x849b, (q15_t)0x21ff, (q15_t)0x8499, (q15_t)0x21f9, (q15_t)0x8498,
  (q15_t)0x21f3, (q15_t)0x8496, (q15_t)0x21ed, (q15_t)0x8494, (q15_t)0x21e7, (q15_t)0x8493, (q15_t)0x21e1, (q15_t)0x8491,
  (q15_t)0x21da, (q15_t)0x848f, (q15_t)0x21d4, (q15_t)0x848e, (q15_t)0x21ce, (q15_t)0x848c, (q15_t)0x21c8, (q15_t)0x848a,
  (q15_t)0x21c2, (q15_t)0x8489, (q15_t)0x21bc, (q15_t)0x8487, (q15_t)0x21b6, (q15_t)0x8486, (q15_t)0x21b0, (q15_t)0x8484,
  (q15_t)0x21aa, (q15_t)0x8482, (q15_t)0x21a4, (q15_t)0x8481, (q15_t)0x219e, (q15_t)0x847f, (q15_t)0x2198, (q15_t)0x847d,
  (q15_t)0x2192, (q15_t)0x847c, (q15_t)0x218c, (q15_t)0x847a, (q15_t)0x2186, (q15_t)0x8478, (q15_t)0x2180, (q15_t)0x8477,
  (q15_t)0x2179, (q15_t)0x8475, (q15_t)0x2173, (q15_t)0x8473, (q15_t)0x216d, (q15_t)0x8472, (q15_t)0x2167, (q15_t)0x8470,
  (q15_t)0x2161, (q15_t)0x846e, (q15_t)0x215b, (q15_t)0x846d, (q15_t)0x2155, (q15_t)0x846b, (q15_t)0x214f, (q15_t)0x846a,
  (q15_t)0x2149, (q15_t)0x8468, (q15_t)0x2143, (q15_t)0x8466, (q15_t)0x213d, (q15_t)0x8465, (q15_t)0x2137, (q15_t)0x8463,
  (q15_t)0x2131, (q15_t)0x8461, (q15_t)0x212b, (q15_t)0x8460, (q15_t)0x2125, (q15_t)0x845e, (q15_t)0x211e, (q15_t)0x845d,
  (q15_t)0x2118, (q15_t)0x845b, (q15_t)0x2112, (q15_t)0x8459, (q15_t)0x210c, (q15_t)0x8458, (q15_t)0x2106, (q15_t)0x8456,
  (q15_t)0x2100, (q15_t)0x8454, (q15_t)0x20fa, (q15_t)0x8453, (q15_t)0x20f4, (q15_t)0x8451, (q15_t)0x20ee, (q15_t)0x8450,
  (q15_t)0x20e8, (q15_t)0x844e, (q15_t)0x20e2, (q15_t)0x844c, (q15_t)0x20dc, (q15_t)0x844b, (q15_t)0x20d6, (q15_t)0x8449,
  (q15_t)0x20d0, (q15_t)0x8447, (q15_t)0x20c9, (q15_t)0x8446, (q15_t)0x20c3, (q15_t)0x8444, (q15_t)0x20bd, (q15_t)0x8443,
  (q15_t)0x20b7, (q15_t)0x8441, (q15_t)0x20b1, (q15_t)0x843f, (q15_t)0x20ab, (q15_t)0x843e, (q15_t)0x20a5, (q15_t)0x843c,
  (q15_t)0x209f, (q15_t)0x843b, (q15_t)0x2099, (q15_t)0x8439, (q15_t)0x2093, (q15_t)0x8437, (q15_t)0x208d, (q15_t)0x8436,
  (q15_t)0x2087, (q15_t)0x8434, (q15_t)0x2081, (q15_t)0x8433, (q15_t)0x207a, (q15_t)0x8431, (q15_t)0x2074, (q15_t)0x842f,
  (q15_t)0x206e, (q15_t)0x842e, (q15_t)0x2068, (q15_t)0x842c, (q15_t)0x2062, (q15_t)0x842b, (q15_t)0x205c, (q15_t)0x8429,
  (q15_t)0x2056, (q15_t)0x8427, (q15_t)0x2050, (q15_t)0x8426, (q15_t)0x204a, (q15_t)0x8424, (q15_t)0x2044, (q15_t)0x8423,
  (q15_t)0x203e, (q15_t)0x8421, (q15_t)0x2038, (q15_t)0x8420, (q15_t)0x2032, (q15_t)0x841e, (q15_t)0x202b, (q15_t)0x841c,
  (q15_t)0x2025, (q15_t)0x841b, (q15_t)0x201f, (q15_t)0x8419, (q15_t)0x2019, (q15_t)0x8418, (q15_t)0x2013, (q15_t)0x8416,
  (q15_t)0x200d, (q15_t)0x8415, (q15_t)0x2007, (q15_t)0x8413, (q15_t)0x2001, (q15_t)0x8411, (q15_t)0x1ffb, (q15_t)0x8410,
  (q15_t)0x1ff5, (q15_t)0x840e, (q15_t)0x1fef, (q15_t)0x840d, (q15_t)0x1fe9, (q15_t)0x840b, (q15_t)0x1fe2, (q15_t)0x840a,
  (q15_t)0x1fdc, (q15_t)0x8408, (q15_t)0x1fd6, (q15_t)0x8406, (q15_t)0x1fd0, (q15_t)0x8405, (q15_t)0x1fca, (q15_t)0x8403,
  (q15_t)0x1fc4, (q15_t)0x8402, (q15_t)0x1fbe, (q15_t)0x8400, (q15_t)0x1fb8, (q15_t)0x83ff, (q15_t)0x1fb2, (q15_t)0x83fd,
  (q15_t)0x1fac, (q15_t)0x83fb, (q15_t)0x1fa6, (q15_t)0x83fa, (q15_t)0x1f9f, (q15_t)0x83f8, (q15_t)0x1f99, (q15_t)0x83f7,
  (q15_t)0x1f93, (q15_t)0x83f5, (q15_t)0x1f8d, (q15_t)0x83f4, (q15_t)0x1f87, (q15_t)0x83f2, (q15_t)0x1f81, (q15_t)0x83f1,
  (q15_t)0x1f7b, (q15_t)0x83ef, (q15_t)0x1f75, (q15_t)0x83ee, (q15_t)0x1f6f, (q15_t)0x83ec, (q15_t)0x1f69, (q15_t)0x83ea,
  (q15_t)0x1f63, (q15_t)0x83e9, (q15_t)0x1f5d, (q15_t)0x83e7, (q15_t)0x1f56, (q15_t)0x83e6, (q15_t)0x1f50, (q15_t)0x83e4,
  (q15_t)0x1f4a, (q15_t)0x83e3, (q15_t)0x1f44, (q15_t)0x83e1, (q15_t)0x1f3e, (q15_t)0x83e0, (q15_t)0x1f38, (q15_t)0x83de,
  (q15_t)0x1f32, (q15_t)0x83dd, (q15_t)0x1f2c, (q15_t)0x83db, (q15_t)0x1f26, (q15_t)0x83da, (q15_t)0x1f20, (q15_t)0x83d8,
  (q15_t)0x1f19, (q15_t)0x83d7, (q15_t)0x1f13, (q15_t)0x83d5, (q15_t)0x1f0d, (q15_t)0x83d3, (q15_t)0x1f07, (q15_t)0x83d2,
  (q15_t)0x1f01, (q15_t)0x83d0, (q15_t)0x1efb, (q15_t)0x83cf, (q15_t)0x1ef5, (q15_t)0x83cd, (q15_t)0x1eef, (q15_t)0x83cc,
  (q15_t)0x1ee9, (q15_t)0x83ca, (q15_t)0x1ee3, (q15_t)0x83c9, (q15_t)0x1edd, (q15_t)0x83c7, (q15_t)0x1ed6, (q15_t)0x83c6,
  (q15_t)0x1ed0, (q15_t)0x83c4, (q15_t)0x1eca, (q15_t)0x83c3, (q15_t)0x1ec4, (q15_t)0x83c1, (q15_t)0x1ebe, (q15_t)0x83c0,
  (q15_t)0x1eb8, (q15_t)0x83be, (q15_t)0x1eb2, (q15_t)0x83bd, (q15_t)0x1eac, (q15_t)0x83bb, (q15_t)0x1ea6, (q15_t)0x83ba,
  (q15_t)0x1ea0, (q15_t)0x83b8, (q15_t)0x1e99, (q15_t)0x83b7, (q15_t)0x1e93, (q15_t)0x83b5, (q15_t)0x1e8d, (q15_t)0x83b4,
  (q15_t)0x1e87, (q15_t)0x83b2, (q15_t)0x1e81, (q15_t)0x83b1, (q15_t)0x1e7b, (q15_t)0x83af, (q15_t)0x1e75, (q15_t)0x83ae,
  (q15_t)0x1e6f, (q15_t)0x83ac, (q15_t)0x1e69, (q15_t)0x83ab, (q15_t)0x1e62, (q15_t)0x83a9, (q15_t)0x1e5c, (q15_t)0x83a8,
  (q15_t)0x1e56, (q15_t)0x83a6, (q15_t)0x1e50, (q15_t)0x83a5, (q15_t)0x1e4a, (q15_t)0x83a3, (q15_t)0x1e44, (q15_t)0x83a2,
  (q15_t)0x1e3e, (q15_t)0x83a0, (q15_t)0x1e38, (q15_t)0x839f, (q15_t)0x1e32, (q15_t)0x839d, (q15_t)0x1e2c, (q15_t)0x839c,
  (q15_t)0x1e25, (q15_t)0x839a, (q15_t)0x1e1f, (q15_t)0x8399, (q15_t)0x1e19, (q15_t)0x8397, (q15_t)0x1e13, (q15_t)0x8396,
  (q15_t)0x1e0d, (q15_t)0x8394, (q15_t)0x1e07, (q15_t)0x8393, (q15_t)0x1e01, (q15_t)0x8392, (q15_t)0x1dfb, (q15_t)0x8390,
  (q15_t)0x1df5, (q15_t)0x838f, (q15_t)0x1dee, (q15_t)0x838d, (q15_t)0x1de8, (q15_t)0x838c, (q15_t)0x1de2, (q15_t)0x838a,
  (q15_t)0x1ddc, (q15_t)0x8389, (q15_t)0x1dd6, (q15_t)0x8387, (q15_t)0x1dd0, (q15_t)0x8386, (q15_t)0x1dca, (q15_t)0x8384,
  (q15_t)0x1dc4, (q15_t)0x8383, (q15_t)0x1dbe, (q15_t)0x8381, (q15_t)0x1db7, (q15_t)0x8380, (q15_t)0x1db1, (q15_t)0x837e,
  (q15_t)0x1dab, (q15_t)0x837d, (q15_t)0x1da5, (q15_t)0x837c, (q15_t)0x1d9f, (q15_t)0x837a, (q15_t)0x1d99, (q15_t)0x8379,
  (q15_t)0x1d93, (q15_t)0x8377, (q15_t)0x1d8d, (q15_t)0x8376, (q15_t)0x1d87, (q15_t)0x8374, (q15_t)0x1d80, (q15_t)0x8373,
  (q15_t)0x1d7a, (q15_t)0x8371, (q15_t)0x1d74, (q15_t)0x8370, (q15_t)0x1d6e, (q15_t)0x836f, (q15_t)0x1d68, (q15_t)0x836d,
  (q15_t)0x1d62, (q15_t)0x836c, (q15_t)0x1d5c, (q15_t)0x836a, (q15_t)0x1d56, (q15_t)0x8369, (q15_t)0x1d50, (q15_t)0x8367,
  (q15_t)0x1d49, (q15_t)0x8366, (q15_t)0x1d43, (q15_t)0x8364, (q15_t)0x1d3d, (q15_t)0x8363, (q15_t)0x1d37, (q15_t)0x8362,
  (q15_t)0x1d31, (q15_t)0x8360, (q15_t)0x1d2b, (q15_t)0x835f, (q15_t)0x1d25, (q15_t)0x835d, (q15_t)0x1d1f, (q15_t)0x835c,
  (q15_t)0x1d18, (q15_t)0x835a, (q15_t)0x1d12, (q15_t)0x8359, (q15_t)0x1d0c, (q15_t)0x8358, (q15_t)0x1d06, (q15_t)0x8356,
  (q15_t)0x1d00, (q15_t)0x8355, (q15_t)0x1cfa, (q15_t)0x8353, (q15_t)0x1cf4, (q15_t)0x8352, (q15_t)0x1cee, (q15_t)0x8350,
  (q15_t)0x1ce8, (q15_t)0x834f, (q15_t)0x1ce1, (q15_t)0x834e, (q15_t)0x1cdb, (q15_t)0x834c, (q15_t)0x1cd5, (q15_t)0x834b,
  (q15_t)0x1ccf, (q15_t)0x8349, (q15_t)0x1cc9, (q15_t)0x8348, (q15_t)0x1cc3, (q15_t)0x8347, (q15_t)0x1cbd, (q15_t)0x8345,
  (q15_t)0x1cb7, (q15_t)0x8344, (q15_t)0x1cb0, (q15_t)0x8342, (q15_t)0x1caa, (q15_t)0x8341, (q15_t)0x1ca4, (q15_t)0x833f,
  (q15_t)0x1c9e, (q15_t)0x833e, (q15_t)0x1c98, (q15_t)0x833d, (q15_t)0x1c92, (q15_t)0x833b, (q15_t)0x1c8c, (q15_t)0x833a,
  (q15_t)0x1c86, (q15_t)0x8338, (q15_t)0x1c7f, (q15_t)0x8337, (q15_t)0x1c79, (q15_t)0x8336, (q15_t)0x1c73, (q15_t)0x8334,
  (q15_t)0x1c6d, (q15_t)0x8333, (q15_t)0x1c67, (q15_t)0x8331, (q15_t)0x1c61, (q15_t)0x8330, (q15_t)0x1c5b, (q15_t)0x832f,
  (q15_t)0x1c55, (q15_t)0x832d, (q15_t)0x1c4e, (q15_t)0x832c, (q15_t)0x1c48, (q15_t)0x832b, (q15_t)0x1c42, (q15_t)0x8329,
  (q15_t)0x1c3c, (q15_t)0x8328, (q15_t)0x1c36, (q15_t)0x8326, (q15_t)0x1c30, (q15_t)0x8325, (q15_t)0x1c2a, (q15_t)0x8324,
  (q15_t)0x1c24, (q15_t)0x8322, (q15_t)0x1c1d, (q15_t)0x8321, (q15_t)0x1c17, (q15_t)0x831f, (q15_t)0x1c11, (q15_t)0x831e,
  (q15_t)0x1c0b, (q15_t)0x831d, (q15_t)0x1c05, (q15_t)0x831b, (q15_t)0x1bff, (q15_t)0x831a, (q15_t)0x1bf9, (q15_t)0x8319,
  (q15_t)0x1bf2, (q15_t)0x8317, (q15_t)0x1bec, (q15_t)0x8316, (q15_t)0x1be6, (q15_t)0x8314, (q15_t)0x1be0, (q15_t)0x8313,
  (q15_t)0x1bda, (q15_t)0x8312, (q15_t)0x1bd4, (q15_t)0x8310, (q15_t)0x1bce, (q15_t)0x830f, (q15_t)0x1bc8, (q15_t)0x830e,
  (q15_t)0x1bc1, (q15_t)0x830c, (q15_t)0x1bbb, (q15_t)0x830b, (q15_t)0x1bb5, (q15_t)0x830a, (q15_t)0x1baf, (q15_t)0x8308,
  (q15_t)0x1ba9, (q15_t)0x8307, (q15_t)0x1ba3, (q15_t)0x8305, (q15_t)0x1b9d, (q15_t)0x8304, (q15_t)0x1b96, (q15_t)0x8303,
  (q15_t)0x1b90, (q15_t)0x8301, (q15_t)0x1b8a, (q15_t)0x8300, (q15_t)0x1b84, (q15_t)0x82ff, (q15_t)0x1b7e, (q15_t)0x82fd,
  (q15_t)0x1b78, (q15_t)0x82fc, (q15_t)0x1b72, (q15_t)0x82fb, (q15_t)0x1b6c, (q15_t)0x82f9, (q15_t)0x1b65, (q15_t)0x82f8,
  (q15_t)0x1b5f, (q15_t)0x82f7, (q15_t)0x1b59, (q15_t)0x82f5, (q15_t)0x1b53, (q15_t)0x82f4, (q15_t)0x1b4d, (q15_t)0x82f3,
  (q15_t)0x1b47, (q15_t)0x82f1, (q15_t)0x1b41, (q15_t)0x82f0, (q15_t)0x1b3a, (q15_t)0x82ef, (q15_t)0x1b34, (q15_t)0x82ed,
  (q15_t)0x1b2e, (q15_t)0x82ec, (q15_t)0x1b28, (q15_t)0x82eb, (q15_t)0x1b22, (q15_t)0x82e9, (q15_t)0x1b1c, (q15_t)0x82e8,
  (q15_t)0x1b16, (q15_t)0x82e7, (q15_t)0x1b0f, (q15_t)0x82e5, (q15_t)0x1b09, (q15_t)0x82e4, (q15_t)0x1b03, (q15_t)0x82e3,
  (q15_t)0x1afd, (q15_t)0x82e1, (q15_t)0x1af7, (q15_t)0x82e0, (q15_t)0x1af1, (q15_t)0x82df, (q15_t)0x1aeb, (q15_t)0x82dd,
  (q15_t)0x1ae4, (q15_t)0x82dc, (q15_t)0x1ade, (q15_t)0x82db, (q15_t)0x1ad8, (q15_t)0x82d9, (q15_t)0x1ad2, (q15_t)0x82d8,
  (q15_t)0x1acc, (q15_t)0x82d7, (q15_t)0x1ac6, (q15_t)0x82d5, (q15_t)0x1ac0, (q15_t)0x82d4, (q15_t)0x1ab9, (q15_t)0x82d3,
  (q15_t)0x1ab3, (q15_t)0x82d1, (q15_t)0x1aad, (q15_t)0x82d0, (q15_t)0x1aa7, (q15_t)0x82cf, (q15_t)0x1aa1, (q15_t)0x82ce,
  (q15_t)0x1a9b, (q15_t)0x82cc, (q15_t)0x1a95, (q15_t)0x82cb, (q15_t)0x1a8e, (q15_t)0x82ca, (q15_t)0x1a88, (q15_t)0x82c8,
  (q15_t)0x1a82, (q15_t)0x82c7, (q15_t)0x1a7c, (q15_t)0x82c6, (q15_t)0x1a76, (q15_t)0x82c4, (q15_t)0x1a70, (q15_t)0x82c3,
  (q15_t)0x1a6a, (q15_t)0x82c2, (q15_t)0x1a63, (q15_t)0x82c1, (q15_t)0x1a5d, (q15_t)0x82bf, (q15_t)0x1a57, (q15_t)0x82be,
  (q15_t)0x1a51, (q15_t)0x82bd, (q15_t)0x1a4b, (q15_t)0x82bb, (q15_t)0x1a45, (q15_t)0x82ba, (q15_t)0x1a3e, (q15_t)0x82b9,
  (q15_t)0x1a38, (q15_t)0x82b7, (q15_t)0x1a32, (q15_t)0x82b6, (q15_t)0x1a2c, (q15_t)0x82b5, (q15_t)0x1a26, (q15_t)0x82b4,
  (q15_t)0x1a20, (q15_t)0x82b2, (q15_t)0x1a1a, (q15_t)0x82b1, (q15_t)0x1a13, (q15_t)0x82b0, (q15_t)0x1a0d, (q15_t)0x82ae,
  (q15_t)0x1a07, (q15_t)0x82ad, (q15_t)0x1a01, (q15_t)0x82ac, (q15_t)0x19fb, (q15_t)0x82ab, (q15_t)0x19f5, (q15_t)0x82a9,
  (q15_t)0x19ef, (q15_t)0x82a8, (q15_t)0x19e8, (q15_t)0x82a7, (q15_t)0x19e2, (q15_t)0x82a6, (q15_t)0x19dc, (q15_t)0x82a4,
  (q15_t)0x19d6, (q15_t)0x82a3, (q15_t)0x19d0, (q15_t)0x82a2, (q15_t)0x19ca, (q15_t)0x82a0, (q15_t)0x19c3, (q15_t)0x829f,
  (q15_t)0x19bd, (q15_t)0x829e, (q15_t)0x19b7, (q15_t)0x829d, (q15_t)0x19b1, (q15_t)0x829b, (q15_t)0x19ab, (q15_t)0x829a,
  (q15_t)0x19a5, (q15_t)0x8299, (q15_t)0x199f, (q15_t)0x8298, (q15_t)0x1998, (q15_t)0x8296, (q15_t)0x1992, (q15_t)0x8295,
  (q15_t)0x198c, (q15_t)0x8294, (q15_t)0x1986, (q15_t)0x8293, (q15_t)0x1980, (q15_t)0x8291, (q15_t)0x197a, (q15_t)0x8290,
  (q15_t)0x1973, (q15_t)0x828f, (q15_t)0x196d, (q15_t)0x828e, (q15_t)0x1967, (q15_t)0x828c, (q15_t)0x1961, (q15_t)0x828b,
  (q15_t)0x195b, (q15_t)0x828a, (q15_t)0x1955, (q15_t)0x8289, (q15_t)0x194e, (q15_t)0x8287, (q15_t)0x1948, (q15_t)0x8286,
  (q15_t)0x1942, (q15_t)0x8285, (q15_t)0x193c, (q15_t)0x8284, (q15_t)0x1936, (q15_t)0x8282, (q15_t)0x1930, (q15_t)0x8281,
  (q15_t)0x192a, (q15_t)0x8280, (q15_t)0x1923, (q15_t)0x827f, (q15_t)0x191d, (q15_t)0x827e, (q15_t)0x1917, (q15_t)0x827c,
  (q15_t)0x1911, (q15_t)0x827b, (q15_t)0x190b, (q15_t)0x827a, (q15_t)0x1905, (q15_t)0x8279, (q15_t)0x18fe, (q15_t)0x8277,
  (q15_t)0x18f8, (q15_t)0x8276, (q15_t)0x18f2, (q15_t)0x8275, (q15_t)0x18ec, (q15_t)0x8274, (q15_t)0x18e6, (q15_t)0x8272,
  (q15_t)0x18e0, (q15_t)0x8271, (q15_t)0x18d9, (q15_t)0x8270, (q15_t)0x18d3, (q15_t)0x826f, (q15_t)0x18cd, (q15_t)0x826e,
  (q15_t)0x18c7, (q15_t)0x826c, (q15_t)0x18c1, (q15_t)0x826b, (q15_t)0x18bb, (q15_t)0x826a, (q15_t)0x18b4, (q15_t)0x8269,
  (q15_t)0x18ae, (q15_t)0x8268, (q15_t)0x18a8, (q15_t)0x8266, (q15_t)0x18a2, (q15_t)0x8265, (q15_t)0x189c, (q15_t)0x8264,
  (q15_t)0x1896, (q15_t)0x8263, (q15_t)0x188f, (q15_t)0x8261, (q15_t)0x1889, (q15_t)0x8260, (q15_t)0x1883, (q15_t)0x825f,
  (q15_t)0x187d, (q15_t)0x825e, (q15_t)0x1877, (q15_t)0x825d, (q15_t)0x1871, (q15_t)0x825b, (q15_t)0x186a, (q15_t)0x825a,
  (q15_t)0x1864, (q15_t)0x8259, (q15_t)0x185e, (q15_t)0x8258, (q15_t)0x1858, (q15_t)0x8257, (q15_t)0x1852, (q15_t)0x8255,
  (q15_t)0x184c, (q15_t)0x8254, (q15_t)0x1845, (q15_t)0x8253, (q15_t)0x183f, (q15_t)0x8252, (q15_t)0x1839, (q15_t)0x8251,
  (q15_t)0x1833, (q15_t)0x8250, (q15_t)0x182d, (q15_t)0x824e, (q15_t)0x1827, (q15_t)0x824d, (q15_t)0x1820, (q15_t)0x824c,
  (q15_t)0x181a, (q15_t)0x824b, (q15_t)0x1814, (q15_t)0x824a, (q15_t)0x180e, (q15_t)0x8248, (q15_t)0x1808, (q15_t)0x8247,
  (q15_t)0x1802, (q15_t)0x8246, (q15_t)0x17fb, (q15_t)0x8245, (q15_t)0x17f5, (q15_t)0x8244, (q15_t)0x17ef, (q15_t)0x8243,
  (q15_t)0x17e9, (q15_t)0x8241, (q15_t)0x17e3, (q15_t)0x8240, (q15_t)0x17dd, (q15_t)0x823f, (q15_t)0x17d6, (q15_t)0x823e,
  (q15_t)0x17d0, (q15_t)0x823d, (q15_t)0x17ca, (q15_t)0x823b, (q15_t)0x17c4, (q15_t)0x823a, (q15_t)0x17be, (q15_t)0x8239,
  (q15_t)0x17b7, (q15_t)0x8238, (q15_t)0x17b1, (q15_t)0x8237, (q15_t)0x17ab, (q15_t)0x8236, (q15_t)0x17a5, (q15_t)0x8234,
  (q15_t)0x179f, (q15_t)0x8233, (q15_t)0x1799, (q15_t)0x8232, (q15_t)0x1792, (q15_t)0x8231, (q15_t)0x178c, (q15_t)0x8230,
  (q15_t)0x1786, (q15_t)0x822f, (q15_t)0x1780, (q15_t)0x822e, (q15_t)0x177a, (q15_t)0x822c, (q15_t)0x1774, (q15_t)0x822b,
  (q15_t)0x176d, (q15_t)0x822a, (q15_t)0x1767, (q15_t)0x8229, (q15_t)0x1761, (q15_t)0x8228, (q15_t)0x175b, (q15_t)0x8227,
  (q15_t)0x1755, (q15_t)0x8226, (q15_t)0x174e, (q15_t)0x8224, (q15_t)0x1748, (q15_t)0x8223, (q15_t)0x1742, (q15_t)0x8222,
  (q15_t)0x173c, (q15_t)0x8221, (q15_t)0x1736, (q15_t)0x8220, (q15_t)0x1730, (q15_t)0x821f, (q15_t)0x1729, (q15_t)0x821e,
  (q15_t)0x1723, (q15_t)0x821c, (q15_t)0x171d, (q15_t)0x821b, (q15_t)0x1717, (q15_t)0x821a, (q15_t)0x1711, (q15_t)0x8219,
  (q15_t)0x170a, (q15_t)0x8218, (q15_t)0x1704, (q15_t)0x8217, (q15_t)0x16fe, (q15_t)0x8216, (q15_t)0x16f8, (q15_t)0x8214,
  (q15_t)0x16f2, (q15_t)0x8213, (q15_t)0x16ec, (q15_t)0x8212, (q15_t)0x16e5, (q15_t)0x8211, (q15_t)0x16df, (q15_t)0x8210,
  (q15_t)0x16d9, (q15_t)0x820f, (q15_t)0x16d3, (q15_t)0x820e, (q15_t)0x16cd, (q15_t)0x820d, (q15_t)0x16c6, (q15_t)0x820b,
  (q15_t)0x16c0, (q15_t)0x820a, (q15_t)0x16ba, (q15_t)0x8209, (q15_t)0x16b4, (q15_t)0x8208, (q15_t)0x16ae, (q15_t)0x8207,
  (q15_t)0x16a8, (q15_t)0x8206, (q15_t)0x16a1, (q15_t)0x8205, (q15_t)0x169b, (q15_t)0x8204, (q15_t)0x1695, (q15_t)0x8203,
  (q15_t)0x168f, (q15_t)0x8201, (q15_t)0x1689, (q15_t)0x8200, (q15_t)0x1682, (q15_t)0x81ff, (q15_t)0x167c, (q15_t)0x81fe,
  (q15_t)0x1676, (q15_t)0x81fd, (q15_t)0x1670, (q15_t)0x81fc, (q15_t)0x166a, (q15_t)0x81fb, (q15_t)0x1664, (q15_t)0x81fa,
  (q15_t)0x165d, (q15_t)0x81f9, (q15_t)0x1657, (q15_t)0x81f8, (q15_t)0x1651, (q15_t)0x81f6, (q15_t)0x164b, (q15_t)0x81f5,
  (q15_t)0x1645, (q15_t)0x81f4, (q15_t)0x163e, (q15_t)0x81f3, (q15_t)0x1638, (q15_t)0x81f2, (q15_t)0x1632, (q15_t)0x81f1,
  (q15_t)0x162c, (q15_t)0x81f0, (q15_t)0x1626, (q15_t)0x81ef, (q15_t)0x161f, (q15_t)0x81ee, (q15_t)0x1619, (q15_t)0x81ed,
  (q15_t)0x1613, (q15_t)0x81ec, (q15_t)0x160d, (q15_t)0x81ea, (q15_t)0x1607, (q15_t)0x81e9, (q15_t)0x1601, (q15_t)0x81e8,
  (q15_t)0x15fa, (q15_t)0x81e7, (q15_t)0x15f4, (q15_t)0x81e6, (q15_t)0x15ee, (q15_t)0x81e5, (q15_t)0x15e8, (q15_t)0x81e4,
  (q15_t)0x15e2, (q15_t)0x81e3, (q15_t)0x15db, (q15_t)0x81e2, (q15_t)0x15d5, (q15_t)0x81e1, (q15_t)0x15cf, (q15_t)0x81e0,
  (q15_t)0x15c9, (q15_t)0x81df, (q15_t)0x15c3, (q15_t)0x81de, (q15_t)0x15bc, (q15_t)0x81dc, (q15_t)0x15b6, (q15_t)0x81db,
  (q15_t)0x15b0, (q15_t)0x81da, (q15_t)0x15aa, (q15_t)0x81d9, (q15_t)0x15a4, (q15_t)0x81d8, (q15_t)0x159d, (q15_t)0x81d7,
  (q15_t)0x1597, (q15_t)0x81d6, (q15_t)0x1591, (q15_t)0x81d5, (q15_t)0x158b, (q15_t)0x81d4, (q15_t)0x1585, (q15_t)0x81d3,
  (q15_t)0x157f, (q15_t)0x81d2, (q15_t)0x1578, (q15_t)0x81d1, (q15_t)0x1572, (q15_t)0x81d0, (q15_t)0x156c, (q15_t)0x81cf,
  (q15_t)0x1566, (q15_t)0x81ce, (q15_t)0x1560, (q15_t)0x81cd, (q15_t)0x1559, (q15_t)0x81cc, (q15_t)0x1553, (q15_t)0x81cb,
  (q15_t)0x154d, (q15_t)0x81c9, (q15_t)0x1547, (q15_t)0x81c8, (q15_t)0x1541, (q15_t)0x81c7, (q15_t)0x153a, (q15_t)0x81c6,
  (q15_t)0x1534, (q15_t)0x81c5, (q15_t)0x152e, (q15_t)0x81c4, (q15_t)0x1528, (q15_t)0x81c3, (q15_t)0x1522, (q15_t)0x81c2,
  (q15_t)0x151b, (q15_t)0x81c1, (q15_t)0x1515, (q15_t)0x81c0, (q15_t)0x150f, (q15_t)0x81bf, (q15_t)0x1509, (q15_t)0x81be,
  (q15_t)0x1503, (q15_t)0x81bd, (q15_t)0x14fc, (q15_t)0x81bc, (q15_t)0x14f6, (q15_t)0x81bb, (q15_t)0x14f0, (q15_t)0x81ba,
  (q15_t)0x14ea, (q15_t)0x81b9, (q15_t)0x14e4, (q15_t)0x81b8, (q15_t)0x14dd, (q15_t)0x81b7, (q15_t)0x14d7, (q15_t)0x81b6,
  (q15_t)0x14d1, (q15_t)0x81b5, (q15_t)0x14cb, (q15_t)0x81b4, (q15_t)0x14c5, (q15_t)0x81b3, (q15_t)0x14be, (q15_t)0x81b2,
  (q15_t)0x14b8, (q15_t)0x81b1, (q15_t)0x14b2, (q15_t)0x81b0, (q15_t)0x14ac, (q15_t)0x81af, (q15_t)0x14a6, (q15_t)0x81ae,
  (q15_t)0x149f, (q15_t)0x81ad, (q15_t)0x1499, (q15_t)0x81ac, (q15_t)0x1493, (q15_t)0x81ab, (q15_t)0x148d, (q15_t)0x81aa,
  (q15_t)0x1487, (q15_t)0x81a9, (q15_t)0x1480, (q15_t)0x81a8, (q15_t)0x147a, (q15_t)0x81a7, (q15_t)0x1474, (q15_t)0x81a6,
  (q15_t)0x146e, (q15_t)0x81a5, (q15_t)0x1468, (q15_t)0x81a4, (q15_t)0x1461, (q15_t)0x81a3, (q15_t)0x145b, (q15_t)0x81a2,
  (q15_t)0x1455, (q15_t)0x81a1, (q15_t)0x144f, (q15_t)0x81a0, (q15_t)0x1449, (q15_t)0x819f, (q15_t)0x1442, (q15_t)0x819e,
  (q15_t)0x143c, (q15_t)0x819d, (q15_t)0x1436, (q15_t)0x819c, (q15_t)0x1430, (q15_t)0x819b, (q15_t)0x142a, (q15_t)0x819a,
  (q15_t)0x1423, (q15_t)0x8199, (q15_t)0x141d, (q15_t)0x8198, (q15_t)0x1417, (q15_t)0x8197, (q15_t)0x1411, (q15_t)0x8196,
  (q15_t)0x140b, (q15_t)0x8195, (q15_t)0x1404, (q15_t)0x8194, (q15_t)0x13fe, (q15_t)0x8193, (q15_t)0x13f8, (q15_t)0x8192,
  (q15_t)0x13f2, (q15_t)0x8191, (q15_t)0x13eb, (q15_t)0x8190, (q15_t)0x13e5, (q15_t)0x818f, (q15_t)0x13df, (q15_t)0x818e,
  (q15_t)0x13d9, (q15_t)0x818d, (q15_t)0x13d3, (q15_t)0x818c, (q15_t)0x13cc, (q15_t)0x818b, (q15_t)0x13c6, (q15_t)0x818a,
  (q15_t)0x13c0, (q15_t)0x8189, (q15_t)0x13ba, (q15_t)0x8188, (q15_t)0x13b4, (q15_t)0x8187, (q15_t)0x13ad, (q15_t)0x8186,
  (q15_t)0x13a7, (q15_t)0x8185, (q15_t)0x13a1, (q15_t)0x8184, (q15_t)0x139b, (q15_t)0x8183, (q15_t)0x1395, (q15_t)0x8182,
  (q15_t)0x138e, (q15_t)0x8181, (q15_t)0x1388, (q15_t)0x8180, (q15_t)0x1382, (q15_t)0x817f, (q15_t)0x137c, (q15_t)0x817e,
  (q15_t)0x1376, (q15_t)0x817d, (q15_t)0x136f, (q15_t)0x817c, (q15_t)0x1369, (q15_t)0x817c, (q15_t)0x1363, (q15_t)0x817b,
  (q15_t)0x135d, (q15_t)0x817a, (q15_t)0x1356, (q15_t)0x8179, (q15_t)0x1350, (q15_t)0x8178, (q15_t)0x134a, (q15_t)0x8177,
  (q15_t)0x1344, (q15_t)0x8176, (q15_t)0x133e, (q15_t)0x8175, (q15_t)0x1337, (q15_t)0x8174, (q15_t)0x1331, (q15_t)0x8173,
  (q15_t)0x132b, (q15_t)0x8172, (q15_t)0x1325, (q15_t)0x8171, (q15_t)0x131f, (q15_t)0x8170, (q15_t)0x1318, (q15_t)0x816f,
  (q15_t)0x1312, (q15_t)0x816e, (q15_t)0x130c, (q15_t)0x816d, (q15_t)0x1306, (q15_t)0x816c, (q15_t)0x12ff, (q15_t)0x816c,
  (q15_t)0x12f9, (q15_t)0x816b, (q15_t)0x12f3, (q15_t)0x816a, (q15_t)0x12ed, (q15_t)0x8169, (q15_t)0x12e7, (q15_t)0x8168,
  (q15_t)0x12e0, (q15_t)0x8167, (q15_t)0x12da, (q15_t)0x8166, (q15_t)0x12d4, (q15_t)0x8165, (q15_t)0x12ce, (q15_t)0x8164,
  (q15_t)0x12c8, (q15_t)0x8163, (q15_t)0x12c1, (q15_t)0x8162, (q15_t)0x12bb, (q15_t)0x8161, (q15_t)0x12b5, (q15_t)0x8160,
  (q15_t)0x12af, (q15_t)0x815f, (q15_t)0x12a8, (q15_t)0x815f, (q15_t)0x12a2, (q15_t)0x815e, (q15_t)0x129c, (q15_t)0x815d,
  (q15_t)0x1296, (q15_t)0x815c, (q15_t)0x1290, (q15_t)0x815b, (q15_t)0x1289, (q15_t)0x815a, (q15_t)0x1283, (q15_t)0x8159,
  (q15_t)0x127d, (q15_t)0x8158, (q15_t)0x1277, (q15_t)0x8157, (q15_t)0x1271, (q15_t)0x8156, (q15_t)0x126a, (q15_t)0x8155,
  (q15_t)0x1264, (q15_t)0x8155, (q15_t)0x125e, (q15_t)0x8154, (q15_t)0x1258, (q15_t)0x8153, (q15_t)0x1251, (q15_t)0x8152,
  (q15_t)0x124b, (q15_t)0x8151, (q15_t)0x1245, (q15_t)0x8150, (q15_t)0x123f, (q15_t)0x814f, (q15_t)0x1239, (q15_t)0x814e,
  (q15_t)0x1232, (q15_t)0x814d, (q15_t)0x122c, (q15_t)0x814c, (q15_t)0x1226, (q15_t)0x814c, (q15_t)0x1220, (q15_t)0x814b,
  (q15_t)0x1219, (q15_t)0x814a, (q15_t)0x1213, (q15_t)0x8149, (q15_t)0x120d, (q15_t)0x8148, (q15_t)0x1207, (q15_t)0x8147,
  (q15_t)0x1201, (q15_t)0x8146, (q15_t)0x11fa, (q15_t)0x8145, (q15_t)0x11f4, (q15_t)0x8145, (q15_t)0x11ee, (q15_t)0x8144,
  (q15_t)0x11e8, (q15_t)0x8143, (q15_t)0x11e1, (q15_t)0x8142, (q15_t)0x11db, (q15_t)0x8141, (q15_t)0x11d5, (q15_t)0x8140,
  (q15_t)0x11cf, (q15_t)0x813f, (q15_t)0x11c9, (q15_t)0x813e, (q15_t)0x11c2, (q15_t)0x813d, (q15_t)0x11bc, (q15_t)0x813d,
  (q15_t)0x11b6, (q15_t)0x813c, (q15_t)0x11b0, (q15_t)0x813b, (q15_t)0x11a9, (q15_t)0x813a, (q15_t)0x11a3, (q15_t)0x8139,
  (q15_t)0x119d, (q15_t)0x8138, (q15_t)0x1197, (q15_t)0x8137, (q15_t)0x1191, (q15_t)0x8137, (q15_t)0x118a, (q15_t)0x8136,
  (q15_t)0x1184, (q15_t)0x8135, (q15_t)0x117e, (q15_t)0x8134, (q15_t)0x1178, (q15_t)0x8133, (q15_t)0x1171, (q15_t)0x8132,
  (q15_t)0x116b, (q15_t)0x8131, (q15_t)0x1165, (q15_t)0x8131, (q15_t)0x115f, (q15_t)0x8130, (q15_t)0x1159, (q15_t)0x812f,
  (q15_t)0x1152, (q15_t)0x812e, (q15_t)0x114c, (q15_t)0x812d, (q15_t)0x1146, (q15_t)0x812c, (q15_t)0x1140, (q15_t)0x812b,
  (q15_t)0x1139, (q15_t)0x812b, (q15_t)0x1133, (q15_t)0x812a, (q15_t)0x112d, (q15_t)0x8129, (q15_t)0x1127, (q15_t)0x8128,
  (q15_t)0x1121, (q15_t)0x8127, (q15_t)0x111a, (q15_t)0x8126, (q15_t)0x1114, (q15_t)0x8126, (q15_t)0x110e, (q15_t)0x8125,
  (q15_t)0x1108, (q15_t)0x8124, (q15_t)0x1101, (q15_t)0x8123, (q15_t)0x10fb, (q15_t)0x8122, (q15_t)0x10f5, (q15_t)0x8121,
  (q15_t)0x10ef, (q15_t)0x8121, (q15_t)0x10e8, (q15_t)0x8120, (q15_t)0x10e2, (q15_t)0x811f, (q15_t)0x10dc, (q15_t)0x811e,
  (q15_t)0x10d6, (q15_t)0x811d, (q15_t)0x10d0, (q15_t)0x811c, (q15_t)0x10c9, (q15_t)0x811c, (q15_t)0x10c3, (q15_t)0x811b,
  (q15_t)0x10bd, (q15_t)0x811a, (q15_t)0x10b7, (q15_t)0x8119, (q15_t)0x10b0, (q15_t)0x8118, (q15_t)0x10aa, (q15_t)0x8117,
  (q15_t)0x10a4, (q15_t)0x8117, (q15_t)0x109e, (q15_t)0x8116, (q15_t)0x1098, (q15_t)0x8115, (q15_t)0x1091, (q15_t)0x8114,
  (q15_t)0x108b, (q15_t)0x8113, (q15_t)0x1085, (q15_t)0x8113, (q15_t)0x107f, (q15_t)0x8112, (q15_t)0x1078, (q15_t)0x8111,
  (q15_t)0x1072, (q15_t)0x8110, (q15_t)0x106c, (q15_t)0x810f, (q15_t)0x1066, (q15_t)0x810f, (q15_t)0x105f, (q15_t)0x810e,
  (q15_t)0x1059, (q15_t)0x810d, (q15_t)0x1053, (q15_t)0x810c, (q15_t)0x104d, (q15_t)0x810b, (q15_t)0x1047, (q15_t)0x810b,
  (q15_t)0x1040, (q15_t)0x810a, (q15_t)0x103a, (q15_t)0x8109, (q15_t)0x1034, (q15_t)0x8108, (q15_t)0x102e, (q15_t)0x8107,
  (q15_t)0x1027, (q15_t)0x8107, (q15_t)0x1021, (q15_t)0x8106, (q15_t)0x101b, (q15_t)0x8105, (q15_t)0x1015, (q15_t)0x8104,
  (q15_t)0x100e, (q15_t)0x8103, (q15_t)0x1008, (q15_t)0x8103, (q15_t)0x1002, (q15_t)0x8102, (q15_t)0xffc, (q15_t)0x8101,
  (q15_t)0xff5, (q15_t)0x8100, (q15_t)0xfef, (q15_t)0x80ff, (q15_t)0xfe9, (q15_t)0x80ff, (q15_t)0xfe3, (q15_t)0x80fe,
  (q15_t)0xfdd, (q15_t)0x80fd, (q15_t)0xfd6, (q15_t)0x80fc, (q15_t)0xfd0, (q15_t)0x80fc, (q15_t)0xfca, (q15_t)0x80fb,
  (q15_t)0xfc4, (q15_t)0x80fa, (q15_t)0xfbd, (q15_t)0x80f9, (q15_t)0xfb7, (q15_t)0x80f8, (q15_t)0xfb1, (q15_t)0x80f8,
  (q15_t)0xfab, (q15_t)0x80f7, (q15_t)0xfa4, (q15_t)0x80f6, (q15_t)0xf9e, (q15_t)0x80f5, (q15_t)0xf98, (q15_t)0x80f5,
  (q15_t)0xf92, (q15_t)0x80f4, (q15_t)0xf8b, (q15_t)0x80f3, (q15_t)0xf85, (q15_t)0x80f2, (q15_t)0xf7f, (q15_t)0x80f2,
  (q15_t)0xf79, (q15_t)0x80f1, (q15_t)0xf73, (q15_t)0x80f0, (q15_t)0xf6c, (q15_t)0x80ef, (q15_t)0xf66, (q15_t)0x80ef,
  (q15_t)0xf60, (q15_t)0x80ee, (q15_t)0xf5a, (q15_t)0x80ed, (q15_t)0xf53, (q15_t)0x80ec, (q15_t)0xf4d, (q15_t)0x80ec,
  (q15_t)0xf47, (q15_t)0x80eb, (q15_t)0xf41, (q15_t)0x80ea, (q15_t)0xf3a, (q15_t)0x80e9, (q15_t)0xf34, (q15_t)0x80e9,
  (q15_t)0xf2e, (q15_t)0x80e8, (q15_t)0xf28, (q15_t)0x80e7, (q15_t)0xf21, (q15_t)0x80e6, (q15_t)0xf1b, (q15_t)0x80e6,
  (q15_t)0xf15, (q15_t)0x80e5, (q15_t)0xf0f, (q15_t)0x80e4, (q15_t)0xf08, (q15_t)0x80e3, (q15_t)0xf02, (q15_t)0x80e3,
  (q15_t)0xefc, (q15_t)0x80e2, (q15_t)0xef6, (q15_t)0x80e1, (q15_t)0xef0, (q15_t)0x80e0, (q15_t)0xee9, (q15_t)0x80e0,
  (q15_t)0xee3, (q15_t)0x80df, (q15_t)0xedd, (q15_t)0x80de, (q15_t)0xed7, (q15_t)0x80dd, (q15_t)0xed0, (q15_t)0x80dd,
  (q15_t)0xeca, (q15_t)0x80dc, (q15_t)0xec4, (q15_t)0x80db, (q15_t)0xebe, (q15_t)0x80db, (q15_t)0xeb7, (q15_t)0x80da,
  (q15_t)0xeb1, (q15_t)0x80d9, (q15_t)0xeab, (q15_t)0x80d8, (q15_t)0xea5, (q15_t)0x80d8, (q15_t)0xe9e, (q15_t)0x80d7,
  (q15_t)0xe98, (q15_t)0x80d6, (q15_t)0xe92, (q15_t)0x80d6, (q15_t)0xe8c, (q15_t)0x80d5, (q15_t)0xe85, (q15_t)0x80d4,
  (q15_t)0xe7f, (q15_t)0x80d3, (q15_t)0xe79, (q15_t)0x80d3, (q15_t)0xe73, (q15_t)0x80d2, (q15_t)0xe6c, (q15_t)0x80d1,
  (q15_t)0xe66, (q15_t)0x80d1, (q15_t)0xe60, (q15_t)0x80d0, (q15_t)0xe5a, (q15_t)0x80cf, (q15_t)0xe53, (q15_t)0x80ce,
  (q15_t)0xe4d, (q15_t)0x80ce, (q15_t)0xe47, (q15_t)0x80cd, (q15_t)0xe41, (q15_t)0x80cc, (q15_t)0xe3a, (q15_t)0x80cc,
  (q15_t)0xe34, (q15_t)0x80cb, (q15_t)0xe2e, (q15_t)0x80ca, (q15_t)0xe28, (q15_t)0x80ca, (q15_t)0xe22, (q15_t)0x80c9,
  (q15_t)0xe1b, (q15_t)0x80c8, (q15_t)0xe15, (q15_t)0x80c7, (q15_t)0xe0f, (q15_t)0x80c7, (q15_t)0xe09, (q15_t)0x80c6,
  (q15_t)0xe02, (q15_t)0x80c5, (q15_t)0xdfc, (q15_t)0x80c5, (q15_t)0xdf6, (q15_t)0x80c4, (q15_t)0xdf0, (q15_t)0x80c3,
  (q15_t)0xde9, (q15_t)0x80c3, (q15_t)0xde3, (q15_t)0x80c2, (q15_t)0xddd, (q15_t)0x80c1, (q15_t)0xdd7, (q15_t)0x80c1,
  (q15_t)0xdd0, (q15_t)0x80c0, (q15_t)0xdca, (q15_t)0x80bf, (q15_t)0xdc4, (q15_t)0x80bf, (q15_t)0xdbe, (q15_t)0x80be,
  (q15_t)0xdb7, (q15_t)0x80bd, (q15_t)0xdb1, (q15_t)0x80bd, (q15_t)0xdab, (q15_t)0x80bc, (q15_t)0xda5, (q15_t)0x80bb,
  (q15_t)0xd9e, (q15_t)0x80bb, (q15_t)0xd98, (q15_t)0x80ba, (q15_t)0xd92, (q15_t)0x80b9, (q15_t)0xd8c, (q15_t)0x80b9,
  (q15_t)0xd85, (q15_t)0x80b8, (q15_t)0xd7f, (q15_t)0x80b7, (q15_t)0xd79, (q15_t)0x80b7, (q15_t)0xd73, (q15_t)0x80b6,
  (q15_t)0xd6c, (q15_t)0x80b5, (q15_t)0xd66, (q15_t)0x80b5, (q15_t)0xd60, (q15_t)0x80b4, (q15_t)0xd5a, (q15_t)0x80b3,
  (q15_t)0xd53, (q15_t)0x80b3, (q15_t)0xd4d, (q15_t)0x80b2, (q15_t)0xd47, (q15_t)0x80b1, (q15_t)0xd41, (q15_t)0x80b1,
  (q15_t)0xd3a, (q15_t)0x80b0, (q15_t)0xd34, (q15_t)0x80af, (q15_t)0xd2e, (q15_t)0x80af, (q15_t)0xd28, (q15_t)0x80ae,
  (q15_t)0xd21, (q15_t)0x80ad, (q15_t)0xd1b, (q15_t)0x80ad, (q15_t)0xd15, (q15_t)0x80ac, (q15_t)0xd0f, (q15_t)0x80ab,
  (q15_t)0xd08, (q15_t)0x80ab, (q15_t)0xd02, (q15_t)0x80aa, (q15_t)0xcfc, (q15_t)0x80aa, (q15_t)0xcf6, (q15_t)0x80a9,
  (q15_t)0xcef, (q15_t)0x80a8, (q15_t)0xce9, (q15_t)0x80a8, (q15_t)0xce3, (q15_t)0x80a7, (q15_t)0xcdd, (q15_t)0x80a6,
  (q15_t)0xcd6, (q15_t)0x80a6, (q15_t)0xcd0, (q15_t)0x80a5, (q15_t)0xcca, (q15_t)0x80a5, (q15_t)0xcc4, (q15_t)0x80a4,
  (q15_t)0xcbd, (q15_t)0x80a3, (q15_t)0xcb7, (q15_t)0x80a3, (q15_t)0xcb1, (q15_t)0x80a2, (q15_t)0xcab, (q15_t)0x80a1,
  (q15_t)0xca4, (q15_t)0x80a1, (q15_t)0xc9e, (q15_t)0x80a0, (q15_t)0xc98, (q15_t)0x80a0, (q15_t)0xc92, (q15_t)0x809f,
  (q15_t)0xc8b, (q15_t)0x809e, (q15_t)0xc85, (q15_t)0x809e, (q15_t)0xc7f, (q15_t)0x809d, (q15_t)0xc79, (q15_t)0x809c,
  (q15_t)0xc72, (q15_t)0x809c, (q15_t)0xc6c, (q15_t)0x809b, (q15_t)0xc66, (q15_t)0x809b, (q15_t)0xc60, (q15_t)0x809a,
  (q15_t)0xc59, (q15_t)0x8099, (q15_t)0xc53, (q15_t)0x8099, (q15_t)0xc4d, (q15_t)0x8098, (q15_t)0xc47, (q15_t)0x8098,
  (q15_t)0xc40, (q15_t)0x8097, (q15_t)0xc3a, (q15_t)0x8096, (q15_t)0xc34, (q15_t)0x8096, (q15_t)0xc2e, (q15_t)0x8095,
  (q15_t)0xc27, (q15_t)0x8095, (q15_t)0xc21, (q15_t)0x8094, (q15_t)0xc1b, (q15_t)0x8093, (q15_t)0xc14, (q15_t)0x8093,
  (q15_t)0xc0e, (q15_t)0x8092, (q15_t)0xc08, (q15_t)0x8092, (q15_t)0xc02, (q15_t)0x8091, (q15_t)0xbfb, (q15_t)0x8090,
  (q15_t)0xbf5, (q15_t)0x8090, (q15_t)0xbef, (q15_t)0x808f, (q15_t)0xbe9, (q15_t)0x808f, (q15_t)0xbe2, (q15_t)0x808e,
  (q15_t)0xbdc, (q15_t)0x808e, (q15_t)0xbd6, (q15_t)0x808d, (q15_t)0xbd0, (q15_t)0x808c, (q15_t)0xbc9, (q15_t)0x808c,
  (q15_t)0xbc3, (q15_t)0x808b, (q15_t)0xbbd, (q15_t)0x808b, (q15_t)0xbb7, (q15_t)0x808a, (q15_t)0xbb0, (q15_t)0x8089,
  (q15_t)0xbaa, (q15_t)0x8089, (q15_t)0xba4, (q15_t)0x8088, (q15_t)0xb9e, (q15_t)0x8088, (q15_t)0xb97, (q15_t)0x8087,
  (q15_t)0xb91, (q15_t)0x8087, (q15_t)0xb8b, (q15_t)0x8086, (q15_t)0xb85, (q15_t)0x8085, (q15_t)0xb7e, (q15_t)0x8085,
  (q15_t)0xb78, (q15_t)0x8084, (q15_t)0xb72, (q15_t)0x8084, (q15_t)0xb6c, (q15_t)0x8083, (q15_t)0xb65, (q15_t)0x8083,
  (q15_t)0xb5f, (q15_t)0x8082, (q15_t)0xb59, (q15_t)0x8082, (q15_t)0xb53, (q15_t)0x8081, (q15_t)0xb4c, (q15_t)0x8080,
  (q15_t)0xb46, (q15_t)0x8080, (q15_t)0xb40, (q15_t)0x807f, (q15_t)0xb3a, (q15_t)0x807f, (q15_t)0xb33, (q15_t)0x807e,
  (q15_t)0xb2d, (q15_t)0x807e, (q15_t)0xb27, (q15_t)0x807d, (q15_t)0xb20, (q15_t)0x807d, (q15_t)0xb1a, (q15_t)0x807c,
  (q15_t)0xb14, (q15_t)0x807b, (q15_t)0xb0e, (q15_t)0x807b, (q15_t)0xb07, (q15_t)0x807a, (q15_t)0xb01, (q15_t)0x807a,
  (q15_t)0xafb, (q15_t)0x8079, (q15_t)0xaf5, (q15_t)0x8079, (q15_t)0xaee, (q15_t)0x8078, (q15_t)0xae8, (q15_t)0x8078,
  (q15_t)0xae2, (q15_t)0x8077, (q15_t)0xadc, (q15_t)0x8077, (q15_t)0xad5, (q15_t)0x8076, (q15_t)0xacf, (q15_t)0x8076,
  (q15_t)0xac9, (q15_t)0x8075, (q15_t)0xac3, (q15_t)0x8075, (q15_t)0xabc, (q15_t)0x8074, (q15_t)0xab6, (q15_t)0x8073,
  (q15_t)0xab0, (q15_t)0x8073, (q15_t)0xaaa, (q15_t)0x8072, (q15_t)0xaa3, (q15_t)0x8072, (q15_t)0xa9d, (q15_t)0x8071,
  (q15_t)0xa97, (q15_t)0x8071, (q15_t)0xa90, (q15_t)0x8070, (q15_t)0xa8a, (q15_t)0x8070, (q15_t)0xa84, (q15_t)0x806f,
  (q15_t)0xa7e, (q15_t)0x806f, (q15_t)0xa77, (q15_t)0x806e, (q15_t)0xa71, (q15_t)0x806e, (q15_t)0xa6b, (q15_t)0x806d,
  (q15_t)0xa65, (q15_t)0x806d, (q15_t)0xa5e, (q15_t)0x806c, (q15_t)0xa58, (q15_t)0x806c, (q15_t)0xa52, (q15_t)0x806b,
  (q15_t)0xa4c, (q15_t)0x806b, (q15_t)0xa45, (q15_t)0x806a, (q15_t)0xa3f, (q15_t)0x806a, (q15_t)0xa39, (q15_t)0x8069,
  (q15_t)0xa33, (q15_t)0x8069, (q15_t)0xa2c, (q15_t)0x8068, (q15_t)0xa26, (q15_t)0x8068, (q15_t)0xa20, (q15_t)0x8067,
  (q15_t)0xa19, (q15_t)0x8067, (q15_t)0xa13, (q15_t)0x8066, (q15_t)0xa0d, (q15_t)0x8066, (q15_t)0xa07, (q15_t)0x8065,
  (q15_t)0xa00, (q15_t)0x8065, (q15_t)0x9fa, (q15_t)0x8064, (q15_t)0x9f4, (q15_t)0x8064, (q15_t)0x9ee, (q15_t)0x8063,
  (q15_t)0x9e7, (q15_t)0x8063, (q15_t)0x9e1, (q15_t)0x8062, (q15_t)0x9db, (q15_t)0x8062, (q15_t)0x9d5, (q15_t)0x8061,
  (q15_t)0x9ce, (q15_t)0x8061, (q15_t)0x9c8, (q15_t)0x8060, (q15_t)0x9c2, (q15_t)0x8060, (q15_t)0x9bc, (q15_t)0x805f,
  (q15_t)0x9b5, (q15_t)0x805f, (q15_t)0x9af, (q15_t)0x805e, (q15_t)0x9a9, (q15_t)0x805e, (q15_t)0x9a2, (q15_t)0x805d,
  (q15_t)0x99c, (q15_t)0x805d, (q15_t)0x996, (q15_t)0x805d, (q15_t)0x990, (q15_t)0x805c, (q15_t)0x989, (q15_t)0x805c,
  (q15_t)0x983, (q15_t)0x805b, (q15_t)0x97d, (q15_t)0x805b, (q15_t)0x977, (q15_t)0x805a, (q15_t)0x970, (q15_t)0x805a,
  (q15_t)0x96a, (q15_t)0x8059, (q15_t)0x964, (q15_t)0x8059, (q15_t)0x95e, (q15_t)0x8058, (q15_t)0x957, (q15_t)0x8058,
  (q15_t)0x951, (q15_t)0x8057, (q15_t)0x94b, (q15_t)0x8057, (q15_t)0x944, (q15_t)0x8057, (q15_t)0x93e, (q15_t)0x8056,
  (q15_t)0x938, (q15_t)0x8056, (q15_t)0x932, (q15_t)0x8055, (q15_t)0x92b, (q15_t)0x8055, (q15_t)0x925, (q15_t)0x8054,
  (q15_t)0x91f, (q15_t)0x8054, (q15_t)0x919, (q15_t)0x8053, (q15_t)0x912, (q15_t)0x8053, (q15_t)0x90c, (q15_t)0x8052,
  (q15_t)0x906, (q15_t)0x8052, (q15_t)0x900, (q15_t)0x8052, (q15_t)0x8f9, (q15_t)0x8051, (q15_t)0x8f3, (q15_t)0x8051,
  (q15_t)0x8ed, (q15_t)0x8050, (q15_t)0x8e6, (q15_t)0x8050, (q15_t)0x8e0, (q15_t)0x804f, (q15_t)0x8da, (q15_t)0x804f,
  (q15_t)0x8d4, (q15_t)0x804f, (q15_t)0x8cd, (q15_t)0x804e, (q15_t)0x8c7, (q15_t)0x804e, (q15_t)0x8c1, (q15_t)0x804d,
  (q15_t)0x8bb, (q15_t)0x804d, (q15_t)0x8b4, (q15_t)0x804c, (q15_t)0x8ae, (q15_t)0x804c, (q15_t)0x8a8, (q15_t)0x804c,
  (q15_t)0x8a2, (q15_t)0x804b, (q15_t)0x89b, (q15_t)0x804b, (q15_t)0x895, (q15_t)0x804a, (q15_t)0x88f, (q15_t)0x804a,
  (q15_t)0x888, (q15_t)0x8049, (q15_t)0x882, (q15_t)0x8049, (q15_t)0x87c, (q15_t)0x8049, (q15_t)0x876, (q15_t)0x8048,
  (q15_t)0x86f, (q15_t)0x8048, (q15_t)0x869, (q15_t)0x8047, (q15_t)0x863, (q15_t)0x8047, (q15_t)0x85d, (q15_t)0x8047,
  (q15_t)0x856, (q15_t)0x8046, (q15_t)0x850, (q15_t)0x8046, (q15_t)0x84a, (q15_t)0x8045, (q15_t)0x843, (q15_t)0x8045,
  (q15_t)0x83d, (q15_t)0x8044, (q15_t)0x837, (q15_t)0x8044, (q15_t)0x831, (q15_t)0x8044, (q15_t)0x82a, (q15_t)0x8043,
  (q15_t)0x824, (q15_t)0x8043, (q15_t)0x81e, (q15_t)0x8042, (q15_t)0x818, (q15_t)0x8042, (q15_t)0x811, (q15_t)0x8042,
  (q15_t)0x80b, (q15_t)0x8041, (q15_t)0x805, (q15_t)0x8041, (q15_t)0x7fe, (q15_t)0x8040, (q15_t)0x7f8, (q15_t)0x8040,
  (q15_t)0x7f2, (q15_t)0x8040, (q15_t)0x7ec, (q15_t)0x803f, (q15_t)0x7e5, (q15_t)0x803f, (q15_t)0x7df, (q15_t)0x803f,
  (q15_t)0x7d9, (q15_t)0x803e, (q15_t)0x7d3, (q15_t)0x803e, (q15_t)0x7cc, (q15_t)0x803d, (q15_t)0x7c6, (q15_t)0x803d,
  (q15_t)0x7c0, (q15_t)0x803d, (q15_t)0x7ba, (q15_t)0x803c, (q15_t)0x7b3, (q15_t)0x803c, (q15_t)0x7ad, (q15_t)0x803b,
  (q15_t)0x7a7, (q15_t)0x803b, (q15_t)0x7a0, (q15_t)0x803b, (q15_t)0x79a, (q15_t)0x803a, (q15_t)0x794, (q15_t)0x803a,
  (q15_t)0x78e, (q15_t)0x803a, (q15_t)0x787, (q15_t)0x8039, (q15_t)0x781, (q15_t)0x8039, (q15_t)0x77b, (q15_t)0x8039,
  (q15_t)0x775, (q15_t)0x8038, (q15_t)0x76e, (q15_t)0x8038, (q15_t)0x768, (q15_t)0x8037, (q15_t)0x762, (q15_t)0x8037,
  (q15_t)0x75b, (q15_t)0x8037, (q15_t)0x755, (q15_t)0x8036, (q15_t)0x74f, (q15_t)0x8036, (q15_t)0x749, (q15_t)0x8036,
  (q15_t)0x742, (q15_t)0x8035, (q15_t)0x73c, (q15_t)0x8035, (q15_t)0x736, (q15_t)0x8035, (q15_t)0x730, (q15_t)0x8034,
  (q15_t)0x729, (q15_t)0x8034, (q15_t)0x723, (q15_t)0x8033, (q15_t)0x71d, (q15_t)0x8033, (q15_t)0x716, (q15_t)0x8033,
  (q15_t)0x710, (q15_t)0x8032, (q15_t)0x70a, (q15_t)0x8032, (q15_t)0x704, (q15_t)0x8032, (q15_t)0x6fd, (q15_t)0x8031,
  (q15_t)0x6f7, (q15_t)0x8031, (q15_t)0x6f1, (q15_t)0x8031, (q15_t)0x6ea, (q15_t)0x8030, (q15_t)0x6e4, (q15_t)0x8030,
  (q15_t)0x6de, (q15_t)0x8030, (q15_t)0x6d8, (q15_t)0x802f, (q15_t)0x6d1, (q15_t)0x802f, (q15_t)0x6cb, (q15_t)0x802f,
  (q15_t)0x6c5, (q15_t)0x802e, (q15_t)0x6bf, (q15_t)0x802e, (q15_t)0x6b8, (q15_t)0x802e, (q15_t)0x6b2, (q15_t)0x802d,
  (q15_t)0x6ac, (q15_t)0x802d, (q15_t)0x6a5, (q15_t)0x802d, (q15_t)0x69f, (q15_t)0x802c, (q15_t)0x699, (q15_t)0x802c,
  (q15_t)0x693, (q15_t)0x802c, (q15_t)0x68c, (q15_t)0x802b, (q15_t)0x686, (q15_t)0x802b, (q15_t)0x680, (q15_t)0x802b,
  (q15_t)0x67a, (q15_t)0x802a, (q15_t)0x673, (q15_t)0x802a, (q15_t)0x66d, (q15_t)0x802a, (q15_t)0x667, (q15_t)0x802a,
  (q15_t)0x660, (q15_t)0x8029, (q15_t)0x65a, (q15_t)0x8029, (q15_t)0x654, (q15_t)0x8029, (q15_t)0x64e, (q15_t)0x8028,
  (q15_t)0x647, (q15_t)0x8028, (q15_t)0x641, (q15_t)0x8028, (q15_t)0x63b, (q15_t)0x8027, (q15_t)0x635, (q15_t)0x8027,
  (q15_t)0x62e, (q15_t)0x8027, (q15_t)0x628, (q15_t)0x8026, (q15_t)0x622, (q15_t)0x8026, (q15_t)0x61b, (q15_t)0x8026,
  (q15_t)0x615, (q15_t)0x8026, (q15_t)0x60f, (q15_t)0x8025, (q15_t)0x609, (q15_t)0x8025, (q15_t)0x602, (q15_t)0x8025,
  (q15_t)0x5fc, (q15_t)0x8024, (q15_t)0x5f6, (q15_t)0x8024, (q15_t)0x5ef, (q15_t)0x8024, (q15_t)0x5e9, (q15_t)0x8023,
  (q15_t)0x5e3, (q15_t)0x8023, (q15_t)0x5dd, (q15_t)0x8023, (q15_t)0x5d6, (q15_t)0x8023, (q15_t)0x5d0, (q15_t)0x8022,
  (q15_t)0x5ca, (q15_t)0x8022, (q15_t)0x5c4, (q15_t)0x8022, (q15_t)0x5bd, (q15_t)0x8021, (q15_t)0x5b7, (q15_t)0x8021,
  (q15_t)0x5b1, (q15_t)0x8021, (q15_t)0x5aa, (q15_t)0x8021, (q15_t)0x5a4, (q15_t)0x8020, (q15_t)0x59e, (q15_t)0x8020,
  (q15_t)0x598, (q15_t)0x8020, (q15_t)0x591, (q15_t)0x8020, (q15_t)0x58b, (q15_t)0x801f, (q15_t)0x585, (q15_t)0x801f,
  (q15_t)0x57f, (q15_t)0x801f, (q15_t)0x578, (q15_t)0x801e, (q15_t)0x572, (q15_t)0x801e, (q15_t)0x56c, (q15_t)0x801e,
  (q15_t)0x565, (q15_t)0x801e, (q15_t)0x55f, (q15_t)0x801d, (q15_t)0x559, (q15_t)0x801d, (q15_t)0x553, (q15_t)0x801d,
  (q15_t)0x54c, (q15_t)0x801d, (q15_t)0x546, (q15_t)0x801c, (q15_t)0x540, (q15_t)0x801c, (q15_t)0x539, (q15_t)0x801c,
  (q15_t)0x533, (q15_t)0x801c, (q15_t)0x52d, (q15_t)0x801b, (q15_t)0x527, (q15_t)0x801b, (q15_t)0x520, (q15_t)0x801b,
  (q15_t)0x51a, (q15_t)0x801b, (q15_t)0x514, (q15_t)0x801a, (q15_t)0x50d, (q15_t)0x801a, (q15_t)0x507, (q15_t)0x801a,
  (q15_t)0x501, (q15_t)0x801a, (q15_t)0x4fb, (q15_t)0x8019, (q15_t)0x4f4, (q15_t)0x8019, (q15_t)0x4ee, (q15_t)0x8019,
  (q15_t)0x4e8, (q15_t)0x8019, (q15_t)0x4e2, (q15_t)0x8018, (q15_t)0x4db, (q15_t)0x8018, (q15_t)0x4d5, (q15_t)0x8018,
  (q15_t)0x4cf, (q15_t)0x8018, (q15_t)0x4c8, (q15_t)0x8017, (q15_t)0x4c2, (q15_t)0x8017, (q15_t)0x4bc, (q15_t)0x8017,
  (q15_t)0x4b6, (q15_t)0x8017, (q15_t)0x4af, (q15_t)0x8016, (q15_t)0x4a9, (q15_t)0x8016, (q15_t)0x4a3, (q15_t)0x8016,
  (q15_t)0x49c, (q15_t)0x8016, (q15_t)0x496, (q15_t)0x8016, (q15_t)0x490, (q15_t)0x8015, (q15_t)0x48a, (q15_t)0x8015,
  (q15_t)0x483, (q15_t)0x8015, (q15_t)0x47d, (q15_t)0x8015, (q15_t)0x477, (q15_t)0x8014, (q15_t)0x471, (q15_t)0x8014,
  (q15_t)0x46a, (q15_t)0x8014, (q15_t)0x464, (q15_t)0x8014, (q15_t)0x45e, (q15_t)0x8014, (q15_t)0x457, (q15_t)0x8013,
  (q15_t)0x451, (q15_t)0x8013, (q15_t)0x44b, (q15_t)0x8013, (q15_t)0x445, (q15_t)0x8013, (q15_t)0x43e, (q15_t)0x8013,
  (q15_t)0x438, (q15_t)0x8012, (q15_t)0x432, (q15_t)0x8012, (q15_t)0x42b, (q15_t)0x8012, (q15_t)0x425, (q15_t)0x8012,
  (q15_t)0x41f, (q15_t)0x8012, (q15_t)0x419, (q15_t)0x8011, (q15_t)0x412, (q15_t)0x8011, (q15_t)0x40c, (q15_t)0x8011,
  (q15_t)0x406, (q15_t)0x8011, (q15_t)0x3ff, (q15_t)0x8011, (q15_t)0x3f9, (q15_t)0x8010, (q15_t)0x3f3, (q15_t)0x8010,
  (q15_t)0x3ed, (q15_t)0x8010, (q15_t)0x3e6, (q15_t)0x8010, (q15_t)0x3e0, (q15_t)0x8010, (q15_t)0x3da, (q15_t)0x800f,
  (q15_t)0x3d4, (q15_t)0x800f, (q15_t)0x3cd, (q15_t)0x800f, (q15_t)0x3c7, (q15_t)0x800f, (q15_t)0x3c1, (q15_t)0x800f,
  (q15_t)0x3ba, (q15_t)0x800e, (q15_t)0x3b4, (q15_t)0x800e, (q15_t)0x3ae, (q15_t)0x800e, (q15_t)0x3a8, (q15_t)0x800e,
  (q15_t)0x3a1, (q15_t)0x800e, (q15_t)0x39b, (q15_t)0x800e, (q15_t)0x395, (q15_t)0x800d, (q15_t)0x38e, (q15_t)0x800d,
  (q15_t)0x388, (q15_t)0x800d, (q15_t)0x382, (q15_t)0x800d, (q15_t)0x37c, (q15_t)0x800d, (q15_t)0x375, (q15_t)0x800c,
  (q15_t)0x36f, (q15_t)0x800c, (q15_t)0x369, (q15_t)0x800c, (q15_t)0x362, (q15_t)0x800c, (q15_t)0x35c, (q15_t)0x800c,
  (q15_t)0x356, (q15_t)0x800c, (q15_t)0x350, (q15_t)0x800b, (q15_t)0x349, (q15_t)0x800b, (q15_t)0x343, (q15_t)0x800b,
  (q15_t)0x33d, (q15_t)0x800b, (q15_t)0x337, (q15_t)0x800b, (q15_t)0x330, (q15_t)0x800b, (q15_t)0x32a, (q15_t)0x800b,
  (q15_t)0x324, (q15_t)0x800a, (q15_t)0x31d, (q15_t)0x800a, (q15_t)0x317, (q15_t)0x800a, (q15_t)0x311, (q15_t)0x800a,
  (q15_t)0x30b, (q15_t)0x800a, (q15_t)0x304, (q15_t)0x800a, (q15_t)0x2fe, (q15_t)0x8009, (q15_t)0x2f8, (q15_t)0x8009,
  (q15_t)0x2f1, (q15_t)0x8009, (q15_t)0x2eb, (q15_t)0x8009, (q15_t)0x2e5, (q15_t)0x8009, (q15_t)0x2df, (q15_t)0x8009,
  (q15_t)0x2d8, (q15_t)0x8009, (q15_t)0x2d2, (q15_t)0x8008, (q15_t)0x2cc, (q15_t)0x8008, (q15_t)0x2c5, (q15_t)0x8008,
  (q15_t)0x2bf, (q15_t)0x8008, (q15_t)0x2b9, (q15_t)0x8008, (q15_t)0x2b3, (q15_t)0x8008, (q15_t)0x2ac, (q15_t)0x8008,
  (q15_t)0x2a6, (q15_t)0x8008, (q15_t)0x2a0, (q15_t)0x8007, (q15_t)0x299, (q15_t)0x8007, (q15_t)0x293, (q15_t)0x8007,
  (q15_t)0x28d, (q15_t)0x8007, (q15_t)0x287, (q15_t)0x8007, (q15_t)0x280, (q15_t)0x8007, (q15_t)0x27a, (q15_t)0x8007,
  (q15_t)0x274, (q15_t)0x8007, (q15_t)0x26d, (q15_t)0x8006, (q15_t)0x267, (q15_t)0x8006, (q15_t)0x261, (q15_t)0x8006,
  (q15_t)0x25b, (q15_t)0x8006, (q15_t)0x254, (q15_t)0x8006, (q15_t)0x24e, (q15_t)0x8006, (q15_t)0x248, (q15_t)0x8006,
  (q15_t)0x242, (q15_t)0x8006, (q15_t)0x23b, (q15_t)0x8005, (q15_t)0x235, (q15_t)0x8005, (q15_t)0x22f, (q15_t)0x8005,
  (q15_t)0x228, (q15_t)0x8005, (q15_t)0x222, (q15_t)0x8005, (q15_t)0x21c, (q15_t)0x8005, (q15_t)0x216, (q15_t)0x8005,
  (q15_t)0x20f, (q15_t)0x8005, (q15_t)0x209, (q15_t)0x8005, (q15_t)0x203, (q15_t)0x8005, (q15_t)0x1fc, (q15_t)0x8004,
  (q15_t)0x1f6, (q15_t)0x8004, (q15_t)0x1f0, (q15_t)0x8004, (q15_t)0x1ea, (q15_t)0x8004, (q15_t)0x1e3, (q15_t)0x8004,
  (q15_t)0x1dd, (q15_t)0x8004, (q15_t)0x1d7, (q15_t)0x8004, (q15_t)0x1d0, (q15_t)0x8004, (q15_t)0x1ca, (q15_t)0x8004,
  (q15_t)0x1c4, (q15_t)0x8004, (q15_t)0x1be, (q15_t)0x8004, (q15_t)0x1b7, (q15_t)0x8003, (q15_t)0x1b1, (q15_t)0x8003,
  (q15_t)0x1ab, (q15_t)0x8003, (q15_t)0x1a4, (q15_t)0x8003, (q15_t)0x19e, (q15_t)0x8003, (q15_t)0x198, (q15_t)0x8003,
  (q15_t)0x192, (q15_t)0x8003, (q15_t)0x18b, (q15_t)0x8003, (q15_t)0x185, (q15_t)0x8003, (q15_t)0x17f, (q15_t)0x8003,
  (q15_t)0x178, (q15_t)0x8003, (q15_t)0x172, (q15_t)0x8003, (q15_t)0x16c, (q15_t)0x8003, (q15_t)0x166, (q15_t)0x8002,
  (q15_t)0x15f, (q15_t)0x8002, (q15_t)0x159, (q15_t)0x8002, (q15_t)0x153, (q15_t)0x8002, (q15_t)0x14d, (q15_t)0x8002,
  (q15_t)0x146, (q15_t)0x8002, (q15_t)0x140, (q15_t)0x8002, (q15_t)0x13a, (q15_t)0x8002, (q15_t)0x133, (q15_t)0x8002,
  (q15_t)0x12d, (q15_t)0x8002, (q15_t)0x127, (q15_t)0x8002, (q15_t)0x121, (q15_t)0x8002, (q15_t)0x11a, (q15_t)0x8002,
  (q15_t)0x114, (q15_t)0x8002, (q15_t)0x10e, (q15_t)0x8002, (q15_t)0x107, (q15_t)0x8002, (q15_t)0x101, (q15_t)0x8002,
  (q15_t)0xfb, (q15_t)0x8001, (q15_t)0xf5, (q15_t)0x8001, (q15_t)0xee, (q15_t)0x8001, (q15_t)0xe8, (q15_t)0x8001,
  (q15_t)0xe2, (q15_t)0x8001, (q15_t)0xdb, (q15_t)0x8001, (q15_t)0xd5, (q15_t)0x8001, (q15_t)0xcf, (q15_t)0x8001,
  (q15_t)0xc9, (q15_t)0x8001, (q15_t)0xc2, (q15_t)0x8001, (q15_t)0xbc, (q15_t)0x8001, (q15_t)0xb6, (q15_t)0x8001,
  (q15_t)0xaf, (q15_t)0x8001, (q15_t)0xa9, (q15_t)0x8001, (q15_t)0xa3, (q15_t)0x8001, (q15_t)0x9d, (q15_t)0x8001,
  (q15_t)0x96, (q15_t)0x8001, (q15_t)0x90, (q15_t)0x8001, (q15_t)0x8a, (q15_t)0x8001, (q15_t)0x83, (q15_t)0x8001,
  (q15_t)0x7d, (q15_t)0x8001, (q15_t)0x77, (q15_t)0x8001, (q15_t)0x71, (q15_t)0x8001, (q15_t)0x6a, (q15_t)0x8001,
  (q15_t)0x64, (q15_t)0x8001, (q15_t)0x5e, (q15_t)0x8001, (q15_t)0x57, (q15_t)0x8001, (q15_t)0x51, (q15_t)0x8001,
  (q15_t)0x4b, (q15_t)0x8001, (q15_t)0x45, (q15_t)0x8001, (q15_t)0x3e, (q15_t)0x8001, (q15_t)0x38, (q15_t)0x8001,
  (q15_t)0x32, (q15_t)0x8001, (q15_t)0x2b, (q15_t)0x8001, (q15_t)0x25, (q15_t)0x8001, (q15_t)0x1f, (q15_t)0x8001,
  (q15_t)0x19, (q15_t)0x8001, (q15_t)0x12, (q15_t)0x8001, (q15_t)0xc, (q15_t)0x8001, (q15_t)0x6, (q15_t)0x8001
};


/**
* \par
* cosFactor tables are generated using the formula : <pre> cos_factors[n] = 2 * cos((2n+1)*pi/(4*N)) </pre>
* \par
* C command to generate the table
* <pre>
* for(i = 0; i< N; i++)
* {
*   cos_factors[i]= 2 * cos((2*i+1)*c/2);
* } </pre>
* \par
* where <code>N</code> is the number of factors to generate and <code>c</code> is <code>pi/(2*N)</code>
* \par
* Then converted to q15 format by multiplying with 2^31 and saturated if required.

*/

static const q15_t ALIGN4 cos_factorsQ15_128[128] = {
  (q15_t)0x7fff, (q15_t)0x7ffa, (q15_t)0x7ff0, (q15_t)0x7fe1, (q15_t)0x7fce, (q15_t)0x7fb5, (q15_t)0x7f97, (q15_t)0x7f75,
  (q15_t)0x7f4d, (q15_t)0x7f21, (q15_t)0x7ef0, (q15_t)0x7eba, (q15_t)0x7e7f, (q15_t)0x7e3f, (q15_t)0x7dfa, (q15_t)0x7db0,
  (q15_t)0x7d62, (q15_t)0x7d0f, (q15_t)0x7cb7, (q15_t)0x7c5a, (q15_t)0x7bf8, (q15_t)0x7b92, (q15_t)0x7b26, (q15_t)0x7ab6,
  (q15_t)0x7a42, (q15_t)0x79c8, (q15_t)0x794a, (q15_t)0x78c7, (q15_t)0x7840, (q15_t)0x77b4, (q15_t)0x7723, (q15_t)0x768e,
  (q15_t)0x75f4, (q15_t)0x7555, (q15_t)0x74b2, (q15_t)0x740b, (q15_t)0x735f, (q15_t)0x72af, (q15_t)0x71fa, (q15_t)0x7141,
  (q15_t)0x7083, (q15_t)0x6fc1, (q15_t)0x6efb, (q15_t)0x6e30, (q15_t)0x6d62, (q15_t)0x6c8f, (q15_t)0x6bb8, (q15_t)0x6adc,
  (q15_t)0x69fd, (q15_t)0x6919, (q15_t)0x6832, (q15_t)0x6746, (q15_t)0x6657, (q15_t)0x6563, (q15_t)0x646c, (q15_t)0x6371,
  (q15_t)0x6271, (q15_t)0x616f, (q15_t)0x6068, (q15_t)0x5f5e, (q15_t)0x5e50, (q15_t)0x5d3e, (q15_t)0x5c29, (q15_t)0x5b10,
  (q15_t)0x59f3, (q15_t)0x58d4, (q15_t)0x57b0, (q15_t)0x568a, (q15_t)0x5560, (q15_t)0x5433, (q15_t)0x5302, (q15_t)0x51ce,
  (q15_t)0x5097, (q15_t)0x4f5e, (q15_t)0x4e21, (q15_t)0x4ce1, (q15_t)0x4b9e, (q15_t)0x4a58, (q15_t)0x490f, (q15_t)0x47c3,
  (q15_t)0x4675, (q15_t)0x4524, (q15_t)0x43d0, (q15_t)0x427a, (q15_t)0x4121, (q15_t)0x3fc5, (q15_t)0x3e68, (q15_t)0x3d07,
  (q15_t)0x3ba5, (q15_t)0x3a40, (q15_t)0x38d8, (q15_t)0x376f, (q15_t)0x3604, (q15_t)0x3496, (q15_t)0x3326, (q15_t)0x31b5,
  (q15_t)0x3041, (q15_t)0x2ecc, (q15_t)0x2d55, (q15_t)0x2bdc, (q15_t)0x2a61, (q15_t)0x28e5, (q15_t)0x2767, (q15_t)0x25e8,
  (q15_t)0x2467, (q15_t)0x22e5, (q15_t)0x2161, (q15_t)0x1fdc, (q15_t)0x1e56, (q15_t)0x1ccf, (q15_t)0x1b47, (q15_t)0x19bd,
  (q15_t)0x1833, (q15_t)0x16a8, (q15_t)0x151b, (q15_t)0x138e, (q15_t)0x1201, (q15_t)0x1072, (q15_t)0xee3, (q15_t)0xd53,
  (q15_t)0xbc3, (q15_t)0xa33, (q15_t)0x8a2, (q15_t)0x710, (q15_t)0x57f, (q15_t)0x3ed, (q15_t)0x25b, (q15_t)0xc9
};

static const q15_t ALIGN4 cos_factorsQ15_512[512] = {
  (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7ffe, (q15_t)0x7ffc, (q15_t)0x7ffb, (q15_t)0x7ff9, (q15_t)0x7ff7,
  (q15_t)0x7ff4, (q15_t)0x7ff2, (q15_t)0x7fee, (q15_t)0x7feb, (q15_t)0x7fe7, (q15_t)0x7fe3, (q15_t)0x7fdf, (q15_t)0x7fda,
  (q15_t)0x7fd6, (q15_t)0x7fd0, (q15_t)0x7fcb, (q15_t)0x7fc5, (q15_t)0x7fbf, (q15_t)0x7fb8, (q15_t)0x7fb1, (q15_t)0x7faa,
  (q15_t)0x7fa3, (q15_t)0x7f9b, (q15_t)0x7f93, (q15_t)0x7f8b, (q15_t)0x7f82, (q15_t)0x7f79, (q15_t)0x7f70, (q15_t)0x7f67,
  (q15_t)0x7f5d, (q15_t)0x7f53, (q15_t)0x7f48, (q15_t)0x7f3d, (q15_t)0x7f32, (q15_t)0x7f27, (q15_t)0x7f1b, (q15_t)0x7f0f,
  (q15_t)0x7f03, (q15_t)0x7ef6, (q15_t)0x7ee9, (q15_t)0x7edc, (q15_t)0x7ecf, (q15_t)0x7ec1, (q15_t)0x7eb3, (q15_t)0x7ea4,
  (q15_t)0x7e95, (q15_t)0x7e86, (q15_t)0x7e77, (q15_t)0x7e67, (q15_t)0x7e57, (q15_t)0x7e47, (q15_t)0x7e37, (q15_t)0x7e26,
  (q15_t)0x7e14, (q15_t)0x7e03, (q15_t)0x7df1, (q15_t)0x7ddf, (q15_t)0x7dcd, (q15_t)0x7dba, (q15_t)0x7da7, (q15_t)0x7d94,
  (q15_t)0x7d80, (q15_t)0x7d6c, (q15_t)0x7d58, (q15_t)0x7d43, (q15_t)0x7d2f, (q15_t)0x7d19, (q15_t)0x7d04, (q15_t)0x7cee,
  (q15_t)0x7cd8, (q15_t)0x7cc2, (q15_t)0x7cab, (q15_t)0x7c94, (q15_t)0x7c7d, (q15_t)0x7c66, (q15_t)0x7c4e, (q15_t)0x7c36,
  (q15_t)0x7c1d, (q15_t)0x7c05, (q15_t)0x7beb, (q15_t)0x7bd2, (q15_t)0x7bb9, (q15_t)0x7b9f, (q15_t)0x7b84, (q15_t)0x7b6a,
  (q15_t)0x7b4f, (q15_t)0x7b34, (q15_t)0x7b19, (q15_t)0x7afd, (q15_t)0x7ae1, (q15_t)0x7ac5, (q15_t)0x7aa8, (q15_t)0x7a8b,
  (q15_t)0x7a6e, (q15_t)0x7a50, (q15_t)0x7a33, (q15_t)0x7a15, (q15_t)0x79f6, (q15_t)0x79d8, (q15_t)0x79b9, (q15_t)0x7999,
  (q15_t)0x797a, (q15_t)0x795a, (q15_t)0x793a, (q15_t)0x7919, (q15_t)0x78f9, (q15_t)0x78d8, (q15_t)0x78b6, (q15_t)0x7895,
  (q15_t)0x7873, (q15_t)0x7851, (q15_t)0x782e, (q15_t)0x780c, (q15_t)0x77e9, (q15_t)0x77c5, (q15_t)0x77a2, (q15_t)0x777e,
  (q15_t)0x775a, (q15_t)0x7735, (q15_t)0x7710, (q15_t)0x76eb, (q15_t)0x76c6, (q15_t)0x76a0, (q15_t)0x767b, (q15_t)0x7654,
  (q15_t)0x762e, (q15_t)0x7607, (q15_t)0x75e0, (q15_t)0x75b9, (q15_t)0x7591, (q15_t)0x7569, (q15_t)0x7541, (q15_t)0x7519,
  (q15_t)0x74f0, (q15_t)0x74c7, (q15_t)0x749e, (q15_t)0x7474, (q15_t)0x744a, (q15_t)0x7420, (q15_t)0x73f6, (q15_t)0x73cb,
  (q15_t)0x73a0, (q15_t)0x7375, (q15_t)0x7349, (q15_t)0x731d, (q15_t)0x72f1, (q15_t)0x72c5, (q15_t)0x7298, (q15_t)0x726b,
  (q15_t)0x723e, (q15_t)0x7211, (q15_t)0x71e3, (q15_t)0x71b5, (q15_t)0x7186, (q15_t)0x7158, (q15_t)0x7129, (q15_t)0x70fa,
  (q15_t)0x70cb, (q15_t)0x709b, (q15_t)0x706b, (q15_t)0x703b, (q15_t)0x700a, (q15_t)0x6fda, (q15_t)0x6fa9, (q15_t)0x6f77,
  (q15_t)0x6f46, (q15_t)0x6f14, (q15_t)0x6ee2, (q15_t)0x6eaf, (q15_t)0x6e7d, (q15_t)0x6e4a, (q15_t)0x6e17, (q15_t)0x6de3,
  (q15_t)0x6db0, (q15_t)0x6d7c, (q15_t)0x6d48, (q15_t)0x6d13, (q15_t)0x6cde, (q15_t)0x6ca9, (q15_t)0x6c74, (q15_t)0x6c3f,
  (q15_t)0x6c09, (q15_t)0x6bd3, (q15_t)0x6b9c, (q15_t)0x6b66, (q15_t)0x6b2f, (q15_t)0x6af8, (q15_t)0x6ac1, (q15_t)0x6a89,
  (q15_t)0x6a51, (q15_t)0x6a19, (q15_t)0x69e1, (q15_t)0x69a8, (q15_t)0x696f, (q15_t)0x6936, (q15_t)0x68fd, (q15_t)0x68c3,
  (q15_t)0x6889, (q15_t)0x684f, (q15_t)0x6815, (q15_t)0x67da, (q15_t)0x679f, (q15_t)0x6764, (q15_t)0x6729, (q15_t)0x66ed,
  (q15_t)0x66b1, (q15_t)0x6675, (q15_t)0x6639, (q15_t)0x65fc, (q15_t)0x65bf, (q15_t)0x6582, (q15_t)0x6545, (q15_t)0x6507,
  (q15_t)0x64c9, (q15_t)0x648b, (q15_t)0x644d, (q15_t)0x640e, (q15_t)0x63cf, (q15_t)0x6390, (q15_t)0x6351, (q15_t)0x6311,
  (q15_t)0x62d2, (q15_t)0x6292, (q15_t)0x6251, (q15_t)0x6211, (q15_t)0x61d0, (q15_t)0x618f, (q15_t)0x614e, (q15_t)0x610d,
  (q15_t)0x60cb, (q15_t)0x6089, (q15_t)0x6047, (q15_t)0x6004, (q15_t)0x5fc2, (q15_t)0x5f7f, (q15_t)0x5f3c, (q15_t)0x5ef9,
  (q15_t)0x5eb5, (q15_t)0x5e71, (q15_t)0x5e2d, (q15_t)0x5de9, (q15_t)0x5da5, (q15_t)0x5d60, (q15_t)0x5d1b, (q15_t)0x5cd6,
  (q15_t)0x5c91, (q15_t)0x5c4b, (q15_t)0x5c06, (q15_t)0x5bc0, (q15_t)0x5b79, (q15_t)0x5b33, (q15_t)0x5aec, (q15_t)0x5aa5,
  (q15_t)0x5a5e, (q15_t)0x5a17, (q15_t)0x59d0, (q15_t)0x5988, (q15_t)0x5940, (q15_t)0x58f8, (q15_t)0x58af, (q15_t)0x5867,
  (q15_t)0x581e, (q15_t)0x57d5, (q15_t)0x578c, (q15_t)0x5742, (q15_t)0x56f9, (q15_t)0x56af, (q15_t)0x5665, (q15_t)0x561a,
  (q15_t)0x55d0, (q15_t)0x5585, (q15_t)0x553a, (q15_t)0x54ef, (q15_t)0x54a4, (q15_t)0x5458, (q15_t)0x540d, (q15_t)0x53c1,
  (q15_t)0x5375, (q15_t)0x5328, (q15_t)0x52dc, (q15_t)0x528f, (q15_t)0x5242, (q15_t)0x51f5, (q15_t)0x51a8, (q15_t)0x515a,
  (q15_t)0x510c, (q15_t)0x50bf, (q15_t)0x5070, (q15_t)0x5022, (q15_t)0x4fd4, (q15_t)0x4f85, (q15_t)0x4f36, (q15_t)0x4ee7,
  (q15_t)0x4e98, (q15_t)0x4e48, (q15_t)0x4df9, (q15_t)0x4da9, (q15_t)0x4d59, (q15_t)0x4d09, (q15_t)0x4cb8, (q15_t)0x4c68,
  (q15_t)0x4c17, (q15_t)0x4bc6, (q15_t)0x4b75, (q15_t)0x4b24, (q15_t)0x4ad2, (q15_t)0x4a81, (q15_t)0x4a2f, (q15_t)0x49dd,
  (q15_t)0x498a, (q15_t)0x4938, (q15_t)0x48e6, (q15_t)0x4893, (q15_t)0x4840, (q15_t)0x47ed, (q15_t)0x479a, (q15_t)0x4746,
  (q15_t)0x46f3, (q15_t)0x469f, (q15_t)0x464b, (q15_t)0x45f7, (q15_t)0x45a3, (q15_t)0x454e, (q15_t)0x44fa, (q15_t)0x44a5,
  (q15_t)0x4450, (q15_t)0x43fb, (q15_t)0x43a5, (q15_t)0x4350, (q15_t)0x42fa, (q15_t)0x42a5, (q15_t)0x424f, (q15_t)0x41f9,
  (q15_t)0x41a2, (q15_t)0x414c, (q15_t)0x40f6, (q15_t)0x409f, (q15_t)0x4048, (q15_t)0x3ff1, (q15_t)0x3f9a, (q15_t)0x3f43,
  (q15_t)0x3eeb, (q15_t)0x3e93, (q15_t)0x3e3c, (q15_t)0x3de4, (q15_t)0x3d8c, (q15_t)0x3d33, (q15_t)0x3cdb, (q15_t)0x3c83,
  (q15_t)0x3c2a, (q15_t)0x3bd1, (q15_t)0x3b78, (q15_t)0x3b1f, (q15_t)0x3ac6, (q15_t)0x3a6c, (q15_t)0x3a13, (q15_t)0x39b9,
  (q15_t)0x395f, (q15_t)0x3906, (q15_t)0x38ab, (q15_t)0x3851, (q15_t)0x37f7, (q15_t)0x379c, (q15_t)0x3742, (q15_t)0x36e7,
  (q15_t)0x368c, (q15_t)0x3631, (q15_t)0x35d6, (q15_t)0x357b, (q15_t)0x351f, (q15_t)0x34c4, (q15_t)0x3468, (q15_t)0x340c,
  (q15_t)0x33b0, (q15_t)0x3354, (q15_t)0x32f8, (q15_t)0x329c, (q15_t)0x3240, (q15_t)0x31e3, (q15_t)0x3186, (q15_t)0x312a,
  (q15_t)0x30cd, (q15_t)0x3070, (q15_t)0x3013, (q15_t)0x2fb5, (q15_t)0x2f58, (q15_t)0x2efb, (q15_t)0x2e9d, (q15_t)0x2e3f,
  (q15_t)0x2de2, (q15_t)0x2d84, (q15_t)0x2d26, (q15_t)0x2cc8, (q15_t)0x2c69, (q15_t)0x2c0b, (q15_t)0x2bad, (q15_t)0x2b4e,
  (q15_t)0x2aef, (q15_t)0x2a91, (q15_t)0x2a32, (q15_t)0x29d3, (q15_t)0x2974, (q15_t)0x2915, (q15_t)0x28b5, (q15_t)0x2856,
  (q15_t)0x27f6, (q15_t)0x2797, (q15_t)0x2737, (q15_t)0x26d8, (q15_t)0x2678, (q15_t)0x2618, (q15_t)0x25b8, (q15_t)0x2558,
  (q15_t)0x24f7, (q15_t)0x2497, (q15_t)0x2437, (q15_t)0x23d6, (q15_t)0x2376, (q15_t)0x2315, (q15_t)0x22b4, (q15_t)0x2254,
  (q15_t)0x21f3, (q15_t)0x2192, (q15_t)0x2131, (q15_t)0x20d0, (q15_t)0x206e, (q15_t)0x200d, (q15_t)0x1fac, (q15_t)0x1f4a,
  (q15_t)0x1ee9, (q15_t)0x1e87, (q15_t)0x1e25, (q15_t)0x1dc4, (q15_t)0x1d62, (q15_t)0x1d00, (q15_t)0x1c9e, (q15_t)0x1c3c,
  (q15_t)0x1bda, (q15_t)0x1b78, (q15_t)0x1b16, (q15_t)0x1ab3, (q15_t)0x1a51, (q15_t)0x19ef, (q15_t)0x198c, (q15_t)0x192a,
  (q15_t)0x18c7, (q15_t)0x1864, (q15_t)0x1802, (q15_t)0x179f, (q15_t)0x173c, (q15_t)0x16d9, (q15_t)0x1676, (q15_t)0x1613,
  (q15_t)0x15b0, (q15_t)0x154d, (q15_t)0x14ea, (q15_t)0x1487, (q15_t)0x1423, (q15_t)0x13c0, (q15_t)0x135d, (q15_t)0x12f9,
  (q15_t)0x1296, (q15_t)0x1232, (q15_t)0x11cf, (q15_t)0x116b, (q15_t)0x1108, (q15_t)0x10a4, (q15_t)0x1040, (q15_t)0xfdd,
  (q15_t)0xf79, (q15_t)0xf15, (q15_t)0xeb1, (q15_t)0xe4d, (q15_t)0xde9, (q15_t)0xd85, (q15_t)0xd21, (q15_t)0xcbd,
  (q15_t)0xc59, (q15_t)0xbf5, (q15_t)0xb91, (q15_t)0xb2d, (q15_t)0xac9, (q15_t)0xa65, (q15_t)0xa00, (q15_t)0x99c,
  (q15_t)0x938, (q15_t)0x8d4, (q15_t)0x86f, (q15_t)0x80b, (q15_t)0x7a7, (q15_t)0x742, (q15_t)0x6de, (q15_t)0x67a,
  (q15_t)0x615, (q15_t)0x5b1, (q15_t)0x54c, (q15_t)0x4e8, (q15_t)0x483, (q15_t)0x41f, (q15_t)0x3ba, (q15_t)0x356,
  (q15_t)0x2f1, (q15_t)0x28d, (q15_t)0x228, (q15_t)0x1c4, (q15_t)0x15f, (q15_t)0xfb, (q15_t)0x96, (q15_t)0x32
};

static const q15_t ALIGN4 cos_factorsQ15_2048[2048] = {
  (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff,
  (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffd, (q15_t)0x7ffd,
  (q15_t)0x7ffd, (q15_t)0x7ffd, (q15_t)0x7ffc, (q15_t)0x7ffc, (q15_t)0x7ffb, (q15_t)0x7ffb, (q15_t)0x7ffb, (q15_t)0x7ffa,
  (q15_t)0x7ffa, (q15_t)0x7ff9, (q15_t)0x7ff9, (q15_t)0x7ff8, (q15_t)0x7ff8, (q15_t)0x7ff7, (q15_t)0x7ff7, (q15_t)0x7ff6,
  (q15_t)0x7ff5, (q15_t)0x7ff5, (q15_t)0x7ff4, (q15_t)0x7ff3, (q15_t)0x7ff3, (q15_t)0x7ff2, (q15_t)0x7ff1, (q15_t)0x7ff0,
  (q15_t)0x7ff0, (q15_t)0x7fef, (q15_t)0x7fee, (q15_t)0x7fed, (q15_t)0x7fec, (q15_t)0x7fec, (q15_t)0x7feb, (q15_t)0x7fea,
  (q15_t)0x7fe9, (q15_t)0x7fe8, (q15_t)0x7fe7, (q15_t)0x7fe6, (q15_t)0x7fe5, (q15_t)0x7fe4, (q15_t)0x7fe3, (q15_t)0x7fe2,
  (q15_t)0x7fe1, (q15_t)0x7fe0, (q15_t)0x7fdf, (q15_t)0x7fdd, (q15_t)0x7fdc, (q15_t)0x7fdb, (q15_t)0x7fda, (q15_t)0x7fd9,
  (q15_t)0x7fd7, (q15_t)0x7fd6, (q15_t)0x7fd5, (q15_t)0x7fd4, (q15_t)0x7fd2, (q15_t)0x7fd1, (q15_t)0x7fd0, (q15_t)0x7fce,
  (q15_t)0x7fcd, (q15_t)0x7fcb, (q15_t)0x7fca, (q15_t)0x7fc9, (q15_t)0x7fc7, (q15_t)0x7fc6, (q15_t)0x7fc4, (q15_t)0x7fc3,
  (q15_t)0x7fc1, (q15_t)0x7fc0, (q15_t)0x7fbe, (q15_t)0x7fbc, (q15_t)0x7fbb, (q15_t)0x7fb9, (q15_t)0x7fb7, (q15_t)0x7fb6,
  (q15_t)0x7fb4, (q15_t)0x7fb2, (q15_t)0x7fb1, (q15_t)0x7faf, (q15_t)0x7fad, (q15_t)0x7fab, (q15_t)0x7fa9, (q15_t)0x7fa8,
  (q15_t)0x7fa6, (q15_t)0x7fa4, (q15_t)0x7fa2, (q15_t)0x7fa0, (q15_t)0x7f9e, (q15_t)0x7f9c, (q15_t)0x7f9a, (q15_t)0x7f98,
  (q15_t)0x7f96, (q15_t)0x7f94, (q15_t)0x7f92, (q15_t)0x7f90, (q15_t)0x7f8e, (q15_t)0x7f8c, (q15_t)0x7f8a, (q15_t)0x7f88,
  (q15_t)0x7f86, (q15_t)0x7f83, (q15_t)0x7f81, (q15_t)0x7f7f, (q15_t)0x7f7d, (q15_t)0x7f7b, (q15_t)0x7f78, (q15_t)0x7f76,
  (q15_t)0x7f74, (q15_t)0x7f71, (q15_t)0x7f6f, (q15_t)0x7f6d, (q15_t)0x7f6a, (q15_t)0x7f68, (q15_t)0x7f65, (q15_t)0x7f63,
  (q15_t)0x7f60, (q15_t)0x7f5e, (q15_t)0x7f5b, (q15_t)0x7f59, (q15_t)0x7f56, (q15_t)0x7f54, (q15_t)0x7f51, (q15_t)0x7f4f,
  (q15_t)0x7f4c, (q15_t)0x7f49, (q15_t)0x7f47, (q15_t)0x7f44, (q15_t)0x7f41, (q15_t)0x7f3f, (q15_t)0x7f3c, (q15_t)0x7f39,
  (q15_t)0x7f36, (q15_t)0x7f34, (q15_t)0x7f31, (q15_t)0x7f2e, (q15_t)0x7f2b, (q15_t)0x7f28, (q15_t)0x7f25, (q15_t)0x7f23,
  (q15_t)0x7f20, (q15_t)0x7f1d, (q15_t)0x7f1a, (q15_t)0x7f17, (q15_t)0x7f14, (q15_t)0x7f11, (q15_t)0x7f0e, (q15_t)0x7f0b,
  (q15_t)0x7f08, (q15_t)0x7f04, (q15_t)0x7f01, (q15_t)0x7efe, (q15_t)0x7efb, (q15_t)0x7ef8, (q15_t)0x7ef5, (q15_t)0x7ef1,
  (q15_t)0x7eee, (q15_t)0x7eeb, (q15_t)0x7ee8, (q15_t)0x7ee4, (q15_t)0x7ee1, (q15_t)0x7ede, (q15_t)0x7eda, (q15_t)0x7ed7,
  (q15_t)0x7ed4, (q15_t)0x7ed0, (q15_t)0x7ecd, (q15_t)0x7ec9, (q15_t)0x7ec6, (q15_t)0x7ec3, (q15_t)0x7ebf, (q15_t)0x7ebb,
  (q15_t)0x7eb8, (q15_t)0x7eb4, (q15_t)0x7eb1, (q15_t)0x7ead, (q15_t)0x7eaa, (q15_t)0x7ea6, (q15_t)0x7ea2, (q15_t)0x7e9f,
  (q15_t)0x7e9b, (q15_t)0x7e97, (q15_t)0x7e94, (q15_t)0x7e90, (q15_t)0x7e8c, (q15_t)0x7e88, (q15_t)0x7e84, (q15_t)0x7e81,
  (q15_t)0x7e7d, (q15_t)0x7e79, (q15_t)0x7e75, (q15_t)0x7e71, (q15_t)0x7e6d, (q15_t)0x7e69, (q15_t)0x7e65, (q15_t)0x7e61,
  (q15_t)0x7e5d, (q15_t)0x7e59, (q15_t)0x7e55, (q15_t)0x7e51, (q15_t)0x7e4d, (q15_t)0x7e49, (q15_t)0x7e45, (q15_t)0x7e41,
  (q15_t)0x7e3d, (q15_t)0x7e39, (q15_t)0x7e34, (q15_t)0x7e30, (q15_t)0x7e2c, (q15_t)0x7e28, (q15_t)0x7e24, (q15_t)0x7e1f,
  (q15_t)0x7e1b, (q15_t)0x7e17, (q15_t)0x7e12, (q15_t)0x7e0e, (q15_t)0x7e0a, (q15_t)0x7e05, (q15_t)0x7e01, (q15_t)0x7dfc,
  (q15_t)0x7df8, (q15_t)0x7df3, (q15_t)0x7def, (q15_t)0x7dea, (q15_t)0x7de6, (q15_t)0x7de1, (q15_t)0x7ddd, (q15_t)0x7dd8,
  (q15_t)0x7dd4, (q15_t)0x7dcf, (q15_t)0x7dca, (q15_t)0x7dc6, (q15_t)0x7dc1, (q15_t)0x7dbc, (q15_t)0x7db8, (q15_t)0x7db3,
  (q15_t)0x7dae, (q15_t)0x7da9, (q15_t)0x7da5, (q15_t)0x7da0, (q15_t)0x7d9b, (q15_t)0x7d96, (q15_t)0x7d91, (q15_t)0x7d8c,
  (q15_t)0x7d87, (q15_t)0x7d82, (q15_t)0x7d7e, (q15_t)0x7d79, (q15_t)0x7d74, (q15_t)0x7d6f, (q15_t)0x7d6a, (q15_t)0x7d65,
  (q15_t)0x7d60, (q15_t)0x7d5a, (q15_t)0x7d55, (q15_t)0x7d50, (q15_t)0x7d4b, (q15_t)0x7d46, (q15_t)0x7d41, (q15_t)0x7d3c,
  (q15_t)0x7d36, (q15_t)0x7d31, (q15_t)0x7d2c, (q15_t)0x7d27, (q15_t)0x7d21, (q15_t)0x7d1c, (q15_t)0x7d17, (q15_t)0x7d11,
  (q15_t)0x7d0c, (q15_t)0x7d07, (q15_t)0x7d01, (q15_t)0x7cfc, (q15_t)0x7cf6, (q15_t)0x7cf1, (q15_t)0x7cec, (q15_t)0x7ce6,
  (q15_t)0x7ce1, (q15_t)0x7cdb, (q15_t)0x7cd5, (q15_t)0x7cd0, (q15_t)0x7cca, (q15_t)0x7cc5, (q15_t)0x7cbf, (q15_t)0x7cb9,
  (q15_t)0x7cb4, (q15_t)0x7cae, (q15_t)0x7ca8, (q15_t)0x7ca3, (q15_t)0x7c9d, (q15_t)0x7c97, (q15_t)0x7c91, (q15_t)0x7c8c,
  (q15_t)0x7c86, (q15_t)0x7c80, (q15_t)0x7c7a, (q15_t)0x7c74, (q15_t)0x7c6e, (q15_t)0x7c69, (q15_t)0x7c63, (q15_t)0x7c5d,
  (q15_t)0x7c57, (q15_t)0x7c51, (q15_t)0x7c4b, (q15_t)0x7c45, (q15_t)0x7c3f, (q15_t)0x7c39, (q15_t)0x7c33, (q15_t)0x7c2d,
  (q15_t)0x7c26, (q15_t)0x7c20, (q15_t)0x7c1a, (q15_t)0x7c14, (q15_t)0x7c0e, (q15_t)0x7c08, (q15_t)0x7c01, (q15_t)0x7bfb,
  (q15_t)0x7bf5, (q15_t)0x7bef, (q15_t)0x7be8, (q15_t)0x7be2, (q15_t)0x7bdc, (q15_t)0x7bd5, (q15_t)0x7bcf, (q15_t)0x7bc9,
  (q15_t)0x7bc2, (q15_t)0x7bbc, (q15_t)0x7bb5, (q15_t)0x7baf, (q15_t)0x7ba8, (q15_t)0x7ba2, (q15_t)0x7b9b, (q15_t)0x7b95,
  (q15_t)0x7b8e, (q15_t)0x7b88, (q15_t)0x7b81, (q15_t)0x7b7a, (q15_t)0x7b74, (q15_t)0x7b6d, (q15_t)0x7b67, (q15_t)0x7b60,
  (q15_t)0x7b59, (q15_t)0x7b52, (q15_t)0x7b4c, (q15_t)0x7b45, (q15_t)0x7b3e, (q15_t)0x7b37, (q15_t)0x7b31, (q15_t)0x7b2a,
  (q15_t)0x7b23, (q15_t)0x7b1c, (q15_t)0x7b15, (q15_t)0x7b0e, (q15_t)0x7b07, (q15_t)0x7b00, (q15_t)0x7af9, (q15_t)0x7af2,
  (q15_t)0x7aeb, (q15_t)0x7ae4, (q15_t)0x7add, (q15_t)0x7ad6, (q15_t)0x7acf, (q15_t)0x7ac8, (q15_t)0x7ac1, (q15_t)0x7aba,
  (q15_t)0x7ab3, (q15_t)0x7aac, (q15_t)0x7aa4, (q15_t)0x7a9d, (q15_t)0x7a96, (q15_t)0x7a8f, (q15_t)0x7a87, (q15_t)0x7a80,
  (q15_t)0x7a79, (q15_t)0x7a72, (q15_t)0x7a6a, (q15_t)0x7a63, (q15_t)0x7a5c, (q15_t)0x7a54, (q15_t)0x7a4d, (q15_t)0x7a45,
  (q15_t)0x7a3e, (q15_t)0x7a36, (q15_t)0x7a2f, (q15_t)0x7a27, (q15_t)0x7a20, (q15_t)0x7a18, (q15_t)0x7a11, (q15_t)0x7a09,
  (q15_t)0x7a02, (q15_t)0x79fa, (q15_t)0x79f2, (q15_t)0x79eb, (q15_t)0x79e3, (q15_t)0x79db, (q15_t)0x79d4, (q15_t)0x79cc,
  (q15_t)0x79c4, (q15_t)0x79bc, (q15_t)0x79b5, (q15_t)0x79ad, (q15_t)0x79a5, (q15_t)0x799d, (q15_t)0x7995, (q15_t)0x798e,
  (q15_t)0x7986, (q15_t)0x797e, (q15_t)0x7976, (q15_t)0x796e, (q15_t)0x7966, (q15_t)0x795e, (q15_t)0x7956, (q15_t)0x794e,
  (q15_t)0x7946, (q15_t)0x793e, (q15_t)0x7936, (q15_t)0x792e, (q15_t)0x7926, (q15_t)0x791e, (q15_t)0x7915, (q15_t)0x790d,
  (q15_t)0x7905, (q15_t)0x78fd, (q15_t)0x78f5, (q15_t)0x78ec, (q15_t)0x78e4, (q15_t)0x78dc, (q15_t)0x78d4, (q15_t)0x78cb,
  (q15_t)0x78c3, (q15_t)0x78bb, (q15_t)0x78b2, (q15_t)0x78aa, (q15_t)0x78a2, (q15_t)0x7899, (q15_t)0x7891, (q15_t)0x7888,
  (q15_t)0x7880, (q15_t)0x7877, (q15_t)0x786f, (q15_t)0x7866, (q15_t)0x785e, (q15_t)0x7855, (q15_t)0x784d, (q15_t)0x7844,
  (q15_t)0x783b, (q15_t)0x7833, (q15_t)0x782a, (q15_t)0x7821, (q15_t)0x7819, (q15_t)0x7810, (q15_t)0x7807, (q15_t)0x77ff,
  (q15_t)0x77f6, (q15_t)0x77ed, (q15_t)0x77e4, (q15_t)0x77db, (q15_t)0x77d3, (q15_t)0x77ca, (q15_t)0x77c1, (q15_t)0x77b8,
  (q15_t)0x77af, (q15_t)0x77a6, (q15_t)0x779d, (q15_t)0x7794, (q15_t)0x778b, (q15_t)0x7782, (q15_t)0x7779, (q15_t)0x7770,
  (q15_t)0x7767, (q15_t)0x775e, (q15_t)0x7755, (q15_t)0x774c, (q15_t)0x7743, (q15_t)0x773a, (q15_t)0x7731, (q15_t)0x7727,
  (q15_t)0x771e, (q15_t)0x7715, (q15_t)0x770c, (q15_t)0x7703, (q15_t)0x76f9, (q15_t)0x76f0, (q15_t)0x76e7, (q15_t)0x76dd,
  (q15_t)0x76d4, (q15_t)0x76cb, (q15_t)0x76c1, (q15_t)0x76b8, (q15_t)0x76af, (q15_t)0x76a5, (q15_t)0x769c, (q15_t)0x7692,
  (q15_t)0x7689, (q15_t)0x767f, (q15_t)0x7676, (q15_t)0x766c, (q15_t)0x7663, (q15_t)0x7659, (q15_t)0x7650, (q15_t)0x7646,
  (q15_t)0x763c, (q15_t)0x7633, (q15_t)0x7629, (q15_t)0x761f, (q15_t)0x7616, (q15_t)0x760c, (q15_t)0x7602, (q15_t)0x75f9,
  (q15_t)0x75ef, (q15_t)0x75e5, (q15_t)0x75db, (q15_t)0x75d1, (q15_t)0x75c8, (q15_t)0x75be, (q15_t)0x75b4, (q15_t)0x75aa,
  (q15_t)0x75a0, (q15_t)0x7596, (q15_t)0x758c, (q15_t)0x7582, (q15_t)0x7578, (q15_t)0x756e, (q15_t)0x7564, (q15_t)0x755a,
  (q15_t)0x7550, (q15_t)0x7546, (q15_t)0x753c, (q15_t)0x7532, (q15_t)0x7528, (q15_t)0x751e, (q15_t)0x7514, (q15_t)0x7509,
  (q15_t)0x74ff, (q15_t)0x74f5, (q15_t)0x74eb, (q15_t)0x74e1, (q15_t)0x74d6, (q15_t)0x74cc, (q15_t)0x74c2, (q15_t)0x74b7,
  (q15_t)0x74ad, (q15_t)0x74a3, (q15_t)0x7498, (q15_t)0x748e, (q15_t)0x7484, (q15_t)0x7479, (q15_t)0x746f, (q15_t)0x7464,
  (q15_t)0x745a, (q15_t)0x744f, (q15_t)0x7445, (q15_t)0x743a, (q15_t)0x7430, (q15_t)0x7425, (q15_t)0x741b, (q15_t)0x7410,
  (q15_t)0x7406, (q15_t)0x73fb, (q15_t)0x73f0, (q15_t)0x73e6, (q15_t)0x73db, (q15_t)0x73d0, (q15_t)0x73c6, (q15_t)0x73bb,
  (q15_t)0x73b0, (q15_t)0x73a5, (q15_t)0x739b, (q15_t)0x7390, (q15_t)0x7385, (q15_t)0x737a, (q15_t)0x736f, (q15_t)0x7364,
  (q15_t)0x7359, (q15_t)0x734f, (q15_t)0x7344, (q15_t)0x7339, (q15_t)0x732e, (q15_t)0x7323, (q15_t)0x7318, (q15_t)0x730d,
  (q15_t)0x7302, (q15_t)0x72f7, (q15_t)0x72ec, (q15_t)0x72e1, (q15_t)0x72d5, (q15_t)0x72ca, (q15_t)0x72bf, (q15_t)0x72b4,
  (q15_t)0x72a9, (q15_t)0x729e, (q15_t)0x7293, (q15_t)0x7287, (q15_t)0x727c, (q15_t)0x7271, (q15_t)0x7266, (q15_t)0x725a,
  (q15_t)0x724f, (q15_t)0x7244, (q15_t)0x7238, (q15_t)0x722d, (q15_t)0x7222, (q15_t)0x7216, (q15_t)0x720b, (q15_t)0x71ff,
  (q15_t)0x71f4, (q15_t)0x71e9, (q15_t)0x71dd, (q15_t)0x71d2, (q15_t)0x71c6, (q15_t)0x71bb, (q15_t)0x71af, (q15_t)0x71a3,
  (q15_t)0x7198, (q15_t)0x718c, (q15_t)0x7181, (q15_t)0x7175, (q15_t)0x7169, (q15_t)0x715e, (q15_t)0x7152, (q15_t)0x7146,
  (q15_t)0x713b, (q15_t)0x712f, (q15_t)0x7123, (q15_t)0x7117, (q15_t)0x710c, (q15_t)0x7100, (q15_t)0x70f4, (q15_t)0x70e8,
  (q15_t)0x70dc, (q15_t)0x70d1, (q15_t)0x70c5, (q15_t)0x70b9, (q15_t)0x70ad, (q15_t)0x70a1, (q15_t)0x7095, (q15_t)0x7089,
  (q15_t)0x707d, (q15_t)0x7071, (q15_t)0x7065, (q15_t)0x7059, (q15_t)0x704d, (q15_t)0x7041, (q15_t)0x7035, (q15_t)0x7029,
  (q15_t)0x701d, (q15_t)0x7010, (q15_t)0x7004, (q15_t)0x6ff8, (q15_t)0x6fec, (q15_t)0x6fe0, (q15_t)0x6fd3, (q15_t)0x6fc7,
  (q15_t)0x6fbb, (q15_t)0x6faf, (q15_t)0x6fa2, (q15_t)0x6f96, (q15_t)0x6f8a, (q15_t)0x6f7d, (q15_t)0x6f71, (q15_t)0x6f65,
  (q15_t)0x6f58, (q15_t)0x6f4c, (q15_t)0x6f3f, (q15_t)0x6f33, (q15_t)0x6f27, (q15_t)0x6f1a, (q15_t)0x6f0e, (q15_t)0x6f01,
  (q15_t)0x6ef5, (q15_t)0x6ee8, (q15_t)0x6edc, (q15_t)0x6ecf, (q15_t)0x6ec2, (q15_t)0x6eb6, (q15_t)0x6ea9, (q15_t)0x6e9c,
  (q15_t)0x6e90, (q15_t)0x6e83, (q15_t)0x6e76, (q15_t)0x6e6a, (q15_t)0x6e5d, (q15_t)0x6e50, (q15_t)0x6e44, (q15_t)0x6e37,
  (q15_t)0x6e2a, (q15_t)0x6e1d, (q15_t)0x6e10, (q15_t)0x6e04, (q15_t)0x6df7, (q15_t)0x6dea, (q15_t)0x6ddd, (q15_t)0x6dd0,
  (q15_t)0x6dc3, (q15_t)0x6db6, (q15_t)0x6da9, (q15_t)0x6d9c, (q15_t)0x6d8f, (q15_t)0x6d82, (q15_t)0x6d75, (q15_t)0x6d68,
  (q15_t)0x6d5b, (q15_t)0x6d4e, (q15_t)0x6d41, (q15_t)0x6d34, (q15_t)0x6d27, (q15_t)0x6d1a, (q15_t)0x6d0c, (q15_t)0x6cff,
  (q15_t)0x6cf2, (q15_t)0x6ce5, (q15_t)0x6cd8, (q15_t)0x6cca, (q15_t)0x6cbd, (q15_t)0x6cb0, (q15_t)0x6ca3, (q15_t)0x6c95,
  (q15_t)0x6c88, (q15_t)0x6c7b, (q15_t)0x6c6d, (q15_t)0x6c60, (q15_t)0x6c53, (q15_t)0x6c45, (q15_t)0x6c38, (q15_t)0x6c2a,
  (q15_t)0x6c1d, (q15_t)0x6c0f, (q15_t)0x6c02, (q15_t)0x6bf5, (q15_t)0x6be7, (q15_t)0x6bd9, (q15_t)0x6bcc, (q15_t)0x6bbe,
  (q15_t)0x6bb1, (q15_t)0x6ba3, (q15_t)0x6b96, (q15_t)0x6b88, (q15_t)0x6b7a, (q15_t)0x6b6d, (q15_t)0x6b5f, (q15_t)0x6b51,
  (q15_t)0x6b44, (q15_t)0x6b36, (q15_t)0x6b28, (q15_t)0x6b1a, (q15_t)0x6b0d, (q15_t)0x6aff, (q15_t)0x6af1, (q15_t)0x6ae3,
  (q15_t)0x6ad5, (q15_t)0x6ac8, (q15_t)0x6aba, (q15_t)0x6aac, (q15_t)0x6a9e, (q15_t)0x6a90, (q15_t)0x6a82, (q15_t)0x6a74,
  (q15_t)0x6a66, (q15_t)0x6a58, (q15_t)0x6a4a, (q15_t)0x6a3c, (q15_t)0x6a2e, (q15_t)0x6a20, (q15_t)0x6a12, (q15_t)0x6a04,
  (q15_t)0x69f6, (q15_t)0x69e8, (q15_t)0x69da, (q15_t)0x69cb, (q15_t)0x69bd, (q15_t)0x69af, (q15_t)0x69a1, (q15_t)0x6993,
  (q15_t)0x6985, (q15_t)0x6976, (q15_t)0x6968, (q15_t)0x695a, (q15_t)0x694b, (q15_t)0x693d, (q15_t)0x692f, (q15_t)0x6921,
  (q15_t)0x6912, (q15_t)0x6904, (q15_t)0x68f5, (q15_t)0x68e7, (q15_t)0x68d9, (q15_t)0x68ca, (q15_t)0x68bc, (q15_t)0x68ad,
  (q15_t)0x689f, (q15_t)0x6890, (q15_t)0x6882, (q15_t)0x6873, (q15_t)0x6865, (q15_t)0x6856, (q15_t)0x6848, (q15_t)0x6839,
  (q15_t)0x682b, (q15_t)0x681c, (q15_t)0x680d, (q15_t)0x67ff, (q15_t)0x67f0, (q15_t)0x67e1, (q15_t)0x67d3, (q15_t)0x67c4,
  (q15_t)0x67b5, (q15_t)0x67a6, (q15_t)0x6798, (q15_t)0x6789, (q15_t)0x677a, (q15_t)0x676b, (q15_t)0x675d, (q15_t)0x674e,
  (q15_t)0x673f, (q15_t)0x6730, (q15_t)0x6721, (q15_t)0x6712, (q15_t)0x6703, (q15_t)0x66f4, (q15_t)0x66e5, (q15_t)0x66d6,
  (q15_t)0x66c8, (q15_t)0x66b9, (q15_t)0x66aa, (q15_t)0x669b, (q15_t)0x668b, (q15_t)0x667c, (q15_t)0x666d, (q15_t)0x665e,
  (q15_t)0x664f, (q15_t)0x6640, (q15_t)0x6631, (q15_t)0x6622, (q15_t)0x6613, (q15_t)0x6603, (q15_t)0x65f4, (q15_t)0x65e5,
  (q15_t)0x65d6, (q15_t)0x65c7, (q15_t)0x65b7, (q15_t)0x65a8, (q15_t)0x6599, (q15_t)0x658a, (q15_t)0x657a, (q15_t)0x656b,
  (q15_t)0x655c, (q15_t)0x654c, (q15_t)0x653d, (q15_t)0x652d, (q15_t)0x651e, (q15_t)0x650f, (q15_t)0x64ff, (q15_t)0x64f0,
  (q15_t)0x64e0, (q15_t)0x64d1, (q15_t)0x64c1, (q15_t)0x64b2, (q15_t)0x64a2, (q15_t)0x6493, (q15_t)0x6483, (q15_t)0x6474,
  (q15_t)0x6464, (q15_t)0x6454, (q15_t)0x6445, (q15_t)0x6435, (q15_t)0x6426, (q15_t)0x6416, (q15_t)0x6406, (q15_t)0x63f7,
  (q15_t)0x63e7, (q15_t)0x63d7, (q15_t)0x63c7, (q15_t)0x63b8, (q15_t)0x63a8, (q15_t)0x6398, (q15_t)0x6388, (q15_t)0x6378,
  (q15_t)0x6369, (q15_t)0x6359, (q15_t)0x6349, (q15_t)0x6339, (q15_t)0x6329, (q15_t)0x6319, (q15_t)0x6309, (q15_t)0x62f9,
  (q15_t)0x62ea, (q15_t)0x62da, (q15_t)0x62ca, (q15_t)0x62ba, (q15_t)0x62aa, (q15_t)0x629a, (q15_t)0x628a, (q15_t)0x627a,
  (q15_t)0x6269, (q15_t)0x6259, (q15_t)0x6249, (q15_t)0x6239, (q15_t)0x6229, (q15_t)0x6219, (q15_t)0x6209, (q15_t)0x61f9,
  (q15_t)0x61e8, (q15_t)0x61d8, (q15_t)0x61c8, (q15_t)0x61b8, (q15_t)0x61a8, (q15_t)0x6197, (q15_t)0x6187, (q15_t)0x6177,
  (q15_t)0x6166, (q15_t)0x6156, (q15_t)0x6146, (q15_t)0x6135, (q15_t)0x6125, (q15_t)0x6115, (q15_t)0x6104, (q15_t)0x60f4,
  (q15_t)0x60e4, (q15_t)0x60d3, (q15_t)0x60c3, (q15_t)0x60b2, (q15_t)0x60a2, (q15_t)0x6091, (q15_t)0x6081, (q15_t)0x6070,
  (q15_t)0x6060, (q15_t)0x604f, (q15_t)0x603f, (q15_t)0x602e, (q15_t)0x601d, (q15_t)0x600d, (q15_t)0x5ffc, (q15_t)0x5fec,
  (q15_t)0x5fdb, (q15_t)0x5fca, (q15_t)0x5fba, (q15_t)0x5fa9, (q15_t)0x5f98, (q15_t)0x5f87, (q15_t)0x5f77, (q15_t)0x5f66,
  (q15_t)0x5f55, (q15_t)0x5f44, (q15_t)0x5f34, (q15_t)0x5f23, (q15_t)0x5f12, (q15_t)0x5f01, (q15_t)0x5ef0, (q15_t)0x5edf,
  (q15_t)0x5ecf, (q15_t)0x5ebe, (q15_t)0x5ead, (q15_t)0x5e9c, (q15_t)0x5e8b, (q15_t)0x5e7a, (q15_t)0x5e69, (q15_t)0x5e58,
  (q15_t)0x5e47, (q15_t)0x5e36, (q15_t)0x5e25, (q15_t)0x5e14, (q15_t)0x5e03, (q15_t)0x5df2, (q15_t)0x5de1, (q15_t)0x5dd0,
  (q15_t)0x5dbf, (q15_t)0x5dad, (q15_t)0x5d9c, (q15_t)0x5d8b, (q15_t)0x5d7a, (q15_t)0x5d69, (q15_t)0x5d58, (q15_t)0x5d46,
  (q15_t)0x5d35, (q15_t)0x5d24, (q15_t)0x5d13, (q15_t)0x5d01, (q15_t)0x5cf0, (q15_t)0x5cdf, (q15_t)0x5cce, (q15_t)0x5cbc,
  (q15_t)0x5cab, (q15_t)0x5c9a, (q15_t)0x5c88, (q15_t)0x5c77, (q15_t)0x5c66, (q15_t)0x5c54, (q15_t)0x5c43, (q15_t)0x5c31,
  (q15_t)0x5c20, (q15_t)0x5c0e, (q15_t)0x5bfd, (q15_t)0x5beb, (q15_t)0x5bda, (q15_t)0x5bc8, (q15_t)0x5bb7, (q15_t)0x5ba5,
  (q15_t)0x5b94, (q15_t)0x5b82, (q15_t)0x5b71, (q15_t)0x5b5f, (q15_t)0x5b4d, (q15_t)0x5b3c, (q15_t)0x5b2a, (q15_t)0x5b19,
  (q15_t)0x5b07, (q15_t)0x5af5, (q15_t)0x5ae4, (q15_t)0x5ad2, (q15_t)0x5ac0, (q15_t)0x5aae, (q15_t)0x5a9d, (q15_t)0x5a8b,
  (q15_t)0x5a79, (q15_t)0x5a67, (q15_t)0x5a56, (q15_t)0x5a44, (q15_t)0x5a32, (q15_t)0x5a20, (q15_t)0x5a0e, (q15_t)0x59fc,
  (q15_t)0x59ea, (q15_t)0x59d9, (q15_t)0x59c7, (q15_t)0x59b5, (q15_t)0x59a3, (q15_t)0x5991, (q15_t)0x597f, (q15_t)0x596d,
  (q15_t)0x595b, (q15_t)0x5949, (q15_t)0x5937, (q15_t)0x5925, (q15_t)0x5913, (q15_t)0x5901, (q15_t)0x58ef, (q15_t)0x58dd,
  (q15_t)0x58cb, (q15_t)0x58b8, (q15_t)0x58a6, (q15_t)0x5894, (q15_t)0x5882, (q15_t)0x5870, (q15_t)0x585e, (q15_t)0x584b,
  (q15_t)0x5839, (q15_t)0x5827, (q15_t)0x5815, (q15_t)0x5803, (q15_t)0x57f0, (q15_t)0x57de, (q15_t)0x57cc, (q15_t)0x57b9,
  (q15_t)0x57a7, (q15_t)0x5795, (q15_t)0x5783, (q15_t)0x5770, (q15_t)0x575e, (q15_t)0x574b, (q15_t)0x5739, (q15_t)0x5727,
  (q15_t)0x5714, (q15_t)0x5702, (q15_t)0x56ef, (q15_t)0x56dd, (q15_t)0x56ca, (q15_t)0x56b8, (q15_t)0x56a5, (q15_t)0x5693,
  (q15_t)0x5680, (q15_t)0x566e, (q15_t)0x565b, (q15_t)0x5649, (q15_t)0x5636, (q15_t)0x5624, (q15_t)0x5611, (q15_t)0x55fe,
  (q15_t)0x55ec, (q15_t)0x55d9, (q15_t)0x55c7, (q15_t)0x55b4, (q15_t)0x55a1, (q15_t)0x558f, (q15_t)0x557c, (q15_t)0x5569,
  (q15_t)0x5556, (q15_t)0x5544, (q15_t)0x5531, (q15_t)0x551e, (q15_t)0x550b, (q15_t)0x54f9, (q15_t)0x54e6, (q15_t)0x54d3,
  (q15_t)0x54c0, (q15_t)0x54ad, (q15_t)0x549a, (q15_t)0x5488, (q15_t)0x5475, (q15_t)0x5462, (q15_t)0x544f, (q15_t)0x543c,
  (q15_t)0x5429, (q15_t)0x5416, (q15_t)0x5403, (q15_t)0x53f0, (q15_t)0x53dd, (q15_t)0x53ca, (q15_t)0x53b7, (q15_t)0x53a4,
  (q15_t)0x5391, (q15_t)0x537e, (q15_t)0x536b, (q15_t)0x5358, (q15_t)0x5345, (q15_t)0x5332, (q15_t)0x531f, (q15_t)0x530c,
  (q15_t)0x52f8, (q15_t)0x52e5, (q15_t)0x52d2, (q15_t)0x52bf, (q15_t)0x52ac, (q15_t)0x5299, (q15_t)0x5285, (q15_t)0x5272,
  (q15_t)0x525f, (q15_t)0x524c, (q15_t)0x5238, (q15_t)0x5225, (q15_t)0x5212, (q15_t)0x51ff, (q15_t)0x51eb, (q15_t)0x51d8,
  (q15_t)0x51c5, (q15_t)0x51b1, (q15_t)0x519e, (q15_t)0x518b, (q15_t)0x5177, (q15_t)0x5164, (q15_t)0x5150, (q15_t)0x513d,
  (q15_t)0x512a, (q15_t)0x5116, (q15_t)0x5103, (q15_t)0x50ef, (q15_t)0x50dc, (q15_t)0x50c8, (q15_t)0x50b5, (q15_t)0x50a1,
  (q15_t)0x508e, (q15_t)0x507a, (q15_t)0x5067, (q15_t)0x5053, (q15_t)0x503f, (q15_t)0x502c, (q15_t)0x5018, (q15_t)0x5005,
  (q15_t)0x4ff1, (q15_t)0x4fdd, (q15_t)0x4fca, (q15_t)0x4fb6, (q15_t)0x4fa2, (q15_t)0x4f8f, (q15_t)0x4f7b, (q15_t)0x4f67,
  (q15_t)0x4f54, (q15_t)0x4f40, (q15_t)0x4f2c, (q15_t)0x4f18, (q15_t)0x4f05, (q15_t)0x4ef1, (q15_t)0x4edd, (q15_t)0x4ec9,
  (q15_t)0x4eb6, (q15_t)0x4ea2, (q15_t)0x4e8e, (q15_t)0x4e7a, (q15_t)0x4e66, (q15_t)0x4e52, (q15_t)0x4e3e, (q15_t)0x4e2a,
  (q15_t)0x4e17, (q15_t)0x4e03, (q15_t)0x4def, (q15_t)0x4ddb, (q15_t)0x4dc7, (q15_t)0x4db3, (q15_t)0x4d9f, (q15_t)0x4d8b,
  (q15_t)0x4d77, (q15_t)0x4d63, (q15_t)0x4d4f, (q15_t)0x4d3b, (q15_t)0x4d27, (q15_t)0x4d13, (q15_t)0x4cff, (q15_t)0x4ceb,
  (q15_t)0x4cd6, (q15_t)0x4cc2, (q15_t)0x4cae, (q15_t)0x4c9a, (q15_t)0x4c86, (q15_t)0x4c72, (q15_t)0x4c5e, (q15_t)0x4c49,
  (q15_t)0x4c35, (q15_t)0x4c21, (q15_t)0x4c0d, (q15_t)0x4bf9, (q15_t)0x4be4, (q15_t)0x4bd0, (q15_t)0x4bbc, (q15_t)0x4ba8,
  (q15_t)0x4b93, (q15_t)0x4b7f, (q15_t)0x4b6b, (q15_t)0x4b56, (q15_t)0x4b42, (q15_t)0x4b2e, (q15_t)0x4b19, (q15_t)0x4b05,
  (q15_t)0x4af1, (q15_t)0x4adc, (q15_t)0x4ac8, (q15_t)0x4ab4, (q15_t)0x4a9f, (q15_t)0x4a8b, (q15_t)0x4a76, (q15_t)0x4a62,
  (q15_t)0x4a4d, (q15_t)0x4a39, (q15_t)0x4a24, (q15_t)0x4a10, (q15_t)0x49fb, (q15_t)0x49e7, (q15_t)0x49d2, (q15_t)0x49be,
  (q15_t)0x49a9, (q15_t)0x4995, (q15_t)0x4980, (q15_t)0x496c, (q15_t)0x4957, (q15_t)0x4942, (q15_t)0x492e, (q15_t)0x4919,
  (q15_t)0x4905, (q15_t)0x48f0, (q15_t)0x48db, (q15_t)0x48c7, (q15_t)0x48b2, (q15_t)0x489d, (q15_t)0x4888, (q15_t)0x4874,
  (q15_t)0x485f, (q15_t)0x484a, (q15_t)0x4836, (q15_t)0x4821, (q15_t)0x480c, (q15_t)0x47f7, (q15_t)0x47e2, (q15_t)0x47ce,
  (q15_t)0x47b9, (q15_t)0x47a4, (q15_t)0x478f, (q15_t)0x477a, (q15_t)0x4765, (q15_t)0x4751, (q15_t)0x473c, (q15_t)0x4727,
  (q15_t)0x4712, (q15_t)0x46fd, (q15_t)0x46e8, (q15_t)0x46d3, (q15_t)0x46be, (q15_t)0x46a9, (q15_t)0x4694, (q15_t)0x467f,
  (q15_t)0x466a, (q15_t)0x4655, (q15_t)0x4640, (q15_t)0x462b, (q15_t)0x4616, (q15_t)0x4601, (q15_t)0x45ec, (q15_t)0x45d7,
  (q15_t)0x45c2, (q15_t)0x45ad, (q15_t)0x4598, (q15_t)0x4583, (q15_t)0x456e, (q15_t)0x4559, (q15_t)0x4544, (q15_t)0x452e,
  (q15_t)0x4519, (q15_t)0x4504, (q15_t)0x44ef, (q15_t)0x44da, (q15_t)0x44c5, (q15_t)0x44af, (q15_t)0x449a, (q15_t)0x4485,
  (q15_t)0x4470, (q15_t)0x445a, (q15_t)0x4445, (q15_t)0x4430, (q15_t)0x441b, (q15_t)0x4405, (q15_t)0x43f0, (q15_t)0x43db,
  (q15_t)0x43c5, (q15_t)0x43b0, (q15_t)0x439b, (q15_t)0x4385, (q15_t)0x4370, (q15_t)0x435b, (q15_t)0x4345, (q15_t)0x4330,
  (q15_t)0x431b, (q15_t)0x4305, (q15_t)0x42f0, (q15_t)0x42da, (q15_t)0x42c5, (q15_t)0x42af, (q15_t)0x429a, (q15_t)0x4284,
  (q15_t)0x426f, (q15_t)0x425a, (q15_t)0x4244, (q15_t)0x422f, (q15_t)0x4219, (q15_t)0x4203, (q15_t)0x41ee, (q15_t)0x41d8,
  (q15_t)0x41c3, (q15_t)0x41ad, (q15_t)0x4198, (q15_t)0x4182, (q15_t)0x416d, (q15_t)0x4157, (q15_t)0x4141, (q15_t)0x412c,
  (q15_t)0x4116, (q15_t)0x4100, (q15_t)0x40eb, (q15_t)0x40d5, (q15_t)0x40bf, (q15_t)0x40aa, (q15_t)0x4094, (q15_t)0x407e,
  (q15_t)0x4069, (q15_t)0x4053, (q15_t)0x403d, (q15_t)0x4027, (q15_t)0x4012, (q15_t)0x3ffc, (q15_t)0x3fe6, (q15_t)0x3fd0,
  (q15_t)0x3fbb, (q15_t)0x3fa5, (q15_t)0x3f8f, (q15_t)0x3f79, (q15_t)0x3f63, (q15_t)0x3f4d, (q15_t)0x3f38, (q15_t)0x3f22,
  (q15_t)0x3f0c, (q15_t)0x3ef6, (q15_t)0x3ee0, (q15_t)0x3eca, (q15_t)0x3eb4, (q15_t)0x3e9e, (q15_t)0x3e88, (q15_t)0x3e73,
  (q15_t)0x3e5d, (q15_t)0x3e47, (q15_t)0x3e31, (q15_t)0x3e1b, (q15_t)0x3e05, (q15_t)0x3def, (q15_t)0x3dd9, (q15_t)0x3dc3,
  (q15_t)0x3dad, (q15_t)0x3d97, (q15_t)0x3d81, (q15_t)0x3d6b, (q15_t)0x3d55, (q15_t)0x3d3e, (q15_t)0x3d28, (q15_t)0x3d12,
  (q15_t)0x3cfc, (q15_t)0x3ce6, (q15_t)0x3cd0, (q15_t)0x3cba, (q15_t)0x3ca4, (q15_t)0x3c8e, (q15_t)0x3c77, (q15_t)0x3c61,
  (q15_t)0x3c4b, (q15_t)0x3c35, (q15_t)0x3c1f, (q15_t)0x3c09, (q15_t)0x3bf2, (q15_t)0x3bdc, (q15_t)0x3bc6, (q15_t)0x3bb0,
  (q15_t)0x3b99, (q15_t)0x3b83, (q15_t)0x3b6d, (q15_t)0x3b57, (q15_t)0x3b40, (q15_t)0x3b2a, (q15_t)0x3b14, (q15_t)0x3afe,
  (q15_t)0x3ae7, (q15_t)0x3ad1, (q15_t)0x3abb, (q15_t)0x3aa4, (q15_t)0x3a8e, (q15_t)0x3a78, (q15_t)0x3a61, (q15_t)0x3a4b,
  (q15_t)0x3a34, (q15_t)0x3a1e, (q15_t)0x3a08, (q15_t)0x39f1, (q15_t)0x39db, (q15_t)0x39c4, (q15_t)0x39ae, (q15_t)0x3998,
  (q15_t)0x3981, (q15_t)0x396b, (q15_t)0x3954, (q15_t)0x393e, (q15_t)0x3927, (q15_t)0x3911, (q15_t)0x38fa, (q15_t)0x38e4,
  (q15_t)0x38cd, (q15_t)0x38b7, (q15_t)0x38a0, (q15_t)0x388a, (q15_t)0x3873, (q15_t)0x385d, (q15_t)0x3846, (q15_t)0x382f,
  (q15_t)0x3819, (q15_t)0x3802, (q15_t)0x37ec, (q15_t)0x37d5, (q15_t)0x37be, (q15_t)0x37a8, (q15_t)0x3791, (q15_t)0x377a,
  (q15_t)0x3764, (q15_t)0x374d, (q15_t)0x3736, (q15_t)0x3720, (q15_t)0x3709, (q15_t)0x36f2, (q15_t)0x36dc, (q15_t)0x36c5,
  (q15_t)0x36ae, (q15_t)0x3698, (q15_t)0x3681, (q15_t)0x366a, (q15_t)0x3653, (q15_t)0x363d, (q15_t)0x3626, (q15_t)0x360f,
  (q15_t)0x35f8, (q15_t)0x35e1, (q15_t)0x35cb, (q15_t)0x35b4, (q15_t)0x359d, (q15_t)0x3586, (q15_t)0x356f, (q15_t)0x3558,
  (q15_t)0x3542, (q15_t)0x352b, (q15_t)0x3514, (q15_t)0x34fd, (q15_t)0x34e6, (q15_t)0x34cf, (q15_t)0x34b8, (q15_t)0x34a1,
  (q15_t)0x348b, (q15_t)0x3474, (q15_t)0x345d, (q15_t)0x3446, (q15_t)0x342f, (q15_t)0x3418, (q15_t)0x3401, (q15_t)0x33ea,
  (q15_t)0x33d3, (q15_t)0x33bc, (q15_t)0x33a5, (q15_t)0x338e, (q15_t)0x3377, (q15_t)0x3360, (q15_t)0x3349, (q15_t)0x3332,
  (q15_t)0x331b, (q15_t)0x3304, (q15_t)0x32ed, (q15_t)0x32d6, (q15_t)0x32bf, (q15_t)0x32a8, (q15_t)0x3290, (q15_t)0x3279,
  (q15_t)0x3262, (q15_t)0x324b, (q15_t)0x3234, (q15_t)0x321d, (q15_t)0x3206, (q15_t)0x31ef, (q15_t)0x31d8, (q15_t)0x31c0,
  (q15_t)0x31a9, (q15_t)0x3192, (q15_t)0x317b, (q15_t)0x3164, (q15_t)0x314c, (q15_t)0x3135, (q15_t)0x311e, (q15_t)0x3107,
  (q15_t)0x30f0, (q15_t)0x30d8, (q15_t)0x30c1, (q15_t)0x30aa, (q15_t)0x3093, (q15_t)0x307b, (q15_t)0x3064, (q15_t)0x304d,
  (q15_t)0x3036, (q15_t)0x301e, (q15_t)0x3007, (q15_t)0x2ff0, (q15_t)0x2fd8, (q15_t)0x2fc1, (q15_t)0x2faa, (q15_t)0x2f92,
  (q15_t)0x2f7b, (q15_t)0x2f64, (q15_t)0x2f4c, (q15_t)0x2f35, (q15_t)0x2f1e, (q15_t)0x2f06, (q15_t)0x2eef, (q15_t)0x2ed8,
  (q15_t)0x2ec0, (q15_t)0x2ea9, (q15_t)0x2e91, (q15_t)0x2e7a, (q15_t)0x2e63, (q15_t)0x2e4b, (q15_t)0x2e34, (q15_t)0x2e1c,
  (q15_t)0x2e05, (q15_t)0x2ded, (q15_t)0x2dd6, (q15_t)0x2dbe, (q15_t)0x2da7, (q15_t)0x2d8f, (q15_t)0x2d78, (q15_t)0x2d60,
  (q15_t)0x2d49, (q15_t)0x2d31, (q15_t)0x2d1a, (q15_t)0x2d02, (q15_t)0x2ceb, (q15_t)0x2cd3, (q15_t)0x2cbc, (q15_t)0x2ca4,
  (q15_t)0x2c8d, (q15_t)0x2c75, (q15_t)0x2c5e, (q15_t)0x2c46, (q15_t)0x2c2e, (q15_t)0x2c17, (q15_t)0x2bff, (q15_t)0x2be8,
  (q15_t)0x2bd0, (q15_t)0x2bb8, (q15_t)0x2ba1, (q15_t)0x2b89, (q15_t)0x2b71, (q15_t)0x2b5a, (q15_t)0x2b42, (q15_t)0x2b2b,
  (q15_t)0x2b13, (q15_t)0x2afb, (q15_t)0x2ae4, (q15_t)0x2acc, (q15_t)0x2ab4, (q15_t)0x2a9c, (q15_t)0x2a85, (q15_t)0x2a6d,
  (q15_t)0x2a55, (q15_t)0x2a3e, (q15_t)0x2a26, (q15_t)0x2a0e, (q15_t)0x29f6, (q15_t)0x29df, (q15_t)0x29c7, (q15_t)0x29af,
  (q15_t)0x2997, (q15_t)0x2980, (q15_t)0x2968, (q15_t)0x2950, (q15_t)0x2938, (q15_t)0x2920, (q15_t)0x2909, (q15_t)0x28f1,
  (q15_t)0x28d9, (q15_t)0x28c1, (q15_t)0x28a9, (q15_t)0x2892, (q15_t)0x287a, (q15_t)0x2862, (q15_t)0x284a, (q15_t)0x2832,
  (q15_t)0x281a, (q15_t)0x2802, (q15_t)0x27eb, (q15_t)0x27d3, (q15_t)0x27bb, (q15_t)0x27a3, (q15_t)0x278b, (q15_t)0x2773,
  (q15_t)0x275b, (q15_t)0x2743, (q15_t)0x272b, (q15_t)0x2713, (q15_t)0x26fb, (q15_t)0x26e4, (q15_t)0x26cc, (q15_t)0x26b4,
  (q15_t)0x269c, (q15_t)0x2684, (q15_t)0x266c, (q15_t)0x2654, (q15_t)0x263c, (q15_t)0x2624, (q15_t)0x260c, (q15_t)0x25f4,
  (q15_t)0x25dc, (q15_t)0x25c4, (q15_t)0x25ac, (q15_t)0x2594, (q15_t)0x257c, (q15_t)0x2564, (q15_t)0x254c, (q15_t)0x2534,
  (q15_t)0x251c, (q15_t)0x2503, (q15_t)0x24eb, (q15_t)0x24d3, (q15_t)0x24bb, (q15_t)0x24a3, (q15_t)0x248b, (q15_t)0x2473,
  (q15_t)0x245b, (q15_t)0x2443, (q15_t)0x242b, (q15_t)0x2413, (q15_t)0x23fa, (q15_t)0x23e2, (q15_t)0x23ca, (q15_t)0x23b2,
  (q15_t)0x239a, (q15_t)0x2382, (q15_t)0x236a, (q15_t)0x2352, (q15_t)0x2339, (q15_t)0x2321, (q15_t)0x2309, (q15_t)0x22f1,
  (q15_t)0x22d9, (q15_t)0x22c0, (q15_t)0x22a8, (q15_t)0x2290, (q15_t)0x2278, (q15_t)0x2260, (q15_t)0x2247, (q15_t)0x222f,
  (q15_t)0x2217, (q15_t)0x21ff, (q15_t)0x21e7, (q15_t)0x21ce, (q15_t)0x21b6, (q15_t)0x219e, (q15_t)0x2186, (q15_t)0x216d,
  (q15_t)0x2155, (q15_t)0x213d, (q15_t)0x2125, (q15_t)0x210c, (q15_t)0x20f4, (q15_t)0x20dc, (q15_t)0x20c3, (q15_t)0x20ab,
  (q15_t)0x2093, (q15_t)0x207a, (q15_t)0x2062, (q15_t)0x204a, (q15_t)0x2032, (q15_t)0x2019, (q15_t)0x2001, (q15_t)0x1fe9,
  (q15_t)0x1fd0, (q15_t)0x1fb8, (q15_t)0x1f9f, (q15_t)0x1f87, (q15_t)0x1f6f, (q15_t)0x1f56, (q15_t)0x1f3e, (q15_t)0x1f26,
  (q15_t)0x1f0d, (q15_t)0x1ef5, (q15_t)0x1edd, (q15_t)0x1ec4, (q15_t)0x1eac, (q15_t)0x1e93, (q15_t)0x1e7b, (q15_t)0x1e62,
  (q15_t)0x1e4a, (q15_t)0x1e32, (q15_t)0x1e19, (q15_t)0x1e01, (q15_t)0x1de8, (q15_t)0x1dd0, (q15_t)0x1db7, (q15_t)0x1d9f,
  (q15_t)0x1d87, (q15_t)0x1d6e, (q15_t)0x1d56, (q15_t)0x1d3d, (q15_t)0x1d25, (q15_t)0x1d0c, (q15_t)0x1cf4, (q15_t)0x1cdb,
  (q15_t)0x1cc3, (q15_t)0x1caa, (q15_t)0x1c92, (q15_t)0x1c79, (q15_t)0x1c61, (q15_t)0x1c48, (q15_t)0x1c30, (q15_t)0x1c17,
  (q15_t)0x1bff, (q15_t)0x1be6, (q15_t)0x1bce, (q15_t)0x1bb5, (q15_t)0x1b9d, (q15_t)0x1b84, (q15_t)0x1b6c, (q15_t)0x1b53,
  (q15_t)0x1b3a, (q15_t)0x1b22, (q15_t)0x1b09, (q15_t)0x1af1, (q15_t)0x1ad8, (q15_t)0x1ac0, (q15_t)0x1aa7, (q15_t)0x1a8e,
  (q15_t)0x1a76, (q15_t)0x1a5d, (q15_t)0x1a45, (q15_t)0x1a2c, (q15_t)0x1a13, (q15_t)0x19fb, (q15_t)0x19e2, (q15_t)0x19ca,
  (q15_t)0x19b1, (q15_t)0x1998, (q15_t)0x1980, (q15_t)0x1967, (q15_t)0x194e, (q15_t)0x1936, (q15_t)0x191d, (q15_t)0x1905,
  (q15_t)0x18ec, (q15_t)0x18d3, (q15_t)0x18bb, (q15_t)0x18a2, (q15_t)0x1889, (q15_t)0x1871, (q15_t)0x1858, (q15_t)0x183f,
  (q15_t)0x1827, (q15_t)0x180e, (q15_t)0x17f5, (q15_t)0x17dd, (q15_t)0x17c4, (q15_t)0x17ab, (q15_t)0x1792, (q15_t)0x177a,
  (q15_t)0x1761, (q15_t)0x1748, (q15_t)0x1730, (q15_t)0x1717, (q15_t)0x16fe, (q15_t)0x16e5, (q15_t)0x16cd, (q15_t)0x16b4,
  (q15_t)0x169b, (q15_t)0x1682, (q15_t)0x166a, (q15_t)0x1651, (q15_t)0x1638, (q15_t)0x161f, (q15_t)0x1607, (q15_t)0x15ee,
  (q15_t)0x15d5, (q15_t)0x15bc, (q15_t)0x15a4, (q15_t)0x158b, (q15_t)0x1572, (q15_t)0x1559, (q15_t)0x1541, (q15_t)0x1528,
  (q15_t)0x150f, (q15_t)0x14f6, (q15_t)0x14dd, (q15_t)0x14c5, (q15_t)0x14ac, (q15_t)0x1493, (q15_t)0x147a, (q15_t)0x1461,
  (q15_t)0x1449, (q15_t)0x1430, (q15_t)0x1417, (q15_t)0x13fe, (q15_t)0x13e5, (q15_t)0x13cc, (q15_t)0x13b4, (q15_t)0x139b,
  (q15_t)0x1382, (q15_t)0x1369, (q15_t)0x1350, (q15_t)0x1337, (q15_t)0x131f, (q15_t)0x1306, (q15_t)0x12ed, (q15_t)0x12d4,
  (q15_t)0x12bb, (q15_t)0x12a2, (q15_t)0x1289, (q15_t)0x1271, (q15_t)0x1258, (q15_t)0x123f, (q15_t)0x1226, (q15_t)0x120d,
  (q15_t)0x11f4, (q15_t)0x11db, (q15_t)0x11c2, (q15_t)0x11a9, (q15_t)0x1191, (q15_t)0x1178, (q15_t)0x115f, (q15_t)0x1146,
  (q15_t)0x112d, (q15_t)0x1114, (q15_t)0x10fb, (q15_t)0x10e2, (q15_t)0x10c9, (q15_t)0x10b0, (q15_t)0x1098, (q15_t)0x107f,
  (q15_t)0x1066, (q15_t)0x104d, (q15_t)0x1034, (q15_t)0x101b, (q15_t)0x1002, (q15_t)0xfe9, (q15_t)0xfd0, (q15_t)0xfb7,
  (q15_t)0xf9e, (q15_t)0xf85, (q15_t)0xf6c, (q15_t)0xf53, (q15_t)0xf3a, (q15_t)0xf21, (q15_t)0xf08, (q15_t)0xef0,
  (q15_t)0xed7, (q15_t)0xebe, (q15_t)0xea5, (q15_t)0xe8c, (q15_t)0xe73, (q15_t)0xe5a, (q15_t)0xe41, (q15_t)0xe28,
  (q15_t)0xe0f, (q15_t)0xdf6, (q15_t)0xddd, (q15_t)0xdc4, (q15_t)0xdab, (q15_t)0xd92, (q15_t)0xd79, (q15_t)0xd60,
  (q15_t)0xd47, (q15_t)0xd2e, (q15_t)0xd15, (q15_t)0xcfc, (q15_t)0xce3, (q15_t)0xcca, (q15_t)0xcb1, (q15_t)0xc98,
  (q15_t)0xc7f, (q15_t)0xc66, (q15_t)0xc4d, (q15_t)0xc34, (q15_t)0xc1b, (q15_t)0xc02, (q15_t)0xbe9, (q15_t)0xbd0,
  (q15_t)0xbb7, (q15_t)0xb9e, (q15_t)0xb85, (q15_t)0xb6c, (q15_t)0xb53, (q15_t)0xb3a, (q15_t)0xb20, (q15_t)0xb07,
  (q15_t)0xaee, (q15_t)0xad5, (q15_t)0xabc, (q15_t)0xaa3, (q15_t)0xa8a, (q15_t)0xa71, (q15_t)0xa58, (q15_t)0xa3f,
  (q15_t)0xa26, (q15_t)0xa0d, (q15_t)0x9f4, (q15_t)0x9db, (q15_t)0x9c2, (q15_t)0x9a9, (q15_t)0x990, (q15_t)0x977,
  (q15_t)0x95e, (q15_t)0x944, (q15_t)0x92b, (q15_t)0x912, (q15_t)0x8f9, (q15_t)0x8e0, (q15_t)0x8c7, (q15_t)0x8ae,
  (q15_t)0x895, (q15_t)0x87c, (q15_t)0x863, (q15_t)0x84a, (q15_t)0x831, (q15_t)0x818, (q15_t)0x7fe, (q15_t)0x7e5,
  (q15_t)0x7cc, (q15_t)0x7b3, (q15_t)0x79a, (q15_t)0x781, (q15_t)0x768, (q15_t)0x74f, (q15_t)0x736, (q15_t)0x71d,
  (q15_t)0x704, (q15_t)0x6ea, (q15_t)0x6d1, (q15_t)0x6b8, (q15_t)0x69f, (q15_t)0x686, (q15_t)0x66d, (q15_t)0x654,
  (q15_t)0x63b, (q15_t)0x622, (q15_t)0x609, (q15_t)0x5ef, (q15_t)0x5d6, (q15_t)0x5bd, (q15_t)0x5a4, (q15_t)0x58b,
  (q15_t)0x572, (q15_t)0x559, (q15_t)0x540, (q15_t)0x527, (q15_t)0x50d, (q15_t)0x4f4, (q15_t)0x4db, (q15_t)0x4c2,
  (q15_t)0x4a9, (q15_t)0x490, (q15_t)0x477, (q15_t)0x45e, (q15_t)0x445, (q15_t)0x42b, (q15_t)0x412, (q15_t)0x3f9,
  (q15_t)0x3e0, (q15_t)0x3c7, (q15_t)0x3ae, (q15_t)0x395, (q15_t)0x37c, (q15_t)0x362, (q15_t)0x349, (q15_t)0x330,
  (q15_t)0x317, (q15_t)0x2fe, (q15_t)0x2e5, (q15_t)0x2cc, (q15_t)0x2b3, (q15_t)0x299, (q15_t)0x280, (q15_t)0x267,
  (q15_t)0x24e, (q15_t)0x235, (q15_t)0x21c, (q15_t)0x203, (q15_t)0x1ea, (q15_t)0x1d0, (q15_t)0x1b7, (q15_t)0x19e,
  (q15_t)0x185, (q15_t)0x16c, (q15_t)0x153, (q15_t)0x13a, (q15_t)0x121, (q15_t)0x107, (q15_t)0xee, (q15_t)0xd5,
  (q15_t)0xbc, (q15_t)0xa3, (q15_t)0x8a, (q15_t)0x71, (q15_t)0x57, (q15_t)0x3e, (q15_t)0x25, (q15_t)0xc

};

static const q15_t ALIGN4 cos_factorsQ15_8192[8192] = {
  (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff,
  (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff,
  (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff,
  (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff,
  (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff, (q15_t)0x7fff,
  (q15_t)0x7fff, (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffe,
  (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffe,
  (q15_t)0x7ffe, (q15_t)0x7ffe, (q15_t)0x7ffd, (q15_t)0x7ffd, (q15_t)0x7ffd, (q15_t)0x7ffd, (q15_t)0x7ffd, (q15_t)0x7ffd,
  (q15_t)0x7ffd, (q15_t)0x7ffd, (q15_t)0x7ffd, (q15_t)0x7ffd, (q15_t)0x7ffd, (q15_t)0x7ffd, (q15_t)0x7ffd, (q15_t)0x7ffc,
  (q15_t)0x7ffc, (q15_t)0x7ffc, (q15_t)0x7ffc, (q15_t)0x7ffc, (q15_t)0x7ffc, (q15_t)0x7ffc, (q15_t)0x7ffc, (q15_t)0x7ffc,
  (q15_t)0x7ffc, (q15_t)0x7ffb, (q15_t)0x7ffb, (q15_t)0x7ffb, (q15_t)0x7ffb, (q15_t)0x7ffb, (q15_t)0x7ffb, (q15_t)0x7ffb,
  (q15_t)0x7ffb, (q15_t)0x7ffb, (q15_t)0x7ffb, (q15_t)0x7ffa, (q15_t)0x7ffa, (q15_t)0x7ffa, (q15_t)0x7ffa, (q15_t)0x7ffa,
  (q15_t)0x7ffa, (q15_t)0x7ffa, (q15_t)0x7ffa, (q15_t)0x7ffa, (q15_t)0x7ff9, (q15_t)0x7ff9, (q15_t)0x7ff9, (q15_t)0x7ff9,
  (q15_t)0x7ff9, (q15_t)0x7ff9, (q15_t)0x7ff9, (q15_t)0x7ff9, (q15_t)0x7ff8, (q15_t)0x7ff8, (q15_t)0x7ff8, (q15_t)0x7ff8,
  (q15_t)0x7ff8, (q15_t)0x7ff8, (q15_t)0x7ff8, (q15_t)0x7ff7, (q15_t)0x7ff7, (q15_t)0x7ff7, (q15_t)0x7ff7, (q15_t)0x7ff7,
  (q15_t)0x7ff7, (q15_t)0x7ff7, (q15_t)0x7ff6, (q15_t)0x7ff6, (q15_t)0x7ff6, (q15_t)0x7ff6, (q15_t)0x7ff6, (q15_t)0x7ff6,
  (q15_t)0x7ff6, (q15_t)0x7ff5, (q15_t)0x7ff5, (q15_t)0x7ff5, (q15_t)0x7ff5, (q15_t)0x7ff5, (q15_t)0x7ff5, (q15_t)0x7ff4,
  (q15_t)0x7ff4, (q15_t)0x7ff4, (q15_t)0x7ff4, (q15_t)0x7ff4, (q15_t)0x7ff4, (q15_t)0x7ff3, (q15_t)0x7ff3, (q15_t)0x7ff3,
  (q15_t)0x7ff3, (q15_t)0x7ff3, (q15_t)0x7ff3, (q15_t)0x7ff2, (q15_t)0x7ff2, (q15_t)0x7ff2, (q15_t)0x7ff2, (q15_t)0x7ff2,
  (q15_t)0x7ff1, (q15_t)0x7ff1, (q15_t)0x7ff1, (q15_t)0x7ff1, (q15_t)0x7ff1, (q15_t)0x7ff1, (q15_t)0x7ff0, (q15_t)0x7ff0,
  (q15_t)0x7ff0, (q15_t)0x7ff0, (q15_t)0x7ff0, (q15_t)0x7fef, (q15_t)0x7fef, (q15_t)0x7fef, (q15_t)0x7fef, (q15_t)0x7fef,
  (q15_t)0x7fee, (q15_t)0x7fee, (q15_t)0x7fee, (q15_t)0x7fee, (q15_t)0x7fee, (q15_t)0x7fed, (q15_t)0x7fed, (q15_t)0x7fed,
  (q15_t)0x7fed, (q15_t)0x7fed, (q15_t)0x7fec, (q15_t)0x7fec, (q15_t)0x7fec, (q15_t)0x7fec, (q15_t)0x7feb, (q15_t)0x7feb,
  (q15_t)0x7feb, (q15_t)0x7feb, (q15_t)0x7feb, (q15_t)0x7fea, (q15_t)0x7fea, (q15_t)0x7fea, (q15_t)0x7fea, (q15_t)0x7fe9,
  (q15_t)0x7fe9, (q15_t)0x7fe9, (q15_t)0x7fe9, (q15_t)0x7fe8, (q15_t)0x7fe8, (q15_t)0x7fe8, (q15_t)0x7fe8, (q15_t)0x7fe8,
  (q15_t)0x7fe7, (q15_t)0x7fe7, (q15_t)0x7fe7, (q15_t)0x7fe7, (q15_t)0x7fe6, (q15_t)0x7fe6, (q15_t)0x7fe6, (q15_t)0x7fe6,
  (q15_t)0x7fe5, (q15_t)0x7fe5, (q15_t)0x7fe5, (q15_t)0x7fe5, (q15_t)0x7fe4, (q15_t)0x7fe4, (q15_t)0x7fe4, (q15_t)0x7fe4,
  (q15_t)0x7fe3, (q15_t)0x7fe3, (q15_t)0x7fe3, (q15_t)0x7fe2, (q15_t)0x7fe2, (q15_t)0x7fe2, (q15_t)0x7fe2, (q15_t)0x7fe1,
  (q15_t)0x7fe1, (q15_t)0x7fe1, (q15_t)0x7fe1, (q15_t)0x7fe0, (q15_t)0x7fe0, (q15_t)0x7fe0, (q15_t)0x7fdf, (q15_t)0x7fdf,
  (q15_t)0x7fdf, (q15_t)0x7fdf, (q15_t)0x7fde, (q15_t)0x7fde, (q15_t)0x7fde, (q15_t)0x7fde, (q15_t)0x7fdd, (q15_t)0x7fdd,
  (q15_t)0x7fdd, (q15_t)0x7fdc, (q15_t)0x7fdc, (q15_t)0x7fdc, (q15_t)0x7fdb, (q15_t)0x7fdb, (q15_t)0x7fdb, (q15_t)0x7fdb,
  (q15_t)0x7fda, (q15_t)0x7fda, (q15_t)0x7fda, (q15_t)0x7fd9, (q15_t)0x7fd9, (q15_t)0x7fd9, (q15_t)0x7fd8, (q15_t)0x7fd8,
  (q15_t)0x7fd8, (q15_t)0x7fd8, (q15_t)0x7fd7, (q15_t)0x7fd7, (q15_t)0x7fd7, (q15_t)0x7fd6, (q15_t)0x7fd6, (q15_t)0x7fd6,
  (q15_t)0x7fd5, (q15_t)0x7fd5, (q15_t)0x7fd5, (q15_t)0x7fd4, (q15_t)0x7fd4, (q15_t)0x7fd4, (q15_t)0x7fd3, (q15_t)0x7fd3,
  (q15_t)0x7fd3, (q15_t)0x7fd2, (q15_t)0x7fd2, (q15_t)0x7fd2, (q15_t)0x7fd1, (q15_t)0x7fd1, (q15_t)0x7fd1, (q15_t)0x7fd0,
  (q15_t)0x7fd0, (q15_t)0x7fd0, (q15_t)0x7fcf, (q15_t)0x7fcf, (q15_t)0x7fcf, (q15_t)0x7fce, (q15_t)0x7fce, (q15_t)0x7fce,
  (q15_t)0x7fcd, (q15_t)0x7fcd, (q15_t)0x7fcd, (q15_t)0x7fcc, (q15_t)0x7fcc, (q15_t)0x7fcc, (q15_t)0x7fcb, (q15_t)0x7fcb,
  (q15_t)0x7fcb, (q15_t)0x7fca, (q15_t)0x7fca, (q15_t)0x7fc9, (q15_t)0x7fc9, (q15_t)0x7fc9, (q15_t)0x7fc8, (q15_t)0x7fc8,
  (q15_t)0x7fc8, (q15_t)0x7fc7, (q15_t)0x7fc7, (q15_t)0x7fc7, (q15_t)0x7fc6, (q15_t)0x7fc6, (q15_t)0x7fc5, (q15_t)0x7fc5,
  (q15_t)0x7fc5, (q15_t)0x7fc4, (q15_t)0x7fc4, (q15_t)0x7fc4, (q15_t)0x7fc3, (q15_t)0x7fc3, (q15_t)0x7fc2, (q15_t)0x7fc2,
  (q15_t)0x7fc2, (q15_t)0x7fc1, (q15_t)0x7fc1, (q15_t)0x7fc0, (q15_t)0x7fc0, (q15_t)0x7fc0, (q15_t)0x7fbf, (q15_t)0x7fbf,
  (q15_t)0x7fbf, (q15_t)0x7fbe, (q15_t)0x7fbe, (q15_t)0x7fbd, (q15_t)0x7fbd, (q15_t)0x7fbd, (q15_t)0x7fbc, (q15_t)0x7fbc,
  (q15_t)0x7fbb, (q15_t)0x7fbb, (q15_t)0x7fbb, (q15_t)0x7fba, (q15_t)0x7fba, (q15_t)0x7fb9, (q15_t)0x7fb9, (q15_t)0x7fb8,
  (q15_t)0x7fb8, (q15_t)0x7fb8, (q15_t)0x7fb7, (q15_t)0x7fb7, (q15_t)0x7fb6, (q15_t)0x7fb6, (q15_t)0x7fb6, (q15_t)0x7fb5,
  (q15_t)0x7fb5, (q15_t)0x7fb4, (q15_t)0x7fb4, (q15_t)0x7fb3, (q15_t)0x7fb3, (q15_t)0x7fb3, (q15_t)0x7fb2, (q15_t)0x7fb2,
  (q15_t)0x7fb1, (q15_t)0x7fb1, (q15_t)0x7fb0, (q15_t)0x7fb0, (q15_t)0x7faf, (q15_t)0x7faf, (q15_t)0x7faf, (q15_t)0x7fae,
  (q15_t)0x7fae, (q15_t)0x7fad, (q15_t)0x7fad, (q15_t)0x7fac, (q15_t)0x7fac, (q15_t)0x7fac, (q15_t)0x7fab, (q15_t)0x7fab,
  (q15_t)0x7faa, (q15_t)0x7faa, (q15_t)0x7fa9, (q15_t)0x7fa9, (q15_t)0x7fa8, (q15_t)0x7fa8, (q15_t)0x7fa7, (q15_t)0x7fa7,
  (q15_t)0x7fa6, (q15_t)0x7fa6, (q15_t)0x7fa6, (q15_t)0x7fa5, (q15_t)0x7fa5, (q15_t)0x7fa4, (q15_t)0x7fa4, (q15_t)0x7fa3,
  (q15_t)0x7fa3, (q15_t)0x7fa2, (q15_t)0x7fa2, (q15_t)0x7fa1, (q15_t)0x7fa1, (q15_t)0x7fa0, (q15_t)0x7fa0, (q15_t)0x7f9f,
  (q15_t)0x7f9f, (q15_t)0x7f9e, (q15_t)0x7f9e, (q15_t)0x7f9d, (q15_t)0x7f9d, (q15_t)0x7f9c, (q15_t)0x7f9c, (q15_t)0x7f9c,
  (q15_t)0x7f9b, (q15_t)0x7f9b, (q15_t)0x7f9a, (q15_t)0x7f9a, (q15_t)0x7f99, (q15_t)0x7f99, (q15_t)0x7f98, (q15_t)0x7f98,
  (q15_t)0x7f97, (q15_t)0x7f97, (q15_t)0x7f96, (q15_t)0x7f96, (q15_t)0x7f95, (q15_t)0x7f95, (q15_t)0x7f94, (q15_t)0x7f94,
  (q15_t)0x7f93, (q15_t)0x7f92, (q15_t)0x7f92, (q15_t)0x7f91, (q15_t)0x7f91, (q15_t)0x7f90, (q15_t)0x7f90, (q15_t)0x7f8f,
  (q15_t)0x7f8f, (q15_t)0x7f8e, (q15_t)0x7f8e, (q15_t)0x7f8d, (q15_t)0x7f8d, (q15_t)0x7f8c, (q15_t)0x7f8c, (q15_t)0x7f8b,
  (q15_t)0x7f8b, (q15_t)0x7f8a, (q15_t)0x7f8a, (q15_t)0x7f89, (q15_t)0x7f89, (q15_t)0x7f88, (q15_t)0x7f87, (q15_t)0x7f87,
  (q15_t)0x7f86, (q15_t)0x7f86, (q15_t)0x7f85, (q15_t)0x7f85, (q15_t)0x7f84, (q15_t)0x7f84, (q15_t)0x7f83, (q15_t)0x7f83,
  (q15_t)0x7f82, (q15_t)0x7f81, (q15_t)0x7f81, (q15_t)0x7f80, (q15_t)0x7f80, (q15_t)0x7f7f, (q15_t)0x7f7f, (q15_t)0x7f7e,
  (q15_t)0x7f7e, (q15_t)0x7f7d, (q15_t)0x7f7c, (q15_t)0x7f7c, (q15_t)0x7f7b, (q15_t)0x7f7b, (q15_t)0x7f7a, (q15_t)0x7f7a,
  (q15_t)0x7f79, (q15_t)0x7f79, (q15_t)0x7f78, (q15_t)0x7f77, (q15_t)0x7f77, (q15_t)0x7f76, (q15_t)0x7f76, (q15_t)0x7f75,
  (q15_t)0x7f75, (q15_t)0x7f74, (q15_t)0x7f73, (q15_t)0x7f73, (q15_t)0x7f72, (q15_t)0x7f72, (q15_t)0x7f71, (q15_t)0x7f70,
  (q15_t)0x7f70, (q15_t)0x7f6f, (q15_t)0x7f6f, (q15_t)0x7f6e, (q15_t)0x7f6d, (q15_t)0x7f6d, (q15_t)0x7f6c, (q15_t)0x7f6c,
  (q15_t)0x7f6b, (q15_t)0x7f6b, (q15_t)0x7f6a, (q15_t)0x7f69, (q15_t)0x7f69, (q15_t)0x7f68, (q15_t)0x7f68, (q15_t)0x7f67,
  (q15_t)0x7f66, (q15_t)0x7f66, (q15_t)0x7f65, (q15_t)0x7f64, (q15_t)0x7f64, (q15_t)0x7f63, (q15_t)0x7f63, (q15_t)0x7f62,
  (q15_t)0x7f61, (q15_t)0x7f61, (q15_t)0x7f60, (q15_t)0x7f60, (q15_t)0x7f5f, (q15_t)0x7f5e, (q15_t)0x7f5e, (q15_t)0x7f5d,
  (q15_t)0x7f5c, (q15_t)0x7f5c, (q15_t)0x7f5b, (q15_t)0x7f5b, (q15_t)0x7f5a, (q15_t)0x7f59, (q15_t)0x7f59, (q15_t)0x7f58,
  (q15_t)0x7f57, (q15_t)0x7f57, (q15_t)0x7f56, (q15_t)0x7f55, (q15_t)0x7f55, (q15_t)0x7f54, (q15_t)0x7f54, (q15_t)0x7f53,
  (q15_t)0x7f52, (q15_t)0x7f52, (q15_t)0x7f51, (q15_t)0x7f50, (q15_t)0x7f50, (q15_t)0x7f4f, (q15_t)0x7f4e, (q15_t)0x7f4e,
  (q15_t)0x7f4d, (q15_t)0x7f4c, (q15_t)0x7f4c, (q15_t)0x7f4b, (q15_t)0x7f4a, (q15_t)0x7f4a, (q15_t)0x7f49, (q15_t)0x7f48,
  (q15_t)0x7f48, (q15_t)0x7f47, (q15_t)0x7f46, (q15_t)0x7f46, (q15_t)0x7f45, (q15_t)0x7f44, (q15_t)0x7f44, (q15_t)0x7f43,
  (q15_t)0x7f42, (q15_t)0x7f42, (q15_t)0x7f41, (q15_t)0x7f40, (q15_t)0x7f40, (q15_t)0x7f3f, (q15_t)0x7f3e, (q15_t)0x7f3e,
  (q15_t)0x7f3d, (q15_t)0x7f3c, (q15_t)0x7f3c, (q15_t)0x7f3b, (q15_t)0x7f3a, (q15_t)0x7f3a, (q15_t)0x7f39, (q15_t)0x7f38,
  (q15_t)0x7f37, (q15_t)0x7f37, (q15_t)0x7f36, (q15_t)0x7f35, (q15_t)0x7f35, (q15_t)0x7f34, (q15_t)0x7f33, (q15_t)0x7f33,
  (q15_t)0x7f32, (q15_t)0x7f31, (q15_t)0x7f31, (q15_t)0x7f30, (q15_t)0x7f2f, (q15_t)0x7f2e, (q15_t)0x7f2e, (q15_t)0x7f2d,
  (q15_t)0x7f2c, (q15_t)0x7f2c, (q15_t)0x7f2b, (q15_t)0x7f2a, (q15_t)0x7f29, (q15_t)0x7f29, (q15_t)0x7f28, (q15_t)0x7f27,
  (q15_t)0x7f27, (q15_t)0x7f26, (q15_t)0x7f25, (q15_t)0x7f24, (q15_t)0x7f24, (q15_t)0x7f23, (q15_t)0x7f22, (q15_t)0x7f21,
  (q15_t)0x7f21, (q15_t)0x7f20, (q15_t)0x7f1f, (q15_t)0x7f1f, (q15_t)0x7f1e, (q15_t)0x7f1d, (q15_t)0x7f1c, (q15_t)0x7f1c,
  (q15_t)0x7f1b, (q15_t)0x7f1a, (q15_t)0x7f19, (q15_t)0x7f19, (q15_t)0x7f18, (q15_t)0x7f17, (q15_t)0x7f16, (q15_t)0x7f16,
  (q15_t)0x7f15, (q15_t)0x7f14, (q15_t)0x7f13, (q15_t)0x7f13, (q15_t)0x7f12, (q15_t)0x7f11, (q15_t)0x7f10, (q15_t)0x7f10,
  (q15_t)0x7f0f, (q15_t)0x7f0e, (q15_t)0x7f0d, (q15_t)0x7f0d, (q15_t)0x7f0c, (q15_t)0x7f0b, (q15_t)0x7f0a, (q15_t)0x7f09,
  (q15_t)0x7f09, (q15_t)0x7f08, (q15_t)0x7f07, (q15_t)0x7f06, (q15_t)0x7f06, (q15_t)0x7f05, (q15_t)0x7f04, (q15_t)0x7f03,
  (q15_t)0x7f02, (q15_t)0x7f02, (q15_t)0x7f01, (q15_t)0x7f00, (q15_t)0x7eff, (q15_t)0x7eff, (q15_t)0x7efe, (q15_t)0x7efd,
  (q15_t)0x7efc, (q15_t)0x7efb, (q15_t)0x7efb, (q15_t)0x7efa, (q15_t)0x7ef9, (q15_t)0x7ef8, (q15_t)0x7ef7, (q15_t)0x7ef7,
  (q15_t)0x7ef6, (q15_t)0x7ef5, (q15_t)0x7ef4, (q15_t)0x7ef3, (q15_t)0x7ef3, (q15_t)0x7ef2, (q15_t)0x7ef1, (q15_t)0x7ef0,
  (q15_t)0x7eef, (q15_t)0x7eef, (q15_t)0x7eee, (q15_t)0x7eed, (q15_t)0x7eec, (q15_t)0x7eeb, (q15_t)0x7eeb, (q15_t)0x7eea,
  (q15_t)0x7ee9, (q15_t)0x7ee8, (q15_t)0x7ee7, (q15_t)0x7ee6, (q15_t)0x7ee6, (q15_t)0x7ee5, (q15_t)0x7ee4, (q15_t)0x7ee3,
  (q15_t)0x7ee2, (q15_t)0x7ee2, (q15_t)0x7ee1, (q15_t)0x7ee0, (q15_t)0x7edf, (q15_t)0x7ede, (q15_t)0x7edd, (q15_t)0x7edd,
  (q15_t)0x7edc, (q15_t)0x7edb, (q15_t)0x7eda, (q15_t)0x7ed9, (q15_t)0x7ed8, (q15_t)0x7ed8, (q15_t)0x7ed7, (q15_t)0x7ed6,
  (q15_t)0x7ed5, (q15_t)0x7ed4, (q15_t)0x7ed3, (q15_t)0x7ed2, (q15_t)0x7ed2, (q15_t)0x7ed1, (q15_t)0x7ed0, (q15_t)0x7ecf,
  (q15_t)0x7ece, (q15_t)0x7ecd, (q15_t)0x7ecc, (q15_t)0x7ecc, (q15_t)0x7ecb, (q15_t)0x7eca, (q15_t)0x7ec9, (q15_t)0x7ec8,
  (q15_t)0x7ec7, (q15_t)0x7ec6, (q15_t)0x7ec6, (q15_t)0x7ec5, (q15_t)0x7ec4, (q15_t)0x7ec3, (q15_t)0x7ec2, (q15_t)0x7ec1,
  (q15_t)0x7ec0, (q15_t)0x7ebf, (q15_t)0x7ebf, (q15_t)0x7ebe, (q15_t)0x7ebd, (q15_t)0x7ebc, (q15_t)0x7ebb, (q15_t)0x7eba,
  (q15_t)0x7eb9, (q15_t)0x7eb8, (q15_t)0x7eb8, (q15_t)0x7eb7, (q15_t)0x7eb6, (q15_t)0x7eb5, (q15_t)0x7eb4, (q15_t)0x7eb3,
  (q15_t)0x7eb2, (q15_t)0x7eb1, (q15_t)0x7eb0, (q15_t)0x7eaf, (q15_t)0x7eaf, (q15_t)0x7eae, (q15_t)0x7ead, (q15_t)0x7eac,
  (q15_t)0x7eab, (q15_t)0x7eaa, (q15_t)0x7ea9, (q15_t)0x7ea8, (q15_t)0x7ea7, (q15_t)0x7ea6, (q15_t)0x7ea6, (q15_t)0x7ea5,
  (q15_t)0x7ea4, (q15_t)0x7ea3, (q15_t)0x7ea2, (q15_t)0x7ea1, (q15_t)0x7ea0, (q15_t)0x7e9f, (q15_t)0x7e9e, (q15_t)0x7e9d,
  (q15_t)0x7e9c, (q15_t)0x7e9b, (q15_t)0x7e9b, (q15_t)0x7e9a, (q15_t)0x7e99, (q15_t)0x7e98, (q15_t)0x7e97, (q15_t)0x7e96,
  (q15_t)0x7e95, (q15_t)0x7e94, (q15_t)0x7e93, (q15_t)0x7e92, (q15_t)0x7e91, (q15_t)0x7e90, (q15_t)0x7e8f, (q15_t)0x7e8e,
  (q15_t)0x7e8d, (q15_t)0x7e8d, (q15_t)0x7e8c, (q15_t)0x7e8b, (q15_t)0x7e8a, (q15_t)0x7e89, (q15_t)0x7e88, (q15_t)0x7e87,
  (q15_t)0x7e86, (q15_t)0x7e85, (q15_t)0x7e84, (q15_t)0x7e83, (q15_t)0x7e82, (q15_t)0x7e81, (q15_t)0x7e80, (q15_t)0x7e7f,
  (q15_t)0x7e7e, (q15_t)0x7e7d, (q15_t)0x7e7c, (q15_t)0x7e7b, (q15_t)0x7e7a, (q15_t)0x7e79, (q15_t)0x7e78, (q15_t)0x7e77,
  (q15_t)0x7e77, (q15_t)0x7e76, (q15_t)0x7e75, (q15_t)0x7e74, (q15_t)0x7e73, (q15_t)0x7e72, (q15_t)0x7e71, (q15_t)0x7e70,
  (q15_t)0x7e6f, (q15_t)0x7e6e, (q15_t)0x7e6d, (q15_t)0x7e6c, (q15_t)0x7e6b, (q15_t)0x7e6a, (q15_t)0x7e69, (q15_t)0x7e68,
  (q15_t)0x7e67, (q15_t)0x7e66, (q15_t)0x7e65, (q15_t)0x7e64, (q15_t)0x7e63, (q15_t)0x7e62, (q15_t)0x7e61, (q15_t)0x7e60,
  (q15_t)0x7e5f, (q15_t)0x7e5e, (q15_t)0x7e5d, (q15_t)0x7e5c, (q15_t)0x7e5b, (q15_t)0x7e5a, (q15_t)0x7e59, (q15_t)0x7e58,
  (q15_t)0x7e57, (q15_t)0x7e56, (q15_t)0x7e55, (q15_t)0x7e54, (q15_t)0x7e53, (q15_t)0x7e52, (q15_t)0x7e51, (q15_t)0x7e50,
  (q15_t)0x7e4f, (q15_t)0x7e4e, (q15_t)0x7e4d, (q15_t)0x7e4c, (q15_t)0x7e4b, (q15_t)0x7e4a, (q15_t)0x7e49, (q15_t)0x7e48,
  (q15_t)0x7e47, (q15_t)0x7e46, (q15_t)0x7e45, (q15_t)0x7e43, (q15_t)0x7e42, (q15_t)0x7e41, (q15_t)0x7e40, (q15_t)0x7e3f,
  (q15_t)0x7e3e, (q15_t)0x7e3d, (q15_t)0x7e3c, (q15_t)0x7e3b, (q15_t)0x7e3a, (q15_t)0x7e39, (q15_t)0x7e38, (q15_t)0x7e37,
  (q15_t)0x7e36, (q15_t)0x7e35, (q15_t)0x7e34, (q15_t)0x7e33, (q15_t)0x7e32, (q15_t)0x7e31, (q15_t)0x7e30, (q15_t)0x7e2f,
  (q15_t)0x7e2e, (q15_t)0x7e2d, (q15_t)0x7e2b, (q15_t)0x7e2a, (q15_t)0x7e29, (q15_t)0x7e28, (q15_t)0x7e27, (q15_t)0x7e26,
  (q15_t)0x7e25, (q15_t)0x7e24, (q15_t)0x7e23, (q15_t)0x7e22, (q15_t)0x7e21, (q15_t)0x7e20, (q15_t)0x7e1f, (q15_t)0x7e1e,
  (q15_t)0x7e1d, (q15_t)0x7e1b, (q15_t)0x7e1a, (q15_t)0x7e19, (q15_t)0x7e18, (q15_t)0x7e17, (q15_t)0x7e16, (q15_t)0x7e15,
  (q15_t)0x7e14, (q15_t)0x7e13, (q15_t)0x7e12, (q15_t)0x7e11, (q15_t)0x7e10, (q15_t)0x7e0e, (q15_t)0x7e0d, (q15_t)0x7e0c,
  (q15_t)0x7e0b, (q15_t)0x7e0a, (q15_t)0x7e09, (q15_t)0x7e08, (q15_t)0x7e07, (q15_t)0x7e06, (q15_t)0x7e05, (q15_t)0x7e04,
  (q15_t)0x7e02, (q15_t)0x7e01, (q15_t)0x7e00, (q15_t)0x7dff, (q15_t)0x7dfe, (q15_t)0x7dfd, (q15_t)0x7dfc, (q15_t)0x7dfb,
  (q15_t)0x7dfa, (q15_t)0x7df8, (q15_t)0x7df7, (q15_t)0x7df6, (q15_t)0x7df5, (q15_t)0x7df4, (q15_t)0x7df3, (q15_t)0x7df2,
  (q15_t)0x7df1, (q15_t)0x7def, (q15_t)0x7dee, (q15_t)0x7ded, (q15_t)0x7dec, (q15_t)0x7deb, (q15_t)0x7dea, (q15_t)0x7de9,
  (q15_t)0x7de8, (q15_t)0x7de6, (q15_t)0x7de5, (q15_t)0x7de4, (q15_t)0x7de3, (q15_t)0x7de2, (q15_t)0x7de1, (q15_t)0x7de0,
  (q15_t)0x7dde, (q15_t)0x7ddd, (q15_t)0x7ddc, (q15_t)0x7ddb, (q15_t)0x7dda, (q15_t)0x7dd9, (q15_t)0x7dd8, (q15_t)0x7dd6,
  (q15_t)0x7dd5, (q15_t)0x7dd4, (q15_t)0x7dd3, (q15_t)0x7dd2, (q15_t)0x7dd1, (q15_t)0x7dd0, (q15_t)0x7dce, (q15_t)0x7dcd,
  (q15_t)0x7dcc, (q15_t)0x7dcb, (q15_t)0x7dca, (q15_t)0x7dc9, (q15_t)0x7dc7, (q15_t)0x7dc6, (q15_t)0x7dc5, (q15_t)0x7dc4,
  (q15_t)0x7dc3, (q15_t)0x7dc2, (q15_t)0x7dc0, (q15_t)0x7dbf, (q15_t)0x7dbe, (q15_t)0x7dbd, (q15_t)0x7dbc, (q15_t)0x7dbb,
  (q15_t)0x7db9, (q15_t)0x7db8, (q15_t)0x7db7, (q15_t)0x7db6, (q15_t)0x7db5, (q15_t)0x7db3, (q15_t)0x7db2, (q15_t)0x7db1,
  (q15_t)0x7db0, (q15_t)0x7daf, (q15_t)0x7dae, (q15_t)0x7dac, (q15_t)0x7dab, (q15_t)0x7daa, (q15_t)0x7da9, (q15_t)0x7da8,
  (q15_t)0x7da6, (q15_t)0x7da5, (q15_t)0x7da4, (q15_t)0x7da3, (q15_t)0x7da2, (q15_t)0x7da0, (q15_t)0x7d9f, (q15_t)0x7d9e,
  (q15_t)0x7d9d, (q15_t)0x7d9c, (q15_t)0x7d9a, (q15_t)0x7d99, (q15_t)0x7d98, (q15_t)0x7d97, (q15_t)0x7d95, (q15_t)0x7d94,
  (q15_t)0x7d93, (q15_t)0x7d92, (q15_t)0x7d91, (q15_t)0x7d8f, (q15_t)0x7d8e, (q15_t)0x7d8d, (q15_t)0x7d8c, (q15_t)0x7d8a,
  (q15_t)0x7d89, (q15_t)0x7d88, (q15_t)0x7d87, (q15_t)0x7d86, (q15_t)0x7d84, (q15_t)0x7d83, (q15_t)0x7d82, (q15_t)0x7d81,
  (q15_t)0x7d7f, (q15_t)0x7d7e, (q15_t)0x7d7d, (q15_t)0x7d7c, (q15_t)0x7d7a, (q15_t)0x7d79, (q15_t)0x7d78, (q15_t)0x7d77,
  (q15_t)0x7d75, (q15_t)0x7d74, (q15_t)0x7d73, (q15_t)0x7d72, (q15_t)0x7d70, (q15_t)0x7d6f, (q15_t)0x7d6e, (q15_t)0x7d6d,
  (q15_t)0x7d6b, (q15_t)0x7d6a, (q15_t)0x7d69, (q15_t)0x7d68, (q15_t)0x7d66, (q15_t)0x7d65, (q15_t)0x7d64, (q15_t)0x7d63,
  (q15_t)0x7d61, (q15_t)0x7d60, (q15_t)0x7d5f, (q15_t)0x7d5e, (q15_t)0x7d5c, (q15_t)0x7d5b, (q15_t)0x7d5a, (q15_t)0x7d59,
  (q15_t)0x7d57, (q15_t)0x7d56, (q15_t)0x7d55, (q15_t)0x7d53, (q15_t)0x7d52, (q15_t)0x7d51, (q15_t)0x7d50, (q15_t)0x7d4e,
  (q15_t)0x7d4d, (q15_t)0x7d4c, (q15_t)0x7d4a, (q15_t)0x7d49, (q15_t)0x7d48, (q15_t)0x7d47, (q15_t)0x7d45, (q15_t)0x7d44,
  (q15_t)0x7d43, (q15_t)0x7d41, (q15_t)0x7d40, (q15_t)0x7d3f, (q15_t)0x7d3e, (q15_t)0x7d3c, (q15_t)0x7d3b, (q15_t)0x7d3a,
  (q15_t)0x7d38, (q15_t)0x7d37, (q15_t)0x7d36, (q15_t)0x7d34, (q15_t)0x7d33, (q15_t)0x7d32, (q15_t)0x7d31, (q15_t)0x7d2f,
  (q15_t)0x7d2e, (q15_t)0x7d2d, (q15_t)0x7d2b, (q15_t)0x7d2a, (q15_t)0x7d29, (q15_t)0x7d27, (q15_t)0x7d26, (q15_t)0x7d25,
  (q15_t)0x7d23, (q15_t)0x7d22, (q15_t)0x7d21, (q15_t)0x7d1f, (q15_t)0x7d1e, (q15_t)0x7d1d, (q15_t)0x7d1b, (q15_t)0x7d1a,
  (q15_t)0x7d19, (q15_t)0x7d17, (q15_t)0x7d16, (q15_t)0x7d15, (q15_t)0x7d13, (q15_t)0x7d12, (q15_t)0x7d11, (q15_t)0x7d0f,
  (q15_t)0x7d0e, (q15_t)0x7d0d, (q15_t)0x7d0b, (q15_t)0x7d0a, (q15_t)0x7d09, (q15_t)0x7d07, (q15_t)0x7d06, (q15_t)0x7d05,
  (q15_t)0x7d03, (q15_t)0x7d02, (q15_t)0x7d01, (q15_t)0x7cff, (q15_t)0x7cfe, (q15_t)0x7cfd, (q15_t)0x7cfb, (q15_t)0x7cfa,
  (q15_t)0x7cf9, (q15_t)0x7cf7, (q15_t)0x7cf6, (q15_t)0x7cf4, (q15_t)0x7cf3, (q15_t)0x7cf2, (q15_t)0x7cf0, (q15_t)0x7cef,
  (q15_t)0x7cee, (q15_t)0x7cec, (q15_t)0x7ceb, (q15_t)0x7ce9, (q15_t)0x7ce8, (q15_t)0x7ce7, (q15_t)0x7ce5, (q15_t)0x7ce4,
  (q15_t)0x7ce3, (q15_t)0x7ce1, (q15_t)0x7ce0, (q15_t)0x7cde, (q15_t)0x7cdd, (q15_t)0x7cdc, (q15_t)0x7cda, (q15_t)0x7cd9,
  (q15_t)0x7cd8, (q15_t)0x7cd6, (q15_t)0x7cd5, (q15_t)0x7cd3, (q15_t)0x7cd2, (q15_t)0x7cd1, (q15_t)0x7ccf, (q15_t)0x7cce,
  (q15_t)0x7ccc, (q15_t)0x7ccb, (q15_t)0x7cca, (q15_t)0x7cc8, (q15_t)0x7cc7, (q15_t)0x7cc5, (q15_t)0x7cc4, (q15_t)0x7cc3,
  (q15_t)0x7cc1, (q15_t)0x7cc0, (q15_t)0x7cbe, (q15_t)0x7cbd, (q15_t)0x7cbc, (q15_t)0x7cba, (q15_t)0x7cb9, (q15_t)0x7cb7,
  (q15_t)0x7cb6, (q15_t)0x7cb5, (q15_t)0x7cb3, (q15_t)0x7cb2, (q15_t)0x7cb0, (q15_t)0x7caf, (q15_t)0x7cad, (q15_t)0x7cac,
  (q15_t)0x7cab, (q15_t)0x7ca9, (q15_t)0x7ca8, (q15_t)0x7ca6, (q15_t)0x7ca5, (q15_t)0x7ca3, (q15_t)0x7ca2, (q15_t)0x7ca1,
  (q15_t)0x7c9f, (q15_t)0x7c9e, (q15_t)0x7c9c, (q15_t)0x7c9b, (q15_t)0x7c99, (q15_t)0x7c98, (q15_t)0x7c97, (q15_t)0x7c95,
  (q15_t)0x7c94, (q15_t)0x7c92, (q15_t)0x7c91, (q15_t)0x7c8f, (q15_t)0x7c8e, (q15_t)0x7c8c, (q15_t)0x7c8b, (q15_t)0x7c8a,
  (q15_t)0x7c88, (q15_t)0x7c87, (q15_t)0x7c85, (q15_t)0x7c84, (q15_t)0x7c82, (q15_t)0x7c81, (q15_t)0x7c7f, (q15_t)0x7c7e,
  (q15_t)0x7c7c, (q15_t)0x7c7b, (q15_t)0x7c79, (q15_t)0x7c78, (q15_t)0x7c77, (q15_t)0x7c75, (q15_t)0x7c74, (q15_t)0x7c72,
  (q15_t)0x7c71, (q15_t)0x7c6f, (q15_t)0x7c6e, (q15_t)0x7c6c, (q15_t)0x7c6b, (q15_t)0x7c69, (q15_t)0x7c68, (q15_t)0x7c66,
  (q15_t)0x7c65, (q15_t)0x7c63, (q15_t)0x7c62, (q15_t)0x7c60, (q15_t)0x7c5f, (q15_t)0x7c5d, (q15_t)0x7c5c, (q15_t)0x7c5a,
  (q15_t)0x7c59, (q15_t)0x7c58, (q15_t)0x7c56, (q15_t)0x7c55, (q15_t)0x7c53, (q15_t)0x7c52, (q15_t)0x7c50, (q15_t)0x7c4f,
  (q15_t)0x7c4d, (q15_t)0x7c4c, (q15_t)0x7c4a, (q15_t)0x7c49, (q15_t)0x7c47, (q15_t)0x7c46, (q15_t)0x7c44, (q15_t)0x7c43,
  (q15_t)0x7c41, (q15_t)0x7c3f, (q15_t)0x7c3e, (q15_t)0x7c3c, (q15_t)0x7c3b, (q15_t)0x7c39, (q15_t)0x7c38, (q15_t)0x7c36,
  (q15_t)0x7c35, (q15_t)0x7c33, (q15_t)0x7c32, (q15_t)0x7c30, (q15_t)0x7c2f, (q15_t)0x7c2d, (q15_t)0x7c2c, (q15_t)0x7c2a,
  (q15_t)0x7c29, (q15_t)0x7c27, (q15_t)0x7c26, (q15_t)0x7c24, (q15_t)0x7c23, (q15_t)0x7c21, (q15_t)0x7c20, (q15_t)0x7c1e,
  (q15_t)0x7c1c, (q15_t)0x7c1b, (q15_t)0x7c19, (q15_t)0x7c18, (q15_t)0x7c16, (q15_t)0x7c15, (q15_t)0x7c13, (q15_t)0x7c12,
  (q15_t)0x7c10, (q15_t)0x7c0f, (q15_t)0x7c0d, (q15_t)0x7c0b, (q15_t)0x7c0a, (q15_t)0x7c08, (q15_t)0x7c07, (q15_t)0x7c05,
  (q15_t)0x7c04, (q15_t)0x7c02, (q15_t)0x7c01, (q15_t)0x7bff, (q15_t)0x7bfd, (q15_t)0x7bfc, (q15_t)0x7bfa, (q15_t)0x7bf9,
  (q15_t)0x7bf7, (q15_t)0x7bf6, (q15_t)0x7bf4, (q15_t)0x7bf3, (q15_t)0x7bf1, (q15_t)0x7bef, (q15_t)0x7bee, (q15_t)0x7bec,
  (q15_t)0x7beb, (q15_t)0x7be9, (q15_t)0x7be8, (q15_t)0x7be6, (q15_t)0x7be4, (q15_t)0x7be3, (q15_t)0x7be1, (q15_t)0x7be0,
  (q15_t)0x7bde, (q15_t)0x7bdc, (q15_t)0x7bdb, (q15_t)0x7bd9, (q15_t)0x7bd8, (q15_t)0x7bd6, (q15_t)0x7bd5, (q15_t)0x7bd3,
  (q15_t)0x7bd1, (q15_t)0x7bd0, (q15_t)0x7bce, (q15_t)0x7bcd, (q15_t)0x7bcb, (q15_t)0x7bc9, (q15_t)0x7bc8, (q15_t)0x7bc6,
  (q15_t)0x7bc5, (q15_t)0x7bc3, (q15_t)0x7bc1, (q15_t)0x7bc0, (q15_t)0x7bbe, (q15_t)0x7bbd, (q15_t)0x7bbb, (q15_t)0x7bb9,
  (q15_t)0x7bb8, (q15_t)0x7bb6, (q15_t)0x7bb5, (q15_t)0x7bb3, (q15_t)0x7bb1, (q15_t)0x7bb0, (q15_t)0x7bae, (q15_t)0x7bac,
  (q15_t)0x7bab, (q15_t)0x7ba9, (q15_t)0x7ba8, (q15_t)0x7ba6, (q15_t)0x7ba4, (q15_t)0x7ba3, (q15_t)0x7ba1, (q15_t)0x7b9f,
  (q15_t)0x7b9e, (q15_t)0x7b9c, (q15_t)0x7b9b, (q15_t)0x7b99, (q15_t)0x7b97, (q15_t)0x7b96, (q15_t)0x7b94, (q15_t)0x7b92,
  (q15_t)0x7b91, (q15_t)0x7b8f, (q15_t)0x7b8d, (q15_t)0x7b8c, (q15_t)0x7b8a, (q15_t)0x7b89, (q15_t)0x7b87, (q15_t)0x7b85,
  (q15_t)0x7b84, (q15_t)0x7b82, (q15_t)0x7b80, (q15_t)0x7b7f, (q15_t)0x7b7d, (q15_t)0x7b7b, (q15_t)0x7b7a, (q15_t)0x7b78,
  (q15_t)0x7b76, (q15_t)0x7b75, (q15_t)0x7b73, (q15_t)0x7b71, (q15_t)0x7b70, (q15_t)0x7b6e, (q15_t)0x7b6c, (q15_t)0x7b6b,
  (q15_t)0x7b69, (q15_t)0x7b67, (q15_t)0x7b66, (q15_t)0x7b64, (q15_t)0x7b62, (q15_t)0x7b61, (q15_t)0x7b5f, (q15_t)0x7b5d,
  (q15_t)0x7b5c, (q15_t)0x7b5a, (q15_t)0x7b58, (q15_t)0x7b57, (q15_t)0x7b55, (q15_t)0x7b53, (q15_t)0x7b52, (q15_t)0x7b50,
  (q15_t)0x7b4e, (q15_t)0x7b4d, (q15_t)0x7b4b, (q15_t)0x7b49, (q15_t)0x7b47, (q15_t)0x7b46, (q15_t)0x7b44, (q15_t)0x7b42,
  (q15_t)0x7b41, (q15_t)0x7b3f, (q15_t)0x7b3d, (q15_t)0x7b3c, (q15_t)0x7b3a, (q15_t)0x7b38, (q15_t)0x7b37, (q15_t)0x7b35,
  (q15_t)0x7b33, (q15_t)0x7b31, (q15_t)0x7b30, (q15_t)0x7b2e, (q15_t)0x7b2c, (q15_t)0x7b2b, (q15_t)0x7b29, (q15_t)0x7b27,
  (q15_t)0x7b25, (q15_t)0x7b24, (q15_t)0x7b22, (q15_t)0x7b20, (q15_t)0x7b1f, (q15_t)0x7b1d, (q15_t)0x7b1b, (q15_t)0x7b19,
  (q15_t)0x7b18, (q15_t)0x7b16, (q15_t)0x7b14, (q15_t)0x7b13, (q15_t)0x7b11, (q15_t)0x7b0f, (q15_t)0x7b0d, (q15_t)0x7b0c,
  (q15_t)0x7b0a, (q15_t)0x7b08, (q15_t)0x7b06, (q15_t)0x7b05, (q15_t)0x7b03, (q15_t)0x7b01, (q15_t)0x7aff, (q15_t)0x7afe,
  (q15_t)0x7afc, (q15_t)0x7afa, (q15_t)0x7af8, (q15_t)0x7af7, (q15_t)0x7af5, (q15_t)0x7af3, (q15_t)0x7af2, (q15_t)0x7af0,
  (q15_t)0x7aee, (q15_t)0x7aec, (q15_t)0x7aeb, (q15_t)0x7ae9, (q15_t)0x7ae7, (q15_t)0x7ae5, (q15_t)0x7ae3, (q15_t)0x7ae2,
  (q15_t)0x7ae0, (q15_t)0x7ade, (q15_t)0x7adc, (q15_t)0x7adb, (q15_t)0x7ad9, (q15_t)0x7ad7, (q15_t)0x7ad5, (q15_t)0x7ad4,
  (q15_t)0x7ad2, (q15_t)0x7ad0, (q15_t)0x7ace, (q15_t)0x7acd, (q15_t)0x7acb, (q15_t)0x7ac9, (q15_t)0x7ac7, (q15_t)0x7ac5,
  (q15_t)0x7ac4, (q15_t)0x7ac2, (q15_t)0x7ac0, (q15_t)0x7abe, (q15_t)0x7abd, (q15_t)0x7abb, (q15_t)0x7ab9, (q15_t)0x7ab7,
  (q15_t)0x7ab5, (q15_t)0x7ab4, (q15_t)0x7ab2, (q15_t)0x7ab0, (q15_t)0x7aae, (q15_t)0x7aac, (q15_t)0x7aab, (q15_t)0x7aa9,
  (q15_t)0x7aa7, (q15_t)0x7aa5, (q15_t)0x7aa3, (q15_t)0x7aa2, (q15_t)0x7aa0, (q15_t)0x7a9e, (q15_t)0x7a9c, (q15_t)0x7a9a,
  (q15_t)0x7a99, (q15_t)0x7a97, (q15_t)0x7a95, (q15_t)0x7a93, (q15_t)0x7a91, (q15_t)0x7a90, (q15_t)0x7a8e, (q15_t)0x7a8c,
  (q15_t)0x7a8a, (q15_t)0x7a88, (q15_t)0x7a87, (q15_t)0x7a85, (q15_t)0x7a83, (q15_t)0x7a81, (q15_t)0x7a7f, (q15_t)0x7a7d,
  (q15_t)0x7a7c, (q15_t)0x7a7a, (q15_t)0x7a78, (q15_t)0x7a76, (q15_t)0x7a74, (q15_t)0x7a72, (q15_t)0x7a71, (q15_t)0x7a6f,
  (q15_t)0x7a6d, (q15_t)0x7a6b, (q15_t)0x7a69, (q15_t)0x7a67, (q15_t)0x7a66, (q15_t)0x7a64, (q15_t)0x7a62, (q15_t)0x7a60,
  (q15_t)0x7a5e, (q15_t)0x7a5c, (q15_t)0x7a5b, (q15_t)0x7a59, (q15_t)0x7a57, (q15_t)0x7a55, (q15_t)0x7a53, (q15_t)0x7a51,
  (q15_t)0x7a4f, (q15_t)0x7a4e, (q15_t)0x7a4c, (q15_t)0x7a4a, (q15_t)0x7a48, (q15_t)0x7a46, (q15_t)0x7a44, (q15_t)0x7a42,
  (q15_t)0x7a41, (q15_t)0x7a3f, (q15_t)0x7a3d, (q15_t)0x7a3b, (q15_t)0x7a39, (q15_t)0x7a37, (q15_t)0x7a35, (q15_t)0x7a34,
  (q15_t)0x7a32, (q15_t)0x7a30, (q15_t)0x7a2e, (q15_t)0x7a2c, (q15_t)0x7a2a, (q15_t)0x7a28, (q15_t)0x7a26, (q15_t)0x7a25,
  (q15_t)0x7a23, (q15_t)0x7a21, (q15_t)0x7a1f, (q15_t)0x7a1d, (q15_t)0x7a1b, (q15_t)0x7a19, (q15_t)0x7a17, (q15_t)0x7a16,
  (q15_t)0x7a14, (q15_t)0x7a12, (q15_t)0x7a10, (q15_t)0x7a0e, (q15_t)0x7a0c, (q15_t)0x7a0a, (q15_t)0x7a08, (q15_t)0x7a06,
  (q15_t)0x7a04, (q15_t)0x7a03, (q15_t)0x7a01, (q15_t)0x79ff, (q15_t)0x79fd, (q15_t)0x79fb, (q15_t)0x79f9, (q15_t)0x79f7,
  (q15_t)0x79f5, (q15_t)0x79f3, (q15_t)0x79f1, (q15_t)0x79f0, (q15_t)0x79ee, (q15_t)0x79ec, (q15_t)0x79ea, (q15_t)0x79e8,
  (q15_t)0x79e6, (q15_t)0x79e4, (q15_t)0x79e2, (q15_t)0x79e0, (q15_t)0x79de, (q15_t)0x79dc, (q15_t)0x79da, (q15_t)0x79d9,
  (q15_t)0x79d7, (q15_t)0x79d5, (q15_t)0x79d3, (q15_t)0x79d1, (q15_t)0x79cf, (q15_t)0x79cd, (q15_t)0x79cb, (q15_t)0x79c9,
  (q15_t)0x79c7, (q15_t)0x79c5, (q15_t)0x79c3, (q15_t)0x79c1, (q15_t)0x79bf, (q15_t)0x79bd, (q15_t)0x79bc, (q15_t)0x79ba,
  (q15_t)0x79b8, (q15_t)0x79b6, (q15_t)0x79b4, (q15_t)0x79b2, (q15_t)0x79b0, (q15_t)0x79ae, (q15_t)0x79ac, (q15_t)0x79aa,
  (q15_t)0x79a8, (q15_t)0x79a6, (q15_t)0x79a4, (q15_t)0x79a2, (q15_t)0x79a0, (q15_t)0x799e, (q15_t)0x799c, (q15_t)0x799a,
  (q15_t)0x7998, (q15_t)0x7996, (q15_t)0x7994, (q15_t)0x7992, (q15_t)0x7991, (q15_t)0x798f, (q15_t)0x798d, (q15_t)0x798b,
  (q15_t)0x7989, (q15_t)0x7987, (q15_t)0x7985, (q15_t)0x7983, (q15_t)0x7981, (q15_t)0x797f, (q15_t)0x797d, (q15_t)0x797b,
  (q15_t)0x7979, (q15_t)0x7977, (q15_t)0x7975, (q15_t)0x7973, (q15_t)0x7971, (q15_t)0x796f, (q15_t)0x796d, (q15_t)0x796b,
  (q15_t)0x7969, (q15_t)0x7967, (q15_t)0x7965, (q15_t)0x7963, (q15_t)0x7961, (q15_t)0x795f, (q15_t)0x795d, (q15_t)0x795b,
  (q15_t)0x7959, (q15_t)0x7957, (q15_t)0x7955, (q15_t)0x7953, (q15_t)0x7951, (q15_t)0x794f, (q15_t)0x794d, (q15_t)0x794b,
  (q15_t)0x7949, (q15_t)0x7947, (q15_t)0x7945, (q15_t)0x7943, (q15_t)0x7941, (q15_t)0x793f, (q15_t)0x793d, (q15_t)0x793b,
  (q15_t)0x7939, (q15_t)0x7937, (q15_t)0x7935, (q15_t)0x7933, (q15_t)0x7931, (q15_t)0x792f, (q15_t)0x792d, (q15_t)0x792b,
  (q15_t)0x7929, (q15_t)0x7927, (q15_t)0x7925, (q15_t)0x7923, (q15_t)0x7921, (q15_t)0x791f, (q15_t)0x791d, (q15_t)0x791a,
  (q15_t)0x7918, (q15_t)0x7916, (q15_t)0x7914, (q15_t)0x7912, (q15_t)0x7910, (q15_t)0x790e, (q15_t)0x790c, (q15_t)0x790a,
  (q15_t)0x7908, (q15_t)0x7906, (q15_t)0x7904, (q15_t)0x7902, (q15_t)0x7900, (q15_t)0x78fe, (q15_t)0x78fc, (q15_t)0x78fa,
  (q15_t)0x78f8, (q15_t)0x78f6, (q15_t)0x78f4, (q15_t)0x78f2, (q15_t)0x78f0, (q15_t)0x78ed, (q15_t)0x78eb, (q15_t)0x78e9,
  (q15_t)0x78e7, (q15_t)0x78e5, (q15_t)0x78e3, (q15_t)0x78e1, (q15_t)0x78df, (q15_t)0x78dd, (q15_t)0x78db, (q15_t)0x78d9,
  (q15_t)0x78d7, (q15_t)0x78d5, (q15_t)0x78d3, (q15_t)0x78d1, (q15_t)0x78ce, (q15_t)0x78cc, (q15_t)0x78ca, (q15_t)0x78c8,
  (q15_t)0x78c6, (q15_t)0x78c4, (q15_t)0x78c2, (q15_t)0x78c0, (q15_t)0x78be, (q15_t)0x78bc, (q15_t)0x78ba, (q15_t)0x78b8,
  (q15_t)0x78b5, (q15_t)0x78b3, (q15_t)0x78b1, (q15_t)0x78af, (q15_t)0x78ad, (q15_t)0x78ab, (q15_t)0x78a9, (q15_t)0x78a7,
  (q15_t)0x78a5, (q15_t)0x78a3, (q15_t)0x78a0, (q15_t)0x789e, (q15_t)0x789c, (q15_t)0x789a, (q15_t)0x7898, (q15_t)0x7896,
  (q15_t)0x7894, (q15_t)0x7892, (q15_t)0x7890, (q15_t)0x788e, (q15_t)0x788b, (q15_t)0x7889, (q15_t)0x7887, (q15_t)0x7885,
  (q15_t)0x7883, (q15_t)0x7881, (q15_t)0x787f, (q15_t)0x787d, (q15_t)0x787a, (q15_t)0x7878, (q15_t)0x7876, (q15_t)0x7874,
  (q15_t)0x7872, (q15_t)0x7870, (q15_t)0x786e, (q15_t)0x786c, (q15_t)0x7869, (q15_t)0x7867, (q15_t)0x7865, (q15_t)0x7863,
  (q15_t)0x7861, (q15_t)0x785f, (q15_t)0x785d, (q15_t)0x785b, (q15_t)0x7858, (q15_t)0x7856, (q15_t)0x7854, (q15_t)0x7852,
  (q15_t)0x7850, (q15_t)0x784e, (q15_t)0x784c, (q15_t)0x7849, (q15_t)0x7847, (q15_t)0x7845, (q15_t)0x7843, (q15_t)0x7841,
  (q15_t)0x783f, (q15_t)0x783c, (q15_t)0x783a, (q15_t)0x7838, (q15_t)0x7836, (q15_t)0x7834, (q15_t)0x7832, (q15_t)0x7830,
  (q15_t)0x782d, (q15_t)0x782b, (q15_t)0x7829, (q15_t)0x7827, (q15_t)0x7825, (q15_t)0x7823, (q15_t)0x7820, (q15_t)0x781e,
  (q15_t)0x781c, (q15_t)0x781a, (q15_t)0x7818, (q15_t)0x7816, (q15_t)0x7813, (q15_t)0x7811, (q15_t)0x780f, (q15_t)0x780d,
  (q15_t)0x780b, (q15_t)0x7808, (q15_t)0x7806, (q15_t)0x7804, (q15_t)0x7802, (q15_t)0x7800, (q15_t)0x77fe, (q15_t)0x77fb,
  (q15_t)0x77f9, (q15_t)0x77f7, (q15_t)0x77f5, (q15_t)0x77f3, (q15_t)0x77f0, (q15_t)0x77ee, (q15_t)0x77ec, (q15_t)0x77ea,
  (q15_t)0x77e8, (q15_t)0x77e5, (q15_t)0x77e3, (q15_t)0x77e1, (q15_t)0x77df, (q15_t)0x77dd, (q15_t)0x77da, (q15_t)0x77d8,
  (q15_t)0x77d6, (q15_t)0x77d4, (q15_t)0x77d2, (q15_t)0x77cf, (q15_t)0x77cd, (q15_t)0x77cb, (q15_t)0x77c9, (q15_t)0x77c6,
  (q15_t)0x77c4, (q15_t)0x77c2, (q15_t)0x77c0, (q15_t)0x77be, (q15_t)0x77bb, (q15_t)0x77b9, (q15_t)0x77b7, (q15_t)0x77b5,
  (q15_t)0x77b2, (q15_t)0x77b0, (q15_t)0x77ae, (q15_t)0x77ac, (q15_t)0x77aa, (q15_t)0x77a7, (q15_t)0x77a5, (q15_t)0x77a3,
  (q15_t)0x77a1, (q15_t)0x779e, (q15_t)0x779c, (q15_t)0x779a, (q15_t)0x7798, (q15_t)0x7795, (q15_t)0x7793, (q15_t)0x7791,
  (q15_t)0x778f, (q15_t)0x778c, (q15_t)0x778a, (q15_t)0x7788, (q15_t)0x7786, (q15_t)0x7783, (q15_t)0x7781, (q15_t)0x777f,
  (q15_t)0x777d, (q15_t)0x777a, (q15_t)0x7778, (q15_t)0x7776, (q15_t)0x7774, (q15_t)0x7771, (q15_t)0x776f, (q15_t)0x776d,
  (q15_t)0x776b, (q15_t)0x7768, (q15_t)0x7766, (q15_t)0x7764, (q15_t)0x7762, (q15_t)0x775f, (q15_t)0x775d, (q15_t)0x775b,
  (q15_t)0x7759, (q15_t)0x7756, (q15_t)0x7754, (q15_t)0x7752, (q15_t)0x774f, (q15_t)0x774d, (q15_t)0x774b, (q15_t)0x7749,
  (q15_t)0x7746, (q15_t)0x7744, (q15_t)0x7742, (q15_t)0x773f, (q15_t)0x773d, (q15_t)0x773b, (q15_t)0x7739, (q15_t)0x7736,
  (q15_t)0x7734, (q15_t)0x7732, (q15_t)0x772f, (q15_t)0x772d, (q15_t)0x772b, (q15_t)0x7729, (q15_t)0x7726, (q15_t)0x7724,
  (q15_t)0x7722, (q15_t)0x771f, (q15_t)0x771d, (q15_t)0x771b, (q15_t)0x7719, (q15_t)0x7716, (q15_t)0x7714, (q15_t)0x7712,
  (q15_t)0x770f, (q15_t)0x770d, (q15_t)0x770b, (q15_t)0x7708, (q15_t)0x7706, (q15_t)0x7704, (q15_t)0x7701, (q15_t)0x76ff,
  (q15_t)0x76fd, (q15_t)0x76fa, (q15_t)0x76f8, (q15_t)0x76f6, (q15_t)0x76f4, (q15_t)0x76f1, (q15_t)0x76ef, (q15_t)0x76ed,
  (q15_t)0x76ea, (q15_t)0x76e8, (q15_t)0x76e6, (q15_t)0x76e3, (q15_t)0x76e1, (q15_t)0x76df, (q15_t)0x76dc, (q15_t)0x76da,
  (q15_t)0x76d8, (q15_t)0x76d5, (q15_t)0x76d3, (q15_t)0x76d1, (q15_t)0x76ce, (q15_t)0x76cc, (q15_t)0x76ca, (q15_t)0x76c7,
  (q15_t)0x76c5, (q15_t)0x76c3, (q15_t)0x76c0, (q15_t)0x76be, (q15_t)0x76bc, (q15_t)0x76b9, (q15_t)0x76b7, (q15_t)0x76b4,
  (q15_t)0x76b2, (q15_t)0x76b0, (q15_t)0x76ad, (q15_t)0x76ab, (q15_t)0x76a9, (q15_t)0x76a6, (q15_t)0x76a4, (q15_t)0x76a2,
  (q15_t)0x769f, (q15_t)0x769d, (q15_t)0x769b, (q15_t)0x7698, (q15_t)0x7696, (q15_t)0x7693, (q15_t)0x7691, (q15_t)0x768f,
  (q15_t)0x768c, (q15_t)0x768a, (q15_t)0x7688, (q15_t)0x7685, (q15_t)0x7683, (q15_t)0x7681, (q15_t)0x767e, (q15_t)0x767c,
  (q15_t)0x7679, (q15_t)0x7677, (q15_t)0x7675, (q15_t)0x7672, (q15_t)0x7670, (q15_t)0x766d, (q15_t)0x766b, (q15_t)0x7669,
  (q15_t)0x7666, (q15_t)0x7664, (q15_t)0x7662, (q15_t)0x765f, (q15_t)0x765d, (q15_t)0x765a, (q15_t)0x7658, (q15_t)0x7656,
  (q15_t)0x7653, (q15_t)0x7651, (q15_t)0x764e, (q15_t)0x764c, (q15_t)0x764a, (q15_t)0x7647, (q15_t)0x7645, (q15_t)0x7642,
  (q15_t)0x7640, (q15_t)0x763e, (q15_t)0x763b, (q15_t)0x7639, (q15_t)0x7636, (q15_t)0x7634, (q15_t)0x7632, (q15_t)0x762f,
  (q15_t)0x762d, (q15_t)0x762a, (q15_t)0x7628, (q15_t)0x7625, (q15_t)0x7623, (q15_t)0x7621, (q15_t)0x761e, (q15_t)0x761c,
  (q15_t)0x7619, (q15_t)0x7617, (q15_t)0x7615, (q15_t)0x7612, (q15_t)0x7610, (q15_t)0x760d, (q15_t)0x760b, (q15_t)0x7608,
  (q15_t)0x7606, (q15_t)0x7604, (q15_t)0x7601, (q15_t)0x75ff, (q15_t)0x75fc, (q15_t)0x75fa, (q15_t)0x75f7, (q15_t)0x75f5,
  (q15_t)0x75f2, (q15_t)0x75f0, (q15_t)0x75ee, (q15_t)0x75eb, (q15_t)0x75e9, (q15_t)0x75e6, (q15_t)0x75e4, (q15_t)0x75e1,
  (q15_t)0x75df, (q15_t)0x75dc, (q15_t)0x75da, (q15_t)0x75d8, (q15_t)0x75d5, (q15_t)0x75d3, (q15_t)0x75d0, (q15_t)0x75ce,
  (q15_t)0x75cb, (q15_t)0x75c9, (q15_t)0x75c6, (q15_t)0x75c4, (q15_t)0x75c1, (q15_t)0x75bf, (q15_t)0x75bc, (q15_t)0x75ba,
  (q15_t)0x75b8, (q15_t)0x75b5, (q15_t)0x75b3, (q15_t)0x75b0, (q15_t)0x75ae, (q15_t)0x75ab, (q15_t)0x75a9, (q15_t)0x75a6,
  (q15_t)0x75a4, (q15_t)0x75a1, (q15_t)0x759f, (q15_t)0x759c, (q15_t)0x759a, (q15_t)0x7597, (q15_t)0x7595, (q15_t)0x7592,
  (q15_t)0x7590, (q15_t)0x758d, (q15_t)0x758b, (q15_t)0x7588, (q15_t)0x7586, (q15_t)0x7584, (q15_t)0x7581, (q15_t)0x757f,
  (q15_t)0x757c, (q15_t)0x757a, (q15_t)0x7577, (q15_t)0x7575, (q15_t)0x7572, (q15_t)0x7570, (q15_t)0x756d, (q15_t)0x756b,
  (q15_t)0x7568, (q15_t)0x7566, (q15_t)0x7563, (q15_t)0x7561, (q15_t)0x755e, (q15_t)0x755c, (q15_t)0x7559, (q15_t)0x7556,
  (q15_t)0x7554, (q15_t)0x7551, (q15_t)0x754f, (q15_t)0x754c, (q15_t)0x754a, (q15_t)0x7547, (q15_t)0x7545, (q15_t)0x7542,
  (q15_t)0x7540, (q15_t)0x753d, (q15_t)0x753b, (q15_t)0x7538, (q15_t)0x7536, (q15_t)0x7533, (q15_t)0x7531, (q15_t)0x752e,
  (q15_t)0x752c, (q15_t)0x7529, (q15_t)0x7527, (q15_t)0x7524, (q15_t)0x7522, (q15_t)0x751f, (q15_t)0x751c, (q15_t)0x751a,
  (q15_t)0x7517, (q15_t)0x7515, (q15_t)0x7512, (q15_t)0x7510, (q15_t)0x750d, (q15_t)0x750b, (q15_t)0x7508, (q15_t)0x7506,
  (q15_t)0x7503, (q15_t)0x7501, (q15_t)0x74fe, (q15_t)0x74fb, (q15_t)0x74f9, (q15_t)0x74f6, (q15_t)0x74f4, (q15_t)0x74f1,
  (q15_t)0x74ef, (q15_t)0x74ec, (q15_t)0x74ea, (q15_t)0x74e7, (q15_t)0x74e4, (q15_t)0x74e2, (q15_t)0x74df, (q15_t)0x74dd,
  (q15_t)0x74da, (q15_t)0x74d8, (q15_t)0x74d5, (q15_t)0x74d2, (q15_t)0x74d0, (q15_t)0x74cd, (q15_t)0x74cb, (q15_t)0x74c8,
  (q15_t)0x74c6, (q15_t)0x74c3, (q15_t)0x74c0, (q15_t)0x74be, (q15_t)0x74bb, (q15_t)0x74b9, (q15_t)0x74b6, (q15_t)0x74b4,
  (q15_t)0x74b1, (q15_t)0x74ae, (q15_t)0x74ac, (q15_t)0x74a9, (q15_t)0x74a7, (q15_t)0x74a4, (q15_t)0x74a1, (q15_t)0x749f,
  (q15_t)0x749c, (q15_t)0x749a, (q15_t)0x7497, (q15_t)0x7495, (q15_t)0x7492, (q15_t)0x748f, (q15_t)0x748d, (q15_t)0x748a,
  (q15_t)0x7488, (q15_t)0x7485, (q15_t)0x7482, (q15_t)0x7480, (q15_t)0x747d, (q15_t)0x747b, (q15_t)0x7478, (q15_t)0x7475,
  (q15_t)0x7473, (q15_t)0x7470, (q15_t)0x746d, (q15_t)0x746b, (q15_t)0x7468, (q15_t)0x7466, (q15_t)0x7463, (q15_t)0x7460,
  (q15_t)0x745e, (q15_t)0x745b, (q15_t)0x7459, (q15_t)0x7456, (q15_t)0x7453, (q15_t)0x7451, (q15_t)0x744e, (q15_t)0x744b,
  (q15_t)0x7449, (q15_t)0x7446, (q15_t)0x7444, (q15_t)0x7441, (q15_t)0x743e, (q15_t)0x743c, (q15_t)0x7439, (q15_t)0x7436,
  (q15_t)0x7434, (q15_t)0x7431, (q15_t)0x742f, (q15_t)0x742c, (q15_t)0x7429, (q15_t)0x7427, (q15_t)0x7424, (q15_t)0x7421,
  (q15_t)0x741f, (q15_t)0x741c, (q15_t)0x7419, (q15_t)0x7417, (q15_t)0x7414, (q15_t)0x7411, (q15_t)0x740f, (q15_t)0x740c,
  (q15_t)0x740a, (q15_t)0x7407, (q15_t)0x7404, (q15_t)0x7402, (q15_t)0x73ff, (q15_t)0x73fc, (q15_t)0x73fa, (q15_t)0x73f7,
  (q15_t)0x73f4, (q15_t)0x73f2, (q15_t)0x73ef, (q15_t)0x73ec, (q15_t)0x73ea, (q15_t)0x73e7, (q15_t)0x73e4, (q15_t)0x73e2,
  (q15_t)0x73df, (q15_t)0x73dc, (q15_t)0x73da, (q15_t)0x73d7, (q15_t)0x73d4, (q15_t)0x73d2, (q15_t)0x73cf, (q15_t)0x73cc,
  (q15_t)0x73ca, (q15_t)0x73c7, (q15_t)0x73c4, (q15_t)0x73c1, (q15_t)0x73bf, (q15_t)0x73bc, (q15_t)0x73b9, (q15_t)0x73b7,
  (q15_t)0x73b4, (q15_t)0x73b1, (q15_t)0x73af, (q15_t)0x73ac, (q15_t)0x73a9, (q15_t)0x73a7, (q15_t)0x73a4, (q15_t)0x73a1,
  (q15_t)0x739f, (q15_t)0x739c, (q15_t)0x7399, (q15_t)0x7396, (q15_t)0x7394, (q15_t)0x7391, (q15_t)0x738e, (q15_t)0x738c,
  (q15_t)0x7389, (q15_t)0x7386, (q15_t)0x7384, (q15_t)0x7381, (q15_t)0x737e, (q15_t)0x737b, (q15_t)0x7379, (q15_t)0x7376,
  (q15_t)0x7373, (q15_t)0x7371, (q15_t)0x736e, (q15_t)0x736b, (q15_t)0x7368, (q15_t)0x7366, (q15_t)0x7363, (q15_t)0x7360,
  (q15_t)0x735e, (q15_t)0x735b, (q15_t)0x7358, (q15_t)0x7355, (q15_t)0x7353, (q15_t)0x7350, (q15_t)0x734d, (q15_t)0x734a,
  (q15_t)0x7348, (q15_t)0x7345, (q15_t)0x7342, (q15_t)0x7340, (q15_t)0x733d, (q15_t)0x733a, (q15_t)0x7337, (q15_t)0x7335,
  (q15_t)0x7332, (q15_t)0x732f, (q15_t)0x732c, (q15_t)0x732a, (q15_t)0x7327, (q15_t)0x7324, (q15_t)0x7321, (q15_t)0x731f,
  (q15_t)0x731c, (q15_t)0x7319, (q15_t)0x7316, (q15_t)0x7314, (q15_t)0x7311, (q15_t)0x730e, (q15_t)0x730b, (q15_t)0x7309,
  (q15_t)0x7306, (q15_t)0x7303, (q15_t)0x7300, (q15_t)0x72fe, (q15_t)0x72fb, (q15_t)0x72f8, (q15_t)0x72f5, (q15_t)0x72f3,
  (q15_t)0x72f0, (q15_t)0x72ed, (q15_t)0x72ea, (q15_t)0x72e8, (q15_t)0x72e5, (q15_t)0x72e2, (q15_t)0x72df, (q15_t)0x72dc,
  (q15_t)0x72da, (q15_t)0x72d7, (q15_t)0x72d4, (q15_t)0x72d1, (q15_t)0x72cf, (q15_t)0x72cc, (q15_t)0x72c9, (q15_t)0x72c6,
  (q15_t)0x72c3, (q15_t)0x72c1, (q15_t)0x72be, (q15_t)0x72bb, (q15_t)0x72b8, (q15_t)0x72b5, (q15_t)0x72b3, (q15_t)0x72b0,
  (q15_t)0x72ad, (q15_t)0x72aa, (q15_t)0x72a8, (q15_t)0x72a5, (q15_t)0x72a2, (q15_t)0x729f, (q15_t)0x729c, (q15_t)0x729a,
  (q15_t)0x7297, (q15_t)0x7294, (q15_t)0x7291, (q15_t)0x728e, (q15_t)0x728c, (q15_t)0x7289, (q15_t)0x7286, (q15_t)0x7283,
  (q15_t)0x7280, (q15_t)0x727e, (q15_t)0x727b, (q15_t)0x7278, (q15_t)0x7275, (q15_t)0x7272, (q15_t)0x726f, (q15_t)0x726d,
  (q15_t)0x726a, (q15_t)0x7267, (q15_t)0x7264, (q15_t)0x7261, (q15_t)0x725f, (q15_t)0x725c, (q15_t)0x7259, (q15_t)0x7256,
  (q15_t)0x7253, (q15_t)0x7250, (q15_t)0x724e, (q15_t)0x724b, (q15_t)0x7248, (q15_t)0x7245, (q15_t)0x7242, (q15_t)0x723f,
  (q15_t)0x723d, (q15_t)0x723a, (q15_t)0x7237, (q15_t)0x7234, (q15_t)0x7231, (q15_t)0x722e, (q15_t)0x722c, (q15_t)0x7229,
  (q15_t)0x7226, (q15_t)0x7223, (q15_t)0x7220, (q15_t)0x721d, (q15_t)0x721b, (q15_t)0x7218, (q15_t)0x7215, (q15_t)0x7212,
  (q15_t)0x720f, (q15_t)0x720c, (q15_t)0x7209, (q15_t)0x7207, (q15_t)0x7204, (q15_t)0x7201, (q15_t)0x71fe, (q15_t)0x71fb,
  (q15_t)0x71f8, (q15_t)0x71f5, (q15_t)0x71f3, (q15_t)0x71f0, (q15_t)0x71ed, (q15_t)0x71ea, (q15_t)0x71e7, (q15_t)0x71e4,
  (q15_t)0x71e1, (q15_t)0x71df, (q15_t)0x71dc, (q15_t)0x71d9, (q15_t)0x71d6, (q15_t)0x71d3, (q15_t)0x71d0, (q15_t)0x71cd,
  (q15_t)0x71ca, (q15_t)0x71c8, (q15_t)0x71c5, (q15_t)0x71c2, (q15_t)0x71bf, (q15_t)0x71bc, (q15_t)0x71b9, (q15_t)0x71b6,
  (q15_t)0x71b3, (q15_t)0x71b0, (q15_t)0x71ae, (q15_t)0x71ab, (q15_t)0x71a8, (q15_t)0x71a5, (q15_t)0x71a2, (q15_t)0x719f,
  (q15_t)0x719c, (q15_t)0x7199, (q15_t)0x7196, (q15_t)0x7194, (q15_t)0x7191, (q15_t)0x718e, (q15_t)0x718b, (q15_t)0x7188,
  (q15_t)0x7185, (q15_t)0x7182, (q15_t)0x717f, (q15_t)0x717c, (q15_t)0x7179, (q15_t)0x7177, (q15_t)0x7174, (q15_t)0x7171,
  (q15_t)0x716e, (q15_t)0x716b, (q15_t)0x7168, (q15_t)0x7165, (q15_t)0x7162, (q15_t)0x715f, (q15_t)0x715c, (q15_t)0x7159,
  (q15_t)0x7156, (q15_t)0x7154, (q15_t)0x7151, (q15_t)0x714e, (q15_t)0x714b, (q15_t)0x7148, (q15_t)0x7145, (q15_t)0x7142,
  (q15_t)0x713f, (q15_t)0x713c, (q15_t)0x7139, (q15_t)0x7136, (q15_t)0x7133, (q15_t)0x7130, (q15_t)0x712d, (q15_t)0x712b,
  (q15_t)0x7128, (q15_t)0x7125, (q15_t)0x7122, (q15_t)0x711f, (q15_t)0x711c, (q15_t)0x7119, (q15_t)0x7116, (q15_t)0x7113,
  (q15_t)0x7110, (q15_t)0x710d, (q15_t)0x710a, (q15_t)0x7107, (q15_t)0x7104, (q15_t)0x7101, (q15_t)0x70fe, (q15_t)0x70fb,
  (q15_t)0x70f8, (q15_t)0x70f6, (q15_t)0x70f3, (q15_t)0x70f0, (q15_t)0x70ed, (q15_t)0x70ea, (q15_t)0x70e7, (q15_t)0x70e4,
  (q15_t)0x70e1, (q15_t)0x70de, (q15_t)0x70db, (q15_t)0x70d8, (q15_t)0x70d5, (q15_t)0x70d2, (q15_t)0x70cf, (q15_t)0x70cc,
  (q15_t)0x70c9, (q15_t)0x70c6, (q15_t)0x70c3, (q15_t)0x70c0, (q15_t)0x70bd, (q15_t)0x70ba, (q15_t)0x70b7, (q15_t)0x70b4,
  (q15_t)0x70b1, (q15_t)0x70ae, (q15_t)0x70ab, (q15_t)0x70a8, (q15_t)0x70a5, (q15_t)0x70a2, (q15_t)0x709f, (q15_t)0x709c,
  (q15_t)0x7099, (q15_t)0x7096, (q15_t)0x7093, (q15_t)0x7090, (q15_t)0x708d, (q15_t)0x708a, (q15_t)0x7087, (q15_t)0x7084,
  (q15_t)0x7081, (q15_t)0x707e, (q15_t)0x707b, (q15_t)0x7078, (q15_t)0x7075, (q15_t)0x7072, (q15_t)0x706f, (q15_t)0x706c,
  (q15_t)0x7069, (q15_t)0x7066, (q15_t)0x7063, (q15_t)0x7060, (q15_t)0x705d, (q15_t)0x705a, (q15_t)0x7057, (q15_t)0x7054,
  (q15_t)0x7051, (q15_t)0x704e, (q15_t)0x704b, (q15_t)0x7048, (q15_t)0x7045, (q15_t)0x7042, (q15_t)0x703f, (q15_t)0x703c,
  (q15_t)0x7039, (q15_t)0x7036, (q15_t)0x7033, (q15_t)0x7030, (q15_t)0x702d, (q15_t)0x702a, (q15_t)0x7027, (q15_t)0x7024,
  (q15_t)0x7021, (q15_t)0x701e, (q15_t)0x701b, (q15_t)0x7018, (q15_t)0x7015, (q15_t)0x7012, (q15_t)0x700f, (q15_t)0x700c,
  (q15_t)0x7009, (q15_t)0x7006, (q15_t)0x7003, (q15_t)0x7000, (q15_t)0x6ffd, (q15_t)0x6ffa, (q15_t)0x6ff7, (q15_t)0x6ff3,
  (q15_t)0x6ff0, (q15_t)0x6fed, (q15_t)0x6fea, (q15_t)0x6fe7, (q15_t)0x6fe4, (q15_t)0x6fe1, (q15_t)0x6fde, (q15_t)0x6fdb,
  (q15_t)0x6fd8, (q15_t)0x6fd5, (q15_t)0x6fd2, (q15_t)0x6fcf, (q15_t)0x6fcc, (q15_t)0x6fc9, (q15_t)0x6fc6, (q15_t)0x6fc3,
  (q15_t)0x6fc0, (q15_t)0x6fbc, (q15_t)0x6fb9, (q15_t)0x6fb6, (q15_t)0x6fb3, (q15_t)0x6fb0, (q15_t)0x6fad, (q15_t)0x6faa,
  (q15_t)0x6fa7, (q15_t)0x6fa4, (q15_t)0x6fa1, (q15_t)0x6f9e, (q15_t)0x6f9b, (q15_t)0x6f98, (q15_t)0x6f95, (q15_t)0x6f91,
  (q15_t)0x6f8e, (q15_t)0x6f8b, (q15_t)0x6f88, (q15_t)0x6f85, (q15_t)0x6f82, (q15_t)0x6f7f, (q15_t)0x6f7c, (q15_t)0x6f79,
  (q15_t)0x6f76, (q15_t)0x6f73, (q15_t)0x6f70, (q15_t)0x6f6c, (q15_t)0x6f69, (q15_t)0x6f66, (q15_t)0x6f63, (q15_t)0x6f60,
  (q15_t)0x6f5d, (q15_t)0x6f5a, (q15_t)0x6f57, (q15_t)0x6f54, (q15_t)0x6f51, (q15_t)0x6f4d, (q15_t)0x6f4a, (q15_t)0x6f47,
  (q15_t)0x6f44, (q15_t)0x6f41, (q15_t)0x6f3e, (q15_t)0x6f3b, (q15_t)0x6f38, (q15_t)0x6f35, (q15_t)0x6f31, (q15_t)0x6f2e,
  (q15_t)0x6f2b, (q15_t)0x6f28, (q15_t)0x6f25, (q15_t)0x6f22, (q15_t)0x6f1f, (q15_t)0x6f1c, (q15_t)0x6f19, (q15_t)0x6f15,
  (q15_t)0x6f12, (q15_t)0x6f0f, (q15_t)0x6f0c, (q15_t)0x6f09, (q15_t)0x6f06, (q15_t)0x6f03, (q15_t)0x6f00, (q15_t)0x6efc,
  (q15_t)0x6ef9, (q15_t)0x6ef6, (q15_t)0x6ef3, (q15_t)0x6ef0, (q15_t)0x6eed, (q15_t)0x6eea, (q15_t)0x6ee7, (q15_t)0x6ee3,
  (q15_t)0x6ee0, (q15_t)0x6edd, (q15_t)0x6eda, (q15_t)0x6ed7, (q15_t)0x6ed4, (q15_t)0x6ed1, (q15_t)0x6ecd, (q15_t)0x6eca,
  (q15_t)0x6ec7, (q15_t)0x6ec4, (q15_t)0x6ec1, (q15_t)0x6ebe, (q15_t)0x6eba, (q15_t)0x6eb7, (q15_t)0x6eb4, (q15_t)0x6eb1,
  (q15_t)0x6eae, (q15_t)0x6eab, (q15_t)0x6ea8, (q15_t)0x6ea4, (q15_t)0x6ea1, (q15_t)0x6e9e, (q15_t)0x6e9b, (q15_t)0x6e98,
  (q15_t)0x6e95, (q15_t)0x6e91, (q15_t)0x6e8e, (q15_t)0x6e8b, (q15_t)0x6e88, (q15_t)0x6e85, (q15_t)0x6e82, (q15_t)0x6e7e,
  (q15_t)0x6e7b, (q15_t)0x6e78, (q15_t)0x6e75, (q15_t)0x6e72, (q15_t)0x6e6f, (q15_t)0x6e6b, (q15_t)0x6e68, (q15_t)0x6e65,
  (q15_t)0x6e62, (q15_t)0x6e5f, (q15_t)0x6e5b, (q15_t)0x6e58, (q15_t)0x6e55, (q15_t)0x6e52, (q15_t)0x6e4f, (q15_t)0x6e4c,
  (q15_t)0x6e48, (q15_t)0x6e45, (q15_t)0x6e42, (q15_t)0x6e3f, (q15_t)0x6e3c, (q15_t)0x6e38, (q15_t)0x6e35, (q15_t)0x6e32,
  (q15_t)0x6e2f, (q15_t)0x6e2c, (q15_t)0x6e28, (q15_t)0x6e25, (q15_t)0x6e22, (q15_t)0x6e1f, (q15_t)0x6e1c, (q15_t)0x6e18,
  (q15_t)0x6e15, (q15_t)0x6e12, (q15_t)0x6e0f, (q15_t)0x6e0c, (q15_t)0x6e08, (q15_t)0x6e05, (q15_t)0x6e02, (q15_t)0x6dff,
  (q15_t)0x6dfb, (q15_t)0x6df8, (q15_t)0x6df5, (q15_t)0x6df2, (q15_t)0x6def, (q15_t)0x6deb, (q15_t)0x6de8, (q15_t)0x6de5,
  (q15_t)0x6de2, (q15_t)0x6ddf, (q15_t)0x6ddb, (q15_t)0x6dd8, (q15_t)0x6dd5, (q15_t)0x6dd2, (q15_t)0x6dce, (q15_t)0x6dcb,
  (q15_t)0x6dc8, (q15_t)0x6dc5, (q15_t)0x6dc1, (q15_t)0x6dbe, (q15_t)0x6dbb, (q15_t)0x6db8, (q15_t)0x6db5, (q15_t)0x6db1,
  (q15_t)0x6dae, (q15_t)0x6dab, (q15_t)0x6da8, (q15_t)0x6da4, (q15_t)0x6da1, (q15_t)0x6d9e, (q15_t)0x6d9b, (q15_t)0x6d97,
  (q15_t)0x6d94, (q15_t)0x6d91, (q15_t)0x6d8e, (q15_t)0x6d8a, (q15_t)0x6d87, (q15_t)0x6d84, (q15_t)0x6d81, (q15_t)0x6d7d,
  (q15_t)0x6d7a, (q15_t)0x6d77, (q15_t)0x6d74, (q15_t)0x6d70, (q15_t)0x6d6d, (q15_t)0x6d6a, (q15_t)0x6d67, (q15_t)0x6d63,
  (q15_t)0x6d60, (q15_t)0x6d5d, (q15_t)0x6d59, (q15_t)0x6d56, (q15_t)0x6d53, (q15_t)0x6d50, (q15_t)0x6d4c, (q15_t)0x6d49,
  (q15_t)0x6d46, (q15_t)0x6d43, (q15_t)0x6d3f, (q15_t)0x6d3c, (q15_t)0x6d39, (q15_t)0x6d36, (q15_t)0x6d32, (q15_t)0x6d2f,
  (q15_t)0x6d2c, (q15_t)0x6d28, (q15_t)0x6d25, (q15_t)0x6d22, (q15_t)0x6d1f, (q15_t)0x6d1b, (q15_t)0x6d18, (q15_t)0x6d15,
  (q15_t)0x6d11, (q15_t)0x6d0e, (q15_t)0x6d0b, (q15_t)0x6d08, (q15_t)0x6d04, (q15_t)0x6d01, (q15_t)0x6cfe, (q15_t)0x6cfa,
  (q15_t)0x6cf7, (q15_t)0x6cf4, (q15_t)0x6cf0, (q15_t)0x6ced, (q15_t)0x6cea, (q15_t)0x6ce7, (q15_t)0x6ce3, (q15_t)0x6ce0,
  (q15_t)0x6cdd, (q15_t)0x6cd9, (q15_t)0x6cd6, (q15_t)0x6cd3, (q15_t)0x6ccf, (q15_t)0x6ccc, (q15_t)0x6cc9, (q15_t)0x6cc5,
  (q15_t)0x6cc2, (q15_t)0x6cbf, (q15_t)0x6cbc, (q15_t)0x6cb8, (q15_t)0x6cb5, (q15_t)0x6cb2, (q15_t)0x6cae, (q15_t)0x6cab,
  (q15_t)0x6ca8, (q15_t)0x6ca4, (q15_t)0x6ca1, (q15_t)0x6c9e, (q15_t)0x6c9a, (q15_t)0x6c97, (q15_t)0x6c94, (q15_t)0x6c90,
  (q15_t)0x6c8d, (q15_t)0x6c8a, (q15_t)0x6c86, (q15_t)0x6c83, (q15_t)0x6c80, (q15_t)0x6c7c, (q15_t)0x6c79, (q15_t)0x6c76,
  (q15_t)0x6c72, (q15_t)0x6c6f, (q15_t)0x6c6c, (q15_t)0x6c68, (q15_t)0x6c65, (q15_t)0x6c62, (q15_t)0x6c5e, (q15_t)0x6c5b,
  (q15_t)0x6c58, (q15_t)0x6c54, (q15_t)0x6c51, (q15_t)0x6c4e, (q15_t)0x6c4a, (q15_t)0x6c47, (q15_t)0x6c44, (q15_t)0x6c40,
  (q15_t)0x6c3d, (q15_t)0x6c39, (q15_t)0x6c36, (q15_t)0x6c33, (q15_t)0x6c2f, (q15_t)0x6c2c, (q15_t)0x6c29, (q15_t)0x6c25,
  (q15_t)0x6c22, (q15_t)0x6c1f, (q15_t)0x6c1b, (q15_t)0x6c18, (q15_t)0x6c15, (q15_t)0x6c11, (q15_t)0x6c0e, (q15_t)0x6c0a,
  (q15_t)0x6c07, (q15_t)0x6c04, (q15_t)0x6c00, (q15_t)0x6bfd, (q15_t)0x6bfa, (q15_t)0x6bf6, (q15_t)0x6bf3, (q15_t)0x6bef,
  (q15_t)0x6bec, (q15_t)0x6be9, (q15_t)0x6be5, (q15_t)0x6be2, (q15_t)0x6bdf, (q15_t)0x6bdb, (q15_t)0x6bd8, (q15_t)0x6bd4,
  (q15_t)0x6bd1, (q15_t)0x6bce, (q15_t)0x6bca, (q15_t)0x6bc7, (q15_t)0x6bc3, (q15_t)0x6bc0, (q15_t)0x6bbd, (q15_t)0x6bb9,
  (q15_t)0x6bb6, (q15_t)0x6bb2, (q15_t)0x6baf, (q15_t)0x6bac, (q15_t)0x6ba8, (q15_t)0x6ba5, (q15_t)0x6ba1, (q15_t)0x6b9e,
  (q15_t)0x6b9b, (q15_t)0x6b97, (q15_t)0x6b94, (q15_t)0x6b90, (q15_t)0x6b8d, (q15_t)0x6b8a, (q15_t)0x6b86, (q15_t)0x6b83,
  (q15_t)0x6b7f, (q15_t)0x6b7c, (q15_t)0x6b79, (q15_t)0x6b75, (q15_t)0x6b72, (q15_t)0x6b6e, (q15_t)0x6b6b, (q15_t)0x6b68,
  (q15_t)0x6b64, (q15_t)0x6b61, (q15_t)0x6b5d, (q15_t)0x6b5a, (q15_t)0x6b56, (q15_t)0x6b53, (q15_t)0x6b50, (q15_t)0x6b4c,
  (q15_t)0x6b49, (q15_t)0x6b45, (q15_t)0x6b42, (q15_t)0x6b3e, (q15_t)0x6b3b, (q15_t)0x6b38, (q15_t)0x6b34, (q15_t)0x6b31,
  (q15_t)0x6b2d, (q15_t)0x6b2a, (q15_t)0x6b26, (q15_t)0x6b23, (q15_t)0x6b20, (q15_t)0x6b1c, (q15_t)0x6b19, (q15_t)0x6b15,
  (q15_t)0x6b12, (q15_t)0x6b0e, (q15_t)0x6b0b, (q15_t)0x6b07, (q15_t)0x6b04, (q15_t)0x6b01, (q15_t)0x6afd, (q15_t)0x6afa,
  (q15_t)0x6af6, (q15_t)0x6af3, (q15_t)0x6aef, (q15_t)0x6aec, (q15_t)0x6ae8, (q15_t)0x6ae5, (q15_t)0x6ae1, (q15_t)0x6ade,
  (q15_t)0x6adb, (q15_t)0x6ad7, (q15_t)0x6ad4, (q15_t)0x6ad0, (q15_t)0x6acd, (q15_t)0x6ac9, (q15_t)0x6ac6, (q15_t)0x6ac2,
  (q15_t)0x6abf, (q15_t)0x6abb, (q15_t)0x6ab8, (q15_t)0x6ab4, (q15_t)0x6ab1, (q15_t)0x6aae, (q15_t)0x6aaa, (q15_t)0x6aa7,
  (q15_t)0x6aa3, (q15_t)0x6aa0, (q15_t)0x6a9c, (q15_t)0x6a99, (q15_t)0x6a95, (q15_t)0x6a92, (q15_t)0x6a8e, (q15_t)0x6a8b,
  (q15_t)0x6a87, (q15_t)0x6a84, (q15_t)0x6a80, (q15_t)0x6a7d, (q15_t)0x6a79, (q15_t)0x6a76, (q15_t)0x6a72, (q15_t)0x6a6f,
  (q15_t)0x6a6b, (q15_t)0x6a68, (q15_t)0x6a64, (q15_t)0x6a61, (q15_t)0x6a5d, (q15_t)0x6a5a, (q15_t)0x6a56, (q15_t)0x6a53,
  (q15_t)0x6a4f, (q15_t)0x6a4c, (q15_t)0x6a48, (q15_t)0x6a45, (q15_t)0x6a41, (q15_t)0x6a3e, (q15_t)0x6a3a, (q15_t)0x6a37,
  (q15_t)0x6a33, (q15_t)0x6a30, (q15_t)0x6a2c, (q15_t)0x6a29, (q15_t)0x6a25, (q15_t)0x6a22, (q15_t)0x6a1e, (q15_t)0x6a1b,
  (q15_t)0x6a17, (q15_t)0x6a14, (q15_t)0x6a10, (q15_t)0x6a0d, (q15_t)0x6a09, (q15_t)0x6a06, (q15_t)0x6a02, (q15_t)0x69ff,
  (q15_t)0x69fb, (q15_t)0x69f8, (q15_t)0x69f4, (q15_t)0x69f1, (q15_t)0x69ed, (q15_t)0x69e9, (q15_t)0x69e6, (q15_t)0x69e2,
  (q15_t)0x69df, (q15_t)0x69db, (q15_t)0x69d8, (q15_t)0x69d4, (q15_t)0x69d1, (q15_t)0x69cd, (q15_t)0x69ca, (q15_t)0x69c6,
  (q15_t)0x69c3, (q15_t)0x69bf, (q15_t)0x69bc, (q15_t)0x69b8, (q15_t)0x69b4, (q15_t)0x69b1, (q15_t)0x69ad, (q15_t)0x69aa,
  (q15_t)0x69a6, (q15_t)0x69a3, (q15_t)0x699f, (q15_t)0x699c, (q15_t)0x6998, (q15_t)0x6995, (q15_t)0x6991, (q15_t)0x698d,
  (q15_t)0x698a, (q15_t)0x6986, (q15_t)0x6983, (q15_t)0x697f, (q15_t)0x697c, (q15_t)0x6978, (q15_t)0x6975, (q15_t)0x6971,
  (q15_t)0x696d, (q15_t)0x696a, (q15_t)0x6966, (q15_t)0x6963, (q15_t)0x695f, (q15_t)0x695c, (q15_t)0x6958, (q15_t)0x6954,
  (q15_t)0x6951, (q15_t)0x694d, (q15_t)0x694a, (q15_t)0x6946, (q15_t)0x6943, (q15_t)0x693f, (q15_t)0x693b, (q15_t)0x6938,
  (q15_t)0x6934, (q15_t)0x6931, (q15_t)0x692d, (q15_t)0x692a, (q15_t)0x6926, (q15_t)0x6922, (q15_t)0x691f, (q15_t)0x691b,
  (q15_t)0x6918, (q15_t)0x6914, (q15_t)0x6910, (q15_t)0x690d, (q15_t)0x6909, (q15_t)0x6906, (q15_t)0x6902, (q15_t)0x68fe,
  (q15_t)0x68fb, (q15_t)0x68f7, (q15_t)0x68f4, (q15_t)0x68f0, (q15_t)0x68ec, (q15_t)0x68e9, (q15_t)0x68e5, (q15_t)0x68e2,
  (q15_t)0x68de, (q15_t)0x68da, (q15_t)0x68d7, (q15_t)0x68d3, (q15_t)0x68d0, (q15_t)0x68cc, (q15_t)0x68c8, (q15_t)0x68c5,
  (q15_t)0x68c1, (q15_t)0x68be, (q15_t)0x68ba, (q15_t)0x68b6, (q15_t)0x68b3, (q15_t)0x68af, (q15_t)0x68ac, (q15_t)0x68a8,
  (q15_t)0x68a4, (q15_t)0x68a1, (q15_t)0x689d, (q15_t)0x6899, (q15_t)0x6896, (q15_t)0x6892, (q15_t)0x688f, (q15_t)0x688b,
  (q15_t)0x6887, (q15_t)0x6884, (q15_t)0x6880, (q15_t)0x687c, (q15_t)0x6879, (q15_t)0x6875, (q15_t)0x6872, (q15_t)0x686e,
  (q15_t)0x686a, (q15_t)0x6867, (q15_t)0x6863, (q15_t)0x685f, (q15_t)0x685c, (q15_t)0x6858, (q15_t)0x6854, (q15_t)0x6851,
  (q15_t)0x684d, (q15_t)0x684a, (q15_t)0x6846, (q15_t)0x6842, (q15_t)0x683f, (q15_t)0x683b, (q15_t)0x6837, (q15_t)0x6834,
  (q15_t)0x6830, (q15_t)0x682c, (q15_t)0x6829, (q15_t)0x6825, (q15_t)0x6821, (q15_t)0x681e, (q15_t)0x681a, (q15_t)0x6816,
  (q15_t)0x6813, (q15_t)0x680f, (q15_t)0x680b, (q15_t)0x6808, (q15_t)0x6804, (q15_t)0x6800, (q15_t)0x67fd, (q15_t)0x67f9,
  (q15_t)0x67f5, (q15_t)0x67f2, (q15_t)0x67ee, (q15_t)0x67ea, (q15_t)0x67e7, (q15_t)0x67e3, (q15_t)0x67df, (q15_t)0x67dc,
  (q15_t)0x67d8, (q15_t)0x67d4, (q15_t)0x67d1, (q15_t)0x67cd, (q15_t)0x67c9, (q15_t)0x67c6, (q15_t)0x67c2, (q15_t)0x67be,
  (q15_t)0x67bb, (q15_t)0x67b7, (q15_t)0x67b3, (q15_t)0x67b0, (q15_t)0x67ac, (q15_t)0x67a8, (q15_t)0x67a5, (q15_t)0x67a1,
  (q15_t)0x679d, (q15_t)0x679a, (q15_t)0x6796, (q15_t)0x6792, (q15_t)0x678e, (q15_t)0x678b, (q15_t)0x6787, (q15_t)0x6783,
  (q15_t)0x6780, (q15_t)0x677c, (q15_t)0x6778, (q15_t)0x6775, (q15_t)0x6771, (q15_t)0x676d, (q15_t)0x6769, (q15_t)0x6766,
  (q15_t)0x6762, (q15_t)0x675e, (q15_t)0x675b, (q15_t)0x6757, (q15_t)0x6753, (q15_t)0x6750, (q15_t)0x674c, (q15_t)0x6748,
  (q15_t)0x6744, (q15_t)0x6741, (q15_t)0x673d, (q15_t)0x6739, (q15_t)0x6736, (q15_t)0x6732, (q15_t)0x672e, (q15_t)0x672a,
  (q15_t)0x6727, (q15_t)0x6723, (q15_t)0x671f, (q15_t)0x671c, (q15_t)0x6718, (q15_t)0x6714, (q15_t)0x6710, (q15_t)0x670d,
  (q15_t)0x6709, (q15_t)0x6705, (q15_t)0x6701, (q15_t)0x66fe, (q15_t)0x66fa, (q15_t)0x66f6, (q15_t)0x66f3, (q15_t)0x66ef,
  (q15_t)0x66eb, (q15_t)0x66e7, (q15_t)0x66e4, (q15_t)0x66e0, (q15_t)0x66dc, (q15_t)0x66d8, (q15_t)0x66d5, (q15_t)0x66d1,
  (q15_t)0x66cd, (q15_t)0x66c9, (q15_t)0x66c6, (q15_t)0x66c2, (q15_t)0x66be, (q15_t)0x66ba, (q15_t)0x66b7, (q15_t)0x66b3,
  (q15_t)0x66af, (q15_t)0x66ab, (q15_t)0x66a8, (q15_t)0x66a4, (q15_t)0x66a0, (q15_t)0x669c, (q15_t)0x6699, (q15_t)0x6695,
  (q15_t)0x6691, (q15_t)0x668d, (q15_t)0x668a, (q15_t)0x6686, (q15_t)0x6682, (q15_t)0x667e, (q15_t)0x667b, (q15_t)0x6677,
  (q15_t)0x6673, (q15_t)0x666f, (q15_t)0x666b, (q15_t)0x6668, (q15_t)0x6664, (q15_t)0x6660, (q15_t)0x665c, (q15_t)0x6659,
  (q15_t)0x6655, (q15_t)0x6651, (q15_t)0x664d, (q15_t)0x664a, (q15_t)0x6646, (q15_t)0x6642, (q15_t)0x663e, (q15_t)0x663a,
  (q15_t)0x6637, (q15_t)0x6633, (q15_t)0x662f, (q15_t)0x662b, (q15_t)0x6627, (q15_t)0x6624, (q15_t)0x6620, (q15_t)0x661c,
  (q15_t)0x6618, (q15_t)0x6615, (q15_t)0x6611, (q15_t)0x660d, (q15_t)0x6609, (q15_t)0x6605, (q15_t)0x6602, (q15_t)0x65fe,
  (q15_t)0x65fa, (q15_t)0x65f6, (q15_t)0x65f2, (q15_t)0x65ef, (q15_t)0x65eb, (q15_t)0x65e7, (q15_t)0x65e3, (q15_t)0x65df,
  (q15_t)0x65dc, (q15_t)0x65d8, (q15_t)0x65d4, (q15_t)0x65d0, (q15_t)0x65cc, (q15_t)0x65c9, (q15_t)0x65c5, (q15_t)0x65c1,
  (q15_t)0x65bd, (q15_t)0x65b9, (q15_t)0x65b5, (q15_t)0x65b2, (q15_t)0x65ae, (q15_t)0x65aa, (q15_t)0x65a6, (q15_t)0x65a2,
  (q15_t)0x659f, (q15_t)0x659b, (q15_t)0x6597, (q15_t)0x6593, (q15_t)0x658f, (q15_t)0x658b, (q15_t)0x6588, (q15_t)0x6584,
  (q15_t)0x6580, (q15_t)0x657c, (q15_t)0x6578, (q15_t)0x6574, (q15_t)0x6571, (q15_t)0x656d, (q15_t)0x6569, (q15_t)0x6565,
  (q15_t)0x6561, (q15_t)0x655d, (q15_t)0x655a, (q15_t)0x6556, (q15_t)0x6552, (q15_t)0x654e, (q15_t)0x654a, (q15_t)0x6546,
  (q15_t)0x6543, (q15_t)0x653f, (q15_t)0x653b, (q15_t)0x6537, (q15_t)0x6533, (q15_t)0x652f, (q15_t)0x652c, (q15_t)0x6528,
  (q15_t)0x6524, (q15_t)0x6520, (q15_t)0x651c, (q15_t)0x6518, (q15_t)0x6514, (q15_t)0x6511, (q15_t)0x650d, (q15_t)0x6509,
  (q15_t)0x6505, (q15_t)0x6501, (q15_t)0x64fd, (q15_t)0x64f9, (q15_t)0x64f6, (q15_t)0x64f2, (q15_t)0x64ee, (q15_t)0x64ea,
  (q15_t)0x64e6, (q15_t)0x64e2, (q15_t)0x64de, (q15_t)0x64db, (q15_t)0x64d7, (q15_t)0x64d3, (q15_t)0x64cf, (q15_t)0x64cb,
  (q15_t)0x64c7, (q15_t)0x64c3, (q15_t)0x64bf, (q15_t)0x64bc, (q15_t)0x64b8, (q15_t)0x64b4, (q15_t)0x64b0, (q15_t)0x64ac,
  (q15_t)0x64a8, (q15_t)0x64a4, (q15_t)0x64a0, (q15_t)0x649c, (q15_t)0x6499, (q15_t)0x6495, (q15_t)0x6491, (q15_t)0x648d,
  (q15_t)0x6489, (q15_t)0x6485, (q15_t)0x6481, (q15_t)0x647d, (q15_t)0x6479, (q15_t)0x6476, (q15_t)0x6472, (q15_t)0x646e,
  (q15_t)0x646a, (q15_t)0x6466, (q15_t)0x6462, (q15_t)0x645e, (q15_t)0x645a, (q15_t)0x6456, (q15_t)0x6453, (q15_t)0x644f,
  (q15_t)0x644b, (q15_t)0x6447, (q15_t)0x6443, (q15_t)0x643f, (q15_t)0x643b, (q15_t)0x6437, (q15_t)0x6433, (q15_t)0x642f,
  (q15_t)0x642b, (q15_t)0x6428, (q15_t)0x6424, (q15_t)0x6420, (q15_t)0x641c, (q15_t)0x6418, (q15_t)0x6414, (q15_t)0x6410,
  (q15_t)0x640c, (q15_t)0x6408, (q15_t)0x6404, (q15_t)0x6400, (q15_t)0x63fc, (q15_t)0x63f9, (q15_t)0x63f5, (q15_t)0x63f1,
  (q15_t)0x63ed, (q15_t)0x63e9, (q15_t)0x63e5, (q15_t)0x63e1, (q15_t)0x63dd, (q15_t)0x63d9, (q15_t)0x63d5, (q15_t)0x63d1,
  (q15_t)0x63cd, (q15_t)0x63c9, (q15_t)0x63c5, (q15_t)0x63c1, (q15_t)0x63be, (q15_t)0x63ba, (q15_t)0x63b6, (q15_t)0x63b2,
  (q15_t)0x63ae, (q15_t)0x63aa, (q15_t)0x63a6, (q15_t)0x63a2, (q15_t)0x639e, (q15_t)0x639a, (q15_t)0x6396, (q15_t)0x6392,
  (q15_t)0x638e, (q15_t)0x638a, (q15_t)0x6386, (q15_t)0x6382, (q15_t)0x637e, (q15_t)0x637a, (q15_t)0x6377, (q15_t)0x6373,
  (q15_t)0x636f, (q15_t)0x636b, (q15_t)0x6367, (q15_t)0x6363, (q15_t)0x635f, (q15_t)0x635b, (q15_t)0x6357, (q15_t)0x6353,
  (q15_t)0x634f, (q15_t)0x634b, (q15_t)0x6347, (q15_t)0x6343, (q15_t)0x633f, (q15_t)0x633b, (q15_t)0x6337, (q15_t)0x6333,
  (q15_t)0x632f, (q15_t)0x632b, (q15_t)0x6327, (q15_t)0x6323, (q15_t)0x631f, (q15_t)0x631b, (q15_t)0x6317, (q15_t)0x6313,
  (q15_t)0x630f, (q15_t)0x630b, (q15_t)0x6307, (q15_t)0x6303, (q15_t)0x62ff, (q15_t)0x62fb, (q15_t)0x62f7, (q15_t)0x62f3,
  (q15_t)0x62f0, (q15_t)0x62ec, (q15_t)0x62e8, (q15_t)0x62e4, (q15_t)0x62e0, (q15_t)0x62dc, (q15_t)0x62d8, (q15_t)0x62d4,
  (q15_t)0x62d0, (q15_t)0x62cc, (q15_t)0x62c8, (q15_t)0x62c4, (q15_t)0x62c0, (q15_t)0x62bc, (q15_t)0x62b8, (q15_t)0x62b4,
  (q15_t)0x62b0, (q15_t)0x62ac, (q15_t)0x62a8, (q15_t)0x62a4, (q15_t)0x62a0, (q15_t)0x629c, (q15_t)0x6298, (q15_t)0x6294,
  (q15_t)0x6290, (q15_t)0x628c, (q15_t)0x6288, (q15_t)0x6284, (q15_t)0x6280, (q15_t)0x627c, (q15_t)0x6278, (q15_t)0x6273,
  (q15_t)0x626f, (q15_t)0x626b, (q15_t)0x6267, (q15_t)0x6263, (q15_t)0x625f, (q15_t)0x625b, (q15_t)0x6257, (q15_t)0x6253,
  (q15_t)0x624f, (q15_t)0x624b, (q15_t)0x6247, (q15_t)0x6243, (q15_t)0x623f, (q15_t)0x623b, (q15_t)0x6237, (q15_t)0x6233,
  (q15_t)0x622f, (q15_t)0x622b, (q15_t)0x6227, (q15_t)0x6223, (q15_t)0x621f, (q15_t)0x621b, (q15_t)0x6217, (q15_t)0x6213,
  (q15_t)0x620f, (q15_t)0x620b, (q15_t)0x6207, (q15_t)0x6203, (q15_t)0x61ff, (q15_t)0x61fb, (q15_t)0x61f7, (q15_t)0x61f3,
  (q15_t)0x61ee, (q15_t)0x61ea, (q15_t)0x61e6, (q15_t)0x61e2, (q15_t)0x61de, (q15_t)0x61da, (q15_t)0x61d6, (q15_t)0x61d2,
  (q15_t)0x61ce, (q15_t)0x61ca, (q15_t)0x61c6, (q15_t)0x61c2, (q15_t)0x61be, (q15_t)0x61ba, (q15_t)0x61b6, (q15_t)0x61b2,
  (q15_t)0x61ae, (q15_t)0x61aa, (q15_t)0x61a6, (q15_t)0x61a1, (q15_t)0x619d, (q15_t)0x6199, (q15_t)0x6195, (q15_t)0x6191,
  (q15_t)0x618d, (q15_t)0x6189, (q15_t)0x6185, (q15_t)0x6181, (q15_t)0x617d, (q15_t)0x6179, (q15_t)0x6175, (q15_t)0x6171,
  (q15_t)0x616d, (q15_t)0x6168, (q15_t)0x6164, (q15_t)0x6160, (q15_t)0x615c, (q15_t)0x6158, (q15_t)0x6154, (q15_t)0x6150,
  (q15_t)0x614c, (q15_t)0x6148, (q15_t)0x6144, (q15_t)0x6140, (q15_t)0x613c, (q15_t)0x6137, (q15_t)0x6133, (q15_t)0x612f,
  (q15_t)0x612b, (q15_t)0x6127, (q15_t)0x6123, (q15_t)0x611f, (q15_t)0x611b, (q15_t)0x6117, (q15_t)0x6113, (q15_t)0x610f,
  (q15_t)0x610a, (q15_t)0x6106, (q15_t)0x6102, (q15_t)0x60fe, (q15_t)0x60fa, (q15_t)0x60f6, (q15_t)0x60f2, (q15_t)0x60ee,
  (q15_t)0x60ea, (q15_t)0x60e6, (q15_t)0x60e1, (q15_t)0x60dd, (q15_t)0x60d9, (q15_t)0x60d5, (q15_t)0x60d1, (q15_t)0x60cd,
  (q15_t)0x60c9, (q15_t)0x60c5, (q15_t)0x60c1, (q15_t)0x60bc, (q15_t)0x60b8, (q15_t)0x60b4, (q15_t)0x60b0, (q15_t)0x60ac,
  (q15_t)0x60a8, (q15_t)0x60a4, (q15_t)0x60a0, (q15_t)0x609c, (q15_t)0x6097, (q15_t)0x6093, (q15_t)0x608f, (q15_t)0x608b,
  (q15_t)0x6087, (q15_t)0x6083, (q15_t)0x607f, (q15_t)0x607b, (q15_t)0x6076, (q15_t)0x6072, (q15_t)0x606e, (q15_t)0x606a,
  (q15_t)0x6066, (q15_t)0x6062, (q15_t)0x605e, (q15_t)0x6059, (q15_t)0x6055, (q15_t)0x6051, (q15_t)0x604d, (q15_t)0x6049,
  (q15_t)0x6045, (q15_t)0x6041, (q15_t)0x603c, (q15_t)0x6038, (q15_t)0x6034, (q15_t)0x6030, (q15_t)0x602c, (q15_t)0x6028,
  (q15_t)0x6024, (q15_t)0x601f, (q15_t)0x601b, (q15_t)0x6017, (q15_t)0x6013, (q15_t)0x600f, (q15_t)0x600b, (q15_t)0x6007,
  (q15_t)0x6002, (q15_t)0x5ffe, (q15_t)0x5ffa, (q15_t)0x5ff6, (q15_t)0x5ff2, (q15_t)0x5fee, (q15_t)0x5fe9, (q15_t)0x5fe5,
  (q15_t)0x5fe1, (q15_t)0x5fdd, (q15_t)0x5fd9, (q15_t)0x5fd5, (q15_t)0x5fd0, (q15_t)0x5fcc, (q15_t)0x5fc8, (q15_t)0x5fc4,
  (q15_t)0x5fc0, (q15_t)0x5fbc, (q15_t)0x5fb7, (q15_t)0x5fb3, (q15_t)0x5faf, (q15_t)0x5fab, (q15_t)0x5fa7, (q15_t)0x5fa3,
  (q15_t)0x5f9e, (q15_t)0x5f9a, (q15_t)0x5f96, (q15_t)0x5f92, (q15_t)0x5f8e, (q15_t)0x5f8a, (q15_t)0x5f85, (q15_t)0x5f81,
  (q15_t)0x5f7d, (q15_t)0x5f79, (q15_t)0x5f75, (q15_t)0x5f70, (q15_t)0x5f6c, (q15_t)0x5f68, (q15_t)0x5f64, (q15_t)0x5f60,
  (q15_t)0x5f5b, (q15_t)0x5f57, (q15_t)0x5f53, (q15_t)0x5f4f, (q15_t)0x5f4b, (q15_t)0x5f46, (q15_t)0x5f42, (q15_t)0x5f3e,
  (q15_t)0x5f3a, (q15_t)0x5f36, (q15_t)0x5f31, (q15_t)0x5f2d, (q15_t)0x5f29, (q15_t)0x5f25, (q15_t)0x5f21, (q15_t)0x5f1c,
  (q15_t)0x5f18, (q15_t)0x5f14, (q15_t)0x5f10, (q15_t)0x5f0c, (q15_t)0x5f07, (q15_t)0x5f03, (q15_t)0x5eff, (q15_t)0x5efb,
  (q15_t)0x5ef7, (q15_t)0x5ef2, (q15_t)0x5eee, (q15_t)0x5eea, (q15_t)0x5ee6, (q15_t)0x5ee2, (q15_t)0x5edd, (q15_t)0x5ed9,
  (q15_t)0x5ed5, (q15_t)0x5ed1, (q15_t)0x5ecc, (q15_t)0x5ec8, (q15_t)0x5ec4, (q15_t)0x5ec0, (q15_t)0x5ebc, (q15_t)0x5eb7,
  (q15_t)0x5eb3, (q15_t)0x5eaf, (q15_t)0x5eab, (q15_t)0x5ea6, (q15_t)0x5ea2, (q15_t)0x5e9e, (q15_t)0x5e9a, (q15_t)0x5e95,
  (q15_t)0x5e91, (q15_t)0x5e8d, (q15_t)0x5e89, (q15_t)0x5e85, (q15_t)0x5e80, (q15_t)0x5e7c, (q15_t)0x5e78, (q15_t)0x5e74,
  (q15_t)0x5e6f, (q15_t)0x5e6b, (q15_t)0x5e67, (q15_t)0x5e63, (q15_t)0x5e5e, (q15_t)0x5e5a, (q15_t)0x5e56, (q15_t)0x5e52,
  (q15_t)0x5e4d, (q15_t)0x5e49, (q15_t)0x5e45, (q15_t)0x5e41, (q15_t)0x5e3c, (q15_t)0x5e38, (q15_t)0x5e34, (q15_t)0x5e30,
  (q15_t)0x5e2b, (q15_t)0x5e27, (q15_t)0x5e23, (q15_t)0x5e1f, (q15_t)0x5e1a, (q15_t)0x5e16, (q15_t)0x5e12, (q15_t)0x5e0e,
  (q15_t)0x5e09, (q15_t)0x5e05, (q15_t)0x5e01, (q15_t)0x5dfd, (q15_t)0x5df8, (q15_t)0x5df4, (q15_t)0x5df0, (q15_t)0x5deb,
  (q15_t)0x5de7, (q15_t)0x5de3, (q15_t)0x5ddf, (q15_t)0x5dda, (q15_t)0x5dd6, (q15_t)0x5dd2, (q15_t)0x5dce, (q15_t)0x5dc9,
  (q15_t)0x5dc5, (q15_t)0x5dc1, (q15_t)0x5dbc, (q15_t)0x5db8, (q15_t)0x5db4, (q15_t)0x5db0, (q15_t)0x5dab, (q15_t)0x5da7,
  (q15_t)0x5da3, (q15_t)0x5d9e, (q15_t)0x5d9a, (q15_t)0x5d96, (q15_t)0x5d92, (q15_t)0x5d8d, (q15_t)0x5d89, (q15_t)0x5d85,
  (q15_t)0x5d80, (q15_t)0x5d7c, (q15_t)0x5d78, (q15_t)0x5d74, (q15_t)0x5d6f, (q15_t)0x5d6b, (q15_t)0x5d67, (q15_t)0x5d62,
  (q15_t)0x5d5e, (q15_t)0x5d5a, (q15_t)0x5d55, (q15_t)0x5d51, (q15_t)0x5d4d, (q15_t)0x5d49, (q15_t)0x5d44, (q15_t)0x5d40,
  (q15_t)0x5d3c, (q15_t)0x5d37, (q15_t)0x5d33, (q15_t)0x5d2f, (q15_t)0x5d2a, (q15_t)0x5d26, (q15_t)0x5d22, (q15_t)0x5d1e,
  (q15_t)0x5d19, (q15_t)0x5d15, (q15_t)0x5d11, (q15_t)0x5d0c, (q15_t)0x5d08, (q15_t)0x5d04, (q15_t)0x5cff, (q15_t)0x5cfb,
  (q15_t)0x5cf7, (q15_t)0x5cf2, (q15_t)0x5cee, (q15_t)0x5cea, (q15_t)0x5ce5, (q15_t)0x5ce1, (q15_t)0x5cdd, (q15_t)0x5cd8,
  (q15_t)0x5cd4, (q15_t)0x5cd0, (q15_t)0x5ccb, (q15_t)0x5cc7, (q15_t)0x5cc3, (q15_t)0x5cbe, (q15_t)0x5cba, (q15_t)0x5cb6,
  (q15_t)0x5cb1, (q15_t)0x5cad, (q15_t)0x5ca9, (q15_t)0x5ca4, (q15_t)0x5ca0, (q15_t)0x5c9c, (q15_t)0x5c97, (q15_t)0x5c93,
  (q15_t)0x5c8f, (q15_t)0x5c8a, (q15_t)0x5c86, (q15_t)0x5c82, (q15_t)0x5c7d, (q15_t)0x5c79, (q15_t)0x5c75, (q15_t)0x5c70,
  (q15_t)0x5c6c, (q15_t)0x5c68, (q15_t)0x5c63, (q15_t)0x5c5f, (q15_t)0x5c5b, (q15_t)0x5c56, (q15_t)0x5c52, (q15_t)0x5c4e,
  (q15_t)0x5c49, (q15_t)0x5c45, (q15_t)0x5c41, (q15_t)0x5c3c, (q15_t)0x5c38, (q15_t)0x5c33, (q15_t)0x5c2f, (q15_t)0x5c2b,
  (q15_t)0x5c26, (q15_t)0x5c22, (q15_t)0x5c1e, (q15_t)0x5c19, (q15_t)0x5c15, (q15_t)0x5c11, (q15_t)0x5c0c, (q15_t)0x5c08,
  (q15_t)0x5c03, (q15_t)0x5bff, (q15_t)0x5bfb, (q15_t)0x5bf6, (q15_t)0x5bf2, (q15_t)0x5bee, (q15_t)0x5be9, (q15_t)0x5be5,
  (q15_t)0x5be0, (q15_t)0x5bdc, (q15_t)0x5bd8, (q15_t)0x5bd3, (q15_t)0x5bcf, (q15_t)0x5bcb, (q15_t)0x5bc6, (q15_t)0x5bc2,
  (q15_t)0x5bbd, (q15_t)0x5bb9, (q15_t)0x5bb5, (q15_t)0x5bb0, (q15_t)0x5bac, (q15_t)0x5ba8, (q15_t)0x5ba3, (q15_t)0x5b9f,
  (q15_t)0x5b9a, (q15_t)0x5b96, (q15_t)0x5b92, (q15_t)0x5b8d, (q15_t)0x5b89, (q15_t)0x5b84, (q15_t)0x5b80, (q15_t)0x5b7c,
  (q15_t)0x5b77, (q15_t)0x5b73, (q15_t)0x5b6e, (q15_t)0x5b6a, (q15_t)0x5b66, (q15_t)0x5b61, (q15_t)0x5b5d, (q15_t)0x5b58,
  (q15_t)0x5b54, (q15_t)0x5b50, (q15_t)0x5b4b, (q15_t)0x5b47, (q15_t)0x5b42, (q15_t)0x5b3e, (q15_t)0x5b3a, (q15_t)0x5b35,
  (q15_t)0x5b31, (q15_t)0x5b2c, (q15_t)0x5b28, (q15_t)0x5b24, (q15_t)0x5b1f, (q15_t)0x5b1b, (q15_t)0x5b16, (q15_t)0x5b12,
  (q15_t)0x5b0e, (q15_t)0x5b09, (q15_t)0x5b05, (q15_t)0x5b00, (q15_t)0x5afc, (q15_t)0x5af7, (q15_t)0x5af3, (q15_t)0x5aef,
  (q15_t)0x5aea, (q15_t)0x5ae6, (q15_t)0x5ae1, (q15_t)0x5add, (q15_t)0x5ad8, (q15_t)0x5ad4, (q15_t)0x5ad0, (q15_t)0x5acb,
  (q15_t)0x5ac7, (q15_t)0x5ac2, (q15_t)0x5abe, (q15_t)0x5ab9, (q15_t)0x5ab5, (q15_t)0x5ab1, (q15_t)0x5aac, (q15_t)0x5aa8,
  (q15_t)0x5aa3, (q15_t)0x5a9f, (q15_t)0x5a9a, (q15_t)0x5a96, (q15_t)0x5a92, (q15_t)0x5a8d, (q15_t)0x5a89, (q15_t)0x5a84,
  (q15_t)0x5a80, (q15_t)0x5a7b, (q15_t)0x5a77, (q15_t)0x5a72, (q15_t)0x5a6e, (q15_t)0x5a6a, (q15_t)0x5a65, (q15_t)0x5a61,
  (q15_t)0x5a5c, (q15_t)0x5a58, (q15_t)0x5a53, (q15_t)0x5a4f, (q15_t)0x5a4a, (q15_t)0x5a46, (q15_t)0x5a41, (q15_t)0x5a3d,
  (q15_t)0x5a39, (q15_t)0x5a34, (q15_t)0x5a30, (q15_t)0x5a2b, (q15_t)0x5a27, (q15_t)0x5a22, (q15_t)0x5a1e, (q15_t)0x5a19,
  (q15_t)0x5a15, (q15_t)0x5a10, (q15_t)0x5a0c, (q15_t)0x5a07, (q15_t)0x5a03, (q15_t)0x59ff, (q15_t)0x59fa, (q15_t)0x59f6,
  (q15_t)0x59f1, (q15_t)0x59ed, (q15_t)0x59e8, (q15_t)0x59e4, (q15_t)0x59df, (q15_t)0x59db, (q15_t)0x59d6, (q15_t)0x59d2,
  (q15_t)0x59cd, (q15_t)0x59c9, (q15_t)0x59c4, (q15_t)0x59c0, (q15_t)0x59bb, (q15_t)0x59b7, (q15_t)0x59b2, (q15_t)0x59ae,
  (q15_t)0x59a9, (q15_t)0x59a5, (q15_t)0x59a1, (q15_t)0x599c, (q15_t)0x5998, (q15_t)0x5993, (q15_t)0x598f, (q15_t)0x598a,
  (q15_t)0x5986, (q15_t)0x5981, (q15_t)0x597d, (q15_t)0x5978, (q15_t)0x5974, (q15_t)0x596f, (q15_t)0x596b, (q15_t)0x5966,
  (q15_t)0x5962, (q15_t)0x595d, (q15_t)0x5959, (q15_t)0x5954, (q15_t)0x5950, (q15_t)0x594b, (q15_t)0x5947, (q15_t)0x5942,
  (q15_t)0x593e, (q15_t)0x5939, (q15_t)0x5935, (q15_t)0x5930, (q15_t)0x592c, (q15_t)0x5927, (q15_t)0x5923, (q15_t)0x591e,
  (q15_t)0x591a, (q15_t)0x5915, (q15_t)0x5911, (q15_t)0x590c, (q15_t)0x5908, (q15_t)0x5903, (q15_t)0x58fe, (q15_t)0x58fa,
  (q15_t)0x58f5, (q15_t)0x58f1, (q15_t)0x58ec, (q15_t)0x58e8, (q15_t)0x58e3, (q15_t)0x58df, (q15_t)0x58da, (q15_t)0x58d6,
  (q15_t)0x58d1, (q15_t)0x58cd, (q15_t)0x58c8, (q15_t)0x58c4, (q15_t)0x58bf, (q15_t)0x58bb, (q15_t)0x58b6, (q15_t)0x58b2,
  (q15_t)0x58ad, (q15_t)0x58a9, (q15_t)0x58a4, (q15_t)0x589f, (q15_t)0x589b, (q15_t)0x5896, (q15_t)0x5892, (q15_t)0x588d,
  (q15_t)0x5889, (q15_t)0x5884, (q15_t)0x5880, (q15_t)0x587b, (q15_t)0x5877, (q15_t)0x5872, (q15_t)0x586e, (q15_t)0x5869,
  (q15_t)0x5864, (q15_t)0x5860, (q15_t)0x585b, (q15_t)0x5857, (q15_t)0x5852, (q15_t)0x584e, (q15_t)0x5849, (q15_t)0x5845,
  (q15_t)0x5840, (q15_t)0x583c, (q15_t)0x5837, (q15_t)0x5832, (q15_t)0x582e, (q15_t)0x5829, (q15_t)0x5825, (q15_t)0x5820,
  (q15_t)0x581c, (q15_t)0x5817, (q15_t)0x5813, (q15_t)0x580e, (q15_t)0x5809, (q15_t)0x5805, (q15_t)0x5800, (q15_t)0x57fc,
  (q15_t)0x57f7, (q15_t)0x57f3, (q15_t)0x57ee, (q15_t)0x57e9, (q15_t)0x57e5, (q15_t)0x57e0, (q15_t)0x57dc, (q15_t)0x57d7,
  (q15_t)0x57d3, (q15_t)0x57ce, (q15_t)0x57c9, (q15_t)0x57c5, (q15_t)0x57c0, (q15_t)0x57bc, (q15_t)0x57b7, (q15_t)0x57b3,
  (q15_t)0x57ae, (q15_t)0x57a9, (q15_t)0x57a5, (q15_t)0x57a0, (q15_t)0x579c, (q15_t)0x5797, (q15_t)0x5793, (q15_t)0x578e,
  (q15_t)0x5789, (q15_t)0x5785, (q15_t)0x5780, (q15_t)0x577c, (q15_t)0x5777, (q15_t)0x5772, (q15_t)0x576e, (q15_t)0x5769,
  (q15_t)0x5765, (q15_t)0x5760, (q15_t)0x575c, (q15_t)0x5757, (q15_t)0x5752, (q15_t)0x574e, (q15_t)0x5749, (q15_t)0x5745,
  (q15_t)0x5740, (q15_t)0x573b, (q15_t)0x5737, (q15_t)0x5732, (q15_t)0x572e, (q15_t)0x5729, (q15_t)0x5724, (q15_t)0x5720,
  (q15_t)0x571b, (q15_t)0x5717, (q15_t)0x5712, (q15_t)0x570d, (q15_t)0x5709, (q15_t)0x5704, (q15_t)0x56ff, (q15_t)0x56fb,
  (q15_t)0x56f6, (q15_t)0x56f2, (q15_t)0x56ed, (q15_t)0x56e8, (q15_t)0x56e4, (q15_t)0x56df, (q15_t)0x56db, (q15_t)0x56d6,
  (q15_t)0x56d1, (q15_t)0x56cd, (q15_t)0x56c8, (q15_t)0x56c4, (q15_t)0x56bf, (q15_t)0x56ba, (q15_t)0x56b6, (q15_t)0x56b1,
  (q15_t)0x56ac, (q15_t)0x56a8, (q15_t)0x56a3, (q15_t)0x569f, (q15_t)0x569a, (q15_t)0x5695, (q15_t)0x5691, (q15_t)0x568c,
  (q15_t)0x5687, (q15_t)0x5683, (q15_t)0x567e, (q15_t)0x5679, (q15_t)0x5675, (q15_t)0x5670, (q15_t)0x566c, (q15_t)0x5667,
  (q15_t)0x5662, (q15_t)0x565e, (q15_t)0x5659, (q15_t)0x5654, (q15_t)0x5650, (q15_t)0x564b, (q15_t)0x5646, (q15_t)0x5642,
  (q15_t)0x563d, (q15_t)0x5639, (q15_t)0x5634, (q15_t)0x562f, (q15_t)0x562b, (q15_t)0x5626, (q15_t)0x5621, (q15_t)0x561d,
  (q15_t)0x5618, (q15_t)0x5613, (q15_t)0x560f, (q15_t)0x560a, (q15_t)0x5605, (q15_t)0x5601, (q15_t)0x55fc, (q15_t)0x55f7,
  (q15_t)0x55f3, (q15_t)0x55ee, (q15_t)0x55ea, (q15_t)0x55e5, (q15_t)0x55e0, (q15_t)0x55dc, (q15_t)0x55d7, (q15_t)0x55d2,
  (q15_t)0x55ce, (q15_t)0x55c9, (q15_t)0x55c4, (q15_t)0x55c0, (q15_t)0x55bb, (q15_t)0x55b6, (q15_t)0x55b2, (q15_t)0x55ad,
  (q15_t)0x55a8, (q15_t)0x55a4, (q15_t)0x559f, (q15_t)0x559a, (q15_t)0x5596, (q15_t)0x5591, (q15_t)0x558c, (q15_t)0x5588,
  (q15_t)0x5583, (q15_t)0x557e, (q15_t)0x5579, (q15_t)0x5575, (q15_t)0x5570, (q15_t)0x556b, (q15_t)0x5567, (q15_t)0x5562,
  (q15_t)0x555d, (q15_t)0x5559, (q15_t)0x5554, (q15_t)0x554f, (q15_t)0x554b, (q15_t)0x5546, (q15_t)0x5541, (q15_t)0x553d,
  (q15_t)0x5538, (q15_t)0x5533, (q15_t)0x552f, (q15_t)0x552a, (q15_t)0x5525, (q15_t)0x5520, (q15_t)0x551c, (q15_t)0x5517,
  (q15_t)0x5512, (q15_t)0x550e, (q15_t)0x5509, (q15_t)0x5504, (q15_t)0x5500, (q15_t)0x54fb, (q15_t)0x54f6, (q15_t)0x54f2,
  (q15_t)0x54ed, (q15_t)0x54e8, (q15_t)0x54e3, (q15_t)0x54df, (q15_t)0x54da, (q15_t)0x54d5, (q15_t)0x54d1, (q15_t)0x54cc,
  (q15_t)0x54c7, (q15_t)0x54c2, (q15_t)0x54be, (q15_t)0x54b9, (q15_t)0x54b4, (q15_t)0x54b0, (q15_t)0x54ab, (q15_t)0x54a6,
  (q15_t)0x54a2, (q15_t)0x549d, (q15_t)0x5498, (q15_t)0x5493, (q15_t)0x548f, (q15_t)0x548a, (q15_t)0x5485, (q15_t)0x5480,
  (q15_t)0x547c, (q15_t)0x5477, (q15_t)0x5472, (q15_t)0x546e, (q15_t)0x5469, (q15_t)0x5464, (q15_t)0x545f, (q15_t)0x545b,
  (q15_t)0x5456, (q15_t)0x5451, (q15_t)0x544d, (q15_t)0x5448, (q15_t)0x5443, (q15_t)0x543e, (q15_t)0x543a, (q15_t)0x5435,
  (q15_t)0x5430, (q15_t)0x542b, (q15_t)0x5427, (q15_t)0x5422, (q15_t)0x541d, (q15_t)0x5418, (q15_t)0x5414, (q15_t)0x540f,
  (q15_t)0x540a, (q15_t)0x5406, (q15_t)0x5401, (q15_t)0x53fc, (q15_t)0x53f7, (q15_t)0x53f3, (q15_t)0x53ee, (q15_t)0x53e9,
  (q15_t)0x53e4, (q15_t)0x53e0, (q15_t)0x53db, (q15_t)0x53d6, (q15_t)0x53d1, (q15_t)0x53cd, (q15_t)0x53c8, (q15_t)0x53c3,
  (q15_t)0x53be, (q15_t)0x53ba, (q15_t)0x53b5, (q15_t)0x53b0, (q15_t)0x53ab, (q15_t)0x53a7, (q15_t)0x53a2, (q15_t)0x539d,
  (q15_t)0x5398, (q15_t)0x5394, (q15_t)0x538f, (q15_t)0x538a, (q15_t)0x5385, (q15_t)0x5380, (q15_t)0x537c, (q15_t)0x5377,
  (q15_t)0x5372, (q15_t)0x536d, (q15_t)0x5369, (q15_t)0x5364, (q15_t)0x535f, (q15_t)0x535a, (q15_t)0x5356, (q15_t)0x5351,
  (q15_t)0x534c, (q15_t)0x5347, (q15_t)0x5343, (q15_t)0x533e, (q15_t)0x5339, (q15_t)0x5334, (q15_t)0x532f, (q15_t)0x532b,
  (q15_t)0x5326, (q15_t)0x5321, (q15_t)0x531c, (q15_t)0x5318, (q15_t)0x5313, (q15_t)0x530e, (q15_t)0x5309, (q15_t)0x5304,
  (q15_t)0x5300, (q15_t)0x52fb, (q15_t)0x52f6, (q15_t)0x52f1, (q15_t)0x52ec, (q15_t)0x52e8, (q15_t)0x52e3, (q15_t)0x52de,
  (q15_t)0x52d9, (q15_t)0x52d5, (q15_t)0x52d0, (q15_t)0x52cb, (q15_t)0x52c6, (q15_t)0x52c1, (q15_t)0x52bd, (q15_t)0x52b8,
  (q15_t)0x52b3, (q15_t)0x52ae, (q15_t)0x52a9, (q15_t)0x52a5, (q15_t)0x52a0, (q15_t)0x529b, (q15_t)0x5296, (q15_t)0x5291,
  (q15_t)0x528d, (q15_t)0x5288, (q15_t)0x5283, (q15_t)0x527e, (q15_t)0x5279, (q15_t)0x5275, (q15_t)0x5270, (q15_t)0x526b,
  (q15_t)0x5266, (q15_t)0x5261, (q15_t)0x525d, (q15_t)0x5258, (q15_t)0x5253, (q15_t)0x524e, (q15_t)0x5249, (q15_t)0x5244,
  (q15_t)0x5240, (q15_t)0x523b, (q15_t)0x5236, (q15_t)0x5231, (q15_t)0x522c, (q15_t)0x5228, (q15_t)0x5223, (q15_t)0x521e,
  (q15_t)0x5219, (q15_t)0x5214, (q15_t)0x520f, (q15_t)0x520b, (q15_t)0x5206, (q15_t)0x5201, (q15_t)0x51fc, (q15_t)0x51f7,
  (q15_t)0x51f3, (q15_t)0x51ee, (q15_t)0x51e9, (q15_t)0x51e4, (q15_t)0x51df, (q15_t)0x51da, (q15_t)0x51d6, (q15_t)0x51d1,
  (q15_t)0x51cc, (q15_t)0x51c7, (q15_t)0x51c2, (q15_t)0x51bd, (q15_t)0x51b9, (q15_t)0x51b4, (q15_t)0x51af, (q15_t)0x51aa,
  (q15_t)0x51a5, (q15_t)0x51a0, (q15_t)0x519c, (q15_t)0x5197, (q15_t)0x5192, (q15_t)0x518d, (q15_t)0x5188, (q15_t)0x5183,
  (q15_t)0x517e, (q15_t)0x517a, (q15_t)0x5175, (q15_t)0x5170, (q15_t)0x516b, (q15_t)0x5166, (q15_t)0x5161, (q15_t)0x515d,
  (q15_t)0x5158, (q15_t)0x5153, (q15_t)0x514e, (q15_t)0x5149, (q15_t)0x5144, (q15_t)0x513f, (q15_t)0x513b, (q15_t)0x5136,
  (q15_t)0x5131, (q15_t)0x512c, (q15_t)0x5127, (q15_t)0x5122, (q15_t)0x511d, (q15_t)0x5119, (q15_t)0x5114, (q15_t)0x510f,
  (q15_t)0x510a, (q15_t)0x5105, (q15_t)0x5100, (q15_t)0x50fb, (q15_t)0x50f7, (q15_t)0x50f2, (q15_t)0x50ed, (q15_t)0x50e8,
  (q15_t)0x50e3, (q15_t)0x50de, (q15_t)0x50d9, (q15_t)0x50d4, (q15_t)0x50d0, (q15_t)0x50cb, (q15_t)0x50c6, (q15_t)0x50c1,
  (q15_t)0x50bc, (q15_t)0x50b7, (q15_t)0x50b2, (q15_t)0x50ad, (q15_t)0x50a9, (q15_t)0x50a4, (q15_t)0x509f, (q15_t)0x509a,
  (q15_t)0x5095, (q15_t)0x5090, (q15_t)0x508b, (q15_t)0x5086, (q15_t)0x5082, (q15_t)0x507d, (q15_t)0x5078, (q15_t)0x5073,
  (q15_t)0x506e, (q15_t)0x5069, (q15_t)0x5064, (q15_t)0x505f, (q15_t)0x505a, (q15_t)0x5056, (q15_t)0x5051, (q15_t)0x504c,
  (q15_t)0x5047, (q15_t)0x5042, (q15_t)0x503d, (q15_t)0x5038, (q15_t)0x5033, (q15_t)0x502e, (q15_t)0x5029, (q15_t)0x5025,
  (q15_t)0x5020, (q15_t)0x501b, (q15_t)0x5016, (q15_t)0x5011, (q15_t)0x500c, (q15_t)0x5007, (q15_t)0x5002, (q15_t)0x4ffd,
  (q15_t)0x4ff8, (q15_t)0x4ff4, (q15_t)0x4fef, (q15_t)0x4fea, (q15_t)0x4fe5, (q15_t)0x4fe0, (q15_t)0x4fdb, (q15_t)0x4fd6,
  (q15_t)0x4fd1, (q15_t)0x4fcc, (q15_t)0x4fc7, (q15_t)0x4fc2, (q15_t)0x4fbe, (q15_t)0x4fb9, (q15_t)0x4fb4, (q15_t)0x4faf,
  (q15_t)0x4faa, (q15_t)0x4fa5, (q15_t)0x4fa0, (q15_t)0x4f9b, (q15_t)0x4f96, (q15_t)0x4f91, (q15_t)0x4f8c, (q15_t)0x4f87,
  (q15_t)0x4f82, (q15_t)0x4f7e, (q15_t)0x4f79, (q15_t)0x4f74, (q15_t)0x4f6f, (q15_t)0x4f6a, (q15_t)0x4f65, (q15_t)0x4f60,
  (q15_t)0x4f5b, (q15_t)0x4f56, (q15_t)0x4f51, (q15_t)0x4f4c, (q15_t)0x4f47, (q15_t)0x4f42, (q15_t)0x4f3d, (q15_t)0x4f39,
  (q15_t)0x4f34, (q15_t)0x4f2f, (q15_t)0x4f2a, (q15_t)0x4f25, (q15_t)0x4f20, (q15_t)0x4f1b, (q15_t)0x4f16, (q15_t)0x4f11,
  (q15_t)0x4f0c, (q15_t)0x4f07, (q15_t)0x4f02, (q15_t)0x4efd, (q15_t)0x4ef8, (q15_t)0x4ef3, (q15_t)0x4eee, (q15_t)0x4ee9,
  (q15_t)0x4ee5, (q15_t)0x4ee0, (q15_t)0x4edb, (q15_t)0x4ed6, (q15_t)0x4ed1, (q15_t)0x4ecc, (q15_t)0x4ec7, (q15_t)0x4ec2,
  (q15_t)0x4ebd, (q15_t)0x4eb8, (q15_t)0x4eb3, (q15_t)0x4eae, (q15_t)0x4ea9, (q15_t)0x4ea4, (q15_t)0x4e9f, (q15_t)0x4e9a,
  (q15_t)0x4e95, (q15_t)0x4e90, (q15_t)0x4e8b, (q15_t)0x4e86, (q15_t)0x4e81, (q15_t)0x4e7c, (q15_t)0x4e78, (q15_t)0x4e73,
  (q15_t)0x4e6e, (q15_t)0x4e69, (q15_t)0x4e64, (q15_t)0x4e5f, (q15_t)0x4e5a, (q15_t)0x4e55, (q15_t)0x4e50, (q15_t)0x4e4b,
  (q15_t)0x4e46, (q15_t)0x4e41, (q15_t)0x4e3c, (q15_t)0x4e37, (q15_t)0x4e32, (q15_t)0x4e2d, (q15_t)0x4e28, (q15_t)0x4e23,
  (q15_t)0x4e1e, (q15_t)0x4e19, (q15_t)0x4e14, (q15_t)0x4e0f, (q15_t)0x4e0a, (q15_t)0x4e05, (q15_t)0x4e00, (q15_t)0x4dfb,
  (q15_t)0x4df6, (q15_t)0x4df1, (q15_t)0x4dec, (q15_t)0x4de7, (q15_t)0x4de2, (q15_t)0x4ddd, (q15_t)0x4dd8, (q15_t)0x4dd3,
  (q15_t)0x4dce, (q15_t)0x4dc9, (q15_t)0x4dc4, (q15_t)0x4dbf, (q15_t)0x4dba, (q15_t)0x4db5, (q15_t)0x4db0, (q15_t)0x4dab,
  (q15_t)0x4da6, (q15_t)0x4da1, (q15_t)0x4d9c, (q15_t)0x4d97, (q15_t)0x4d92, (q15_t)0x4d8d, (q15_t)0x4d88, (q15_t)0x4d83,
  (q15_t)0x4d7e, (q15_t)0x4d79, (q15_t)0x4d74, (q15_t)0x4d6f, (q15_t)0x4d6a, (q15_t)0x4d65, (q15_t)0x4d60, (q15_t)0x4d5b,
  (q15_t)0x4d56, (q15_t)0x4d51, (q15_t)0x4d4c, (q15_t)0x4d47, (q15_t)0x4d42, (q15_t)0x4d3d, (q15_t)0x4d38, (q15_t)0x4d33,
  (q15_t)0x4d2e, (q15_t)0x4d29, (q15_t)0x4d24, (q15_t)0x4d1f, (q15_t)0x4d1a, (q15_t)0x4d15, (q15_t)0x4d10, (q15_t)0x4d0b,
  (q15_t)0x4d06, (q15_t)0x4d01, (q15_t)0x4cfc, (q15_t)0x4cf7, (q15_t)0x4cf2, (q15_t)0x4ced, (q15_t)0x4ce8, (q15_t)0x4ce3,
  (q15_t)0x4cde, (q15_t)0x4cd9, (q15_t)0x4cd4, (q15_t)0x4ccf, (q15_t)0x4cca, (q15_t)0x4cc5, (q15_t)0x4cc0, (q15_t)0x4cbb,
  (q15_t)0x4cb6, (q15_t)0x4cb1, (q15_t)0x4cac, (q15_t)0x4ca7, (q15_t)0x4ca2, (q15_t)0x4c9d, (q15_t)0x4c98, (q15_t)0x4c93,
  (q15_t)0x4c8e, (q15_t)0x4c88, (q15_t)0x4c83, (q15_t)0x4c7e, (q15_t)0x4c79, (q15_t)0x4c74, (q15_t)0x4c6f, (q15_t)0x4c6a,
  (q15_t)0x4c65, (q15_t)0x4c60, (q15_t)0x4c5b, (q15_t)0x4c56, (q15_t)0x4c51, (q15_t)0x4c4c, (q15_t)0x4c47, (q15_t)0x4c42,
  (q15_t)0x4c3d, (q15_t)0x4c38, (q15_t)0x4c33, (q15_t)0x4c2e, (q15_t)0x4c29, (q15_t)0x4c24, (q15_t)0x4c1f, (q15_t)0x4c1a,
  (q15_t)0x4c14, (q15_t)0x4c0f, (q15_t)0x4c0a, (q15_t)0x4c05, (q15_t)0x4c00, (q15_t)0x4bfb, (q15_t)0x4bf6, (q15_t)0x4bf1,
  (q15_t)0x4bec, (q15_t)0x4be7, (q15_t)0x4be2, (q15_t)0x4bdd, (q15_t)0x4bd8, (q15_t)0x4bd3, (q15_t)0x4bce, (q15_t)0x4bc9,
  (q15_t)0x4bc4, (q15_t)0x4bbe, (q15_t)0x4bb9, (q15_t)0x4bb4, (q15_t)0x4baf, (q15_t)0x4baa, (q15_t)0x4ba5, (q15_t)0x4ba0,
  (q15_t)0x4b9b, (q15_t)0x4b96, (q15_t)0x4b91, (q15_t)0x4b8c, (q15_t)0x4b87, (q15_t)0x4b82, (q15_t)0x4b7d, (q15_t)0x4b77,
  (q15_t)0x4b72, (q15_t)0x4b6d, (q15_t)0x4b68, (q15_t)0x4b63, (q15_t)0x4b5e, (q15_t)0x4b59, (q15_t)0x4b54, (q15_t)0x4b4f,
  (q15_t)0x4b4a, (q15_t)0x4b45, (q15_t)0x4b40, (q15_t)0x4b3b, (q15_t)0x4b35, (q15_t)0x4b30, (q15_t)0x4b2b, (q15_t)0x4b26,
  (q15_t)0x4b21, (q15_t)0x4b1c, (q15_t)0x4b17, (q15_t)0x4b12, (q15_t)0x4b0d, (q15_t)0x4b08, (q15_t)0x4b03, (q15_t)0x4afd,
  (q15_t)0x4af8, (q15_t)0x4af3, (q15_t)0x4aee, (q15_t)0x4ae9, (q15_t)0x4ae4, (q15_t)0x4adf, (q15_t)0x4ada, (q15_t)0x4ad5,
  (q15_t)0x4ad0, (q15_t)0x4acb, (q15_t)0x4ac5, (q15_t)0x4ac0, (q15_t)0x4abb, (q15_t)0x4ab6, (q15_t)0x4ab1, (q15_t)0x4aac,
  (q15_t)0x4aa7, (q15_t)0x4aa2, (q15_t)0x4a9d, (q15_t)0x4a97, (q15_t)0x4a92, (q15_t)0x4a8d, (q15_t)0x4a88, (q15_t)0x4a83,
  (q15_t)0x4a7e, (q15_t)0x4a79, (q15_t)0x4a74, (q15_t)0x4a6f, (q15_t)0x4a6a, (q15_t)0x4a64, (q15_t)0x4a5f, (q15_t)0x4a5a,
  (q15_t)0x4a55, (q15_t)0x4a50, (q15_t)0x4a4b, (q15_t)0x4a46, (q15_t)0x4a41, (q15_t)0x4a3b, (q15_t)0x4a36, (q15_t)0x4a31,
  (q15_t)0x4a2c, (q15_t)0x4a27, (q15_t)0x4a22, (q15_t)0x4a1d, (q15_t)0x4a18, (q15_t)0x4a12, (q15_t)0x4a0d, (q15_t)0x4a08,
  (q15_t)0x4a03, (q15_t)0x49fe, (q15_t)0x49f9, (q15_t)0x49f4, (q15_t)0x49ef, (q15_t)0x49e9, (q15_t)0x49e4, (q15_t)0x49df,
  (q15_t)0x49da, (q15_t)0x49d5, (q15_t)0x49d0, (q15_t)0x49cb, (q15_t)0x49c6, (q15_t)0x49c0, (q15_t)0x49bb, (q15_t)0x49b6,
  (q15_t)0x49b1, (q15_t)0x49ac, (q15_t)0x49a7, (q15_t)0x49a2, (q15_t)0x499c, (q15_t)0x4997, (q15_t)0x4992, (q15_t)0x498d,
  (q15_t)0x4988, (q15_t)0x4983, (q15_t)0x497e, (q15_t)0x4978, (q15_t)0x4973, (q15_t)0x496e, (q15_t)0x4969, (q15_t)0x4964,
  (q15_t)0x495f, (q15_t)0x495a, (q15_t)0x4954, (q15_t)0x494f, (q15_t)0x494a, (q15_t)0x4945, (q15_t)0x4940, (q15_t)0x493b,
  (q15_t)0x4936, (q15_t)0x4930, (q15_t)0x492b, (q15_t)0x4926, (q15_t)0x4921, (q15_t)0x491c, (q15_t)0x4917, (q15_t)0x4911,
  (q15_t)0x490c, (q15_t)0x4907, (q15_t)0x4902, (q15_t)0x48fd, (q15_t)0x48f8, (q15_t)0x48f2, (q15_t)0x48ed, (q15_t)0x48e8,
  (q15_t)0x48e3, (q15_t)0x48de, (q15_t)0x48d9, (q15_t)0x48d3, (q15_t)0x48ce, (q15_t)0x48c9, (q15_t)0x48c4, (q15_t)0x48bf,
  (q15_t)0x48ba, (q15_t)0x48b4, (q15_t)0x48af, (q15_t)0x48aa, (q15_t)0x48a5, (q15_t)0x48a0, (q15_t)0x489b, (q15_t)0x4895,
  (q15_t)0x4890, (q15_t)0x488b, (q15_t)0x4886, (q15_t)0x4881, (q15_t)0x487c, (q15_t)0x4876, (q15_t)0x4871, (q15_t)0x486c,
  (q15_t)0x4867, (q15_t)0x4862, (q15_t)0x485c, (q15_t)0x4857, (q15_t)0x4852, (q15_t)0x484d, (q15_t)0x4848, (q15_t)0x4843,
  (q15_t)0x483d, (q15_t)0x4838, (q15_t)0x4833, (q15_t)0x482e, (q15_t)0x4829, (q15_t)0x4823, (q15_t)0x481e, (q15_t)0x4819,
  (q15_t)0x4814, (q15_t)0x480f, (q15_t)0x4809, (q15_t)0x4804, (q15_t)0x47ff, (q15_t)0x47fa, (q15_t)0x47f5, (q15_t)0x47ef,
  (q15_t)0x47ea, (q15_t)0x47e5, (q15_t)0x47e0, (q15_t)0x47db, (q15_t)0x47d5, (q15_t)0x47d0, (q15_t)0x47cb, (q15_t)0x47c6,
  (q15_t)0x47c1, (q15_t)0x47bb, (q15_t)0x47b6, (q15_t)0x47b1, (q15_t)0x47ac, (q15_t)0x47a7, (q15_t)0x47a1, (q15_t)0x479c,
  (q15_t)0x4797, (q15_t)0x4792, (q15_t)0x478d, (q15_t)0x4787, (q15_t)0x4782, (q15_t)0x477d, (q15_t)0x4778, (q15_t)0x4773,
  (q15_t)0x476d, (q15_t)0x4768, (q15_t)0x4763, (q15_t)0x475e, (q15_t)0x4758, (q15_t)0x4753, (q15_t)0x474e, (q15_t)0x4749,
  (q15_t)0x4744, (q15_t)0x473e, (q15_t)0x4739, (q15_t)0x4734, (q15_t)0x472f, (q15_t)0x4729, (q15_t)0x4724, (q15_t)0x471f,
  (q15_t)0x471a, (q15_t)0x4715, (q15_t)0x470f, (q15_t)0x470a, (q15_t)0x4705, (q15_t)0x4700, (q15_t)0x46fa, (q15_t)0x46f5,
  (q15_t)0x46f0, (q15_t)0x46eb, (q15_t)0x46e6, (q15_t)0x46e0, (q15_t)0x46db, (q15_t)0x46d6, (q15_t)0x46d1, (q15_t)0x46cb,
  (q15_t)0x46c6, (q15_t)0x46c1, (q15_t)0x46bc, (q15_t)0x46b6, (q15_t)0x46b1, (q15_t)0x46ac, (q15_t)0x46a7, (q15_t)0x46a1,
  (q15_t)0x469c, (q15_t)0x4697, (q15_t)0x4692, (q15_t)0x468d, (q15_t)0x4687, (q15_t)0x4682, (q15_t)0x467d, (q15_t)0x4678,
  (q15_t)0x4672, (q15_t)0x466d, (q15_t)0x4668, (q15_t)0x4663, (q15_t)0x465d, (q15_t)0x4658, (q15_t)0x4653, (q15_t)0x464e,
  (q15_t)0x4648, (q15_t)0x4643, (q15_t)0x463e, (q15_t)0x4639, (q15_t)0x4633, (q15_t)0x462e, (q15_t)0x4629, (q15_t)0x4624,
  (q15_t)0x461e, (q15_t)0x4619, (q15_t)0x4614, (q15_t)0x460e, (q15_t)0x4609, (q15_t)0x4604, (q15_t)0x45ff, (q15_t)0x45f9,
  (q15_t)0x45f4, (q15_t)0x45ef, (q15_t)0x45ea, (q15_t)0x45e4, (q15_t)0x45df, (q15_t)0x45da, (q15_t)0x45d5, (q15_t)0x45cf,
  (q15_t)0x45ca, (q15_t)0x45c5, (q15_t)0x45c0, (q15_t)0x45ba, (q15_t)0x45b5, (q15_t)0x45b0, (q15_t)0x45aa, (q15_t)0x45a5,
  (q15_t)0x45a0, (q15_t)0x459b, (q15_t)0x4595, (q15_t)0x4590, (q15_t)0x458b, (q15_t)0x4586, (q15_t)0x4580, (q15_t)0x457b,
  (q15_t)0x4576, (q15_t)0x4570, (q15_t)0x456b, (q15_t)0x4566, (q15_t)0x4561, (q15_t)0x455b, (q15_t)0x4556, (q15_t)0x4551,
  (q15_t)0x454b, (q15_t)0x4546, (q15_t)0x4541, (q15_t)0x453c, (q15_t)0x4536, (q15_t)0x4531, (q15_t)0x452c, (q15_t)0x4526,
  (q15_t)0x4521, (q15_t)0x451c, (q15_t)0x4517, (q15_t)0x4511, (q15_t)0x450c, (q15_t)0x4507, (q15_t)0x4501, (q15_t)0x44fc,
  (q15_t)0x44f7, (q15_t)0x44f2, (q15_t)0x44ec, (q15_t)0x44e7, (q15_t)0x44e2, (q15_t)0x44dc, (q15_t)0x44d7, (q15_t)0x44d2,
  (q15_t)0x44cd, (q15_t)0x44c7, (q15_t)0x44c2, (q15_t)0x44bd, (q15_t)0x44b7, (q15_t)0x44b2, (q15_t)0x44ad, (q15_t)0x44a7,
  (q15_t)0x44a2, (q15_t)0x449d, (q15_t)0x4497, (q15_t)0x4492, (q15_t)0x448d, (q15_t)0x4488, (q15_t)0x4482, (q15_t)0x447d,
  (q15_t)0x4478, (q15_t)0x4472, (q15_t)0x446d, (q15_t)0x4468, (q15_t)0x4462, (q15_t)0x445d, (q15_t)0x4458, (q15_t)0x4452,
  (q15_t)0x444d, (q15_t)0x4448, (q15_t)0x4443, (q15_t)0x443d, (q15_t)0x4438, (q15_t)0x4433, (q15_t)0x442d, (q15_t)0x4428,
  (q15_t)0x4423, (q15_t)0x441d, (q15_t)0x4418, (q15_t)0x4413, (q15_t)0x440d, (q15_t)0x4408, (q15_t)0x4403, (q15_t)0x43fd,
  (q15_t)0x43f8, (q15_t)0x43f3, (q15_t)0x43ed, (q15_t)0x43e8, (q15_t)0x43e3, (q15_t)0x43dd, (q15_t)0x43d8, (q15_t)0x43d3,
  (q15_t)0x43cd, (q15_t)0x43c8, (q15_t)0x43c3, (q15_t)0x43bd, (q15_t)0x43b8, (q15_t)0x43b3, (q15_t)0x43ad, (q15_t)0x43a8,
  (q15_t)0x43a3, (q15_t)0x439d, (q15_t)0x4398, (q15_t)0x4393, (q15_t)0x438d, (q15_t)0x4388, (q15_t)0x4383, (q15_t)0x437d,
  (q15_t)0x4378, (q15_t)0x4373, (q15_t)0x436d, (q15_t)0x4368, (q15_t)0x4363, (q15_t)0x435d, (q15_t)0x4358, (q15_t)0x4353,
  (q15_t)0x434d, (q15_t)0x4348, (q15_t)0x4343, (q15_t)0x433d, (q15_t)0x4338, (q15_t)0x4333, (q15_t)0x432d, (q15_t)0x4328,
  (q15_t)0x4323, (q15_t)0x431d, (q15_t)0x4318, (q15_t)0x4313, (q15_t)0x430d, (q15_t)0x4308, (q15_t)0x4302, (q15_t)0x42fd,
  (q15_t)0x42f8, (q15_t)0x42f2, (q15_t)0x42ed, (q15_t)0x42e8, (q15_t)0x42e2, (q15_t)0x42dd, (q15_t)0x42d8, (q15_t)0x42d2,
  (q15_t)0x42cd, (q15_t)0x42c8, (q15_t)0x42c2, (q15_t)0x42bd, (q15_t)0x42b7, (q15_t)0x42b2, (q15_t)0x42ad, (q15_t)0x42a7,
  (q15_t)0x42a2, (q15_t)0x429d, (q15_t)0x4297, (q15_t)0x4292, (q15_t)0x428d, (q15_t)0x4287, (q15_t)0x4282, (q15_t)0x427c,
  (q15_t)0x4277, (q15_t)0x4272, (q15_t)0x426c, (q15_t)0x4267, (q15_t)0x4262, (q15_t)0x425c, (q15_t)0x4257, (q15_t)0x4251,
  (q15_t)0x424c, (q15_t)0x4247, (q15_t)0x4241, (q15_t)0x423c, (q15_t)0x4237, (q15_t)0x4231, (q15_t)0x422c, (q15_t)0x4226,
  (q15_t)0x4221, (q15_t)0x421c, (q15_t)0x4216, (q15_t)0x4211, (q15_t)0x420c, (q15_t)0x4206, (q15_t)0x4201, (q15_t)0x41fb,
  (q15_t)0x41f6, (q15_t)0x41f1, (q15_t)0x41eb, (q15_t)0x41e6, (q15_t)0x41e0, (q15_t)0x41db, (q15_t)0x41d6, (q15_t)0x41d0,
  (q15_t)0x41cb, (q15_t)0x41c6, (q15_t)0x41c0, (q15_t)0x41bb, (q15_t)0x41b5, (q15_t)0x41b0, (q15_t)0x41ab, (q15_t)0x41a5,
  (q15_t)0x41a0, (q15_t)0x419a, (q15_t)0x4195, (q15_t)0x4190, (q15_t)0x418a, (q15_t)0x4185, (q15_t)0x417f, (q15_t)0x417a,
  (q15_t)0x4175, (q15_t)0x416f, (q15_t)0x416a, (q15_t)0x4164, (q15_t)0x415f, (q15_t)0x415a, (q15_t)0x4154, (q15_t)0x414f,
  (q15_t)0x4149, (q15_t)0x4144, (q15_t)0x413f, (q15_t)0x4139, (q15_t)0x4134, (q15_t)0x412e, (q15_t)0x4129, (q15_t)0x4124,
  (q15_t)0x411e, (q15_t)0x4119, (q15_t)0x4113, (q15_t)0x410e, (q15_t)0x4108, (q15_t)0x4103, (q15_t)0x40fe, (q15_t)0x40f8,
  (q15_t)0x40f3, (q15_t)0x40ed, (q15_t)0x40e8, (q15_t)0x40e3, (q15_t)0x40dd, (q15_t)0x40d8, (q15_t)0x40d2, (q15_t)0x40cd,
  (q15_t)0x40c8, (q15_t)0x40c2, (q15_t)0x40bd, (q15_t)0x40b7, (q15_t)0x40b2, (q15_t)0x40ac, (q15_t)0x40a7, (q15_t)0x40a2,
  (q15_t)0x409c, (q15_t)0x4097, (q15_t)0x4091, (q15_t)0x408c, (q15_t)0x4086, (q15_t)0x4081, (q15_t)0x407c, (q15_t)0x4076,
  (q15_t)0x4071, (q15_t)0x406b, (q15_t)0x4066, (q15_t)0x4060, (q15_t)0x405b, (q15_t)0x4056, (q15_t)0x4050, (q15_t)0x404b,
  (q15_t)0x4045, (q15_t)0x4040, (q15_t)0x403a, (q15_t)0x4035, (q15_t)0x4030, (q15_t)0x402a, (q15_t)0x4025, (q15_t)0x401f,
  (q15_t)0x401a, (q15_t)0x4014, (q15_t)0x400f, (q15_t)0x4009, (q15_t)0x4004, (q15_t)0x3fff, (q15_t)0x3ff9, (q15_t)0x3ff4,
  (q15_t)0x3fee, (q15_t)0x3fe9, (q15_t)0x3fe3, (q15_t)0x3fde, (q15_t)0x3fd8, (q15_t)0x3fd3, (q15_t)0x3fce, (q15_t)0x3fc8,
  (q15_t)0x3fc3, (q15_t)0x3fbd, (q15_t)0x3fb8, (q15_t)0x3fb2, (q15_t)0x3fad, (q15_t)0x3fa7, (q15_t)0x3fa2, (q15_t)0x3f9d,
  (q15_t)0x3f97, (q15_t)0x3f92, (q15_t)0x3f8c, (q15_t)0x3f87, (q15_t)0x3f81, (q15_t)0x3f7c, (q15_t)0x3f76, (q15_t)0x3f71,
  (q15_t)0x3f6b, (q15_t)0x3f66, (q15_t)0x3f61, (q15_t)0x3f5b, (q15_t)0x3f56, (q15_t)0x3f50, (q15_t)0x3f4b, (q15_t)0x3f45,
  (q15_t)0x3f40, (q15_t)0x3f3a, (q15_t)0x3f35, (q15_t)0x3f2f, (q15_t)0x3f2a, (q15_t)0x3f24, (q15_t)0x3f1f, (q15_t)0x3f1a,
  (q15_t)0x3f14, (q15_t)0x3f0f, (q15_t)0x3f09, (q15_t)0x3f04, (q15_t)0x3efe, (q15_t)0x3ef9, (q15_t)0x3ef3, (q15_t)0x3eee,
  (q15_t)0x3ee8, (q15_t)0x3ee3, (q15_t)0x3edd, (q15_t)0x3ed8, (q15_t)0x3ed2, (q15_t)0x3ecd, (q15_t)0x3ec7, (q15_t)0x3ec2,
  (q15_t)0x3ebd, (q15_t)0x3eb7, (q15_t)0x3eb2, (q15_t)0x3eac, (q15_t)0x3ea7, (q15_t)0x3ea1, (q15_t)0x3e9c, (q15_t)0x3e96,
  (q15_t)0x3e91, (q15_t)0x3e8b, (q15_t)0x3e86, (q15_t)0x3e80, (q15_t)0x3e7b, (q15_t)0x3e75, (q15_t)0x3e70, (q15_t)0x3e6a,
  (q15_t)0x3e65, (q15_t)0x3e5f, (q15_t)0x3e5a, (q15_t)0x3e54, (q15_t)0x3e4f, (q15_t)0x3e49, (q15_t)0x3e44, (q15_t)0x3e3e,
  (q15_t)0x3e39, (q15_t)0x3e33, (q15_t)0x3e2e, (q15_t)0x3e28, (q15_t)0x3e23, (q15_t)0x3e1d, (q15_t)0x3e18, (q15_t)0x3e12,
  (q15_t)0x3e0d, (q15_t)0x3e07, (q15_t)0x3e02, (q15_t)0x3dfc, (q15_t)0x3df7, (q15_t)0x3df1, (q15_t)0x3dec, (q15_t)0x3de6,
  (q15_t)0x3de1, (q15_t)0x3ddb, (q15_t)0x3dd6, (q15_t)0x3dd0, (q15_t)0x3dcb, (q15_t)0x3dc5, (q15_t)0x3dc0, (q15_t)0x3dba,
  (q15_t)0x3db5, (q15_t)0x3daf, (q15_t)0x3daa, (q15_t)0x3da4, (q15_t)0x3d9f, (q15_t)0x3d99, (q15_t)0x3d94, (q15_t)0x3d8e,
  (q15_t)0x3d89, (q15_t)0x3d83, (q15_t)0x3d7e, (q15_t)0x3d78, (q15_t)0x3d73, (q15_t)0x3d6d, (q15_t)0x3d68, (q15_t)0x3d62,
  (q15_t)0x3d5d, (q15_t)0x3d57, (q15_t)0x3d52, (q15_t)0x3d4c, (q15_t)0x3d47, (q15_t)0x3d41, (q15_t)0x3d3c, (q15_t)0x3d36,
  (q15_t)0x3d31, (q15_t)0x3d2b, (q15_t)0x3d26, (q15_t)0x3d20, (q15_t)0x3d1b, (q15_t)0x3d15, (q15_t)0x3d10, (q15_t)0x3d0a,
  (q15_t)0x3d04, (q15_t)0x3cff, (q15_t)0x3cf9, (q15_t)0x3cf4, (q15_t)0x3cee, (q15_t)0x3ce9, (q15_t)0x3ce3, (q15_t)0x3cde,
  (q15_t)0x3cd8, (q15_t)0x3cd3, (q15_t)0x3ccd, (q15_t)0x3cc8, (q15_t)0x3cc2, (q15_t)0x3cbd, (q15_t)0x3cb7, (q15_t)0x3cb2,
  (q15_t)0x3cac, (q15_t)0x3ca7, (q15_t)0x3ca1, (q15_t)0x3c9b, (q15_t)0x3c96, (q15_t)0x3c90, (q15_t)0x3c8b, (q15_t)0x3c85,
  (q15_t)0x3c80, (q15_t)0x3c7a, (q15_t)0x3c75, (q15_t)0x3c6f, (q15_t)0x3c6a, (q15_t)0x3c64, (q15_t)0x3c5f, (q15_t)0x3c59,
  (q15_t)0x3c53, (q15_t)0x3c4e, (q15_t)0x3c48, (q15_t)0x3c43, (q15_t)0x3c3d, (q15_t)0x3c38, (q15_t)0x3c32, (q15_t)0x3c2d,
  (q15_t)0x3c27, (q15_t)0x3c22, (q15_t)0x3c1c, (q15_t)0x3c16, (q15_t)0x3c11, (q15_t)0x3c0b, (q15_t)0x3c06, (q15_t)0x3c00,
  (q15_t)0x3bfb, (q15_t)0x3bf5, (q15_t)0x3bf0, (q15_t)0x3bea, (q15_t)0x3be5, (q15_t)0x3bdf, (q15_t)0x3bd9, (q15_t)0x3bd4,
  (q15_t)0x3bce, (q15_t)0x3bc9, (q15_t)0x3bc3, (q15_t)0x3bbe, (q15_t)0x3bb8, (q15_t)0x3bb3, (q15_t)0x3bad, (q15_t)0x3ba7,
  (q15_t)0x3ba2, (q15_t)0x3b9c, (q15_t)0x3b97, (q15_t)0x3b91, (q15_t)0x3b8c, (q15_t)0x3b86, (q15_t)0x3b80, (q15_t)0x3b7b,
  (q15_t)0x3b75, (q15_t)0x3b70, (q15_t)0x3b6a, (q15_t)0x3b65, (q15_t)0x3b5f, (q15_t)0x3b5a, (q15_t)0x3b54, (q15_t)0x3b4e,
  (q15_t)0x3b49, (q15_t)0x3b43, (q15_t)0x3b3e, (q15_t)0x3b38, (q15_t)0x3b33, (q15_t)0x3b2d, (q15_t)0x3b27, (q15_t)0x3b22,
  (q15_t)0x3b1c, (q15_t)0x3b17, (q15_t)0x3b11, (q15_t)0x3b0c, (q15_t)0x3b06, (q15_t)0x3b00, (q15_t)0x3afb, (q15_t)0x3af5,
  (q15_t)0x3af0, (q15_t)0x3aea, (q15_t)0x3ae4, (q15_t)0x3adf, (q15_t)0x3ad9, (q15_t)0x3ad4, (q15_t)0x3ace, (q15_t)0x3ac9,
  (q15_t)0x3ac3, (q15_t)0x3abd, (q15_t)0x3ab8, (q15_t)0x3ab2, (q15_t)0x3aad, (q15_t)0x3aa7, (q15_t)0x3aa2, (q15_t)0x3a9c,
  (q15_t)0x3a96, (q15_t)0x3a91, (q15_t)0x3a8b, (q15_t)0x3a86, (q15_t)0x3a80, (q15_t)0x3a7a, (q15_t)0x3a75, (q15_t)0x3a6f,
  (q15_t)0x3a6a, (q15_t)0x3a64, (q15_t)0x3a5e, (q15_t)0x3a59, (q15_t)0x3a53, (q15_t)0x3a4e, (q15_t)0x3a48, (q15_t)0x3a42,
  (q15_t)0x3a3d, (q15_t)0x3a37, (q15_t)0x3a32, (q15_t)0x3a2c, (q15_t)0x3a26, (q15_t)0x3a21, (q15_t)0x3a1b, (q15_t)0x3a16,
  (q15_t)0x3a10, (q15_t)0x3a0b, (q15_t)0x3a05, (q15_t)0x39ff, (q15_t)0x39fa, (q15_t)0x39f4, (q15_t)0x39ee, (q15_t)0x39e9,
  (q15_t)0x39e3, (q15_t)0x39de, (q15_t)0x39d8, (q15_t)0x39d2, (q15_t)0x39cd, (q15_t)0x39c7, (q15_t)0x39c2, (q15_t)0x39bc,
  (q15_t)0x39b6, (q15_t)0x39b1, (q15_t)0x39ab, (q15_t)0x39a6, (q15_t)0x39a0, (q15_t)0x399a, (q15_t)0x3995, (q15_t)0x398f,
  (q15_t)0x398a, (q15_t)0x3984, (q15_t)0x397e, (q15_t)0x3979, (q15_t)0x3973, (q15_t)0x396d, (q15_t)0x3968, (q15_t)0x3962,
  (q15_t)0x395d, (q15_t)0x3957, (q15_t)0x3951, (q15_t)0x394c, (q15_t)0x3946, (q15_t)0x3941, (q15_t)0x393b, (q15_t)0x3935,
  (q15_t)0x3930, (q15_t)0x392a, (q15_t)0x3924, (q15_t)0x391f, (q15_t)0x3919, (q15_t)0x3914, (q15_t)0x390e, (q15_t)0x3908,
  (q15_t)0x3903, (q15_t)0x38fd, (q15_t)0x38f7, (q15_t)0x38f2, (q15_t)0x38ec, (q15_t)0x38e7, (q15_t)0x38e1, (q15_t)0x38db,
  (q15_t)0x38d6, (q15_t)0x38d0, (q15_t)0x38ca, (q15_t)0x38c5, (q15_t)0x38bf, (q15_t)0x38ba, (q15_t)0x38b4, (q15_t)0x38ae,
  (q15_t)0x38a9, (q15_t)0x38a3, (q15_t)0x389d, (q15_t)0x3898, (q15_t)0x3892, (q15_t)0x388c, (q15_t)0x3887, (q15_t)0x3881,
  (q15_t)0x387c, (q15_t)0x3876, (q15_t)0x3870, (q15_t)0x386b, (q15_t)0x3865, (q15_t)0x385f, (q15_t)0x385a, (q15_t)0x3854,
  (q15_t)0x384e, (q15_t)0x3849, (q15_t)0x3843, (q15_t)0x383d, (q15_t)0x3838, (q15_t)0x3832, (q15_t)0x382d, (q15_t)0x3827,
  (q15_t)0x3821, (q15_t)0x381c, (q15_t)0x3816, (q15_t)0x3810, (q15_t)0x380b, (q15_t)0x3805, (q15_t)0x37ff, (q15_t)0x37fa,
  (q15_t)0x37f4, (q15_t)0x37ee, (q15_t)0x37e9, (q15_t)0x37e3, (q15_t)0x37dd, (q15_t)0x37d8, (q15_t)0x37d2, (q15_t)0x37cc,
  (q15_t)0x37c7, (q15_t)0x37c1, (q15_t)0x37bc, (q15_t)0x37b6, (q15_t)0x37b0, (q15_t)0x37ab, (q15_t)0x37a5, (q15_t)0x379f,
  (q15_t)0x379a, (q15_t)0x3794, (q15_t)0x378e, (q15_t)0x3789, (q15_t)0x3783, (q15_t)0x377d, (q15_t)0x3778, (q15_t)0x3772,
  (q15_t)0x376c, (q15_t)0x3767, (q15_t)0x3761, (q15_t)0x375b, (q15_t)0x3756, (q15_t)0x3750, (q15_t)0x374a, (q15_t)0x3745,
  (q15_t)0x373f, (q15_t)0x3739, (q15_t)0x3734, (q15_t)0x372e, (q15_t)0x3728, (q15_t)0x3723, (q15_t)0x371d, (q15_t)0x3717,
  (q15_t)0x3712, (q15_t)0x370c, (q15_t)0x3706, (q15_t)0x3701, (q15_t)0x36fb, (q15_t)0x36f5, (q15_t)0x36f0, (q15_t)0x36ea,
  (q15_t)0x36e4, (q15_t)0x36df, (q15_t)0x36d9, (q15_t)0x36d3, (q15_t)0x36ce, (q15_t)0x36c8, (q15_t)0x36c2, (q15_t)0x36bc,
  (q15_t)0x36b7, (q15_t)0x36b1, (q15_t)0x36ab, (q15_t)0x36a6, (q15_t)0x36a0, (q15_t)0x369a, (q15_t)0x3695, (q15_t)0x368f,
  (q15_t)0x3689, (q15_t)0x3684, (q15_t)0x367e, (q15_t)0x3678, (q15_t)0x3673, (q15_t)0x366d, (q15_t)0x3667, (q15_t)0x3662,
  (q15_t)0x365c, (q15_t)0x3656, (q15_t)0x3650, (q15_t)0x364b, (q15_t)0x3645, (q15_t)0x363f, (q15_t)0x363a, (q15_t)0x3634,
  (q15_t)0x362e, (q15_t)0x3629, (q15_t)0x3623, (q15_t)0x361d, (q15_t)0x3618, (q15_t)0x3612, (q15_t)0x360c, (q15_t)0x3606,
  (q15_t)0x3601, (q15_t)0x35fb, (q15_t)0x35f5, (q15_t)0x35f0, (q15_t)0x35ea, (q15_t)0x35e4, (q15_t)0x35df, (q15_t)0x35d9,
  (q15_t)0x35d3, (q15_t)0x35cd, (q15_t)0x35c8, (q15_t)0x35c2, (q15_t)0x35bc, (q15_t)0x35b7, (q15_t)0x35b1, (q15_t)0x35ab,
  (q15_t)0x35a6, (q15_t)0x35a0, (q15_t)0x359a, (q15_t)0x3594, (q15_t)0x358f, (q15_t)0x3589, (q15_t)0x3583, (q15_t)0x357e,
  (q15_t)0x3578, (q15_t)0x3572, (q15_t)0x356c, (q15_t)0x3567, (q15_t)0x3561, (q15_t)0x355b, (q15_t)0x3556, (q15_t)0x3550,
  (q15_t)0x354a, (q15_t)0x3544, (q15_t)0x353f, (q15_t)0x3539, (q15_t)0x3533, (q15_t)0x352e, (q15_t)0x3528, (q15_t)0x3522,
  (q15_t)0x351c, (q15_t)0x3517, (q15_t)0x3511, (q15_t)0x350b, (q15_t)0x3506, (q15_t)0x3500, (q15_t)0x34fa, (q15_t)0x34f4,
  (q15_t)0x34ef, (q15_t)0x34e9, (q15_t)0x34e3, (q15_t)0x34de, (q15_t)0x34d8, (q15_t)0x34d2, (q15_t)0x34cc, (q15_t)0x34c7,
  (q15_t)0x34c1, (q15_t)0x34bb, (q15_t)0x34b6, (q15_t)0x34b0, (q15_t)0x34aa, (q15_t)0x34a4, (q15_t)0x349f, (q15_t)0x3499,
  (q15_t)0x3493, (q15_t)0x348d, (q15_t)0x3488, (q15_t)0x3482, (q15_t)0x347c, (q15_t)0x3476, (q15_t)0x3471, (q15_t)0x346b,
  (q15_t)0x3465, (q15_t)0x3460, (q15_t)0x345a, (q15_t)0x3454, (q15_t)0x344e, (q15_t)0x3449, (q15_t)0x3443, (q15_t)0x343d,
  (q15_t)0x3437, (q15_t)0x3432, (q15_t)0x342c, (q15_t)0x3426, (q15_t)0x3420, (q15_t)0x341b, (q15_t)0x3415, (q15_t)0x340f,
  (q15_t)0x340a, (q15_t)0x3404, (q15_t)0x33fe, (q15_t)0x33f8, (q15_t)0x33f3, (q15_t)0x33ed, (q15_t)0x33e7, (q15_t)0x33e1,
  (q15_t)0x33dc, (q15_t)0x33d6, (q15_t)0x33d0, (q15_t)0x33ca, (q15_t)0x33c5, (q15_t)0x33bf, (q15_t)0x33b9, (q15_t)0x33b3,
  (q15_t)0x33ae, (q15_t)0x33a8, (q15_t)0x33a2, (q15_t)0x339c, (q15_t)0x3397, (q15_t)0x3391, (q15_t)0x338b, (q15_t)0x3385,
  (q15_t)0x3380, (q15_t)0x337a, (q15_t)0x3374, (q15_t)0x336e, (q15_t)0x3369, (q15_t)0x3363, (q15_t)0x335d, (q15_t)0x3357,
  (q15_t)0x3352, (q15_t)0x334c, (q15_t)0x3346, (q15_t)0x3340, (q15_t)0x333b, (q15_t)0x3335, (q15_t)0x332f, (q15_t)0x3329,
  (q15_t)0x3324, (q15_t)0x331e, (q15_t)0x3318, (q15_t)0x3312, (q15_t)0x330c, (q15_t)0x3307, (q15_t)0x3301, (q15_t)0x32fb,
  (q15_t)0x32f5, (q15_t)0x32f0, (q15_t)0x32ea, (q15_t)0x32e4, (q15_t)0x32de, (q15_t)0x32d9, (q15_t)0x32d3, (q15_t)0x32cd,
  (q15_t)0x32c7, (q15_t)0x32c2, (q15_t)0x32bc, (q15_t)0x32b6, (q15_t)0x32b0, (q15_t)0x32aa, (q15_t)0x32a5, (q15_t)0x329f,
  (q15_t)0x3299, (q15_t)0x3293, (q15_t)0x328e, (q15_t)0x3288, (q15_t)0x3282, (q15_t)0x327c, (q15_t)0x3276, (q15_t)0x3271,
  (q15_t)0x326b, (q15_t)0x3265, (q15_t)0x325f, (q15_t)0x325a, (q15_t)0x3254, (q15_t)0x324e, (q15_t)0x3248, (q15_t)0x3243,
  (q15_t)0x323d, (q15_t)0x3237, (q15_t)0x3231, (q15_t)0x322b, (q15_t)0x3226, (q15_t)0x3220, (q15_t)0x321a, (q15_t)0x3214,
  (q15_t)0x320e, (q15_t)0x3209, (q15_t)0x3203, (q15_t)0x31fd, (q15_t)0x31f7, (q15_t)0x31f2, (q15_t)0x31ec, (q15_t)0x31e6,
  (q15_t)0x31e0, (q15_t)0x31da, (q15_t)0x31d5, (q15_t)0x31cf, (q15_t)0x31c9, (q15_t)0x31c3, (q15_t)0x31bd, (q15_t)0x31b8,
  (q15_t)0x31b2, (q15_t)0x31ac, (q15_t)0x31a6, (q15_t)0x31a1, (q15_t)0x319b, (q15_t)0x3195, (q15_t)0x318f, (q15_t)0x3189,
  (q15_t)0x3184, (q15_t)0x317e, (q15_t)0x3178, (q15_t)0x3172, (q15_t)0x316c, (q15_t)0x3167, (q15_t)0x3161, (q15_t)0x315b,
  (q15_t)0x3155, (q15_t)0x314f, (q15_t)0x314a, (q15_t)0x3144, (q15_t)0x313e, (q15_t)0x3138, (q15_t)0x3132, (q15_t)0x312d,
  (q15_t)0x3127, (q15_t)0x3121, (q15_t)0x311b, (q15_t)0x3115, (q15_t)0x3110, (q15_t)0x310a, (q15_t)0x3104, (q15_t)0x30fe,
  (q15_t)0x30f8, (q15_t)0x30f3, (q15_t)0x30ed, (q15_t)0x30e7, (q15_t)0x30e1, (q15_t)0x30db, (q15_t)0x30d6, (q15_t)0x30d0,
  (q15_t)0x30ca, (q15_t)0x30c4, (q15_t)0x30be, (q15_t)0x30b8, (q15_t)0x30b3, (q15_t)0x30ad, (q15_t)0x30a7, (q15_t)0x30a1,
  (q15_t)0x309b, (q15_t)0x3096, (q15_t)0x3090, (q15_t)0x308a, (q15_t)0x3084, (q15_t)0x307e, (q15_t)0x3079, (q15_t)0x3073,
  (q15_t)0x306d, (q15_t)0x3067, (q15_t)0x3061, (q15_t)0x305b, (q15_t)0x3056, (q15_t)0x3050, (q15_t)0x304a, (q15_t)0x3044,
  (q15_t)0x303e, (q15_t)0x3039, (q15_t)0x3033, (q15_t)0x302d, (q15_t)0x3027, (q15_t)0x3021, (q15_t)0x301b, (q15_t)0x3016,
  (q15_t)0x3010, (q15_t)0x300a, (q15_t)0x3004, (q15_t)0x2ffe, (q15_t)0x2ff8, (q15_t)0x2ff3, (q15_t)0x2fed, (q15_t)0x2fe7,
  (q15_t)0x2fe1, (q15_t)0x2fdb, (q15_t)0x2fd6, (q15_t)0x2fd0, (q15_t)0x2fca, (q15_t)0x2fc4, (q15_t)0x2fbe, (q15_t)0x2fb8,
  (q15_t)0x2fb3, (q15_t)0x2fad, (q15_t)0x2fa7, (q15_t)0x2fa1, (q15_t)0x2f9b, (q15_t)0x2f95, (q15_t)0x2f90, (q15_t)0x2f8a,
  (q15_t)0x2f84, (q15_t)0x2f7e, (q15_t)0x2f78, (q15_t)0x2f72, (q15_t)0x2f6d, (q15_t)0x2f67, (q15_t)0x2f61, (q15_t)0x2f5b,
  (q15_t)0x2f55, (q15_t)0x2f4f, (q15_t)0x2f4a, (q15_t)0x2f44, (q15_t)0x2f3e, (q15_t)0x2f38, (q15_t)0x2f32, (q15_t)0x2f2c,
  (q15_t)0x2f27, (q15_t)0x2f21, (q15_t)0x2f1b, (q15_t)0x2f15, (q15_t)0x2f0f, (q15_t)0x2f09, (q15_t)0x2f03, (q15_t)0x2efe,
  (q15_t)0x2ef8, (q15_t)0x2ef2, (q15_t)0x2eec, (q15_t)0x2ee6, (q15_t)0x2ee0, (q15_t)0x2edb, (q15_t)0x2ed5, (q15_t)0x2ecf,
  (q15_t)0x2ec9, (q15_t)0x2ec3, (q15_t)0x2ebd, (q15_t)0x2eb7, (q15_t)0x2eb2, (q15_t)0x2eac, (q15_t)0x2ea6, (q15_t)0x2ea0,
  (q15_t)0x2e9a, (q15_t)0x2e94, (q15_t)0x2e8e, (q15_t)0x2e89, (q15_t)0x2e83, (q15_t)0x2e7d, (q15_t)0x2e77, (q15_t)0x2e71,
  (q15_t)0x2e6b, (q15_t)0x2e65, (q15_t)0x2e60, (q15_t)0x2e5a, (q15_t)0x2e54, (q15_t)0x2e4e, (q15_t)0x2e48, (q15_t)0x2e42,
  (q15_t)0x2e3c, (q15_t)0x2e37, (q15_t)0x2e31, (q15_t)0x2e2b, (q15_t)0x2e25, (q15_t)0x2e1f, (q15_t)0x2e19, (q15_t)0x2e13,
  (q15_t)0x2e0e, (q15_t)0x2e08, (q15_t)0x2e02, (q15_t)0x2dfc, (q15_t)0x2df6, (q15_t)0x2df0, (q15_t)0x2dea, (q15_t)0x2de5,
  (q15_t)0x2ddf, (q15_t)0x2dd9, (q15_t)0x2dd3, (q15_t)0x2dcd, (q15_t)0x2dc7, (q15_t)0x2dc1, (q15_t)0x2dbb, (q15_t)0x2db6,
  (q15_t)0x2db0, (q15_t)0x2daa, (q15_t)0x2da4, (q15_t)0x2d9e, (q15_t)0x2d98, (q15_t)0x2d92, (q15_t)0x2d8d, (q15_t)0x2d87,
  (q15_t)0x2d81, (q15_t)0x2d7b, (q15_t)0x2d75, (q15_t)0x2d6f, (q15_t)0x2d69, (q15_t)0x2d63, (q15_t)0x2d5e, (q15_t)0x2d58,
  (q15_t)0x2d52, (q15_t)0x2d4c, (q15_t)0x2d46, (q15_t)0x2d40, (q15_t)0x2d3a, (q15_t)0x2d34, (q15_t)0x2d2f, (q15_t)0x2d29,
  (q15_t)0x2d23, (q15_t)0x2d1d, (q15_t)0x2d17, (q15_t)0x2d11, (q15_t)0x2d0b, (q15_t)0x2d05, (q15_t)0x2cff, (q15_t)0x2cfa,
  (q15_t)0x2cf4, (q15_t)0x2cee, (q15_t)0x2ce8, (q15_t)0x2ce2, (q15_t)0x2cdc, (q15_t)0x2cd6, (q15_t)0x2cd0, (q15_t)0x2ccb,
  (q15_t)0x2cc5, (q15_t)0x2cbf, (q15_t)0x2cb9, (q15_t)0x2cb3, (q15_t)0x2cad, (q15_t)0x2ca7, (q15_t)0x2ca1, (q15_t)0x2c9b,
  (q15_t)0x2c96, (q15_t)0x2c90, (q15_t)0x2c8a, (q15_t)0x2c84, (q15_t)0x2c7e, (q15_t)0x2c78, (q15_t)0x2c72, (q15_t)0x2c6c,
  (q15_t)0x2c66, (q15_t)0x2c61, (q15_t)0x2c5b, (q15_t)0x2c55, (q15_t)0x2c4f, (q15_t)0x2c49, (q15_t)0x2c43, (q15_t)0x2c3d,
  (q15_t)0x2c37, (q15_t)0x2c31, (q15_t)0x2c2b, (q15_t)0x2c26, (q15_t)0x2c20, (q15_t)0x2c1a, (q15_t)0x2c14, (q15_t)0x2c0e,
  (q15_t)0x2c08, (q15_t)0x2c02, (q15_t)0x2bfc, (q15_t)0x2bf6, (q15_t)0x2bf0, (q15_t)0x2beb, (q15_t)0x2be5, (q15_t)0x2bdf,
  (q15_t)0x2bd9, (q15_t)0x2bd3, (q15_t)0x2bcd, (q15_t)0x2bc7, (q15_t)0x2bc1, (q15_t)0x2bbb, (q15_t)0x2bb5, (q15_t)0x2bb0,
  (q15_t)0x2baa, (q15_t)0x2ba4, (q15_t)0x2b9e, (q15_t)0x2b98, (q15_t)0x2b92, (q15_t)0x2b8c, (q15_t)0x2b86, (q15_t)0x2b80,
  (q15_t)0x2b7a, (q15_t)0x2b74, (q15_t)0x2b6f, (q15_t)0x2b69, (q15_t)0x2b63, (q15_t)0x2b5d, (q15_t)0x2b57, (q15_t)0x2b51,
  (q15_t)0x2b4b, (q15_t)0x2b45, (q15_t)0x2b3f, (q15_t)0x2b39, (q15_t)0x2b33, (q15_t)0x2b2d, (q15_t)0x2b28, (q15_t)0x2b22,
  (q15_t)0x2b1c, (q15_t)0x2b16, (q15_t)0x2b10, (q15_t)0x2b0a, (q15_t)0x2b04, (q15_t)0x2afe, (q15_t)0x2af8, (q15_t)0x2af2,
  (q15_t)0x2aec, (q15_t)0x2ae6, (q15_t)0x2ae1, (q15_t)0x2adb, (q15_t)0x2ad5, (q15_t)0x2acf, (q15_t)0x2ac9, (q15_t)0x2ac3,
  (q15_t)0x2abd, (q15_t)0x2ab7, (q15_t)0x2ab1, (q15_t)0x2aab, (q15_t)0x2aa5, (q15_t)0x2a9f, (q15_t)0x2a99, (q15_t)0x2a94,
  (q15_t)0x2a8e, (q15_t)0x2a88, (q15_t)0x2a82, (q15_t)0x2a7c, (q15_t)0x2a76, (q15_t)0x2a70, (q15_t)0x2a6a, (q15_t)0x2a64,
  (q15_t)0x2a5e, (q15_t)0x2a58, (q15_t)0x2a52, (q15_t)0x2a4c, (q15_t)0x2a47, (q15_t)0x2a41, (q15_t)0x2a3b, (q15_t)0x2a35,
  (q15_t)0x2a2f, (q15_t)0x2a29, (q15_t)0x2a23, (q15_t)0x2a1d, (q15_t)0x2a17, (q15_t)0x2a11, (q15_t)0x2a0b, (q15_t)0x2a05,
  (q15_t)0x29ff, (q15_t)0x29f9, (q15_t)0x29f3, (q15_t)0x29ee, (q15_t)0x29e8, (q15_t)0x29e2, (q15_t)0x29dc, (q15_t)0x29d6,
  (q15_t)0x29d0, (q15_t)0x29ca, (q15_t)0x29c4, (q15_t)0x29be, (q15_t)0x29b8, (q15_t)0x29b2, (q15_t)0x29ac, (q15_t)0x29a6,
  (q15_t)0x29a0, (q15_t)0x299a, (q15_t)0x2994, (q15_t)0x298e, (q15_t)0x2989, (q15_t)0x2983, (q15_t)0x297d, (q15_t)0x2977,
  (q15_t)0x2971, (q15_t)0x296b, (q15_t)0x2965, (q15_t)0x295f, (q15_t)0x2959, (q15_t)0x2953, (q15_t)0x294d, (q15_t)0x2947,
  (q15_t)0x2941, (q15_t)0x293b, (q15_t)0x2935, (q15_t)0x292f, (q15_t)0x2929, (q15_t)0x2923, (q15_t)0x291d, (q15_t)0x2918,
  (q15_t)0x2912, (q15_t)0x290c, (q15_t)0x2906, (q15_t)0x2900, (q15_t)0x28fa, (q15_t)0x28f4, (q15_t)0x28ee, (q15_t)0x28e8,
  (q15_t)0x28e2, (q15_t)0x28dc, (q15_t)0x28d6, (q15_t)0x28d0, (q15_t)0x28ca, (q15_t)0x28c4, (q15_t)0x28be, (q15_t)0x28b8,
  (q15_t)0x28b2, (q15_t)0x28ac, (q15_t)0x28a6, (q15_t)0x28a0, (q15_t)0x289a, (q15_t)0x2895, (q15_t)0x288f, (q15_t)0x2889,
  (q15_t)0x2883, (q15_t)0x287d, (q15_t)0x2877, (q15_t)0x2871, (q15_t)0x286b, (q15_t)0x2865, (q15_t)0x285f, (q15_t)0x2859,
  (q15_t)0x2853, (q15_t)0x284d, (q15_t)0x2847, (q15_t)0x2841, (q15_t)0x283b, (q15_t)0x2835, (q15_t)0x282f, (q15_t)0x2829,
  (q15_t)0x2823, (q15_t)0x281d, (q15_t)0x2817, (q15_t)0x2811, (q15_t)0x280b, (q15_t)0x2805, (q15_t)0x27ff, (q15_t)0x27f9,
  (q15_t)0x27f3, (q15_t)0x27ee, (q15_t)0x27e8, (q15_t)0x27e2, (q15_t)0x27dc, (q15_t)0x27d6, (q15_t)0x27d0, (q15_t)0x27ca,
  (q15_t)0x27c4, (q15_t)0x27be, (q15_t)0x27b8, (q15_t)0x27b2, (q15_t)0x27ac, (q15_t)0x27a6, (q15_t)0x27a0, (q15_t)0x279a,
  (q15_t)0x2794, (q15_t)0x278e, (q15_t)0x2788, (q15_t)0x2782, (q15_t)0x277c, (q15_t)0x2776, (q15_t)0x2770, (q15_t)0x276a,
  (q15_t)0x2764, (q15_t)0x275e, (q15_t)0x2758, (q15_t)0x2752, (q15_t)0x274c, (q15_t)0x2746, (q15_t)0x2740, (q15_t)0x273a,
  (q15_t)0x2734, (q15_t)0x272e, (q15_t)0x2728, (q15_t)0x2722, (q15_t)0x271c, (q15_t)0x2716, (q15_t)0x2710, (q15_t)0x270a,
  (q15_t)0x2704, (q15_t)0x26fe, (q15_t)0x26f8, (q15_t)0x26f2, (q15_t)0x26ec, (q15_t)0x26e7, (q15_t)0x26e1, (q15_t)0x26db,
  (q15_t)0x26d5, (q15_t)0x26cf, (q15_t)0x26c9, (q15_t)0x26c3, (q15_t)0x26bd, (q15_t)0x26b7, (q15_t)0x26b1, (q15_t)0x26ab,
  (q15_t)0x26a5, (q15_t)0x269f, (q15_t)0x2699, (q15_t)0x2693, (q15_t)0x268d, (q15_t)0x2687, (q15_t)0x2681, (q15_t)0x267b,
  (q15_t)0x2675, (q15_t)0x266f, (q15_t)0x2669, (q15_t)0x2663, (q15_t)0x265d, (q15_t)0x2657, (q15_t)0x2651, (q15_t)0x264b,
  (q15_t)0x2645, (q15_t)0x263f, (q15_t)0x2639, (q15_t)0x2633, (q15_t)0x262d, (q15_t)0x2627, (q15_t)0x2621, (q15_t)0x261b,
  (q15_t)0x2615, (q15_t)0x260f, (q15_t)0x2609, (q15_t)0x2603, (q15_t)0x25fd, (q15_t)0x25f7, (q15_t)0x25f1, (q15_t)0x25eb,
  (q15_t)0x25e5, (q15_t)0x25df, (q15_t)0x25d9, (q15_t)0x25d3, (q15_t)0x25cd, (q15_t)0x25c7, (q15_t)0x25c1, (q15_t)0x25bb,
  (q15_t)0x25b5, (q15_t)0x25af, (q15_t)0x25a9, (q15_t)0x25a3, (q15_t)0x259d, (q15_t)0x2597, (q15_t)0x2591, (q15_t)0x258b,
  (q15_t)0x2585, (q15_t)0x257f, (q15_t)0x2579, (q15_t)0x2573, (q15_t)0x256d, (q15_t)0x2567, (q15_t)0x2561, (q15_t)0x255b,
  (q15_t)0x2555, (q15_t)0x254f, (q15_t)0x2549, (q15_t)0x2543, (q15_t)0x253d, (q15_t)0x2537, (q15_t)0x2531, (q15_t)0x252b,
  (q15_t)0x2525, (q15_t)0x251f, (q15_t)0x2519, (q15_t)0x2513, (q15_t)0x250c, (q15_t)0x2506, (q15_t)0x2500, (q15_t)0x24fa,
  (q15_t)0x24f4, (q15_t)0x24ee, (q15_t)0x24e8, (q15_t)0x24e2, (q15_t)0x24dc, (q15_t)0x24d6, (q15_t)0x24d0, (q15_t)0x24ca,
  (q15_t)0x24c4, (q15_t)0x24be, (q15_t)0x24b8, (q15_t)0x24b2, (q15_t)0x24ac, (q15_t)0x24a6, (q15_t)0x24a0, (q15_t)0x249a,
  (q15_t)0x2494, (q15_t)0x248e, (q15_t)0x2488, (q15_t)0x2482, (q15_t)0x247c, (q15_t)0x2476, (q15_t)0x2470, (q15_t)0x246a,
  (q15_t)0x2464, (q15_t)0x245e, (q15_t)0x2458, (q15_t)0x2452, (q15_t)0x244c, (q15_t)0x2446, (q15_t)0x2440, (q15_t)0x243a,
  (q15_t)0x2434, (q15_t)0x242e, (q15_t)0x2428, (q15_t)0x2422, (q15_t)0x241c, (q15_t)0x2416, (q15_t)0x2410, (q15_t)0x240a,
  (q15_t)0x2404, (q15_t)0x23fd, (q15_t)0x23f7, (q15_t)0x23f1, (q15_t)0x23eb, (q15_t)0x23e5, (q15_t)0x23df, (q15_t)0x23d9,
  (q15_t)0x23d3, (q15_t)0x23cd, (q15_t)0x23c7, (q15_t)0x23c1, (q15_t)0x23bb, (q15_t)0x23b5, (q15_t)0x23af, (q15_t)0x23a9,
  (q15_t)0x23a3, (q15_t)0x239d, (q15_t)0x2397, (q15_t)0x2391, (q15_t)0x238b, (q15_t)0x2385, (q15_t)0x237f, (q15_t)0x2379,
  (q15_t)0x2373, (q15_t)0x236d, (q15_t)0x2367, (q15_t)0x2361, (q15_t)0x235b, (q15_t)0x2355, (q15_t)0x234e, (q15_t)0x2348,
  (q15_t)0x2342, (q15_t)0x233c, (q15_t)0x2336, (q15_t)0x2330, (q15_t)0x232a, (q15_t)0x2324, (q15_t)0x231e, (q15_t)0x2318,
  (q15_t)0x2312, (q15_t)0x230c, (q15_t)0x2306, (q15_t)0x2300, (q15_t)0x22fa, (q15_t)0x22f4, (q15_t)0x22ee, (q15_t)0x22e8,
  (q15_t)0x22e2, (q15_t)0x22dc, (q15_t)0x22d6, (q15_t)0x22d0, (q15_t)0x22ca, (q15_t)0x22c4, (q15_t)0x22bd, (q15_t)0x22b7,
  (q15_t)0x22b1, (q15_t)0x22ab, (q15_t)0x22a5, (q15_t)0x229f, (q15_t)0x2299, (q15_t)0x2293, (q15_t)0x228d, (q15_t)0x2287,
  (q15_t)0x2281, (q15_t)0x227b, (q15_t)0x2275, (q15_t)0x226f, (q15_t)0x2269, (q15_t)0x2263, (q15_t)0x225d, (q15_t)0x2257,
  (q15_t)0x2251, (q15_t)0x224a, (q15_t)0x2244, (q15_t)0x223e, (q15_t)0x2238, (q15_t)0x2232, (q15_t)0x222c, (q15_t)0x2226,
  (q15_t)0x2220, (q15_t)0x221a, (q15_t)0x2214, (q15_t)0x220e, (q15_t)0x2208, (q15_t)0x2202, (q15_t)0x21fc, (q15_t)0x21f6,
  (q15_t)0x21f0, (q15_t)0x21ea, (q15_t)0x21e4, (q15_t)0x21dd, (q15_t)0x21d7, (q15_t)0x21d1, (q15_t)0x21cb, (q15_t)0x21c5,
  (q15_t)0x21bf, (q15_t)0x21b9, (q15_t)0x21b3, (q15_t)0x21ad, (q15_t)0x21a7, (q15_t)0x21a1, (q15_t)0x219b, (q15_t)0x2195,
  (q15_t)0x218f, (q15_t)0x2189, (q15_t)0x2183, (q15_t)0x217c, (q15_t)0x2176, (q15_t)0x2170, (q15_t)0x216a, (q15_t)0x2164,
  (q15_t)0x215e, (q15_t)0x2158, (q15_t)0x2152, (q15_t)0x214c, (q15_t)0x2146, (q15_t)0x2140, (q15_t)0x213a, (q15_t)0x2134,
  (q15_t)0x212e, (q15_t)0x2128, (q15_t)0x2121, (q15_t)0x211b, (q15_t)0x2115, (q15_t)0x210f, (q15_t)0x2109, (q15_t)0x2103,
  (q15_t)0x20fd, (q15_t)0x20f7, (q15_t)0x20f1, (q15_t)0x20eb, (q15_t)0x20e5, (q15_t)0x20df, (q15_t)0x20d9, (q15_t)0x20d3,
  (q15_t)0x20cc, (q15_t)0x20c6, (q15_t)0x20c0, (q15_t)0x20ba, (q15_t)0x20b4, (q15_t)0x20ae, (q15_t)0x20a8, (q15_t)0x20a2,
  (q15_t)0x209c, (q15_t)0x2096, (q15_t)0x2090, (q15_t)0x208a, (q15_t)0x2084, (q15_t)0x207e, (q15_t)0x2077, (q15_t)0x2071,
  (q15_t)0x206b, (q15_t)0x2065, (q15_t)0x205f, (q15_t)0x2059, (q15_t)0x2053, (q15_t)0x204d, (q15_t)0x2047, (q15_t)0x2041,
  (q15_t)0x203b, (q15_t)0x2035, (q15_t)0x202e, (q15_t)0x2028, (q15_t)0x2022, (q15_t)0x201c, (q15_t)0x2016, (q15_t)0x2010,
  (q15_t)0x200a, (q15_t)0x2004, (q15_t)0x1ffe, (q15_t)0x1ff8, (q15_t)0x1ff2, (q15_t)0x1fec, (q15_t)0x1fe5, (q15_t)0x1fdf,
  (q15_t)0x1fd9, (q15_t)0x1fd3, (q15_t)0x1fcd, (q15_t)0x1fc7, (q15_t)0x1fc1, (q15_t)0x1fbb, (q15_t)0x1fb5, (q15_t)0x1faf,
  (q15_t)0x1fa9, (q15_t)0x1fa3, (q15_t)0x1f9c, (q15_t)0x1f96, (q15_t)0x1f90, (q15_t)0x1f8a, (q15_t)0x1f84, (q15_t)0x1f7e,
  (q15_t)0x1f78, (q15_t)0x1f72, (q15_t)0x1f6c, (q15_t)0x1f66, (q15_t)0x1f60, (q15_t)0x1f59, (q15_t)0x1f53, (q15_t)0x1f4d,
  (q15_t)0x1f47, (q15_t)0x1f41, (q15_t)0x1f3b, (q15_t)0x1f35, (q15_t)0x1f2f, (q15_t)0x1f29, (q15_t)0x1f23, (q15_t)0x1f1d,
  (q15_t)0x1f16, (q15_t)0x1f10, (q15_t)0x1f0a, (q15_t)0x1f04, (q15_t)0x1efe, (q15_t)0x1ef8, (q15_t)0x1ef2, (q15_t)0x1eec,
  (q15_t)0x1ee6, (q15_t)0x1ee0, (q15_t)0x1ed9, (q15_t)0x1ed3, (q15_t)0x1ecd, (q15_t)0x1ec7, (q15_t)0x1ec1, (q15_t)0x1ebb,
  (q15_t)0x1eb5, (q15_t)0x1eaf, (q15_t)0x1ea9, (q15_t)0x1ea3, (q15_t)0x1e9c, (q15_t)0x1e96, (q15_t)0x1e90, (q15_t)0x1e8a,
  (q15_t)0x1e84, (q15_t)0x1e7e, (q15_t)0x1e78, (q15_t)0x1e72, (q15_t)0x1e6c, (q15_t)0x1e66, (q15_t)0x1e5f, (q15_t)0x1e59,
  (q15_t)0x1e53, (q15_t)0x1e4d, (q15_t)0x1e47, (q15_t)0x1e41, (q15_t)0x1e3b, (q15_t)0x1e35, (q15_t)0x1e2f, (q15_t)0x1e29,
  (q15_t)0x1e22, (q15_t)0x1e1c, (q15_t)0x1e16, (q15_t)0x1e10, (q15_t)0x1e0a, (q15_t)0x1e04, (q15_t)0x1dfe, (q15_t)0x1df8,
  (q15_t)0x1df2, (q15_t)0x1deb, (q15_t)0x1de5, (q15_t)0x1ddf, (q15_t)0x1dd9, (q15_t)0x1dd3, (q15_t)0x1dcd, (q15_t)0x1dc7,
  (q15_t)0x1dc1, (q15_t)0x1dbb, (q15_t)0x1db4, (q15_t)0x1dae, (q15_t)0x1da8, (q15_t)0x1da2, (q15_t)0x1d9c, (q15_t)0x1d96,
  (q15_t)0x1d90, (q15_t)0x1d8a, (q15_t)0x1d84, (q15_t)0x1d7d, (q15_t)0x1d77, (q15_t)0x1d71, (q15_t)0x1d6b, (q15_t)0x1d65,
  (q15_t)0x1d5f, (q15_t)0x1d59, (q15_t)0x1d53, (q15_t)0x1d4c, (q15_t)0x1d46, (q15_t)0x1d40, (q15_t)0x1d3a, (q15_t)0x1d34,
  (q15_t)0x1d2e, (q15_t)0x1d28, (q15_t)0x1d22, (q15_t)0x1d1c, (q15_t)0x1d15, (q15_t)0x1d0f, (q15_t)0x1d09, (q15_t)0x1d03,
  (q15_t)0x1cfd, (q15_t)0x1cf7, (q15_t)0x1cf1, (q15_t)0x1ceb, (q15_t)0x1ce4, (q15_t)0x1cde, (q15_t)0x1cd8, (q15_t)0x1cd2,
  (q15_t)0x1ccc, (q15_t)0x1cc6, (q15_t)0x1cc0, (q15_t)0x1cba, (q15_t)0x1cb3, (q15_t)0x1cad, (q15_t)0x1ca7, (q15_t)0x1ca1,
  (q15_t)0x1c9b, (q15_t)0x1c95, (q15_t)0x1c8f, (q15_t)0x1c89, (q15_t)0x1c83, (q15_t)0x1c7c, (q15_t)0x1c76, (q15_t)0x1c70,
  (q15_t)0x1c6a, (q15_t)0x1c64, (q15_t)0x1c5e, (q15_t)0x1c58, (q15_t)0x1c51, (q15_t)0x1c4b, (q15_t)0x1c45, (q15_t)0x1c3f,
  (q15_t)0x1c39, (q15_t)0x1c33, (q15_t)0x1c2d, (q15_t)0x1c27, (q15_t)0x1c20, (q15_t)0x1c1a, (q15_t)0x1c14, (q15_t)0x1c0e,
  (q15_t)0x1c08, (q15_t)0x1c02, (q15_t)0x1bfc, (q15_t)0x1bf6, (q15_t)0x1bef, (q15_t)0x1be9, (q15_t)0x1be3, (q15_t)0x1bdd,
  (q15_t)0x1bd7, (q15_t)0x1bd1, (q15_t)0x1bcb, (q15_t)0x1bc4, (q15_t)0x1bbe, (q15_t)0x1bb8, (q15_t)0x1bb2, (q15_t)0x1bac,
  (q15_t)0x1ba6, (q15_t)0x1ba0, (q15_t)0x1b9a, (q15_t)0x1b93, (q15_t)0x1b8d, (q15_t)0x1b87, (q15_t)0x1b81, (q15_t)0x1b7b,
  (q15_t)0x1b75, (q15_t)0x1b6f, (q15_t)0x1b68, (q15_t)0x1b62, (q15_t)0x1b5c, (q15_t)0x1b56, (q15_t)0x1b50, (q15_t)0x1b4a,
  (q15_t)0x1b44, (q15_t)0x1b3d, (q15_t)0x1b37, (q15_t)0x1b31, (q15_t)0x1b2b, (q15_t)0x1b25, (q15_t)0x1b1f, (q15_t)0x1b19,
  (q15_t)0x1b13, (q15_t)0x1b0c, (q15_t)0x1b06, (q15_t)0x1b00, (q15_t)0x1afa, (q15_t)0x1af4, (q15_t)0x1aee, (q15_t)0x1ae8,
  (q15_t)0x1ae1, (q15_t)0x1adb, (q15_t)0x1ad5, (q15_t)0x1acf, (q15_t)0x1ac9, (q15_t)0x1ac3, (q15_t)0x1abd, (q15_t)0x1ab6,
  (q15_t)0x1ab0, (q15_t)0x1aaa, (q15_t)0x1aa4, (q15_t)0x1a9e, (q15_t)0x1a98, (q15_t)0x1a91, (q15_t)0x1a8b, (q15_t)0x1a85,
  (q15_t)0x1a7f, (q15_t)0x1a79, (q15_t)0x1a73, (q15_t)0x1a6d, (q15_t)0x1a66, (q15_t)0x1a60, (q15_t)0x1a5a, (q15_t)0x1a54,
  (q15_t)0x1a4e, (q15_t)0x1a48, (q15_t)0x1a42, (q15_t)0x1a3b, (q15_t)0x1a35, (q15_t)0x1a2f, (q15_t)0x1a29, (q15_t)0x1a23,
  (q15_t)0x1a1d, (q15_t)0x1a17, (q15_t)0x1a10, (q15_t)0x1a0a, (q15_t)0x1a04, (q15_t)0x19fe, (q15_t)0x19f8, (q15_t)0x19f2,
  (q15_t)0x19eb, (q15_t)0x19e5, (q15_t)0x19df, (q15_t)0x19d9, (q15_t)0x19d3, (q15_t)0x19cd, (q15_t)0x19c7, (q15_t)0x19c0,
  (q15_t)0x19ba, (q15_t)0x19b4, (q15_t)0x19ae, (q15_t)0x19a8, (q15_t)0x19a2, (q15_t)0x199b, (q15_t)0x1995, (q15_t)0x198f,
  (q15_t)0x1989, (q15_t)0x1983, (q15_t)0x197d, (q15_t)0x1977, (q15_t)0x1970, (q15_t)0x196a, (q15_t)0x1964, (q15_t)0x195e,
  (q15_t)0x1958, (q15_t)0x1952, (q15_t)0x194b, (q15_t)0x1945, (q15_t)0x193f, (q15_t)0x1939, (q15_t)0x1933, (q15_t)0x192d,
  (q15_t)0x1926, (q15_t)0x1920, (q15_t)0x191a, (q15_t)0x1914, (q15_t)0x190e, (q15_t)0x1908, (q15_t)0x1901, (q15_t)0x18fb,
  (q15_t)0x18f5, (q15_t)0x18ef, (q15_t)0x18e9, (q15_t)0x18e3, (q15_t)0x18dc, (q15_t)0x18d6, (q15_t)0x18d0, (q15_t)0x18ca,
  (q15_t)0x18c4, (q15_t)0x18be, (q15_t)0x18b8, (q15_t)0x18b1, (q15_t)0x18ab, (q15_t)0x18a5, (q15_t)0x189f, (q15_t)0x1899,
  (q15_t)0x1893, (q15_t)0x188c, (q15_t)0x1886, (q15_t)0x1880, (q15_t)0x187a, (q15_t)0x1874, (q15_t)0x186e, (q15_t)0x1867,
  (q15_t)0x1861, (q15_t)0x185b, (q15_t)0x1855, (q15_t)0x184f, (q15_t)0x1848, (q15_t)0x1842, (q15_t)0x183c, (q15_t)0x1836,
  (q15_t)0x1830, (q15_t)0x182a, (q15_t)0x1823, (q15_t)0x181d, (q15_t)0x1817, (q15_t)0x1811, (q15_t)0x180b, (q15_t)0x1805,
  (q15_t)0x17fe, (q15_t)0x17f8, (q15_t)0x17f2, (q15_t)0x17ec, (q15_t)0x17e6, (q15_t)0x17e0, (q15_t)0x17d9, (q15_t)0x17d3,
  (q15_t)0x17cd, (q15_t)0x17c7, (q15_t)0x17c1, (q15_t)0x17bb, (q15_t)0x17b4, (q15_t)0x17ae, (q15_t)0x17a8, (q15_t)0x17a2,
  (q15_t)0x179c, (q15_t)0x1795, (q15_t)0x178f, (q15_t)0x1789, (q15_t)0x1783, (q15_t)0x177d, (q15_t)0x1777, (q15_t)0x1770,
  (q15_t)0x176a, (q15_t)0x1764, (q15_t)0x175e, (q15_t)0x1758, (q15_t)0x1752, (q15_t)0x174b, (q15_t)0x1745, (q15_t)0x173f,
  (q15_t)0x1739, (q15_t)0x1733, (q15_t)0x172c, (q15_t)0x1726, (q15_t)0x1720, (q15_t)0x171a, (q15_t)0x1714, (q15_t)0x170e,
  (q15_t)0x1707, (q15_t)0x1701, (q15_t)0x16fb, (q15_t)0x16f5, (q15_t)0x16ef, (q15_t)0x16e8, (q15_t)0x16e2, (q15_t)0x16dc,
  (q15_t)0x16d6, (q15_t)0x16d0, (q15_t)0x16ca, (q15_t)0x16c3, (q15_t)0x16bd, (q15_t)0x16b7, (q15_t)0x16b1, (q15_t)0x16ab,
  (q15_t)0x16a4, (q15_t)0x169e, (q15_t)0x1698, (q15_t)0x1692, (q15_t)0x168c, (q15_t)0x1686, (q15_t)0x167f, (q15_t)0x1679,
  (q15_t)0x1673, (q15_t)0x166d, (q15_t)0x1667, (q15_t)0x1660, (q15_t)0x165a, (q15_t)0x1654, (q15_t)0x164e, (q15_t)0x1648,
  (q15_t)0x1642, (q15_t)0x163b, (q15_t)0x1635, (q15_t)0x162f, (q15_t)0x1629, (q15_t)0x1623, (q15_t)0x161c, (q15_t)0x1616,
  (q15_t)0x1610, (q15_t)0x160a, (q15_t)0x1604, (q15_t)0x15fd, (q15_t)0x15f7, (q15_t)0x15f1, (q15_t)0x15eb, (q15_t)0x15e5,
  (q15_t)0x15de, (q15_t)0x15d8, (q15_t)0x15d2, (q15_t)0x15cc, (q15_t)0x15c6, (q15_t)0x15c0, (q15_t)0x15b9, (q15_t)0x15b3,
  (q15_t)0x15ad, (q15_t)0x15a7, (q15_t)0x15a1, (q15_t)0x159a, (q15_t)0x1594, (q15_t)0x158e, (q15_t)0x1588, (q15_t)0x1582,
  (q15_t)0x157b, (q15_t)0x1575, (q15_t)0x156f, (q15_t)0x1569, (q15_t)0x1563, (q15_t)0x155c, (q15_t)0x1556, (q15_t)0x1550,
  (q15_t)0x154a, (q15_t)0x1544, (q15_t)0x153d, (q15_t)0x1537, (q15_t)0x1531, (q15_t)0x152b, (q15_t)0x1525, (q15_t)0x151e,
  (q15_t)0x1518, (q15_t)0x1512, (q15_t)0x150c, (q15_t)0x1506, (q15_t)0x14ff, (q15_t)0x14f9, (q15_t)0x14f3, (q15_t)0x14ed,
  (q15_t)0x14e7, (q15_t)0x14e0, (q15_t)0x14da, (q15_t)0x14d4, (q15_t)0x14ce, (q15_t)0x14c8, (q15_t)0x14c1, (q15_t)0x14bb,
  (q15_t)0x14b5, (q15_t)0x14af, (q15_t)0x14a9, (q15_t)0x14a2, (q15_t)0x149c, (q15_t)0x1496, (q15_t)0x1490, (q15_t)0x148a,
  (q15_t)0x1483, (q15_t)0x147d, (q15_t)0x1477, (q15_t)0x1471, (q15_t)0x146b, (q15_t)0x1464, (q15_t)0x145e, (q15_t)0x1458,
  (q15_t)0x1452, (q15_t)0x144c, (q15_t)0x1445, (q15_t)0x143f, (q15_t)0x1439, (q15_t)0x1433, (q15_t)0x142d, (q15_t)0x1426,
  (q15_t)0x1420, (q15_t)0x141a, (q15_t)0x1414, (q15_t)0x140e, (q15_t)0x1407, (q15_t)0x1401, (q15_t)0x13fb, (q15_t)0x13f5,
  (q15_t)0x13ef, (q15_t)0x13e8, (q15_t)0x13e2, (q15_t)0x13dc, (q15_t)0x13d6, (q15_t)0x13d0, (q15_t)0x13c9, (q15_t)0x13c3,
  (q15_t)0x13bd, (q15_t)0x13b7, (q15_t)0x13b1, (q15_t)0x13aa, (q15_t)0x13a4, (q15_t)0x139e, (q15_t)0x1398, (q15_t)0x1391,
  (q15_t)0x138b, (q15_t)0x1385, (q15_t)0x137f, (q15_t)0x1379, (q15_t)0x1372, (q15_t)0x136c, (q15_t)0x1366, (q15_t)0x1360,
  (q15_t)0x135a, (q15_t)0x1353, (q15_t)0x134d, (q15_t)0x1347, (q15_t)0x1341, (q15_t)0x133b, (q15_t)0x1334, (q15_t)0x132e,
  (q15_t)0x1328, (q15_t)0x1322, (q15_t)0x131b, (q15_t)0x1315, (q15_t)0x130f, (q15_t)0x1309, (q15_t)0x1303, (q15_t)0x12fc,
  (q15_t)0x12f6, (q15_t)0x12f0, (q15_t)0x12ea, (q15_t)0x12e4, (q15_t)0x12dd, (q15_t)0x12d7, (q15_t)0x12d1, (q15_t)0x12cb,
  (q15_t)0x12c4, (q15_t)0x12be, (q15_t)0x12b8, (q15_t)0x12b2, (q15_t)0x12ac, (q15_t)0x12a5, (q15_t)0x129f, (q15_t)0x1299,
  (q15_t)0x1293, (q15_t)0x128d, (q15_t)0x1286, (q15_t)0x1280, (q15_t)0x127a, (q15_t)0x1274, (q15_t)0x126d, (q15_t)0x1267,
  (q15_t)0x1261, (q15_t)0x125b, (q15_t)0x1255, (q15_t)0x124e, (q15_t)0x1248, (q15_t)0x1242, (q15_t)0x123c, (q15_t)0x1235,
  (q15_t)0x122f, (q15_t)0x1229, (q15_t)0x1223, (q15_t)0x121d, (q15_t)0x1216, (q15_t)0x1210, (q15_t)0x120a, (q15_t)0x1204,
  (q15_t)0x11fd, (q15_t)0x11f7, (q15_t)0x11f1, (q15_t)0x11eb, (q15_t)0x11e5, (q15_t)0x11de, (q15_t)0x11d8, (q15_t)0x11d2,
  (q15_t)0x11cc, (q15_t)0x11c5, (q15_t)0x11bf, (q15_t)0x11b9, (q15_t)0x11b3, (q15_t)0x11ad, (q15_t)0x11a6, (q15_t)0x11a0,
  (q15_t)0x119a, (q15_t)0x1194, (q15_t)0x118d, (q15_t)0x1187, (q15_t)0x1181, (q15_t)0x117b, (q15_t)0x1175, (q15_t)0x116e,
  (q15_t)0x1168, (q15_t)0x1162, (q15_t)0x115c, (q15_t)0x1155, (q15_t)0x114f, (q15_t)0x1149, (q15_t)0x1143, (q15_t)0x113d,
  (q15_t)0x1136, (q15_t)0x1130, (q15_t)0x112a, (q15_t)0x1124, (q15_t)0x111d, (q15_t)0x1117, (q15_t)0x1111, (q15_t)0x110b,
  (q15_t)0x1105, (q15_t)0x10fe, (q15_t)0x10f8, (q15_t)0x10f2, (q15_t)0x10ec, (q15_t)0x10e5, (q15_t)0x10df, (q15_t)0x10d9,
  (q15_t)0x10d3, (q15_t)0x10cc, (q15_t)0x10c6, (q15_t)0x10c0, (q15_t)0x10ba, (q15_t)0x10b4, (q15_t)0x10ad, (q15_t)0x10a7,
  (q15_t)0x10a1, (q15_t)0x109b, (q15_t)0x1094, (q15_t)0x108e, (q15_t)0x1088, (q15_t)0x1082, (q15_t)0x107b, (q15_t)0x1075,
  (q15_t)0x106f, (q15_t)0x1069, (q15_t)0x1063, (q15_t)0x105c, (q15_t)0x1056, (q15_t)0x1050, (q15_t)0x104a, (q15_t)0x1043,
  (q15_t)0x103d, (q15_t)0x1037, (q15_t)0x1031, (q15_t)0x102a, (q15_t)0x1024, (q15_t)0x101e, (q15_t)0x1018, (q15_t)0x1012,
  (q15_t)0x100b, (q15_t)0x1005, (q15_t)0xfff, (q15_t)0xff9, (q15_t)0xff2, (q15_t)0xfec, (q15_t)0xfe6, (q15_t)0xfe0,
  (q15_t)0xfd9, (q15_t)0xfd3, (q15_t)0xfcd, (q15_t)0xfc7, (q15_t)0xfc0, (q15_t)0xfba, (q15_t)0xfb4, (q15_t)0xfae,
  (q15_t)0xfa8, (q15_t)0xfa1, (q15_t)0xf9b, (q15_t)0xf95, (q15_t)0xf8f, (q15_t)0xf88, (q15_t)0xf82, (q15_t)0xf7c,
  (q15_t)0xf76, (q15_t)0xf6f, (q15_t)0xf69, (q15_t)0xf63, (q15_t)0xf5d, (q15_t)0xf56, (q15_t)0xf50, (q15_t)0xf4a,
  (q15_t)0xf44, (q15_t)0xf3e, (q15_t)0xf37, (q15_t)0xf31, (q15_t)0xf2b, (q15_t)0xf25, (q15_t)0xf1e, (q15_t)0xf18,
  (q15_t)0xf12, (q15_t)0xf0c, (q15_t)0xf05, (q15_t)0xeff, (q15_t)0xef9, (q15_t)0xef3, (q15_t)0xeec, (q15_t)0xee6,
  (q15_t)0xee0, (q15_t)0xeda, (q15_t)0xed3, (q15_t)0xecd, (q15_t)0xec7, (q15_t)0xec1, (q15_t)0xeba, (q15_t)0xeb4,
  (q15_t)0xeae, (q15_t)0xea8, (q15_t)0xea1, (q15_t)0xe9b, (q15_t)0xe95, (q15_t)0xe8f, (q15_t)0xe89, (q15_t)0xe82,
  (q15_t)0xe7c, (q15_t)0xe76, (q15_t)0xe70, (q15_t)0xe69, (q15_t)0xe63, (q15_t)0xe5d, (q15_t)0xe57, (q15_t)0xe50,
  (q15_t)0xe4a, (q15_t)0xe44, (q15_t)0xe3e, (q15_t)0xe37, (q15_t)0xe31, (q15_t)0xe2b, (q15_t)0xe25, (q15_t)0xe1e,
  (q15_t)0xe18, (q15_t)0xe12, (q15_t)0xe0c, (q15_t)0xe05, (q15_t)0xdff, (q15_t)0xdf9, (q15_t)0xdf3, (q15_t)0xdec,
  (q15_t)0xde6, (q15_t)0xde0, (q15_t)0xdda, (q15_t)0xdd3, (q15_t)0xdcd, (q15_t)0xdc7, (q15_t)0xdc1, (q15_t)0xdba,
  (q15_t)0xdb4, (q15_t)0xdae, (q15_t)0xda8, (q15_t)0xda1, (q15_t)0xd9b, (q15_t)0xd95, (q15_t)0xd8f, (q15_t)0xd88,
  (q15_t)0xd82, (q15_t)0xd7c, (q15_t)0xd76, (q15_t)0xd6f, (q15_t)0xd69, (q15_t)0xd63, (q15_t)0xd5d, (q15_t)0xd56,
  (q15_t)0xd50, (q15_t)0xd4a, (q15_t)0xd44, (q15_t)0xd3d, (q15_t)0xd37, (q15_t)0xd31, (q15_t)0xd2b, (q15_t)0xd24,
  (q15_t)0xd1e, (q15_t)0xd18, (q15_t)0xd12, (q15_t)0xd0b, (q15_t)0xd05, (q15_t)0xcff, (q15_t)0xcf9, (q15_t)0xcf2,
  (q15_t)0xcec, (q15_t)0xce6, (q15_t)0xce0, (q15_t)0xcd9, (q15_t)0xcd3, (q15_t)0xccd, (q15_t)0xcc7, (q15_t)0xcc0,
  (q15_t)0xcba, (q15_t)0xcb4, (q15_t)0xcae, (q15_t)0xca7, (q15_t)0xca1, (q15_t)0xc9b, (q15_t)0xc95, (q15_t)0xc8e,
  (q15_t)0xc88, (q15_t)0xc82, (q15_t)0xc7c, (q15_t)0xc75, (q15_t)0xc6f, (q15_t)0xc69, (q15_t)0xc63, (q15_t)0xc5c,
  (q15_t)0xc56, (q15_t)0xc50, (q15_t)0xc4a, (q15_t)0xc43, (q15_t)0xc3d, (q15_t)0xc37, (q15_t)0xc31, (q15_t)0xc2a,
  (q15_t)0xc24, (q15_t)0xc1e, (q15_t)0xc18, (q15_t)0xc11, (q15_t)0xc0b, (q15_t)0xc05, (q15_t)0xbff, (q15_t)0xbf8,
  (q15_t)0xbf2, (q15_t)0xbec, (q15_t)0xbe6, (q15_t)0xbdf, (q15_t)0xbd9, (q15_t)0xbd3, (q15_t)0xbcd, (q15_t)0xbc6,
  (q15_t)0xbc0, (q15_t)0xbba, (q15_t)0xbb4, (q15_t)0xbad, (q15_t)0xba7, (q15_t)0xba1, (q15_t)0xb9b, (q15_t)0xb94,
  (q15_t)0xb8e, (q15_t)0xb88, (q15_t)0xb81, (q15_t)0xb7b, (q15_t)0xb75, (q15_t)0xb6f, (q15_t)0xb68, (q15_t)0xb62,
  (q15_t)0xb5c, (q15_t)0xb56, (q15_t)0xb4f, (q15_t)0xb49, (q15_t)0xb43, (q15_t)0xb3d, (q15_t)0xb36, (q15_t)0xb30,
  (q15_t)0xb2a, (q15_t)0xb24, (q15_t)0xb1d, (q15_t)0xb17, (q15_t)0xb11, (q15_t)0xb0b, (q15_t)0xb04, (q15_t)0xafe,
  (q15_t)0xaf8, (q15_t)0xaf2, (q15_t)0xaeb, (q15_t)0xae5, (q15_t)0xadf, (q15_t)0xad8, (q15_t)0xad2, (q15_t)0xacc,
  (q15_t)0xac6, (q15_t)0xabf, (q15_t)0xab9, (q15_t)0xab3, (q15_t)0xaad, (q15_t)0xaa6, (q15_t)0xaa0, (q15_t)0xa9a,
  (q15_t)0xa94, (q15_t)0xa8d, (q15_t)0xa87, (q15_t)0xa81, (q15_t)0xa7b, (q15_t)0xa74, (q15_t)0xa6e, (q15_t)0xa68,
  (q15_t)0xa62, (q15_t)0xa5b, (q15_t)0xa55, (q15_t)0xa4f, (q15_t)0xa48, (q15_t)0xa42, (q15_t)0xa3c, (q15_t)0xa36,
  (q15_t)0xa2f, (q15_t)0xa29, (q15_t)0xa23, (q15_t)0xa1d, (q15_t)0xa16, (q15_t)0xa10, (q15_t)0xa0a, (q15_t)0xa04,
  (q15_t)0x9fd, (q15_t)0x9f7, (q15_t)0x9f1, (q15_t)0x9eb, (q15_t)0x9e4, (q15_t)0x9de, (q15_t)0x9d8, (q15_t)0x9d1,
  (q15_t)0x9cb, (q15_t)0x9c5, (q15_t)0x9bf, (q15_t)0x9b8, (q15_t)0x9b2, (q15_t)0x9ac, (q15_t)0x9a6, (q15_t)0x99f,
  (q15_t)0x999, (q15_t)0x993, (q15_t)0x98d, (q15_t)0x986, (q15_t)0x980, (q15_t)0x97a, (q15_t)0x973, (q15_t)0x96d,
  (q15_t)0x967, (q15_t)0x961, (q15_t)0x95a, (q15_t)0x954, (q15_t)0x94e, (q15_t)0x948, (q15_t)0x941, (q15_t)0x93b,
  (q15_t)0x935, (q15_t)0x92f, (q15_t)0x928, (q15_t)0x922, (q15_t)0x91c, (q15_t)0x915, (q15_t)0x90f, (q15_t)0x909,
  (q15_t)0x903, (q15_t)0x8fc, (q15_t)0x8f6, (q15_t)0x8f0, (q15_t)0x8ea, (q15_t)0x8e3, (q15_t)0x8dd, (q15_t)0x8d7,
  (q15_t)0x8d1, (q15_t)0x8ca, (q15_t)0x8c4, (q15_t)0x8be, (q15_t)0x8b7, (q15_t)0x8b1, (q15_t)0x8ab, (q15_t)0x8a5,
  (q15_t)0x89e, (q15_t)0x898, (q15_t)0x892, (q15_t)0x88c, (q15_t)0x885, (q15_t)0x87f, (q15_t)0x879, (q15_t)0x872,
  (q15_t)0x86c, (q15_t)0x866, (q15_t)0x860, (q15_t)0x859, (q15_t)0x853, (q15_t)0x84d, (q15_t)0x847, (q15_t)0x840,
  (q15_t)0x83a, (q15_t)0x834, (q15_t)0x82e, (q15_t)0x827, (q15_t)0x821, (q15_t)0x81b, (q15_t)0x814, (q15_t)0x80e,
  (q15_t)0x808, (q15_t)0x802, (q15_t)0x7fb, (q15_t)0x7f5, (q15_t)0x7ef, (q15_t)0x7e9, (q15_t)0x7e2, (q15_t)0x7dc,
  (q15_t)0x7d6, (q15_t)0x7cf, (q15_t)0x7c9, (q15_t)0x7c3, (q15_t)0x7bd, (q15_t)0x7b6, (q15_t)0x7b0, (q15_t)0x7aa,
  (q15_t)0x7a4, (q15_t)0x79d, (q15_t)0x797, (q15_t)0x791, (q15_t)0x78a, (q15_t)0x784, (q15_t)0x77e, (q15_t)0x778,
  (q15_t)0x771, (q15_t)0x76b, (q15_t)0x765, (q15_t)0x75f, (q15_t)0x758, (q15_t)0x752, (q15_t)0x74c, (q15_t)0x745,
  (q15_t)0x73f, (q15_t)0x739, (q15_t)0x733, (q15_t)0x72c, (q15_t)0x726, (q15_t)0x720, (q15_t)0x71a, (q15_t)0x713,
  (q15_t)0x70d, (q15_t)0x707, (q15_t)0x700, (q15_t)0x6fa, (q15_t)0x6f4, (q15_t)0x6ee, (q15_t)0x6e7, (q15_t)0x6e1,
  (q15_t)0x6db, (q15_t)0x6d5, (q15_t)0x6ce, (q15_t)0x6c8, (q15_t)0x6c2, (q15_t)0x6bb, (q15_t)0x6b5, (q15_t)0x6af,
  (q15_t)0x6a9, (q15_t)0x6a2, (q15_t)0x69c, (q15_t)0x696, (q15_t)0x690, (q15_t)0x689, (q15_t)0x683, (q15_t)0x67d,
  (q15_t)0x676, (q15_t)0x670, (q15_t)0x66a, (q15_t)0x664, (q15_t)0x65d, (q15_t)0x657, (q15_t)0x651, (q15_t)0x64a,
  (q15_t)0x644, (q15_t)0x63e, (q15_t)0x638, (q15_t)0x631, (q15_t)0x62b, (q15_t)0x625, (q15_t)0x61f, (q15_t)0x618,
  (q15_t)0x612, (q15_t)0x60c, (q15_t)0x605, (q15_t)0x5ff, (q15_t)0x5f9, (q15_t)0x5f3, (q15_t)0x5ec, (q15_t)0x5e6,
  (q15_t)0x5e0, (q15_t)0x5da, (q15_t)0x5d3, (q15_t)0x5cd, (q15_t)0x5c7, (q15_t)0x5c0, (q15_t)0x5ba, (q15_t)0x5b4,
  (q15_t)0x5ae, (q15_t)0x5a7, (q15_t)0x5a1, (q15_t)0x59b, (q15_t)0x594, (q15_t)0x58e, (q15_t)0x588, (q15_t)0x582,
  (q15_t)0x57b, (q15_t)0x575, (q15_t)0x56f, (q15_t)0x569, (q15_t)0x562, (q15_t)0x55c, (q15_t)0x556, (q15_t)0x54f,
  (q15_t)0x549, (q15_t)0x543, (q15_t)0x53d, (q15_t)0x536, (q15_t)0x530, (q15_t)0x52a, (q15_t)0x523, (q15_t)0x51d,
  (q15_t)0x517, (q15_t)0x511, (q15_t)0x50a, (q15_t)0x504, (q15_t)0x4fe, (q15_t)0x4f8, (q15_t)0x4f1, (q15_t)0x4eb,
  (q15_t)0x4e5, (q15_t)0x4de, (q15_t)0x4d8, (q15_t)0x4d2, (q15_t)0x4cc, (q15_t)0x4c5, (q15_t)0x4bf, (q15_t)0x4b9,
  (q15_t)0x4b2, (q15_t)0x4ac, (q15_t)0x4a6, (q15_t)0x4a0, (q15_t)0x499, (q15_t)0x493, (q15_t)0x48d, (q15_t)0x487,
  (q15_t)0x480, (q15_t)0x47a, (q15_t)0x474, (q15_t)0x46d, (q15_t)0x467, (q15_t)0x461, (q15_t)0x45b, (q15_t)0x454,
  (q15_t)0x44e, (q15_t)0x448, (q15_t)0x441, (q15_t)0x43b, (q15_t)0x435, (q15_t)0x42f, (q15_t)0x428, (q15_t)0x422,
  (q15_t)0x41c, (q15_t)0x415, (q15_t)0x40f, (q15_t)0x409, (q15_t)0x403, (q15_t)0x3fc, (q15_t)0x3f6, (q15_t)0x3f0,
  (q15_t)0x3ea, (q15_t)0x3e3, (q15_t)0x3dd, (q15_t)0x3d7, (q15_t)0x3d0, (q15_t)0x3ca, (q15_t)0x3c4, (q15_t)0x3be,
  (q15_t)0x3b7, (q15_t)0x3b1, (q15_t)0x3ab, (q15_t)0x3a4, (q15_t)0x39e, (q15_t)0x398, (q15_t)0x392, (q15_t)0x38b,
  (q15_t)0x385, (q15_t)0x37f, (q15_t)0x378, (q15_t)0x372, (q15_t)0x36c, (q15_t)0x366, (q15_t)0x35f, (q15_t)0x359,
  (q15_t)0x353, (q15_t)0x34c, (q15_t)0x346, (q15_t)0x340, (q15_t)0x33a, (q15_t)0x333, (q15_t)0x32d, (q15_t)0x327,
  (q15_t)0x321, (q15_t)0x31a, (q15_t)0x314, (q15_t)0x30e, (q15_t)0x307, (q15_t)0x301, (q15_t)0x2fb, (q15_t)0x2f5,
  (q15_t)0x2ee, (q15_t)0x2e8, (q15_t)0x2e2, (q15_t)0x2db, (q15_t)0x2d5, (q15_t)0x2cf, (q15_t)0x2c9, (q15_t)0x2c2,
  (q15_t)0x2bc, (q15_t)0x2b6, (q15_t)0x2af, (q15_t)0x2a9, (q15_t)0x2a3, (q15_t)0x29d, (q15_t)0x296, (q15_t)0x290,
  (q15_t)0x28a, (q15_t)0x283, (q15_t)0x27d, (q15_t)0x277, (q15_t)0x271, (q15_t)0x26a, (q15_t)0x264, (q15_t)0x25e,
  (q15_t)0x258, (q15_t)0x251, (q15_t)0x24b, (q15_t)0x245, (q15_t)0x23e, (q15_t)0x238, (q15_t)0x232, (q15_t)0x22c,
  (q15_t)0x225, (q15_t)0x21f, (q15_t)0x219, (q15_t)0x212, (q15_t)0x20c, (q15_t)0x206, (q15_t)0x200, (q15_t)0x1f9,
  (q15_t)0x1f3, (q15_t)0x1ed, (q15_t)0x1e6, (q15_t)0x1e0, (q15_t)0x1da, (q15_t)0x1d4, (q15_t)0x1cd, (q15_t)0x1c7,
  (q15_t)0x1c1, (q15_t)0x1ba, (q15_t)0x1b4, (q15_t)0x1ae, (q15_t)0x1a8, (q15_t)0x1a1, (q15_t)0x19b, (q15_t)0x195,
  (q15_t)0x18e, (q15_t)0x188, (q15_t)0x182, (q15_t)0x17c, (q15_t)0x175, (q15_t)0x16f, (q15_t)0x169, (q15_t)0x162,
  (q15_t)0x15c, (q15_t)0x156, (q15_t)0x150, (q15_t)0x149, (q15_t)0x143, (q15_t)0x13d, (q15_t)0x137, (q15_t)0x130,
  (q15_t)0x12a, (q15_t)0x124, (q15_t)0x11d, (q15_t)0x117, (q15_t)0x111, (q15_t)0x10b, (q15_t)0x104, (q15_t)0xfe,
  (q15_t)0xf8, (q15_t)0xf1, (q15_t)0xeb, (q15_t)0xe5, (q15_t)0xdf, (q15_t)0xd8, (q15_t)0xd2, (q15_t)0xcc,
  (q15_t)0xc5, (q15_t)0xbf, (q15_t)0xb9, (q15_t)0xb3, (q15_t)0xac, (q15_t)0xa6, (q15_t)0xa0, (q15_t)0x99,
  (q15_t)0x93, (q15_t)0x8d, (q15_t)0x87, (q15_t)0x80, (q15_t)0x7a, (q15_t)0x74, (q15_t)0x6d, (q15_t)0x67,
  (q15_t)0x61, (q15_t)0x5b, (q15_t)0x54, (q15_t)0x4e, (q15_t)0x48, (q15_t)0x41, (q15_t)0x3b, (q15_t)0x35,
  (q15_t)0x2f, (q15_t)0x28, (q15_t)0x22, (q15_t)0x1c, (q15_t)0x15, (q15_t)0xf, (q15_t)0x9, (q15_t)0x3
};

/**
 * @} end of DCT4_IDCT4_Table group
 */

/**
 * @addtogroup DCT4_IDCT4
 * @{
 */

/**
 * @brief  Initialization function for the Q15 DCT4/IDCT4.
 * @param[in,out] *S         points to an instance of Q15 DCT4/IDCT4 structure.
 * @param[in]     *S_RFFT    points to an instance of Q15 RFFT/RIFFT structure.
 * @param[in]     *S_CFFT    points to an instance of Q15 CFFT/CIFFT structure.
 * @param[in]     N          length of the DCT4.
 * @param[in]     Nby2       half of the length of the DCT4.
 * @param[in]     normalize  normalizing factor.
 * @return  	  arm_status function returns ARM_MATH_SUCCESS if initialization is successful or ARM_MATH_ARGUMENT_ERROR if <code>N</code> is not a supported transform length.
 * \par Normalizing factor:
 * The normalizing factor is <code>sqrt(2/N)</code>, which depends on the size of transform <code>N</code>.
 * Normalizing factors in 1.15 format are mentioned in the table below for different DCT sizes:
 * \image html dct4NormalizingQ15Table.gif
 */

arm_status arm_dct4_init_q15(
  arm_dct4_instance_q15 * S,
  arm_rfft_instance_q15 * S_RFFT,
  arm_cfft_radix4_instance_q15 * S_CFFT,
  uint16_t N,
  uint16_t Nby2,
  q15_t normalize)
{
  /*  Initialise the default arm status */
  arm_status status = ARM_MATH_SUCCESS;

  /* Initializing the pointer array with the weight table base addresses of different lengths */
  q15_t *twiddlePtr[4] = { (q15_t *) WeightsQ15_128, (q15_t *) WeightsQ15_512,
    (q15_t *) WeightsQ15_2048, (q15_t *) WeightsQ15_8192
  };

  /* Initializing the pointer array with the cos factor table base addresses of different lengths */
  q15_t *pCosFactor[4] =
    { (q15_t *) cos_factorsQ15_128, (q15_t *) cos_factorsQ15_512,
    (q15_t *) cos_factorsQ15_2048, (q15_t *) cos_factorsQ15_8192
  };

  /* Initialize the DCT4 length */
  S->N = N;

  /* Initialize the half of DCT4 length */
  S->Nby2 = Nby2;

  /* Initialize the DCT4 Normalizing factor */
  S->normalize = normalize;

  /* Initialize Real FFT Instance */
  S->pRfft = S_RFFT;

  /* Initialize Complex FFT Instance */
  S->pCfft = S_CFFT;

  switch (N)
  {
    /* Initialize the table modifier values */
  case 8192U:
    S->pTwiddle = twiddlePtr[3];
    S->pCosFactor = pCosFactor[3];
    break;
  case 2048U:
    S->pTwiddle = twiddlePtr[2];
    S->pCosFactor = pCosFactor[2];
    break;
  case 512U:
    S->pTwiddle = twiddlePtr[1];
    S->pCosFactor = pCosFactor[1];
    break;
  case 128U:
    S->pTwiddle = twiddlePtr[0];
    S->pCosFactor = pCosFactor[0];
    break;
  default:
    status = ARM_MATH_ARGUMENT_ERROR;
  }

  /* Initialize the RFFT/RIFFT */
  arm_rfft_init_q15(S->pRfft, S->N, 0U, 1U);

  /* return the status of DCT4 Init function */
  return (status);
}

/**
 * @} end of DCT4_IDCT4 group
 */

