//=================================================================
// The main program file.
//
// Copyright 2020 Georgia Tech.  All rights reserved.
// The materials provided by the instructor in this course are for
// the use of the students currently enrolled in the course.
// Copyrighted course materials may not be further disseminated.
// This file must not be made publicly available anywhere.
//==================================================================

// External libs
#include <stdlib.h>

// Project includes
#include "globals.h"
#include "hardware.h"
#include "compost_pile_public.h"
#include "fruit_public.h"
#include "player_public.h"
#include "doubly_linked_list.h"

//For sound components
////AnalogOut DACout(p18);
//PwmOut pwmout(p26);
////PwmOut speaker(p25);

////wave_player waver(&DACout);
//SDFileSystem sd(p5, p6, p7, p8, "sd"); // mosi, miso, sck, cs

DigitalOut myled1(LED1);
DigitalOut myled2(LED2);
DigitalOut myled3(LED3);
DigitalOut myled4(LED4);

// ===User implementations start===
bool touch(boundingBox knife, boundingBox fruit);
int fruit_contact(void); // iterate through fruits and see if any collided
void compost_pile_update(void);
void incScore(void);

void playSound(char * wav);
void playNotes(void);
void set_random_seed(Timer);
int tempScore = 0;
bool gameWon = false;
bool gameLost = false;
bool thrown = false;

int main()
{
    GameInputs inputs; 
    // First things first: initialize hardware
    ASSERT_P(hardware_init() == ERROR_NONE, "Hardware init failed!");
    pc.printf("Program Starting\n");

    // ===User implementations start===
    // Game state variables
    
    // Timer to measure game update speed (secondarily used to generate random seed)
    Timer t;
    int dt; // delta time
    set_random_seed(t);
 
    //pwmout.period(1.0/4000.0);
    //playSound("/sd/wavfiles/BUZZER.wav");//test the sound convert to thread
    //wait(0.1);
    //playNotes();
    
    //initialize functions
    compost_pile_init();
    fruit_init();
    player_init();        
    pc.printf("Initialization complete\n");
    uLCD.printf("SCORE:");
    uLCD.filled_rectangle(70, 0, 120, 8, RED);
    int is_bomb;
    
    while(!gameWon && !gameLost)
    {
        t.start();
        fruit_generator();
        player_draw(0x0000FF);
        player_knife_draw();   
        inputs = read_inputs();
        if (inputs.ay>=0.5) player_moveUp();            
        if (inputs.ay<= -0.5) player_moveDown();                
        if (inputs.b3) {
            player_throw();
        }
        knife_move();
        if (player_get_info().knife_boundingBox.topLeft.x >= 127) {
            player_knife_return();
        }
        compost_pile_update();
        is_bomb = fruit_contact();
        if (tempScore == 5) {
            gameWon = true;
            continue;
        }
        if (is_bomb == 1) {
            gameLost = true;
            continue;
        }
        if (get_compost_tallest_height() > 60) {
            gameLost = true;
            continue;
        }
        
        t.stop();
        dt = t.read_ms();
        if (dt < 100) wait_ms(100 - dt);
    }
    destroyList(get_fruit_list());
    
    if (gameLost) {
        uLCD.cls();
        uLCD.printf("YOU LOSE!\n\n\n\n\nRESTART TO \nPLAY AGAIN");
    }
    
    if (gameWon) {
        uLCD.cls();
        uLCD.printf("YOU WIN! \n\n\n\n\nRESTART TO \nPLAY AGAIN");
    }
    return 0;
}
    // ===User implementations end===

// ===User implementations start===
void incScore(void) {
    tempScore++;
    if (tempScore == 10) {
        gameWon = true;
        return;
    }
    uLCD.filled_rectangle(70, 0, tempScore*10 + 70, 8, GREEN);
}
    
    


bool touch(boundingBox knife, boundingBox fruit) {
    if (fruit.topLeft.x > knife.bottomRight.x+3){
        return false;
    } else if (fruit.topLeft.y > knife.bottomRight.y+3) {
        return false;
    } else if (knife.topLeft.x > fruit.bottomRight.x+3) {
        return false;
    } else if (knife.topLeft.y > fruit.bottomRight.y+3) {
        return false;
    }
    return true;
}
    
/* This function iterate through the fruit list, checking if anyone one of them is contacting 
*  with the player's knife. Don't forget to check if the "fruit" is the bomb, since we 
*  generated bombs using the fruit_generator. 
*/ 
int fruit_contact(void) {
    DLinkedList* fruitList = get_fruit_list();
    LLNode* curr = fruitList -> head;
    FRUIT* fruit;
    while (curr != NULL) {
        fruit = (FRUIT*)(curr->data);
        if ((touch(player_get_info().knife_boundingBox, fruit->box))&&(player_get_info().player_has_knife == 0)){
            if ((fruit->type) == 0) {
                return 1;
            }
            fruit->status = FRUIT_SLICED;
            player_knife_return();
            incScore();
            return 0;
        }
        curr = curr-> next;
    }
    return 0;
}

/** Call compost_pile_update() repeatedly in your game-loop. ex: main()
    This function iterate through the fruit list. Add fruits that reach the bottom of the screen
    to the compost pile.
    @return  Number of remaining cities. You might end the game when all cities are demolished.
*/
void compost_pile_update(void){
    DLinkedList* fruitList = get_fruit_list();
    LLNode* curr = fruitList -> head;
    FRUIT* fruit;
    int index;
    while (curr != NULL) {
        fruit = (FRUIT*)(curr->data);
        if ((fruit->box.topLeft.y > 127 - get_compost_tallest_height())&&(fruit->type != 0)) {
            index = (fruit->box.topLeft.x)/11;
            compost_add(index);
            fruit->status = FRUIT_SLICED;
        }
        curr = curr -> next;
    }
    draw_compost();
}

//fcn to play a wav
void playSound(char* wav)
{
    //open wav file
    FILE *wave_file;
    wave_file=fopen(wav,"r");
    
    if(wave_file != NULL) 
    {
        printf("File opened successfully\n");

        //play wav file
        printf("Sound playing...\n");
        waver.play(wave_file);
    
        //close wav file
        printf("Sound stopped...\n");
        fclose(wave_file);
        return;
    }
    
    printf("Could not open file for reading - %s\n", wav);
        return;
}

/* Wanna hear some annoying sounds?*/
void playNotes(void)
{
    int i;
// generate a 500Hz tone using PWM hardware output
    speaker.period(1.0/500.0); // 500hz period
    speaker =0.5; //50% duty cycle - max volume
    wait(3);
    speaker=0.0; // turn off audio
    wait(2);
// generate a short 150Hz tone using PWM hardware output
// something like this can be used for a button click effect for feedback
    for (i=0; i<10; i++) {
        speaker.period(1.0/150.0); // 500hz period
        speaker =0.25; //25% duty cycle - mid range volume
        wait(.02);
        speaker=0.0; // turn off audio
        wait(0.5);
    }

// sweep up in frequency by changing the PWM period
    for (i=0; i<8000; i=i+100) {
        speaker.period(1.0/float(i));
        speaker=0.25;
        wait(.1);
    }
    wait(2);

// two tone police siren effect -  two periods or two frequencies
// increase volume - by changing the PWM duty cycle
    for (i=0; i<26; i=i+2) {
        speaker.period(1.0/969.0);
        speaker = float(i)/50.0;
        wait(.5);
        speaker.period(1.0/800.0);
        wait(.5);
    }
    // decrease volume
    for (i=25; i>=0; i=i-2) {
        speaker.period(1.0/969.0);
        speaker = float(i)/50.0;
        wait(.5);
        speaker.period(1.0/800.0);
        wait(.5);
    }
    speaker =0.0;
    wait(2);
}

void set_random_seed(Timer t) {
    GameInputs inputs; 
    t.start();
    uLCD.printf("Push any button to start.\n");
    while(1){
      inputs = read_inputs();
      if (inputs.b1 || inputs.b2 || inputs.b3) break;
      }
    uLCD.cls();
    t.stop();
    int seed = t.read_ms();    
    srand(seed);
}

// ===User implementations end===
