/* Internal temperature sensor class. Adapted from AnalogIn.h, from:
 * mbed Microcontroller Library
 * Copyright (c) 2006-2013 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#ifndef INTERNAL_TEMP_H
#define INTERNAL_TEMP_H

#include "platform.h"

#if DEVICE_ANALOGIN

#include "analogin_api.h"

void temp_init(analogin_t *obj) {
    obj->adc = (ADCName)16;

    // Enable ADC1 clock source.
    RCC->APB2ENR |= RCC_APB2ENR_ADC1EN;

    // Enable the ADC
    ADC1->CR2 |= ADC_CR2_ADON;
    
    // Enable internal temperature sensor.
    ADC->CCR |= ADC_CCR_TSVREFE;
    
    // Set longest sample time just to be safe.
    ADC1->SMPR1 |= 0x1c;
}

static inline uint32_t temp_read(analogin_t *obj) {
    // Select the appropriate channel
    ADC1->SQR3 = (int) obj->adc;

    // Start conversion
    ADC1->CR2 |= ADC_CR2_SWSTART;

    // Wait for conversion to finish
    while (!(ADC1->SR & ADC_SR_EOC));

    uint32_t data = ADC1->DR;
    return data; // 12 bit
}

static inline uint32_t temp_read_u32(analogin_t *obj) {
    uint32_t value;
    value = temp_read(obj);
    return value;
}

uint16_t temperature_read_u16(analogin_t *obj) {
    uint32_t value = temp_read_u32(obj);

    return (value << 4) | ((value >> 8) & 0x000F); // 12 bit
}

float temperature_read(analogin_t *obj) {
    uint32_t value = temp_read_u32(obj);
    return (float)value * (1.0f / (float)0xFFF); //Check range
}

class InternalTemperature {

public:
    InternalTemperature() {
        temp_init(&_adc);
    }

    float read() {
        return temperature_read(&_adc);
    }

    unsigned short read_u16() {
        return temperature_read_u16(&_adc);
    }

#ifdef MBED_OPERATORS
    operator float() {
        return read();
    }
#endif

protected:
    analogin_t _adc;
};


#endif

#endif

