#include "SDFileSystem.h"
#include "MPU9250.h"
#include "InterruptManager.h"

#define BLENANO

/*      Pin Definitions     */
#ifdef BLENANO
    #define MOSI                p9
    #define MISO                p11 
    #define SCLK                p8 
    #define CS_SD               p10 
    #define CS_MPU              p29 
    #define BUTTON              p28
    #define MIC                 p5
    #define LOAD_CELL           p4
    #define LED                 p15
#else
    #define MOSI                p20
    #define MISO                p22 
    #define SCLK                p25 
    #define CS_SD               p23 
    #define CS_MPU              p21 
    #define BUTTON              p17
    #define MIC                 p5
    #define LOAD_CELL           p4
    #define LED                 p15
#endif

/*      Un/comment to determine which tests will be run     */
#define IMU_TEST
//#define MIC_TEST
//#define LOAD_TEST

/*      Parameters for sampling     */
#define FILE_NAME           "trick"
#define SAMPLE_RATE         1000       

/*      Objects used        */
DigitalOut      led(LED);
AnalogIn        mic(MIC);
InterruptIn     button(BUTTON);
Ticker          timer;
SPI             spi(MOSI, MISO, SCLK); 
SDFileSystem    sd(MOSI, MISO, SCLK, CS_SD, "sd");
mpu9250_spi     imu(spi , CS_MPU);

/*      Flags for sampling      */
#define IMU_FLAG            (1<<1U)
#define MIC_FLAG            (1<<2U)
#define CREATE_FILE         (1<<3U)
#define CLOSE_FILE          (1<<4U)

/*      Status      */
#define START_SAMPLE        1
#define END_SAMPLE          0

static uint16_t status = 0;
static int button_status = START_SAMPLE;   
static uint16_t file_count = 0;      

static float sample_rate = (1/(float)SAMPLE_RATE) ;     

static FILE *fp = NULL;
static char file_name[17];

static void openNextFile(void)
{

    if( fp ){
        fclose(fp);    
    }
    
    snprintf(file_name, sizeof(file_name), "/sd/%s_%02d.txt", FILE_NAME, file_count++); 
    fp = fopen(file_name, "w");
    fprintf(fp, "Sample Rate: %d Hz\n", SAMPLE_RATE);
    
}

static void button_interrupt(void)
{
    
    //CREATE FILE
    if ( button_status == START_SAMPLE ){
        status |= CREATE_FILE;
        button_status = END_SAMPLE;
        
    //CLOSE FILE    
    } else if ( button_status == END_SAMPLE ){
        status |= CLOSE_FILE;
        button_status = START_SAMPLE;
    }

}

static void timer_interrupt(void)
{
    
    if ( button_status == END_SAMPLE ){
        //ENABLE SAMPLE FLAGS
        #ifdef IMU_TEST
            status |= IMU_FLAG;
        #endif
        
        #ifdef MIC_TEST
            status |= MIC_FLAG;
        #endif
        
        #ifdef LOAD_TEST
            status |= LOAD_FLAG;
        #endif
    }
    
}

int main()
{

    #ifdef IMU_TEST
    //initialize imu
    imu.init(1, BITS_DLPF_CFG_2100HZ_NOLPF);
    #endif
    
    //set up button as active low to trigger file creation
    button.fall(&button_interrupt);
    timer.attach(&timer_interrupt, sample_rate );
    led.write(0);
    
    snprintf(file_name, sizeof(file_name), "/sd/%s_%02d.txt", FILE_NAME, file_count++); 
    fp = fopen(file_name, "w");
    fprintf(fp, "Sample Rate: %d Hz\n", SAMPLE_RATE);
    
    while(1){
        
        if ( status & CREATE_FILE ){
            led.write(1);    
            openNextFile();
            status ^= CREATE_FILE;
        }
        
        #ifdef IMU_TEST
        if ( status & IMU_FLAG ){
            led.write(!led.read());
            imu.read_all();
            
            #ifndef MIC_TEST
            fprintf(fp, "%5d\t%5d\t%5d\t%5d\t%5d\t%5d\t%5d\t%5d\t%5d\n", (signed int) imu.gyroscope_data[0],
                                                                         (signed int) imu.gyroscope_data[1],
                                                                         (signed int) imu.gyroscope_data[2],
                                                                         (signed int) imu.accelerometer_data[0],
                                                                         (signed int) imu.accelerometer_data[1],
                                                                         (signed int) imu.accelerometer_data[2],
                                                                         (signed int) imu.Magnetometer[0],
                                                                         (signed int) imu.Magnetometer[1],
                                                                         (signed int) imu.Magnetometer[2] );
            #endif
            
            status ^= IMU_FLAG;
        }
        #endif
        
        #ifdef MIC_TEST
        if ( status & MIC_FLAG ){
            led.write(!led.read());
            
            #ifndef IMU_TEST
            fprintf(fp, "%d\n", mic.read_u16());
            
            #else
            fprintf(fp, "%5d\t%5d\t%5d\t%5d\t%5d\t%5d\t%5d\t%5d\t%5d\t%4d\n", (signed int) imu.gyroscope_data[0],
                                                                              (signed int) imu.gyroscope_data[1],
                                                                              (signed int) imu.gyroscope_data[2],
                                                                              (signed int) imu.accelerometer_data[0],
                                                                              (signed int) imu.accelerometer_data[1],
                                                                              (signed int) imu.accelerometer_data[2],
                                                                              (signed int) imu.Magnetometer[0],
                                                                              (signed int) imu.Magnetometer[1],
                                                                              (signed int) imu.Magnetometer[2],
                                                                              mic.read_u16() );
            #endif
            
            status ^= MIC_FLAG;
            
        }
        #endif
        
        if ( status & CLOSE_FILE ){
            led.write(0);
            fclose(fp);
            status ^= CLOSE_FILE;
        }
        
    }
    
}

