#include "mbed.h"
#include "nRF2401LP.h"

SPI spi_2(p5, p6, p7); // mosi, miso, sclk
SPI spi_1(p11, p12, p13); // mosi, miso, sclk

DigitalOut csn_2(p8);
DigitalOut ce_2(p9);

DigitalOut csn_1(p14);
DigitalOut ce_1(p15);

LocalFileSystem local("local");  
FILE *fp;

char * regNames[] = 
{
  "CONFIG",
  "EN_AA",
  "EN_RXADDR",
  "SETUP_AW",
  "SETUP_RETR",
  "RF_CH",
  "RF_SETUP",
  "STATUS",
  "OBSERVE_TX",
  "RPD",
  "RX_ADDR_P0",
  "RX_ADDR_P1",
  "RX_ADDR_P2",
  "RX_ADDR_P3",
  "RX_ADDR_P4",
  "RX_ADDR_P5",
  "TX_ADDR",
  "RX_PW_P0",
  "RX_PW_P1",
  "RX_PW_P2",
  "RX_PW_P3",
  "RX_PW_P4",
  "RX_PW_P5",
  "FIFO_STATUS"
  };
 

void readReg(SPI *spi_p, DigitalOut *csn_p, int regIdx, int byteCnt, unsigned int *reg_p);

void writeReg(SPI *spi_p, DigitalOut *csn_p, int regIdx, int byteCnt, unsigned int *reg_p)
{
    int idx;
    // Executable in power down or standby modes only.
    
    (*csn_p) = 0;
    wait_ms(1);
    (*csn_p) = 1;
    wait_ms(1);
    (*csn_p) = 0;
    wait_ms(1);
    
    spi_p->write(NRF24L01P_SPI_CMD_WR_REG | regIdx);
    for (idx=0; idx<byteCnt; idx++)
    {
         wait_ms(1);
         fprintf(fp, "\n <%s> : Writing 0x%x to register <%d> \n", 
                 __FUNCTION__, reg_p[idx], regIdx);
         spi_p->write(reg_p[idx]);
    }
    
    wait_ms(1);
    
    // Deselect the device
    (*csn_p) = 1;      
}    

int selectChannel(SPI *spi_p, DigitalOut *csn_p, int chnIdx)
{
   unsigned int byte = chnIdx & 0x7f;
   
   fprintf(fp, "\n <%s> : Chan<%d>/<%d MHZ> \n", __FUNCTION__, chnIdx, 2400 + chnIdx);
   
   // The RF channel frequency is set by the RF_CH register according to the following 
   // formula:   F0= 2400 + RF_CH    [In MHz]
   // The programming resolution of the RF channel frequency setting is 1MHz.
   
   writeReg(spi_p, csn_p, NRF24L01P_REG_RF_CH, 1, &byte); 
   
   return 0;
}


int selectDataRate(SPI *spi_p, DigitalOut *csn_p, int dataRateKbps)
{
   unsigned int regVal;
   
   // The air data rate is set by the RF_DR bit in the RF_SETUP register. A transmitter 
   // and a receiver must be programmed with the same air data rate to communicate with 
   // each other.
   
   readReg(spi_p, csn_p, NRF24L01P_REG_RF_SETUP, 1, &regVal);
   
   switch (dataRateKbps)
   {           
      case NRF24L01P_DR_1_MBPS:     
           fprintf(fp, "\n 1000 Kbps data rate selected ... \n");
           // Set RF_DR_LOW to 0 and RF_DR_HIGH to 0
           regVal &= ~(NRF24L01P_RF_SETUP_REG_DR_LOW_BIT | NRF24L01P_RF_SETUP_REG_DR_HIGH_BIT);
           break;
           
      case NRF24L01P_DR_2_MBPS:        
           fprintf(fp, "\n 2000 Kbps data rate selected ... \n");
           // Set RF_DR_LOW to 0 and RF_DR_HIGH to 1
           regVal |=  NRF24L01P_RF_SETUP_REG_DR_HIGH_BIT; 
           regVal &= ~NRF24L01P_RF_SETUP_REG_DR_LOW_BIT;
           break;
       
      case NRF24L01P_DR_250_KBPS:         
      default:
           fprintf(fp, "\n 250 Kbps data rate selected ... \n");
           // Set RF_DR_LOW to 1 and RF_DR_HIGH to 0
           regVal |=  NRF24L01P_RF_SETUP_REG_DR_LOW_BIT;   
           regVal &= ~NRF24L01P_RF_SETUP_REG_DR_HIGH_BIT;        
           break;         
   }

   writeReg(spi_p, csn_p, NRF24L01P_REG_RF_SETUP, 1, &regVal); 
   
   return 1;
}



int selectTxPower(SPI *spi_p, DigitalOut *csn_p, int txPower)
{
   unsigned int regVal;
   
   readReg(spi_p, csn_p, NRF24L01P_REG_RF_SETUP, 1, &regVal);

   regVal &= ~(REG_BIT_1 | REG_BIT_2);
   switch (txPower)
   {
      case NRF24L01P_TX_PWR_MINUS_18_DB:           
           break;

      case NRF24L01P_TX_PWR_MINUS_12_DB:
           regVal |= REG_BIT_1;
           break;

      case NRF24L01P_TX_PWR_MINUS_6_DB:
           regVal |= REG_BIT_2;
           break;

      case NRF24L01P_TX_PWR_ZERO_DB:     
           regVal |= REG_BIT_1;
           regVal |= REG_BIT_2;
      default:
           break;         
   }

   writeReg(spi_p, csn_p, NRF24L01P_REG_RF_SETUP, 1, &regVal); 
   
   return 1;
}

int setRadioOpnMode(SPI *spi_p, DigitalOut *csn_p, int radioMode)
{
   unsigned int regVal;

   readReg(spi_p, csn_p, NRF24L01P_REG_CONFIG, 1, &regVal);
  
   switch (radioMode)
   {
      case NRF24L01P_MODE_POWER_DOWN:
           regVal &= ~REG_BIT_1;
           break;

      case NRF24L01P_MODE_STANDBY:  // POWER UP
           regVal |= REG_BIT_1;
           break;
      
      case NRF24L01P_MODE_TX:
           regVal &= ~REG_BIT_0;
           break;

      case NRF24L01P_MODE_RX:
           regVal |= REG_BIT_0;
           break;

      default:
           return 0;
   }

   writeReg(spi_p, csn_p, NRF24L01P_REG_CONFIG, 1, &regVal);

   return 1;
}

int setUpPktRx(SPI *spi_p, DigitalOut *csn_p, DigitalOut *ce_p)
{
    unsigned int regVal;
    unsigned int nodeAddr[5] = {0x15, 0x26, 0x37, 0x48, 0x59}; 
  
    // Select RX by setting the PRIM_RX bit in the CONFIG register to high. 
   
    readReg(spi_p, csn_p, NRF24L01P_REG_STATUS, 1, &regVal); 
    fprintf(fp, "\n <%s> : Status <0x%x> \n", __FUNCTION__, regVal);

    readReg(spi_p, csn_p, NRF24L01P_REG_FIFO_STATUS, 1, &regVal); 
    fprintf(fp, "\n <%s> : FIFO Status <0x%x> \n", __FUNCTION__, regVal); 
    
    // writeReg(spi_p, csn_p, NRF24L01P_REG_TX_ADDR, 5, nodeAddr);
 
    (*ce_p) = 0;
    
    regVal = 32;
    writeReg(spi_p, csn_p, NRF24L01P_REG_RX_PW_P0, 1, &regVal);
    writeReg(spi_p, csn_p, NRF24L01P_REG_RX_PW_P1, 1, &regVal);
    writeReg(spi_p, csn_p, NRF24L01P_REG_RX_PW_P2, 1, &regVal);
    writeReg(spi_p, csn_p, NRF24L01P_REG_RX_PW_P3, 1, &regVal);
    writeReg(spi_p, csn_p, NRF24L01P_REG_RX_PW_P4, 1, &regVal);
    
    wait_ms(100); 
    
    setRadioOpnMode(spi_p, csn_p, NRF24L01P_MODE_STANDBY);
  
    wait_ms(100);    
  
    // Set config bit PRIM_RX high
    setRadioOpnMode(spi_p, csn_p, NRF24L01P_MODE_RX);

    wait_ms(100);

    readReg(spi_p, csn_p, NRF24L01P_REG_CONFIG, 1, &regVal); 
    fprintf(fp, "\n <%s> : Config <0x%x> \n", __FUNCTION__, regVal);  
    
    regVal = 0x3f;
    writeReg(spi_p, csn_p, NRF24L01P_REG_EN_RX_ADDR, 1, &regVal);
    // Start Active RX mode by setting CE high. After 130s nRF24L01+ monitors the 
   // air for incoming communication.   
    wait_ms(100);
    (*ce_p) = 1;
    
    wait_ms(100);
    
    readReg(spi_p, csn_p, NRF24L01P_REG_STATUS, 1, &regVal); 
    fprintf(fp, "\n <%s> : Status <0x%x> \n", __FUNCTION__, regVal);

    readReg(spi_p, csn_p, NRF24L01P_REG_FIFO_STATUS, 1, &regVal); 
    fprintf(fp, "\n <%s> : FIFO Status <0x%x> \n", __FUNCTION__, regVal); 
     
    return 0; 
}

int txPacket(SPI *spi_p, DigitalOut *csn_p, DigitalOut *ce_p, unsigned int *txBuff, unsigned int pktLen)
{
   unsigned int regVal;

    /*
     * The TX mode is an active mode for transmitting packets. To enter this mode, 
     * the nRF24L01+ must have the PWR_UP bit set high, PRIM_RX bit set low, a 
     * payload in the TX FIFO and a high pulse on the CE for more than 10s.
     */

    /*
     * You can write to the TX FIFO using these three commands; 
     * W_TX_PAYLOAD and W_TX_PAYLOAD_NO_ACK in PTX mode and W_ACK_PAYLOAD in PRX mode.
     */

    readReg(spi_p, csn_p, NRF24L01P_REG_STATUS, 1, &regVal); 
    fprintf(fp, "\n <%s> : Status <0x%x> \n", __FUNCTION__, regVal);

    readReg(spi_p, csn_p, NRF24L01P_REG_FIFO_STATUS, 1, &regVal); 
    fprintf(fp, "\n <%s> : Status <0x%x> \n", __FUNCTION__, regVal);


    (*ce_p) = 0;
   
    setRadioOpnMode(spi_p, csn_p, NRF24L01P_MODE_STANDBY);
 
    wait_ms(10);

    // Set config bit PRIM_RX low
    setRadioOpnMode(spi_p, csn_p, NRF24L01P_MODE_TX);

    wait_ms(10);
    
    selectTxPower(spi_p, csn_p, NRF24L01P_TX_PWR_ZERO_DB);

    readReg(spi_p, csn_p, NRF24L01P_REG_CONFIG, 1, &regVal); 
    fprintf(fp, "\n <%s> : Config <0x%x> \n", __FUNCTION__, regVal);
    
    // Now clock the payload into the nRF24L01+
    
    writeReg(spi_p, csn_p, NRF24L01P_SPI_CMD_WR_TX_PYLD, pktLen, txBuff); 
    
    // A high pulse on CE starts the transmission. The minimum pulse width on CE is 10s.

    readReg(spi_p, csn_p, NRF24L01P_REG_STATUS, 1, &regVal); 
    fprintf(fp, "\n <%s> : Status <0x%x> \n", __FUNCTION__, regVal);

    readReg(spi_p, csn_p, NRF24L01P_REG_FIFO_STATUS, 1, &regVal); 
    fprintf(fp, "\n <%s> : FIFO Status <0x%x> \n", __FUNCTION__, regVal);

    (*ce_p) = 1;
 
    wait_ms(1);

    (*ce_p) = 0;

    wait_ms(1);

    readReg(spi_p, csn_p, NRF24L01P_REG_STATUS, 1, &regVal); 
    fprintf(fp, "\n <%s> : Status <0x%x> \n", __FUNCTION__, regVal);

    readReg(spi_p, csn_p, NRF24L01P_REG_FIFO_STATUS, 1, &regVal); 
    fprintf(fp, "\n <%s> : FIFO Status <0x%x> \n", __FUNCTION__, regVal);

    return 0;
}


int getSelectedChannel(SPI *spi_p, DigitalOut *csn_p)
{
    unsigned int byte;
    readReg(spi_p, csn_p, NRF24L01P_REG_RF_CH, 1, &byte);
    return byte;
}

void readReg( SPI *spi_p, DigitalOut *csn_p, int regIdx, int byteCnt, unsigned int *reg_p)
{
    int idx;
    
    (*csn_p) = 0;
    wait_ms(1);
    (*csn_p) = 1;
    wait_ms(1);
    (*csn_p) = 0;
    wait_ms(1);


    spi_p->write(NRF24L01P_SPI_CMD_RD_REG | regIdx);
 
    for (idx=0; idx<byteCnt; idx++)
    {
        // Send a dummy byte to receive the contents of the register
        reg_p[idx] = spi_p->write(0x00);
    }
    
    wait_ms(1);
    
    // Deselect the device
    (*csn_p) = 1;   
}

int main() 
{
  
    unsigned int byte, idx;
    int regIdx;
    unsigned int spi_1_rxAddrP0[RF24L01P_RX_ADDR_REG_LEN];
    unsigned int spi_2_rxAddrP0[RF24L01P_RX_ADDR_REG_LEN];
    unsigned int spi_1_txAddr[RF24L01P_RX_ADDR_REG_LEN];
    unsigned int spi_2_txAddr[RF24L01P_RX_ADDR_REG_LEN];
    unsigned int txBuff[32] = {0x12, 0x34, 0x56, 0x78};
    unsigned int rxBuff[32];
    unsigned int addr[5];

    for (idx=0; idx<10; idx++)
    {
       ce_1 = 0x1;
       ce_2 = 0x1;
       csn_1 = 0x1;
       csn_2 = 0x1;
       wait_ms(100);
       ce_1 = 0x0;
       ce_2 = 0x0;
       csn_1 = 0x0;
       csn_2 = 0x0;
       wait_ms(100);
    }

    csn_1 = 0x1;
    csn_2 = 0x1;
    ce_1 = 0x0;
    ce_2 = 0x0;

    fp = fopen("/local/spi.dat", "w");  
    if (fp == NULL)
        return 1;
        
    ce_1 = 0x0;
    ce_2 = 0x0;        
        
    // Setup the spi for 8 bit data, high steady state clock,
    // second edge capture, with a 1kHz clock rate
    spi_1.format(8, 0);   // CPOL 0 and CPHA 0    
    spi_1.frequency(1000000);
   
    spi_2.format(8, 0);   // CPOL 0 and CPHA 0    
    spi_2.frequency(1000000);   
    
    readReg(&spi_1, &ce_1, NRF24L01P_REG_RF_SETUP, 1, &byte);
    fprintf(fp, "\n SPI1 :  RF Seup : Value<0x%x> \n", (unsigned int)byte);
    
    selectChannel(&spi_1, &csn_1, 15);
    selectChannel(&spi_2, &csn_2, 15);

    byte = 0x0;
    writeReg(&spi_1, &csn_1, NRF24L01P_REG_EN_AA, 1, &byte); 
    writeReg(&spi_2, &csn_2, NRF24L01P_REG_EN_AA, 1, &byte); 
   
    readReg(&spi_2, &csn_2, NRF24L01P_REG_RX_ADDR_P0, 5, &addr[0]);
    for (idx=0; idx<5; idx++)
         fprintf(fp, "\n SP1_2 : RX_ADDR_P0 <%d> : <0x%x> ", idx, addr[idx]); 
         
    readReg(&spi_1, &csn_1, NRF24L01P_REG_TX_ADDR, 5, &addr[0]);
    for (idx=0; idx<5; idx++)
         fprintf(fp, "\n SP1_1 : TX_ADDR <%d> : <0x%x> ", idx, addr[idx]); 
            
    for (regIdx = NRF24L01P_REG_CONFIG; regIdx <= NRF24L01P_REG_FIFO_STATUS; regIdx++)
    {
         readReg(&spi_1, &csn_1, regIdx, 1, &byte);
         fprintf(fp, "\n <1> SPI_1 : Reg<%d>/<%s> : Value<0x%x> \n", regIdx, regNames[regIdx], (unsigned int)byte);
    }
        
    readReg(&spi_1, &csn_1, NRF24L01P_REG_DYNPD, 1, &byte);
    fprintf(fp, "\n <1> SPI_1 : Reg<%d> : Value<0x%x> \n", NRF24L01P_REG_DYNPD, (unsigned int)byte);
    
    readReg(&spi_1, &csn_1, NRF24L01P_REG_FEATURE, 1, &byte);
    fprintf(fp, "\n <1> SPI_1 : Reg<%d> : Value<0x%x> \n", NRF24L01P_REG_FEATURE, (unsigned int)byte);    

    for (regIdx = NRF24L01P_REG_CONFIG; regIdx <= NRF24L01P_REG_FIFO_STATUS; regIdx++)
    {
         readReg(&spi_2, &csn_2, regIdx, 1, &byte);
         fprintf(fp, "\n <1> SPI_2 : Reg<%d>/<%s> : Value<0x%x> \n", regIdx, regNames[regIdx], (unsigned int)byte);
    }        
   
    readReg(&spi_2, &csn_2, NRF24L01P_REG_DYNPD, 1, &byte);
    fprintf(fp, "\n <1> SPI_2 : Reg<%d> : Value<0x%x> \n", NRF24L01P_REG_DYNPD, (unsigned int)byte);
    
    readReg(&spi_2, &csn_2, NRF24L01P_REG_FEATURE, 1, &byte);
    fprintf(fp, "\n <1> SPI_2 : Reg<%d> : Value<0x%x> \n", NRF24L01P_REG_FEATURE, (unsigned int)byte);    

   
    
    selectDataRate(&spi_1, &csn_1, NRF24L01P_DR_250_KBPS); 
    selectDataRate(&spi_2, &csn_2, NRF24L01P_DR_250_KBPS);
    
    setUpPktRx(&spi_2, &csn_2, &ce_2);
    
    txPacket(&spi_1, &csn_1, &ce_1, txBuff, 32);
    
    wait_ms(1000);
    
    // ce_2 = 0;
    
#if 0    
    readReg(&spi_2, &csn_2, NRF24L01P_RE_RX_ADDR_P0, 5, &spi_2_rxAddrP0[0]);
    for (idx=0; idx<RF24L01P_RX_ADDR_REG_LEN; idx++)
    {
         fprintf(fp, "\n SPI_2 : Reg<%d>/Idx<%d> : Value<0x%x> \n", 
                 NRF24L01P_REG_RX_ADDR_P0, idx, (unsigned int)spi_1_rxAddrP0[idx]);
    }
    
    readReg(&spi_2, &csn_2, NRF24L01P_REG_RX_ADDR_P0, 5, &spi_2_rxAddrP0[0]);
    for (idx=0; idx<RF24L01P_RX_ADDR_REG_LEN; idx++)
    {
         fprintf(fp, "\n SPI_2 : Reg<%d>/Idx<%d> : Value<0x%x> \n", 
                 NRF24L01P_REG_RX_ADDR_P0, idx, (unsigned int)spi_2_rxAddrP0[idx]);
    }
    
    readReg(&spi_1, &csn_1, NRF24L01P_REG_TX_ADDR, 5, &spi_1_txAddr[0]);
    for (idx=0; idx<RF24L01P_TX_ADDR_REG_LEN; idx++)
    {
         fprintf(fp, "\n SPI_1 : Reg<%d>/Idx<%d> : Value<0x%x> \n", 
                 NRF24L01P_REG_TX_ADDR, idx, (unsigned int)spi_2_rxAddrP0[idx]);
    }
    
    readReg(&spi_2, &csn_2, NRF24L01P_REG_TX_ADDR, 5, &spi_2_txAddr[0]);
    for (idx=0; idx<RF24L01P_TX_ADDR_REG_LEN; idx++)
    {
         fprintf(fp, "\n SPI_2 : Reg<%d>/Idx<%d> : Value<0x%x> \n", 
                 NRF24L01P_REG_TX_ADDR, idx, (unsigned int)spi_2_rxAddrP0[idx]);
    }    
#endif

    for (regIdx = NRF24L01P_REG_CONFIG; regIdx <= NRF24L01P_REG_FIFO_STATUS; regIdx++)
    {
         readReg(&spi_1, &csn_1, regIdx, 1, &byte);
         fprintf(fp, "\n <1> SPI_1 : Reg<%d>/<%s> : Value<0x%x> \n", regIdx, regNames[regIdx], (unsigned int)byte);
    }
    
    for (regIdx = NRF24L01P_REG_CONFIG; regIdx <= NRF24L01P_REG_FIFO_STATUS; regIdx++)
    {
         readReg(&spi_2, &csn_2, regIdx, 1, &byte);
         fprintf(fp, "\n <1> SPI_2 : Reg<%d>/<%s> : Value<0x%x> \n", regIdx, regNames[regIdx], (unsigned int)byte);
    }        
  

#if 0  
    readReg(&spi_1, &csn_1, NRF24L01P_REG_RF_CH, 1, &byte);
    fprintf(fp, "\n SPI_1 : Reg<%d> : Value<0x%x> \n", NRF24L01P_REG_RF_CH, (unsigned int)byte); 
    selectChannel(&spi_1, &csn_1, 5);
    readReg(&spi_1, &csn_1, NRF24L01P_REG_RF_CH, 1, &byte);
    fprintf(fp, "\n SPI_1 : Reg<%d> : Value<0x%x> \n", NRF24L01P_REG_RF_CH, (unsigned int)byte); 
    
    readReg(&spi_1, &csn_1, NRF24L01P_REG_RF_SETUP, 1, &byte);
    fprintf(fp, "\n SPI_1 : Reg<%d> : Value<0x%x> \n", NRF24L01P_REG_RF_SETUP, (unsigned int)byte); 
    selectDataRate(&spi_1, &csn_1, NRF24L01P_DR_250_KBPS);
    readReg(&spi_1, &csn_1, NRF24L01P_REG_RF_SETUP, 1, &byte);
    fprintf(fp, "\n SPI_1 : Reg<%d> : Value<0x%x> \n", NRF24L01P_REG_RF_SETUP, (unsigned int)byte); 
       
    fprintf(fp, "\n SPI_1 : Selected Channel <%d> \n", getSelectedChannel(&spi_1, &csn_1)); 
    
    fprintf(fp, "\n SPI_2 : Selected Channel <%d> \n", getSelectedChannel(&spi_2, &csn_2)); 
 #endif
    
 #if 0
    for (regIdx = NRF24L01P_REG_CONFIG; regIdx <= NRF24L01P_REG_FIFO_STATUS; regIdx++)
    {
    // Every new command must be started by a high to low transition on CSN.

    // Select the device by seting chip select low

    csn_1 = 0;
    wait_ms(1);
    csn_1 = 1;
    wait_ms(1);
    csn_1 = 0;
    wait_ms(1);

    spi_1.write(NRF24L01P_SPI_CMD_RD_REG | regIdx);

    // Send a dummy byte to receive the contents of the register
    byte = spi_1.write(0x00);
   
    wait_ms(1);
    
    // Deselect the device
    csn_1 = 1;
    

    fprintf(fp, "\n SPI_1 : Reg<%d> : Value<0x%x> \n", regIdx, (unsigned int)byte);
    }
    
    spi_2.format(8, 0);   // CPOL 0 and CPHA 0
    
    spi_2.frequency(1000000);
    
    for (regIdx = NRF24L01P_REG_CONFIG; regIdx <= NRF24L01P_REG_FIFO_STATUS; regIdx++)
    {
    // Every new command must be started by a high to low transition on CSN.

    // Select the device by seting chip select low

    csn_2 = 0;
    wait_ms(1);
    csn_2 = 1;
    wait_ms(1);
    csn_2 = 0;
    wait_ms(1);

    spi_2.write(NRF24L01P_SPI_CMD_RD_REG | regIdx);

    // Send a dummy byte to receive the contents of the register
    byte = spi_2.write(0x00);
   
    wait_ms(1);
    
    // Deselect the device
    csn_2 = 1;
    

    fprintf(fp, "\n SPI_2 : Reg<%d> : Value<0x%x> \n", regIdx, (unsigned int)byte);
    }
 #endif
     
    fclose(fp);
}

