/*
    FreeRTOS V6.0.3 - Copyright (C) 2010 Real Time Engineers Ltd.

    ***************************************************************************
    *                                                                         *
    * If you are:                                                             *
    *                                                                         *
    *    + New to FreeRTOS,                                                   *
    *    + Wanting to learn FreeRTOS or multitasking in general quickly       *
    *    + Looking for basic training,                                        *
    *    + Wanting to improve your FreeRTOS skills and productivity           *
    *                                                                         *
    * then take a look at the FreeRTOS eBook                                  *
    *                                                                         *
    *        "Using the FreeRTOS Real Time Kernel - a Practical Guide"        *
    *                  http://www.FreeRTOS.org/Documentation                  *
    *                                                                         *
    * A pdf reference manual is also available.  Both are usually delivered   *
    * to your inbox within 20 minutes to two hours when purchased between 8am *
    * and 8pm GMT (although please allow up to 24 hours in case of            *
    * exceptional circumstances).  Thank you for your support!                *
    *                                                                         *
    ***************************************************************************

    This file is part of the FreeRTOS distribution.

    FreeRTOS is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License (version 2) as published by the Free Software Foundation AND MODIFIED BY the FreeRTOS exception.
    ***NOTE*** The exception to the GPL is included to allow you to distribute a combined work that includes FreeRTOS without being obliged to provide the
    source code for proprietary components outside of the FreeRTOS kernel.
    FreeRTOS is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
    more details. You should have received a copy of the GNU General Public License and the FreeRTOS license exception along with FreeRTOS; if not it can be viewed here: http://www.freertos.org/a00114.html and also obtained
    by writing to Richard Barry, contact details for whom are available on the FreeRTOS WEB site.

    1 tab == 4 spaces!

    http://www.FreeRTOS.org - Documentation, latest information, license and contact details.

    http://www.SafeRTOS.com - A version that is certified for use in safety critical systems.

    http://www.OpenRTOS.com - Commercial support, development, porting, licensing and training services.
*/

//#error The batch file Demo\CORTEX_LPC1768_GCC_RedSuite\CreateProjectDirectoryStructure.bat must be executed before the first build.  After executing the batch file hit F5 to refrech the Eclipse project, then delete this line.

/*
 * Creates all the demo application tasks, then starts the scheduler.  The WEB documentation provides more details of the standard demo application tasks
 * (which just exist to test the kernel port and provide an example of how to use each FreeRTOS API function).
 *
 * In addition to the standard demo tasks, the following tasks and tests are defined and/or created within this file:
 *
 * "Check" hook -  This only executes fully every five seconds from the tick hook.  Its main function is to check that all the standard demo tasks are
 * still operational.  The status can be viewed using on the Task Stats page served by the WEB server.
 *
 * "uIP" task -  This is the task that handles the uIP stack.  All TCP/IP processing is performed in this task.
 * 
 * "USB" task - Enumerates the USB device as a CDC class, then echoes back all received characters with a configurable offset (for example, if the offset
 * is 1 and 'A' is received then 'B' will be sent back).  A dumb terminal such as Hyperterminal can be used to talk to the USB task.
 */
/*----------------------------------------------------------------------------------------------------------*/
/*
 *  Modified for mbed IDE development environment
 *  By Kenji Arai / JH1PJL on October 28th,2010
 *      October 28th,2010
 */

#include <stdio.h>

#include "FreeRTOS.h"
#include "task.h"

#include "integer.h"
#include "BlockQ.h"
#include "blocktim.h"
#include "flash.h"
#include "partest.h"
#include "semtest.h"
#include "PollQ.h"
#include "GenQTest.h"
#include "QPeek.h"
#include "queue.h"
#include "recmutex.h"



/* Task priorities. */
#define mainQUEUE_POLL_PRIORITY                ( tskIDLE_PRIORITY + 2 )
#define mainSEM_TEST_PRIORITY                ( tskIDLE_PRIORITY + 1 )
#define mainBLOCK_Q_PRIORITY                ( tskIDLE_PRIORITY + 2 )
#define mainUIP_TASK_PRIORITY                ( tskIDLE_PRIORITY + 3 )
#define mainINTEGER_TASK_PRIORITY           ( tskIDLE_PRIORITY )
#define mainGEN_QUEUE_TASK_PRIORITY            ( tskIDLE_PRIORITY )
#define mainFLASH_TASK_PRIORITY                ( tskIDLE_PRIORITY + 2 )

/* LED */
#define LED1 0
#define LED2 1
#define LED3 2
#define LED4 3

// FLASH
#define FLASH_SETUP           1
#define FLASHCFG_Val          0x0000303A

// Define clocks
#define XTAL        (12000000UL)        /* Oscillator frequency               */
#define OSC_CLK     (      XTAL)        /* Main oscillator frequency          */
#define RTC_CLK     (   32000UL)        /* RTC oscillator frequency           */
#define IRC_OSC     ( 4000000UL)        /* Internal RC oscillator frequency   */

/*----------------------------------------------------------------------------------------------------------*/
// Configure the hardware for mbed board
static void prvSetupHardware( void );
static void prvSetupSystem( void );

//Control tasks for JH1PJL
void vTask1( void *pvParameters );
void vTask2( void *pvParameters );
void vTask3( void *pvParameters );
void vTask4( void *pvParameters );
void vTask5( void *pvParameters );
void vTask6( void *pvParameters );
void vTask7( void *pvParameters );

/*----------------------------------------------------------------------------------------------------------*/
uint32_t SystemFrequency;    /*!< System Clock Frequency (Core Clock)  */

unsigned portBASE_TYPE uxHiWtrMrk_tsk1, uxHiWtrMrk_tsk2;
unsigned portBASE_TYPE uxHiWtrMrk_tsk3, uxHiWtrMrk_tsk4, uxHiWtrMrk_tsk5;
unsigned portBASE_TYPE uxHiWtrMrk_tsk6, uxHiWtrMrk_tsk7;

////////////////////////////// Constant Data /////////////////////////////////////////////////////////////////
/* ---------------------------< Copyright >---------------------------------------------------------------- */
const uint8_t copyright[]    = "Arai,Kenji / JH1PJL(c)2010 kenjia@sannet.ne.jp "__DATE__" (" __TIME__ ")";
#if ( USE_XPRESSO == 1 )
/* ---------------------------< Board >---------------------------------------------------------------- */
const uint8_t board[]        = "LPCXpresso LPC1768";
/*----------------------------------------------------------------------------------------------------------*/
#elif ( USE_MBED == 1 )
/* ---------------------------< Board >---------------------------------------------------------------- */
const uint8_t board[]        = "med LPC1768";
#else
/* ---------------------------< Board >---------------------------------------------------------------- */
const uint8_t board[]        = "No identification";
#endif

int main( void ){
//    char cIPAddress[ 16 ]; 
    /* Configure the hardware for mbed board */
    prvSetupHardware();
    prvSetupSystem();

    /* Start the standard demo tasks.  These are just here to exercise the kernel port and provide examples of how the FreeRTOS API can be used. */
    vStartBlockingQueueTasks( mainBLOCK_Q_PRIORITY );
    vCreateBlockTimeTasks();
    vStartSemaphoreTasks( mainSEM_TEST_PRIORITY );
    vStartPolledQueueTasks( mainQUEUE_POLL_PRIORITY );
    vStartIntegerMathTasks( mainINTEGER_TASK_PRIORITY );
    vStartGenericQueueTasks( mainGEN_QUEUE_TASK_PRIORITY );
    vStartQueuePeekTasks();
    vStartRecursiveMutexTasks();

    // ??? Task
    xTaskCreate( vTask1, ( signed char * ) "Task1", ( ( unsigned short ) 96 ), ( void * ) NULL, tskIDLE_PRIORITY, NULL );
    // ??? Task
    xTaskCreate( vTask2, ( signed char * ) "Task2", ( ( unsigned short ) 96 ), ( void * ) NULL, tskIDLE_PRIORITY, NULL );
    // ??? Task
    xTaskCreate( vTask3, ( signed char * ) "Task3", ( ( unsigned short ) 96 ), ( void * ) NULL, tskIDLE_PRIORITY, NULL );
    // ??? Task
    xTaskCreate( vTask4, ( signed char * ) "Task4", ( ( unsigned short ) 96 ), ( void * ) NULL, tskIDLE_PRIORITY, NULL );
    // ??? Task
    xTaskCreate( vTask5, ( signed char * ) "Task5", ( ( unsigned short ) 96 ), ( void * ) NULL, tskIDLE_PRIORITY, NULL );
    // ??? Task
    xTaskCreate( vTask6, ( signed char * ) "Task6", ( ( unsigned short ) 96 ), ( void * ) NULL, tskIDLE_PRIORITY, NULL );
    // ??? Task
    xTaskCreate( vTask7, ( signed char * ) "Task7", ( ( unsigned short ) 96 ), ( void * ) NULL, tskIDLE_PRIORITY, NULL );

    /* Start the scheduler. */
    vTaskStartScheduler();

    /* Will only get here if there was insufficient memory to create the idle task.  The idle task is created within vTaskStartScheduler(). */
    for( ;; );
}
/*----------------------------------------------------------------------------------------------------------*/

/***** TASK #1 ******/
// ????
void vTask1 ( void *pvParameters ){
    portTickType xLastCheckTime;
    portTickType xDelayTime;

    xDelayTime = 125 / portTICK_RATE_MS;
    xLastCheckTime = xTaskGetTickCount();
    while(1){
        vTaskDelayUntil( &xLastCheckTime, xDelayTime );
        vParTestToggleLED( LED1 );
        uxHiWtrMrk_tsk1 = uxTaskGetStackHighWaterMark( NULL );
    }
}
/*----------------------------------------------------------------------------------------------------------*/

/***** TASK #2 ******/
// ???
void vTask2 ( void *pvParameters ){
    portTickType xLastCheckTime;
    portTickType xDelayTime;

    xDelayTime = 250 / portTICK_RATE_MS;
    xLastCheckTime = xTaskGetTickCount();
    while(1){
        vTaskDelayUntil( &xLastCheckTime, xDelayTime );
        vParTestToggleLED( LED2 );
        uxHiWtrMrk_tsk2 = uxTaskGetStackHighWaterMark( NULL );
    }
}
/*----------------------------------------------------------------------------------------------------------*/

/***** TASK #3 ******/
// ???
void vTask3 ( void *pvParameters ){
    portTickType xLastCheckTime;
    portTickType xDelayTime;

    xDelayTime = 500 / portTICK_RATE_MS;
    xLastCheckTime = xTaskGetTickCount();
    while(1){
        vTaskDelayUntil( &xLastCheckTime, xDelayTime );
        vParTestToggleLED( LED3 );
        uxHiWtrMrk_tsk2 = uxTaskGetStackHighWaterMark( NULL );
    }
}
/*----------------------------------------------------------------------------------------------------------*/

/***** TASK #4 ******/
// ???
void vTask4(void *pvParameters) {
    portTickType xLastCheckTime;
    portTickType xDelayTime;

    xDelayTime = 1000 / portTICK_RATE_MS;
    xLastCheckTime = xTaskGetTickCount();
    while(1){
        vTaskDelayUntil( &xLastCheckTime, xDelayTime );
        vParTestToggleLED( LED4 );
        uxHiWtrMrk_tsk2 = uxTaskGetStackHighWaterMark( NULL );
    }
}
/*----------------------------------------------------------------------------------------------------------*/

/***** TASK #5 ******/
// ?????
void vTask5 ( void *pvParameters ){

    vTaskDelay( 100 / portTICK_RATE_MS );            // Wait 0.1sec
    while(1){
        vTaskDelay( 10000 / portTICK_RATE_MS );        // Wait 10sec
        uxHiWtrMrk_tsk5 = uxTaskGetStackHighWaterMark( NULL );
    }
}
/*----------------------------------------------------------------------------------------------------------*/

/***** TASK #6 ******/
// ????
void vTask6(void *pvParameters) {
    portTickType xLastCheckTime;
    portTickType xDelayTime;

    vTaskDelay(  200 / portTICK_RATE_MS );    // Wait
    xDelayTime = 25 / portTICK_RATE_MS;
    xLastCheckTime = xTaskGetTickCount();    // Need to initialize time prior to the first call to vTaskDelayUntil()
    while(1){
        vTaskDelayUntil( &xLastCheckTime, xDelayTime );
        uxHiWtrMrk_tsk6 = uxTaskGetStackHighWaterMark( NULL );
    }
}

/***** TASK #7******/
// ????
void vTask7(void *pvParameters) {
    portTickType xLastCheckTime;
    portTickType xDelayTime;

    xDelayTime = 10 / portTICK_RATE_MS;    // 10mS interval
    xLastCheckTime = xTaskGetTickCount();
    while(1){
        vTaskDelayUntil( &xLastCheckTime, xDelayTime );
        uxHiWtrMrk_tsk7 = uxTaskGetStackHighWaterMark( NULL );
    }
}
/*----------------------------------------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------------------------------------*/

char *pcGetTaskStatusMessage( void ){
    /* Not bothered about a critical section here. */
    return 0;
}
/*----------------------------------------------------------------------------------------------------------*/

void prvSetupSystem( void ){
    ;
}
/*----------------------------------------------------------------------------------------------------------*/

void prvSetupHardware( void ){
    /* Disable peripherals power. */
    LPC_SC->PCONP = 0;

    /* Enable GPIO power. */
    LPC_SC->PCONP = PCONP_PCGPIO;

    /* Disable TPIU. */
    LPC_PINCON->PINSEL10 = 0;

    if ( LPC_SC->PLL0STAT & ( 1 << 25 ) ){
        /* Enable PLL, disconnected. */
        LPC_SC->PLL0CON = 1;
        LPC_SC->PLL0FEED = PLLFEED_FEED1;
        LPC_SC->PLL0FEED = PLLFEED_FEED2;
    }
    
    /* Disable PLL, disconnected. */
    LPC_SC->PLL0CON = 0;
    LPC_SC->PLL0FEED = PLLFEED_FEED1;
    LPC_SC->PLL0FEED = PLLFEED_FEED2;
        
    /* Enable main OSC. */
    LPC_SC->SCS |= 0x20;
    while( !( LPC_SC->SCS & 0x40 ) );
    
    /* select main OSC, 12MHz, as the PLL clock source. */
    LPC_SC->CLKSRCSEL = 0x1;
    
    LPC_SC->PLL0CFG = 0x20031;
    LPC_SC->PLL0FEED = PLLFEED_FEED1;
    LPC_SC->PLL0FEED = PLLFEED_FEED2;
          
    /* Enable PLL, disconnected. */
    LPC_SC->PLL0CON = 1;
    LPC_SC->PLL0FEED = PLLFEED_FEED1;
    LPC_SC->PLL0FEED = PLLFEED_FEED2;
    
    /* Set clock divider. */
    LPC_SC->CCLKCFG = 0x03;
    
    /* Configure flash accelerator. */
    LPC_SC->FLASHCFG = 0x403a;
    
    /* Check lock bit status. */
    while( ( ( LPC_SC->PLL0STAT & ( 1 << 26 ) ) == 0 ) );
        
    /* Enable and connect. */
    LPC_SC->PLL0CON = 3;
    LPC_SC->PLL0FEED = PLLFEED_FEED1;
    LPC_SC->PLL0FEED = PLLFEED_FEED2;
    while( ( ( LPC_SC->PLL0STAT & ( 1 << 25 ) ) == 0 ) );
    
    
    /* Configure the clock for the USB. */
      
    if( LPC_SC->PLL1STAT & ( 1 << 9 ) )
    {
        /* Enable PLL, disconnected. */
        LPC_SC->PLL1CON = 1;
        LPC_SC->PLL1FEED = PLLFEED_FEED1;
        LPC_SC->PLL1FEED = PLLFEED_FEED2;
    }
    
    /* Disable PLL, disconnected. */
    LPC_SC->PLL1CON = 0;
    LPC_SC->PLL1FEED = PLLFEED_FEED1;
    LPC_SC->PLL1FEED = PLLFEED_FEED2;
    
    LPC_SC->PLL1CFG = 0x23;
    LPC_SC->PLL1FEED = PLLFEED_FEED1;
    LPC_SC->PLL1FEED = PLLFEED_FEED2;
          
    /* Enable PLL, disconnected. */
    LPC_SC->PLL1CON = 1;
    LPC_SC->PLL1FEED = PLLFEED_FEED1;
    LPC_SC->PLL1FEED = PLLFEED_FEED2;
    while( ( ( LPC_SC->PLL1STAT & ( 1 << 10 ) ) == 0 ) );
    
    /* Enable and connect. */
    LPC_SC->PLL1CON = 3;
    LPC_SC->PLL1FEED = PLLFEED_FEED1;
    LPC_SC->PLL1FEED = PLLFEED_FEED2;
    while( ( ( LPC_SC->PLL1STAT & ( 1 << 9 ) ) == 0 ) );

    /*  Setup the peripheral bus to be the same as the PLL output (64 MHz). */
    LPC_SC->PCLKSEL0 = 0x05555555;

    /* Porting from system_LPC17xx.c void SystemInit()                            */
    /* Determine clock frequency according to clock register values             */
    if (((LPC_SC->PLL0STAT >> 24) & 3) == 3) {/* If PLL0 enabled and connected      */
        switch (LPC_SC->CLKSRCSEL & 0x03) {
            case 0:                           /* Internal RC oscillator => PLL0     */
            case 3:                           /* Reserved, default to Internal RC   */
                    SystemFrequency = (IRC_OSC *
                              (((2 * ((LPC_SC->PLL0STAT & 0x7FFF) + 1))) /
                              (((LPC_SC->PLL0STAT >> 16) & 0xFF) + 1))   /
                              ((LPC_SC->CCLKCFG & 0xFF)+ 1));
                    break;
            case 1:                           /* Main oscillator => PLL0            */
                    SystemFrequency = (OSC_CLK *
                              (((2 * ((LPC_SC->PLL0STAT & 0x7FFF) + 1))) /
                              (((LPC_SC->PLL0STAT >> 16) & 0xFF) + 1))   /
                              ((LPC_SC->CCLKCFG & 0xFF)+ 1));
                    break;
            case 2:                           /* RTC oscillator => PLL0             */
                    SystemFrequency = (RTC_CLK *
                              (((2 * ((LPC_SC->PLL0STAT & 0x7FFF) + 1))) /
                              (((LPC_SC->PLL0STAT >> 16) & 0xFF) + 1))   /
                              ((LPC_SC->CCLKCFG & 0xFF)+ 1));
                    break;
        }
    } else {
        switch (LPC_SC->CLKSRCSEL & 0x03) {
            case 0:                           /* Internal RC oscillator => PLL0     */
            case 3:                           /* Reserved, default to Internal RC   */
                    SystemFrequency = IRC_OSC / ((LPC_SC->CCLKCFG & 0xFF)+ 1);
                    break;
            case 1:                           /* Main oscillator => PLL0            */
                    SystemFrequency = OSC_CLK / ((LPC_SC->CCLKCFG & 0xFF)+ 1);
                    break;
            case 2:                           /* RTC oscillator => PLL0             */
                    SystemFrequency = RTC_CLK / ((LPC_SC->CCLKCFG & 0xFF)+ 1);
                    break;
        }
      }

    #if (FLASH_SETUP == 1)                  /* Flash Accelerator Setup            */
      LPC_SC->FLASHCFG  = FLASHCFG_Val;
    #endif

    /* Configure the LEDs. */
    vParTestInitialise();
}

/*----------------------------------------------------------------------------------------------------------*/

void vConfigureTimerForRunTimeStats( void ){
    const unsigned long TCR_COUNT_RESET = 2, CTCR_CTM_TIMER = 0x00, TCR_COUNT_ENABLE = 0x01;

    /* This function configures a timer that is used as the time base when collecting run time
     *  statistical information - basically the percentage     of CPU time that each task is utilizing.
     *  It is called automatically when the scheduler is started
     *  (assuming configGENERATE_RUN_TIME_STATS is set to 1). */

    /* Power up and feed the timer. */
    LPC_SC->PCONP |= 0x02UL;
    LPC_SC->PCLKSEL0 = (LPC_SC->PCLKSEL0 & (~(0x3<<2))) | (0x01 << 2);

    /* Reset Timer 0 */
    LPC_TIM0->TCR = TCR_COUNT_RESET;

    /* Just count up. */
    LPC_TIM0->CTCR = CTCR_CTM_TIMER;

    /* Prescale to a frequency that is good enough to get a decent resolution,
    but not too fast so as to overflow all the time. */
    LPC_TIM0->PR =  ( configCPU_CLOCK_HZ / 10000UL ) - 1UL;

    /* Start the counter. */
    LPC_TIM0->TCR = TCR_COUNT_ENABLE;
}
/*----------------------------------------------------------------------------------------------------------*/

