#include "mbed.h"
#include "KT0915.h"

KT0915::KT0915(PinName sda,PinName scl):
    _KT0915(sda,scl)
{
    _KT0915.frequency(100000);
}

void KT0915::init(void)
{
    write_reg(0x02, 0b00000000, 0b00000111);//FM space:100KHz, R/L mute:disable
    write_reg(0x04, 0b11100000, 0b10010000);//FM/AM/device mute:disable, bass:disable DAC cap:60uF
    write_reg(0x05, 0b10011000, 0b00100000);//Mono:enable, de-emphasis:50uF, blend:disable
    write_reg(0x0A, 0b00000100, 0b00000000);//LDO:highest, FM AFC:enable
    write_reg(0x0C, 0b00000000, 0b00101100);//FM wide freq:enable
    write_reg(0x0F, 0b10001000, 0b00000000);//Stanby:disable, volume:0
    read_reg(0x16,&data1,&data2);
    write_reg(0x16,data1&0b000111111, 0b11000010);//mode:FM, use internal defined band, audio gain:0dB, AM AFC:enable
    write_reg(0x22, 0b10100010, 0b11101100);//AM AGC fast win:fastest, AM AGC short win:slowest
    //AM bandwidth:6KHz, AM gain:12dB, Left Inverse Control:enable
    read_reg(0x23, &data1, &data2);
    write_reg(0x23, (data1 | 0b00011100), data2);//low th:B
    write_reg(0x2E, 0b00101000, 0b10001100);//softmute:fast, AM softmute start level:0b100
    //softmute target volume:0b0100, softmute mode:RSSI, FM softmute start threshold:0b100
    write_reg(0x33, 0b01010100, 0b00000001);//AM space:9KHz
    read_reg(0x3F, &data1, &data2);
    write_reg(0x3F, data1, ((data2 & 0b10001000) | 0b00010011));//RF AGC patch
}

int KT0915::getID(void)
{
    read_reg(0x01,&data1,&data2);
    return (data1<<8)+data2;
}

void KT0915::set_xtal(int xtal)
{
    read_reg(0x16,&data1,&data2);
    if(xtal==32) {
        write_reg(0x16,data1&0b11100000,data2);
    } else if(xtal==38) {
        write_reg(0x16,(data1&0b11100000)|0b00001001,data2);
    } else if(xtal==12000) {
        write_reg(0x16,(data1&0b11100000)|0b00010011,data2);
    } else if(xtal==24000) {
        write_reg(0x16,(data1&0b11100000)|0b00010111,data2);
    }
}

void KT0915::set_space(bool mode,int space)
{
    if(mode==0) {
        read_reg(0x33,&data1,&data2);
        read_reg(0x22,&data1,&data2);
        if(space==1) {
            write_reg(0x33,data1&0b00111111,data2);
            write_reg(0x22,data1,data2&0b00111111);
        } else if(space==9) {
            write_reg(0x33,(data1&0b00111111)|0b01000000,data2);
            write_reg(0x22,data1,data2|0b11000000);
        } else if(space==10) {
            write_reg(0x33,data1|0b11000000,data2);
            write_reg(0x22,data1,data2|0b11000000);
        }
    } else {
        read_reg(0x02,&data1,&data2);
        if(space==50) {
            write_reg(0x02,data1,(data2&0b11110011)|0b00001000);
        } else if(space==100) {
            write_reg(0x02,data1,(data2&0b11110011)|0b00000100);
        } else if(space==200) {
            write_reg(0x02,data1,data2&0b11110011);
        }
    }
}

void KT0915::set_vol(int vol)
{
    read_reg(0x0F, &data1, &data2);
    if ((data2 & 0b00011111) != vol) {
        write_reg(0x0F, data1, (data2 >> 5) * 0b100000 + vol);
    }
}

void KT0915::set_freq(bool mode,int freq)
{
    int f_upper, f_lower, freq2;
    read_reg(0x16,&data1,&data2);
    if (mode == 0) {
        write_reg(0x16, data1|0b10000000,data2);
        f_upper = (freq >> 8 | 0b10000000);
        f_lower = freq & 0b11111111;
        write_reg(0x17, f_upper, f_lower);
    }
    if (mode == 1) {
        write_reg(0x16, data1&0b01111111,data2);
        freq2 = freq / 5;
        f_upper = (freq2 >> 8 | 0b10000000);
        f_lower = freq2 & 0b11111111;
        write_reg(0x03, f_upper, f_lower);
    }
}

int KT0915::get_rssi(bool mode)
{
    int rssi;
    if (mode == 0) {
        read_reg(0x24, &data1, &data2);
        rssi = -90 + ( data1 & 0b00011111) * 3;
    }
    if (mode == 1) {
        read_reg(0x12, &data1, &data2);
        rssi = -100 + (data2 >> 3) * 3;
    }
    return rssi;
}

int KT0915::get_snr(bool mode)
{
    int snr;
    if (mode == 1) {
        read_reg(0x14, &data1, &data2);
        snr = (data1 & 0b00011111) * 0b100 + (data2 >> 6);
    }
    return snr;
}

bool KT0915::get_ready(void)
{
    read_reg(0x14,&data1,&data2);
    return (data1&0b00100000)>>5;
}

void KT0915::cali(void)   //re_cali
{
    write_reg(0x10,0b01100000,0b00000000);
}


void KT0915::write_reg(int memory_address,int byte1,int byte2)
{
    char cmd[3]= {memory_address,byte1,byte2};
    _KT0915.write(KT0915_address,cmd,3);
}

void KT0915::read_reg(int memory_address,int *byte1,int *byte2)
{
    char cmd[2];
    cmd[0]=memory_address;
    _KT0915.write(KT0915_address,cmd,1,true);
    _KT0915.read(KT0915_address|1,cmd,2);
    *byte1=cmd[0];
    *byte2=cmd[1];
}
