///////////////////////////////////////////////////////////////////////////////
// NJE10XCtrl: NJE-105/106 control lib   by rinos 2010
///////////////////////////////////////////////////////////////////////////////

#ifndef __NJE10XCTRL_H__
#define __NJE10XCTRL_H__

#include "mbed.h"

////////////////////////////////////////////////////////////////////////////////
// NJE10XCtrl Lib

class NJE10XCtrl {
    // defines /////////////////////////////////////////////////////////////////
public:
    // Attributes
    typedef enum {
        ATTR_GREEN,
        ATTR_RED,
        ATTR_YELLOW,
    } Attr1;

    typedef enum {
        ATTR_SCROLL,
        ATTR_SCROLL_R,
        ATTR_STOP,
        ATTR_FIX,
    } Attr2;

    typedef enum {
        ATTR_NORMAL,
        ATTR_BLINK,
        ATTR_REVERSE,
        ATTR_BLINK_REVERSE,
    } Attr3;
    
    // NJE command parameter
    typedef enum {
        OP_NORMAL,
        OP_AUTO,
        OP_POWERSAVE,
    } OpMode;

    typedef enum {
        FREEMODE_ON,
        FREEMODE_OFF,
    } FreeMode;

    typedef enum {
        SCROLL_SLOW,
        SCROLL_MIDDLE,
        SCROLL_FAST,
    } ScrollSpeed;

    typedef enum {
        BLINK_OFF,
        BLINK_SLOW,
        BLINK_MIDDLE,
        BLINK_FAST,
    } BlinkSpeed;

    // error code
    typedef enum {
        S_OK,
        S_WRITE_FAILED,
        S_BUFFER_OVERFLOW,
        S_INVALID_ID,
        S_INVALID_PARAM,
        
        S_NOT_SUPPORT, // not yet...
    } Status;

private:
    static const int MAX_DATA_BUF = 128;

    Serial    m_port;
    char m_buf[MAX_DATA_BUF + 1];
    int  m_size;
    

    // internal funcs
    Status sendCmd(const char* str);
    Status sendCmd(const char* str, int len);

	// Invalid method
protected:
	NJE10XCtrl(const NJE10XCtrl& v);
	const NJE10XCtrl& operator =(const NJE10XCtrl& v);

public:
    NJE10XCtrl(PinName tx_pin = p9, PinName rx_pin = p10);
    ~NJE10XCtrl();
    
    // Append Str/Attr to the internal buffer //////////////////////////////////
    Status add(char ch);                // append char to the internal buffer
    Status add(const char* str);        // append string to the internal buffer
    Status addAttr(Attr1 a1 = ATTR_GREEN, Attr2 a2 = ATTR_SCROLL, Attr3 a3 = ATTR_NORMAL);
    Status clear();                        // clear internal buffer
    int size() const { return m_size; }
    int left() const { return MAX_DATA_BUF - m_size; }
    
    // Set/Del Normal message (01-99) //////////////////////////////////////////
    Status setMessage(int id, const char* msg = 0,        // use internal buffer if msg is NULL
                    Attr1 a1 = ATTR_GREEN, Attr2 a2 = ATTR_SCROLL, Attr3 a3 = ATTR_NORMAL);
    Status delMessage(int id);                            // del message (1 to 99)
    Status setTitle(const char* msg = 0);                // use internal buffer if msg is NULL
    Status delTitle();                                    // del message title

    // Add/Clear free message (Max 20) /////////////////////////////////////////
    Status addFreeMessage(const char* msg = 0);    // use internal buffer if msg is NULL
    Status clearFreeMessage();                    // clear all free message
    Status addFreeCredit(const char* msg = 0);    // use internal buffer if msg is NULL
    Status clearFreeCredit();                    // clear all free credit

    // NJE control commands ////////////////////////////////////////////////////
    Status reset();
    
    // NJE show status /////////////////////////////////////////////////////////
    Status showTime();
    Status showContactCredit();
    Status showNewsCredit();
    Status showFreeCredit();

    // NJES commands ///////////////////////////////////////////////////////////
    Status setOpMode(OpMode mode = OP_NORMAL);
    Status setFreeMode(FreeMode mode = FREEMODE_ON);
    Status setScrollSpeed(ScrollSpeed speed = SCROLL_MIDDLE);
    Status setBlinkSpeed(BlinkSpeed speed = BLINK_FAST);
    Status setStopTime(int sec = 5);
};

#endif
