#include "NJE10XCtrl.h"

////////////////////////////////////////////////////////////////////////////////
// Local funcs

static bool myputc(Serial& port, int ch){
    while(!port.writeable());
    return port.putc(ch) != EOF;
}

static bool myputs(Serial& port, const char* str, size_t len){
    while(len--){
        if(!myputc(port, *str++)) return false;
    }
    return true;
}

static int myCheckMsg(const char*& msg, const char* buf, int size){
    if(msg) return strlen(msg);
    msg = buf;
    return size;
}

static char GetAttr1Char(NJE10XCtrl::Attr1 a1){
    return (char(a1) & 3) + 'A';
    
}

static char GetAttr2Char(NJE10XCtrl::Attr2 a2, NJE10XCtrl::Attr3 a3){
    const char A2[4] = {'A', 'E', 'I', 'W'};
    return A2[a2 & 3] + (a3 & 3);
}
    
////////////////////////////////////////////////////////////////////////////////
// NJE10XCtrl class
NJE10XCtrl::NJE10XCtrl(PinName tx_pin, PinName rx_pin) : m_port(tx_pin,  rx_pin), m_size(0) {
    m_port.baud(9600);
    m_port.format(8, Serial::None, 1);
    m_buf[0] = 0;
}

NJE10XCtrl::~NJE10XCtrl(){
}

////////////////////////////////////////////////////////////////////////////////
// NJE10XCtrl private funcs

const char NJE_DATA_HEAD[10] = {
    0x0d, 0x0a, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01
};

NJE10XCtrl::Status NJE10XCtrl::sendCmd(const char * str, int size){
    /* for debug
    char buf[128];
    memcpy(buf, str, size);
    buf[size] = 0;
    printf("Send '%s'\n", buf);
	*/
    
    if(    myputs(m_port, NJE_DATA_HEAD,    sizeof(NJE_DATA_HEAD)) &&
        myputs(m_port, str,            size) &&
        myputs(m_port, NJE_DATA_HEAD,    2)){
            return S_OK;
    }
    return S_WRITE_FAILED;
}

NJE10XCtrl::Status NJE10XCtrl::sendCmd(const char * str){
    return sendCmd(str, strlen(str));
};


// Append Str/Attr to the internal buffer ///////////////////////////////////////

NJE10XCtrl::Status NJE10XCtrl::add(char ch){
    // TODO: Escape command ( '~', "NJE...")
    if(m_size >= MAX_DATA_BUF) return S_BUFFER_OVERFLOW;
    m_buf[m_size++] = ch;
    m_buf[m_size]   = 0;
    return S_OK;
}

NJE10XCtrl::Status NJE10XCtrl::add(const char* str){
    while(*str) if(add(*str++)) return S_BUFFER_OVERFLOW;
    return S_OK;
}

NJE10XCtrl::Status NJE10XCtrl::addAttr(Attr1 a1, Attr2 a2, Attr3 a3){
    if(    add('~') == S_OK &&
        add(GetAttr1Char(a1)) == S_OK &&
        add(GetAttr2Char(a2, a3)) == S_OK &&
        add('~') == S_OK) return S_OK;

    return S_BUFFER_OVERFLOW; // faile
}
NJE10XCtrl::Status NJE10XCtrl::clear(){
    m_size = 0;
    m_buf[0] = 0;
    return S_OK;
}


// Set/Del Normal message (01-99) //////////////////////////////////////////////
NJE10XCtrl::Status NJE10XCtrl::setMessage(int id, const char* msg, Attr1 a1, Attr2 a2, Attr3 a3){
    if(id < 1 || 99 < id) return S_INVALID_ID;
    int len = myCheckMsg(msg, m_buf, m_size);
    
    const char CMD[]    = "]011A110%02d01010000%c%c%s";
    const int  CMD_SIZE    = 8 + 2 + 8 + 2;
    if(len > MAX_DATA_BUF - CMD_SIZE) return S_BUFFER_OVERFLOW;

    char buf[MAX_DATA_BUF];
    char c1 = GetAttr1Char(a1);
    char c2 = GetAttr2Char(a2, a3);

    return sendCmd(buf, sprintf(buf, CMD, id, c1, c2, msg));
}
NJE10XCtrl::Status NJE10XCtrl::delMessage(int id){
    if(id < 1 || 99 < id) return S_INVALID_ID;

    char buf[MAX_DATA_BUF];
    return sendCmd(buf, sprintf(buf, "]011A210%02d", id));
}

NJE10XCtrl::Status NJE10XCtrl::setTitle(const char* msg){
    int len = myCheckMsg(msg, m_buf, m_size);
    
    const char CMD[]    = "]011A11000%s";
    const int  CMD_SIZE    = 8 + 2;
    if(len > MAX_DATA_BUF - CMD_SIZE) return S_BUFFER_OVERFLOW;

    char buf[MAX_DATA_BUF];
    return sendCmd(buf, sprintf(buf, CMD, msg));
}
NJE10XCtrl::Status NJE10XCtrl::delTitle(){
    return sendCmd("]011A210");
}

// Add/Clear free message //////////////////////////////////////////////////////
NJE10XCtrl::Status NJE10XCtrl::addFreeMessage(const char* msg){
    int len = myCheckMsg(msg, m_buf, m_size);
    return sendCmd(msg, len);
}
NJE10XCtrl::Status NJE10XCtrl::clearFreeMessage(){
    return sendCmd("NJEM2");
}

NJE10XCtrl::Status NJE10XCtrl::addFreeCredit(const char* msg){
    int len = myCheckMsg(msg, m_buf, m_size);

    const char CMD[]    = "NJEC1%s";
    const int  CMD_SIZE    = 5;
    
    if(len > MAX_DATA_BUF - CMD_SIZE) return S_BUFFER_OVERFLOW;

    char buf[MAX_DATA_BUF];
    return sendCmd(buf, sprintf(buf, CMD, msg));
}
NJE10XCtrl::Status NJE10XCtrl::clearFreeCredit(){
    return sendCmd("NJEC2");
}

// NJE control commands ////////////////////////////////////////////////////////

NJE10XCtrl::Status NJE10XCtrl::reset(){
    return sendCmd("NJER");
}

// NJE show status /////////////////////////////////////////////////////////////
NJE10XCtrl::Status NJE10XCtrl::showTime(){
    return sendCmd("NJET");
}
NJE10XCtrl::Status NJE10XCtrl::showContactCredit(){
    return sendCmd("NJEV1");
}
NJE10XCtrl::Status NJE10XCtrl::showNewsCredit(){
    return sendCmd("NJEV2");
}
NJE10XCtrl::Status NJE10XCtrl::showFreeCredit(){
    return sendCmd("NJEV3");
}

// NJED commands ///////////////////////////////////////////////////////////////
NJE10XCtrl::Status NJE10XCtrl::setOpMode(OpMode mode){
    const char CMD[]    = "NJES02%02d";
    char buf[MAX_DATA_BUF];
    return sendCmd(buf, sprintf(buf, CMD, mode));
}

NJE10XCtrl::Status NJE10XCtrl::setFreeMode(FreeMode mode){
    const char CMD[]    = "NJES05%02d";
    char buf[MAX_DATA_BUF];
    return sendCmd(buf, sprintf(buf, CMD, mode));
}

NJE10XCtrl::Status NJE10XCtrl::setScrollSpeed(ScrollSpeed speed){
    const char CMD[]    = "NJES06%02d";
    char buf[MAX_DATA_BUF];
    return sendCmd(buf, sprintf(buf, CMD, speed));
}

NJE10XCtrl::Status NJE10XCtrl::setBlinkSpeed(BlinkSpeed speed){
    const char CMD[]    = "NJES07%02d";
    char buf[MAX_DATA_BUF];
    return sendCmd(buf, sprintf(buf, CMD, speed));
}

NJE10XCtrl::Status NJE10XCtrl::setStopTime(int sec){
    if(sec > 10) return S_INVALID_PARAM;
    const char CMD[]    = "NJES08%02d";
    char buf[MAX_DATA_BUF];
    return sendCmd(buf, sprintf(buf, CMD, sec));
}
