/* mbed Microcontroller Library
 * Copyright (c) 2006-2014 ARM Limited
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <events/mbed_events.h>
#include <mbed.h>
#include "ble/BLE.h"
#include "ble/Gap.h"
#include "my-ble-definition.h"

/** BLE service definition. don't edit this */
const static char     DEVICE_NAME[] = MY_DEVICE_NAME;
const static uint32_t trigger_flags = MY_TRIGGERS;
static EventQueue eventQueue(/* event count */ 16 * EVENTS_EVENT_SIZE);
UUID MY_SERVICE_UUID("4c592000-0000-0000-0000-000000000000");
MyBLEservice* MY_BLE;

char *ble_error_string(const ble_error_t error) {
    switch(error) {
    case BLE_ERROR_BUFFER_OVERFLOW:
        return ((char*)"The requested action would cause a buffer overflow and has been aborted.");
    case BLE_ERROR_NOT_IMPLEMENTED:
        return ((char*)"Requested a feature that isn't yet implemented or isn't supported by the target HW.");
    case BLE_ERROR_PARAM_OUT_OF_RANGE:
        return ((char*)"One of the supplied parameters is outside the valid range.");
    case BLE_ERROR_INVALID_PARAM:
        return ((char*)"One of the supplied parameters is invalid.");
    case BLE_STACK_BUSY:
        return ((char*)"The stack is busy.");
    case BLE_ERROR_INVALID_STATE:
        return ((char*)"Invalid state.");
    case BLE_ERROR_NO_MEM:
        return ((char*)"Out of memory.");
    case BLE_ERROR_OPERATION_NOT_PERMITTED:
        return ((char*)"The operation requested is not permitted.");
    case BLE_ERROR_INITIALIZATION_INCOMPLETE:
        return ((char*)"The BLE subsystem has not completed its initialization.");
    case BLE_ERROR_ALREADY_INITIALIZED:
        return ((char*)"The BLE system has already been initialized.");
    case BLE_ERROR_UNSPECIFIED:
        return ((char*)"Unknown error.");
    case BLE_ERROR_INTERNAL_STACK_FAILURE:
        return ((char*)"The platform-specific stack failed.");
    default:
        return ((char*)"Unknown error");
    }
}

static void onConnectionCallback(const Gap::ConnectionCallbackParams_t *params)
{
    set_led(BLEST_LED, true);

#ifdef MY_BLE_DEBUG
    DPRINTF("Connected: %x", params->peerAddr[Gap::ADDR_LEN - 1]);
    for (int i = Gap::ADDR_LEN - 2; i >= 0; i--)
    {
        DPRINTF(":%x", params->peerAddr[i]);
    }
    DPRINTF("\r\n");
#endif
}

void disconnectionCallback(const Gap::DisconnectionCallbackParams_t *params)
{
    ble_error_t error;
    set_led(BLEST_LED, false);
    error = BLE::Instance().gap().startAdvertising();

#ifdef MY_BLE_DEBUG
    if (error == BLE_ERROR_NONE) {
        DPRINTF("Disconnecte and start advertising.\r\n");
    } else {
        DPRINTF("Disconnected, but advertising failed.\r\n");
        DPRINTF(" %s\r\n", ble_error_string(error))
    }
#endif
}

void timerCallback(void) {
    set_led(SYSST_LED, true);
    wait_ms(50);
    set_led(SYSST_LED, false);

    eventQueue.call(my_ble_timer);
}

void int1_rise(void) {
    eventQueue.call(my_ble_int1_rise);
}

void int1_fall(void) {
    eventQueue.call(my_ble_int1_fall);
}

void int2_rise(void) {
    eventQueue.call(my_ble_int2_rise);
}

void int2_fall(void) {
    eventQueue.call(my_ble_int2_fall);
}

void ble_received(const uint8_t *buf) {
    eventQueue.call(my_ble_received, *((uint32_t *)buf));
}

/**
 * This function is called when the ble initialization process has failled
 */
void onBleInitError(BLE &ble, ble_error_t error)
{
    DPRINTF("BLE initialize error.\r\n");
    DPRINTF(" %s\r\n", ble_error_string(error));
}

/**
 * Callback triggered when the ble initialization process has finished
 */
void bleInitComplete(BLE::InitializationCompleteCallbackContext *params)
{
    BLE&        ble   = params->ble;
    ble_error_t error = params->error;

    if (error != BLE_ERROR_NONE) {
        /* In case of error, forward the error handling to onBleInitError */
        onBleInitError(ble, error);
        return;
    }

    /* Ensure that it is the default instance of BLE */
    if(ble.getInstanceID() != BLE::DEFAULT_INSTANCE) {
        return;
    }

    /* return value is void */
    ble.gap().onConnection(onConnectionCallback);
    /* return value is void */
    ble.gap().onDisconnection(disconnectionCallback);

    /* Setup primary service */
    MY_BLE = new MyBLEservice(ble);
    if (MY_BLE == NULL) {
        return;
    }

    /* set write callback */
    MY_BLE->set_write_callback(ble_received);

    /* setup user program */
    if (my_ble_init() != MY_SUCCESS) {
        return;
    }

    /* Setup advertising */
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::COMPLETE_LIST_128BIT_SERVICE_IDS,
        MY_SERVICE_UUID.getBaseUUID(),
        UUID::LENGTH_OF_LONG_UUID);
    ble.gap().accumulateAdvertisingPayload(
        GapAdvertisingData::SHORTENED_LOCAL_NAME,
        reinterpret_cast<const uint8_t *>(DEVICE_NAME),
        sizeof(DEVICE_NAME));

    ble.gap().setDeviceName(reinterpret_cast<const uint8_t *>(DEVICE_NAME));
    ble.gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    ble.gap().setAdvertisingInterval(1000); /* 1000ms */
    ble.gap().startAdvertising();

#ifdef MY_BLE_DEBUG
    DPRINTF("Successfully started!\r\n");
#endif
}

void scheduleBleEventsProcessing(BLE::OnEventsToProcessCallbackContext* context) {
    BLE &ble = BLE::Instance();
    eventQueue.call(Callback<void()>(&ble, &BLE::processEvents));
}

int main()
{
    DPRINTF("Start %s program\r\n", MY_DEVICE_NAME);

    /* setup callback */
    if (trigger_flags & MY_TRIGGER_TIMER) {
        eventQueue.call_every((uint32_t)(MY_TIMER_INTERVAL) * 1000, timerCallback);
    }
    if (trigger_flags & MY_TRIGGER_INT1_RISE) {
        set_isr(MY_INT1_RISE, int1_rise);
    }
    if (trigger_flags & MY_TRIGGER_INT1_FALL) {
        set_isr(MY_INT1_FALL, int1_fall);
    }
    if (trigger_flags & MY_TRIGGER_INT2_RISE) {
        set_isr(MY_INT2_RISE, int2_rise);
    }
    if (trigger_flags & MY_TRIGGER_INT2_FALL) {
        set_isr(MY_INT2_FALL, int2_fall);
    }

    BLE &ble = BLE::Instance();
    ble.onEventsToProcess(scheduleBleEventsProcessing);
    ble.init(bleInitComplete);
    eventQueue.dispatch_forever();

    return 0;
}
