/******************************************************************************
 *
 *  Copyright 2018 ASAHI KASEI MICRODEVICES CORPORATION, Japan
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 ******************************************************************************/
#pragma once
#include "ble/BLE.h"
#include "my-ble-hardware.h"

extern UUID MY_SERVICE_UUID;

class MyBLEservice {
public:
    enum {
        UUID_READ_DATA1 = 0x2001,
        UUID_READ_DATA2 = 0x2002,
        UUID_WRITE = 0x1001
    };

    static const unsigned BLE_WRITE_BUF_LEN = 20;

public:
    /**
     * @brief   EnvironmentalService constructor.
     * @param   _ble Reference to BLE device.
     */
    MyBLEservice(BLE& _ble) :
        ble(_ble),
        data1Characteristic(UUID_READ_DATA1, &u32_data1, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        data2Characteristic(UUID_READ_DATA2, &u32_data2, GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        writeCharacteristic(UUID_WRITE, write_buf)
    {
        static bool serviceAdded = false; /* We should only ever need to add the information service once. */
        if (serviceAdded) {
            return;
        }

        /* set callback */
        writeCharacteristic.setWriteAuthorizationCallback(this, &MyBLEservice::onWritten);

        GattCharacteristic *charTable[] = { &data1Characteristic,
                                            &data2Characteristic,
                                            &writeCharacteristic };

        GattService myService(MY_SERVICE_UUID, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));

        ble.gattServer().addService(myService);
        serviceAdded = true;
    }

    /**
     * @brief   Update data1 characteristic.
     * @param   data New value.
     */
    void update_data1(uint32_t data)
    {
        u32_data1 = data;
        ble.gattServer().write(data1Characteristic.getValueHandle(), (uint8_t *) &u32_data1, sizeof(uint32_t));
    }

    /**
     * @brief   Update data2 characteristic.
     * @param   data New value.
     */
    void update_data2(uint32_t data)
    {
        u32_data2 = data;
        ble.gattServer().write(data1Characteristic.getValueHandle(), (uint8_t *)&u32_data2, sizeof(uint32_t));
    }

    void set_write_callback(Callback<void(const uint8_t *)> fp)
    {
        if (fp) {
            this->wfp = fp;
        }
    }

private:
    /**
     * @brief   Update data1 characteristic.
     * @param   data New value.
     */
    void onWritten(GattWriteAuthCallbackParams *param)
    {
        uint16_t len = param->len;

        if (len > BLE_WRITE_BUF_LEN) {
            len = BLE_WRITE_BUF_LEN;
        }
        memset(write_buf, 0x00, BLE_WRITE_BUF_LEN);
        memcpy(write_buf, param->data, len);
        ble.gattServer().write(writeCharacteristic.getValueHandle(), (uint8_t *)&write_buf, sizeof(write_buf));

        if (this->wfp) {
            this->wfp(this->write_buf);
        }
    }

    BLE& ble;

    uint32_t u32_data1;
    uint32_t u32_data2;
    uint8_t  write_buf[BLE_WRITE_BUF_LEN];

    Callback<void(const uint8_t *buf)> wfp;

    ReadOnlyGattCharacteristic<uint32_t> data1Characteristic;
    ReadOnlyGattCharacteristic<uint32_t> data2Characteristic;
    WriteOnlyArrayGattCharacteristic<uint8_t, BLE_WRITE_BUF_LEN> writeCharacteristic;
};

