//Whack-a-mole with LCD and touch keypad

#include "mbed.h"
#include "uLCD_4DGL.h"  /*LCD*/
#include <mpr121.h>     /*touch keypad*/

#include <stdio.h>      /* printf, scanf, puts, NULL */
#include <stdlib.h>     /* srand, rand */
#include <time.h>       /* time */
#include <math.h>       /* log2 */



#define REFRESH_TIME_MS 2000

//Mole locations
#define X1 1
#define X2 43
#define X3 85
#define Y1 0
#define Y2 30
#define Y3 60
#define Y4 90

PwmOut vib(p21); //vibration motor
Serial pc(USBTX, USBRX);
uLCD_4DGL lcd(p28, p27, p30);

AnalogIn random_analog(A0);

// Create the interrupt receiver object on pin 26 -->> for keypad
InterruptIn interrupt(p26);

// Setup the i2c bus on pins 28 and 27 -->> for keypad
I2C i2c(p9, p10);

// Setup the Mpr121:
// constructor(i2c object, i2c address of the mpr121) -->> for keypad
Mpr121 mpr121(&i2c, Mpr121::ADD_VSS);

//Timer
Timer t; //for function use
Timer gametime; //for int main() use
Timer again; //for restart countdown

//global variables accessable to all functions
//mole locations
int hole = 0;   //hole number
int x = 0;      //coordinate values (not actual pixel values)
int y = 0;
//score
int value = 0;
int keynum = 0;
int hit = 0;
int miss = 0;
int highest = 0;

bool isMoleAlive = 0;


// initializes screen with 12 empty mole holes
void allempty() 
{
    lcd.cls();
    
    //choose empty mole hole picture and intialize variables for while function
    lcd.set_sector_address(0x0000, 0x0000);
    int i = 0;
    int k = 1;
    int x = 0;
    int y = 1;
    
    //print 12 empty mole holes
    while(i < 12)
    {   
        lcd.locate(0,15);
        lcd.display_image(x, y);
        if ( k < 3 ) 
        {   x += 42;
            k++;        }
        else 
        {   k = 1;
            x = 0; 
            y += 30;    }   
        i++;
    }
    
    //count down to start of game
    int z = 3;
    lcd.locate(4, 15);
    lcd.printf("Get Ready!");
    wait(1);
    while (z > 0)
    {
        lcd.locate(4, 15);
        lcd.printf("     %d       ", z);
        wait(1);
        z--;
    }
    lcd.locate(5, 15);
    lcd.printf("   Go!     ");
    wait(0.5);
}    

//revised modulus operator created to work correctly with negative modulus
int mod (int x1, int x2)
{
    if(x1 < 0) { return (x1 % x2)+x2; }
    else { return x1 % x2; }
}

//pop / show the mole
void popup()
{
    isMoleAlive = 1;
    
    //generate random int 0 to 11
    hole = rand() % 12;
    
    //use random number to match x and y coordinate of touchpad
    if (hole < 4)       { x = 0;}
    else if (hole < 8)  { x = 1;}
    else                { x = 2;}
    y = mod(4-(1+mod(hole, 4)), 4);    
        
    //pop up mole corresponding to that hole number
    lcd.set_sector_address(0x0000, 0x0005);
    lcd.display_image(x*42, y*30);
    
    //testing purposes only
        //lcd.locate(0,15);
        //lcd.printf("r:%d,   x:%d,   y:%d\n", hole, x, y);
    
    highest++; //highest possible number of hits
    
    lcd.locate(0,15);
    //lcd.printf("H:%d Hit:%d, Miss:%d", keynum, hit, miss);
}

//hide a mole
void hide()
{
    isMoleAlive = 0;
    vib = 0;
    //reseting the mole hole to empty hole
    lcd.set_sector_address(0x0000, 0x0000);
    lcd.display_image(x*42, y*30);
}

//to convert keypad
int mylog2(int arg)
{
        //creating log 2 
        int squared = 1;
        int tmp = 0; //log 2 value
        for ( ; squared < arg; )  //( squared != arg )
        {
            squared = squared * 2;
            tmp++;
        }
        if (squared == arg)
        {
            return tmp;     //keynum = tmp;
        }
        return keynum;      //return same in case more than one pad is touched.
}
    
void keyInterrupt() {
    
    value=mpr121.read(0x00);
    value +=mpr121.read(0x01)<<8;
    // LED demo mod by J. Hamblen
    pc.printf("MPR value: %d \r\n", value);
    
    
    //Did the hammer hit the mole?
    if (value != 0)     //ignore the release
    {
        //Converting to keynum
        keynum = mylog2(value);
        
        //hit or miss
        if ( keynum == hole && isMoleAlive )   //If the hammer hit the mole
        {
            hit++;
            vib = 1;
            lcd.set_sector_address(0x0000, 0x000A);
            lcd.display_image(x*42, y*30);
            isMoleAlive = 0;
        }
        else
        {
            miss++;
        }
        

    }
    
}



//initialize lcd screen
void setup()
{
    interrupt.fall(&keyInterrupt);
    interrupt.mode(PullUp);
    
    srand (random_analog * 100000);
    
    lcd.baudrate(3000000);
    lcd.background_color(0x029142);
    lcd.textbackground_color(0x029142);
    lcd.cls();
    lcd.media_init();
    lcd.color(WHITE);
    lcd.set_font(FONT_7X8);
    lcd.text_bold(ON);
}

//printing score
void printScore()
{
    lcd.locate(0,15);
    lcd.printf("->%d ", keynum);
    lcd.locate(8,15);           
    lcd.printf("+%d  ", hit);
    lcd.locate(15,15);
    lcd.printf("-%d  ", miss);
}

//function to endgame and print final score
void endgame()
{
    //displaying final score
    lcd.cls();
    lcd.text_width(1.5); 
    lcd.text_height(1.5);
    lcd.color(WHITE);
    lcd.locate(0,1);
    lcd.printf("    FINAL SCORE\n");
    lcd.printf("------------------\n");
    lcd.locate(1,4);
    lcd.printf("%d HITS", hit);
    lcd.locate(1,7);
    lcd.printf("%d MISSES \n\n------------------", miss);
    
    //print a commentary based on player's score
    double grade = 100*(hit-(0.25*miss))/highest;
    lcd.locate(0,12);
    if (grade < 20) {lcd.printf("  WOW.. YOU SUCK");}
    else if (grade < 50) {lcd.printf("    TRY HARDER");}
    else if (grade < 80) {lcd.printf("   YOU DID OKAY");}
    else if (grade < 90) {lcd.printf("    NICE JOB!!");}
    else if (grade < 100) {lcd.printf("    GRRRREAT!!");}
    else if (grade == 100) 
        {
            lcd.locate(0, 11);
            lcd.printf("    /\\/\\/\\/\\/\\ \n");
            lcd.printf("   |          |\n");
            lcd.printf("   | MOLE GOD |\n");
            lcd.printf("   |__________|\n");
        }
}

//function to restart another game
bool restart()
{
    
    bool replay = false;
    
    //reset score
    hit = miss = highest = 0;
    
    //clear screen for restart prompt
    lcd.cls();
    lcd.locate(0,1);
    lcd.text_height(2);
    lcd.text_width(2);
    lcd.printf(" ANOTHER   GAME?  \nPRESS ANY   KEY   \n");
    
    again.start();
    int countdown;
    //waiting for key press
    while(again.read() < 10 && replay == false)
    {
        countdown = 10 - (int)again.read();
        //countdown: user has 10 seconds to replay 
        lcd.locate(0,7);
        lcd.printf("    %d    ", countdown);
                
        //check for key press
        if(value != 0)
        {
            //user has pressed a key and wants to play again!
            replay = true;
            break;
        }
        
    }
    
    again.stop();
    again.reset();
    return replay;
}


//////////////////////////////////////////////////
////////////####main####//////////////
int main()
{   
    bool keepplaying = 1;
    
    //initializing...
    setup(); 
    
    while(keepplaying == true)
    {   //starting the game - 30 seconds for testing
        allempty();
        gametime.reset();
        gametime.start();
        float timelimit;
        float overalltime = 20;
        
        while (gametime.read() < overalltime) 
        { 
            popup();
            
            //reduce time limit as game progresses
            if (gametime.read() < overalltime/3)       { timelimit = 2; }
            else if (gametime.read() < overalltime/2)  { timelimit = 1.6;   }
            else if (gametime.read() < overalltime/1.5)  { timelimit = 1.2; }
            else if (gametime.read() < overalltime/1.3)  { timelimit = 1;   }
            else if (gametime.read() < overalltime/1.1)  { timelimit = 0.8; }
            else if (gametime.read() < overalltime)  { timelimit = 0.65;} 
                
            wait(timelimit);
            
            if (isMoleAlive == 1)
            {
                miss++;
            }
            
            hide();
            printScore();
            
        }
            
            gametime.stop();
            endgame();
            wait(3);
            keepplaying = restart();
            
            
    }
        
    lcd.cls();
    lcd.locate(0,2);
    lcd.text_height(2);
    lcd.text_width(2);
    lcd.printf(" THANKS\n");
    lcd.printf("   FOR\n\n");
    lcd.printf(" PLAYING\n");
    
}

    
    

