
// ##################################################################################
// XPL with TFT Screen Application
// ##################################################################################

// generate XPL debug level output on the serial line
//#define DEBUG_OUTPUT_XPL

// turn OFF dependency on being connected to ethernet
#define CONNECTED_OFF

// -----------------------------------------------------------
// MBED Headers
#include "mbed.h"
#include "EthernetInterface.h"
#include <math.h>

// -----------------------------------------------------------
// Graphics Headers
#include "ILI9341.h"
#include "SPI_STMPE610.h"
#include "Arial12x12.h"
#include "Arial24x23.h"
#include "Arial28x28.h"
#include "Arial43x48_numb.h"

// -----------------------------------------------------------
// My Headers
#include "Watchdog.h"
#include "xPL.h"
#include "nihh_utils.h"

// -----------------------------------------------------------
// XPL defines
#define UDP_PORT            3865
#define UDP_BUFFER_SIZE     500

#define XPL_NAME            "NUCLEO1"

// -----------------------------------------------------------
// global variables
Watchdog wd;
xPL xpl;

DigitalOut led1(LED1);
DigitalOut led2(LED2);
DigitalOut led3(LED3);
DigitalOut led4(LED4);

Serial logger(SERIAL_TX, SERIAL_RX);

EthernetInterface   gEth;
UDPSocket           _socket;
UDPSocket           _socketSender;
char                in_buffer[UDP_BUFFER_SIZE];


#define PIN_RESET_TFT   PC_13 /* place holder */

// ------------- FROM WORKING VERSION

#define PIN_MOSI        D10   // WAS D11 !!!!!!!!!!!!!!!   CLASHES WITH ETHERNET ON NUCLEO !!!
#define PIN_MISO        D12
#define PIN_SCLK        D13
#define PIN_CS_TFT      D5
#define PIN_DC_TFT      D6
#define PIN_BL_TFT      D7
#define PIN_CS_SD       D4

// =====================================

#define PIN_XP          A2
#define PIN_XM          A0
#define PIN_YP          A3
#define PIN_YM          A1


struct point {
    int x;
    int y;
};

void calibrate(void);
bool getPixel(point& p);
point toPixel(point p);

typedef enum { YES, MAYBE, NO } TOUCH;

TOUCH getTouch(point& p);

int readTouch(PinName p, PinName m, PinName a, PinName i);


int x_off = 22071;
int y_off = 18707;
int pp_tx = 373;
int pp_ty = 297;

// =====================================

DigitalOut backlight(PIN_BL_TFT) ;

ILI9341 TFT(SPI_8, 10000000, PIN_MOSI, PIN_MISO, PIN_SCLK, PIN_CS_TFT, PIN_RESET_TFT, PIN_DC_TFT, "Seeed2.8") ;

int page = 1 ;
int numPage = 3 ;


// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// ++++++++   TFT SCREEN FUNCTIONS
// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

int readTouch(PinName p, PinName m, PinName a, PinName i)
{
    DigitalOut _p(p);
    _p = 1;
    DigitalOut _m(m);
    _m = 0;
    AnalogIn   _a(a);
    AnalogIn   _i(i); // this pin has to be high Z (DigitalIn may also work)
    wait_us(10);
    return _a.read_u16();
}

TOUCH getTouch(point& p)
{
    int y2 = readTouch(PIN_XP,PIN_XM,PIN_YP,PIN_YM);
    int x2 = readTouch(PIN_YP,PIN_YM,PIN_XP,PIN_XM);
    int y1 = readTouch(PIN_XP,PIN_XM,PIN_YP,PIN_YM);
    int x1 = readTouch(PIN_YP,PIN_YM,PIN_XP,PIN_XM);
    int xd = x1 - x2;
    int yd = y1 - y2;
    xd = (xd > 0) ? xd : -xd;
    yd = (yd > 0) ? xd : -xd;
    p.x = x1 + x2;
    p.y = y1 + y2;
    
    const int th = 8000;
    const int df =  100;
    TOUCH touch;
    if (x1 < th || x2 < th ||
            y1 < th || y2 < th) {
        p.x = 0;
        p.y = 0;
        touch = NO;
    } else if (xd > df || yd > df) {
        touch = MAYBE;
    } else {
        touch = YES;
    }
    //TFT.locate(0,50);
    //TFT.printf("x: %6i y: %6i",p.x,p.y);
    return touch;
}

void calibrate(void)
{
    int i;
    int a = 0,b = 0,c = 0, d = 0;
    int pos_x = 0, pos_y = 0;
    point p;
    
    backlight = 0 ;
    //TFT.BusEnable(true) ;
    
    TFT.background(Black);
    wait(0.1) ;
    TFT.foreground(White);
    wait(0.1) ;
    TFT.cls() ;
    wait(0.1) ;
    TFT.set_font((unsigned char*) Arial12x12);
    TFT.locate(90,0);
    TFT.printf("Graphics");
    
    TFT.line(0,3,6,3,White);
    TFT.line(3,0,3,6,White);
    
    backlight = 1 ;
    
    //if (font)
    //{
        // get the center of the screen
        pos_x = TFT.columns() / 2 - 3;
        pos_x = pos_x * Arial12x12[1];
        pos_y = (TFT.rows() / 2) - 1;
        pos_y = pos_y * Arial12x12[2];
        TFT.locate(pos_x,pos_y);
        TFT.printf("press cross    ");
        TFT.locate(pos_x,pos_y + Arial12x12[2]);
        TFT.printf("to calibrate   ");
    //}
    
    for (i=0; i<5; i++) {
        while (getTouch(p) != YES)
            /*nothing*/;
        a += p.x;
        b += p.y;
    }
    a = a / 5;
    b = b / 5;
    //if (font)
    //{
        TFT.locate(pos_x,pos_y);
        TFT.printf("ok             ");
        TFT.locate(pos_x,pos_y + Arial12x12[2]);
        TFT.printf("release touch  ");
    //}
    while (getTouch(p) != NO)
        /*nothing*/;
    
    TFT.cls();
    TFT.line(TFT.width() -5, TFT.height() - 8,TFT.width() - 5,TFT.height() -1,White);   // paint cross
    TFT.line(TFT.width() - 8,TFT.height() - 5,TFT.width() - 1,TFT.height() - 5,White);
    //if (font)
    //{
        TFT.locate(pos_x,pos_y);
        TFT.printf("press cross    ");
        TFT.locate(pos_x,pos_y + Arial12x12[2]);
        TFT.printf("to calibrate   ");
    //}
    for (i=0; i<5; i++) {
        while (getTouch(p) != YES)
            /*nothing*/;
        c+= p.x;
        d+= p.y;
    }
    c = c / 5;
    d = d / 5;
    x_off = a;
    y_off = b;
    i = c-a;  // delta x
    pp_tx = i / (TFT.width() - 6);
    i = d-b;  // delta y
    pp_ty = i / (TFT.height() - 6);
    //if (font)
    //{
        TFT.locate(pos_x,pos_y);
        TFT.printf("Calibrated     ");
        TFT.locate(pos_x,pos_y + Arial12x12[2]);
        TFT.printf("x %6i %4i", x_off, pp_tx);
        TFT.locate(pos_x,pos_y + 2*Arial12x12[2]);
        TFT.printf("y %6i %4i", y_off, pp_ty);
    //}
    while (getTouch(p) != NO)
        /*nothing*/;
    TFT.cls();
    
    //TFT.BusEnable(false) ;
    
    printf("x_off:%6i pp_tx:%4i \n\r", x_off, pp_tx);
    printf("y_off:%6i pp_ty:%4i \n\r", y_off, pp_ty);
}

point toPixel(point p)
{
    p.x -= x_off;
    p.x /= pp_tx;
    int w = TFT.width();
    if (p.x > w) p.x = w;
    if (p.x < 0) p.x = 0;
    p.y -= y_off;
    p.y /= pp_ty;
    int h = TFT.height();
    if (p.y > h) p.y = h;
    if (p.y < 0) p.y = 0;
    return (p);
}

bool getPixel(point& p)
{
    TOUCH touch = getTouch(p);
    p = toPixel(p);
    return touch == YES;
}


// ===================================================

void initTFT(void)
{
    //Configure the display driver
    //TFT.BusEnable(true) ;
    TFT.FastWindow(true) ;
    TFT.background(Black);
    TFT.foreground(White);
    wait(0.01) ;
    TFT.cls();
    //TFT.BusEnable(false) ;
}

void screen1(void) // Welcome Screen
{
    printf("screen1\r\n" );
    
    //TFT.BusEnable(true) ;
    backlight = 0 ;
    TFT.background(White) ;
    wait(0.1) ;
    TFT.cls() ;
    wait(0.1) ;
    
    TFT.set_font((unsigned char*) Arial24x23);
    TFT.foreground(Red) ;
    TFT.locate(80, 40) ;
    TFT.printf("MBED") ;
    TFT.foreground(Blue);
    TFT.locate(60, 80) ;
    TFT.printf("2.8\"TFT") ; 
    TFT.locate(40, 120) ;
    TFT.printf("with touch") ;
    TFT.foreground(Black);
    TFT.set_font((unsigned char*) Arial12x12);
    TFT.foreground(Blue) ;
    TFT.locate(30, 180) ;
    TFT.printf("This program is running on") ;
    TFT.locate(30, 200) ;
    TFT.printf("ST Nucleo F411RE with") ;
    TFT.locate(30, 220) ;
    TFT.printf("a program developed in mbed") ;
    TFT.foreground(Green) ;
    TFT.locate(30, 260) ;
    TFT.printf("To advance demo page, touch") ;
    TFT.locate(30, 280) ;
    TFT.printf("and hold right side of screen") ;
    TFT.locate(30, 300) ;
    TFT.printf("until the next screen starts") ;
    //TFT.BusEnable(false) ;
    backlight = 1 ;
}

void screen2(void) // Graphics
{
    printf("screen2\r\n" );
    
    //Draw some graphics
    int i, x[2], y[2] ;
    backlight = 0 ;
    //TFT.BusEnable(true) ;
    TFT.background(Black);
    wait(0.1) ;
    TFT.foreground(White);
    wait(0.1) ;
    TFT.cls() ;
    wait(0.1) ;
    TFT.set_font((unsigned char*) Arial12x12);
    TFT.locate(90,0);
    TFT.printf("Graphics");
    
    x[0] = 25 ; x[1] = 224 ;
    y[0] = 20 ; y[1] = 219 ;
    for (i = 20 ; i < 220 ; i += 10) {
        TFT.line(i+5, y[0], i+5, y[1], Blue) ;
        TFT.line(x[0], i, x[1], i, Blue) ;
    }
    TFT.line(125, y[0], 125, y[1], Green) ;
    TFT.line(x[0], 120, x[1], 120, Green) ;
    TFT.rect(x[0],y[0], x[1], y[1], Green) ;
    TFT.locate(10, 20) ;
    TFT.printf("V") ;
    TFT.locate(0, 115) ;
    TFT.printf("0.0") ;
    TFT.locate(115, 225) ;
    TFT.printf("0.0") ;
    TFT.locate(215, 225) ;
    TFT.printf("T") ;

    double s;
    for (int i = x[0]; i < 225; i++) {
        s = 40 * sin((long double)i / 20);
        TFT.pixel(i, 120 + (int)s, White);
    }
    
    TFT.fillrect(10, 240, 229, 309, White) ;
    TFT.rect(10, 240, 229, 309, Red) ;
    TFT.rect(11, 241, 228, 308, Red) ;
    
    TFT.background(White) ;
    TFT.foreground(Black) ;
    TFT.locate(20, 250) ;
    TFT.printf("With QVGA resolution") ;
    TFT.locate(20, 270) ;
    TFT.printf("simple graphics drawing") ;
    TFT.locate(20, 290) ;
    TFT.printf("capability is provided") ;
    //TFT.BusEnable(false) ;
    backlight = 1 ;
}    

void screen3(void) // Graphics
{
    printf("screen3\r\n" );
    
    //int rowX;
    int rowY;
    //int x = Terminal6x8[1];
    int y = Terminal6x8[2];
    
    //Draw some text
    backlight = 0 ;
    
    TFT.background(Black);
    wait(0.1) ;
    TFT.foreground(White);
    wait(0.1) ;
    TFT.cls() ;
    wait(0.1) ;
    
    backlight = 1 ;
    
    TFT.set_font((unsigned char*) Terminal6x8);
    
    for( rowY=0; rowY<100; ++rowY )
        {
        TFT.locate(0, (rowY%25)*y);
        TFT.printf("%3d TFT width = %d, height = %d", rowY, TFT.width(), TFT.height()) ;
        }
    
}

void incPage(void)
{
    page++ ;
    if (page >= numPage) {
        page = 0 ;
    }
}

void decPage(void) 
{
    page-- ;
    if (page < 0) {
        page = numPage - 1 ;
    }
}

// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// ++++++++   ETHERNET 
// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
// +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

// -----------------------------------------------------------
// 
// -----------------------------------------------------------
void terminal_error_state( void )
{
    led1=0;
    led2=0;
    led3=0;
    logger.printf("\r\n\r\nJUST TO BE CLEAR - THIS IS BAD BAD NEWS !!! \r\n");
    logger.printf("\r\n\r\n---- WAITING FOR THE WATCHDOG TO RESET US ---- \r\n");
    while( 1 )
        {
        led1 = !led1;
        led2 = !led2;
        led3 = !led3;
        wait(0.3);
        }
}

// -----------------------------------------------------------
// 
// -----------------------------------------------------------
static bool DeviceConnect()
{
    int retries = 10;
    int rc = 0;
    
    while (retries--) 
        {
        rc = gEth.connect();
        if (rc == 0) {
            // done
            return true;
            }
        else
            {
            logger.printf("Connecting... ERR %d\r\n", rc);
            }
        }
    
    logger.printf("Connecting... FAILED\r\n");
    return false;
}

// -----------------------------------------------------------
// 
// -----------------------------------------------------------
#ifndef CONNECTED_OFF_NOPE
static bool DeviceEthStartup()
{
   int rc = 0;
   
   // get ethernet up !
    if( !DeviceConnect() ) terminal_error_state();
    
    // Output the network address
    const char *ip = gEth.get_ip_address();
    const char *netmask = gEth.get_netmask();
    const char *gateway = gEth.get_gateway();
    logger.printf("IP address: %s\r\n", ip ? ip : "None");
    logger.printf("Netmask: %s\r\n", netmask ? netmask : "None");
    logger.printf("Gateway: %s\r\n", gateway ? gateway : "None");
    
    // setup a UDP listener
    _socket.set_blocking(true);
    _socket.set_timeout(3000);
    //_socket.set_broadcasting(true);
    
    // open the socket
    rc = _socket.open((NetworkInterface*)&gEth);
    if( rc != 0)
        {
        logger.printf("UDPSocket RECEIVER OPEN ERROR   (rc=%d)\r\n", rc);
        terminal_error_state();
        }
    
    // bind the XPL port
    rc = _socket.bind(UDP_PORT);
    logger.printf("Binding port %d\r\n", UDP_PORT);
    if( rc != 0)
        {
        logger.printf("UDPSocket BIND ERROR   (rc=%d)\r\n", rc);
        terminal_error_state();
        }
    
    // open the sender socket
    rc = _socketSender.open((NetworkInterface*)&gEth);
    if( rc != 0)
        {
        logger.printf("UDPSocket SENDER OPEN ERROR   (rc=%d)\r\n", rc);
        terminal_error_state();
        }
    //_socketSender.set_broadcasting(true);
    
    return true;
}
#endif

// -----------------------------------------------------------
// 
// -----------------------------------------------------------
void SendUdPMessage(char *buffer)
{
    #ifndef CONNECTED_OFF
        // send the UDP data as broadcast
        int rc = _socketSender.sendto( "255.255.255.255", UDP_PORT, (const void *)buffer, strlen( buffer ) );
        
        #ifdef DEBUG_OUTPUT_XPL
            logger.printf("[SEND rc=%d] buffer[%d]\r\n%s\r\n", rc, strlen(buffer), buffer);
        #endif
        
    #else
        logger.printf("[OFFLINE] XPLsend [%d]\r\n%s\r\n", strlen(buffer), buffer);
    #endif
}


// -----------------------------------------------------------
// 
// -----------------------------------------------------------
void SendXPLint(char *Device, int iCurrent )
{
//    xPL_Message _message;
//    _message.SetTarget( "*" );
//    _message.AddCommand("device", "xxxxx");
//    _message.AddCommand("current", "0");
//    xpl.SendMessage( &_message, true );
}


// -----------------------------------------------------------
// 
// -----------------------------------------------------------
void SendXPLString(char *Device, char *szCurrent )
{
//    xPL_Message _message;
//    _message.SetTarget( "*" );
//    _message.AddCommand("device", "xxxxx");
//    _message.AddCommand("current", "0");
//    xpl.SendMessage( &_message, true );
}


// -----------------------------------------------------------
// 
// -----------------------------------------------------------
void AfterParseAction(xPL_Message * message)
{
    // is this a message for me ?
    if (xpl.TargetIsMe(message))
        {
        logger.printf("XPL: MSG IS FOR ME !");
        
        logger.printf("%s\r\n", message->toString());
        
        // is this a control message?
        if (message->IsSchema("control", "basic"))
            {
            logger.printf("XPL: {control.basic}");
            }
        }
    
    //logger.printf("XPL: [%s]\r\n", message->source.device_id );
    logger.printf("XPL: [%s] %d items\r\n", message->source.device_id, message->command_count );
    
    // do we have any data ???  maybe the parse failed....
    if( message->command_count == 0 )
        {
        // 
        logger.printf("XPL: no data?\r\n%s\r\n", in_buffer );
        }
    else
        {
        // output the message data  
        for (short i=0; i<message->command_count; i++)
            {
            // grab each item and do something with it
            //logger.printf("%s=%s\r\n", message->command[i].name, message->command[i].value);
            }
        logger.printf("XPL: %d items processed\r\n", message->command_count );
        }
    
    // output the full msg!
    //logger.printf("%s\r\n", message->toString());
}



// -----------------------------------------------------------
// 
// -----------------------------------------------------------
//void Receiver_Thread(void const *argument)
void Receiver_Thread()
{
    SocketAddress   _sAddr;
    int             iNoDataCount = 0;
    
    // loop forever
    while(1)
        {
        // receive from the udp socket
        led2 = 0;
        
        #ifndef CONNECTED_OFF
            int n = _socket.recvfrom(&_sAddr, in_buffer, UDP_BUFFER_SIZE);
            //led2 = 1;
            
            // see what weve got !
            if (n == NSAPI_ERROR_WOULD_BLOCK)
                {
                // timeout with no data...
                ++iNoDataCount;
                //logger.printf("-- NoData (TIMEOUT)\r\n");
                led2 = 0;
                }
            else if (n < 0)
                {
                // some other error
                ++iNoDataCount;
                //logger.printf("-- NoData (ERR) %i\r\n", n);
                led2 = 0;
                }
            else if( n > 0)
                {
                // good data :)
                iNoDataCount = 0;
                in_buffer[n] = '\0';
                // logger.printf("[%d bytes] %s:%d\r\n", n, _sAddr.get_ip_address(), _sAddr.get_port());
                // logger.printf("%s\r\n", in_buffer);
                led2 = 1;
                
                // try and parse the msg
                xpl.ParseInputMessage(in_buffer);
                }
            
            // have we stopped receiving data ?!?!?
            if( iNoDataCount > 20 )
                {
                logger.printf("ERROR: No UDP data repeatidly...\r\n");
                terminal_error_state();
                }
        #endif
        
        // stroke the WD
        wd.Service();
        
        ThisThread::sleep_for(300);
        }
}


// -----------------------------------------------------------
// 
// -----------------------------------------------------------

Thread thread1;

int main()
{
    const char *ip;
    char xpl_ip[16];
    int iAppLoop=0;
    
    point p;
    
    
    int prevPage = 99 ;
    bool waitTouch = false ;
    
    // FIRST THINGS FIRST!!!  set the WD timeout interval...
    // enough to get up and running!
    wd.Configure(1000.0);
    
    // setup the pc serial logger
    logger.baud(115200);
    logger.printf("\r\n\r\n<<<<<<<<< Basic UDP XPL Listener >>>>>>>>>>\r\n");
    
    
    
    // Fireup the TFT Screen
    initTFT() ;
    printf("TFT Started {width = %d, height = %d}\n\r", TFT.width(), TFT.height()) ;
    //calibrate();
    screen1() ;
    
    
    led1 = 0;
    led2 = 0;
    led3 = 0;
    led4 = 0;
    
    // WatchDog Startup Information
    logger.printf("\r\n------------------------------------------------------\r\n" );
    logger.printf("[SystemCoreClock] %d Hz\r\n", SystemCoreClock);
    logger.printf("[Reset reason] ");
    if (__HAL_RCC_GET_FLAG(RCC_FLAG_IWDGRST)) logger.printf("Independant-Watchdog  ");
    if (__HAL_RCC_GET_FLAG(RCC_FLAG_WWDGRST)) logger.printf("Window-Watchdog  ");
    if (__HAL_RCC_GET_FLAG(RCC_FLAG_LPWRRST)) logger.printf("Low-Power-Reset  ");
    if (__HAL_RCC_GET_FLAG(RCC_FLAG_SFTRST)) logger.printf("Software-Reset  ");
    if (__HAL_RCC_GET_FLAG(RCC_FLAG_PORRST)) logger.printf("Power-On/Down-Reset  ");
    if (__HAL_RCC_GET_FLAG(RCC_FLAG_PINRST)) logger.printf("Reset-pin  ");
    if (__HAL_RCC_GET_FLAG(RCC_FLAG_BORRST)) logger.printf("Brownout  ");
    if (__HAL_RCC_GET_FLAG(RCC_FLAG_LSIRDY)) logger.printf("Low-speed-internal-clock-ready  ");
    __HAL_RCC_CLEAR_RESET_FLAGS();
    logger.printf("\r\n------------------------------------------------------\r\n");
    
    if (wd.WatchdogCausedReset())
        {
        logger.printf("<<Watchdog caused reset>>\r\n");
        led3 = 1;
        }
    else
        {
        logger.printf("<<Normal Startup>>\r\n");
        led1 = 1;
        led2 = 1;
        led3 = 0;
        }
    logger.printf("------------------------------------------------------\r\n\r\n");
    
    
    
    ThisThread::sleep_for(100);
    
    screen2() ;
    
    #ifndef CONNECTED_OFF
        // Fireup the Ethernet
        DeviceEthStartup();
        
        // process the devices ip adress into the XPL instance ID
        ip = gEth.get_ip_address();
        strcpy( xpl_ip, ip );
        UTILS_strip_char_in_string( xpl_ip, '.' );
    #endif
    
    screen1() ;
    
    // now set the WD to something to run with - between receving data
    wd.Service();
    wd.Configure(300.0);
    
    // setup xpl
    xpl.SendExternal = &SendUdPMessage;         // pointer to the send callback
    xpl.AfterParseAction = &AfterParseAction;   // pointer to a post parsing action callback 
    xpl.SetSource( "NIHH", XPL_NAME, xpl_ip);   // parameters for hearbeat message

    // kick off the XPL receiver thread to run in the background
    //Thread thread1(Receiver_Thread, NULL, osPriorityNormal); //, DEFAULT_STACK_SIZE);   //osPriorityHigh
//    thread1.set_priority(osPriorityLow);
//    thread1.start( Receiver_Thread );
    
    #ifndef CONNECTED_OFF
        // i like to do this on startup to announce ourself
        logger.printf("XPL: STARTED {VendorID:%s, DeviceID:%s, InstanceID:%s}\n\r", "NIHH", XPL_NAME, xpl_ip );
        xpl.SendHBeat();
    #endif
    
    // Fireup the TFT Screen
    //initTFT() ;
    //printf("TFT Started {width = %d, height = %d}\n\r", TFT.width(), TFT.height()) ;
    //calibrate();
    
    
    
    
    // start running the main processing stuff here
    while (true) 
        {
            

        printf("page %d of %d\r\n", page, numPage );
        
        switch(page) {
        case 0:
            if (prevPage != page) {
                screen1() ;
            }
            waitTouch = true ;
            break ;
        case 1:
            if (prevPage != page) {
                screen2() ; 
            }
            waitTouch = true ;
            break ;
        case 2:
            if (prevPage != page) {
                screen3() ; 
            }
            waitTouch = true ;
            break ;
        default:
            page = 0 ; 
            break ;
        }
        prevPage = page ;
            
        
        do  {
            
            ThisThread::sleep_for(10);
            
            // allow xpl to do its thing...
            #ifndef CONNECTED_OFF
                xpl.Process();
            #endif
            
            if( getPixel(p) ) 
                {
                printf("TFT Touch x = %d, y = %d\n\r", p.x, p.y) ;
                
                if (p.x < 100) 
                    { // left
                    decPage() ;
                    } 
                else if (p.x > 150) 
                    { // right
                    incPage() ;
                    }
                waitTouch = false ;
                }
            
            // SIMPLE TEST - every 50 loops send a test XPL msg
            ++iAppLoop;
            if( iAppLoop >= 20000 )
                {
                xPL_Message _message;
                
                /*
                _message.SetSchema( "clock", "update" );
                _message.SetTarget( "*" );
                _message.type = XPL_STAT;
                _message.hop = 1;
                _message.AddCommand("time", "20180730113315");
                xpl.SendMessage( (xPL_Message *)&_message, true );
                */
                
                _message.SetSchema( "sensor", "basic" );
                _message.SetTarget( "*" );
                _message.type = XPL_STAT;
                _message.hop = 1;
                _message.AddCommand("device", "test1");
                _message.AddCommand("type", "test");
                _message.AddCommand("current", "1");
                xpl.SendMessage( (xPL_Message *)&_message, true );
                
                logger.printf("<<<<<< Sample Message Sent >>>>>>\r\n");
                
                // reset
                iAppLoop = 0;
                }
            
            } while(waitTouch != false) ;
        
        }
}
