// Copyright 2009 Richard Parker

#ifndef MBED_HMC6352_H
#define MBED_HMC6352_H

#include "mbed.h"

class HMC6352
{
public:
    // Enum to make the returned mode more readable.
    enum Mode {
        Standby = 0x0,
        Query = 0x1,
        Continuous = 0x2,
        None = 0x3
    };
    
    // Enum to make the returned output type more readable.
    enum Output {
        Heading = 0x0,
        RawX = 0x1,
        RawY = 0x2,
        X = 0x3,        
        Y = 0x4,
        Unknown = 0x5
    };

    HMC6352(PinName sda, PinName scl);
    ~HMC6352();

    // Address.
    inline int address() { return _address; }
    inline void setAddress(int address) { _address = address; }
    
    // Version.
    char getVersion();
    void setVersion(const char version);
    
    // Time delay.
    char getTimeDelay();
    void setTimeDelay(const char delay);

    // Summing.
    char getSumming();
    void setSumming(const char samples);
    
    // Heading.
    float getHeading();
    
    // Control
    void reset();
    
    void wakeUp();
    void goToSleep();
    
    void startCalibrate();
    void endCalibrate();
    void calibrate(int delay = 20);
    
    // Get the mode section of tyhe operation byte from RAM.
    HMC6352::Mode getMode();
    void setMode(HMC6352::Mode mode);
    
    // Save the operation byte in ram to eeprom.
    void saveOperation();  
    
    // Get the current output mode from the RAM register.
    HMC6352::Output getOutput();
    void setOutput(HMC6352::Output output);

    inline float version() const { return 0.1; }

private:
    // Clears the contents of the buffer to 0x0
    void _clearBuffer();
    
    // Converts the values in the lower two bytes of the buffer to a heading.
    float _bufferToHeading();
    
    // Read or write to memory.
    void _readFromMemory(const char address, const bool eeprom);
    void _writeToMemory(const char address, const char data, const bool eeprom);

    // Set up a buffer to read and write data. The maximum amount of data 
    // returned or sent is 3 bytes.
    char _buffer[3];

    // Set up the i2c communication channel object.
    I2C _i2c;
       
    // To hold the address currently being used.
    int _address;

};

#endif