// Copyright 2010 Richard Parker

#ifndef MBED_EATOUCH_H
#define MBED_EATOUCH_H

#include "mbed.h"

class EALCD;

/**
 * Class to handle the touch screen on the LCD board.
 * @author Richard Parker
 */
class EATouch
{
public:
    struct matrix {
        short xOffset;
        float xScale;
        short yOffset;
        float yScale;
    };

    EATouch(EALCD& lcd);
    
    ~EATouch();
    
    /**
     * Start the calibration routine.
     * Calibration is a 3 point process to set up internal conversion measurements.
     * @return    bool    The status as to whether calibration was successful or whether timed 
     *                    out etc.
     */
    bool calibrate();
    
    /**
     * Set the threshold value to register a press after.
     * @param    threshold    The value over which the value read is considered a press.
     */
    inline void setThreshold(unsigned short threshold) { _threshold = threshold; }
    
    
    inline unsigned short threshold() { return _threshold; }

    /**
     * Check for a press.
     * @param    x        The x value of the touch in pixels (converted using calibration settings).
     * @param    y        The y value of the touch in pixels (converted using calibration settings).
     * @param    pressed  Boolean describing if a touch is detected. If pressed is false then x 
     *                    and y are undefined.
     */
    void touch(short& x, short& y, bool& pressed);
    
    bool save();
    
    bool load();
       
private:
    EALCD& _lcd;
    
    EATouch::matrix _cal_matrix;
    
    void _raw(short& x, short& y, bool& pressed);

    /**
     * Value to hold the threshold which is used to decided if pressed or not.
     */
    unsigned int _threshold;
   
    /**
     * Take in x and y values from touch panel and convert to pixel position.
     * Uses the calibration values taht are set using the calibrate() function.
     * @param    iX    Input x value.
     * @param    iY    Input y value.
     * @param    oX    Output x value.
     * @param    oY    Output y value.
     */
    void _convert(  short iX, 
                    short iY, 
                    short& oX,
                    short& oY);
                    
    bool _captureCalibrationPoint(  short iX, 
                    short iY, 
                    short& oX,
                    short& oY);
    
};

#endif