#include "mbed.h"
#include "EATouch.h"

#include "EALCD.h"
#include "../graphics/EAColor.h"
#include "../graphics/EABrush.h"

#define EATOUCH_X1        40
#define EATOUCH_Y1        40
#define EATOUCH_X2        280
#define EATOUCH_Y2        200
#define EATOUCH_RADIUS    10
#define EATOUCH_TIMEOUT   5000

EATouch::EATouch(EALCD& lcd)
:   _lcd(lcd),
    _threshold(500)
{
}

EATouch::~EATouch()
{
}

bool EATouch::calibrate()
{
    // Draw the first point.
    short x1;
    short y1;
    short x2;
    short y2;
       
    // Capture valid calibration points.
    if ((_captureCalibrationPoint(EATOUCH_X1, EATOUCH_Y1, x1, y1) == true)
        &&
        (_captureCalibrationPoint(EATOUCH_X2, EATOUCH_Y2, x2, y2) == true))
    {
        // If captured valid calibration points now convert for matrix.
        _cal_matrix.xScale = (float)(EATOUCH_X2 - EATOUCH_X1)/(x2 - x1);
        _cal_matrix.xOffset = EATOUCH_X1 - _cal_matrix.xScale*x1;
        
        _cal_matrix.yScale = (float)(EATOUCH_Y2 - EATOUCH_Y1)/(y2 - y1);
        _cal_matrix.yOffset = EATOUCH_Y1 - _cal_matrix.yScale*y1;       
        
        // Now try to store for later.
        save();
        
        return true;
    } else {
        return false;
    }
}

void EATouch::_raw(short& x, short& y, bool& pressed)
{
    unsigned short z1 = 0;
    unsigned short z2 = 0;   
    
    unsigned short tx;
    unsigned short ty;

    short ax;
    short ay;
    short ap;
    
    // Get the first sample.
    _lcd._getTouch(tx, ty, z1, z2); 
    ax = tx;
    ay = ty;
    ap = ty*(z2/z1 - 1);
    
    // Get other samples.
    for (int i = 0; i < 0; i++)
    {
        _lcd._getTouch(tx, ty, z1, z2); 
        
        ax = (ax + tx)/2;
        ay = (ay + ty)/2;
        ap = (ap + (ty*(z2/z1 - 1)))/2;
    }
   
    // Assign values back out.
    x = ax;
    y = ay;
   
    // Check that pressure value is above the threshold.
    if (ap > threshold())
    {        
        // Switch x over to increase in desired direction - the same as the pixels.
        x = 4095 - x;
        
        pressed = true;
    } else {
        pressed = false;  
        x = 0;
        y = 0;
    }
}

void EATouch::_convert(short iX, short iY, short& oX, short& oY)
{
    // Use matrix values to convert the values.
    oX = iX*_cal_matrix.xScale + _cal_matrix.xOffset;
    oY = iY*_cal_matrix.yScale + _cal_matrix.yOffset;
}

bool EATouch::_captureCalibrationPoint(short iX, short iY, short& oX, short& oY)
{
    EAColor white = EAColor(0xFF, 0xFF, 0xFF);
    EAColor black = EAColor(0x00, 0x00, 0x00);

    EABrush brush = _lcd.brush();
    EAPen pen = _lcd.pen();
    
    bool pressed = false;
    
    short x = 0;
    short y = 0;
    
    Timer t;

    // Set up brushes and pens.
    brush.setColor(white);  
    _lcd.setBrush(brush);
    pen.setColor(white);  
    _lcd.setPen(pen);
    
    // Clear the screen ready for retrieving calibration data.
    brush.setColor(black);  
    _lcd.setBrush(brush);
    _lcd.clearScreen();

    // Draw circle to hit.
    _lcd.drawFilledEllipse(iX-EATOUCH_RADIUS, iY-EATOUCH_RADIUS, EATOUCH_RADIUS*2, EATOUCH_RADIUS*2);
        
    // Now wait for press or timeout.
    t.start();
    while (t.read_ms() < EATOUCH_TIMEOUT)
    {
        _raw(y, x, pressed);
        
        if (pressed == true)
        {
            break;
        }
        
        wait(0.2);
    }
    
    // Check if time out or pressed.
    brush.setColor(black);  
    _lcd.setBrush(brush);

    if (pressed == true)
    {
        oX = x;
        oY = y;
    
        _lcd.clearScreen();

        return true;
    } else {
        // Timed out.
        oX = 0;
        oY = 0;
    
        _lcd.clearScreen();

        return false;
    }
}


void EATouch::touch(short& x, short& y, bool& pressed)
{
    short tx = 0;
    short ty = 0;
    bool tp = false;  
    
    // Get the values.
    _raw(tx, ty, tp);
    _convert(tx, ty, x, y);
       
    if (tp == true)
    {
        // Pressure threshold says pressed check coordinates (don't register 
        // if seems to be off screen)
        if ((x < 0)
            ||
            (x >= _lcd.width())
            ||
            (y < 0)
            ||
            (y >= _lcd.height()))
        {
            pressed = false;
            x = 0;
            y = 0;
        } else {   
            pressed = true;
        }
    } else {
        pressed = false;
        x = 0;
        y = 0;
    }
}

bool EATouch::load()
{
    FILE* fp = fopen("/local/touch.cfg", "r");
    if (fp == NULL)
    {
        return false;
    }
    
    if (fread(&_cal_matrix, sizeof(EATouch::matrix), 1, fp) != 1)
    {
        fclose(fp);
        return false;
    }
    
    // Close the file.
    fclose(fp);
    
    return true; 
}

bool EATouch::save()
{
    FILE* fp = fopen("/local/touch.cfg", "w");
    if (fp == NULL)
    {
        return false;
    }
    
    if (fwrite(&_cal_matrix, sizeof(EATouch::matrix), 1, fp) != 1)
    {
        fclose(fp);
        return false;
    }
    
    // Close the file.
    fclose(fp);
    
    return true; 
}

