// Copyright 2010 Richard Parker

#ifndef MBED_EALCD_H
#define MBED_EALCD_H

#include "mbed.h"

#include "../graphics/EAPen.h"
#include "../graphics/EABrush.h"
#include "../fonts/EAFont.h"
#include "../images/EAImage.h"

#ifndef PI
#define PI 3.14593
#endif

class EATouch;
class EAImage;

/**
 * Class encapsulating the EA LCD.
 * @author Richard Parker
 */
class EALCD
{
public:
    friend EATouch;
    friend EAImage;

    EALCD(  PinName LED_PWM,
            PinName LCD_RS, 
            PinName LCD_SO, 
            PinName LCD_SI, 
            PinName LCD_SCL,
            PinName LCD_CS,
            PinName TCH_CS);
    ~EALCD();
    
    /**
     * Sets the brightness of the display by adjusting the backlight power.
     * @param    percentage    A value between 0.0 - 1.0 (fully off - fully on).
     */
    void setBrightness(float percentage);
    inline float brightness() { return (1-_backlight); }
    
    inline void setMaxBrightness(float percentage) { _maxBright = percentage; setBrightness(_maxBright); }
    inline float maxBrightness() { return _maxBright; }
    
    void noop();
    
    void clearScreen();
    void drawPoint(short x, short y);
    void drawLine(short x0, short y0, short x1, short y1);
    void drawFilledRect(short x, short y, unsigned short w, unsigned short h);
    void drawRect(short x, short y, unsigned short w, unsigned short h);
    void drawEllipse(short x, short y, unsigned short w, unsigned short h);
    void drawFilledEllipse(short x, short y, unsigned short w, unsigned short h);
    void drawImage(unsigned short x, unsigned short y, EAImage& img);
    void drawText(unsigned short x, unsigned short y, const std::string& text); 
    
    inline unsigned short width() const { return 320; }
    inline unsigned short height() const { return 240; }
    
    inline void setPen(const EAPen& pen) { _pen = pen; }
    inline const EAPen& pen() { return _pen; }
      
    inline void setBrush(const EABrush& brush) { _brush = brush; }
    inline const EABrush& brush() { return _brush; }

    inline void setFont(const EAFont& font) { _font = font; }
    inline const EAFont& font() { return _font; }
    
    inline float version() const { return 0.6; }

protected:
    void _getTouch(unsigned short& x, unsigned short& y, unsigned short& z1, unsigned short& z2);

private:
    /**
     * PWM control for the backlight.
     */
    PwmOut _backlight;
    
    /**
     * 4 wire spi interface for spi comms.
     */
    SPI _comm;
    DigitalOut _rs;
    DigitalOut _cs;
    
    /**
     * Chip select for the touch screen.
     */
    DigitalOut _tch_cs;
       
    /**
     * Graphic helper to store the current pen.
     */
    EAPen _pen;
    
    /**
     * Graphic helper to store the current brush.
     */
    EABrush _brush;
    
    /**
     * Graphic helper to store the current font.
     */
    EAFont _font;
    
    float _maxBright;

    void _swap(short& i, short& j);

    void _writeToDisplay(unsigned short data);
    void _writeToRegister(unsigned short addr, unsigned short data);
    
    void _initialise();

    void _moveTo(short x, short y);
    void _window(short x, short y, unsigned short w, unsigned short h);
    
    void _drawPoint(short x, short y, unsigned short c);
    void _drawLine(short x0, short y0, short x0, short y0, unsigned short c);
    void _drawEllipse(short cX, short cY, unsigned short w, unsigned short h, bool filled);
    void _draw4EllipsePoints(short cX, short cY, short x, short y, bool filled);

};

#endif