#include "mbed.h"
#include "EATouchManager.h"

#include "../screen/EATouch.h"
#include "../screen/EALCD.h"
#include "EAHitBox.h"

EATouchManager::EATouchManager(EALCD& lcd, EATouch& touch)
:   _touch(touch),
    _lcd(lcd),
    _head(NULL),
    _count(0),
    _timeout(2000)
{
}

EATouchManager::~EATouchManager()
{
    // Free any memory associated with the list.
    clearHitBoxes();
}

void EATouchManager::tick()
{
    short x = 0;
    short y = 0;
    bool p = false;
    
    // Get the touch.
    _touch.touch(x, y, p);
      
    // Check if hit target.
    if (p == true)
    {
        _doHits(x, y);
        
        // Reset the timeout count.
        _watchReset();
    }
    
    // Check the timeout. If reached then turn off the screen.
    _watchCheck();
}

void EATouchManager::_watchReset()
{
    _count = 0;
    
    // If the lcd is off then turn it back on as the count has been reset.
    if (_lcd.brightness() < 0.01)
    {
        float step = (_lcd.maxBrightness() - _lcd.brightness())/10;
    
        for(int i = 0; i < 10; i++)
        {
            _lcd.setBrightness(_lcd.brightness() + step);
            wait(0.2);
        }
    }
}

void EATouchManager::_watchCheck()
{  
    // If the count has passed the timeout then turn the lcd off if it is on.
    // Otherwise increment the count.
    if (_count > _timeout)
    {
        if (_lcd.brightness() >= 0.01)
        {
            float step = _lcd.brightness()/10;
        
            for(int i = 0; i < 10; i++)
            {
                _lcd.setBrightness(_lcd.brightness() - step);
                wait(0.2);
            }            
        }
    } else {
        _count++;
    }
}

void EATouchManager::_doHits(short x, short y)
{
    EAHitBox* current = _head;
    
    while (current != NULL)
    {
        // Do the check this will fire the pointer function if it does.
        current->checkContains(x, y);
        
        // Get the next item in the linked list.
        current = current->next();
    }
}

void EATouchManager::deleteHitBox(EAHitBox* box)
{
    EAHitBox* current = _head;
    EAHitBox* last = NULL;
    EAHitBox* next = NULL;
    
    // If not valid then just return.
    if (box == NULL)
    {
        return;
    }
    
    while (current != NULL)
    {
        // Get the next item in the linked list.
        next = current->next();
        
        // If the pointer matches the one that want to delete then first
        // remove from the list, put the link in from previous to next
        // and actually delet the data.
        if (box == current)
        {
            if ((next != NULL) && (last == NULL))
            {
                // If there is an item after but none before then the next 
                // becomes the root item.
                _head = next;
            } 
            else if ((next != NULL) && (last != NULL)) 
            {
                // There was a last and there is a next so need to stitch 
                // list together removing the current.
                last->setNext(next);
            }
            else if ((next == NULL) && (last == NULL))
            {
                // If there is no item after this one then nothing else to do.
                _head = NULL;
            }
            else if ((next == NULL) && (last != NULL))
            {
                // Last is now the end of the list.
                last->setNext(NULL);
            }
                
            // Actually get rid of memory.
            delete current;
            current = NULL;
            break;                
        }
        
        // Swap pointers over.
        last = current;
        current = next;
    }
}

EAHitBox* EATouchManager::createHitBox()
{
    EAHitBox* box = new EAHitBox();
    
    // If able to create then add the pointer to the list and return.
    if (box != NULL)
    {
        box->setNext(_head);
        _head = box;
    }
    
    return box;
}

void EATouchManager::clearHitBoxes()
{
    EAHitBox* current = _head;
    EAHitBox* next = NULL;
       
    // Delete each hit box in turn.
    while (current != NULL)
    { 
        next = current->next();
        
        delete current;
        current = NULL;
        
        current = next;
    }
    
    // Reset the root.
    _head = NULL;

}


