// Copyright 2010 Richard Parker

#include "mbed.h"
#include "EAFontCache.h"

EAFontCache::EAFontCache()
{
}

EAFontCache::~EAFontCache()
{
    // Free any memory associated with the list.
    _clearCacheEntries();
}

EAFontCache& EAFontCache::instance()
{
    // Only instantiated when getInstance() is called.
    static EAFontCache instance; 
    return instance;
}

void EAFontCache::_clearCacheEntries()
{
    std::vector<EAFontCacheEntry*>::iterator it;
    it = _entries.begin();

    while (it != _entries.end())
    {
        delete *it;
        
        _entries.erase(it);
            
        it = _entries.begin();
    }    
}

void EAFontCache::_deleteCacheEntry(EAFontCacheEntry* entry)
{
    if (entry == NULL)
    {
        return;
    }

    std::vector<EAFontCacheEntry*>::iterator it;
    it = _entries.begin();

    while (it != _entries.end())
    {
        if (*it == entry)
        {
            delete *it;
        
            _entries.erase(it);
            
            // Can only have one.
            break;
        }
    
        ++it;
    }    
}

EAFontCacheEntry* EAFontCache::_createCacheEntry()
{
    EAFontCacheEntry* newEntry = new EAFontCacheEntry();
    
    _entries.push_back(newEntry);
    
    return newEntry;
}

EAFontCacheEntry* EAFontCache::getCharacters(const std::string& path)
{
    if (path.empty() == true)
    {
        return NULL;
    }

    std::vector<EAFontCacheEntry*>::iterator it;
    it = _entries.begin();

    // Try to find if we do then increment the references.
    while (it != _entries.end())
    {
        if ((*it)->path() == path)
        {
            // Match so return the value.
            (*it)->incReferences();
            break;
        }
    
        ++it;
    }     
            
    if (it == _entries.end())
    {
        // Not found so want to make new entry.
        EAFontCacheEntry* current = _createCacheEntry();

        if (current != NULL)
        {
            if (current->load(path) == false)
            {
                _deleteCacheEntry(current);
                return NULL;
            } else {
                // Increment the reference count.
                current->incReferences();
                return current;
            }
        } else {
            return NULL;
        }
    } else {          
        return *it;
    }
}

void EAFontCache::returnCharacters(EAFontCacheEntry* entry)
{
    if (entry == NULL)
    {
        return;
    }

    std::vector<EAFontCacheEntry*>::iterator it;
    it = _entries.begin();

    // Try to find if we do then increment the references.
    while (it != _entries.end())
    {
        if ((*it) == entry)
        {
            // Match so decrement the references.
            entry->decReferences();
                       
            if (entry->references() <= 0)
            {
                _deleteCacheEntry(entry);
            }
            
            break;
        }
        
        ++it;
    }    
}
