#ifndef __MPL3115A2_H__
#define __MPL3115A2_H__
#include <mbed.h>

/// Class to provide simple control of the Freescale MPL311A2 pressure and 
/// temperature sensor. The class allows the user to read the pressure as
/// barometric, altimeter or raw and the temperature in Celsius.
///        
/// Reading can be taken either on demand or continuously using either polling
/// or interrupts to wait for the next reading. 
///
/// The correct modes should be get before either calling getReadings for
/// single readings or activate to start continuous readings.
class MPL3115A2 {
public:
    /// Enum for how the pressure data is to be read.
    enum DataMode {
        BarometricMode, ///< Barometric presure in Pascals
        AltimeterMode,  ///< Altitude in meters
        RawMode         ///< Raw values from internal DACs
    };
    
    /// Enum for how consecutive readings are made
    enum ReadMode {
        Single,     ///< Make a single reading when calling getReadings, chip remains in standby the rest of the time
        Polling,    ///< Make consecutive readings by polling isDataReady, chip is active all the time.
        Interupt    ///< Make consecutive readings with an interrupt raised when data is ready, chip is active all the time.
    };
    
    /// Enum to allow specifying which of the chips two interrupt pins to use.
    enum InterruptPin {
        INT1,   ///< Pin called INT1 on chip
        INT2    ///< Pin called INT2 on chip
    };
    
    /// Create a new instance of the class using the specified pins for the I2C bus.
    /// Once initialised the chip is in standby mode.
    /// @param sda The pin name for the sda line of the I2C bus.
    /// @param scl The pin name for the scl line of the I2C bus.
    MPL3115A2(PinName sda, PinName scl);
    
    /// Read the chip id, this is a fixed value of 0xC4.
    /// @returns Chip id.
    int getId(void);
    
    /// Set the how the pressure is read. The default is BarometricMode if to
    /// be change then is must be called before getReadings or activate.
    /// @param mode New mode to read pressure in.
    void setDataMode(DataMode mode);
    
    /// Get the current setting for the pressure reading mode.
    /// @returns Current pressure reading mode.
    DataMode getDataMode(void);
    
    /// Set the how get continuous reading. The default is Single if to
    /// be change then is must be called before getReadings or activate.
    /// @param mode New continuous reading mode.
    void setReadMode(ReadMode mode);
    
    /// Get the current setting for the continuous reading mode.
    /// @returns Current continuous reading mode.
    ReadMode getReadMode(void);
    
    /// Set the number of samples to be used for each reading.
    /// The number must be a power of two between 1 and 128.
    /// @param samples Number of samples per reading.
    void setOverSampling(int samples);
    
    /// Get the number of samples to be used for each reading.
    /// @returns Number of samples per reading.
    int getOverSampling(void);
    
    /// Sets the number of seconds between consecutive readings.
    /// The number must be a power of two between 1 and 32768.
    /// @param period Number of seconds between consecutive readings.
    void setSamplingPeriod(int period);
    
    /// Get the number of seconds between consecutive readings.
    /// @returns Seconds between samples.
    int getSamplingPeriod(void);
    
    /// Set which of the chips two interrupt pins should be used to signal
    /// that the data is ready. It is up to the user to attach the interrupt
    /// handler to the correct MCU pin and then call either getStatus or 
    /// getReadings within the handler to clear the interrupt.
    /// @param intPin Chip interrupt pin to signal data ready interrupts.
    void setDataReadyInt(InterruptPin intPin);
    
    /// Get the configured interrupt pin used to signal that the data 
    /// is ready.
    /// @returns Current data ready interrupt pin.
    InterruptPin getDataReadyInt(void);
    
    /// Read the chips STATUS register. The register contains data ready and
    /// overwrite flags.
    /// @returns Value of the STATUS register.
    int getStatus(void);
    
    /// Returns true if there is data available to read.
    /// @returns True is data is available to read otherwise false.
    bool isDataReady(void);
    
    /// Put the chip into active state to allow taking of consecutive
    /// samples. How often the samples are taken depends on the sampling
    /// period, the maximum frequency is every one second and the minimum
    /// every 9 hours.
    void activate();
    
    /// Gives the active state of the chip
    /// @returns True if the chip is in the active state otherwise false.
    bool isActive();
    
    /// Puts the chip into the standby state and stops automatic readings.
    void standby();
    
    /// Wait for and get the next readings for pressure and temperature.
    /// If the reading mode is Single then the chip is setup and a wait
    /// loop entered until the data is available.
    /// If the reading mode is Polling then the wait loop is entered until
    /// the next reading is available.
    /// If the reading mode is Interrupt then the wait loop will exit imediately
    /// as the data is already available. This will clear the interrupt.
    /// @param pres Reference to variable to hold the pressure reading in Pascals
    /// meters depending on the data mode.
    /// @param temp Reference to variable to hold the temperature reading in
    /// Celsius.
    /// @returns True is the reading was sucessful otherwise false. The reading will
    /// be unsuccessful if the data mode is raw.
    bool getReadings(float &pres, float &temp);
    
private:
    float convTemperature(char *data);
    float convAltimeter(char *data);
    float convPressure(char *data);
    
    void readRegs(char addr, char *data, int len);
    void writeRegs(const char *data, int len);
    
private:
    I2C m_i2c;
    int m_addr;
    
    ReadMode m_readMode;
    DataMode m_dataMode;
    int m_overSample;
    int m_samplePeriod;
    InterruptPin m_dataReadInt;
};

#endif