#include <mbed.h>
#include "mpl3115a2.h"

namespace {
const char STATUS = 0x00;
const char OUT_P_MSB = 0x01;
const char OUT_T_MSB = 0x04;
const char WHO_AM_I = 0x0C;
const char SYSMOD = 0x11;
const char PT_DATA_CFG = 0x13;
const char CTRL_REG1 = 0x26;
const char CTRL_REG2 = 0x27;
const char CTRL_REG3 = 0x28;
const char CTRL_REG4 = 0x29;
const char CTRL_REG5 = 0x2A;

// Values for bits in STATUS
const char STATUS_TDR = 0x02;
const char STATUS_PDR = 0x04;
const char STATUS_TPDR = 0x08;
const char STATUS_DR_MASK = 0x0E;
const char STATUS_TOW = 0x20;
const char STATUS_POW = 0x40;
const char STATUS_TPOW = 0x80;
const char STATUS_OW_MASK = 0xE0;

// Values for bits in PT_DATA_CFG
const char PT_DATA_CFG_TDEFE = 0x01;
const char PT_DATA_CFG_PDEFE = 0x02;
const char PT_DATA_CFG_DREM = 0x04;

// Values for bits in CTRL_REG1
const char CTRL_REG1_SBYB = 0x01;
const char CTRL_REG1_OSB = 0x02;
const char CTRL_REG1_RST = 0x04;
const char CTRL_REG1_OS_MASK = 0x38;
const char CTRL_REG1_RAW = 0x40;
const char CTRL_REG1_ALT = 0x80;

// Values for bits in CTRL_REG2
const char CTRL_REG2_ST_MASK = 0x0F;
const char CTRL_REG2_ALARM_SEL = 0x10;
const char CTRL_REG2_LOAD_OUTPUT = 0x20;

// Values for bits in CTRL_REG4
const char CTRL_REG4_INT_EN_TCHG = 0x01;
const char CTRL_REG4_INT_EN_PCHG = 0x02;
const char CTRL_REG4_INT_EN_TTH = 0x04;
const char CTRL_REG4_INT_EN_PTH = 0x08;
const char CTRL_REG4_INT_EN_TW = 0x10;
const char CTRL_REG4_INT_EN_PW = 0x20;
const char CTRL_REG4_INT_EN_FIFO = 0x40;
const char CTRL_REG4_INT_EN_DRDY = 0x80;

// Values for bits in CTRL_REG5
const char CTRL_REG5_INT_CFG_TCHG = 0x01;
const char CTRL_REG5_INT_CFG_PW = 0x20;
const char CTRL_REG5_INT_CFG_TTH = 0x04;
const char CTRL_REG5_INT_CFG_PTH = 0x08;
const char CTRL_REG5_INT_CFG_TW = 0x10;
const char CTRL_REG5_INT_CFG_PCHG = 0x02;
const char CTRL_REG5_INT_CFG_FIFO = 0x40;
const char CTRL_REG5_INT_CFG_DRDY = 0x80;
};

MPL3115A2::MPL3115A2(PinName sda, PinName scl) :
    m_i2c(sda, scl),
    m_addr(0x60<<1),
    m_readMode(Single),
    m_dataMode(BarometricMode),
    m_overSample(0),
    m_samplePeriod(0),
    m_dataReadInt(INT2)
{
    // Reset the chip so we know where we are.
    char data[2] = { CTRL_REG1, CTRL_REG1_RST};
    writeRegs(data, 2);    
    wait(0.1);
    
    // Configure to get data ready events.
    data[0] = PT_DATA_CFG;
    data[1] = PT_DATA_CFG_TDEFE | PT_DATA_CFG_PDEFE | PT_DATA_CFG_DREM;
    writeRegs(data, 2);    
}

void MPL3115A2::setDataMode(DataMode mode) {
    m_dataMode = mode;
}

MPL3115A2::DataMode MPL3115A2::getDataMode(void) {
    return m_dataMode;
}

void MPL3115A2::setReadMode(ReadMode mode) {
    m_readMode = mode;
}

MPL3115A2::ReadMode MPL3115A2::getReadMode(void) {
    return m_readMode;
}

void MPL3115A2::setOverSampling(int samples) {
    int pow = 0;
    while (((samples % 2) != 1) && (pow < 8)) {
        ++pow;
        samples = samples >> 1;
    }
    m_overSample = pow;
}

int MPL3115A2::getOverSampling(void) {
    return 1 << m_overSample;
}

void MPL3115A2::setSamplingPeriod(int period) {
    int pow = 0;
    while (((period % 2) != 1) && (pow < 16)) {
        ++pow;
        period = period >> 1;
    }
    m_samplePeriod = pow;
}

int MPL3115A2::getSamplingPeriod(void) {
    return 1 << m_samplePeriod;
}

void MPL3115A2::setDataReadyInt(InterruptPin intPin) {
    m_dataReadInt = intPin;
}

MPL3115A2::InterruptPin MPL3115A2::getDataReadyInt(void) {
    return m_dataReadInt;
}

int MPL3115A2::getId(void) {
    char id;
    
    // Read the WHO_AM_I register to get the value.
    readRegs(WHO_AM_I, &id, 1);
    return id;
}

int MPL3115A2::getStatus(void) {
    char status;
    
    readRegs(STATUS, &status, 1);
    return status;
}

bool MPL3115A2::isDataReady(void) {
    char status;
    
    // Read the STATUS register and and in the data ready mask
    // to get a boolean.
    readRegs(STATUS, &status, 1);
    return (status & STATUS_DR_MASK) != 0;
};

void MPL3115A2::activate() {
    if (m_readMode != Single) {
        char reg1 = CTRL_REG1_SBYB;
        char reg2 = 0;
        char reg4 = 0;
        char reg5 = 0;
        
        // Set the data mode  and over sampling correctly
        switch (m_dataMode) {
        case BarometricMode:
            // nothing to set.
            break;
            
        case AltimeterMode:
            reg1 |= CTRL_REG1_ALT;
            break;
            
        case RawMode:
            reg1 |= CTRL_REG1_RAW;
            break;
        }
        reg1 |= (m_overSample << 3) & CTRL_REG1_OS_MASK;
        
        // Set the sampling period
        reg2 = m_samplePeriod & CTRL_REG2_ST_MASK;
        
        // If we are using interupts then configure the
        // interrupt registers.
        if (m_readMode == Interupt) {
            reg4 = CTRL_REG4_INT_EN_DRDY;
            if (m_dataReadInt == INT1) {
                reg5 = CTRL_REG5_INT_CFG_DRDY;
            }
        }
        
        char data[6] = { CTRL_REG1, reg1, reg2, 0, reg4, reg5 };
        writeRegs(data, 6);
    }
}

bool MPL3115A2::isActive() {
    char status;
    
    // read the active status from the SYSMOD register
    readRegs(SYSMOD, &status, 1);
    return status != 0;
}

void MPL3115A2::standby() {
    
    // Clear the CTRL_REG1 to put the chip into standby.
    char data[2] = { CTRL_REG1, 0 };
    writeRegs(data, 2);
}

    
bool MPL3115A2::getReadings(float &pres, float &temp) {
    if (m_dataMode == RawMode) {
        // Raw mode not supported.
        return false;
    }
    
    // If we're in single read mode then we need to initiate 
    // a new read cycle.
    if (m_readMode == Single) {
        // Set the OSB bit to initiate a new reading.
        char value = CTRL_REG1_OSB;
        
        // Add in oversampling
        value |= (m_overSample << 3) & CTRL_REG1_OS_MASK;
        
        // Set the pressure reading mode to altimeter if needed.
        if (m_dataMode == AltimeterMode) {
            value |= CTRL_REG1_ALT;
        }
        
        // Write the new value to CTRL_REG1 to kick things off.
        char data[2] = { CTRL_REG1, value};
        writeRegs(data, 2);    
    }
    
    int status = 0;
    
    // Loop until the data is ready.
    while ((status & STATUS_DR_MASK) == 0) {
        status = getStatus();
        printf("0x%X-", status);
    }
    
    // Read the pressure and temperature data all at once
    // and then convert it into real values.
    char data[5];
    readRegs(OUT_P_MSB, &data[0], 5);
    if (m_dataMode == AltimeterMode) {
        pres = convAltimeter(&data[0]);
    } else {
        pres = convPressure(&data[0]);
    }
    temp = convTemperature(&data[3]);
    
    return true;
}


float MPL3115A2::convTemperature(char *data) {
    short temp;
    float ftemp;
    
    // Move the signed byte up to the top of the short so
    // that it is in the right place and or in the bottom
    // 8bits.
    temp = (data[0] << 8) | data[1];
    
    // shift out the bottom 4 0 bits and convert
    // the integer temp into a float value by
    // dividing by 16.
    ftemp = (temp >> 4) / 16.0f;
    return ftemp;
}

float MPL3115A2::convAltimeter(char *data) {
    int altInt;
    float altFloat;
    
    // Put the 3 byte value in the upper 24 bits so the sign
    // is correct.
    altInt = (data[0] << 24) | (data[1] << 16) | (data[2] << 8);
    
    // Shift down extending the sign as we go and then
    // divide by 16 to get fractions.
    altFloat = (altInt >> 12) / 16.0f;
    
    return altFloat;
}

float MPL3115A2::convPressure(char *data) {
    int presInt;
    float fprs;
    
    // Or in the three bytes containing the 18bit int
    // into the top 18bits so the sign bit is correct.
    presInt = (data[0] << 24) | (data[1] << 16) | (data[2] << 8);

    // Shift the in into the correct position and then
    // divide by 4 to generate the fractional part.
    fprs = (presInt >> 12) / 4.0f;
    
    return fprs;
}

void MPL3115A2::readRegs(char addr, char *data, int len) {
    m_i2c.write(m_addr, &addr, 1, true);
    m_i2c.read(m_addr, data, len);
}

void MPL3115A2::writeRegs(const char *data, int len) {
    m_i2c.write(m_addr, data, len);
}

