#ifndef PRCMPL3115A2_H
#define RPCMPL3115A2_H

#include "rpc.h"
#include "mpl3115a2.h"

/// RPC wrapper class for MPL3115A2 driver class.
/// This class provides a simple single shot interface to the 
/// MPL3115A2 allowing the user to get the altimeter/barometric
/// mode and number of samples per reading.
class RpcMPL3115A2 : public mbed::RPC {
public:
    /// Create a new instance of the class using the specified pins for the I2C bus.
    /// @param sda The pin name for the sda line of the I2C bus.
    /// @param scl The pin name for the scl line of the I2C bus.
    RpcMPL3115A2(PinName sda, PinName scl, const char *name=NULL);

    /// Read the chip id, this is a fixed value of 0xC4. This method is
    /// linked to the RPC method method "id".
    /// @returns Chip id.
    int id(void);
    
    /// Set the how the pressure is read. The default is BarometricMode if 
    /// it is to be change then is must be called before getReadings or activate.
    /// This method is linked to to the RPC method "writeMode".
    /// @param mode New mode to read pressure in.
    void setMode(int mode);

    /// Get the current setting for the pressure reading mode.
    /// This method is linked to the RPC method "readMode".
    /// @returns Current pressure reading mode.
    int getMode(void);
    
    /// Set the number of samples to be used for each reading.
    /// The number must be a power of two between 1 and 128.
    /// This method is linked to the RPC method "writeSamples".
    /// @param samples Number of samples per reading.
    void setOverSampling(int rate);
    
    /// Get the number of samples to be used for each reading.
    /// This method is linked to the RPC method "readSamples".
    /// @returns Number of samples per reading.
    int getOverSampling();
    
    /// Reads both pressure and temperature. Returns pressure 
    /// in the units set by mode and the temperature in Celsius,
    /// the values are separated by a space.
    /// This method is linked to the RPC method "read".
    void read(Arguments*, Reply*);

    /// Override of RPC method to return array of methods
    /// that can be called through RPC.
    /// @returns array of method structures terminated by a
    /// null structure.
    virtual const struct rpc_method *get_rpc_methods();
    
    /// Override of RPC method to return the RPC name of the class
    /// for this object.
    /// @returns RPC name of class for this instance.
    virtual const char *get_rpc_class_name();
    
    /// Get the predefined RPC class structure for this
    /// class. Allows using template to register class
    /// this RPC.
    /// @returns Point to RPC class information structure.
    static struct rpc_class *get_rpc_class();
    
private:
    MPL3115A2 chip;
};

#endif