#pragma once
#include "mbed.h"

// small wrapper for an endless rotary encoder via ISR on clk pin
// attach a callback which sends -1 on left turn and +1 on right turn
// if clk pin is shared, temporary disable IRQ via enable() function
class RotaryEncoder
{
public:
  RotaryEncoder(PinName clk, PinName dt, Callback<void(int)> fnCB)
    : _clk(clk)
    , _dt(dt)
    , _fnCB(fnCB)
    {
        _clk.mode(PullNone);  // device is already PullUp
        _clk.fall([this]()
            {
                _dtState = (integrate() << 1) > 1<<_readDTCountShift;
            });
        // a very naive implementation also works with first IRQ flank, but it's very instable
        _clk.rise([this]()
            {
                bool bIn =  (integrate() << 1) > 1<<_readDTCountShift;
                if(_dtState && !bIn)
                    _fnCB(1);
                else if(!_dtState && bIn)
                    _fnCB(-1);
            });
    }

    void enable(bool bEnable)
    {
        bEnable ? _clk.enable_irq() : _clk.disable_irq();
    }
    
private:
    
    // read _dt pin multiple times to reduce jitter
    int integrate() const
    {
        int n = 0;
        for(int i = 0; i < 1<<_readDTCountShift; ++i)
            n += _dt.read();
        return n;
    }
    
    const int _readDTCountShift = 4;
    InterruptIn _clk;
    mutable DigitalIn _dt;
    Callback<void(int)> const _fnCB;
    bool _dtState;
};