#pragma once
#include "mbed.h"

 //! Set to 1 to enable backbuffering, prefer when using a lot of pixel drawings, which should be updated all at once, only tranfers pages that have been changed.
 //! Advantage: Highest update speed; detects pages which are up2date.
 //! Disadvantage: increased RAM usage; backbuffer synchronization
#define USE_BACKBUFFER 11

//! on LPC if ETH is not used, use 2. page of SRAM for buffers
#define LPC1768_USE_ETH_BUFFER 1
 
 // Some measures @400kHz I2C freq:
 // Copy whole display buffer: ~40fps
 // Render 1 page of text and copy to display: ~220fps
 // Render 2 pages of text and copy to display: ~130fps
 // Still able to work with 1.2MHz I2C freq:
 // Copy whole display buffer: ~90fps
 // Render 1 page of text and copy to display: ~420fps
 // Render 2 pages of text and copy to display: ~280fps

/**
    Create an object to communicate with a SH1106 display.
    This class is not a generic library but tries to get the most out of the hardware.
 */
class I2CDisplay_SH1106
{
public:
    enum class Direction
    {
        HorizontalUp,
        HorizontalDown,
        VerticalUp,
        VerticalDown
    };
    enum class PixelUpdateMode : uint8_t
    {
        DirectDraw,    //!< high refresh speed for small area updates, largest transfer overhead
        CacheOnly,     //!< just write to cache memory,  updateDisplay() must be explictly called
        UpdateDisplay, //!< many pixels are transferred at once, good for updating whole display
    };

    enum class I2CSpeed : uint8_t
    {
        Normal,     //!<  100kHz; just for compatibility, e.g. if multiple device are on same bus with incompatible speeds)
        Fast,       //!<  400kHz specification
        FastPlus,   //!< 1000kHz; high speed spec
        SuperFast,  //!< 1200kHz; out of specs, but still works; above this freq bits get lost
    };
    
    /**
         Create and initializes the object.
         Sets the maximum frequency and some other display options.
         Clears the memory and switches the display on (if not cleared, random pixels are set).
         @param sda  the I2C SDA pin
         @param scl the IC2 clock pin
         @param address the address of the device
         @param contrast the initial contrast of the display
         @param speed the speed of the I2C bus
     */
    I2CDisplay_SH1106(PinName sda=I2C_SDA1, PinName scl=I2C_SCL1, uint8_t address=0x78, uint8_t contrast=0, I2CSpeed speed=I2CSpeed::FastPlus);
    ~I2CDisplay_SH1106();

    enum class PlotterMode : uint8_t
    {
        No,
        Directional1, // left to right
        Directional2, // alternating
    };
    /**
        In most cases changes are written to local memory first. This function is required to force an update to the display.
        @param pm The style to refresh the display
        @param plotterDelay Just applies if pm is set, defines the "speed" of the plot
     */
    void update(PlotterMode pm=PlotterMode::No, uint16_t plotterDelay=1000);
    /**
        Clear the display
     */
    void clear();
    
    /**
        Copies an in memory bitmap to the display. The bitmap must have size of the display
    */
    void copyBitmap(char const* pBmp, bool forceUpdate = true, PlotterMode pm=PlotterMode::No, uint16_t plotDelay=1000);
    /**
        Sets or clears a pixel at any display position.
        No parameter checks, know what you do.
     */
    void setPixel(uint8_t x, uint8_t y, bool bSet, PixelUpdateMode m = PixelUpdateMode::CacheOnly);
    /**
        test of scrolling, still in alpha version
     */    
    void scroll(uint8_t distance, uint32_t delayMS=20, uint8_t step=1);

    //void writeText(char column, char page, const char *font_address, const char *text, const uint8_t size);
    /** renders a single char to the default buffer with custom font (currently just Adafruit glcdfont)
     */
    void renderChar(uint8_t x, uint8_t y, char c);
    /** renders a line of text to the default buffer with custom font (currently just Adafruit glcdfont)
     */
    void renderLine(uint8_t x, uint8_t y, char const* t, uint8_t len);

    void selectFont(const char* font, uint8_t w, uint8_t h);
    
// direct display commands    
public:
    
    /** inverts the display, no effect on buffer
    */
    void invert(bool i);
    
    /** define the pixel intensity, range 0-255
    */
    void contrast(uint8_t c);

    /** how the display is built in
    */
    void direction(Direction dir);
    
    /** enable/disable display output
    */
    void enable(bool enable);

    // readable from outside, e.g. if required for loops    
    uint8_t const _width;
    uint8_t const _height;
};
