#include "I2CDisplay_SH1106.h"
#include "hal/i2c_api.h"

#define SSD1106_SETLOWCOLUMN 0x00
#define SSD1106_SETHIGHCOLUMN 0x10
#define SSD1106_SETSTARTLINE 0x40
#define SSD1106_SETCONTRAST 0x81
#define SSD1106_CHARGEPUMP 0x8D  // SSD1106_EXTERNALVCC (don't know the effect)
#define SSD1106_SEGREMAP 0xA0      // effect not known
#define SSD1106_DISPLAYALLON_RESUME 0xA4
#define SSD1106_DISPLAYALLON 0xA5
#define SSD1106_NORMALDISPLAY 0xA6
#define SSD1106_INVERTDISPLAY 0xA7
#define SSD1106_SETMULTIPLEX 0xA8
#define SSD1106_DISPLAYOFF 0xAE
#define SSD1106_DISPLAYON 0xAF
#define SSD1106_SETPAGEADDRESS 0xB0
#define SSD1106_COMSCANINC 0xC0 // output scan direction (flip display v)
#define SSD1106_COMSCANDEC 0xC8 // output scan direction (page0 is where pins are)
#define SSD1106_SETDISPLAYOFFSET 0xD3
#define SSD1106_SETDISPLAYCLOCKDIV 0xD5
#define SSD1106_SETPRECHARGE 0xD9
#define SSD1106_SETCOMPINS 0xDA
#define SSD1106_SETVCOMDETECT 0xDB

#define SSD1106_READMODIFYWRITE_BEGIN 0xE0
#define SSD1106_READMODIFYWRITE_END 0xEE
#define SSD1106_EXTERNALVCC 0x1
#define SSD1106_SWITCHCAPVCC 0x2

#define WIDTH 128
#define HEIGHT 64
#define NUM_PAGES (HEIGHT/8)
#define RAW_OFFSET (1 + 2)
#define RAW_WIDTH (WIDTH+RAW_OFFSET+2)

#define DATAMODE 0x40
#define COMMANDMODE 0x00

#if LPC1768_USE_ETH_BUFFER
// there are 2 section 16kB each, which can be adressed by AHBSRAMx. Normally used for CAN/Ethernet.
// As second section is directly behind the allocated buffer can overlap
#if USE_BACKBUFFER
static char internalData[133*64/8*2+32] __attribute__((section("AHBSRAM0")));
#else
static char internalData[133*64/8+32] __attribute__((section("AHBSRAM0")));
#endif 
#else //LPC1768_USE_ETH_BUFFER
#if USE_BACKBUFFER
static char internalData[133*64/8*2+32];
#else
static char internalData[133*64/8+32];
#endif 
#endif

struct I2CWrapper : public i2c_t
{
    inline void command(uint8_t c)
    {
        char data[2] { COMMANDMODE, c };
        i2c_write(this, _address, data, sizeof(data), 1);
    }

    inline void command(uint8_t c1, uint8_t c2) // a 2 byte command
    {
        char data[3] { COMMANDMODE, c1, c2 };
        i2c_write(this, _address, data, sizeof(data), 1);
    }

    void commands(uint8_t count, uint8_t const* pCommands) // multiple commands at once
    {
        char data[1] { COMMANDMODE };
        i2c_write(this, _address, data, sizeof(data), 0); // activate command mode
        // send all commands
        for (uint8_t i = 0; i < count; ++i) {
            i2c_byte_write(this, pCommands[i]);
        }

        i2c_stop(this); // finished
    }

    void copyBitmap(char const* pBmp)
    {
        auto pBuffer = (char*)displayBuffer+RAW_OFFSET;
        for(uint8_t r=0; r<NUM_PAGES; ++r, pBuffer += RAW_WIDTH, pBmp += WIDTH)
            std::copy(pBmp, pBmp+WIDTH, pBuffer);
#if USE_BACKBUFFER
        _backBufferSkipped = true;
#endif
    }

    uint8_t* getBuffer()
    {
    #if USE_BACKBUFFER
        if(_backBufferSkipped)
        {
            std::copy(displayBuffer, displayBuffer+sizeof(displayBuffer), displayBackBuffer);
            _backBufferSkipped = false;
        }
        return displayBackBuffer;
    #else
        return displayBuffer;
    #endif
    }

    // transfers a complete page (RAW_WIDTH bytes)
    inline void transferPage(uint8_t pageIdx, char const* pPage)
    {
        command(SSD1106_SETPAGEADDRESS | pageIdx);
        i2c_write(this, _address, pPage, RAW_WIDTH, 1);
    }
    
    // copies data from displayBuffer to I2C paged memory
    void update()
    {
        command(SSD1106_SETSTARTLINE | 0x0);
        auto pPage = (char*)displayBuffer;
#if USE_BACKBUFFER
        if(_backBufferSkipped)
        {
            for(uint8_t r=0; r<NUM_PAGES; ++r, pPage += RAW_WIDTH)
            {
                transferPage(r, pPage);
            }
            return;
        }
#endif        
        for(uint8_t r=0; r<NUM_PAGES; ++r, pPage += RAW_WIDTH)
        {
#if USE_BACKBUFFER
            // internal SRAM is fast, so compare before transfer
            auto pBackPage = (char*)displayBackBuffer+r*RAW_WIDTH;
            if(std::equal(pPage, pPage+RAW_WIDTH, pBackPage))
                continue; // nothing to copy, next page please
            std::copy(pBackPage, pBackPage+RAW_WIDTH, pPage);
#endif
            transferPage(r, pPage);
        }
    };
    
    inline void beginReadModifyWrite(uint8_t x, uint8_t page)
    {
        uint8_t const commandsToSend[] {
               static_cast<uint8_t>(SSD1106_SETPAGEADDRESS | page)
             , static_cast<uint8_t>(SSD1106_SETLOWCOLUMN | (x & 0xF))
             , static_cast<uint8_t>(SSD1106_SETHIGHCOLUMN | (x >> 4))
             , SSD1106_READMODIFYWRITE_BEGIN };
        commands(sizeof(commandsToSend), commandsToSend);
    }

    inline void endReadModifyWrite()
    {
        command(SSD1106_READMODIFYWRITE_END);
    }
    
    inline void setPixel(uint8_t x, uint8_t y, bool bSet, I2CDisplay_SH1106::PixelUpdateMode m = I2CDisplay_SH1106::PixelUpdateMode::CacheOnly)
    {
        uint8_t page = y >>  3;
        uint8_t yMod = y & 7;
        x += RAW_OFFSET;
        uint8_t& v = getBuffer()[x + page*RAW_WIDTH];

#define _BV(bit) (1<<(bit))
        v = (v & ~_BV(yMod)) | bSet * _BV(yMod);
#undef _BV
    
        switch(m)
        {
        case I2CDisplay_SH1106::PixelUpdateMode::DirectDraw:
        {
            beginReadModifyWrite(x, page);
            char c[2] { DATAMODE, (char)v };   // write 1 byte in data mode
            i2c_write(this, _address, c, sizeof(c), 1);
            endReadModifyWrite();
            break;
        }
        case I2CDisplay_SH1106::PixelUpdateMode::UpdateDisplay:
            update();
            break;
        case I2CDisplay_SH1106::PixelUpdateMode::CacheOnly: break;
        }
    }

    void updatePlotterMode(int pm, uint16_t plotUpdateDelay)
    {
        uint8_t xS = RAW_OFFSET;
        uint8_t xE = WIDTH+RAW_OFFSET;
        int8_t dir = 1;
        for(uint8_t page = 0; page < 8; ++page)
        {
            uint8_t* pBuffer = displayBuffer+page*RAW_WIDTH;
            if(dir == 1)
            {
                for(uint8_t x = xS; x < xE; ++x)
                {
                    beginReadModifyWrite(x, page);  // unfortunately we can't do this outside loop, because we need to switch between command and data mode
                    // draw plotter pen
                    char c[2] { DATAMODE, pBuffer[x] };
                    i2c_write(this, _address, c, sizeof(c), 1);
                    endReadModifyWrite();
                    wait_us(plotUpdateDelay);
                }
            }
            else
            {
                for(uint8_t x = xE-1; x >= xS; --x)
                {
                    beginReadModifyWrite(x, page);  // unfortunately we can't do this outside loop, because we need to switch between command and data mode
                    // draw plotter pen
                    char c[2] { DATAMODE, 0xFF };
                    //i2c_write(this, _address, c, sizeof(c), 1);
                    wait_us(plotUpdateDelay);  // if this value is too small, we won't see the 'plotter' position
                    // draw pixels below pen
                    c[1] = pBuffer[x];
                    i2c_write(this, _address, c, sizeof(c), 1);
                    endReadModifyWrite();
                }
            }
            
            if(pm == 2)
                dir = -dir;
        }
    }

    void drawFontColumn(uint8_t x, uint8_t y, uint16_t fontPos)
    {
        auto c = pFont[fontPos];
        for(uint8_t j=0; j<hFont; ++j)
            setPixel(x, y + j, c & (1<<j), I2CDisplay_SH1106::PixelUpdateMode::CacheOnly);
    }
    
    uint8_t displayBuffer[RAW_WIDTH*NUM_PAGES];
#if USE_BACKBUFFER
    uint8_t displayBackBuffer[RAW_WIDTH*NUM_PAGES];
    bool _backBufferSkipped;
#endif
    uint8_t _address;
    const char* pFont;
    uint8_t wFont;
    uint8_t hFont;
};

I2CDisplay_SH1106::I2CDisplay_SH1106(PinName sda, PinName scl, uint8_t address, uint8_t contrast, I2CSpeed speed)
    : _width(WIDTH)
    , _height(HEIGHT)
{
    static_assert(sizeof(I2CWrapper) <= sizeof(internalData), "internalData too small");
    I2CWrapper* pI2C = reinterpret_cast<I2CWrapper*>(internalData);

    i2c_init(pI2C, sda, scl);
    const uint8_t freqMul[] = {10, 40, 100, 120};
    i2c_frequency(pI2C, 10000*freqMul[static_cast<int>(speed)]);
    const bool bExternalVCC = true;

    uint8_t const commandsToSend[] {
          SSD1106_DISPLAYOFF
        , SSD1106_SETDISPLAYCLOCKDIV, 0x80                                // the suggested ratio 0x80
        , SSD1106_SETMULTIPLEX, static_cast<uint8_t>(_height-1)
        , SSD1106_SETDISPLAYOFFSET, 0x00                                  // no offset
        , SSD1106_SETSTARTLINE | 0x0            // line #0
        , SSD1106_CHARGEPUMP, bExternalVCC ? 0x10 : 0x14
        , SSD1106_SEGREMAP | 0x1
        , SSD1106_COMSCANDEC
        , SSD1106_SETCOMPINS, static_cast<uint8_t>(_height == 32 ? 0x02 : 0x12)
        , SSD1106_SETCONTRAST, contrast
        , SSD1106_SETPRECHARGE, bExternalVCC ? 0x22 : 0xF1 // copied, not tested
        , SSD1106_SETVCOMDETECT, 0x40
        , SSD1106_DISPLAYALLON_RESUME
        , SSD1106_NORMALDISPLAY };

    pI2C->_address = address;
    pI2C->commands(sizeof(commandsToSend), commandsToSend);

    clear();
    pI2C->update();
    pI2C->command(SSD1106_DISPLAYON);
}

I2CDisplay_SH1106::~I2CDisplay_SH1106()
{
    I2CWrapper* pI2C = reinterpret_cast<I2CWrapper*>(internalData);
    i2c_reset(pI2C);
}

void I2CDisplay_SH1106::setPixel(uint8_t x, uint8_t y, bool bSet, PixelUpdateMode m)
{
    I2CWrapper* pI2C = reinterpret_cast<I2CWrapper*>(internalData);
    pI2C->setPixel(x, y, bSet, m);
}

void I2CDisplay_SH1106::update(I2CDisplay_SH1106::PlotterMode pm, uint16_t plotUpdateDelay)
{
    I2CWrapper* pI2C = reinterpret_cast<I2CWrapper*>(internalData);
    pm != I2CDisplay_SH1106::PlotterMode::No
         ? pI2C->updatePlotterMode(static_cast<int>(pm), plotUpdateDelay)
         : pI2C->update();
}

void I2CDisplay_SH1106::clear()
{
    I2CWrapper* pI2C = reinterpret_cast<I2CWrapper*>(internalData);
    // clear buffer and set first column to datamode
    uint8_t* pBuffer;
#if USE_BACKBUFFER
    pI2C->_backBufferSkipped = false;
    pBuffer = pI2C->displayBackBuffer;
#else
    pBuffer = pI2C->displayBuffer;
#endif
    std::fill(pBuffer, pBuffer+sizeof(I2CWrapper::displayBuffer), 0x00);
    for(int i=0; i<NUM_PAGES; ++i)
      pBuffer[i*RAW_WIDTH] = DATAMODE;
}

void I2CDisplay_SH1106::scroll(uint8_t distance, uint32_t delayMS, uint8_t step)
{
    I2CWrapper* pI2C = reinterpret_cast<I2CWrapper*>(internalData);
    for(uint8_t y=0;y<distance; y+=step)
    {
        pI2C->command(SSD1106_SETSTARTLINE | y);
        thread_sleep_for(delayMS);
    }
    // TODO: synchronize with local buffer
}

void I2CDisplay_SH1106::copyBitmap(char const* pBmp, bool forceUpdate, I2CDisplay_SH1106::PlotterMode pm, uint16_t plotDelay)
{
    I2CWrapper* pI2C = reinterpret_cast<I2CWrapper*>(internalData);
    pI2C->copyBitmap(pBmp);
    if(forceUpdate)
        update(pm, plotDelay);
}

void I2CDisplay_SH1106::contrast(uint8_t c)
{
    I2CWrapper* pI2C = reinterpret_cast<I2CWrapper*>(internalData);
    pI2C->command(SSD1106_SETCONTRAST, c);
}

void I2CDisplay_SH1106::enable(bool enable)
{
    I2CWrapper* pI2C = reinterpret_cast<I2CWrapper*>(internalData);
    pI2C->command(enable ? SSD1106_DISPLAYON : SSD1106_DISPLAYOFF);
}

void I2CDisplay_SH1106::renderChar(uint8_t x, uint8_t y, char c)
{
    I2CWrapper* pI2C = reinterpret_cast<I2CWrapper*>(internalData);
    uint16_t pos = ((uint16_t)c)*pI2C->wFont;
    for(uint8_t i=0; i<pI2C->wFont; ++i)
    {
        pI2C->drawFontColumn(x + i, y, pos + i);
    }
}

void I2CDisplay_SH1106::renderLine(uint8_t x, uint8_t y, char const* t, uint8_t len)
{
    I2CWrapper* pI2C = reinterpret_cast<I2CWrapper*>(internalData);
    auto xMax = min(len*(pI2C->wFont + 1), WIDTH);
    for(; x<xMax; x+=pI2C->wFont + 1, ++t)
    {
        renderChar(x, y, *t);
        pI2C->drawFontColumn(x+pI2C->wFont, y, 0);
    }
}

void I2CDisplay_SH1106::selectFont(const char* font, uint8_t w, uint8_t h)
{
    I2CWrapper* pI2C = reinterpret_cast<I2CWrapper*>(internalData);
    pI2C->pFont = font;
    pI2C->wFont = w;
    pI2C->hFont = h;
}

void I2CDisplay_SH1106::direction(Direction dir)
{
    I2CWrapper* pI2C = reinterpret_cast<I2CWrapper*>(internalData);
    pI2C->command(static_cast<int>(dir) & 1 ? SSD1106_COMSCANDEC : SSD1106_COMSCANINC
                 ,static_cast<int>(dir) & 1 ? SSD1106_COMSCANDEC : SSD1106_COMSCANINC);
}
