#include "MMA8451Q.h"

#define REG_STATUS        0x00
#define REG_WHO_AM_I      0x0D
#define REG_CTRL_REG_1    0x2A
#define REG_CTRL_REG_2    0x2B
#define REG_CTRL_REG_4    0x2D
#define REG_CTRL_REG_5    0x2E
#define REG_INT_SRC       0x0C
#define REG_FF_MT_CFG     0x15
#define REG_FF_MT_SRC     0x16
#define REG_FF_MT_THS     0x17
#define REG_FF_MT_CNT     0x18
#define REG_DBCNTM        0x11
#define REG_DBNCE         0x12
#define REG_BKFR          0x13
#define REG_P_L_THS       0x14
#define REG_PL_STATUS     0x10

#define REG_OUT_X_MSB     0x01
#define REG_OUT_Y_MSB     0x03
#define REG_OUT_Z_MSB     0x05

#define UINT14_MAX        16383

#define ZYXDR           0x08
#define ZDR             0x04
#define YDR             0x02
#define XDR             0x01

void (*MMA8451Q_usr2_fptr)(void);               // Pointers to user function called after
void (*MMA8451Q_usr1_fptr)(void);               // IRQ assertion.

//
InterruptIn MMA8451Q_Int1( PTA14);      // INT1
InterruptIn MMA8451Q_Int2( PTA15);      // INT2

MMA8451Q::MMA8451Q(PinName sda, PinName scl, int addr) : m_i2c(sda, scl), m_addr(addr) {
    
    MMA8451Q_Int1.fall( NULL);
    MMA8451Q_Int2.fall( NULL);
    MMA8451Q_usr2_fptr = NULL;
    MMA8451Q_usr1_fptr = NULL;    

    Reset();    
    Active();    
}

MMA8451Q::~MMA8451Q() 
{
     MMA8451Q_Int1.fall( NULL);
     MMA8451Q_Int2.fall( NULL);
     MMA8451Q_usr2_fptr = NULL;
     MMA8451Q_usr1_fptr = NULL;
}

void MMA8451Q::Reset( void)
{
    // Soft reset
    uint8_t data[2] = {REG_CTRL_REG_2, 0x40};
    writeRegs(data, 2);
    wait( 0.1);
}


void MMA8451Q::MotionDetection( void(*fptr)(void))
{
    Reset();
    
    //data sheet MMA8451Q.pdf strana 45
    // 6.1 Example Steps for Configuring Motion Detection
    // X or Y > 3g using MFF Function 4g, 100 Hz ODR, Normal Mode
    // Step 1: Put the device into Standby Mode: Register 0x2A CTRL_REG1
    unsigned char data[2] = {REG_CTRL_REG_1, 0x18}; // Set the device in 100 Hz ODR, Standby
    writeRegs(data, 2);

    //data sheet MMA8451Q.pdf strana 32
    // Step 2: Set Configuration Register for Motion Detection by setting the “OR” condition OAE = 1, enabling
    // X, Y, and the latch
    data[0] = REG_FF_MT_CFG;
    data[1] = 0xD8;
    writeRegs(data, 2);

    //data sheet MMA8451Q.pdf strana 34
    // Step 3: Threshold Setting Value for the Motion detection of > 2g
    // Note: The step count is 0.063g/ count
    // • 1g/0.063g = 15.8; //Round up to 16
    data[0] = REG_FF_MT_THS;
    data[1] = 0x08;
    writeRegs(data, 2);
    
    //data sheet MMA8451Q.pdf strana 32
    // Step 4: Set the debounce counter to eliminate false readings for 100 Hz sample rate with a requirement
    // of 100 ms timer.
    // Note: 100 ms/10 ms (steps) = 10 counts
    data[0] = REG_FF_MT_CNT;
    data[1] = 0x0A;
    writeRegs(data, 2);
    
    //data sheet MMA8451Q.pdf strana 48
    // Step 5: Enable Motion/Freefall Interrupt Function in the System (CTRL_REG4)
    data[0] = REG_CTRL_REG_4;
    data[1] = 0x04;
    writeRegs(data, 2);
    
    //data sheet MMA8451Q.pdf strana 48
    // Step 6: Route the Motion/Freefall Interrupt Function to INT2 hardware pin (CTRL_REG5)
    data[0] = REG_CTRL_REG_5;
    data[1] = 0x00;
    writeRegs(data, 2);
    
    
    // Step 7: Put the device in Active Mode
    data[0] = REG_CTRL_REG_1;
    data[1] = 0x19;
    writeRegs(data, 2);

    MMA8451Q_usr2_fptr = fptr;
    MMA8451Q_Int2.fall( this, &MMA8451Q::Motion_IRQ);
}


void MMA8451Q::Motion_IRQ( void)
{
    unsigned char t;
    
    readRegs( REG_INT_SRC, &t, 1);
    //
    if ( (t & 0x04) == 0x04) {
        
        readRegs( REG_FF_MT_SRC, &t, 1);
     
        MMA8451Q_usr2_fptr();
    }
}

void MMA8451Q::Active( void)
{
    unsigned char t;
    
    // Activate the peripheral
    readRegs(REG_CTRL_REG_1, &t, 1);
    unsigned char data[2] = {REG_CTRL_REG_1, t|0x01};
    writeRegs(data, 2);
}

void MMA8451Q::Standby( void)
{
    unsigned char t;
    
    // Standby
    readRegs(REG_CTRL_REG_1, &t, 1);
    unsigned char data[2] = {REG_CTRL_REG_1, t&0xFE};
    writeRegs(data, 2);
}

float MMA8451Q::getAccX() {
    return (float(getAccAxis(REG_OUT_X_MSB))/4096.0);
}

float MMA8451Q::getAccY() {
    return (float(getAccAxis(REG_OUT_Y_MSB))/4096.0);
}

float MMA8451Q::getAccZ() {
    return (float(getAccAxis(REG_OUT_Z_MSB))/4096.0);
}

void MMA8451Q::getAccAllAxis(float * res) {
    res[0] = getAccX();
    res[1] = getAccY();
    res[2] = getAccZ();
}

int16_t MMA8451Q::getAccAxis(uint8_t addr) {
    int16_t acc;
    uint8_t res[2];
    readRegs(addr, res, 2);

    acc = (res[0] << 6) | (res[1] >> 2);
    if (acc > UINT14_MAX/2)
        acc -= UINT14_MAX;

    return acc;
}

void MMA8451Q::readRegs(int addr, uint8_t * data, int len) {
    char t[1] = {addr};
    m_i2c.write(m_addr, t, 1, true);
    m_i2c.read(m_addr, (char *)data, len);
}

void MMA8451Q::writeRegs(uint8_t * data, int len) {
    m_i2c.write(m_addr, (char *)data, len);
}
