/* mbed Microcontroller Library
 * Copyright (c) 2017 u-blox
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
 
 
/* This example program for the u-blox C030 board.
 * The UbloxATCellularInterface or OnboardCellularInterface uses it
 * to make a simple HTTP connection to a STC IOT server.
 * Progress may be monitored with a serial terminal running at 9600 baud or at https://ksu.aep1.iot.stcs.com.sa 
 * The LED on the C030 board will turn green when this program is
 * operating correctly, pulse blue when a sockets operation is completed
 * and turn red if there is a failure.
 
 * Modified by: Muhammad Faheem Khan (Digital Solutions, STCS) 
 */
 
#include "mbed.h"
#include "UbloxATCellularInterface.h"
#include "UbloxATCellularInterfaceN2xx.h"
#include "LM75B.h"   // Temperature Sensor Library. Import more libraries from https://os.mbed.com/components/mbed-Application-Shield/
#include "C12832.h"  // LCD Library. Import more libraries from https://os.mbed.com/components/mbed-Application-Shield/

// Sensor Pin and Object Activation
C12832 lcd(D11, D13, D12, D7, D10);
LM75B sensor(D14,D15); // 1st Temperature sensor
//AnalogIn pot2 (A1);   // 2nd Potentiometer sensor
AnalogIn moisture(A0);
#define INTERFACE_CLASS  UbloxATCellularInterface  //This interface is used to initiate modem commands for sending data to sever

// The credentials of the SIM in the board.  If PIN checking is enabled
// for your SIM card you must set this to the required PIN.
#define PIN "0000"
#define APN         "jtm2m"   //APN remains the same for all the kits
#define USERNAME    NULL
#define PASSWORD    NULL
#define TCP_SERVER "integration.campus.stcs.io"   //STC IoT parsing server address

// LEDs
DigitalOut ledRed(LED1, 1);
DigitalOut ledGreen(LED2, 1);
DigitalOut ledBlue(LED3, 1);

// The user button
volatile bool buttonPressed = false;


float map(float in, float inMin, float inMax, float outMin, float outMax) {
  // check it's within the range
  if (inMin<inMax) { 
    if (in <= inMin) 
      return outMin;
    if (in >= inMax)
      return outMax;
  } else {  // cope with input range being backwards.
    if (in >= inMin) 
      return outMin;
    if (in <= inMax)
      return outMax;
  }
  // calculate how far into the range we are
  float scale = (in-inMin)/(inMax-inMin);
  // calculate the output.
  return outMin + scale*(outMax-outMin);
}

static void good() {
    ledGreen = 0;
    ledBlue = 1;
    ledRed = 1;
}

static void bad() {
    ledRed = 0;
    ledGreen = 1;
    ledBlue = 1;
}

static void event() {
    ledBlue = 0;
    ledRed = 1;
    ledGreen = 1;
}

static void pulseEvent() {
    event();
    wait_ms(500);
    good();
}

static void ledOff() {
    ledBlue = 1;
    ledRed = 1;
    ledGreen = 1;
}
static void cbButton()
{
    buttonPressed = true;
    pulseEvent();
}

int main()
{
    INTERFACE_CLASS *interface = new INTERFACE_CLASS();
    // If you need to debug the cellular interface, comment out the
    // instantiation above and uncomment the one below.
    // For the N2xx interface, change xxx to MBED_CONF_UBLOX_CELL_BAUD_RATE,
    // while for the non-N2xx interface change it to MBED_CONF_UBLOX_CELL_N2XX_BAUD_RATE.
//    INTERFACE_CLASS *interface = new INTERFACE_CLASS(MDMTXD, MDMRXD,
//                                                     xxx,
//                                                     true);
#ifndef TARGET_UBLOX_C030_N211
    TCPSocket sockTcp;   // for 1st sensor
    TCPSocket sockTcp1;  // for 2nd Sensor
#endif
   
    SocketAddress tcpServer;
    char deviceID[20] = "357520073946010";
    char buf[1024];  // this buffer is used to send POST data to platform
    char temp[50]; // this buffer will be used to store sensor values
    char soil[50]; // this buffer will be used to store 2nd sensor values 
    int x;

#ifdef TARGET_UBLOX_C027
    // No user button on C027
    InterruptIn userButton(NC);
#else
    InterruptIn userButton(SW0);
#endif
    
    // Attach a function to the user button
    userButton.rise(&cbButton);
    
    good();
    printf("Starting up, please wait up to 180 seconds for network registration to complete...\n");
    interface->set_credentials(APN, USERNAME, PASSWORD);
    for (x = 0; interface->connect(PIN) != 0; x++) {
        if (x > 0) {
            bad();
            printf("Retrying (have you checked that an antenna is plugged in and your APN is correct?)...\n");
        }
    }
    pulseEvent();
    
    printf("Getting the IP address of \"integration.campus.stcs.io\"...\n");
    if (interface->gethostbyname(TCP_SERVER, &tcpServer) == 0) {
        pulseEvent();

        tcpServer.set_port(1880);
        printf("\"integration.campus.stcs.io\" address: %s on port %d.\n", tcpServer.get_ip_address(), tcpServer.get_port());

        printf("Performing socket operations in a loop (until the user button is pressed on C030)...\n");
        while (!buttonPressed) {
            
#ifndef TARGET_UBLOX_C030_N211
        lcd.cls();
        lcd.locate(0,3);
        lcd.printf("Connecting");
        
           // TCP Sockets
            printf("=== TCP ===\n");
            printf("Opening a TCP socket...\n");
            if (sockTcp.open(interface) == 0 && sockTcp1.open(interface) == 0 ) {
                pulseEvent();
                printf("TCP socket open.\n");
                sockTcp.set_timeout(10000);
                printf("Connecting socket to %s on port %d...\n", tcpServer.get_ip_address(), tcpServer.get_port());
                if (sockTcp.connect(tcpServer) == 0 && sockTcp1.connect(tcpServer) == 0) {
                    pulseEvent();
                    
                    
                      lcd.cls();
                      lcd.locate(0,3);
                      lcd.printf("Connected");
                      wait(2.0);
                    printf("Connected, sending HTTP GET request to %s over socket...\n", TCP_SERVER);
                    
                    
                    ////////////////// SENDING 1st SENSOR VALUE /////////////////////
                    
                      lcd.cls();
                      lcd.locate(0,3);
                    
                   
                    snprintf(temp,sizeof temp,"Temperature=%.1f", sensor.temp());  //Sensor Key "Temperature" and Value is Sensor Data
                   
                    lcd.printf("%s", temp); // Display Temperature value on LCD
                         
                    
                    snprintf(buf, sizeof buf, "%s%s%s%s%s%s", "POST /ksu/?deviceID=",deviceID,"&",temp,"&unit=C"," HTTP/1.0\r\n\r\n");
                    
                    printf("Output string is %s",buf);
                                  
                    // Note: since this is a short string we can send it in one go as it will
                    // fit within the default buffer sizes.  Normally you should call sock.send()
                    // in a loop until your entire buffer has been sent.
                    if (sockTcp.send((void *) buf, strlen(buf)) == (int) strlen(buf)) {
                        pulseEvent();
                        printf("Socket send completed, waiting for response...\n");
                        x = sockTcp.recv(buf, sizeof (buf));
                        if (x > 0) {
                            pulseEvent();
                            printf("Received %d byte response from server on TCP socket:\n"
                                   "----------------------------------------------------\n%.*s"
                                   "----------------------------------------------------\n",
                                    x, x, buf);
                        }
                    }
                    
                    
                    
                    
                    
                    
                    /////////////////////////////////// Sending 2nd Sensor Value //////////////////////
                    
                    
                     float final_2 = moisture.read() * 3300;
                      uint16_t meas = (moisture.read_u16()); 
                     float final_1 = (float) 3300 / 65535 * (float) meas;
                     float newval = map(final_1,660,770,100,0);
                    
                    
                      lcd.cls();
                      lcd.locate(0,3);
                                
                    snprintf(soil,sizeof soil,"Moisture=%.1f", (float)map(final_1,660,770,100,0));  //Sensor Key "Moisture" and Value is Sensor Data
                   
                    lcd.printf("%s", soil); // Display Moisture value on LCD
                    wait(2.0);
                    
                    
                    snprintf(buf, sizeof buf, "%s%s%s%s%s%s", "POST /ksu/?deviceID=",deviceID,"&",soil,"&unit=%"," HTTP/1.0\r\n\r\n");
                    
                    printf("Output string is %s",buf);
                                  
                    // Note: since this is a short string we can send it in one go as it will
                    // fit within the default buffer sizes.  Normally you should call sock.send()
                    // in a loop until your entire buffer has been sent.
                    if (sockTcp1.send((void *) buf, strlen(buf)) == (int) strlen(buf)) {
                        pulseEvent();
                        printf("Socket send completed, waiting for response...\n");
                        x = sockTcp1.recv(buf, sizeof (buf));
                        if (x > 0) {
                            pulseEvent();
                            printf("Received %d byte response from server on TCP socket:\n"
                                   "----------------------------------------------------\n%.*s"
                                   "----------------------------------------------------\n",
                                    x, x, buf);
                        }
                    }
                    
                }
                      lcd.cls();
                      lcd.locate(0,3);
                      lcd.printf("Resend after 10s");
                      wait(1.0);
                      printf("Closing socket...\n");
                      sockTcp.close();
                      sockTcp1.close();
                      pulseEvent();
                      printf("Socket closed.\n");
            }
#endif
            wait_ms(10000);  //Loop delay in milliseconds. Keep it more than 10 seconds to save data usage on cellular
#ifndef TARGET_UBLOX_C027
            printf("[Checking if user button has been pressed]\n");
#endif
        }
        
        pulseEvent();
        printf("User button was pressed, stopping...\n");
        interface->disconnect();
        ledOff();
        printf("Stopped.\n");
    } 
    else {
        bad();
        printf("Unable to get IP address of \"integraton.campus.stcs.io\".\n");
    }
}


// End Of File
