/* mbed Microcontroller Library
 * Copyright (c) 2019 ARM Limited
 * SPDX-License-Identifier: Apache-2.0
 *
 * I2C Master code for Cypress PSoC 6 communicating with Arduino
 * =============================================================
 *
 * This module sets the PSoC 6 up as the master on the I2C bus
 * and periodically collects a set of environment data from an arduino
 * running the code below:
 */ 
// 
// //I2C SLAVE CODE
// //I2C Communication between Two Arduino
// //CircuitDigest
// //Pramoth.T
// 
// #include<Wire.h>                          //Library for I2C Communication // // functions
// #include <Adafruit_Sensor.h>
// #include <DHT.h>
// #include <DHT_U.h>
// 
// #define DHTPIN 4     // Digital pin connected to the DHT sensor 
// // Feather HUZZAH ESP8266 note: use pins 3, 4, 5, 12, 13 or 14 --
// // Pin 15 can work but DHT must be disconnected during program upload.
// 
// // Uncomment the type of sensor in use:
// #define DHTTYPE    DHT11     // DHT 11
// //#define DHTTYPE    DHT22     // DHT 22 (AM2302)
// //#define DHTTYPE    DHT21     // DHT 21 (AM2301)
// 
// // See guide for details on sensor wiring and usage:
// //   https://learn.adafruit.com/dht/overview
// 
// DHT_Unified dht(DHTPIN, DHTTYPE);
// 
// uint32_t delayMS;
// #include <PreciseLM35.h>
// #define TESTING false
// 
// const int pinLM35 = A2;
// PreciseLM35 lm35(pinLM35, DEFAULT);
// 
// 
// unsigned char txDataPacket[10];
// unsigned char rxCommandPacket[8];
// unsigned char rxCommandNum;
// unsigned char rxCommand = 0;
// bool SW1Pressed = 0;
// bool SW2Pressed = 0;
// 
// int blueLed = 13;
// int SW1Pin = 2;
// int SW2Pin = 3;
// 
// void setup()
// 
// {
//   pinMode(SW1Pin, INPUT);
//   pinMode(SW2Pin, INPUT);
//   pinMode(blueLed, OUTPUT);
//   dht.begin();
//   sensor_t sensor;
//   dht.temperature().getSensor(&sensor);
//   dht.humidity().getSensor(&sensor);
//   delayMS = sensor.min_delay / 1000;
//   Serial.begin(115200);                     //Begins Serial Communication at // 9600 baud rate
//   Wire.begin(8);                          //Begins I2C communication with // // Slave Address as 8 at pin (A4,A5)
//   Wire.onReceive(receiveEvent);           //Function call when Slave receives // value from master
//   Wire.onRequest(requestEvent);           //Function call when Master request // value from Slave
// }
// 
// void loop(void)
// {
//   unsigned char SW1Debounced;
//   unsigned char SW2Debounced;
//   int temperature = 20;
//   while (1) {
//     if (rxCommandNum || TESTING) {
// 
// 
//       Serial.println("Slave Received Command From Master:");   //Prints in // // Serial Monitor
//       Serial.print(rxCommand);
//       Serial.print(" - ");
//       Serial.println(rxCommandNum);
//       for (int i = 0; i < 8; i++) {
//         Serial.print( rxCommandPacket[i] );
//         Serial.print( " - " );
//       }
//       Serial.println(" ");
//       for (int i = 0; i < 10; i++) {
//         Serial.print( txDataPacket[i] );
//         Serial.print( " - " );
//       }
//       Serial.println(" ");
//       rxCommandNum = 0;
//     }
//     sensors_event_t event;
//     int potvalue = map( analogRead(A0), 0, 1023, 0, 100);                   // // Reads analog value from POT (0-5V)
//     int lightlevel = map(analogRead(A1), 0, 1023, 0, 100);                  // // Ambient light level
//     int humidityInt = 101;
//     temperature = ((int)((lm35.readCelsius()) * 10));                   // // // Ambient Temperature
//     dht.humidity().getEvent(&event);
//     if (isnan(event.relative_humidity)) {
//       Serial.println(F("Error reading humidity!"));
//     }
//     else {
//       humidityInt = (int)event.relative_humidity;
//       Serial.print(F("Humidity: "));
//       Serial.print(humidityInt);
//       Serial.println(F("%"));
//     }
//     txDataPacket[0] = potvalue;
//     txDataPacket[1] = 0;   // or potvalue >> 8;
//     txDataPacket[2] = lightlevel;
//     txDataPacket[3] = 0; // or lightlevel >> 8;
//     txDataPacket[4] = temperature;
//     txDataPacket[5] = temperature >> 8;
//     SW1Debounced = (SW1Debounced << 1) | digitalRead(SW1Pin);
//     if (SW1Debounced == 0) SW1Pressed = 1;
//     SW2Debounced = (SW2Debounced << 1) | digitalRead(SW2Pin);
//     if (SW2Debounced == 0) SW2Pressed = 1;
//     txDataPacket[6] = SW1Pressed;
//     txDataPacket[7] = SW2Pressed;
//     txDataPacket[8] = humidityInt;
//     txDataPacket[9] = 0;
//     digitalWrite(blueLed, rxCommandPacket[1] & 1);
//     delay(delayMS);
//     if (TESTING) delay(2000);
//   }
// }
// 
// void receiveEvent (int howMany)                    //This Function is called // when Slave receives value from master
// { for (int i = 0; i < howMany; i++) {
//     rxCommandPacket[i] = Wire.read();                    //Used to read value // received from master and store in variable SlaveReceived
//   }
//   rxCommandNum = howMany;
//   rxCommand = rxCommandPacket[0];
// } * 

#include "mbed.h"
#include "platform/mbed_thread.h"
#include "TextLCD.h"
#include "GUI.h"
#include "cy8ckit_028_tft.h"


#define UNO_CMD (0x10) // Command byte to Uno
#define UNO_CMD_CONF (0x01) // Configuration data
#define UNO_ADDR     (0x10) // Uno i2C address

//I2C i2c(P8_1, P8_0);
I2C i2c(I2C_SDA, I2C_SCL);
TextLCD_I2C lcd(&i2c, 0x7e, TextLCD::LCD16x2 /*, TextLCD::WS0010*/);                  // I2C exp: I2C bus, PCF8574 Slaveaddress, LCD Type
TextLCD_I2C lcd2(&i2c, 0x4e, TextLCD::LCD16x2 /*, TextLCD::WS0010*/);                  // I2C exp: I2C bus, PCF8574 Slaveaddress, LCD Type
DigitalOut myled(LED1);

//Serial pc(SERIAL_TX, SERIAL_RX);
Serial          pc(USBTX, USBRX);

volatile char TempCelsiusDisplay[] = "+abc.d C";

int main()
{

    char data_write[8];
    char data_read[10];
    char buffer[120];
//    i2c.frequency(400000);
    /* Initialise display */
    GUI_Init();
    GUI_Clear();

    lcd.cls();
    lcd.setBacklight(TextLCD::LightOn);
    lcd.setCursor(TextLCD::CurOff_BlkOff);
    lcd2.cls();
    lcd2.setBacklight(TextLCD::LightOn);
    lcd2.setCursor(TextLCD::CurOff_BlkOff);

    /* Configure the Temperature sensor device STLM75:
    - Thermostat mode Interrupt
    - Fault tolerance: 0
    */
    data_write[0] = UNO_CMD;
    data_write[1] = 0x08;

    GUI_SetFont(GUI_FONT_10_1);
    GUI_SetTextAlign(GUI_TA_LEFT);
    int status = i2c.write(UNO_ADDR, data_write, 2, 0);
    if (status != 0) { // Error
        GUI_DispStringAt("I2C Error", 0, 220);
        while (1) {
            myled = !myled;
            ThisThread::sleep_for(200);
        }
    }
    GUI_DispStringAt("I2C Connected", 0, 220);
    GUI_SetFont(GUI_FONT_20B_1);
    GUI_DispStringAt("Data from Arduino", 0, 0);

    while (1) {
        // Read Arduino data
        data_write[0] = UNO_CMD;
        i2c.write(UNO_ADDR, data_write, 2, 1); // no stop
        if (i2c.read(UNO_ADDR, data_read, 10, 0)) {
            GUI_SetFont(GUI_FONT_10_1);
            GUI_DispStringAt("I2C receive error  ", 0, 220);
        } else {
            GUI_SetFont(GUI_FONT_10_1);
            GUI_DispStringAt("I2C Data Received  ", 0, 220);

            // read TEN bytes
            // pot value - msb-1, lsb-0
            // light level - msb-3, lsb-2
            // temperature - msb-5, lsb-4
            // sw1 state - 6 - 0-255 based on how long pressed
            // sw2 state - 7 - 0-255 based on how long pressed
            // Humidity - msb-9, lsb-8

            // Calculate temperature value in Celcius
            int tempval = (int)((int)data_read[5] << 8) | data_read[4];
            if (tempval < 0) {
                TempCelsiusDisplay[0] = '-';
            } else {
                TempCelsiusDisplay[0] = '+';
            }

            // Integer part
            TempCelsiusDisplay[1] = (tempval / 1000) + 0x30;
            TempCelsiusDisplay[2] = ((tempval % 1000) / 100) + 0x30;
            TempCelsiusDisplay[3] = ((tempval % 1000) % 100 / 10) + 0x30;
            TempCelsiusDisplay[4] = '.';
            TempCelsiusDisplay[5] = ((tempval % 1000) % 100 % 10) + 0x30;

            //Switches

            unsigned char SW1State = data_read[6];
            unsigned char SW2State = data_read[7];

            //light level
            int lightDisplay = (int)((int)data_read[3] << 8) | data_read[2];

            //potentiometer value
            int potValue = (int)((int)data_read[1] << 8) | data_read[0];
            
            // humidity value
            int humidity = (int)((int)data_read[9] << 8) | data_read[8];

            // Display result
            pc.printf("temp = %s, Light is %3d%%, Hum is %3d%%%s%s\n", TempCelsiusDisplay, lightDisplay, humidity, SW1State?", SW1 pressed":"", SW2State?", SW2 pressed":"");

            lcd.locate(0,0);
            lcd.printf("Temp Lght Hum 12");
            lcd.locate(0,1);
            lcd.printf(" %2dC  %2d%c %2d%c %s%s", tempval/10, lightDisplay, 0x25, humidity, 0x25, SW1State?"1":"_", SW2State?"2":"_");
            lcd2.locate(0,0);
            lcd2.printf("Temp Lght Pot 12");
            lcd2.locate(0,1);
            lcd2.printf(" %2dC  %2d%c %2d%c %s%s", tempval/10, lightDisplay, 0x25, potValue, 0x25, SW1State?"1":"_", SW2State?"2":"_");

            sprintf(buffer,"Temp is %2fC \nLight Level is %2d%c \nPot Value is %2d%c \nHumidity is %2d%c \n%s \n%s ", tempval/10, lightDisplay, 0x25, potValue, 0x25, humidity, 0x25, SW1State?"SW1 Pressed ":"SW1 Released", SW2State?"SW2 Pressed ":"SW2 Released");
            GUI_SetFont(GUI_FONT_20_1);
            GUI_DispStringAt(buffer, 0, 40);
        }
        myled = !myled;
        data_write[1] = data_write[1] ^ 0x01;
        thread_sleep_for(250);
    }

}

