/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


#include "mbed.h"
#include "AppRTCSettings.h"
#include "lpc_swim_font.h"

/******************************************************************************
 * Defines and typedefs
 *****************************************************************************/
 
#define BTN_OFF    20
 
/******************************************************************************
 * Private variables
 *****************************************************************************/

// Ugly but needed for callbacks
static AppRTCSettings* theApp = NULL;

/******************************************************************************
 * Private functions
 *****************************************************************************/

static void buttonClicked(uint32_t x)
{
  bool* done = (bool*)x;
  *done = true;
}

static void fieldClicked(uint32_t x)
{
  if (theApp != NULL) {
    theApp->setActiveField(x);
  }
}

static void increaseValue(uint32_t x)
{
  AppRTCSettings* app = (AppRTCSettings*)x;
  app->modifyValue(1);
}

static void decreaseValue(uint32_t x)
{
  AppRTCSettings* app = (AppRTCSettings*)x;
  app->modifyValue(-1);
}

static void nextField(uint32_t x)
{
  AppRTCSettings* app = (AppRTCSettings*)x;
  app->changeActiveField(true);
}

static void prevField(uint32_t x)
{
  AppRTCSettings* app = (AppRTCSettings*)x;
  app->changeActiveField(false);
}

void AppRTCSettings::draw()
{
    // Prepare fullscreen
    swim_window_open(_win, 
                   _disp->width(), _disp->height(),         // full size
                   (COLOR_T*)_fb,
                   0,0,_disp->width()-1, _disp->height()-1, // window position and size
                   1,                                       // border
                   GREEN,WHITE,BLACK);/*WHITE, RED,  BLACK);*/                     // colors: pen, backgr, forgr
    swim_set_pen_color(_win, BLACK);
    swim_set_title(_win, "Real Time Clock Settings", GREEN);
    swim_set_pen_color(_win, BLACK);

    ImageButton* ib;
    
    Resource* rOk = _res[Resource_Ok_button];
    Resource* rCancel = _res[Resource_Cancel_button];
    ib =  new ImageButton(_win->fb, _win->xpmax - 2*BTN_OFF - rCancel->width() -rOk->width(), _win->ypmax - BTN_OFF - rOk->height(), rOk->width(), rOk->height());
    ib->loadImages(rOk);
    ib->draw();
    _buttons[ButtonOk] = ib;    
    ib =  new ImageButton(_win->fb, _win->xpmax - BTN_OFF - rCancel->width(), _win->ypmax - BTN_OFF - rCancel->height(), rCancel->width(), rCancel->height());
    ib->loadImages(rCancel);
    ib->draw();
    _buttons[ButtonCancel] = ib;    

    int arrowW = _res[Resource_ArrowUp_button]->width();
    int arrowH = _res[Resource_ArrowUp_button]->height();
    ib =  new ImageButton(_win->fb, 300, 40, arrowW, arrowH);
    ib->loadImages( _res[Resource_ArrowUp_button]);
    ib->setAction(increaseValue, (uint32_t)this);
    ib->draw();
    _buttons[ButtonUp] = ib;    
    ib =  new ImageButton(_win->fb, 300, 40+arrowH+arrowH, arrowW, arrowH);
    ib->loadImages( _res[Resource_ArrowDown_button]);
    ib->setAction(decreaseValue, (uint32_t)this);
    ib->draw();
    _buttons[ButtonDown] = ib;
    ib =  new ImageButton(_win->fb, 300-arrowW/2-10, 40+arrowH, arrowW, arrowH);
    ib->loadImages(_res[Resource_ArrowLeft_button]);
    ib->setAction(prevField, (uint32_t)this);
    ib->draw();
    _buttons[ButtonLeft] = ib;
    ib =  new ImageButton(_win->fb, 300+arrowW/2+10, 40+arrowH, arrowW, arrowH);
    ib->loadImages(_res[Resource_ArrowRight_button]);
    ib->setAction(nextField, (uint32_t)this);
    ib->draw();
    _buttons[ButtonRight] = ib;
    
    // To avoid having each DigitButton deallocate the shared image
    void* pointerToFree = _digitImage.pointerToFree;
    _digitImage.pointerToFree = NULL;

    addDateFields(0, 20);
    addTimeFields(0, 20+65);
    
    // Restore shared image so that it will be deallocated during teardown
    _digitImage.pointerToFree = pointerToFree;    
    
  for (int i = 0; i < NumButtons; i++) {
    _buttons[i]->draw();
  }
  
  markField(_activeField, true);
}

void AppRTCSettings::addDateFields(int xoff, int yoff)
{
    DigitButton* db;
    
    int fontY = yoff;
    int y = fontY + swim_get_font_height(_win) + 2;
    int btny = y-7;
    y += _win->ypvmin; // compensate for title bar 
    int x = xoff+20;
    int idx = ButtonYear;
    int btnw;
    
    swim_put_text_xy(_win, "Year", x, fontY);
    btnw = 65;
    db =  new DigitButton(_win->fb, x, y, btnw, 34);
    db->loadImages(&_digitImage);
    db->setNumDigits(4);
    db->setValue(_values[idx]);
    db->setAction(fieldClicked, idx);
    _buttons[idx++] = db;
    
    x += btnw + 20;
    btnw = 45;
    swim_put_text_xy(_win, "Month", x, fontY);
    db =  new DigitButton(_win->fb, x, y, btnw, 34);
    db->loadImages(&_digitImage);
    db->setNumDigits(2);
    db->setValue(_values[idx]);
    db->setAction(fieldClicked, idx);
    _buttons[idx++] = db;
    swim_put_box(_win, x-13, btny+20, x-2, btny+23);
    
    x += btnw + 10;    
    swim_put_text_xy(_win, "Day", x, fontY);
    db =  new DigitButton(_win->fb, x, y, btnw, 34);
    db->loadImages(&_digitImage);
    db->setNumDigits(2);
    db->setValue(_values[idx]);
    db->setAction(fieldClicked, idx);
    _buttons[idx++] = db;
    swim_put_box(_win, x-13, btny+20, x-2, btny+23);
}

void AppRTCSettings::addTimeFields(int xoff, int yoff)
{
    DigitButton* db;
    
    int fontY = yoff;
    int y = fontY + swim_get_font_height(_win) + 2;
    int btny = y-7;
    y += _win->ypvmin; // compensate for title bar 
    int x = xoff+20;
    int idx = ButtonHour;
    int btnw = 45;
    
    swim_put_text_xy(_win, "Hour", x, fontY);
    db =  new DigitButton(_win->fb, x, y, btnw, 34);
    db->loadImages(&_digitImage);
    db->setNumDigits(2);
    db->setValue(_values[idx]);
    db->setAction(fieldClicked, idx);
    _buttons[idx++] = db;
    
    x += btnw + 10;    
    swim_put_text_xy(_win, "Minutes", x, fontY);
    db =  new DigitButton(_win->fb, x, y, btnw, 34);
    db->loadImages(&_digitImage);
    db->setNumDigits(2);
    db->setValue(_values[idx]);
    db->setAction(fieldClicked, idx);
    _buttons[idx++] = db;
    swim_put_box(_win, x-13, btny+22, x-2, btny+25);
    
    x += btnw + 10;    
    swim_put_text_xy(_win, "Seconds", x, fontY);
    db =  new DigitButton(_win->fb, x, y, btnw, 34);
    db->loadImages(&_digitImage);
    db->setNumDigits(2);
    db->setValue(_values[idx]);
    db->setAction(fieldClicked, idx);
    _buttons[idx++] = db;
    swim_put_box(_win, x-13, btny+22, x-2, btny+25);
}

void AppRTCSettings::markField(int field, bool active)
{
  COLOR_T oldPen = _win->pen;
  COLOR_T oldFill = _win->fill;
  _win->fill = active ? BLACK : _win->bkg;
  _win->pen = active ? BLACK : _win->bkg;
  if (field >= 0 && field < NumFields) {
    int x0, y0, x1, y1;
    _buttons[field]->bounds(x0,y0,x1,y1);
    y1 -= _win->ypvmin+1;
    x0--;
    swim_put_box(_win, x0, y1, x1, y1+3);
  }
  _win->fill = oldFill;
  _win->pen = oldPen;
}

/******************************************************************************
 * Public functions
 *****************************************************************************/

AppRTCSettings::AppRTCSettings() : _disp(NULL), _win(NULL), _fb(NULL), _activeField(0)
{
  for (int i = 0; i < NumButtons; i++) {
    _buttons[i] = NULL;
  }
  for (int i = 0; i < NumResources; i++) {
    _res[i] = NULL;
  }
  time_t rawtime;
  struct tm * timeinfo;

  time (&rawtime);
  timeinfo = localtime (&rawtime);
  
  _values[ButtonYear] = timeinfo->tm_year + 1900;
  _values[ButtonMonth] = timeinfo->tm_mon + 1;
  _values[ButtonDay] = timeinfo->tm_mday;
  _values[ButtonHour] = timeinfo->tm_hour;
  _values[ButtonMinute] = timeinfo->tm_min;
  _values[ButtonSecond] = timeinfo->tm_sec;
  
  _digitImage.pointerToFree = NULL;
  
  theApp = this;
}

AppRTCSettings::~AppRTCSettings()
{
    theApp = NULL;
    teardown();
}

bool AppRTCSettings::setup()
{
    _disp = DMBoard::instance().display();
    _win = (SWIM_WINDOW_T*)malloc(sizeof(SWIM_WINDOW_T));
    _fb = _disp->allocateFramebuffer();
    int res = Image::decode(_res[Resource_Digits], Image::RES_16BIT, &_digitImage);

    return (_win != NULL && _fb != NULL && res == 0);
}

void AppRTCSettings::runToCompletion()
{
    // Alternative 1: use the calling thread's context to run in
    bool done = false;
    bool abort = false;
    draw();
    _buttons[ButtonOk]->setAction(buttonClicked, (uint32_t)&done);
    _buttons[ButtonCancel]->setAction(buttonClicked, (uint32_t)&abort);
    void* oldFB = _disp->swapFramebuffer(_fb);
    
    // Wait for touches
    TouchPanel* touch = DMBoard::instance().touchPanel();
    touch_coordinate_t coord;

    int lastPressed = NumButtons;
    Timer t;
    t.start();
    int repeatAt = 0;

    uint32_t maxDelay = osWaitForever; 
    while(!done && !abort) {
      Thread::signal_wait(0x1, maxDelay);
      if (touch->read(coord) == TouchPanel::TouchError_Ok) {
        for (int i = 0; i < NumButtons; i++) {
          if (_buttons[i]->handle(coord.x, coord.y, coord.z > 0)) {
            _buttons[i]->draw();
            if (_buttons[i]->pressed()) {
              lastPressed = i; // new button pressed
              t.reset();
              repeatAt = 1000;
            }
          }
        }
        if (lastPressed == ButtonUp || lastPressed == ButtonDown) {
          maxDelay = 10;
          if (_buttons[lastPressed]->pressed() && t.read_ms() > repeatAt) {
            modifyValue((lastPressed == ButtonUp) ? 1 : -1);
            repeatAt = t.read_ms()+(200/(t.read_ms()/1000));
          }
        } else {
          maxDelay = osWaitForever;
        }
      }
    }

    if (!abort) {
      RtosLog* log = DMBoard::instance().logger();
      log->printf("New time: %04d-%02d-%02d  %02d:%02d:%02d\n", _values[0], _values[1],
          _values[2], _values[3], _values[4], _values[5]);
        tm t;
        t.tm_year = _values[ButtonYear] - 1900;  // years since 1900
        t.tm_mon = _values[ButtonMonth] - 1;      // month is 0..11
        t.tm_mday = _values[ButtonDay];
        t.tm_hour = _values[ButtonHour];
        t.tm_min = _values[ButtonMinute];
        t.tm_sec = _values[ButtonSecond];
        set_time(mktime(&t));
    }
    
    // User has clicked the button, restore the original FB
    _disp->swapFramebuffer(oldFB);
    swim_window_close(_win);
}

bool AppRTCSettings::teardown()
{
    if (_win != NULL) {
        free(_win);
        _win = NULL;
    }
    if (_fb != NULL) {
        free(_fb);
        _fb = NULL;
    }
    for (int i = 0; i < NumButtons; i++) {
        if (_buttons[i] != NULL) {
            delete _buttons[i];
            _buttons[i] = NULL;
        }
    }
    if (_digitImage.pointerToFree != NULL) {
        free(_digitImage.pointerToFree);
        _digitImage.pointerToFree = NULL;
    }
    return true;
}

void AppRTCSettings::modifyValue(int mod)
{
  uint32_t v = _values[_activeField];
  switch (_activeField) {
      case ButtonYear:
          // 1970..2100
          _values[_activeField] = (((v - 1970 + (2100-1970)) + mod) % (2100-1970)) + 1970;
          break;
      case ButtonMonth:
          // 1..12
          _values[_activeField] = (((v - 1 + 12) + mod) % 12) + 1;
          break;
      case ButtonDay:
          // 1..31
          _values[_activeField] = (((v - 1 + 31) + mod) % 31) + 1;
          break;
      case ButtonHour:
          // 0..23
          _values[_activeField] = (v + 24 + mod) % 24;
          break;
      case ButtonMinute:
      case ButtonSecond:
          // 0..59
          _values[_activeField] = (v + 60 + mod) % 60;
          break;
  }
  ((DigitButton*)_buttons[_activeField])->setValue(_values[_activeField]);
}

void AppRTCSettings::changeActiveField(bool next)
{
  markField(_activeField, false);
  if (next) {
    _activeField = (_activeField+1) % NumFields;
  } else {
    _activeField = (_activeField+NumFields-1) % NumFields;
  }
  markField(_activeField, true);
}

void AppRTCSettings::setActiveField(uint32_t newField)
{
  if (_activeField != newField && newField < NumFields) {
    markField(_activeField, false);
    _activeField = newField;
    markField(_activeField, true);
  }
}

void AppRTCSettings::addResource(Resources id, Resource* res)
{
    _res[id] = res;
}


