/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


#include "mbed.h"
#include "AppTemplate.h"
#include "lpc_swim_font.h"
#include "lpc_colors.h"
#include "image_data.h"

/******************************************************************************
 * Defines and typedefs
 *****************************************************************************/
 
#define BTN_WIDTH  65
#define BTN_HEIGHT 40
#define BTN_OFF    20

/******************************************************************************
 * Local variables
 *****************************************************************************/


/******************************************************************************
 * Private functions
 *****************************************************************************/

static void buttonClicked(uint32_t x)
{
  bool* done = (bool*)x;
  *done = true;
}

void AppTemplate::draw()
{
    // Prepare fullscreen
    swim_window_open(_win, 
                   _disp->width(), _disp->height(),         // full size
                   (COLOR_T*)_fb,
                   0,0,_disp->width()-1, _disp->height()-1, // window position and size
                   0,                                       // border
                   BLUE, WHITE, BLACK);                    // colors: pen, backgr, forgr
    
    // Show a message
    swim_put_text_xy(_win, "Hello World!", 100, 100);
    
    // Create the OK button that will end this application
    _btn = new ImageButton(_win->fb, _win->xpmax - BTN_OFF - BTN_WIDTH, _win->ypmax - BTN_OFF - BTN_HEIGHT, BTN_WIDTH, BTN_HEIGHT);
    _btn->loadImages(img_ok, img_size_ok);
    _btn->draw();
}

/******************************************************************************
 * Public functions
 *****************************************************************************/

AppTemplate::AppTemplate() : _disp(NULL), _win(NULL), _fb(NULL), _btn(NULL)
{
}

AppTemplate::~AppTemplate()
{
    teardown();
}

bool AppTemplate::setup()
{
    _disp = DMBoard::instance().display();
    _win = (SWIM_WINDOW_T*)malloc(sizeof(SWIM_WINDOW_T));
    _fb = _disp->allocateFramebuffer();

    return (_win != NULL && _fb != NULL);
}

void AppTemplate::runToCompletion()
{
    // Draw on the frame buffer
    draw();
    
    // Configure the button to call buttonClicked() when clicked
    bool done = false;
    _btn->setAction(buttonClicked, (uint32_t)&done);
    
    // Switch to "our" frame buffer, but save the old one so it can be 
    // restored when we exit our application
    void* oldFB = _disp->swapFramebuffer(_fb);
    
    TouchPanel* touch = DMBoard::instance().touchPanel();
    touch_coordinate_t coord;
    while(!done) {
      // wait for a new touch signal (signal is sent from AppLauncher,
      // which listens for touch events)
      Thread::signal_wait(0x1);
      if (touch->read(coord) == TouchPanel::TouchError_Ok) {
        
        // you can do something where with the touch event if you like
        // for example
        //
        //  swim_put_circle(_win, coord.x, coord.y, 2, 1);
        //
        
        
        // See if the touch event effects the button
        if (_btn->handle(coord.x, coord.y, coord.z > 0)) {
          _btn->draw();
        }
      }
    }
    
    // User has clicked the button, restore the original frame buffer
    _disp->swapFramebuffer(oldFB);
    swim_window_close(_win);
}

bool AppTemplate::teardown()
{
    if (_win != NULL) {
        free(_win);
        _win = NULL;
    }
    if (_fb != NULL) {
        free(_fb);
        _fb = NULL;
    }
    if (_btn != NULL) {
        delete _btn;
        _btn = NULL;
    }
    return true;
}

