#include "mbed.h"

/*****Globals*****/

// Motor Selectors
#define x 'x'
#define y 'y'
#define f 'f'

// Direction Selectors
#define cw 0 // = 0
#define ccw 1 // = 1

// Motor Speed
#define xMotorPeriod 4 // 25kHz
#define yMotorPeriod 4 // 25kHz
#define fMotorPeriod 15

#define xFreq 25000
#define yFreq 25000

class Drive{
    public:
        int stepCount, position, distance, dir, period;
        Drive(PinName pulse, PinName direction, int motorPeriod) : _pulse(pulse), _direction(direction) {
            _pulse.write(0.0f);
            _direction = 0;
            distance = 1;
            stepCount = 0;
            period = motorPeriod;
            dir = cw;
        }
        void on(){
            _pulse.period_us(period);   // set period
            _pulse.write(0.50f);              // 50% duty cycle  
        }
        // edit members
        void setDistance(int value){
            distance = value;
        }
        void setDirection(int value){
            dir = value;
        }
        void stepIncr(){
            stepCount++;
            if(stepCount == distance){ // stop motor
                _pulse.write(0.0f);
                stepCount = 0;
            }
        }
        void resetCount(){
            stepCount = 0;
        }
    private:
        PwmOut _pulse;
        DigitalOut _direction;
};

// Motors
Drive xMotor(p7, p10, xMotorPeriod);
Drive yMotor(p8, p11, yMotorPeriod);
Drive fMotor(p9, p12, fMotorPeriod);

InterruptIn xStepRead(p1);
InterruptIn yStepRead(p2);
InterruptIn fStepRead(p3);

// LEDs
DigitalOut myled1(LED1);
DigitalOut myled2(LED2);
DigitalOut myled3(LED3);
DigitalOut myled4(LED4);

/*****Functions*****/

// motor drive functions
void motorDrive(int steps, char motor, int direction){
    // Input int pulse for distance // 1 pulse =
    // Input char motor as the drive motor
    // Return value of 0 means freq input was negative, 1 success
    // No more than one motor can run at a time (PWM limit)

    /* Example
    int main(){ // Generates a 25 kHz PWM for 15 seconds
        motorPWM(500, 'x');
        wait(15.0);
        motorStop('x');
    }
    */
    switch(motor){
        case 'x':   // x track motor
            xMotor.resetCount();
            xMotor.setDistance(steps);
            xMotor.setDirection(direction);

            xStepRead.rise(&xMotor, &Drive::stepIncr);
            wait_us(5);
            xMotor.on();
            break;
        case 'y':   // y track motor    
            yMotor.resetCount();
            yMotor.setDistance(steps);
            yMotor.setDirection(direction);
            
            yStepRead.rise(&yMotor, &Drive::stepIncr);
            wait_us(5);
            yMotor.on();   
            break;
        case 'f':   // friction motor
            fMotor.resetCount();
            fMotor.setDistance(steps);
            fMotor.setDirection(direction);
            
            fStepRead.rise(&fMotor, &Drive::stepIncr);
            wait_us(5);
            fMotor.on();
            break;
    }
}

int main() {
    myled1 = 1;
    myled2 = 1;
    myled3 = 1;
    myled4 = 1;

    while(1){
        motorDrive(xFreq*5, x, cw);
        wait(10.0);
        motorDrive(yFreq*5, y, cw);
        wait(10.0);
    }
}