/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

#ifndef RESOURCE_H
#define RESOURCE_H

#include "mbed.h"
#include "Image.h"

/**
 * A resource container to pass around instead of using hardcoded data.
 *
 * Having a Resource class makes it possible to hide the source from the
 * user so that e.g. AppColorPicker doesn't need to know if the OK button
 * is loaded from an array or from a file system. The AppColorPicker only
 * need the resource. The Image class takes care of the loading.
 *
 * @code
 * #include "mbed.h"
 * #include "Resource.h"
 * #include "Image.h"
 *
 * const unsigned char ok_image1[] = { 137,80,78,71, ... };
 * int ok_image1_sz = sizeof(ok_image1);
 *
 * int main(void) {
 *    // initialize the display
 *    ...
 *
 *    // create the resources
 *    Resource resOk(ok_image1, ok_image1_sz, 40, 40);
 *    Resource resCancel("/qspi/cancel.png", 40, 40);
 *
 *    // add resource to application
 *    AppNetwork app ...;
 *    app.setResource(AppNetwork::Resource_Ok_Button, &resOk);
 *    app.setResource(AppNetwork::Resource_Cancel_Button, &resCancel);
 *
 *    // run application
 *    app.setup();
 *    app.runToCompletion();
 *
 *    ...
 * }
 * @endcode
 */
class Resource {
public:

    /** Create a new file resource.
     *
     *  The width and height are only guidelines and are often not used.
     *  The purpose is to allow e.g. the AppColorPicker to know the size
     *  of the image before it is loaded from the file system. It can also
     *  be used as a guide by the AppLauncher.
     *
     *  @param filename  the resource location
     *  @param width     the width of the image
     *  @param height    the height of the image
     */
    Resource(const char* filename, int width, int height);

    /** Create a new resource from a data array.
     *
     *  The width and height are only guidelines and are often not used.
     *  The purpose is to allow e.g. the AppColorPicker to know the size
     *  of the image before it is loaded from the file system. It can also
     *  be used as a guide by the AppLauncher.
     *
     *  @param data      the resource
     *  @param dataSize  number of bytes in data
     *  @param width     the width of the image
     *  @param height    the height of the image
     */
    Resource(const unsigned char* data, const unsigned int dataSize, int width, int height);
    ~Resource();

    /** Returns the width of the resource.
     *
     *  The width and height are only guidelines and are often not used.
     *  The purpose is to allow e.g. the AppColorPicker to know the size
     *  of the image before it is loaded from the file system. It can also
     *  be used as a guide by the AppLauncher.
     *
     *  @return the width
     */
    int width() { return _width; }

    /** Returns the height of the resource.
     *
     *  The width and height are only guidelines and are often not used.
     *  The purpose is to allow e.g. the AppColorPicker to know the size
     *  of the image before it is loaded from the file system. It can also
     *  be used as a guide by the AppLauncher.
     *
     *  @return the height
     */
    int height() { return _height; }

private:
    friend class Image;
    
    int _width;
    int _height;
    bool _isFile;
    char* _filename;
    const unsigned char* _data;
    unsigned int _dataSize;

    Image::ImageData_t _img;
};

#endif
