/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
#ifndef IMAGEBUTTON_H
#define IMAGEBUTTON_H

#include "Clickable.h"
#include "Image.h"
#include "Resource.h"

/**
 * The ImageButton is used in the same way as the Button so see it for an example
 */
class ImageButton : public Clickable {
public:

    /** Creates a new button
     *
     *  This button will use a SWIM window to draw on. That window will use
     *  part of the full size frame buffer to draw on.
     *
     *  @param fb      the frame buffer
     *  @param x       the upper left corner of the button
     *  @param y       the upper left corner of the button
     *  @param width   the width of the button
     *  @param height  the height of the button
     *  @param caption optional text to put below the image
     *  @param color   text color
     */
  ImageButton(COLOR_T* fb, uint16_t x, uint16_t y, uint16_t width, uint16_t height, 
              const char* caption=NULL, COLOR_T color=BLACK);
  virtual ~ImageButton();

    /** Loads the mandatory "normal" state image and the optional "pressed" state image
     *
     *  @param imgUp   the file with the image for the normal state
     *  @param imgDown the file with the image for the pressed state (or NULL to use the same)
     *
     *  @returns
     *       true on success
     *       false on failure
     */
  bool loadImages(const char* imgUp, const char* imgDown = 0);

    /** Loads the mandatory "normal" state image and the optional "pressed" state image
     *
     *  @param imgUp      the image for the normal state
     *  @param imgUpSize  the size of the imgUp data
     *  @param imgDown    the image for the pressed state (or NULL to use the same)
     *  @param imgUpSize  the size of the imgDown data
     *
     *  @returns
     *       true on success
     *       false on failure
     */
  bool loadImages(const unsigned char* imgUp, unsigned int imgUpSize, 
                  const unsigned char* imgDown = 0, unsigned int imgDownSize = 0);

    /** Loads the mandatory "normal" state image and the optional "pressed" state image
     *
     *  @param resImgUp   the resource for the normal state image
     *  @param resImgDown the resource for the pressed state image (or NULL to use the same)
     *
     *  @returns
     *       true on success
     *       false on failure
     */
  bool loadImages(Resource* resImgUp, Resource* resImgDown = 0);

    /** Specifys a color that will be considered transparent (i.e. will not be drawn)
     *  @param tColor  the transparent color
     */
  void setTransparency(COLOR_T tColor);

    /** Draws the button (on a new framebuffer if one is specified)
     *  @param fb      the frame buffer
     */
  virtual void draw(COLOR_T* fb = 0);

private:
  Image::ImageData_t _imgUp;
  Image::ImageData_t _imgDown;
  char* _caption;
  COLOR_T _captionColor;
  bool _transparent;
  COLOR_T _transparentColor;
};

#endif /* IMAGEBUTTON_H */
