/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 
#include "mbed.h"
#include "Renderer.h"
#include "DMBoard.h"

/******************************************************************************
 * Defines and typedefs
 *****************************************************************************/

#if !defined(MIN)
  #define MIN(__a, __b)  (((__a)<(__b))?(__a):(__b))
#endif

/******************************************************************************
 * Public functions
 *****************************************************************************/

Renderer::Renderer()
{
  memset(layers, 0, sizeof(layerinfo_t)*MaxNumLayers);
  for (int i = 0; i < MaxNumLayers; i++) {
    layers[i].used = false;
    order[i] = NULL;
  }

  numRegisteredLayers = 0;
  activeBackBuffer = 0;
  display = DMBoard::instance().display();

  this->screenWidth = display->width();
  this->screenHeight = display->height();
  this->screenPixels = this->screenWidth * this->screenHeight;

  // Assume screen->bpp == Bpp_16
  this->screenBytes = display->bytesPerPixel() * this->screenPixels;

  ImageBackBuffer[0] = (image_t)display->allocateFramebuffer();
  ImageBackBuffer[1] = (image_t)display->allocateFramebuffer();
  if ((ImageBackBuffer[0] == NULL) || (ImageBackBuffer[1] == NULL)) {
    printf("Failed to allocate buffer(s) for Renderer. Halting...\n");
    while(1) {
      ;
    }
  }
  memset(ImageBackBuffer[0], 0xff, screenBytes);
  memset(ImageBackBuffer[1], 0xff, screenBytes);
}

Renderer::~Renderer()
{
  free(ImageBackBuffer[0]);
  free(ImageBackBuffer[1]);
  for (int i = 0; i < MaxNumLayers; i++) {
    if (layers[i].used) {
      free(layers[i].tmp);
      delete(layers[i].lock);
    }
  }
}

uint32_t Renderer::registerUser(int layer, int xoff, int yoff, int width, int height)
{
  setupMutex.lock();
  if (numRegisteredLayers < MaxNumLayers) {
    for (int i = 0; i < MaxNumLayers; i++) {
      if (!layers[i].used) {
        // found a spot
        layers[i].used = true;
        layers[i].x0 = xoff;
        layers[i].y0 = yoff;
        layers[i].x1 = MIN(xoff+width, screenWidth);
        layers[i].y1 = MIN(yoff+height, screenHeight);
        layers[i].width = width;
        layers[i].clippedHeight = layers[i].y1 - layers[i].y0;
        layers[i].zorder = layer;
        layers[i].signalId = 1;//(1<<i);
        layers[i].lock = new Mutex();
        layers[i].newData = NULL;
        layers[i].activeData = NULL;
        layers[i].fullscreen = false;
        if ((xoff == 0) && (yoff == 0) && (width == screenWidth) && (height == screenHeight)) {
          layers[i].fullscreen = true;
        }

        layers[i].tmp = (uint16_t*)malloc(width*layers[i].clippedHeight*2);
        if (layers[i].tmp != NULL) {
          memset(layers[i].tmp, 0, width*layers[i].clippedHeight*2);
        }


        // sort the order
        for (int j = 0; j < MaxNumLayers; j++) {
          if (order[j] == NULL) {
            // no more layers so add the new one to the top
            order[j] = &(layers[i]);
            break;
          }
          if (order[j]->zorder > layer) {
            // should insert the new layer here
            for (int k = numRegisteredLayers; k > j; k--) {
              order[k] = order[k-1];
            }
            order[j] = &(layers[i]);
            break;
          }
        }

        // Cause a repaint of all layers. It does not have to be immediate
        // as for unregisterUser() - it is enough that it is done next time.
        order[0]->activeData = NULL;

        numRegisteredLayers++;

        setupMutex.unlock();
        return (uint32_t)(&(layers[i]));
      }
    }
  }
  setupMutex.unlock();
  return 0;
}

uint32_t Renderer::registerFullscreenUser(int layer)
{
  return registerUser(layer, 0, 0, screenWidth, screenHeight);
}

void Renderer::unregisterUser(uint32_t handle)
{
  setupMutex.lock();
  if (handle != 0) {
    layerinfo_t* layer = (layerinfo_t*)handle;
    for (int i = 0; i < MaxNumLayers; i++) {
      if (order[i] == layer) {
        layer->used = false;
        free(layer->tmp);
        delete layer->lock;

        // move all layers "down" one step
        for (int j = i+1; j<numRegisteredLayers; j++) {
          order[j-1] = order[j];
        }
        order[numRegisteredLayers-1] = NULL;
        numRegisteredLayers--;

        // cause a repaint
        if (order[0] != NULL) {
          order[0]->activeData = NULL;

          // Very important that the signal is not sent while the lock is held
          // as it will cause the renderer thread be able to take it also
          setupMutex.unlock();
          osSignalSet(threadId, layer->signalId);
          return;
        } else {
          // no longer anything to show, clear back buffers
          memset(ImageBackBuffer[0], 0xff, screenBytes);
          memset(ImageBackBuffer[1], 0xff, screenBytes);
        }

        break;
      }
    }
  }
  setupMutex.unlock();
}

void Renderer::setFramebuffer(uint32_t handle, const uint16_t* data)
{
  layerinfo_t* layer = (layerinfo_t*)handle;

  // make sure that the render thread is not using the data when
  // we change it
  layer->lock->lock();
  layer->newData = data;
  layer->activeData = NULL;
  layer->lock->unlock();

  // notify the renderer that there is new data for our layer
  osSignalSet(threadId, layer->signalId);
}

void Renderer::render()
{
  //save this thread's ID so that it may be signalled from 
  //unregisterUser() and setFramebuffer()
  threadId = Thread::gettid();
    
  int mask = 1;//(1<<MaxNumLayers) - 1;
  while(true)
  {
    osEvent ev = Thread::signal_wait(mask);
    if (ev.status == osEventSignal) {
      setupMutex.lock();
      for (int i = 0; i < numRegisteredLayers; i++) {
        if (order[i]->activeData != order[i]->newData) {
        //if (order[i]->signalId & ev.value.signals) {

          int n = (activeBackBuffer + 1) & 1;

//          if (i == 0) {
//            // TEMPORARY
//            board->setFrameBuffer((uint32_t)ImageBackBuffer[n]);
//            if (order[i]->activeData != NULL && order[i]->newData != NULL) {
//              for (int y = 0; y < order[i]->clippedHeight; y++) {
//                for (int x = 0; x < order[i]->width; x++) {
//                  if (order[i]->activeData[y*order[i]->width+x] != order[i]->newData[y*order[i]->width+x]) {
//                    printf("Difference in x,y {%d,%d} active is 0x%04x, new is 0x%04x\n", x, y,
//                           order[i]->activeData[y*order[i]->width+x], order[i]->newData[y*order[i]->width+x]);
//                  }
//                }
//              }
//            }
//          }

          // Layer i have new data
          // redraw this layer and all on top of it
          for (int j = i; j < numRegisteredLayers; j++) {
            layerinfo_t* layer = order[j];
            layer->lock->lock();
            layer->activeData = layer->newData;
            memcpy(layer->tmp, layer->newData, layer->width*layer->clippedHeight*2);
            layer->lock->unlock();
            if (layer->activeData != NULL) {
              if (layer->fullscreen) {
                memcpy(ImageBackBuffer[n], layer->tmp, screenBytes);
              } else {
                for (int y = 0; y < layer->clippedHeight; y++) {
                  memcpy(ImageBackBuffer[n] + ((y + layer->y0) * screenWidth) + layer->x0,
                         layer->tmp + (y * layer->width),
                         (layer->x1 - layer->x0) * 2);
                }
              }
            }
          }

          //board->setFrameBuffer((uint32_t)ImageBackBuffer[n]);
          display->setFramebuffer(ImageBackBuffer[n]);
          if (i == (numRegisteredLayers - 1)) {
            // only top layer changed, no need to copy entire back buffer
          } else {
            // more layers exist on top so the back buffer must be updated
            memcpy(ImageBackBuffer[activeBackBuffer], ImageBackBuffer[n], screenBytes);
          }
          activeBackBuffer = n;
          break;
        }
      }
      setupMutex.unlock();
    }
  }
}
