/*
 *  Copyright 2014 Embedded Artists AB
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

#ifndef APP_H
#define APP_H

/**
 * Base class for all Apps. The idea is that you can have multiple
 * Apps in your program. Each App has three funtions:
 *
 *  setup() is called to let the App load all needed resources
 *
 *  runToCompletion() is called and will not return until the App exits
 *
 *  teardown() is called to let the App free allocated resources
 *
 * For an example of Apps in use, see the AppLauncher class
 */
class App {
public:
	virtual ~App() {};

    /** Prepare the App before it is started.
     *
     *  This function can be implemented to allocate memory, load startup
     *  images or other time consuming preparations.
     *
     *  The return value is to let the caller now if the application can
     *  be started or not.
     *
     *  @param x       the touched x coordinate
     *  @param y       the touched y coordinate
     *  @param pressed true if the user pressed the display
     *
     *  @returns
     *       true if the app can be started
     *       false if the setup failed
     */
    virtual bool setup() { return true; }
    
    /** Runs the App to completion
     *
     *  This function should not return until the App has finished.
     */
    virtual void runToCompletion() = 0;
    
    /** Cleanup
     *
     *  Implement to free up the memory allocated in setup().
     *
     *  @returns
     *       true if the teardown finished successfully
     *       false if the teardown failed
     */
    virtual bool teardown() { return true; }
};

#endif
